using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using Volpe.Cafe.Utils;
using Volpe.XlLib;
using Volpe.Cafe.UI.Properties;
using System.Reflection;
namespace Volpe.Cafe.UI
{
    public sealed class Program
    {
        #region 
        private Program()
        {
        }
        #endregion
        #region 
        [STAThread]
        static void Main(string[] args)
        {
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);
            AppDomain.CurrentDomain.AssemblyResolve += new ResolveEventHandler(CurrentDomain_AssemblyResolve);
            Console.WriteLine("Starting run at:  {0}", DateTime.Now.ToString());
            Program.Run();
            Console.WriteLine("Exiting model at: {0}", DateTime.Now.ToString());
        }
        static void Run()
        {
            try
            {
                if (Program.ShowWarningsScreen(true) == DialogResult.OK)
                {
                    int xlCheck = XlUtilities.CompatibilityCheck();
                    string message = null;
                    if (xlCheck == -1)
                    {   
                        message = "A version of Excel could not be found on the current system.  The model will not be " +
                            "able to load input files or generate output reports.\n\nPress OK to continue running the model.";
                    }
                    else if (xlCheck == 0)
                    {   
                        message = "The version of Excel installed on the current system is not supported by the model.  " +
                            "Errors may occur when loading input files or generating output reports.\n\n" +
                            "Press OK to continue running the model.";
                    }
                    if (message != null)
                    {
                        RtfDialog.ShowDialog("Compatibility Warning", message,
                            new Font(Cafe2d.DefaultFont.FontFamily, 10, FontStyle.Regular), Color.DarkRed, PromptOption.Ok, true, false);
                    }
                    Application.Run(new Cafe2d());
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "CAFE Model Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                Console.WriteLine(ex.ToString());
            }
        }
        static Assembly CurrentDomain_AssemblyResolve(object sender, ResolveEventArgs args)
        {
            string resource = "Volpe.Cafe.UI.Resources.DLL." + (new AssemblyName(args.Name)).Name + ".dll";
            Stream stream = Assembly.GetExecutingAssembly().GetManifestResourceStream(resource);
            byte[] data = new byte[stream.Length];
            stream.Read(data, 0, data.Length);
            return Assembly.Load(data);
        }
        internal static DialogResult ShowWarningsScreen(bool showCancelButton)
        {
            MemoryStream warnStream = new MemoryStream(Resources.Warnings_rtf);
            PromptOption promptOption = (showCancelButton) ? PromptOption.OkCancel : PromptOption.Ok;
            DialogResult dr = RtfDialog.ShowDialogFromStream("CAFE Model Warnings", warnStream, promptOption, false, false);
            warnStream.Close();
            return dr;
        }
        internal static DialogResult ShowNoticeScreen()
        {
            MemoryStream noticeStream = new MemoryStream(Resources.Notice_rtf);
            DialogResult dr = RtfDialog.ShowDialogFromStream("CAFE Model Notice", noticeStream, PromptOption.Ok, false, true);
            noticeStream.Close();
            return dr;
        }
        internal static DialogResult ShowAboutScreen()
        {
            MemoryStream stream = new MemoryStream(Resources.About_rtf);
            DialogResult dr = RtfDialog.ShowDialogFromStream("About CAFE Model", stream, PromptOption.Ok, true, false);
            stream.Close();
            return dr;
        }
        #endregion
    }
}

