﻿#region << Using Directives >>
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
#endregion

namespace Volpe.Cafe.UI
{
    /// <summary>
    /// Provides a customized navigation button.
    /// </summary>
    public class NavButton : Control
    {

        #region /*** Events ***/

        /// <summary>Occurs when the control is clicked by the left mouse button.</summary>
        public event EventHandler MouseLeftClick;

        #endregion


        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="NavButton"/> class.
        /// </summary>
        public NavButton()
        {
            this.InitializeComponent();
            this.DoubleBuffered = true;
        }
        /// <summary> 
        /// Clean up any resources being used.
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
        }

        #endregion


        #region /*** Methods ***/

        #region /* Methods for raising events */

        /// <summary>
        /// Raises the <see cref="MouseLeftClick"/> event.
        /// </summary>
        /// <param name="e">Event data to pass to the event.</param>
        protected virtual void OnMouseLeftClick(MouseEventArgs e)
        {
            if (this.MouseLeftClick != null)
            {
                this.MouseLeftClick(this, e);
            }
        }

        #endregion

        #region /* Custom painting methods */

        /// <summary>
        /// Raises the <see cref="Control.Paint"/> event.
        /// </summary>
        /// <param name="e">A <see cref="PaintEventArgs"/> that contains the event data.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            //
            this.PaintImage (e.Graphics);
            this.PaintText  (e.Graphics);
            this.PaintBorder(e.Graphics);
        }
        /// <summary>
        /// Paints the background of the control.
        /// </summary>
        /// <param name="e">A <see cref="PaintEventArgs"/> that contains the event data.</param>
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            base.OnPaintBackground(e);
            //
            Graphics  g    = e.Graphics;
            Rectangle rect = new Rectangle(0, 0, this.Width, this.Height);
            //
            if (this._mouseHover) { g.FillRectangle(new SolidBrush(this._hoverBackColor), rect); }
            else                  { g.FillRectangle(new SolidBrush(this.BackColor      ), rect); }
        }
        void PaintImage(Graphics g)
        {
            if (this._showImage && this._image != null)
            {
                int w = this.Width;
                int h = this.Height;
                Rectangle rect = new Rectangle((this._imageLeftAlign) ? 0 : w - h, 0, h, h);
                g.DrawImage(this._image, rect);
                //int lineX = (this._imageLeftAlign) ? h + 4 : w - h - 4;
                //e.Graphics.DrawLine(Pens.Gray, lineX, 2, lineX, h - 2);
            }
        }
        void PaintText(Graphics g)
        {
            if (this._showText)
            {
                int w = this.Width;
                int h = this.Height;
                int tX = (this._showImage && this._imageLeftAlign) ? h + 4 : 0;
                int tW = (this._showImage) ? w - h - 4 : w;
                RectangleF textRect = new RectangleF(tX, 2, tW, h - 4);
                StringFormat sf = new StringFormat();
                if (!this._textLeftAlign) { sf.FormatFlags |= StringFormatFlags.DirectionRightToLeft; }
                g.DrawString(this.Text, this.Font, new SolidBrush(this.ForeColor), textRect, sf);
            }
        }
        void PaintBorder(Graphics g)
        {
            Rectangle rect = new Rectangle(0, 0, this.Width, this.Height);
            //
            if      (this._mouseDown ) { ControlPaint.DrawBorder(g, rect, Control.DefaultBackColor, ButtonBorderStyle.Inset); }
            else if (this._mouseHover) { ControlPaint.DrawBorder(g, rect, this._hoverBorderColor  , this._hoverBorderStyle ); }
            else                       { ControlPaint.DrawBorder(g, rect, this._borderColor       , this._borderStyle      ); }
        }

        #endregion

        #region /* Mouse hover/down support methods */

        void ProcessMouseMove(int x, int y)
        {
            if (this.IsOutOfBounds(x, y))
            {   // check if mouse is out-of-bounds
                this.ProcessMouseLeave();
            }
            else if (!this._mouseHover)
            {
                this.BeginUpdate();
                this._mouseHover = true;
                this.BringToFront();
                this.EndUpdate();
            }
        }
        void ProcessMouseLeave()
        {
            if (this._mouseHover)
            {
                this.BeginUpdate();
                this._mouseHover = false;
                this._mouseDown = false;
                this.EndUpdate();
            }
        }
        void ProcessLMouseDown(int x, int y)
        {
            if (!this._mouseDown)
            {
                this.BeginUpdate();
                this._mouseDown = true;
                this.EndUpdate();
            }
        }
        void ProcessLMouseUp(int x, int y)
        {
            if (this._mouseDown)
            {
                this.BeginUpdate();
                this._mouseDown = false;
                this.ProcessMouseLeave();
                this.EndUpdate();
                this.OnMouseLeftClick(new MouseEventArgs(MouseButtons.Left, 1, x, y, 0));
            }
        }

        #endregion

        /// <summary>
        /// Raises the <see cref="Control.SizeChanged"/> event.
        /// </summary>
        /// <param name="e">An <see cref="EventArgs"/> that contains the event data.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            this.Invalidate();
            base.OnSizeChanged(e);
        }

        /// <summary>
        /// Processes windows messages.
        /// </summary>
        /// <param name="m">The windows message.</param>
        protected override void WndProc(ref Message m)
        {
            IntPtr lParam = m.LParam;
            // gets the low x and y coordinates of the mouse
            // (applicable to supported mouse events only)
            int x = Win32.LoWord(lParam.ToInt32());
            int y = Win32.HiWord(lParam.ToInt32());

            // intercept some messages
            switch ((Win32.Messages)m.Msg)
            {
                case Win32.Messages.WM_MOUSEMOVE:
                    this.ProcessMouseMove(x, y);
                    break;

                case Win32.Messages.WM_MOUSELEAVE:
                    this.ProcessMouseLeave();
                    break;

                case Win32.Messages.WM_LBUTTONDOWN:
                    this.ProcessLMouseDown(x, y);
                    break;

                case Win32.Messages.WM_LBUTTONUP:
                    this.ProcessLMouseUp(x, y);
                    break;
            }

            try
            {   // pass the message on to the base
                base.WndProc(ref m);
            }
            catch (Exception ex) { Console.WriteLine(ex.ToString()); }
        }
        /// <summary>
        /// Forces the control to invalidate its client area and immediately redraw itself and any child controls.
        /// </summary>
        public override void Refresh()
        {
            this.ProcessMouseLeave();
            base.Refresh();
        }

        void InitializeComponent()
        {
            this.Text = this.Name;
            this._showText = true;
            this._textLeftAlign = true;
            //
            this._image = null;
            this._showImage = true;
            this._imageLeftAlign = true;
            //
            this._borderStyle = ButtonBorderStyle.Solid;
            this._borderColor = SystemColors.ControlDark;
            //
            this._hoverBorderStyle = ButtonBorderStyle.Outset;
            this._hoverBorderColor = SystemColors.Control;
            this._hoverBackColor   = SystemColors.Control;
        }

        void BeginUpdate()
        {
            Win32.BeginUpdate(this, ref this._updateCount, ref this._eventMask);
        }
        void EndUpdate()
        {
            Win32.EndUpdate(this, ref this._updateCount, ref this._eventMask);
            base.Refresh();
        }

        /// <summary>
        /// Gets whether the specified position of the mouse cursor, relative to this control, is out of bounds.
        /// </summary>
        /// <param name="pt">The relative position of the mouse cursor.</param>
        /// <returns>true, if the position of the mouse cursor is out of bounds; false, otherwise.</returns>
        bool IsOutOfBounds(Point pt)
        {
            return this.IsOutOfBounds(pt.X, pt.Y);
        }
        /// <summary>
        /// Gets whether the specified position of the mouse cursor, relative to this control, is out of bounds.
        /// </summary>
        /// <param name="x">The relative X position of the mouse cursor.</param>
        /// <param name="y">The relative Y position of the mouse cursor.</param>
        /// <returns>true, if the position of the mouse cursor is out of bounds; false, otherwise.</returns>
        bool IsOutOfBounds(int x, int y)
        {
            int xl = 0, yl = 0, xr = this.Width, yr = this.Height;
//            if      (this.BorderStyle == BorderStyle.FixedSingle) { xl--; yl--; xr += 2; yr += 2; }
//            else if (this.BorderStyle == BorderStyle.Fixed3D    ) { xl--; yl--; }
            return (x < xl || y < yl || x > xr || y > yr);
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets or sets whether the text should be displayed on the control.</summary>
        [Category("Appearance")]
        [Description("Indicates whether the text should be displayed on the control.")]
        [DefaultValue(true)]
        public bool ShowText { get { return this._showText; } set { this._showText = value; base.Refresh(); } }
        /// <summary>Gets or sets whether the text should be left-aligned (true), or right-aligned (false).</summary>
        [Category("Appearance")]
        [Description("Indicates whether the text should be left-aligned (true), or right-aligned (false).")]
        [DefaultValue(true)]
        public bool TextLeftAlign { get { return this._textLeftAlign; } set { this._textLeftAlign = value; base.Refresh(); } }

        /// <summary>Gets or sets the image associated with the control.</summary>
        [Category("Appearance")]
        [Description("The image associated with the control.")]
        [DefaultValue(null)]
        public Image Image { get { return this._image; } set { this._image = value; base.Refresh(); } }
        /// <summary>Gets or sets whether an image should be displayed on this control.</summary>
        [Category("Appearance")]
        [Description("Indicates whether an image should be displayed on the control.")]
        [DefaultValue(true)]
        public bool ShowImage { get { return this._showImage; } set { this._showImage = value; base.Refresh(); } }
        /// <summary>Gets or sets whether an image should be left-aligned (true), or right-aligned (false).</summary>
        [Category("Appearance")]
        [Description("Indicates whether an image should be left-aligned (true), or right-aligned (false).")]
        [DefaultValue(true)]
        public bool ImageLeftAlign { get { return this._imageLeftAlign; } set { this._imageLeftAlign = value; base.Refresh(); } }

        /// <summary>Gets or sets the border style for the control.</summary>
        [Category("Appearance")]
        [Description("Determines whether the control has a border.")]
        [DefaultValue(typeof(ButtonBorderStyle), "Solid")]
        public ButtonBorderStyle BorderStyle { get { return this._borderStyle; } set { this._borderStyle = value; base.Refresh(); } }
        /// <summary>Gets or sets the border color of the control.</summary>
        [Category("Appearance")]
        [Description("If the control has a visible border, indicates the border color for the control.")]
        [DefaultValue(typeof(Color), "ControlDark")]
        public Color BorderColor { get { return this._borderColor; } set { this._borderColor = value; base.Refresh(); } }

        /// <summary>Gets or sets the border style for the control, when the mouse is hovering over it.</summary>
        [Category("Appearance")]
        [Description("The border style for the control, when the mouse is hovering over it.")]
        [DefaultValue(typeof(ButtonBorderStyle), "Outset")]
        public ButtonBorderStyle HoverBorderStyle { get { return this._hoverBorderStyle; } set { this._hoverBorderStyle = value; } }
        /// <summary>Gets or sets the border color of the control, when the mouse is hovering over it.</summary>
        [Category("Appearance")]
        [Description("The border color of the control, when the mouse is hovering over it.")]
        [DefaultValue(typeof(Color), "Control")]
        public Color HoverBorderColor { get { return this._hoverBorderColor; } set { this._hoverBorderColor = value; } }
        /// <summary>Gets or sets the background color of the control, when the mouse is hovering over it.</summary>
        [Category("Appearance")]
        [Description("The background color of the control, when the mouse is hovering over it.")]
        [DefaultValue(typeof(Color), "Control")]
        public Color HoverBackColor { get { return this._hoverBackColor; } set { this._hoverBackColor = value; } }

        #endregion


        #region /*** Variables ***/

        bool _showText;
        bool _textLeftAlign;

        Image _image;
        bool  _showImage;
        bool  _imageLeftAlign;

        ButtonBorderStyle _borderStyle;
        Color             _borderColor;

        ButtonBorderStyle _hoverBorderStyle;
        Color             _hoverBorderColor;
        Color             _hoverBackColor;

        bool _mouseDown = false;
        bool _mouseHover = false;

        int _updateCount = 0;
        int _eventMask = 0;

        #endregion

    }
}
