﻿#region << Using Directives >>
using System;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.IO.Compression;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using Volpe.Cafe.Data;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;
#endregion

namespace Volpe.Cafe.UI.Panels
{
    /// <summary>
    /// Provides a graphical user interface (GUI) for a single session of the CAFE Model.
    /// </summary>
    public class SessionPanel : UserControl
    {

        #region /*** Nested Types ***/

        sealed class SessionMessages
        {
            public const string CAFEModelError = "CAFE Model Error";
            public const string CAFEModel      = "CAFE Model";

            public const string NEW_STATUS_SessionCreated = "New session created.";

            public const string OPEN_FileNotFound = "Error opening session file.\n\nThe specified file cannot be found on disk.";
            public const string OPEN_SessionNotValid = "Error opening session file.\n\nThe specified file may not represent a valid session, or the session format is no longer supported.";
            public const string OPEN_STATUS_SelectSessionFile = "Please select the session file you would like to open ...";
            public const string OPEN_STATUS_LoadingSession = "Loading session ...";
            public const string OPEN_STATUS_SessionLoadingFailed = "Session loading failed.";
            public const string OPEN_STATUS_SessionNotLoaded = "Session not loaded.";
            public const string OPEN_STATUS_SessionLoaded = "Session loaded.";
            public const string OPEN_STATUS_SessionLoadedReadOnly = "Session loaded as read-only.";

            public const string SAVE_SessionRunning = "The session is currently running.\n\nPlease wait for modeling to complete, or stop the session before saving again.";
            public const string SAVE_STATUS_SavingSession = "Saving session ...";
            public const string SAVE_STATUS_SessionSaved = "Session saved.";

            public const string CLOSE_SessionRunning = "The session is currently running.\n\nPlease wait for modeling to complete, or stop the session before closing.";
            public const string CLOSE_SessionNotSaved = "The session has not been saved yet.\n\nWould you like to save the changes before closing?";
            public const string CLOSE_CloseSession = "A CAFE Model session is currently open and will be closed.\n\nAre you sure you would like to proceed?";
            public const string CLOSE_STATUS_SessionClosed = "Session closed.";

            public const string START_SessionRunning = "The session is already running.";
            public const string START_NoComplianceModel = "The session cannot be run since no valid compliance model has been selected.\n\nPlease open the Modeling Settings window to select a compliance model you wish to run.";
            public const string START_SessionReadOnly = "The session cannot be run because it was opened as read-only.";

            public const string STOP_SessionNotRunning = "The session is not currently running.";

            //public const string REPORTS_SessionReporting = "The reports for the session are already being generated.";
            //public const string REPORTS_ComplianceModelNotRun = "The reports for the session cannot be generated because the compliance model was not run or did not complete successfully.";
            //public const string REPORTS_Error = "Error generating reports.\n\nOne or more of the modeling reports could not be generated.  Please restart the application and try again.";

            //public const string REPORTS_STATUS_Starting = "Preparing to generate reports ...";
            //public const string REPORTS_STATUS_GeneratingReports = "Generating Reports ...";
            //public const string REPORTS_STATUS_Stopped = "Reporting Stopped.";
            //public const string REPORTS_STATUS_Completed = "Reporting Completed!";

            //public const string REPORTS_SessionNotReporting = "The reports for the session are not currently being generated.";
            //public const string REPORTS_STATUS_CancelReports = "Canceling Reports (this may take some time) ...";

            public const string COMPLIANCE_STATUS_StartRequested = "Starting the Modeling Process ...";
            public const string COMPLIANCE_STATUS_Running        = "Model Running ...";
            public const string COMPLIANCE_STATUS_StopRequested  = "Stopping the Modeling Process ...";
            public const string COMPLIANCE_STATUS_Stopped        = "Modeling Stopped.";
            public const string COMPLIANCE_STATUS_Completed      = "Modeling Completed!";
            public const string COMPLIANCE_STATUS_Unstarted      = "The Modeling Process has not Started.";

        }

        #endregion

        #region /*** Events ***/

        /// <summary>Occurs when the session state and/or progress is being updated.</summary>
        [Browsable(true)]
        [Category("Behavior")]
        public event EventHandler SessionUpdating;
        /// <summary>Occurs when the name of the session changes.</summary>
        [Browsable(true)]
        [Category("Behavior")]
        public event EventHandler SessionNameChanged;

        #endregion

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new empty CAFE Model session.
        /// </summary>
        public SessionPanel()
        {
            this.InitializeComponent();
        }
        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
        }

        #endregion

        #region /*** Methods ***/

        #region /* Windows Form Designer generated code */

        void InitializeComponent()
        {
            this.ux_splitter = new System.Windows.Forms.Splitter();
            this.ux_progress = new System.Windows.Forms.RichTextBox();
            this.ux_statusPanel = new System.Windows.Forms.Panel();
            this.ux_additionalInfo = new System.Windows.Forms.RichTextBox();
            this.ux_runtime = new System.Windows.Forms.Label();
            this.ux_progressPanel = new System.Windows.Forms.Panel();
            this.ux_statusPanel.SuspendLayout();
            this.ux_progressPanel.SuspendLayout();
            this.SuspendLayout();
            // 
            // ux_splitter
            // 
            this.ux_splitter.Dock = System.Windows.Forms.DockStyle.Bottom;
            this.ux_splitter.Location = new System.Drawing.Point(0, 211);
            this.ux_splitter.Name = "ux_splitter";
            this.ux_splitter.Size = new System.Drawing.Size(472, 2);
            this.ux_splitter.TabIndex = 5;
            this.ux_splitter.TabStop = false;
            // 
            // ux_progress
            // 
            this.ux_progress.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom)
                        | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_progress.BackColor = System.Drawing.SystemColors.Window;
            this.ux_progress.BorderStyle = System.Windows.Forms.BorderStyle.None;
            this.ux_progress.DetectUrls = false;
            this.ux_progress.Font = new System.Drawing.Font("Lucida Console", 12F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.ux_progress.Location = new System.Drawing.Point(8, 8);
            this.ux_progress.Name = "ux_progress";
            this.ux_progress.ReadOnly = true;
            this.ux_progress.Size = new System.Drawing.Size(456, 192);
            this.ux_progress.TabIndex = 1;
            this.ux_progress.Text = "Ready";
            this.ux_progress.WordWrap = false;
            // 
            // ux_statusPanel
            // 
            this.ux_statusPanel.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_statusPanel.Controls.Add(this.ux_additionalInfo);
            this.ux_statusPanel.Controls.Add(this.ux_runtime);
            this.ux_statusPanel.Dock = System.Windows.Forms.DockStyle.Bottom;
            this.ux_statusPanel.Location = new System.Drawing.Point(0, 213);
            this.ux_statusPanel.Name = "ux_statusPanel";
            this.ux_statusPanel.Size = new System.Drawing.Size(472, 80);
            this.ux_statusPanel.TabIndex = 2;
            // 
            // ux_additionalInfo
            // 
            this.ux_additionalInfo.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom)
                        | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_additionalInfo.BackColor = System.Drawing.SystemColors.Control;
            this.ux_additionalInfo.BorderStyle = System.Windows.Forms.BorderStyle.None;
            this.ux_additionalInfo.Location = new System.Drawing.Point(8, 32);
            this.ux_additionalInfo.Name = "ux_additionalInfo";
            this.ux_additionalInfo.ReadOnly = true;
            this.ux_additionalInfo.Size = new System.Drawing.Size(456, 40);
            this.ux_additionalInfo.TabIndex = 1;
            this.ux_additionalInfo.TabStop = false;
            this.ux_additionalInfo.Text = "";
            this.ux_additionalInfo.WordWrap = false;
            // 
            // ux_runtime
            // 
            this.ux_runtime.AutoSize = true;
            this.ux_runtime.Location = new System.Drawing.Point(8, 8);
            this.ux_runtime.Name = "ux_runtime";
            this.ux_runtime.Size = new System.Drawing.Size(58, 13);
            this.ux_runtime.TabIndex = 0;
            this.ux_runtime.Text = "Runtime: 0";
            // 
            // ux_progressPanel
            // 
            this.ux_progressPanel.BackColor = System.Drawing.Color.White;
            this.ux_progressPanel.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_progressPanel.Controls.Add(this.ux_progress);
            this.ux_progressPanel.Dock = System.Windows.Forms.DockStyle.Fill;
            this.ux_progressPanel.Location = new System.Drawing.Point(0, 0);
            this.ux_progressPanel.Name = "ux_progressPanel";
            this.ux_progressPanel.Size = new System.Drawing.Size(472, 211);
            this.ux_progressPanel.TabIndex = 6;
            // 
            // SessionPanel
            // 
            this.Controls.Add(this.ux_progressPanel);
            this.Controls.Add(this.ux_splitter);
            this.Controls.Add(this.ux_statusPanel);
            this.Name = "SessionPanel";
            this.Size = new System.Drawing.Size(472, 293);
            this.ux_statusPanel.ResumeLayout(false);
            this.ux_statusPanel.PerformLayout();
            this.ux_progressPanel.ResumeLayout(false);
            this.ResumeLayout(false);

        }

        #endregion

        #region /* Overriden Methods */

        /// <summary>
        /// Raises the <see cref="Control.SizeChanged"/> event.
        /// </summary>
        /// <param name="e">Event data to pass to the event.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            //float zoom = this.Width / 400.0F;
            //if (zoom < 0.75F) { zoom = 0.75F; } else if (zoom > 3.0F ) { zoom = 3.0F ; }
            //if (this.ux_progress.ZoomFactor != zoom) { this.ux_progress.ZoomFactor = zoom; }
            // call OnSizeChanged from base class
            base.OnSizeChanged(e);
        }

        #endregion

        #region /* Methods for raising events */

        /// <summary>
        /// Raises the <see cref="SessionUpdating"/> event.
        /// </summary>
        protected virtual void OnSessionUpdating()
        {
            if (this.SessionUpdating != null) { this.SessionUpdating(this, EventArgs.Empty); }
        }
        /// <summary>
        /// Raises the <see cref="SessionNameChanged"/> event.
        /// </summary>
        protected virtual void OnSessionNameChanged()
        {
            if (this.SessionNameChanged != null) { this.SessionNameChanged(this, EventArgs.Empty); }
        }

        #endregion

        /// <summary>
        /// Creates a new empty session.
        /// </summary>
        public void CreateNewSession()
        {
            this.CreateNewSession(true);
        }
        /// <summary>
        /// Creates a new empty session.
        /// </summary>
        /// <param name="makeVisible">true, to make the new session visible in the parent; false, otherwise.</param>
        public void CreateNewSession(bool makeVisible)
        {
            if (this._isOpen && !this.CloseSession(true)) { return; }

            this._compliance      = null;
            this._data            = null;
            this._settings        = new ModelingSettings();
            //
            this._index           = SessionCount++;
            this._path            = null;
            this._name            = null;
            this._status          = null;
            this._refreshInterval = RefreshIntervalNormal;
            this._isOpen          = true;
            this._readOnly        = false;
            this._saveRequired    = true;
            this._stopRequested   = false;

            // update gui ...
            this.UpdateSessionName();
            this.RefreshSession();
            if (makeVisible) { this.Visible = true; }
            this.UpdateStatus(SessionMessages.NEW_STATUS_SessionCreated);
        }

        /// <summary>
        /// Displays an <see cref="OpenFileDialog"/> to the user, and opens the session at the specified file on disk.
        /// </summary>
        /// <returns>true, if the session was opened; false, otherwise.</returns>
        public bool OpenSession()
        {
            return this.OpenSession(false);
        }
        /// <summary>
        /// Displays an <see cref="OpenFileDialog"/> to the user, and opens the session at the specified file on disk.
        /// </summary>
        /// <param name="silent">true, to silently accept all prompts; false, otherwise.</param>
        /// <returns>true, if the session was opened; false, otherwise.</returns>
        public bool OpenSession(bool silent)
        {
            return this.OpenSession_Internal(null, true, silent);
        }
        /// <summary>
        /// Opens the session at the specified path.
        /// </summary>
        /// <param name="path">The path of the session to load.</param>
        /// <returns>true, if the session was opened; false, otherwise.</returns>
        public bool OpenSession(string path)
        {
            return this.OpenSession(path, false);
        }
        /// <summary>
        /// Opens the session at the specified path.
        /// </summary>
        /// <param name="path">The path of the session to load.</param>
        /// <param name="silent">true, to silently accept all prompts; false, otherwise.</param>
        /// <returns>true, if the session was opened; false, otherwise.</returns>
        public bool OpenSession(string path, bool silent)
        {
            return this.OpenSession_Internal(path, false, silent);
        }
        //--------------------------------------------------------------------------------------//
        // Opens the session specified by _path.  If _path is not valid, an exception will be   //
        // thrown.                                                                              //
        //                                                                                      //
        // Params:                                                                              //
        //  path        -- the path to open, or null, if displayOpen is true                    //
        //  displayOpen -- true, to display the "Open" dialog                                   //
        //  silent      -- true, to silently accept all prompts; false, otherwise               //
        // -------------------------------------------------------------------------------------//
        bool OpenSession_Internal(string path, bool displayOpen, bool silent)
        {
            if (this._isOpen && !this.CloseSession(true)) { return false; }

            if (displayOpen)
            {   // prompt for session file to open
                this.UpdateStatus(SessionMessages.OPEN_STATUS_SelectSessionFile);
                OpenFileDialog dlg = new OpenFileDialog();
                dlg.Filter = "All Files (*.*)|*.*|CAFE Model Session Data (*.cmsd)|*.cmsd";
                dlg.FilterIndex = 2;
                dlg.Title = "Please select the session file you would like to open:";
                if (dlg.ShowDialog() == DialogResult.OK) { this._path = dlg.FileName; }
                else { this.UpdateStatus(SessionMessages.OPEN_STATUS_SessionNotLoaded); return false; }
            }
            else
            {   // use specified session file path
                this._path = path;
            }

            // perform some error-checking ...
            if (!File.Exists(this._path))
            {
                this.UpdateStatus(SessionMessages.OPEN_STATUS_SessionLoadingFailed);
                if (!silent) { this.ShowErrorPrompt(SessionMessages.OPEN_FileNotFound); }
                return false;
            }

            // -------------------------------------------------- \\
            // open the session file ...
            this.UpdateStatus(SessionMessages.OPEN_STATUS_LoadingSession);
            //
            BinaryFormatter bf = new BinaryFormatter();
            Stream dataStream = null;
            GZipStream gzip = null;
            try
            {   // decompress the session file
                dataStream = new MemoryStream();
                gzip = new GZipStream(new FileStream(this._path, FileMode.Open, FileAccess.Read, FileShare.Read), CompressionMode.Decompress);
                byte[] gzipBuf = new byte[4096];
                int gzipBufCount = 0;
                do
                {
                    dataStream.Write(gzipBuf, 0, gzipBufCount);
                    gzipBufCount = gzip.Read(gzipBuf, 0, gzipBuf.Length);
                } while (gzipBufCount > 0);

                // deserialize session data
                dataStream.Seek(0, SeekOrigin.Begin);
                object[] sessionData = (object[])bf.Deserialize(dataStream);

                // copy session data to local vars
                this._compliance      = (ICompliance     )sessionData[0];
                this._data            = (Industry        )sessionData[1];
                this._settings        = (ModelingSettings)sessionData[2];
                this._refreshInterval = (int             )sessionData[3];

                // update runtime varaibles ...
                // (if session was opened read-only, a save will be required prior to running the model)
                FileInfo fileInfo   = new FileInfo(this._path);
                //
                this._isOpen        = true;
                this._readOnly      = fileInfo.IsReadOnly;
                this._saveRequired  = this._readOnly;
                this._stopRequested = false;

                // if session was opened as read/write, mark session file as read-only to prevent other instances from taking control
                if (!this._readOnly) { fileInfo.IsReadOnly = true; }

                // update gui ...
                this.UpdateSessionName();
                this.RefreshSession();
                this.UpdateStatus((this._readOnly) ?
                    SessionMessages.OPEN_STATUS_SessionLoadedReadOnly :
                    SessionMessages.OPEN_STATUS_SessionLoaded);

                // add to recent items lists
                RecentItems.AddEntry(this._path);
                return true;
            }
            catch (Exception ex)
            {   // update gui
                this.UpdateStatus(SessionMessages.OPEN_STATUS_SessionLoadingFailed);
                // write exception to console and show error prompt
                Console.WriteLine(ex.ToString());
                if (!silent) { this.ShowErrorPrompt(SessionMessages.OPEN_SessionNotValid); }
                return false;
            }
            finally
            {   // close the any opened streams
                if (dataStream != null) { dataStream.Close(); }
                if (gzip       != null) { gzip      .Close(); }
            }
        }

        /// <summary>
        /// Closes the session by cleaning up runtime variables.  If the session is running or needs saving, a user-prompt will
        /// be generated.
        /// </summary>
        /// <returns>true, if the session was closed; false, otherwise.</returns>
        public bool CloseSession()
        {
            return this.CloseSession(false);
        }
        /// <summary>
        /// Closes the session by cleaning up runtime variables.  If the session is running or needs saving, a user-prompt will
        /// be generated.
        /// </summary>
        /// <param name="confirmClose">Whether to display a close confirmation dialog to the user before closing the session.</param>
        /// <returns>true, if the session was closed; false, otherwise.</returns>
        public bool CloseSession(bool confirmClose)
        {
            if (!this._isOpen) { return true; }

            if (this.Running)
            {   // the session is running
                MessageBox.Show(SessionMessages.CLOSE_SessionRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK, MessageBoxIcon.Information);
                return false;
            }

            if (this._saveRequired || confirmClose)
            {   // the session hasn't been saved or close-confirmation is required -- prompt user
                DialogResult dr;
                if (this._saveRequired)
                {   // session needs saving
                    dr = MessageBox.Show(SessionMessages.CLOSE_SessionNotSaved, SessionMessages.CAFEModel,
                        MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question, MessageBoxDefaultButton.Button3);
                }
                else
                {   // session needs close confirmation
                    dr = MessageBox.Show(SessionMessages.CLOSE_CloseSession, SessionMessages.CAFEModel,
                        MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2);
                }
                //
                if (dr == DialogResult.Cancel || (!this._saveRequired && dr == DialogResult.No))
                {   // user selected to cancel closing
                    return false;
                }
                else if (this._saveRequired && dr == DialogResult.Yes)
                {   // user selected to save the session before closing
                    if (!this.SaveSession()) { return false; }
                }
            }

            // continue with session closing
            //
            // clear out read-only setting on the session file, but only if session was opened as read/write
            if (!this._readOnly && File.Exists(this._path))
            {
                (new FileInfo(this._path)).IsReadOnly = false;
            }
            //
            // clear certain elements ...
            this._compliance      = null;
            this._data            = null;
            this._settings        = null;
            //
            this._index           = -1;
            this._path            = null;
            this._name            = null;
            this._status          = null;
            this._refreshInterval = -1;
            this._isOpen          = false;
            this._readOnly        = false;
            this._saveRequired    = false;
            this._stopRequested   = false;

            // force garbage collection to ensure resources held by this session are released
            GC.Collect();
            GC.WaitForPendingFinalizers();
            GC.Collect();

            // update gui ...
            this.UpdateSessionName();
            //this.RefreshSession();
            this.UpdateStatus(SessionMessages.CLOSE_STATUS_SessionClosed);

            // session closed
            return true;
        }

        /// <summary>
        /// Saves the session to disk.  If the session has not been previously saved, a <see cref="SaveFileDialog"/> will be
        /// displayed.
        /// </summary>
        /// <returns>true, if the session was saved; false, otherwise.</returns>
        public bool SaveSession()
        {
            return this.SaveSession_Internal(false, null);
        }
        /// <summary>
        /// Saves the session to disk, using the specified session name.  If the session has not been previously saved, a
        /// <see cref="SaveFileDialog"/> will be displayed.
        /// </summary>
        /// <param name="name">The name to assign to the session.</param>
        /// <returns>true, if the session was saved; false, otherwise.</returns>
        public bool SaveSession(string name)
        {
            if (name != null && name.Trim() != string.Empty && name != this.SessionName)
            {
                this._saveRequired = true;
            }
            return this.SaveSession_Internal(false, name);
        }
        /// <summary>
        /// Displays a <see cref="SaveFileDialog"/> to the user, and saves the session to the specified file on disk.
        /// </summary>
        /// <returns>true, if the session was saved; false, otherwise.</returns>
        public bool SaveSessionAs()
        {
            return this.SaveSession_Internal(true, null);
        }
        // -------------------------------------------------------------------------------------//
        // displaySaveAs -- true, to display the "Save As" dialog                               //
        // name          -- the name to assign to the session file                              //
        // -------------------------------------------------------------------------------------//
        bool SaveSession_Internal(bool displaySaveAs, string name)
        {
            if (!this._isOpen || (!displaySaveAs && !this._saveRequired)) { return false; }
            if (this.Running)
            {
                MessageBox.Show(SessionMessages.SAVE_SessionRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK, MessageBoxIcon.Information);
                return false;
            }

            // append path with the new session name
            bool silent = false;
            if (name != null)
            {
                name = name.Trim();
                if (File.Exists(this._path) && name != string.Empty)
                {
                    this._path = Path.GetDirectoryName(this._path) + "\\" + name + ".cmsd";
                    silent = true;
                }
            }

            // remove read-only flag if saving to existing session file
            if (File.Exists(this._path) && !this._readOnly)
            {
                (new FileInfo(this._path)).IsReadOnly = false;
            }

            // display the save as dialog, if required
            if (displaySaveAs || this._path == null || (!silent && (!File.Exists(this._path) ||
                (File.GetAttributes(this._path) & FileAttributes.ReadOnly) == FileAttributes.ReadOnly)))
            {
                SaveFileDialog dlg = new SaveFileDialog();
                dlg.FileName = File.Exists(this._path) ? this._path : (name != null) ? name + ".cmsd" : this.SessionName + ".cmsd";
                dlg.Filter = "CAFE Model Session Data (*.cmsd)|*.cmsd";
                dlg.Title = "Please specify the location where you would like to save the '" +
                    ((name != null) ? name : this.SessionName) + "' session:";
                if (dlg.ShowDialog() == DialogResult.OK)
                {   // update the session path -- save is performed below ...
                    this._path = dlg.FileName;
                }
                else
                {   // user canceled save -- mark the session file as read-only to prevent other instances from taking control
                    if (File.Exists(this._path) && !this._readOnly)
                    {
                        File.SetAttributes(this._path, FileAttributes.ReadOnly);
                    }
                    return false;
                }
            }

            // -------------------------------------------------- \\
            // save the session ...
            this.UpdateStatus(SessionMessages.SAVE_STATUS_SavingSession);
            //
            // copy local vars to the session data object
            object[] sessionData = new object[4];
            ICompliance complianceModel = null;
            if (this._compliance != null)
            {
                complianceModel              = this._compliance.CreateNew();
                complianceModel.UserNotes    = this._compliance.UserNotes;
                complianceModel.UserKeywords = this._compliance.UserKeywords;
            }
            sessionData[0] = complianceModel;
            sessionData[1] = this._data;
            sessionData[2] = this._settings;
            sessionData[3] = this._refreshInterval;

            // serialize the session to memory
            MemoryStream dataStream = new MemoryStream();
            BinaryFormatter bf = new BinaryFormatter();
            bf.Serialize(dataStream, sessionData);
            dataStream.Seek(0, SeekOrigin.Begin);
            // compress the serialized session data
            GZipStream gzip = new GZipStream(new FileStream(this._path, FileMode.Create, FileAccess.ReadWrite, FileShare.None), CompressionMode.Compress);
            byte[] gzipBuf = new byte[4096];
            int gzipBufCount = 0;
            do
            {
                gzip.Write(gzipBuf, 0, gzipBufCount);
                gzipBufCount = dataStream.Read(gzipBuf, 0, gzipBuf.Length);
            } while (gzipBufCount > 0);
            gzip.Close();
            dataStream.Close();
            // mark the saved session file as read-only to prevent other instances from taking control
            File.SetAttributes(this._path, FileAttributes.ReadOnly);

            // update gui ...
            this._saveRequired = false;
            this._readOnly     = false;
            //
            this.UpdateSessionName();
            this.RefreshSession();
            this.UpdateStatus(SessionMessages.SAVE_STATUS_SessionSaved);

            // add to recent items lists
            RecentItems.AddEntry(this._path);
            //
            return true;
        }

        /// <summary>
        /// Starts the compliance modeling process for the current session.
        /// </summary>
        public void StartModeling()
        {
            if (!this._isOpen) { return; }

            if (this.Running)
            {
                MessageBox.Show(SessionMessages.START_SessionRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK, MessageBoxIcon.Information);
                return;
            }
            if (this.ReadOnly)
            {
                MessageBox.Show(SessionMessages.START_SessionReadOnly, SessionMessages.CAFEModel, MessageBoxButtons.OK, MessageBoxIcon.Information);
                return;
            }

            // perform some error-checking ...
            if (this._compliance == null)
            {
                MessageBox.Show(SessionMessages.START_NoComplianceModel, SessionMessages.CAFEModelError, MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation);
                return;
            }

            // -------------------------------------------------- \\
            // start the modeling process ...
            this._stopRequested = false;

            // add compliance model event handlers
            this._compliance.ModelingChanged += new ModelingEventHandler(this.Compliance_ModelingChanged);
            this._compliance.Prompt          += new PromptEventHandler  (this.Compliance_Prompt);

            // create copies of data and settings for compliance
            Industry         complianceData     = (this._data     == null) ? null : this._data    .Clone();
            ModelingSettings complianceSettings = (this._settings == null) ? null : this._settings.Clone();
            // append output settings with the session name
            if (complianceSettings != null) { complianceSettings.OutputSettings.OutputPath += ("\\" + this.SessionName); }
            // start modeling
            this._compliance.Start(complianceData, complianceSettings);

            // refresh session and start the refresh thread
            this.RefreshSession();
            this.InitRefreshSessionThread();
        }
        /// <summary>
        /// Stops the compliance modeling process for the current session.
        /// </summary>
        public void StopModeling()
        {
            if (!this._isOpen) { return; }

            if (!this.Running)
            {
                MessageBox.Show(SessionMessages.STOP_SessionNotRunning, SessionMessages.CAFEModel, MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
                return;
            }

            // -------------------------------------------------- \\
            // stop the modeling process ...
            if (this._stopRequested)
            {
                this._compliance.Abort(false);
                this._stopRequested = false;
            }
            else
            {
                this._compliance.Abort(true);
                this._stopRequested = true;
                this.InitWaitToStopThread();
            }
        }

        // ----- event handlers from the compliance model -----
        void Compliance_ModelingChanged(object sender, ModelingEventArgs e)
        {
            // get modeling state
            ModelingState state = e.State;

            // update status
            string status = string.Empty;
            //
            if      (state == ModelingState.StartRequested) { status = SessionMessages.COMPLIANCE_STATUS_StartRequested; }
            else if (state == ModelingState.Running       ) { status = SessionMessages.COMPLIANCE_STATUS_Running;        }
            else if (state == ModelingState.StopRequested ) { status = SessionMessages.COMPLIANCE_STATUS_StopRequested;  }
            else if (state == ModelingState.Stopped       ) { status = SessionMessages.COMPLIANCE_STATUS_Stopped;        }
            else if (state == ModelingState.Completed     ) { status = SessionMessages.COMPLIANCE_STATUS_Completed;      }
            else if (state == ModelingState.Unstarted     ) { status = SessionMessages.COMPLIANCE_STATUS_Unstarted;      }

            if (state == ModelingState.Stopped || state == ModelingState.Completed)
            {   // remove event handlers if model stopped or completed
                this._compliance.ModelingChanged -= new ModelingEventHandler(this.Compliance_ModelingChanged);
                this._compliance.Prompt          -= new PromptEventHandler  (this.Compliance_Prompt);
            }

            this.RefreshSession();
            this.UpdateStatus(status);
        }
        void Compliance_Prompt(object sender, PromptEventArgs e)
        {
            MessageBoxButtons buttons =
                (e.PromptOption == PromptOption.OkCancel   ) ? MessageBoxButtons.OKCancel    :
                (e.PromptOption == PromptOption.YesNoCancel) ? MessageBoxButtons.YesNoCancel :
                (e.PromptOption == PromptOption.YesNo      ) ? MessageBoxButtons.YesNo       : MessageBoxButtons.OK;
            MessageBoxDefaultButton defaultButton =
                (e.PromptOption == PromptOption.YesNoCancel) ? MessageBoxDefaultButton.Button3 :
                (e.PromptOption == PromptOption.YesNo ||
                 e.PromptOption == PromptOption.OkCancel   ) ? MessageBoxDefaultButton.Button2 : MessageBoxDefaultButton.Button1;

            DialogResult dr =  MessageBox.Show(e.Message, e.Title, buttons, MessageBoxIcon.Information, defaultButton);
            e.PromptResult =
                (dr == DialogResult.OK    ) ? PromptResult.Ok     :
                (dr == DialogResult.Cancel) ? PromptResult.Cancel :
                (dr == DialogResult.Yes   ) ? PromptResult.Yes    :
                (dr == DialogResult.No    ) ? PromptResult.No     : PromptResult.None;
        }

        // ----- threading methods for UI support -----
        void InitRefreshSessionThread()
        {
            Thread refreshThread = new Thread(new ThreadStart(this.RefreshSessionThread));
            refreshThread.Name = "Session-" + this.SessionName + "::SessionRefreshThread";
            refreshThread.Start();
        }
        void RefreshSessionThread()
        {
            while (this.Running)
            {
                for (int i = 0; i < this._refreshInterval / SessionPanel.MinRefreshInterval; i++)
                {
                    Thread.Sleep(SessionPanel.MinRefreshInterval);
                    if (!this.Running) { break; }
                }
                // if refresh interval is -1 (paused), sleep for half a second, and loop again
                if (this._refreshInterval == -1) { Thread.Sleep(500); } else { this.RefreshSession(); }
            }
            //
            // final refresh upon completion
            this.RefreshSession();
        }
        void InitWaitToStopThread()
        {
            Thread waitToStopThread = new Thread(new ThreadStart(this.WaitToStopThread));
            waitToStopThread.Name = "Session-" + this.SessionName + "::WaitToStopThread";
            waitToStopThread.Start();
        }
        void WaitToStopThread()
        {
            // wait for 5 seconds before canceling stop request
            for (int i = 0; i < 10; i++)
            {
                Thread.Sleep(500);
                if (!this._stopRequested || !this.Running) { return; }
            }
            this._stopRequested = false;
        }

        // ----- support methods -----
        /// <summary>
        /// Refreshes the session progress and status information.
        /// </summary>
        public void RefreshSession()
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new EmptyEventHandler(this.RefreshSession));
                return;
            }
            //
            if (this.Running)
            {   // obtain status from the compliance model
                IModelingProgress progress = this._compliance.Progress;
                // set progress ...
                StringBuilder sb = new StringBuilder();
                if (this._refreshInterval == -1)
                {
                    sb.Append("Automatic refreshing is currently paused.\n(Press F5 to refresh manually.)\n\n");
                }
                if (progress == null || (progress.ProgressString == null && progress.Scenario == null))
                {
                    if (this._refreshInterval != -1) { sb.Append("Refreshing compliance model progress ..."); }
                }
                else
                {
                    if (this._stopRequested) { sb.Append("Waiting to stop ...\n(Press stop again to terminate immediately.)\n\n"); }
                    //
                    if (progress.ProgressString != null)
                    {
                        sb.Append(progress.ProgressString);
                    }
                    else
                    {
                        sb.Append("Scenario: "); sb.Append(Interaction.GetTitleCase(progress.Scenario.ToString())); sb.Append("\n");
                        //
                        if (progress.ModelYear == null) { sb.Append("Model Year: N/A\n"); }
                        else { sb.Append("Model Year: "); sb.Append(progress.ModelYear); sb.Append("\n"); }
                        //
                        Manufacturer mfr = progress.Manufacturer;
                        if (mfr == null)
                        {
                            sb.Append("Manufacturer: N/A\n");
                        }
                        else
                        {
                            Manufacturer.CModelData mmd = progress.Manufacturer.ModelData;
                            //
                            sb.Append("Manufacturer: "); sb.Append(Interaction.GetTitleCase(progress.Manufacturer.Description.Name, 4));
                            sb.Append("\n");
                            sb.Append("         pSTND  STND  CAFE  Credits  Fines  Costs\n");
                            sb.Append("    PC :  "); this.RefreshSession_Helper_UpdateRC(sb, mmd, RegulatoryClass.PassengerCar );
                            sb.Append("    LT :  "); this.RefreshSession_Helper_UpdateRC(sb, mmd, RegulatoryClass.LightTruck   );
                            sb.Append("    2b3:  "); this.RefreshSession_Helper_UpdateRC(sb, mmd, RegulatoryClass.LightTruck2b3);
                        }
                    }
                }
                // update the progress text-box
                string progressString = sb.ToString();
                if (this.ux_progress.Text != progressString)
                {
                    this.ux_progress.Text = progressString;
                }
                // set additional data (if any)
                string additionalInfoString = (progress == null || progress.AdditionalInfo == null) ? string.Empty : progress.AdditionalInfo.ToString();
                if (this.ux_additionalInfo.Text != additionalInfoString)
                {
                    this.ux_additionalInfo.Text = additionalInfoString;
                }
                // update runtime
                this.ux_runtime.Text = Interaction.GetTimeString(this._compliance.Runtime);
            }
            else
            {
                this.ux_additionalInfo.Clear();
            }
        }
        void RefreshSession_Helper_UpdateRC(StringBuilder sb, Manufacturer.CModelData mmd, RegulatoryClass regClass)
        {
            if (double.IsNaN(mmd.CAFE[regClass]))
            {
                sb.Append(" -     -     -        -      -      -\n");
            }
            else
            {
                double d;
                //
                d = mmd.PreliminaryStandard[regClass];
                if (d < 10 || (d >= 100 && d < 1000)) { sb.Append(" "); }
                sb.Append((d < 100) ? d.ToString("0.0") : d.ToString("0")); sb.Append("  ");
                //
                d = mmd.Standard[regClass];
                if (d < 10 || (d >= 100 && d < 1000)) { sb.Append(" "); }
                sb.Append((d < 100) ? d.ToString("0.0") : d.ToString("0")); sb.Append("  ");
                //
                d = mmd.CAFE[regClass];
                if (d < 10 || (d >= 100 && d < 1000)) { sb.Append(" "); }
                sb.Append((d < 100) ? d.ToString("0.0") : d.ToString("0")); sb.Append("   ");
                //
                this.RefreshSession_Helper_FormatValue(sb, mmd.Credits [regClass]);
                this.RefreshSession_Helper_FormatValue(sb, mmd.Fines   [regClass]);
                this.RefreshSession_Helper_FormatValue(sb, mmd.TechCost[regClass]);
                sb.Append("\n");
            }
        }
        void RefreshSession_Helper_FormatValue(StringBuilder sb, double d)
        {
            double abs_d = Math.Abs(d);
            string unit = "  ";
            //
            if      (abs_d > 1000000000) { d /= 1000000000; unit = " b"; }
            else if (abs_d >    1000000) { d /=    1000000; unit = " m"; }
            else if (abs_d >       1000) { d /=       1000; unit = " k"; }
            //
            d = Math.Round(d);
            if      (d < -100) {                   }
            else if (d <  -10) { sb.Append(  " "); }
            else if (d <    0) { sb.Append( "  "); }
            else if (d <   10) { sb.Append("   "); }
            else if (d <  100) { sb.Append( "  "); }
            else               { sb.Append(  " "); }
            //
            sb.Append(d.ToString("0"));
            sb.Append(unit);
            sb.Append(' ');
        }
        /// <summary>
        /// Updates the status bar in the parent form.
        /// </summary>
        void UpdateStatus(string value)
        {
            this._status = value;
            // raise session updating event
            this.OnSessionUpdating();
        }

        void UpdateSessionName()
        {
            if (this._isOpen)
            {
                this._name = (this._path == null) ? this.GetGenericSessionName() : Path.GetFileNameWithoutExtension(this._path);
                this.Text = this._name + ((this._readOnly) ? " [Read Only]" : string.Empty);
            }
            else { this.Text = this._name = string.Empty; }
            //
            this.OnSessionNameChanged();
        }
        string GetGenericSessionName()
        {
            return "Session " + this._index.ToString();
        }

        void ShowErrorPrompt(string message)
        {
            MessageBox.Show(message, SessionMessages.CAFEModelError, MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the name of the session.</summary>
        [Browsable(false)]
        public string SessionName { get { return this._name; } }
        /// <summary>Gets the location of the session file, where the session data was loaded from or can be saved to.</summary>
        [Browsable(false)]
        public string SessionPath { get { return this._path; } }

        /// <summary>Gets or sets the compliance model that will be used during the modeling process.</summary>
        /// <remarks>Selecting a different compliance model will force the current session to restart.</remarks>
        [Browsable(false)]
        public ICompliance Compliance { get { return this._compliance; } set { this._compliance = value; this.SaveRequired = true; } }
        /// <summary>Gets or sets the modeling data associated with the current session.</summary>
        [Browsable(false)]
        public Industry Data { get { return this._data; } set { this._data = value; this.SaveRequired = true; } }
        /// <summary>Gets or sets the modeling settings associated with the current session.</summary>
        [Browsable(false)]
        public ModelingSettings Settings { get { return this._settings; } /* set { this._settings = value; this.SaveRequired = true; } */ }

        /// <summary>Gets whether the session is currently executing the compliance model simulation.</summary>
        [Browsable(false)]
        public bool Running { get { return (this._compliance != null && this._compliance.Running); } }

        /// <summary>Gets the status of the session, which should be displayed in the parent form's status bar.</summary>
        [Browsable(false)]
        public string Status { get { return this._status; } }
        /// <summary>Gets or set how frequently, in milliseconds, the session refreshes itself based on the latest progress
        ///   information of the compliance model.  Use -1 to pause refreshing; otherwise, the refresh interval will be converted
        ///   to be a minimum <see cref="MinRefreshInterval"/> or a multiple of <see cref="MinRefreshInterval"/>.</summary>
        [Browsable(false)]
        public int RefreshInterval
        {
            get { return this._refreshInterval; }
            set
            {
                if (value == -1) { this._refreshInterval = -1; }    // -1 indicates pause refreshing
                else
                {
                    this._refreshInterval = (value < MinRefreshInterval) ? MinRefreshInterval :
                        value - (value % MinRefreshInterval);       // make multiple of MinRefreshInterval
                }
                this.SaveRequired = true;
            }
        }

        /// <summary>Gets whether the session is open.</summary>
        [Browsable(false)]
        public bool IsOpen { get { return this._isOpen; } }
        /// <summary>Gets whether the session has been closed.</summary>
        [Browsable(false)]
        public bool IsClosed { get { return !this._isOpen; } }
        /// <summary>Gets whether the session was open as read-only.</summary>
        [Browsable(false)]
        public bool ReadOnly { get { return this._readOnly; } }
        /// <summary>Gets or sets whether the session needs to be saved.</summary>
        [Browsable(false)]
        public bool SaveRequired { get { return this._saveRequired; } set { this._saveRequired = value; this.RefreshSession(); } }

        #endregion

        #region /*** Variables ***/

        #region /* UI Variables */

        Panel       ux_progressPanel;
        RichTextBox ux_progress;
        Splitter    ux_splitter;
        Panel       ux_statusPanel;
        Label       ux_runtime;
        RichTextBox ux_additionalInfo;

        #endregion

        // ----- constants -----
        /// <summary>Represents a realtime refresh interval of 125 ms, or 8 times per second.</summary>
        public const int RefreshIntervalRealtime = 125;
        /// <summary>Represents a high refresh interval of 500 ms, or twice per second.</summary>
        public const int RefreshIntervalHigh = 500;
        /// <summary>Represents a normal refresh interval of 2000 ms, or once every two seconds.</summary>
        public const int RefreshIntervalNormal = 2000;
        /// <summary>Represents a low refresh interval of 4000 ms, or once every four seconds.</summary>
        public const int RefreshIntervalLow = 4000;
        /// <summary>Specifies that refreshing is paused.</summary>
        public const int RefreshIntervalPaused = -1;

        /// <summary>Represents the smallest allowed refresh interval, in milliseconds.</summary>
        public const int MinRefreshInterval = 125;

        static int SessionCount = 1;

        int    _index;
        string _path;
        string _name;   // the name of the session

        // modeling & reporting variables
        ICompliance      _compliance;
        Industry         _data;
        ModelingSettings _settings;

        // runtime variables
        string _status;
        int    _refreshInterval;    // a value of -1 indicates that session refreshing is paused
        bool   _isOpen;
        bool   _readOnly;           // indicates that the session was open as read-only
        bool   _saveRequired;
        bool   _stopRequested;

        #endregion

    }
}
