﻿#region << Using Directives >>
using System;
using Volpe.Cafe.Generic;
using Volpe.Cafe.Utils;
#endregion

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents an object used for storing a single effects value.
    /// </summary>
    [Serializable]
    public class EffectsValue : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        EffectsValue(EffectsValue prototype)
        {
            this.Value1 = prototype.Value1.Clone();
            this.Value2 = prototype.Value2.Clone();
            this.AnnualValue1 = new VCFuelValue[Interaction.CY];
            for (int i = 0; i < prototype.AnnualValue1.Length; i++)
            {
                this.AnnualValue1[i] = prototype.AnnualValue1[i].Clone();
            }
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="EffectsValue"/> class.
        /// </summary>
        public EffectsValue()
        {
            this.Value1 = new VCFuelValue();
            this.Value2 = new RCFuelValue();
            this.AnnualValue1 = new VCFuelValue[Interaction.CY];
            for (int i = 0; i < Interaction.CY; i++)
            {
                this.AnnualValue1[i] = new VCFuelValue();
            }
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EffectsValue"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EffectsValue"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EffectsValue"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EffectsValue"/>.</returns>
        public EffectsValue Clone()
        {
            return new EffectsValue(this);
        }

        #endregion

        /// <summary>
        /// Clears all values in this <see cref="EffectsValue"/> instance.
        /// </summary>
        public void Clear()
        {
            this.Value1.Clear();
            this.Value2.Clear();
            for (int i = 0; i < this.AnnualValue1.Length; i++)
            {
                this.AnnualValue1[i].Clear();
                //this.AnnualValue2[i].Clear();
            }
        }

        /// <summary>
        /// Appends the specified value to the effects values represented by the specified vehicle class, fuel type, and vehicle
        /// age. The new value will be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        /// <param name="vehClass">The vehicle class at which to append the value.</param>
        /// <param name="fuelType">The fuel type at which to append the value.</param>
        /// <param name="age">The vehicle age at which to append the value.</param>
        public void AppendValue(double value, VehicleClass vehClass, FuelType fuelType, int age)
        {
            this.Value1           [vehClass, fuelType] += value;
            this.AnnualValue1[age][vehClass, fuelType] += value;
        }
        /// <summary>
        /// Appends the specified value to the effects values represented by the specified vehicle class, regulatory class, fuel
        /// type, and vehicle age. The new value will be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        /// <param name="vehClass">The vehicle class at which to append the value.</param>
        /// <param name="regClass">The regulatory class at which to append the value.</param>
        /// <param name="fuelType">The fuel type at which to append the value.</param>
        /// <param name="age">The vehicle age at which to append the value.</param>
        public void AppendValue(double value, VehicleClass vehClass, RegulatoryClass regClass, FuelType fuelType, int age)
        {
            this.Value1           [vehClass, fuelType] += value;
            this.Value2           [regClass, fuelType] += value;
            this.AnnualValue1[age][vehClass, fuelType] += value;
            //this.AnnualValue2[age][regClass, fuelType] += value;
        }
        /// <summary>
        /// Appends all effects values in the specified <see cref="EffectsValue"/> instance to this instance. The new value will
        /// be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        public void AppendValue(EffectsValue value)
        {
            this.AppendValues(value);
        }
        /// <summary>
        /// Appends all effects values in the specified <see cref="EffectsValue"/> array to this instance. The new values will be
        /// added to the existing one.
        /// </summary>
        /// <param name="values">One or more values to append. This value may not be null.</param>
        public void AppendValues(params EffectsValue[] values)
        {
            for (int i = 0; i < values.Length; i++)
            {
                this.Value1 += values[i].Value1;
                this.Value2 += values[i].Value2;
                for (int j = 0; j < this.AnnualValue1.Length; j++)
                {
                    this.AnnualValue1[j] += values[i].AnnualValue1[j];
                    //values[i].AnnualValue2[j] += this.AnnualValue2[j];
                }
            }
        }

        /// <summary>
        /// Gets the effects value, for each fuel type, for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class). Only one of vehicle class or regulatory class must be used for disaggregation, with the other
        /// being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class for which to get the value, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class for which to get the value, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <returns>The effects value for the specified vehicle/regulatory class.</returns>
        public FuelValue GetValue(VehicleClass vehClass, RegulatoryClass regClass)
        {
            if (vehClass != VehicleClass   .None) { return this.Value1.GetValue(vehClass); }
            if (regClass != RegulatoryClass.None) { return this.Value2.GetValue(regClass); }
            return FuelValue.Zero;
        }
        /// <summary>
        /// Gets the effects value for either the specified vehicle class (if wishing to obtain the value aggregated by vehicle
        /// class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as well as
        /// the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with the
        /// other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class for which to get the value, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class for which to get the value, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type for which to get the value.</param>
        /// <returns>The effects value for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetValue(VehicleClass vehClass, RegulatoryClass regClass, FuelType fuelType)
        {
            return this.GetValue(vehClass, regClass)[fuelType];
        }
        /// <summary>
        /// Gets the effects value, for each fuel type, for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified vehicle age. Only one of vehicle class or regulatory class must be used
        /// for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="age">The vehicle age for which to get the value.</param>
        /// <param name="vehClass">The vehicle class for which to get the value, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">Not supported. Reserved in case of future need.</param>
        /// <returns>The effects value for the specified vehicle age and vehicle/regulatory class.</returns>
        public FuelValue GetAnnualValue(int age, VehicleClass vehClass, RegulatoryClass regClass)
        {
            if (vehClass != VehicleClass.None) { return this.AnnualValue1[age].GetValue(vehClass); }
            //if (regClass != RegulatoryClass.None) { return this.AnnualValue2[age].GetValue(regClass); }
            return FuelValue.Zero;
        }
        /// <summary>
        /// Gets the effects value for either the specified vehicle class (if wishing to obtain the value aggregated by vehicle
        /// class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as well as
        /// the specified vehicle age and fuel type. Only one of vehicle class or regulatory class must be used for disaggregation,
        /// with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="age">The vehicle age for which to get the value.</param>
        /// <param name="vehClass">The vehicle class for which to get the value, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">Not supported. Reserved in case of future need.</param>
        /// <param name="fuelType">The fuel type for which to get the value.</param>
        /// <returns>The effects value for the specified vehicle age, vehicle/regulatory class, and fuel type.</returns>
        public double GetAnnualValue(int age, VehicleClass vehClass, RegulatoryClass regClass, FuelType fuelType)
        {
            return this.GetAnnualValue(age, vehClass, regClass)[fuelType];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the effects value, aggregated by vehicle class and fuel type.</summary>
        public VCFuelValue Value1 { get; private set; }
        /// <summary>Gets the effects value, aggregated by regulatory class and fuel type.</summary>
        public RCFuelValue Value2 { get; private set; }
        /// <summary>Gets the annual effects value, aggregated by vehicle class and fuel type.</summary>
        public VCFuelValue[] AnnualValue1 { get; private set; }
        ///// <summary>Gets the annual effects value, aggregated by regulatory class and fuel type.</summary>
        //public RCFuelValue[] AnnualValue2 { get; private set; }

        #endregion

    }
}
