using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using Volpe.Cafe.IO;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Generic;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents the overall industry, consisting of all manufacturers, vehicles, engines, and transmissions.
    /// </summary>
    /// <seealso cref="Manufacturer"/>
    /// <seealso cref="Vehicle"/>
    /// <seealso cref="Engine"/>
    /// <seealso cref="Transmission"/>
    /// <seealso cref="Platform"/>
    [Serializable]
    public sealed class Industry : ICloneable
    {

        #region /*** Nested Types ***/

        /// <summary>
        /// Provides compliance model data for the entire <see cref="Industry"/>, which is updated during runtime.
        /// </summary>
        [Serializable]
        public sealed class CModelData
        {

            #region /*** Constructors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Industry.CModelData"/> class.
            /// </summary>
            internal CModelData() { }

            #endregion


            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Industry.CModelData"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Industry.CModelData"/>.</returns>
            internal Industry.CModelData Clone()
            {
                Industry.CModelData imd = new Industry.CModelData();
                //
                if (this.TechUsedSales != null)
                {
                    imd.TechUsedSales = new RCDouble[this.TechUsedSales.Length];
                    for (int i = 0; i < this.TechUsedSales.Length; i++)
                    {
                        imd.TechUsedSales[i] = this.TechUsedSales[i].Clone();
                    }
                }
                //
                imd.MinFP = this.MinFP;
                imd.MaxFP = this.MaxFP;
                imd.MinCW = this.MinCW;
                imd.MaxCW = this.MaxCW;
                //
                return imd;
            }

            /// <summary>
            /// Updates min and max FP/CW counters for the entire industry.
            /// </summary>
            internal void UpdateFPCW(List<Manufacturer> mfrs)
            {
                // reset ...
                double minFP = 99999; double minCW = 99999;
                double maxFP = -1;    double maxCW = -1;
                // and calculate ...
                for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
                {   // get mfr and scan each vehicle
                    Manufacturer mfr = mfrs[i];
                    //
                    for (int j = 0, vehCount = mfr.VehicleCount; j < vehCount; j++)
                    {
                        Vehicle.CDescription vd = mfr.Vehicles[j].Description;
                        double fp = vd.Footprint, cw = vd.CurbWeight;
                        if (fp < minFP) { minFP = fp; } else if (fp > maxFP) { maxFP = fp; }
                        if (cw < minCW) { minCW = cw; } else if (cw > maxCW) { maxCW = cw; }
                    } // next j (vehicle)
                } // next i (mfr)
                //
                // update class-level vars
                this.MinFP = minFP; this.MaxFP = maxFP;
                this.MinCW = minCW; this.MaxCW = maxCW;
            }

            #endregion


            #region /*** Variables ***/

            /// <summary>Specifies the overall sales volumes of technologies that are currently in use by all vehicles in the entire
            ///   industry fleet.  The volumes are represented for each technology, per regulatory class.</summary>
            /// <remarks>The technology used sales volumes should be reinitialized and calculated each time before starting a new
            ///   model year, based on the current "tech-used" states of the entire industry fleet.  The industry volumes <b>should
            ///   not</b> be updated during the compliance modeling process whenever a new technology is applied to a
            ///   <see cref="Vehicle"/>.</remarks>
            public RCDouble[] TechUsedSales;

            // TODO:  The min/max FP and CW calc needs to change to be by reg-class; however, for this to happen, the "UpdateFPCW" calc needs to happen during modeling, instead of input loading

            /// <summary>Specifies the smallest footprint value (in sq. ft.) among all vehicles produced in the entire industry.</summary>
            public double MinFP;
            /// <summary>Specifies the largest footprint value (in sq. ft.) among all vehicles produced in the entire industry.</summary>
            public double MaxFP;

            /// <summary>Specifies the smallest curb weight value (in pounds) among all vehicles produced in the entire industry.</summary>
            public double MinCW;
            /// <summary>Specifies the largest curb weight value (in pounds) among all vehicles produced in the entire industry.</summary>
            public double MaxCW;

            #endregion

        }

        #endregion


        #region /*** Constructors ***/

        // Private constructor used for internal usage (such as cloning).
        Industry() { }
        // Internal constructor used for loading input files.
        internal Industry(List<Manufacturer> mfrs, int minYear, int maxYear)
        {
            this._modelData     = new CModelData();
            this._manufacturers = mfrs;
            //
            this._minYear = new ModelYear(minYear);
            this._maxYear = new ModelYear(maxYear);
        }

        #endregion


        #region /*** Methods ***/

        #region /* ICloneable Members  */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Industry"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Industry"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Industry"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Industry"/>.</returns>
        public Industry Clone()
        {
            Industry ind = new Industry();
            //
            ind._modelData     = this._modelData.Clone();
            ind._manufacturers = this.CloneManufacturers(this._manufacturers);
            //
            ind._minYear       = new ModelYear(this._minYear.Year);
            ind._maxYear       = new ModelYear(this._maxYear.Year);
            //
            return ind;
        }
        List<Manufacturer> CloneManufacturers(List<Manufacturer> list)
        {
            // initialize a new list instance
            List<Manufacturer> newList = new List<Manufacturer>(list.Count);
            // traverse the current list and clone each element
            for (int i = 0, count = list.Count; i < count; i++)
            {
                newList.Add((Manufacturer)(list[i].Clone()));
            }
            // return the new list
            return newList;
        }

        #endregion

        /// <summary>
        /// Summarizes this <see cref="Industry"/> instance to the specified <see cref="LogWriter"/>.
        /// </summary>
        /// <param name="writer">The <see cref="LogWriter"/> where to write the summary logs.</param>
        public void WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            writer.Summary.WriteLine("---------- Modeling Data ----------");
            writer.Summary.WriteLine("--- Model Years Analyzed ---\n  "
                + this._minYear.ToString() + " to " + this._maxYear.ToString() + "\n");
            writer.Summary.WriteLine("--- Valid Manufacturers ---");
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                Manufacturer mfr = this._manufacturers[i];
                string mfrSummary = "  " + mfr.Index + ". Code=" + mfr.Description.Code + ", Name=" + mfr.Description.Name;
                mfrSummary += "\n      Vehicles=" + mfr.VehicleCount;
                for (int j = 0; j < VCValue<object>.Classes.Length; j++)
                {
                    mfrSummary += ((j == 0) ? " (" : ", ") + VCValue<object>.Names[j] + "=" +
                        this.Log_GetVehCountByClass(mfr.Vehicles, VCValue<object>.Classes[j]);
                }
                mfrSummary += ")";
                mfrSummary += "\n      Engines=" + mfr.Engines.Count;
                mfrSummary += "\n      Transmissions=" + mfr.Transmissions.Count;
                writer.Summary.WriteLine(mfrSummary);
            }
            writer.Summary.WriteLine();
            writer.Summary.WriteLine("--- Industry Summary ---");
            int engs = 0, trns = 0, vehs = 0;
            VCValue<int> vehsByClass = new VCValue<int>();
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                Manufacturer mfr = this._manufacturers[i];
                engs += mfr.Engines.Count;
                trns += mfr.Transmissions.Count;
                vehs += mfr.VehicleCount;
                foreach(VehicleClass vehClass in VCValue<object>.Classes)
                {
                    vehsByClass[vehClass] += this.Log_GetVehCountByClass(mfr.Vehicles, vehClass);
                }
            }
            writer.Summary.WriteLine("  Total Manufacturers Analyzed:  " + this._manufacturers.Count);
            writer.Summary.WriteLine("  Total Vehicles Analyzed:  " + vehs);
            for (int i = 0; i < VCValue<object>.Classes.Length; i++)
            {
                string vcName = VCValue<object>.Names[i];
                vcName = "    " + vcName + ":" + new string(' ', vcName.Length - 2);
                writer.Summary.WriteLine(vcName + vehsByClass.Items[i]);
            }
            writer.Summary.WriteLine("  Total Engines Analyzed:   " + engs);
            writer.Summary.WriteLine("  Total Transmissions Analyzed:  " + trns);
            writer.Summary.WriteLine();
        }
        int Log_GetVehCountByClass(List<Vehicle> vehs, VehicleClass vehClass)
        {
            int count = 0;
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {
                if (vehs[i].VehicleClass == vehClass) { count++; }
            }
            return count;
        }

        /// <summary>
        /// Updates min and max FP/CW properties based on the current state of all <see cref="Vehicle"/>s (for all model years)
        /// in the <see cref="Industry"/>.
        /// </summary>
        public void UpdateMinMaxFpCw()
        {
            this._modelData.UpdateFPCW(this._manufacturers);
        }

        /// <summary>
        /// Calculates and returns total employment hours for all vehicles produced within the <see cref="Industry"/>, specified
        /// as thousand labor-hours, for the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to compute employment hours.</param>
        /// <param name="regClass">The regulatory class based on which to determine the compliance modeling data to return.
        ///   Alternatively, specify -1 to return the sum of the compliance modeling data of all regulatory classes, not including
        ///   the unregulated class, or -2 to return the sum of the compliance modeling data of all regulatory classes, including
        ///   the unregulated class.</param>
        /// <returns>The total employment hours for all vehicles produced within the <see cref="Industry"/>, specified as thousand
        ///   labor-hours.</returns>
        public double CalcLaborHours(ModelYear year, RegulatoryClass regClass)
        {
            double hours = 0;
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                hours += this._manufacturers[i].CalcLaborHours(year, regClass);
            }
            return hours;
        }
        /// <summary>
        /// Calculates and returns total employment hours for all vehicles produced within the <see cref="Industry"/>, specified
        /// as thousand labor-hours, for the specified range of model years.
        /// </summary>
        /// <param name="minYear">The lower bound model year for which to compute employment hours.</param>
        /// <param name="maxYear">The upper bound model year for which to compute employment hours.</param>
        /// <param name="regClass">The regulatory class based on which to determine the compliance modeling data to return.
        ///   Alternatively, specify -1 to return the sum of the compliance modeling data of all regulatory classes, not including
        ///   the unregulated class, or -2 to return the sum of the compliance modeling data of all regulatory classes, including
        ///   the unregulated class.</param>
        /// <returns>The total employment hours for all vehicles produced within the <see cref="Industry"/>, specified as thousand
        ///   labor-hours.</returns>
        public double CalcLaborHours(ModelYear minYear, ModelYear maxYear, RegulatoryClass regClass)
        {
            double hours = 0;
            for (int i = 0, mfrCount = this._manufacturers.Count; i < mfrCount; i++)
            {
                hours += this._manufacturers[i].CalcLaborHours(minYear, maxYear, regClass);
            }
            return hours;
        }

        bool HasValues(bool[] value)
        {
            // return true if ANY elements in value are TRUE
            if (value == null) { return false; }
            for (int i = 0; i < value.Length; i++)
            {
                if (value[i]) { return true; }
            }
            return false;
        }

        #endregion


        #region /*** Properties ***/

        // ----- industry characteristics -----
        /// <summary>Gets compliance model data for the <see cref="Industry"/>, which is updated during runtime.</summary>
        public Industry.CModelData ModelData { get { return this._modelData; } }

        /// <summary>Gets a list of all <see cref="Manufacturer"/>s within this <see cref="Industry"/>.</summary>
        public List<Manufacturer> Manufacturers { get { return this._manufacturers; } }
        /// <summary>Gets the total count of <see cref="Manufacturer"/>s within this <see cref="Industry"/>.</summary>
        public int ManufacturerCount { get { return (this._manufacturers == null) ? 0 : this._manufacturers.Count; } }
        /// <summary>Gets a list of <see cref="Vehicle"/>s produced within this <see cref="Industry"/>.</summary>
        public List<Vehicle> Vehicles
        {
            get
            {
                // scan each manufacturer and add it's vehicles to the list
                List<Vehicle> list = new List<Vehicle>(this.VehicleCount);
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    list.AddRange(this._manufacturers[i].Vehicles);
                }
                return list;
            }
        }
        /// <summary>Gets a list of <see cref="Engine"/>s produced within this <see cref="Industry"/>.</summary>
        public List<Engine> Engines
        {
            get
            {
                // scan each manufacturer and add it's engines to the list
                List<Engine> list = new List<Engine>(this.EngineCount);
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    list.AddRange(this._manufacturers[i].Engines);
                }
                return list;
            }
        }
        /// <summary>Gets a list of <see cref="Transmission"/>s produced within this <see cref="Industry"/>.</summary>
        public List<Transmission> Transmissions
        {
            get
            {
                // scan each manufacturer and add it's transmissions to the list
                List<Transmission> list = new List<Transmission>(this.TransmissionCount);
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    list.AddRange(this._manufacturers[i].Transmissions);
                }
                return list;
            }
        }
        /// <summary>Gets a list of <see cref="Platform"/>s produced within this <see cref="Industry"/>.</summary>
        public List<Platform> Platforms
        {
            get
            {
                // scan each manufacturer and add it's platforms to the list
                List<Platform> list = new List<Platform>(this.PlatformCount);
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    list.AddRange(this._manufacturers[i].Platforms);
                }
                return list;
            }
        }
        /// <summary>Gets the total count of <see cref="Vehicle"/>s produced within this <see cref="Industry"/>.</summary>
        public int VehicleCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].VehicleCount;
                }
                return count;
            }
        }
        /// <summary>Gets the total count of <see cref="Engine"/>s within this <see cref="Industry"/>.</summary>
        public int EngineCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].Engines.Count;
                }
                return count;
            }
        }
        /// <summary>Gets the total count of <see cref="Transmission"/>s within this <see cref="Industry"/>.</summary>
        public int TransmissionCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].Transmissions.Count;
                }
                return count;
            }
        }
        /// <summary>Gets the total count of <see cref="Platform"/>s within this <see cref="Industry"/>.</summary>
        public int PlatformCount
        {
            get
            {
                int count = 0;
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    count += this._manufacturers[i].Platforms.Count;
                }
                return count;
            }
        }

        /// <summary>Gets the variety of regulatory classes represented within this <see cref="Industry"/>.</summary>
        public RCValue<bool> RegClasses
        {
            get
            {
                if (this._regClasses != null) { return this._regClasses; }
                //
                RCValue<bool> regClasses = new RCValue<bool>();
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    List<Vehicle> mfrVehs = this._manufacturers[i].Vehicles;
                    for (int j = 0; j < mfrVehs.Count; j++)
                    {
                        if (mfrVehs[j].RegClass != RegulatoryClass.None)
                        {
                            this._regClasses[mfrVehs[j].RegClass] = true;
                        }
                    }
                }
                if (this.HasValues(regClasses.Items)) { this._regClasses = regClasses; }
                return regClasses;
            }
        }
        /// <summary>Gets the variety of vehicle classes represented within this <see cref="Industry"/>.</summary>
        public VCValue<bool> VehClasses
        {
            get
            {
                if (this._vehClasses != null) { return this._vehClasses; }
                //
                VCValue<bool> vehClasses = new VCValue<bool>();
                for (int i = 0; i < this._manufacturers.Count; i++)
                {
                    List<Vehicle> mfrVehs = this._manufacturers[i].Vehicles;
                    for (int j = 0; j < mfrVehs.Count; j++)
                    {
                        if (mfrVehs[j].VehicleClass != VehicleClass.None)
                        {
                            vehClasses[mfrVehs[j].VehicleClass] = true;
                        }
                    }
                }
                if (this.HasValues(vehClasses.Items)) { this._vehClasses = vehClasses; }
                return vehClasses;
            }
        }

        // ----- industry model year info -----
        /// <summary>Gets the minimum year available for the compliance modeling process.</summary>
        public ModelYear MinYear { get { return this._minYear; } }
        /// <summary>Gets the maximum year available for the compliance modeling process.</summary>
        public ModelYear MaxYear { get { return this._maxYear; } }

        #endregion


        #region /*** Variables ***/

        // ----- industry characteristics -----
        Industry.CModelData _modelData;
        List<Manufacturer> _manufacturers;

        // ----- industry model year info -----
        ModelYear _minYear;
        ModelYear _maxYear;

        // other runtime variables
        RCValue<bool> _regClasses;
        VCValue<bool> _vehClasses;

        #endregion

    }
}
