﻿#region << Using Directives >>
using System;
using Volpe.Cafe.Generic;
#endregion

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents an object that stores compliance modeling data, as <see cref="FuelValue"/> values, for each regulatory class.
    /// </summary>
    [Serializable]
    public class RCFuelValue : RCValue<FuelValue>, ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="RCFuelValue"/> class.
        /// </summary>
        public RCFuelValue() : base() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCFuelValue"/> class using the specified initial value for all members.
        /// </summary>
        public RCFuelValue(FuelValue initialValue) : base(initialValue, initialValue, initialValue) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCFuelValue"/> class using the specified values.
        /// </summary>
        public RCFuelValue(FuelValue passengerCar, FuelValue lightTruck, FuelValue lightTruck2b3) :
            base(passengerCar, lightTruck, lightTruck2b3) { }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value that is the sum of the two specified <see cref="RCFuelValue"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the two specified <see cref="RCFuelValue"/> values.</returns>
        public static RCFuelValue operator +(RCFuelValue value1, RCFuelValue value2)
        {
            RCFuelValue result = new RCFuelValue();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] + value2.Items[i];
            }
            return result;
        }
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the sum of the specified <see cref="RCFuelValue"/> value and a
        ///// <see cref="Double"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the sum.</param>
        ///// <param name="value2">The second value to include in the sum.</param>
        ///// <returns>The sum of the specified <see cref="RCFuelValue"/> value and a <see cref="Double"/> value.</returns>
        //public static RCFuelValue operator +(RCFuelValue value1, double value2)
        //{
        //    return value1 + new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the sum of the specified <see cref="RCFuelValue"/> value and a
        ///// <see cref="Double"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the sum.</param>
        ///// <param name="value2">The second value to include in the sum.</param>
        ///// <returns>The sum of the specified <see cref="RCFuelValue"/> value and a <see cref="Double"/> value.</returns>
        //public static RCFuelValue operator +(double value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) + value2;
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the sum of the specified <see cref="RCFuelValue"/> value and an
        ///// <see cref="Int32"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the sum.</param>
        ///// <param name="value2">The second value to include in the sum.</param>
        ///// <returns>The sum of the specified <see cref="RCFuelValue"/> value and an <see cref="Int32"/> value.</returns>
        //public static RCFuelValue operator +(RCFuelValue value1, int value2)
        //{
        //    return value1 + new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the sum of the specified <see cref="RCFuelValue"/> value and an
        ///// <see cref="Int32"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the sum.</param>
        ///// <param name="value2">The second value to include in the sum.</param>
        ///// <returns>The sum of the specified <see cref="RCFuelValue"/> value and an <see cref="Int32"/> value.</returns>
        //public static RCFuelValue operator +(int value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) + value2;
        //}

        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCFuelValue operator -(RCFuelValue value1, RCFuelValue value2)
        {
            RCFuelValue result = new RCFuelValue();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] - value2.Items[i];
            }
            return result;
        }
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the difference of the second value subtracted from the first value.
        ///// </summary>
        ///// <param name="value1">The value to subtract from.</param>
        ///// <param name="value2">The value to subtract.</param>
        ///// <returns>The difference of the second value subtracted from the first value.</returns>
        //public static RCFuelValue operator -(RCFuelValue value1, double value2)
        //{
        //    return value1 - new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the difference of the second value subtracted from the first value.
        ///// </summary>
        ///// <param name="value1">The value to subtract from.</param>
        ///// <param name="value2">The value to subtract.</param>
        ///// <returns>The difference of the second value subtracted from the first value.</returns>
        //public static RCFuelValue operator -(double value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) - value2;
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the difference of the second value subtracted from the first value.
        ///// </summary>
        ///// <param name="value1">The value to subtract from.</param>
        ///// <param name="value2">The value to subtract.</param>
        ///// <returns>The difference of the second value subtracted from the first value.</returns>
        //public static RCFuelValue operator -(RCFuelValue value1, int value2)
        //{
        //    return value1 - new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the difference of the second value subtracted from the first value.
        ///// </summary>
        ///// <param name="value1">The value to subtract from.</param>
        ///// <param name="value2">The value to subtract.</param>
        ///// <returns>The difference of the second value subtracted from the first value.</returns>
        //public static RCFuelValue operator -(int value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) - value2;
        //}

        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value that is the product of the two specified <see cref="RCFuelValue"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the two specified <see cref="RCFuelValue"/> values.</returns>
        public static RCFuelValue operator *(RCFuelValue value1, RCFuelValue value2)
        {
            RCFuelValue result = new RCFuelValue();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] * value2.Items[i];
            }
            return result;
        }
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the product of the specified <see cref="RCFuelValue"/> value and a
        ///// <see cref="Double"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the product.</param>
        ///// <param name="value2">The second value to include in the product</param>
        ///// <returns>The product of the specified <see cref="RCFuelValue"/> value and a <see cref="Double"/> value.</returns>
        //public static RCFuelValue operator *(RCFuelValue value1, double value2)
        //{
        //    return value1 * new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the product of the specified <see cref="RCFuelValue"/> value and a
        ///// <see cref="Double"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the product.</param>
        ///// <param name="value2">The second value to include in the product</param>
        ///// <returns>The product of the specified <see cref="RCFuelValue"/> value and a <see cref="Double"/> value.</returns>
        //public static RCFuelValue operator *(double value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) * value2;
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the product of the specified <see cref="RCFuelValue"/> value and an
        ///// <see cref="Int32"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the product.</param>
        ///// <param name="value2">The second value to include in the product</param>
        ///// <returns>The product of the specified <see cref="RCFuelValue"/> value and an <see cref="Int32"/> value.</returns>
        //public static RCFuelValue operator *(RCFuelValue value1, int value2)
        //{
        //    return value1 * new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the product of the specified <see cref="RCFuelValue"/> value and an
        ///// <see cref="Int32"/> value.
        ///// </summary>
        ///// <param name="value1">The first value to include in the product.</param>
        ///// <param name="value2">The second value to include in the product</param>
        ///// <returns>The product of the specified <see cref="RCFuelValue"/> value and an <see cref="Int32"/> value.</returns>
        //public static RCFuelValue operator *(int value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) * value2;
        //}

        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCFuelValue operator /(RCFuelValue value1, RCFuelValue value2)
        {
            RCFuelValue result = new RCFuelValue();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] / value2.Items[i];
            }
            return result;
        }
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the first value divided by the second value.
        ///// </summary>
        ///// <param name="value1">The value of the numerator.</param>
        ///// <param name="value2">The value of the denominator.</param>
        ///// <returns>The first value divided by the second value.</returns>
        //public static RCFuelValue operator /(RCFuelValue value1, double value2)
        //{
        //    return value1 / new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the first value divided by the second value.
        ///// </summary>
        ///// <param name="value1">The value of the numerator.</param>
        ///// <param name="value2">The value of the denominator.</param>
        ///// <returns>The first value divided by the second value.</returns>
        //public static RCFuelValue operator /(double value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) / value2;
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the first value divided by the second value.
        ///// </summary>
        ///// <param name="value1">The value of the numerator.</param>
        ///// <param name="value2">The value of the denominator.</param>
        ///// <returns>The first value divided by the second value.</returns>
        //public static RCFuelValue operator /(RCFuelValue value1, int value2)
        //{
        //    return value1 / new RCFuelValue(value2);
        //}
        ///// <summary>
        ///// Returns a new <see cref="RCFuelValue"/> value that is the first value divided by the second value.
        ///// </summary>
        ///// <param name="value1">The value of the numerator.</param>
        ///// <param name="value2">The value of the denominator.</param>
        ///// <returns>The first value divided by the second value.</returns>
        //public static RCFuelValue operator /(int value1, RCFuelValue value2)
        //{
        //    return new RCFuelValue(value1) / value2;
        //}

        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value, which is the copy of the specified <see cref="RCFuelValue"/> value.
        /// </summary>
        /// <param name="value">The value to copy.</param>
        /// <returns>A copy of the specified <see cref="RCFuelValue"/> value.</returns>
        public static RCFuelValue operator +(RCFuelValue value)
        {
            return RCFuelValue.Zero + value;
        }
        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value, which is the negative of the specified <see cref="RCFuelValue"/> value.
        /// </summary>
        /// <param name="value">The value to negate.</param>
        /// <returns>The specified <see cref="RCFuelValue"/> value negated.</returns>
        public static RCFuelValue operator -(RCFuelValue value)
        {
            return RCFuelValue.Zero - value;
        }
        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value, which is one more than the specified value (increment).
        /// </summary>
        /// <param name="value">The value to increment by one.</param>
        /// <returns>The specified value, incremented by one.</returns>
        public static RCFuelValue operator ++(RCFuelValue value)
        {
            return value + RCFuelValue.One;
        }
        /// <summary>
        /// Returns a new <see cref="RCFuelValue"/> value, which is one less than the specified value (decrement).
        /// </summary>
        /// <param name="value">The value to decrement by one.</param>
        /// <returns>The specified value, decremented by one.</returns>
        public static RCFuelValue operator --(RCFuelValue value)
        {
            return value - RCFuelValue.One;
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="RCFuelValue"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCFuelValue"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a copy of the current <see cref="RCFuelValue"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCFuelValue"/>.</returns>
        public new RCFuelValue Clone()
        {
            RCFuelValue value = new RCFuelValue();
            this.CopyTo(value);
            return value;
        }

        #endregion

        /// <summary>
        /// Returns the string representation of this <see cref="RCFuelValue"/> instance, using the specified format for all elements.
        /// </summary>
        /// <param name="format">A format string.</param>
        /// <returns>The string representation of the <see cref="RCFuelValue"/> instance, using the specified format.</returns>
        public string ToString(string format)
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString(format));
            }
            return "{" + s + "}";
        }
        /// <summary>
        /// Returns the string representation of this <see cref="RCFuelValue"/> instance, whose elements are rounded with the specified precision.
        /// </summary>
        /// <param name="digits">The rounding precision; e.g. number of digits after the period.</param>
        /// <returns>The string representation of the <see cref="RCFuelValue"/> instance.</returns>
        public string ToString(int digits)
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString(digits));
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Returns the compliance modeling data value for the specified regulatory class, or the <see cref="Total"/> of all
        /// regulatory classes if <see cref="RegulatoryClass.All"/> is specified.
        /// </summary>
        /// <param name="regClass">The regulatory class for which to obtain the value.</param>
        /// <returns>The compliance modeling data value for the specified regulatory class.</returns>
        public FuelValue GetValue(RegulatoryClass regClass)
        {
            return (regClass == RegulatoryClass.All) ? this.Total : this[regClass];
        }

        #endregion

        #region /*** Properties ***/

        //----- static values -----
        /// <summary>Gets the <see cref="RCFuelValue"/> value whose elements are all 0.</summary>
        public static RCFuelValue Zero { get { return new RCFuelValue(FuelValue.Zero); } }
        /// <summary>Gets the <see cref="RCFuelValue"/> value whose elements are all 1.</summary>
        public static RCFuelValue One { get { return new RCFuelValue(FuelValue.One); } }

        //----- instance values -----
        /// <summary>Gets or sets the compliance modeling data value based on the given regulatory class and fuel type.</summary>
        /// <param name="regClass">The regulatory class for which to get or set the compliance modeling data.</param>
        /// <param name="fuelType">The fuel type for which to get or set the compliance modeling data.</param>
        /// <returns>The compliance modeling data value based on the given regulatory class and fuel type.</returns>
        /// <exception cref="System.ArgumentException">The value specified does not represent a valid regulatory class.</exception>
        public double this[RegulatoryClass regClass, FuelType fuelType]
        {
            get { return this.Items[this.GetIndex(regClass)][fuelType]; }
            set { this.Items[this.GetIndex(regClass)][fuelType] = value; }
        }

        /// <summary>Gets the sum of the compliance modeling data of all regulatory classes.</summary>
        public FuelValue Total
        {
            get
            {
                FuelValue total = FuelValue.Zero;
                for (int i = 0; i < this.Items.Length; i++)
                {
                    total += this.Items[i];
                }
                return total;
            }
        }

        #endregion

    }
}
