using System;
using System.Collections.Generic;
using Volpe.Cafe;
using Volpe.Cafe.Utils;
using RC = Volpe.Cafe.RegulatoryClass;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents a single vehicle model.
    /// </summary>
    [Serializable]
    public sealed class Vehicle
    {

        #region /*** Nested Types ***/

        /// <summary>
        /// Provides a description of engineering characteristics for a <see cref="Vehicle"/>.
        /// </summary>
        [Serializable]
        public sealed class CDescription
        {

            #region /*** Constructors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Vehicle.CDescription"/> class.
            /// </summary>
            internal CDescription() { }

            #endregion


            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Vehicle.CDescription"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Vehicle.CDescription"/>.</returns>
            internal CDescription Clone()
            {
                Vehicle.CDescription vd = new Vehicle.CDescription();
                //
                vd.Code                  = this.Code;
                vd.Manufacturer          = this.Manufacturer;
                vd.Brand                 = this.Brand;
                vd.Model                 = this.Model;
                vd.Nameplate             = this.Nameplate;
                vd.Platform              = this.Platform;
                vd.EngineCode            = this.EngineCode;
                vd.TransmissionCode      = this.TransmissionCode;
                vd.Origin                = this.Origin;
                vd.BaseFE                = this.BaseFE;
                vd.FuelEconomy           = this.FuelEconomy;
                vd.FuelShare             = this.FuelShare;
                vd.Sales                 = (double[])Interaction.CloneArray(this.Sales, typeof(double));
                vd.Msrp                  = this.Msrp;
                vd.RegulatoryIndicator   = this.RegulatoryIndicator;
                vd.TechnologyClass       = this.TechnologyClass;
                vd.EngineTechnologyClass = this.EngineTechnologyClass;
                vd.SafetyClass           = this.SafetyClass;
                vd.ZEVCandidate          = this.ZEVCandidate;
                vd.Style                 = this.Style;
                vd.Structure             = this.Structure;
                vd.Drive                 = this.Drive;
                vd.Wheelbase             = this.Wheelbase;
                vd.Footprint             = this.Footprint;
                vd.GliderWeightMR0       = this.GliderWeightMR0;
                vd.CurbWeight            = this.CurbWeight;
                vd.GVWR                  = this.GVWR;
                vd.GCWR                  = this.GCWR;
                vd.MaxGVWRToCW           = this.MaxGVWRToCW;
                vd.MaxGCWRToGVWR         = this.MaxGCWRToGVWR;
                vd.FuelCapacity          = this.FuelCapacity;
                vd.VehiclePower          = this.VehiclePower;
                vd.VehicleTorque         = this.VehicleTorque;
                vd.VehicleRange          = this.VehicleRange;
                vd.HybridType            = this.HybridType;
                vd.RefreshYears          = (int[])Interaction.CloneArray(this.RefreshYears , typeof(int));
                vd.RedesignYears         = (int[])Interaction.CloneArray(this.RedesignYears, typeof(int));
                vd.EmploymentHours       = this.EmploymentHours;
                //
                vd.UsedTechnologies      = (int[])Interaction.CloneArray(this.UsedTechnologies     , typeof(int));
                vd.AvailableTechnologies = (int[])Interaction.CloneArray(this.AvailableTechnologies, typeof(int));
                //
                return vd;
            }

            /// <summary>
            /// Returns the string representation of this <see cref="Vehicle.CDescription"/> instance.
            /// </summary>
            /// <returns>The string representation of the <see cref="Vehicle.CDescription"/> instance.</returns>
            public override string ToString()
            {
                return (this.Model + "(Eng: " + this.EngineCode + ", Trn: " + this.TransmissionCode + ")");
            }

            /// <summary>
            /// Calculates the average vehicle sales, averaged accross all model years where sales exist.
            /// </summary>
            /// <returns>The average vehicle sales for all model years.</returns>
            public double CalcAverageSales()
            {
                int    count = 0;
                double sales = 0;
                for (int i = 0; i < this.Sales.Length; i++)
                {
                    if (this.Sales[i] > 0)
                    {
                        sales += this.Sales[i];
                        count++;
                    }
                }
                return (count == 0) ? 0 : sales / count;
            }
            /// <summary>
            /// Calculates the average vehicle MSRP, averaged accross all model years where MSRP values exist.
            /// </summary>
            /// <returns>The average vehicle MSRP for all model years.</returns>
            public double CalcAverageMSRP()
            {
                return this.Msrp;
            }

            #endregion


            #region /*** Properties ***/

            /// <summary>Gets the payload capacity of the vehicle, which is computed as GVWR - Curb Weight.</summary>
            public double PayloadCapacity { get { return this.GVWR - this.CurbWeight; } }
            /// <summary>Gets the towing capacity of the vehicle, which is computed as GCWR - GVWR.</summary>
            public double TowingCapacity { get { return this.GCWR - this.GVWR; } }

            /// <summary>Gets the loaded vehcle weight, which is computed as Curb Weight + 300 lbs.</summary>
            public double LVW { get { return this.CurbWeight + 300; } }
            /// <summary>Gets the adjusted loaded vehcle weight, which is computed as (Curb Weight + GVWR) / 2.</summary>
            public double ALVW { get { return (this.CurbWeight + this.GVWR) / 2; } }

            #endregion


            #region /*** Variables ***/

            /// <summary>Represents the internal manufacturer specific vehicle code.</summary>
            public int Code;
            /// <summary>Specifies the manufacturer name that produces this vehicle.</summary>
            public string Manufacturer;
            /// <summary>Specifies the brand of the vehicle.</summary>
            public string Brand;
            /// <summary>Specifies the model name of the vehicle.</summary>
            public string Model;
            /// <summary>Specifies the nameplate of the vehicle.</summary>
            public string Nameplate;
            /// <summary>Specifies the platform of the vehicle.</summary>
            public string Platform;

            /// <summary>Represents the internal manufacturer specific engine code of the engine that the vehicle uses.</summary>
            public int EngineCode;
            /// <summary>Represents the internal manufacturer specific transmission code of the transmission that the vehicle uses.</summary>
            public int TransmissionCode;
            /// <summary>Represents whether the vehicle is domestic, imported, or other.</summary>
            public char Origin;

            /// <summary>Represents the product-plan fuel economy of the vehicle as it was read in from the input file. The base
            ///   fuel economy value will remain unaffected whenever the vehicle's fuel economy changes.</summary>
            public FuelValue BaseFE;
            /// <summary>Represents the CAFE fuel economy rating of the vehicle for each fuel type.</summary>
            public FuelValue FuelEconomy;
            /// <summary>Represents the percent share that the vehicle runs on each fuel type.</summary>
            public FuelValue FuelShare;

            /// <summary>Specifies the vehicle's forecast of sales for each model year.</summary>
            public double[] Sales;
            /// <summary>Specifies the vehicle's manufacturer suggested retail price.</summary>
            public double Msrp;

            /// <summary>Specifies whether the vehicle should be regulated as a Light Truck (LT) or a Passenger Car (PC).</summary>
            public string RegulatoryIndicator;
            /// <summary>Represents the technology class assignment of the vehicle.</summary>
            public string TechnologyClass;
            /// <summary>Represents the technology cost class assignment of the vehicle.</summary>
            public string EngineTechnologyClass;
            /// <summary>Represents the safety class or classes of a vehicle.</summary>
            public string SafetyClass;
            /// <summary>Specifies whether the vehicle is a preferred candiate for ZEV technology application.</summary>
            public HEVType ZEVCandidate;

            /// <summary>Specifies the vehicle style.</summary>
            public string Style;
            /// <summary>Specifies the vehicle structure.</summary>
            public string Structure;
            /// <summary>Represents the drive type of the vehicle.</summary>
            public char Drive;
            /// <summary>Represents the vehicle's wheelbase.</summary>
            public double Wheelbase;
            /// <summary>Represents the vehicle footprint.</summary>
            public double Footprint;
            /// <summary>Represents the glider weight of the vehicle (negating any mass reduction).</summary>
            public double GliderWeightMR0;
            /// <summary>Represents the curb weight of the vehicle.</summary>
            public double CurbWeight;
            /// <summary>Represents the Gross Vehicle Weight Rating of the vehicle.</summary>
            public double GVWR;
            /// <summary>Represents the Gross Combined Weight Rating of the vehicle.</summary>
            public double GCWR;
            /// <summary>Represents the maximum ratio of GVWR to Curb Weight allowed for the vehicle.</summary>
            public double MaxGVWRToCW;
            /// <summary>Represents the maximum ratio of GCWR to GVWR allowed for the vehicle.</summary>
            public double MaxGCWRToGVWR;
            ///// <summary>Represents the payload capacity of the vehicle.</summary>
            //public double PayloadCapacity;
            ///// <summary>Represents the towing capacity of the vehicle.</summary>
            //public double TowingCapacity;
            /// <summary>Represents the fuel capacity of the vehicle.</summary>
            public double FuelCapacity;
            /// <summary>Represents the overall power rating of the vehicle, including engine and motor power.</summary>
            public double VehiclePower;
            /// <summary>Represents the overall torque of the vehicle, including engine and motor torque.</summary>
            public double VehicleTorque;
            /// <summary>Represents the overall range of the vehicle, in miles, including engine and motor range.</summary>
            public double VehicleRange;
            /// <summary>Represents the hybridization type of the vehicle, if applicable.</summary>
            public string HybridType;

            /// <summary>Represents an array of last and projected next refresh (or freshening) years of the vehicle.</summary>
            public int[] RefreshYears;
            /// <summary>Represents an array of last and projected next redesign years of the vehicle.</summary>
            public int[] RedesignYears;
            /// <summary>Represents the average employement hours per unit of the vehicle.</summary>
            public double EmploymentHours;

            /// <summary>Represents the indexes of technologies initially installed on the baseline vehicle.</summary>
            public int[] UsedTechnologies;
            /// <summary>Represents the indexes of technologies initially available for applicability on the baseline vehicle.</summary>
            public int[] AvailableTechnologies;

            #endregion

        }

        /// <summary>
        /// Provides compliance model data for a <see cref="Vehicle"/>, which is updated during runtime.
        /// </summary>
        [Serializable]
        public sealed class CModelData
        {

            #region /*** Constructors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Vehicle.CModelData"/> class.
            /// </summary>
            internal CModelData()
            {
                this.TechLingerYear = new List<ModelYear>();
            }

            #endregion


            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Vehicle.CModelData"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Vehicle.CModelData"/>.</returns>
            internal Vehicle.CModelData Clone()
            {
                Vehicle.CModelData cv = new Vehicle.CModelData();
                //
                cv.TechCost              = this.TechCost;
                cv.TechCostTimeBased     = (double[])Interaction.CloneArray(this.TechCostTimeBased, typeof(double));
                cv.RegCost               = this.RegCost;
                cv.DiscCost              = this.DiscCost;
                cv.ConsumerValuation     = this.ConsumerValuation;
                cv.RelativeLossOfValue   = this.RelativeLossOfValue;
                cv.MaintenanceCost       = this.MaintenanceCost;
                cv.RepairCost            = this.RepairCost;
                cv.TaxesAndFees          = this.TaxesAndFees;
                cv.FinancingCost         = this.FinancingCost;
                cv.InsuranceCost         = this.InsuranceCost;
                cv.TaxCredit             = this.TaxCredit;
                cv.OffCycleCredits       = this.OffCycleCredits;
                cv.ZEVCredits            = this.ZEVCredits;
                cv.SalesOverFE           = this.SalesOverFE;
                cv.TechAvailable         = (bool[])Interaction.CloneArray(this.TechAvailable     , typeof(bool));
                cv.TechEnabled           = (bool[])Interaction.CloneArray(this.TechEnabled       , typeof(bool));
                cv.TechUsed              = (bool[])Interaction.CloneArray(this.TechUsed          , typeof(bool));
                cv.TechInherited         = (bool[])Interaction.CloneArray(this.TechInherited     , typeof(bool));
                cv.TechApplied           = (bool[])Interaction.CloneArray(this.TechApplied       , typeof(bool));
                cv.TechAppliedYear       = (int [])Interaction.CloneArray(this.TechAppliedYear   , typeof(int ));
                cv.TechSuperseded        = (bool[])Interaction.CloneArray(this.TechSuperseded    , typeof(bool));
                cv.TechSupersededYear    = (int [])Interaction.CloneArray(this.TechSupersededYear, typeof(int ));
                cv.TechIgnored           = (bool[])Interaction.CloneArray(this.TechIgnored       , typeof(bool));
                cv.FCAdjKey              = this.FCAdjKey;
                for (int i = 0; i < this.TechLingerYear.Count; i++)
                {
                    cv.TechLingerYear.Add(this.TechLingerYear[i]);
                }
                //
                return cv;
            }

            #endregion


            #region /*** Variables ***/

            /// <summary>Represents the technology costs accumulated by the <see cref="Vehicle"/>.</summary>
            public double TechCost;
            /// <summary>Represents the technology costs accumulated by the <see cref="Vehicle"/> from time-based technologies
            ///   only. This variable is useful when recalculating vehicle costs at the beginning of a model year to adjust for
            ///   learning.</summary>
            public double[] TechCostTimeBased;
            /// <summary>Represents the regulatory costs (fines + tech costs) accumulated by the <see cref="Vehicle"/>.</summary>
            public double RegCost;
            /// <summary>Represents the discounted technology costs accumulated by the <see cref="Vehicle"/>.</summary>
            public double DiscCost;
            /// <summary>Represents the changes in consumer valuation due to application of technologies accumulated by the <see cref="Vehicle"/>.</summary>
            public double ConsumerValuation;
            /// <summary>Represents the relative loss in value to the consumer due to lower operating life of a <see cref="Vehicle"/> that was converted to a pure EV.</summary>
            public double RelativeLossOfValue;
            /// <summary>Represents the technology maintenance costs accumulated by the <see cref="Vehicle"/>.</summary>
            public double MaintenanceCost;
            /// <summary>Represents the technology repair costs accumulated by the <see cref="Vehicle"/>.</summary>
            public double RepairCost;

            /// <summary>Specifies the amount a buyer is expected to pay in taxes and fees for purchasing the <see cref="Vehicle"/>.</summary>
            public double TaxesAndFees;
            /// <summary>Specifies the amount a buyer is expected to pay for financing the <see cref="Vehicle"/>.</summary>
            public double FinancingCost;
            /// <summary>Specifies the amount a buyer is expected to pay for insuring the <see cref="Vehicle"/>.</summary>
            public double InsuranceCost;

            /// <summary>Represents the tax break (in $) realized by the consumer for purchasing the <see cref="Vehicle"/>.</summary>
            public double TaxCredit;

            /// <summary>Represents the off cycle credits accumulated by the <see cref="Vehicle"/>.</summary>
            public double OffCycleCredits;

            /// <summary>Represents the ZEVs credits accumulated by the <see cref="Vehicle"/>.</summary>
            public double ZEVCredits;

            /// <summary>Represents the sales to fuel economy ratio for the <see cref="Vehicle"/> for the model year being analyzed.</summary>
            public double SalesOverFE;

            /// <summary>Specifies that the technology is generally available for application to a <see cref="Vehicle"/>.</summary>
            /// <remarks>General availability does not necessarily imply that the technology may be applied in a given model year.
            ///   Technologies applied in previous years or model year specific overrides may render the technology inapplicable in
            ///   subsequent years.  This variable should be initialized once before starting a new scenario.</remarks>
            public bool[] TechAvailable;
            /// <summary>Specifies that the technology is available for application to a <see cref="Vehicle"/> in the model year
            ///   that is being analyzed, taking into account model year specific technology overrides.</summary>
            /// <remarks>This variable should be initialized each time before starting a new model year.</remarks>
            public bool[] TechEnabled;

            /// <summary>Specifies that the technology is currently being used by the <see cref="Vehicle"/>, <see cref="Engine"/>, or
            ///   <see cref="Transmission"/>.</summary>
            /// <remarks>This variable should be initialized once before starting a new scenario.  This variable will be updated
            ///   whenever a new technology is applied to a <see cref="Vehicle"/> during the compliance modeling process.</remarks>
            public bool[] TechUsed;
            /// <summary>Specifies that the technology has been inherited on the <see cref="Vehicle"/> from its <see cref="Engine"/>,
            ///   <see cref="Transmission"/>, or <see cref="Platform"/>.</summary>
            /// <remarks>This variable should be initialized once before starting a new scenario.  This variable will be updated
            ///   whenever a new technology is inherited on a <see cref="Vehicle"/> during the compliance modeling process.</remarks>
            public bool[] TechInherited;
            /// <summary>Specifies that the technology has been applied during the modeling process and is currently being used by
            ///   the <see cref="Vehicle"/>, <see cref="Engine"/>, or <see cref="Transmission"/>.</summary>
            /// <remarks>This variable should be initialized once before starting a new scenario.  This variable will be updated
            ///   whenever a new technology is applied to a <see cref="Vehicle"/> during the compliance modeling process.</remarks>
            public bool[] TechApplied;
            /// <summary>Specifies the model year when the technology was applied to a vehicle during the modeling process.</summary>
            public int[] TechAppliedYear;
            /// <summary>Specifies that the technology has been used by the <see cref="Vehicle"/>, <see cref="Engine"/>, or
            ///   <see cref="Transmission"/> at some point, but was later superseded by another technology.  Technologies can be
            ///   marked as superseded if they were present in the <see cref="Manufacturer"/>'s product plan, or applied during the
            ///   modeling process.</summary>
            /// <remarks>This variable should be initialized once before starting a new scenario.  This variable will be updated
            ///   whenever a technology that was applied to a <see cref="Vehicle"/> during the compliance modeling process supersedes
            ///   another technology that is already present on that <see cref="Vehicle"/>.</remarks>
            public bool[] TechSuperseded;
            /// <summary>Specifies the model year when the technology was superseded on a vehicle by another technology during
            ///   the modeling process.</summary>
            public int[] TechSupersededYear;

            /// <summary>Specifies that the technology has been turned off for the <see cref="Vehicle"/>, and will no longer be used
            ///   in the model year that is being analyzed.</summary>
            /// <remarks>This variable should be initialized each time before starting a new model year.</remarks>
            public bool[] TechIgnored;

            /// <summary>Represents a technology key used for looking up fuel consumption adjustment factors.</summary>
            public ulong FCAdjKey;

            /// <summary>Specifies the most recent redesign year for which there are lingering technologies.</summary>
            public List<ModelYear> TechLingerYear;

            #endregion

        }

        #endregion


        #region /*** Constructors ***/

        // Private constructor used for internal usage (such as cloning).
        Vehicle()
        {
            // vehicle characteristics
            this.Description  = null;
            this.ModelData    = null;
            this.Manufacturer = null;
            this.Engine       = null;
            this.Transmission = null;
            this.Platform     = null;
            this.LegacyEng    = null;
            this.LegacyTrn    = null;

            // regulatory class assignments
            this.RegClass        = RegulatoryClass.None;
            this.TechnologyClass = -1;
            this.SafetyClass     = SafetyClass    .None;
            this.VehicleClass    = VehicleClass   .None;
            this.VehicleStyle    = VehicleStyle   .None;
            this.HEVType         = HEVType        .None;

            // vehicle model year info
            this.MinYear = null;
            this.MaxYear = null;
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="Vehicle"/> class, using the specified vehicle <see cref="Description"/>.
        /// </summary>
        /// <param name="description">A description of engineering characteristics for this <see cref="Vehicle"/>.</param>
        internal Vehicle(Vehicle.CDescription description) : this()
        {
            this.Description = description;
            this.ModelData = new Vehicle.CModelData();
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Vehicle"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Vehicle"/>.</returns>
        internal Vehicle Clone()
        {
            Vehicle veh = new Vehicle();
            //
            // vehicle characteristics
            veh.Description  = this.Description.Clone();
            veh.ModelData    = this.ModelData  .Clone();
            veh.Manufacturer = this.Manufacturer;
            veh.Engine       = this.Engine;
            veh.Transmission = this.Transmission;
            veh.Platform     = this.Platform;
            // clone legacy eng & trn, if those were removed (occurs when a veh is converted to strong HEV, plug-in HEV, EV, or FCV)
            if (this.LegacyEng != null) { veh.LegacyEng = this.LegacyEng.Clone(); }
            if (this.LegacyTrn != null) { veh.LegacyTrn = this.LegacyTrn.Clone(); }
            // regulatory class assignments
            veh.RegClass              = this.RegClass;
            veh.TechnologyClass       = this.TechnologyClass;
            veh.EngineTechnologyClass = this.EngineTechnologyClass;
            veh.SafetyClass           = this.SafetyClass;
            veh.VehicleClass          = this.VehicleClass;
            veh.VehicleStyle          = this.VehicleStyle;
            // hybrid/electric type
            veh.HEVType               = this.HEVType;
            // vehicle model year info
            veh.MinYear               = new ModelYear(this.MinYear.Year);
            veh.MaxYear               = new ModelYear(this.MaxYear.Year);
            veh.ProductionYears       = this.ProductionYears;
            //
            return veh;
        }

        /// <summary>
        /// Initializes the <see cref="Vehicle"/> by setting the <see cref="Manufacturer"/> reference and parsing the
        /// <see cref="Engine"/> and <see cref="Transmission"/> lists.
        /// </summary>
        internal void Initialize(Manufacturer manufacturer)
        {
            // set manufacturer reference
            this.Manufacturer = manufacturer;

            // clear out references
            this.Engine       = null;
            this.Transmission = null;

            // scan each engine and transmission codes to get their references
            Engine      [] engs = manufacturer.Engines      .ToArray();
            Transmission[] trns = manufacturer.Transmissions.ToArray();

            // scan engines (if vehicle has a valid engine code)
            if (this.Description.EngineCode > 0)
            {
                for (int i = 0; i < engs.Length; i++)
                {
                    if (engs[i].Description.Code == this.Description.EngineCode)
                    {   // found match--set engine reference
                        this.Engine = engs[i];
                        // also add the vehicle to the engine's vehicles list
                        this.Engine.Vehicles.Add(this);
                        break;
                    }
                }
            }

            // scan transmissions (if vehicle has a valid transmission code)
            if (this.Description.TransmissionCode > 0)
            {
                for (int i = 0; i < trns.Length; i++)
                {
                    if (trns[i].Description.Code == this.Description.TransmissionCode)
                    {   // found match--set transmission reference
                        this.Transmission = trns[i];
                        // also add the vehicle to the transmission's vehicles list
                        this.Transmission.Vehicles.Add(this);
                        break;
                    }
                }
            }

            // look for min, max, and total production years
            int yearCount = this.Description.Sales.Length;
            int minYear = -1;
            int maxYear = -1;
            int ttlYear = 0;
            for (int i = 0; i < yearCount; i++)
            {
                if (this.Description.Sales[i] > 0)
                {
                    if (minYear == -1) { minYear = i; }
                    maxYear = i;
                    ttlYear++;
                }
            }
            if (ttlYear == 0)
            {
                throw new Exception("Vehicle (Code:" + this.Description.Code + ",Mfr:" + this.Description.Manufacturer +
                    "):  does not have any valid production years.");
            }
            this.MinYear = ModelYear.NewModelYearFromIndex(minYear);
            this.MaxYear = ModelYear.NewModelYearFromIndex(maxYear);
            this.ProductionYears = ttlYear;

            //-------------------------------------------------------------------------------------------------//
            // check for valid engine and transmission                                                         //
            //  - electricity-only or hydrogen-only vehicles should not have an engine or transmission present //
            //  - strong hybrid vehicles should not have a transmission present                                //
            //-------------------------------------------------------------------------------------------------//
            bool isEV_FCV = (this.Description.FuelEconomy.PrimaryFuel == FuelType.Electricity ||
                this.Description.FuelEconomy.PrimaryFuel == FuelType.Hydrogen);
            bool isHEV = false;
            // check the tech-used states on the modeling data
            if (this.ModelData.TechUsed != null)
            {
                for (int i = 0; i < TechnologyIndexes.TechnologyCount; i++)
                {
                    if (this.ModelData.TechUsed[i])
                    {
                        if (TechnologyIndexes.IsConversionToSHEV(i) ||
                            TechnologyIndexes.IsConversionToPHEV(i)) { isHEV = true; }
                        if (TechnologyIndexes.IsConversionToEV  (i) ||
                            TechnologyIndexes.IsConversionToFCV (i)) { isEV_FCV = true; }
                    }
                }
            }
            // check used-techs from the product plan
            for (int i = 0; i < this.Description.UsedTechnologies.Length; i++)
            {
                if (TechnologyIndexes.IsConversionToSHEV(this.Description.UsedTechnologies[i]) ||
                    TechnologyIndexes.IsConversionToPHEV(this.Description.UsedTechnologies[i])) { isHEV = true; }
                if (TechnologyIndexes.IsConversionToEV  (this.Description.UsedTechnologies[i]) ||
                    TechnologyIndexes.IsConversionToFCV (this.Description.UsedTechnologies[i])) { isEV_FCV = true; }
            }
            if (!(isEV_FCV || isHEV))
            {
                string hdr = "Vehicle (Code:" + this.Description.Code + ",Mfr:" + this.Description.Manufacturer + "):  ";
                if (this.Engine       == null) { throw new Exception("\n\n" + hdr + "Engine Code is not valid."); }
                if (this.Transmission == null) { throw new Exception("\n\n" + hdr + "Transmission Code is not valid."); }
            }

            // assign vehicle style/class, technology class, and safety class
            this.AssignVehStyleAndVehClass();
            this.AssignTechnologyClass();
            this.AssignSafetyClass();
        }
        void AssignVehStyleAndVehClass()
        {
            // assign vehicle style
            switch (this.Description.Style.ToUpper())
            {
                case "CONVERTIBLE"  : this.VehicleStyle = VehicleStyle.Convertible ; break;
                case "COUPE"        : this.VehicleStyle = VehicleStyle.Coupe       ; break;
                case "HATCHBACK"    : this.VehicleStyle = VehicleStyle.Hatchback   ; break;
                case "SEDAN"        : this.VehicleStyle = VehicleStyle.Sedan       ; break;
                case "WAGON"        : this.VehicleStyle = VehicleStyle.Wagon       ; break;

                case "SPORT UTILITY": this.VehicleStyle = VehicleStyle.SportUtility; break;
                case "MINIVAN"      : this.VehicleStyle = VehicleStyle.Minivan     ; break;
                case "VAN"          : this.VehicleStyle = VehicleStyle.Van         ; break;
                case "PASSENGER VAN": this.VehicleStyle = VehicleStyle.PassengerVan; break;
                case "CARGO VAN"    : this.VehicleStyle = VehicleStyle.CargoVan    ; break;

                case "PICKUP"       : this.VehicleStyle = VehicleStyle.Pickup      ; break;
                case "LARGE PICKUP" : this.VehicleStyle = VehicleStyle.LargePickup ; break;
                case "CHASSIS CAB"  : this.VehicleStyle = VehicleStyle.ChassisCab  ; break;
                case "CUTAWAY"      : this.VehicleStyle = VehicleStyle.Cutaway     ; break;
                default             : this.VehicleStyle = VehicleStyle.None        ; break;
            }
            // assign vehicle class
            if (this.VehicleStyle == VehicleStyle.Convertible || this.VehicleStyle == VehicleStyle.Coupe ||
                this.VehicleStyle == VehicleStyle.Hatchback   || this.VehicleStyle == VehicleStyle.Sedan ||
                this.VehicleStyle == VehicleStyle.Wagon)
            {   // the vehicle is a passenger car
                this.VehicleClass = VehicleClass.LDV;
            }
            else
            {   // the vehicle is a light truck
                this.VehicleClass = (this.Description.GVWR >= 10001) ? VehicleClass.LDT3  :
                                    (this.Description.GVWR >=  8501) ? VehicleClass.LDT2b :
                                    (this.Description.GVWR >=  6001) ? VehicleClass.LDT2a :
                                                                       VehicleClass.LDT1;
            }
        }
        void AssignTechnologyClass()
        {
            // set tech class to "none" initially
            this.TechnologyClass = -1;
            // scan each class in description to look for a match
            for (int i = 0; i < TechnologyClasses.ClassCount; i++)
            {
                if (Interaction.StringCompare(this.Description.TechnologyClass, TechnologyClasses.ClassNames[i], true))
                {
                    this.TechnologyClass = i;
                }
            }
            for (int i = 0; i < TechnologyClasses.EngineClassCount; i++)
            {
                if (Interaction.StringCompare(this.Description.EngineTechnologyClass, TechnologyClasses.EngineClassNames[i], true))
                {
                    this.EngineTechnologyClass = i;
                }
            }
        }
        void AssignSafetyClass()
        {
            // assign safety class
            switch (this.Description.SafetyClass.ToUpper())
            {
                case "PC": this.SafetyClass = SafetyClass.PC  ; break;
                case "LT": this.SafetyClass = SafetyClass.LT  ; break;
                case "CM": this.SafetyClass = SafetyClass.CM  ; break;
                default  : this.SafetyClass = SafetyClass.None; break;
            }
        }

        /// <summary>
        /// Sets the <see cref="Engine"/> reference for the <see cref="Vehicle"/>.
        /// </summary>
        internal void SetEngine(Engine engine)
        {
            this.Engine = engine;
            this.Description.EngineCode = engine.Description.Code;
        }
        /// <summary>
        /// Sets the <see cref="Transmission"/> reference for the <see cref="Vehicle"/>.
        /// </summary>
        internal void SetTransmission(Transmission transmission)
        {
            this.Transmission = transmission;
            this.Description.TransmissionCode = transmission.Description.Code;
        }
        /// <summary>
        /// Sets the <see cref="Platform"/> reference for the <see cref="Vehicle"/>.
        /// </summary>
        internal void SetPlatform(Platform platform)
        {
            this.Platform = platform;
            this.Description.Platform = platform.Name;
        }

        /// <summary>
        /// Removes an engine from the vehicle.  The vehicle's engine should only be removed after the vehicle is converted to
        /// pure electric or fuel cell.
        /// </summary>
        public void RemoveEngine()
        {
            if (this.Engine != null)
            {   // save description of the legacy engine
                this.LegacyEng = this.Engine.Description.Clone();
                //
                // remove the engine
                this.Engine.RemoveVehicle(this);
                //
                this.Description.EngineCode = -1;
                this.Engine = null;
            }
        }
        /// <summary>
        /// Removes a transmission from the vehicle.  The vehicle's transmission should only be removed after the vehicle is
        /// hybridized, or converted to pure electric or fuel cell.
        /// </summary>
        public void RemoveTransmission()
        {
            if (this.Transmission != null)
            {   // save description of the legacy transmission
                this.LegacyTrn = this.Transmission.Description.Clone();
                //
                // remove the transmission
                this.Transmission.RemoveVehicle(this);
                //
                this.Description.TransmissionCode = -1;
                this.Transmission = null;
            }
        }

        /// <summary>
        /// Determines the redesign state of the <see cref="Vehicle"/> for the specified model year.
        /// </summary>
        /// <param name="modelYear">The model year for which to check the <see cref="Vehicle"/>'s redesign state.</param>
        /// <param name="atRedesign">When this method returns, specifies whether the <see cref="Vehicle"/> is at
        ///   redesign for the year specified.</param>
        /// <param name="inShadowOfRedesign">When this method returns, specifies whether the <see cref="Vehicle"/> is
        ///   in shadow of redesign for the year specified.</param>
        void GetRedesignState(ModelYear modelYear, out bool atRedesign, out bool inShadowOfRedesign)
        {
            atRedesign = inShadowOfRedesign = false;    // set as false initially
            int year = modelYear.Year;
            int[] redesignYears = this.Description.RedesignYears;
            for (int i = 0, count = redesignYears.Length; i < count; i++)
            {
                if (redesignYears[i] == year) { atRedesign = true; break; }
            }
            if (!atRedesign)
            {
                for (int i = 0, count = redesignYears.Length; i < count; i++)
                {
                    int delta = Math.Abs(redesignYears[i] - year);    // years before or after redesign
                    if (0 < delta && delta < 5) { inShadowOfRedesign = true; break; }
                }
            }
        }
        /// <summary>
        /// Determines the refresh state of the <see cref="Vehicle"/> for the specified model year.
        /// </summary>
        /// <param name="modelYear">The model year for which to check the <see cref="Vehicle"/>'s refresh state.</param>
        /// <param name="atRefresh">When this method returns, specifies whether the <see cref="Vehicle"/> is at refresh
        ///   for the year specified.</param>
        /// <param name="inShadowOfRefresh">When this method returns, specifies whether the <see cref="Vehicle"/> is in
        ///   shadow of refresh for the year specified.</param>
        void GetRefreshState(ModelYear modelYear, out bool atRefresh, out bool inShadowOfRefresh)
        {
            atRefresh = inShadowOfRefresh = false;
            int year = modelYear.Year;
            int[] refreshYears = this.Description.RefreshYears;
            for (int i = 0, count = refreshYears.Length; i < count; i++)
            {
                if (refreshYears[i] == year) { atRefresh = true; break; }
            }
            if (!atRefresh)
            {
                for (int i = 0, count = refreshYears.Length; i < count; i++)
                {
                    int delta = Math.Abs(refreshYears[i] - year);    // years before or after refresh
                    if (0 < delta && delta < 2) { inShadowOfRefresh = true; break; }
                }
            }
        }
        /// <summary>
        /// Determines whether the vehicle is at redesign for the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to check the <see cref="Vehicle"/>'s redesign state.</param>
        /// <returns>true, if the vehicle is at redesign for the specified model year; false, otherwise.</returns>
        public bool IsAtRedesign(ModelYear year)
        {
            bool atRedesign, shadow;
            this.GetRedesignState(year, out atRedesign, out shadow);
            return atRedesign;
        }
        /// <summary>
        /// Determines whether the vehicle is at refresh for the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to check the <see cref="Vehicle"/>'s refresh state.</param>
        /// <returns>true, if the vehicle is at refresh for the specified model year; false, otherwise.</returns>
        public bool IsAtRefresh(ModelYear year)
        {
            bool atRefresh, shadow;
            this.GetRefreshState(year, out atRefresh, out shadow);
            return atRefresh;
        }
        /// <summary>
        /// Returns the latest redesign year for this vehicle, preceding the specified model year.
        /// </summary>
        /// <param name="year">The model year to search for.</param>
        /// <returns>The latest redesign year for the vehicle, preceding the specified model year.</returns>
        public int GetLastRedesign(int year)
        {
            return this.GetLastRefreshOrRedesign(this.Description.RedesignYears, year);
        }
        /// <summary>
        /// Returns the latest refresh year for this vehicle, preceding the specified model year.
        /// </summary>
        /// <param name="year">The model year to search for.</param>
        /// <returns>The latest refresh year for the vehicle, preceding the specified model year.</returns>
        public int GetLastRefresh(int year)
        {
            return this.GetLastRefreshOrRedesign(this.Description.RefreshYears, year);
        }
        int GetLastRefreshOrRedesign(int[] years, int year)
        {
            int last = -1;
            for (int i = 0; i < years.Length; i++)
            {
                if (year > years[i]) { last = years[i]; } else { break; }
            }
            return last;
        }

        /// <summary>
        /// Determines whether the vehicle is a leader of its engine.
        /// </summary>
        /// <returns>true, if the vehicle is an engine leader; false, otherwise.</returns>
        public bool IsEngineLeader()
        {
            return this.IsPlatformLeader(this.Engine);
        }
        /// <summary>
        /// Determines whether the vehicle is a leader of its transmission.
        /// </summary>
        /// <returns>true, if the vehicle is an transmission leader; false, otherwise.</returns>
        public bool IsTransmissionLeader()
        {
            return this.IsPlatformLeader(this.Transmission);
        }
        /// <summary>
        /// Determines whether the vehicle is a leader of its platform.
        /// </summary>
        /// <returns>true, if the vehicle is a platform leader; false, otherwise.</returns>
        public bool IsPlatformLeader()
        {
            return this.IsPlatformLeader(this.Platform);
        }
        bool IsPlatformLeader(Component c)
        {
            return (c != null && c.GetPlatformLeader() == this);
        }

        /// <summary>
        /// Calculates and returns total employment hours for the <see cref="Vehicle"/>, specified as thousand labor-hours, for
        /// the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to compute employment hours.</param>
        /// <returns>The total employment hours for the <see cref="Vehicle"/>, specified as thousand labor-hours.</returns>
        public double CalcLaborHours(ModelYear year)
        {
            return this.CalcLaborHours(year.Index);
        }
        /// <summary>
        /// Calculates and returns total employment hours for the <see cref="Vehicle"/>, specified as thousand labor-hours, for
        /// the specified range of model years.
        /// </summary>
        /// <param name="minYear">The lower bound model year for which to compute employment hours.</param>
        /// <param name="maxYear">The upper bound model year for which to compute employment hours.</param>
        /// <returns>The total employment hours for the <see cref="Vehicle"/>, specified as thousand labor-hours.</returns>
        public double CalcLaborHours(ModelYear minYear, ModelYear maxYear)
        {
            double hours = 0;
            for (int i = minYear.Index; i <= maxYear.Index; i++)
            {
                hours += this.CalcLaborHours(i);
            }
            return hours;
        }
        double CalcLaborHours(int yrIndex)
        {
            return this.Description.EmploymentHours * this.Description.Sales[yrIndex] / 1000;
        }

        #endregion


        #region /*** Properties ***/

        // ----- vehicle characteristics -----
        /// <summary>Gets a description of engineering characteristics for the <see cref="Vehicle"/>.</summary>
        public Vehicle.CDescription Description { get; private set; }
        /// <summary>Gets compliance model data for the <see cref="Vehicle"/>, which is updated during runtime.</summary>
        public Vehicle.CModelData ModelData { get; private set; }

        /// <summary>Gets the <see cref="Manufacturer"/> of this <see cref="Vehicle"/>.</summary>
        public Manufacturer Manufacturer { get; private set; }
        /// <summary>Gets the <see cref="Engine"/> used by this <see cref="Vehicle"/>.</summary>
        public Engine Engine { get; private set; }
        /// <summary>Gets the <see cref="Transmission"/> used by this <see cref="Vehicle"/>.</summary>
        public Transmission Transmission { get; private set; }
        /// <summary>Gets the <see cref="Platform"/> of this <see cref="Vehicle"/>.</summary>
        public Platform Platform { get; private set; }
        /// <summary>Gets the description of the legacy engine that was removed from the <see cref="Vehicle"/> when it was
        ///   converted to an Electric Vehicle or a Fuel Cell Vehicle.</summary>
        public Engine.CDescription LegacyEng { get; private set; }
        /// <summary>Gets the description of the legacy transmission that was removed from the <see cref="Vehicle"/> when it was
        ///   converted to a Strong Hybrid, a Plug-in Hybrid, an Electric Vehicle, or a Fuel Cell Vehicle.</summary>
        public Transmission.CDescription LegacyTrn { get; private set; }

        // ----- regulatory class assignments -----
        /// <summary>Gets or sets the CAFE regulatory class assignment of the <see cref="Vehicle"/>.</summary>
        public RegulatoryClass RegClass { get; set; }
        /// <summary>Gets the technology class assignment associated with the <see cref="Vehicle"/>.</summary>
        public int TechnologyClass { get; private set; }
        /// <summary>Gets the engine technology class assignment associated with the <see cref="Vehicle"/>.</summary>
        public int EngineTechnologyClass { get; private set; }
        /// <summary>Gets the name of the technology class assignment associated with the <see cref="Vehicle"/>.</summary>
        internal string TechClassString { get { return TechnologyClasses.ClassNames[this.TechnologyClass]; } }
        /// <summary>Gets the name of the technology cost class assignment associated with the <see cref="Vehicle"/>.</summary>
        internal string EngineTechClassString { get { return TechnologyClasses.EngineClassNames[this.EngineTechnologyClass]; } }
        /// <summary>Gets the <see cref="SafetyClass"/> assignment of the <see cref="Vehicle"/>.</summary>
        public SafetyClass SafetyClass { get; private set; }
        /// <summary>Gets the classification of the <see cref="Vehicle"/>, whether the <see cref="Vehicle"/> is a passenger car,
        ///   class 1 or class 2a truck (light truck), or class 2b or class 3 truck (heavy light truck).</summary>
        public VehicleClass VehicleClass { get; private set; }
        /// <summary>Gets the style of the <see cref="Vehicle"/>.</summary>
        public VehicleStyle VehicleStyle { get; private set; }

        // ----- type of hybrid/electric -----
        /// <summary>Gets or sets the vehicle's hybrid/electric type.</summary>
        public HEVType HEVType { get; set; }
        /// <summary>Gets the string representation of the vehicle's hybrid/electric type.</summary>
        public string HEVTypeString
        {
            get
            {
                switch (this.HEVType)
                {
                    case HEVType.MildHybrid  : return "MHEV";
                    case HEVType.StrongHybrid: return "SHEV";
                    case HEVType.PlugInHybrid: return "PHEV";
                    case HEVType.PureElectric: return "BEV";
                    case HEVType.FuelCell    : return "FCV";
                    default:                   return "Conventional";
                }
            }
        }

        // ----- vehicle model year info -----
        /// <summary>Gets the lowest <see cref="ModelYear"/> for which the <see cref="Vehicle"/> has production data.</summary>
        public ModelYear MinYear { get; private set; }
        /// <summary>Gets the highest <see cref="ModelYear"/> for which the <see cref="Vehicle"/> has production data.</summary>
        public ModelYear MaxYear { get; private set; }
        /// <summary>Gets the the total number of model years for which the <see cref="Vehicle"/> has production data.</summary>
        public int ProductionYears { get; private set; }

        #endregion

    }
}
