﻿#region << Using Directives >>
using System;
#endregion

namespace Volpe.Cafe.IO
{
    /// <summary>
    /// The exception that is thrown when an error occurs while loading or parsing an input file, or generating a modeling report
    /// or writing an output file.
    /// </summary>
    [Serializable]
    public class IOException : Exception
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="IOException"/> class, using the specified file name that was being
        /// processed when the exception occured.
        /// </summary>
        /// <param name="fileName">The name of the file that was being processed when the exception occured.</param>
        public IOException(string fileName) : this(fileName, string.Empty) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="IOException"/> class, using the specified file name that was being
        /// processed when the exception occured, and the message describing the reason and/or nature of the exception.
        /// </summary>
        /// <param name="fileName">The name of the file that was being processed when the exception occured.</param>
        /// <param name="message">The error message that explains the reason and/or nature of the exception.</param>
        public IOException(string fileName, string message) : this(fileName, message, null) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="IOException"/> class, using the specified file name that was being
        /// processed when the exception occured, the message describing the reason and/or nature of the exception, as well as a
        /// reference to the inner exception that is the cause of this exception.
        /// </summary>
        /// <param name="fileName">The name of the file that was being processed when the exception occured.</param>
        /// <param name="message">The error message that explains the reason and/or nature of the exception.</param>
        /// <param name="innerException">The exception that is the cause of the current exception.</param>
        public IOException(string fileName, string message, Exception innerException) : this(fileName, message, innerException, 0) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="IOException"/> class, using the specified file name that was being
        /// processed when the exception occured, the message describing the reason and/or nature of the exception, as well as a
        /// reference to the inner exception that is the cause of this exception.
        /// (*** This constructor must be called by sub-classes for proper initialization. ***)
        /// </summary>
        /// <param name="fileName">The name of the file that was being processed when the exception occured.</param>
        /// <param name="message">The error message that explains the reason and/or nature of the exception.</param>
        /// <param name="innerException">The exception that is the cause of the current exception.</param>
        /// <param name="readWriteFlags">Indicates type of I/O exception; this field is useful for sub-classes.  A different
        ///   header string will be appended to the front of the generated exception message based on the value of this field.
        ///   The following flags are available:  0 = error processing file, 1 = error reading file, 2 = error writing file.</param>
        protected IOException(string fileName, string message, Exception innerException, int readWriteFlags)
            : base(((readWriteFlags == 2) ? ErrorWritingFile : (readWriteFlags == 1) ? ErrorReadingFile : ErrorProcessingFile) +
            fileName + ((message == null || message.Trim() == string.Empty) ? "" : "\nAdditional information: " + message), innerException)
        {
            this._fileName = fileName;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the name of the file that was being processed when the exception occured.</summary>
        public string FileName { get { return this._fileName; } }

        #endregion


        #region /*** Variables ***/

        // ----- constants -----
        internal const string ErrorProcessingFile = "Error processing file: ";
        internal const string ErrorReadingFile = "Error reading file: ";
        internal const string ErrorWritingFile = "Error writing file: ";

        // ----- runtime variables -----
        string _fileName;

        #endregion

    }
}
