#region << Using Directives >>
using System;
using System.Collections.Generic;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;
using TI = Volpe.Cafe.TechnologyIndexes;
using TC = Volpe.Cafe.TechnologyClasses;
using Volpe.Cafe.HelperClasses;
#endregion

namespace Volpe.Cafe.IO.InputParsers
{
    /// <summary>
    /// Provides an input parser for loading and parsing Excel Technologies files.
    /// </summary>
    sealed class XlTechnologiesParser : XlParser
    {

        #region /*** Ctors ***/

        internal XlTechnologiesParser(string path, string password) : base(path, password, true) { }

        #endregion


        #region /*** Methods ***/

        protected override void ParseInternal()
        {
            // begin parsing ...
            string[] sheets      = this.SheetNames;
            string[] names       = new string[] { "Technologies", "FC Adjustments", "Cost Adjustments" };
            int   [] indexes     = new int   [] { -1, -1, -1 };
            int   [] attrIndexes = new int   [TC.ClassCount];
            int   [] costIndexes = new int   [TC.EngineClassCount];
            for (int i = 0; i < attrIndexes.Length; i++) { attrIndexes[i] = -1; }
            for (int i = 0; i < costIndexes.Length; i++) { costIndexes[i] = -1; }

            // scan each worksheet name, looking for a match
            for (int i = 0; i < sheets.Length; i++)
            {
                string sheet = sheets[i].Trim().ToUpper();
                for (int j = 0; j < names.Length; j++)
                {
                    if (Interaction.StringCompare(sheet, names[j], true)) { indexes[j] = i; break; }
                }
                for (int j = 0; j < TC.ClassCount; j++)
                {
                    if (Interaction.StringCompare(sheet, TC.ClassNames[j], true)) { attrIndexes[j] = i; break; }
                }
                for (int j = 0; j < TC.EngineClassCount; j++)
                {
                    if (Interaction.StringCompare(sheet, TC.EngineClassNames[j], true)) { costIndexes[j] = i; break; }
                }
            }

            // check if all sheets were found
            if (!this.VerifyIndexes("Technologies workbook", attrIndexes, TC.      ClassNames, 0, attrIndexes.Length) ||
                !this.VerifyIndexes("Technologies workbook", costIndexes, TC.EngineClassNames, 0, costIndexes.Length) ||
                !this.VerifyIndexes("Technologies workbook",     indexes,               names, 0,     indexes.Length)) { return; }

            //-------------------------------------------//
            // nothing is missing--proceed to parse data //
            //-------------------------------------------//
            var techList       = new TechnologyInfo[TechnologyIndexes.TechnologyCount];
            var fcAdjFactors   = new Dictionary<ulong, double[]>();
            var costAdjFactors = new Dictionary<ulong, double[]>();
            var referencePower = new Dictionary<ulong, double[]>();
            var fcPwrFactors   = new Dictionary<ulong, double[]>();
            //
            // load the main technology definitions worksheet
            this.LoadTechnologies(indexes[0], techList);
            // load the technology attributes for each tech-class
            for (int i = 0; i < TC.ClassCount; i++)
            {
                this.LoadTechnologyAttributes(attrIndexes[i], i, techList);
            }
            // load the technology costs for each tech-class (non-engine technologies)
            for (int i = 0; i < TC.ClassCount; i++)
            {
                this.LoadTechnologyCosts(attrIndexes[i], i, techList, false);
            }
            // load the technology costs for each eng-tech-class (engine-level technologies)
            for (int i = 0; i < TC.EngineClassCount; i++)
            {
                this.LoadTechnologyCosts(costIndexes[i], i, techList, true);
            }
            // load adjustment factors
            this.LoadTechnologyAdjFactors(indexes[1], fcAdjFactors  , TC.ClassNames      ); // FC
            this.LoadTechnologyAdjFactors(indexes[2], costAdjFactors, TC.EngineClassNames); // Cost
            //
            this._techs = new Technologies(techList, fcAdjFactors, costAdjFactors);
        }
        void LoadTechnologies(int wsIndex, TechnologyInfo[] techList)
        {
            // activate the worksheet
            this.ActivateWorksheet(wsIndex);
            int rows = this.Rows;
            int cols = this.Columns;

            string[] names   = new string[] { "Name", "Phase-in Cap", "PC OCC", "LT OCC", "2b3 OCC", "ZEV Credits" };
            int   [] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }

            //----------------------------------------//
            // perform some verifications of the data //
            //----------------------------------------//
            //
            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column = this.GetString(1, i);
                for (int j = 0; j < names.Length; j++)
                {
                    if (Interaction.StringCompare(column, names[j], true)) { indexes[j] = i; break; }
                }
            }
            //
            // check if all columns were found
            if (!this.VerifyIndexes(this.SheetNames[wsIndex] + " worksheet", indexes, names)) { return; }

            //-------------------------//
            // read-in each technology //
            //-------------------------//
            //
            // skip the first two rows, which are headers
            for (int i = 2; i < rows; i++)
            {   // initialize new technology instance
                string name = this.GetString(i, indexes[0]);
                TechnologyInfo tech = new TechnologyInfo(name);

                // --- load technology at i-th row ---
                if (tech.Index == -1)
                {
                    this.LogError("On the main Technologies worksheet, the technology with name \"" + tech.Name +
                        "\" is undefined.");
                    return;
                }
                // parse global technology attributes
                tech.PhaseInCap     = this.GetDouble(i, indexes[1]);
                tech.OffCycleCredit = new RCDouble(
                                      this.GetDouble(i, indexes[2]),
                                      this.GetDouble(i, indexes[3]),
                                      this.GetDouble(i, indexes[4]));
                tech.ZEVCredits     = this.GetDouble(i, indexes[5]);

                // save technology at a specific index
                techList[tech.Index] = tech;
            }

            // verify that all of the technologies defined in the model were loaded from the input file
            for (int i = 0; i < techList.Length; i++)
            {
                if (techList[i] == null)
                {
                    string techName = TI.GetName(i);
                    this.LogError("On the main Technologies worksheet, missing technology definition for \"" + techName + "\".");
                }
            }
        }
        void LoadTechnologyAttributes(int wsIndex, int techClass, TechnologyInfo[] techList)
        {
            // activate the worksheet
            this.ActivateWorksheet(wsIndex);
            int rows = this.Rows;
            int cols = this.Columns;

            string[] names   = new string[] {
                "Name"              , "Applicable"  , "Year Avail"    , "Year Retired"  , "FC"              ,
                "Secondary FC"      , "Secondary FS", "Electric Range", "Electric Power", "Delta Weight (%)",
                "Delta Weight (lbs)", "Consumer Valuation" };
            int   [] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }

            //----------------------------------------//
            // perform some verifications of the data //
            //----------------------------------------//
            //
            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column = this.GetString(1, i);
                for (int j = 0; j < names.Length; j++)
                {
                    if (Interaction.StringCompare(column, names[j], true)) { indexes[j] = i; break; }
                }
            }
            //
            // check if all columns were found
            if (!this.VerifyIndexes(this.SheetNames[wsIndex] + " worksheet", indexes, names)) { return; }

            //-------------------------//
            // read-in each technology //
            //-------------------------//
            //
            // skip the first two rows, which are headers
            for (int i = 2; i < rows; i++)
            {   // initialize new technology-attributes instance
                TechnologyAttributes ta = new TechnologyAttributes();

                // --- load technology at i-th row ---
                //
                // first, get the abbr, engine size the tech applies to (if applicable), and tech index
                string name  = this.GetString(i, indexes[0]);
                int    index = TI.GetIndex(name);   // get the technology index
                if (index == -1)
                {
                    this.LogError("On the " + this.SheetNames[wsIndex] + " worksheet, the technology with the name \"" +
                        name + "\" is undefined.");
                    return;
                }

                // parse technology attributes
                ta.Applicable        =              this.GetBool  (i, indexes[ 1]);
                ta.YearAvailable     =              this.GetInt32 (i, indexes[ 2]);
                ta.YearRetired       =              this.GetInt32 (i, indexes[ 3]);
                ta.FC                =              this.GetDouble(i, indexes[ 4]);
                ta.FCSecondary       =              this.GetDouble(i, indexes[ 5]);
                ta.FSSecondary       =              this.GetDouble(i, indexes[ 6]);
                ta.ElectricRange     =              this.GetDouble(i, indexes[ 7]);
                ta.ElectricPower     =              this.GetDouble(i, indexes[ 8]);
                ta.DeltaWeightPCT    =              this.GetDouble(i, indexes[ 9]);
                ta.DeltaWeightLBS    =              this.GetDouble(i, indexes[10]);
                ta.ConsumerValuation =              this.GetDouble(i, indexes[11]);

                // add attributes to the technology specified by index
                techList[index].SetAttributes(techClass, ta);
            }
        }
        void LoadTechnologyCosts(int wsIndex, int techClass, TechnologyInfo[] techList, bool isEngClass)
        {
            // activate the worksheet
            this.ActivateWorksheet(wsIndex);
            int rows = this.Rows;
            int cols = this.Columns;

            string[] namesMain   = new string[] { "Name" };
            int   [] indexesMain = new int[namesMain.Length];
            for (int i = 0; i < indexesMain.Length; i++) { indexesMain[i] = -1; }
            //
            string[] namesCost   = new string[] { "Cost Table", "Maint. Table", "Repair Table", "Stranded Capital Table" };
            int   [] indexesCost = new int[namesCost.Length];
            for (int i = 0; i < indexesCost.Length; i++) { indexesCost[i] = -1; }
            int costTableCount = 0, maintTableCount = 0, repairTableCount = 0, strandedCapitalCount = 0; // total number cols in various "cost" tables

            //----------------------------------------//
            // perform some verifications of the data //
            //----------------------------------------//
            //
            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column = this.GetString(1, i);
                if (indexesCost[0] == -1 && Interaction.StringStartsWith(column, "C2", true) &&
                                            Interaction.StringCompare(this.GetString(0, i), namesCost[0], false))
                {   // look for cost table columns
                    indexesCost[0] = i; // first learning cost table column found
                    while (Interaction.StringStartsWith(column, "C2", true))
                    {   // count the number of adjacent learning cost table columns
                        costTableCount++; i++;
                        if (i == cols) { break; }
                        column = this.GetString(1, i);
                    }
                }
                if (indexesCost[1] == -1 && Interaction.StringStartsWith(column, "M2", true) &&
                                            Interaction.StringCompare(this.GetString(0, i), namesCost[1], false))
                {   // look for maintanance cost table columns
                    indexesCost[1] = i; // first maintanance cost table column found
                    while (Interaction.StringStartsWith(column, "M2", true))
                    {   // count the number of adjacent maintanance cost table columns
                        maintTableCount++; i++;
                        if (i == cols) { break; }
                        column = this.GetString(1, i);
                    }
                }
                if (indexesCost[2] == -1 && Interaction.StringStartsWith(column, "R2", true) &&
                                            Interaction.StringCompare(this.GetString(0, i), namesCost[2], false))
                {   // look for repair cost table columns
                    indexesCost[2] = i;    // first repair cost table column found
                    while (Interaction.StringStartsWith(column, "R2", true))
                    {   // count the number of adjacent repair cost table columns
                        repairTableCount++; i++;
                        if (i == cols) { break; }
                        column = this.GetString(1, i);
                    }
                }
                if (indexesCost[3] == -1 && Interaction.StringStartsWith(column, "SC-", false) &&
                                            Interaction.StringCompare(this.GetString(0, i), namesCost[3], false))
                {   // look for stranded capital table columns
                    indexesCost[3] = i;    // first stranded capital table column found
                    while (Interaction.StringStartsWith(column, "SC-", false))
                    {   // count the number of adjacent stranded capital table columns
                        strandedCapitalCount++; i++;
                        if (i == cols) { break; }
                        column = this.GetString(1, i);
                    }
                }
                // look for all other columns
                for (int j = 0; j < namesMain.Length; j++)
                {
                    if (Interaction.StringCompare(column, namesMain[j], true)) { indexesMain[j] = i; break; }
                }
            }
            //
            // check if all columns were found
            if (!this.VerifyIndexes(this.SheetNames[wsIndex] + " worksheet", indexesCost, namesCost)) { return; }
            //
            // compute starting cost/maintanance/repair table year
            int costTableMinYear = Interaction.GetInt32(
                Interaction.StringRemoveWhitespacePunctuationAndSpecialChars(this.GetString(1, indexesCost[0])).Replace("C", ""));
            if (costTableMinYear < ModelYear.MinYear) { this.LogError("On the '" + this.SheetNames[wsIndex] + "' worksheet, the Cost Table section is not valid."); }
            //
            int maintTableMinYear = Interaction.GetInt32(
                Interaction.StringRemoveWhitespacePunctuationAndSpecialChars(this.GetString(1, indexesCost[1])).Replace("M", ""));
            if (maintTableMinYear < ModelYear.MinYear) { this.LogError("On the '" + this.SheetNames[wsIndex] + "' worksheet, the Maintance Table section is not valid."); }
            //
            int repairTableMinYear = Interaction.GetInt32(
                Interaction.StringRemoveWhitespacePunctuationAndSpecialChars(this.GetString(1, indexesCost[2])).Replace("R", ""));
            if (repairTableMinYear < ModelYear.MinYear) { this.LogError("On the '" + this.SheetNames[wsIndex] + "' worksheet, the Repair Table section is not valid."); }
            //
            // at this point, if parsing errors were encountered, stop trying to load file and return ...
            if (this.HasErrors) { return; }

            //-------------------------//
            // read-in each technology //
            //-------------------------//
            //
            // skip the first two rows, which are headers
            for (int i = 2; i < rows; i++)
            {   // initialize new technology-attributes instance
                TechnologyCosts tc = new TechnologyCosts();

                // --- load technology at i-th row ---
                //
                // first, get the abbr, engine size the tech applies to (if applicable), and tech index
                string name  = this.GetString(i, indexesMain[0]);
                int    index = TI.GetIndex(name);   // get the technology index
                if (index == -1)
                {
                    this.LogError("On the " + this.SheetNames[wsIndex] + " worksheet, the technology with the name \"" +
                        name + "\" is undefined.");
                    return;
                }

                // costs are defined differently based on tech type
                //  - engine technologies are specified by engine class
                //  - non-engine technologies are specified by tech class
                // skip techs if tech level and class do not match
                if (TI.IsEngineLevel(index) != isEngClass) { continue; }

                // parse learning cost table
                int costTableStart = costTableMinYear - ModelYear.MinYear;
                tc.CostTable = new double[costTableStart + costTableCount];
                for (int j = 0; j < costTableCount; j++)
                {
                    tc.CostTable[costTableStart + j] = this.GetDouble(i, indexesCost[0] + j);
                }
                // parse maintanance cost table
                int maintTableStart = maintTableMinYear - ModelYear.MinYear;
                tc.MaintenanceCostTable = new double[maintTableStart + maintTableCount];
                for (int j = 0; j < maintTableCount; j++)
                {
                    tc.MaintenanceCostTable[maintTableStart + j] = this.GetDouble(i, indexesCost[1] + j);
                }
                // parse repair cost table
                int repairTableStart = repairTableMinYear - ModelYear.MinYear;
                tc.RepairCostTable = new double[repairTableStart + repairTableCount];
                for (int j = 0; j < repairTableCount; j++)
                {
                    tc.RepairCostTable[repairTableStart + j] = this.GetDouble(i, indexesCost[2] + j);
                }
                // parse stranded capital table
                tc.StrandedCapital = new double[strandedCapitalCount];
                for (int j = 0; j < strandedCapitalCount; j++)
                {
                    tc.StrandedCapital[j] = this.GetDouble(i, indexesCost[3] + j);
                }

                // add attributes to the technology at a specific index
                techList[index].SetCosts(techClass, tc);
            }
        }
        void LoadTechnologyAdjFactors(int wsIndex, Dictionary<ulong, double[]> adjFactors, string[] colNames)
        {
            // activate the worksheet
            this.ActivateWorksheet(wsIndex);
            int rows = this.Rows;
            int cols = this.Columns;

            string[] names = new string[colNames.Length + 1];
            names[0] = "Technologies";
            Array.Copy(colNames, 0, names, 1, colNames.Length);
            int[] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }

            //----------------------------------------//
            // perform some verifications of the data //
            //----------------------------------------//
            //
            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column = this.GetString(1, i);
                for (int j = 0; j < names.Length; j++)
                {
                    if (Interaction.StringCompare(column, names[j], true)) { indexes[j] = i; break; }
                }
            }
            //
            // check if all columns were found
            if (!this.VerifyIndexes(this.SheetNames[wsIndex] + " worksheet", indexes, names)) { return; }

            //-------------------------//
            // read-in each adj-factor //
            //-------------------------//
            //
            // skip the first two rows, which are headers
            for (int i = 2; i < rows; i++)
            {   // get string of techs -- if empty, ignore row
                string techString = this.GetString(i, indexes[0]).Trim();
                if (techString == string.Empty) { continue; }

                // get tech list and compute key
                string[] techs = techString.Split(';', ',');
                ulong key = 0;
                for (int j = 0; j < techs.Length; j++)
                {
                    string techName = techs[j].Trim();
                    if (techName != string.Empty)
                    {
                        int techIndex = TI.GetIndex(techName);
                        if (techIndex == -1)
                        {
                            this.LogError("On the " + this.SheetNames[wsIndex] + " worksheet, the technology with the name \"" +
                                techName + "\" is undefined.");
                            return;
                        }
                        key |= KeyHelper.Keys[techIndex];
                    }
                }

                // build array of values
                double[] values = new double[colNames.Length];
                for (int j = 0; j < values.Length; j++)
                {
                    values[j] = this.GetDouble(i, indexes[1 + j]);
                }

                // add key/adj-factors to dataset
                adjFactors.Add(key, values);
            }
        }

        double[] GetDblArr(int row, int column)
        {
            string values = this.GetString(row, column);
            if (values == null || values == string.Empty) { return new double[0]; }
            //
            string[] s = values.Split(',', ';');
            double[] d = new double[s.Length];
            for (int i = 0; i < s.Length; i++)
            {
                if (!Interaction.IsNumeric(s[i])) { return null; }
                d[i] = double.Parse(s[i]);
            }
            return d;
        }

        #endregion


        #region /*** Variables ***/

        internal Technologies _techs;

        #endregion

    }
}
