﻿#region << Using Directives >>
using System;
using System.IO;
using System.Text;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
#endregion

namespace Volpe.Cafe.IO.Reporting.CSV
{
    /// <summary>
    /// Provides a base class for generating reports where fields are delimited by a specific character.
    /// </summary>
    [Serializable]
    public abstract class DelimitedReportingBase
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="DelimitedReportingBase"/> class.
        /// </summary>
        /// <param name="path">The output path of the report.</param>
        /// <param name="append">Determines whether data is to be appended to the file.  If the file exists and append is false,
        ///   the file is overwritten. If the file exists and append is true, the data is appended to the file. Otherwise, a new
        ///   file is created.</param>
        /// <param name="delimiter">A character used to delimit the fields written to the report.</param>
        public DelimitedReportingBase(string path, bool append, char delimiter)
        {
            this._disposed        = false;
            this._path            = path;
            this._stream          = new StreamWriter(path, append);
            this._newLine         = this._stream.NewLine.ToCharArray();
            this._delimiter       = delimiter;
            this._delimiterString = delimiter.ToString();
            this._buffer          = new StringBuilder(4096);
            //
            if (!append) { this._stream.Write(this.GetHeader()); }
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Returns the header used for the report. The report's header is written as the first line immediately after the stream
        /// is opened.
        /// </summary>
        protected abstract string GetHeader();

        /// <summary>
        /// Parses the data into the report.
        /// </summary>
        /// <param name="scen">The scenario to report on.</param>
        /// <param name="year">The model year to report on.</param>
        /// <param name="compliance">The compliance model that invoked the report generator.</param>
        public void ParseData(Scenario scen, ModelYear year, ICompliance compliance)
        {
            try   { this.ParseData_Internal(scen, year, compliance); }
            catch { this.Close(); }
        }
        /// <summary>
        /// Parses the data into the report.
        /// </summary>
        /// <param name="scen">The scenario to report on.</param>
        /// <param name="year">The model year to report on.</param>
        /// <param name="compliance">The compliance model that invoked the report generator.</param>
        protected abstract void ParseData_Internal(Scenario scen, ModelYear year, ICompliance compliance);

        /// <summary>
        /// Writes the text representation of an object to the report by calling <see cref="object.ToString"/> on that object.
        /// </summary>
        /// <param name="value">The object to write to the report. If value is null, only the <see cref="Delimiter"/> character
        ///   is written.</param>
        protected void Write(object value)
        {
            if (value == null) { this.Write(string.Empty); }
            else               { this.Write(value.ToString()); }
        }
        /// <summary>
        /// Writes the text representation of a <see cref="Boolean"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Boolean"/> value to write to the report.</param>
        protected void Write(bool value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of a <see cref="Decimal"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Decimal"/> value to write to the report.</param>
        protected void Write(decimal value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of a <see cref="Double"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Double"/> value to write to the report.</param>
        protected void Write(double value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of a <see cref="Single"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Single"/> value to write to the report.</param>
        protected void Write(float value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of an <see cref="Int64"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Int64"/> value to write to the report.</param>
        protected void Write(long value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of an <see cref="Int32"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Int32"/> value to write to the report.</param>
        protected void Write(int value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of an <see cref="Int16"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Int16"/> value to write to the report.</param>
        protected void Write(short value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of a <see cref="Byte"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="Byte"/> value to write to the report.</param>
        protected void Write(byte value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of a <see cref="SByte"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="SByte"/> value to write to the report.</param>
        protected void Write(sbyte value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of an <see cref="UInt64"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="UInt64"/> value to write to the report.</param>
        protected void Write(ulong value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of an <see cref="UInt32"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="UInt32"/> value to write to the report.</param>
        protected void Write(uint value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes the text representation of an <see cref="UInt16"/> value to the report.
        /// </summary>
        /// <param name="value">The <see cref="UInt16"/> value to write to the report.</param>
        protected void Write(ushort value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes a character to the report.
        /// </summary>
        /// <param name="value">The character to write to the report.</param>
        protected void Write(char value)
        {
            this.Write(value.ToString());
        }
        /// <summary>
        /// Writes a string to the report.
        /// </summary>
        /// <param name="value">The string to write to the report. If value is null, only the <see cref="Delimiter"/> character
        ///   is written.</param>
        protected void Write(string value)
        {
            if (value != null)
            {
                if (value.Contains("\""))
                {
                    value = "\"" + value.Replace("\"", "\"\"") + "\"";
                }
                else if (value.Contains(this._delimiterString) || value.Contains("\r") || value.Contains("\n"))
                {
                    value = "\"" + value + "\"";
                }
            }
            // write to buffer until line terminator is supplied
            this._buffer.Append(value);
            this._buffer.Append(this._delimiter);
        }

        /// <summary>
        /// Writes the specified count of empty "cells" to the report. Each cell will be delimited with the specified
        /// <see cref="Delimiter"/> character.
        /// </summary>
        /// <param name="count">The number of empty cells to write to the report.</param>
        protected void WriteEmpty(int count)
        {
            for (int i = 0; i < count; i++)
            {
                this._buffer.Append(this._delimiter);
            }
        }

        /// <summary>
        /// Finalizes the current row in the report and prepares a new one for writing.
        /// </summary>
        protected void NewRow()
        {
            if (this._buffer.Length != 0)
            {
                this._stream.WriteLine();
                this._stream.Write(this._buffer.ToString(0, this._buffer.Length - 1));
                this._buffer.Length = 0;
            }
        }

        /// <summary>
        /// Closes the report.
        /// </summary>
        public void Close()
        {
            this.NewRow();
            this._stream.Close();
            this._disposed = true;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets whether the report has been closed and its resources released.</summary>
        public bool Disposed { get { return this._disposed; } }

        /// <summary>Gets the location of the report.</summary>
        public string Path { get { return this._path; } }
        /// <summary>Gets the filename of the report.</summary>
        public string FileName { get { return System.IO.Path.GetFileName(this._path); } }
        /// <summary>Gets the friendly name of the report.</summary>
        public abstract string ReportName { get; }

        /// <summary>Gets the character used to delimit the fields written to the report.</summary>
        public char Delimiter { get { return this._delimiter; } }

        #endregion


        #region /*** Variables ***/

        bool          _disposed;     // set to true after the stream has been closed
        string        _path;
        StreamWriter  _stream;
        char[]        _newLine;
        char          _delimiter;
        string        _delimiterString;
        StringBuilder _buffer;

        #endregion

    }
}
