#region << Using Directives >>
using System;
using System.Windows.Forms;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.IO.Reporting.CSV;
using Volpe.Cafe.IO.Reporting.XL;
using Volpe.Cafe.Settings;
using Volpe.Cafe.UI;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides a standard interface on which to base each compliance model implementation.
    /// </summary>
    [ModelDescription("ICompliance", "Provides a standard interface on which to base each compliance model implementation.", 1.0F)]
    public interface ICompliance
    {

        #region /*** Events ***/

        /// <summary>Occurs whenever the compliance model displays a prompt, which requires user action.</summary>
        event PromptEventHandler Prompt;

        /// <summary>Occurs after the compliance model has started.</summary>
        event ModelingEventHandler ModelingStarted;
        /// <summary>Occurs after the compliance model has stopped, but did not complete successfully.  This typically indicates
        ///   that the user has aborted the modeling process or an exception in the code has occured.</summary>
        event ModelingEventHandler ModelingStopped;
        /// <summary>Occurs after the compliance model has completed successfully.</summary>
        event ModelingEventHandler ModelingCompleted;
        /// <summary>Occurs after the compliance model has started, stopped, or completed.  This event is generated along side
        ///   <see cref="ModelingStarted"/>, <see cref="ModelingStopped"/>, and <see cref="ModelingCompleted"/> events.  The
        ///   <see cref="ModelingChanged"/> event also occurs when a model start or stop has been requested by the user.</summary>
        event ModelingEventHandler ModelingChanged;

        /// <summary>Occurs before the processing of a single scenario (along with any pre-processing) has started.</summary>
        event ModelingProgressEventHandler ScenarioStarted;
        /// <summary>Occurs after the processing of a single scenario (along with any post-processing) has completed.</summary>
        event ModelingProgressEventHandler ScenarioCompleted;
        /// <summary>Occurs before the processing of a single model year (along with any pre-processing) has started.</summary>
        event ModelingProgressEventHandler ModelYearStarted;
        /// <summary>Occurs after the processing of a single model year (along with any post-processing) has completed.</summary>
        event ModelingProgressEventHandler ModelYearCompleted;
        /// <summary>Occurs before the processing of a single manufacturer (along with any pre-processing) has started.</summary>
        event ModelingProgressEventHandler ManufacturerStarted;
        /// <summary>Occurs after the processing of a single manufacturer (along with any post-processing) has completed.</summary>
        event ModelingProgressEventHandler ManufacturerCompleted;

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Creates a new instance with the same underlying type as this <see cref="ICompliance"/>.
        /// </summary>
        /// <returns>A new <see cref="ICompliance"/> object with the same underlying type as the current instance.</returns>
        ICompliance CreateNew();

        /// <summary>
        /// Starts compliance modeling in a new thread, using the specified modeling data and settings.
        /// </summary>
        /// <param name="data">The modeling data to use for compliance modeling.</param>
        /// <param name="settings">The modeling settings to use for compliance modeling.</param>
        /// <remarks>
        /// A compliance model should always be started in a separate thread.
        /// </remarks>
        void Start(Industry data, ModelingSettings settings);

        /// <summary>
        /// Aborts all threads spawned by this <see cref="ICompliance"/> instance, and terminates compliance modeling immediately.
        /// </summary>
        void Abort();
        /// <summary>
        /// Aborts all threads spawned by this <see cref="ICompliance"/> instance, and terminates compliance modeling, optionally
        /// waiting for the pending operations to complete.
        /// </summary>
        /// <param name="abortWhenReady">If true, specifies that the pending operations will be completed prior to aborting the
        ///   modeling process; otherwise, modeling will be terminated immediately.</param>
        void Abort(bool abortWhenReady);

        /// <summary>
        /// Returns an array of "working" or "final" compliance modeling data for the specified scenario.
        /// </summary>
        /// <param name="scen">The scenario for which to obtain the array of compliance modeling data.</param>
        /// <returns>An array of compliance modeling data for the specified scenario.</returns>
        /// <remarks>
        /// The array returned by this method should be initialized to a length capable of storing the greatest model year index.
        /// If, however, the compliance model has not yet been run, the return value may be null.  Once modeling begins, if the
        /// compliance model has not started processing the specific scenario, the return value should be an array where each
        /// element is null.  If the compliance model has not processed a specific model year of the scenario, the return value
        /// should be an array where only the unprocessed model years are null.
        /// </remarks>
        Industry[] GetData(Scenario scen);
        /// <summary>
        /// Returns the "working" or "final" compliance modeling data for the specified scenario and model year.
        /// </summary>
        /// <param name="scen">The scenario for which to obtain the compliance modeling data.</param>
        /// <param name="year">The model year for which to obtain the compliance modeling data.</param>
        /// <returns>The compliance modeling data for the specified scenario and model year.</returns>
        /// <remarks>
        /// If the compliance model has not processed a specific scenario and/or model year, the return value should be null.
        /// </remarks>
        Industry GetData(Scenario scen, ModelYear year);

        /// <summary>
        /// Returns the effects modeling data for the entire industry for the specified scenario and model year.
        /// </summary>
        /// <param name="scen">The scenario for which to obtain the effects modeling data.</param>
        /// <param name="year">The model year for which to obtain the effects modeling data.</param>
        /// <returns>The effects modeling data for the entire industry for the specified scenario and model year.</returns>
        EffectsData GetEffectsData(Scenario scen, int year);
        ///// <summary>
        ///// Returns the effects modeling data for the specified scenario, model year, and manufacturer.
        ///// </summary>
        ///// <param name="scen">The scenario for which to obtain the effects modeling data.</param>
        ///// <param name="year">The model year for which to obtain the effects modeling data.</param>
        ///// <param name="mfrCode">The code of the manufacturer for which to obtain the effects modeling data.</param>
        ///// <returns>The effects modeling data for the specified scenario, model year, and manufacturer.</returns>
        //EffectsData GetEffectsData(Scenario scen, int year, int mfrCode);

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets whether the <see cref="ICompliance"/> instance is valid to begin modeling.</summary>
        /// <remarks>Generally, the <see cref="ICompliance"/> will be invalid if its <see cref="Data"/> or <see cref="Settings"/>
        ///   are missing or not valid.</remarks>
        bool IsValid { get; }

        /// <summary>Gets the default model inputs that are required for modeling.</summary>
        RequiredModelInputs RequiredModelInputs { get; }

        /// <summary>Gets the product plan modeling data associated with this <see cref="ICompliance"/> instance, which is the
        ///   unaltered input data being used for compliance modeling.</summary>
        Industry Data { get; }

        /// <summary>Gets an array of "working" or "final" modeling data objects for the baseline scenario, by model year.</summary>
        Industry[] BaselineData { get; }
        /// <summary>Gets an array of "working" or "final" modeling data objects for the current scenario, by model year.  This
        ///   value should be reset before starting a new scenario.  At the end of the scenario, this value should contain the
        ///   final state of the industry for each model year.</summary>
        Industry[] ScenarioData { get; }

        /// <summary>Gets the modeling settings associated with this <see cref="ICompliance"/> instance, which are the settings
        ///   being used for compliance modeling.</summary>
        ModelingSettings Settings { get; }

        /// <summary>Gets the <see cref="LogWriter"/> to use for writing log files generated during compliance modeling.</summary>
        LogWriter LogWriter { get; }
        /// <summary>Gets the <see cref="XlReportGenerator"/> to use for generating Excel reports for the compliance model.</summary>
        XlReportGenerator XlReportGenerator { get; }
        /// <summary>Gets the <see cref="CsvReportGenerator"/> to use for generating CSV reports for the compliance model.</summary>
        CsvReportGenerator CsvReportGenerator { get; }

        /// <summary>Gets the current state of the modeling process.</summary>
        ModelingState State { get; }
        /// <summary>Gets the state of the modeling process, whether the compliance model is currently running.</summary>
        bool Running { get; }
        /// <summary>Gets the state of the modeling process, whether the compliance model has been stopped.</summary>
        bool Stopped { get; }
        /// <summary>Gets the state of the modeling process, whether the compliance model has completed.</summary>
        bool Completed { get; }
        /// <summary>Gets the state of the modeling process, whether the compliance model has stopped or completed.</summary>
        bool StoppedOrCompleted { get; }

        /// <summary>Gets the current modeling progress, or null if compliance modeling has not started.</summary>
        IModelingProgress Progress { get; }
        /// <summary>Gets the timestamp of when the modeling process was started (or restarted).</summary>
        DateTime StartTime { get; }
        /// <summary>Gets the timestamp of when the modeling process has stopped or completed.</summary>
        DateTime StopTime { get; }
        /// <summary>Gets the amount of time, in milliseconds, that the compliance model has been running thus far or since it
        ///   was last restarted.</summary>
        long Runtime { get; }

        /// <summary>Gets an array of <see cref="ModelYear"/>s avaiable for compliance modeling. The 0-th index in the array
        ///   represents the first <see cref="ModelYear"/> available.</summary>
        ModelYear[] ModelYears { get; }

        /// <summary>Gets the minimum year avaiable for compliance modeling.</summary>
        int MinYear { get; }
        /// <summary>Gets the maximum year avaiable for compliance modeling.</summary>
        int MaxYear { get; }

        /// <summary>Gets the minimum year avaiable for effects modeling.</summary>
        int MinEffectsYear { get; }
        /// <summary>Gets the maximum year avaiable for effects modeling.</summary>
        int MaxEffectsYear { get; }

        /// <summary>Gets the last error message generated by the compliance model.</summary>
        string LastErrorMessage { get; }

        /// <summary>Gets or sets the additional user specified notes associated with the compliance model.</summary>
        string UserNotes { get; set; }
        /// <summary>Gets or sets the user specified keywords (or tags) associated with the compliance model.</summary>
        string UserKeywords { get; set; }

        #endregion

    }
}
