using System;
using System.Collections;
using System.Collections.Generic;
using Volpe.Cafe;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides methods to assist the compliance modeling process in performing market simulation calculations, such as
    /// assigning the regulatory costs to vehicles and recalculating vehicle sales.
    /// </summary>
    [Serializable]
    public class MarketSimulation
    {

        #region /*** Methods ***/

        /// <summary>
        /// Allocates the regulatory costs for all vehicles within the industry for the given model year.
        /// </summary>
        /// <param name="data">The modeling data used for modeling.</param>
        /// <param name="settings">The modeling settings used for modeling.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year for which to calculate the regulatory costs.</param>
        /// <remarks>
        /// This method implements the cost allocation model for a single model year, by using the inputs that specify the cost
        /// allocation strategy assumed for each manufacturer.
        /// </remarks>
        public static void AllocateCosts(Industry data, ModelingSettings settings, Scenario scen, ModelYear year)
        {
            List<Manufacturer> mfrs = data.Manufacturers;
            for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
            {
                Manufacturer.CModelData mmd = mfrs[i].ModelData;
                List<Vehicle> mfrVehs = mfrs[i].Vehicles;
                int       mfrVehCount = mfrVehs.Count;

                double mfrSales = 0, mfrSalesRevenue = 0, mfrTechCost = 0;
                double mfrFines = mmd.Fines.Total;
                int mfrStrategy = 0;

                // sum-up mfr totals of all valid vehicles
                for (int j = 0; j < mfrVehCount; j++)
                {
                    Vehicle.CDescription vd = mfrVehs[j].Description;
                    mfrSales        += vd.Sales[year.Index];
                    mfrSalesRevenue += vd.Sales[year.Index] * vd.Msrp;
                    mfrTechCost     += vd.Sales[year.Index] * mfrVehs[j].ModelData.TechCost;
                }

                // calculate reg-costs based on mfr's cost allocation strategy
                if (mfrStrategy == 3)
                {   // distribute both technology costs and fines evenly
                    for (int j = 0; j < mfrVehCount; j++)
                    {
                        mfrVehs[j].ModelData.RegCost = (mfrTechCost + mfrFines) / mfrSales;
                    }
                } // end if (strategy 3)
                else if (mfrStrategy == 1 || (mfrTechCost == 0 && mfrFines > 0))
                {   // distribute both technology costs and fines based on the share of aggregate sales revenue
                    double pctIncrease = (mfrTechCost + mfrFines) / mfrSalesRevenue;
                    for (int j = 0; j < mfrVehCount; j++)
                    {
                        mfrVehs[j].ModelData.RegCost = pctIncrease * mfrVehs[j].Description.Msrp;
                    }
                } // end else if (strategy 1)
                else
                {   // the else clause is normally executed when the strategy is 0
                    //
                    // first, allocate technology costs on an as-incurred basis
                    for (int j = 0; j < mfrVehCount; j++)
                    {
                        Vehicle.CModelData vmd = mfrVehs[j].ModelData;
                        vmd.RegCost = vmd.TechCost;
                    }
                    // then, allocate any fines
                    if (mfrFines > 0)
                    {
                        double[] vehPseudoFines = new double[mfrVehCount];
                        double   mfrPseudoFines = 0;

                        // get vehicle and manufacturer pseudo-fines
                        for (int j = 0; j < mfrVehCount; j++)
                        {
                            vehPseudoFines[j] = GetPseudoFine(mfrs[i], mfrVehs[j], scen, year, settings);
                            mfrPseudoFines   += vehPseudoFines[j];
                        }
                        // allocate fines based on share of total pseudo-fines
                        for (int j = 0; j < mfrVehCount; j++)
                        {
                            double vehSales = mfrVehs[j].Description.Sales[year.Index];
                            if (vehPseudoFines[j] != 0 && vehSales != 0)
                            {
                                mfrVehs[j].ModelData.RegCost += (mfrFines * (vehPseudoFines[j] / mfrPseudoFines)) / vehSales;
                            }
                        }
                    } // end if
                } // end else (strategy 0, or other)

                // aggregate reg-cost to the manufacturer level
                mmd.RegCost.Clear();
                for (int j = 0; j < mfrVehCount; j++)
                {
                    Vehicle.CDescription vd = mfrVehs[j].Description;
                    mmd.RegCost[mfrVehs[j].RegClass] += mfrVehs[j].ModelData.RegCost * vd.Sales[year.Index];
                }
            } // next i (next mfr)
        }
        /// <summary>
        /// Calculates and returns the pseudo fine for the specified vehicle.
        /// </summary>
        /// <param name="mfr">The manufacturer of the vehicle, whose pseudo fine to calculate.</param>
        /// <param name="veh">The vehicle whose pseudo fine to calculate.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The calculated pseudo fine for the specified vehicle.</returns>
        static double GetPseudoFine(Manufacturer mfr, Vehicle veh, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            double vehSales = veh.Description.Sales[year.Index];
            double vehFE    = Standards.GetAverageFuelEconomy(veh, scen, year, settings, true, true, true);
            double standard = mfr.ModelData.Standard[veh.RegClass];
            return Math.Max(0, scen.ScenInfo[veh.RegClass].FineRate[year.Index] * vehSales * (standard - vehFE));
        }

        #endregion

    }
}
