using System;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Data;

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Provides attributes relating to a single vehicle technology.
    /// </summary>
    [Serializable]
    public sealed class TechnologyAttributes
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="TechnologyAttributes"/> class.
        /// </summary>
        internal TechnologyAttributes() { }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="TechnologyAttributes"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="TechnologyAttributes"/>.</returns>
        internal TechnologyAttributes Clone()
        {
            TechnologyAttributes ta = new TechnologyAttributes();
            //
            ta.Applicable        = this.Applicable;
            ta.YearAvailable     = this.YearAvailable;
            ta.YearRetired       = this.YearRetired;
            //
            ta.FC                = this.FC;
            ta.FCSecondary       = this.FCSecondary;
            ta.FSSecondary       = this.FSSecondary;
            //
            ta.ElectricRange     = this.ElectricRange;
            ta.ElectricPower     = this.ElectricPower;
            ta.DeltaWeightPCT    = this.DeltaWeightPCT;
            ta.DeltaWeightLBS    = this.DeltaWeightLBS;
            ta.ConsumerValuation = this.ConsumerValuation;
            //
            return ta;
        }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies whether the current technology can be used with a specific <see cref="TechnologyClass"/>.</summary>
        public bool Applicable;
        /// <summary>Specifies the model year when the technology becomes available for applicability.</summary>
        public int YearAvailable;
        /// <summary>Specifies the last model year that the technology is available for applicability.</summary>
        public int YearRetired;

        /// <summary>Specifies the fuel consumption improvement estimate of the current technology.
        ///   For dual fuel vehicles (such as PHEVs), this value speficies the fuel consumption improvement when operating on
        ///   primary fuel (such as gasoline).</summary>
        public double FC;
        /// <summary>Specifies the fuel consumption improvement estimate of the current technology when operating on secondary
        ///   fuel. This value applies only when technologies that convert a vehicle to dual-fuel are used (such as PHEV).</summary>
        public double FCSecondary;
        /// <summary>Specifies the amount of miles the vehicle is expected to run on the secondary fuel after applying the
        ///   current technology, only if the current technology converts that vehicle to dual fuel.</summary>
        public double FSSecondary;

        /// <summary>Specifies what the range, in miles, of an electric vehicle would be when operating on a battery, as a
        ///   result of applying the current technology.  This field should only be specified for PHEV and EV technologies.</summary>
        public double ElectricRange;
        /// <summary>Specifies what the moter power, in HP-equivalent, of an electric vehicle would be when operating on a
        ///   battery, as a result of applying the current technology.  This field should only be specified for PHEV and EV
        ///   technologies.</summary>
        public double ElectricPower;
        /// <summary>Specifies the percentage by which the vehicle's weight changes as a result of applying the current
        ///   technology.</summary>
        public double DeltaWeightPCT;
        /// <summary>Specifies the amount of pounds by which the vehicle's weight changes as a result of applying the current
        ///   technology.</summary>
        public double DeltaWeightLBS;
        /// <summary>Specifies what the estimated changes in consumer valuation of a vehicle would be after applying the current
        ///   technology.</summary>
        public double ConsumerValuation;

        #endregion

    }
}
