﻿#region << Using Directives >>
using System;
using System.Collections.Generic;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Generic;
//
using VehClass = Volpe.Cafe.VehicleClass;
using RegClass = Volpe.Cafe.RegulatoryClass;
#endregion

namespace Volpe.Cafe.Data
{
    #region /*** Helper Classes ***/

    /// <summary>
    /// Provides various vehicle characteristic data, average fuel economy, on-road fleet, and vehicle miles traveled, which
    /// serve as the basis for the modeling effects calculations.
    /// </summary>
    [Serializable]
    public sealed class GeneralEffects : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        GeneralEffects(GeneralEffects prototype)
        {
            this.Sales             = prototype.Sales            .Clone();   // sales and vehicle metrics
            this.CurbWeight        = prototype.CurbWeight       .Clone();
            this.Footprint         = prototype.Footprint        .Clone();
            this.WorkFactor        = prototype.WorkFactor       .Clone();
            this.RatedFuelEconomy  = prototype.RatedFuelEconomy .Clone();   // fuel economy
            this.OnRoadFuelEconomy = prototype.OnRoadFuelEconomy.Clone();
            this.FuelShare         = prototype.FuelShare        .Clone();
            this.Fleet             = prototype.Fleet            .Clone();   // fleet/vmt
            this.VMT               = prototype.VMT              .Clone();
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="GeneralEffects"/> class.
        /// </summary>
        public GeneralEffects()
        {
            this.Sales             = new EffectsValue();    // sales and vehicle metrics
            this.CurbWeight        = new EffectsValue();
            this.Footprint         = new EffectsValue();
            this.WorkFactor        = new EffectsValue();
            this.RatedFuelEconomy  = new EffectsValue();    // fuel economy
            this.OnRoadFuelEconomy = new EffectsValue();
            this.FuelShare         = new EffectsValue();
            this.Fleet             = new EffectsValue();    // fleet/vmt
            this.VMT               = new EffectsValue();
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="GeneralEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="GeneralEffects"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="GeneralEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="GeneralEffects"/>.</returns>
        public GeneralEffects Clone()
        {
            return new GeneralEffects(this);
        }

        #endregion

        /// <summary>
        /// Appends all general effects values in the specified <see cref="GeneralEffects"/> instance to this instance. The new
        /// value will be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        public void AppendValue(GeneralEffects value)
        {
            this.Sales            .AppendValue(value.Sales            );
            this.CurbWeight       .AppendValue(value.CurbWeight       );
            this.Footprint        .AppendValue(value.Footprint        );
            this.WorkFactor       .AppendValue(value.WorkFactor       );
            this.RatedFuelEconomy .AppendValue(value.RatedFuelEconomy );
            this.OnRoadFuelEconomy.AppendValue(value.OnRoadFuelEconomy);
            this.FuelShare        .AppendValue(value.FuelShare        );
            this.Fleet            .AppendValue(value.Fleet            );
            this.VMT              .AppendValue(value.VMT              );
        }

        /// <summary>
        /// Returns the total vehicle sales for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with
        /// the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The total vehicle sales for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetSales(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.Sales.GetValue(vehClass, regClass, fuelType);
        }

        /// <summary>
        /// Returns the average vehicle curb weight for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The average vehicle curb weight for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetCurbWeight(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.CalcArithmeticMean(vehClass, regClass, fuelType, this.CurbWeight, this.Sales);
        }
        /// <summary>
        /// Returns the average vehicle footprint for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The average vehicle footprint for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetFootprint(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.CalcArithmeticMean(vehClass, regClass, fuelType, this.Footprint, this.Sales);
        }
        /// <summary>
        /// Returns the average vehicle work factor for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The average vehicle work factor for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetWorkFactor(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.CalcArithmeticMean(vehClass, regClass, fuelType, this.WorkFactor, this.Sales);
        }
        /// <summary>
        /// Helper method for GetCurbWeight(...), GetFootprint(...), and GetWorkFactor(...).
        /// </summary>
        double CalcArithmeticMean(VehClass vehClass, RegClass regClass, FuelType fuelType, EffectsValue value, EffectsValue sales)
        {
            if (vehClass == VehClass.All && fuelType == FuelType.All)
            {
                double sum = 0;
                foreach (VehClass vc in VCValue<object>.Classes)
                {
                    double s = sales.GetValue(vc, RegClass.None, fuelType);
                    if (s != 0) { sum += value.GetValue(vc, RegClass.None).ArithmeticMean(sales.GetValue(vc, RegClass.None)) * s; }
                }
                return sum / sales.GetValue(VehClass.All, RegClass.None, FuelType.All);
            }
            else if (vehClass == VehClass.All)
            {
                double sum = 0;
                foreach (VehClass vc in VCValue<object>.Classes)
                {
                    double s = sales.GetValue(vc, RegClass.None, fuelType);
                    if (s != 0) { sum += value.GetValue(vc, RegClass.None, fuelType) * s; }
                }
                return sum / sales.GetValue(VehClass.All, RegClass.None, fuelType);
            }
            else if (regClass == RegClass.All && fuelType == FuelType.All)
            {
                double sum = 0;
                foreach (RegClass rc in RCValue<object>.Classes)
                {
                    double s = sales.GetValue(VehClass.None , rc, fuelType);
                    if (s != 0) { sum += value.GetValue(VehClass.None, rc).ArithmeticMean(sales.GetValue(VehClass.None, rc)) * s; }
                }
                return sum / sales.GetValue(VehClass.None, RegClass.All, FuelType.All);
            }
            else if (regClass == RegClass.All)
            {
                double sum = 0;
                foreach (RegClass rc in RCValue<object>.Classes)
                {
                    double s = sales.GetValue(VehClass.None, rc, fuelType);
                    if (s != 0) { sum += value.GetValue(VehClass.None, rc, fuelType) * s; }
                }
                return sum / sales.GetValue(VehClass.None, RegClass.All, fuelType);
            }
            else if (fuelType == FuelType.All)
            {
                return value.GetValue(vehClass, regClass).ArithmeticMean(sales.GetValue(vehClass, regClass));
            }
            else
            {
                return value.GetValue(vehClass, regClass, fuelType);
            }
        }

        /// <summary>
        /// Returns the average rated fuel economy for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <param name="energyEffects">An object containing the associated energy consumption effects data.</param>
        /// <returns>The average rated fuel economy for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetRatedFE(VehClass vehClass, RegClass regClass, FuelType fuelType, EnergyConsumptionEffects energyEffects)
        {
            return this.CalcFuelEconomy(vehClass, regClass, fuelType, this.RatedFuelEconomy, energyEffects.RatedGallons);
        }
        /// <summary>
        /// Returns the average on-road fuel economy for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <param name="energyEffects">An object containing the associated energy consumption effects data.</param>
        /// <returns>The average on-road fuel economy for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetOnRoadFE(VehClass vehClass, RegClass regClass, FuelType fuelType, EnergyConsumptionEffects energyEffects)
        {
            return this.CalcFuelEconomy(vehClass, regClass, fuelType, this.OnRoadFuelEconomy, energyEffects.Gallons);
        }
        /// <summary>
        /// Helper method for GetRatedFE(...) and GetOnRoadFE(...).
        /// </summary>
        double CalcFuelEconomy(VehClass vehClass, RegClass regClass, FuelType fuelType, EffectsValue value, EffectsValue gallons)
        {
            if (vehClass == VehClass.All || regClass == RegClass.All || fuelType == FuelType.All)
            {
                return this.VMT.GetValue(vehClass, regClass, fuelType) / gallons.GetValue(vehClass, regClass, fuelType);
            }
            return value.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the average fuel share for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with
        /// the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The average fuel share for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetFuelShare(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            if (fuelType == FuelType.All) { return 1; }
            if (vehClass == VehClass.All || regClass == RegClass.All)
            {
                return this.VMT.GetValue(vehClass, regClass, fuelType) / this.VMT.GetValue(vehClass, regClass, FuelType.All);
            }
            return this.FuelShare.GetValue(vehClass, regClass, fuelType);
        }

        /// <summary>
        /// Returns the vehicle on-road fleet (the survival weighted vehicle sales) for either the specified vehicle class (if
        /// wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the
        /// value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory
        /// class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The vehicle on-road fleet (the survival weighted vehicle sales) for the specified vehicle/regulatory class
        ///   and fuel type.</returns>
        public double GetFleet(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.Fleet.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the total vehicle miles traveled for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The total vehicle miles traveled for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetVMT(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.VMT.GetValue(vehClass, regClass, fuelType);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the total vehicle sales.</summary>
        public EffectsValue Sales { get; private set; }
        /// <summary>Gets the average vehicle curb weight.</summary>
        public EffectsValue CurbWeight { get; private set; }
        /// <summary>Gets the average vehicle footprint.</summary>
        public EffectsValue Footprint { get; private set; }
        /// <summary>Gets the average vehicle work factor.</summary>
        public EffectsValue WorkFactor { get; private set; }

        /// <summary>Gets the average rated fuel economy.</summary>
        public EffectsValue RatedFuelEconomy { get; private set; }
        /// <summary>Gets the average on-road fuel economy.</summary>
        public EffectsValue OnRoadFuelEconomy { get; private set; }
        /// <summary>Gets the average fuel share.</summary>
        public EffectsValue FuelShare { get; private set; }

        /// <summary>Gets the vehicle on-road fleet (the survival weighted vehicle sales).</summary>
        public EffectsValue Fleet { get; private set; }
        /// <summary>Gets the total vehicle miles traveled.</summary>
        public EffectsValue VMT { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides modeling effects for energy consumption from vehicle operation.
    /// </summary>
    [Serializable]
    public sealed class EnergyConsumptionEffects : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        EnergyConsumptionEffects(EnergyConsumptionEffects prototype)
        {
            this.BTU          = prototype.BTU         .Clone();
            this.Gallons      = prototype.Gallons     .Clone();
            this.RatedGallons = prototype.RatedGallons.Clone();
            this.NativeUnits  = prototype.NativeUnits .Clone();
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="EnergyConsumptionEffects"/> class.
        /// </summary>
        public EnergyConsumptionEffects()
        {
            this.BTU          = new EffectsValue();
            this.Gallons      = new EffectsValue();
            this.RatedGallons = new EffectsValue();
            this.NativeUnits  = new EffectsValue();
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EnergyConsumptionEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EnergyConsumptionEffects"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EnergyConsumptionEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EnergyConsumptionEffects"/>.</returns>
        public EnergyConsumptionEffects Clone()
        {
            return new EnergyConsumptionEffects(this);
        }

        #endregion

        /// <summary>
        /// Appends all energy consumption values in the specified <see cref="EnergyConsumptionEffects"/> instance to this
        /// instance. The new value will be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        public void AppendValue(EnergyConsumptionEffects value)
        {
            this.BTU         .AppendValue(value.BTU         );
            this.Gallons     .AppendValue(value.Gallons     );
            this.RatedGallons.AppendValue(value.RatedGallons);
            this.NativeUnits .AppendValue(value.NativeUnits );
        }

        /// <summary>
        /// Returns the energy use in BTU for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with
        /// the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The energy use in BTU for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetBTU(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.BTU.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the fuel consumption in liguid gallons or gasoline equivalent gallons (GEG) for either the specified vehicle
        /// class (if wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to
        /// obtain the value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or
        /// regulatory class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/>
        /// or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The fuel consumption in liguid gallons or gasoline equivalent gallons (GEG) for the specified
        ///   vehicle/regulatory class and fuel type.</returns>
        public double GetGallons(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.Gallons.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the fuel consumption in liguid gallons or gasoline equivalent gallons (GEG), using the rated vehicle fuel
        /// economy instead of the on-road fuel economy, for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The fuel consumption in liguid gallons or gasoline equivalent gallons (GEG), using the rated vehicle fuel
        ///   economy instead of the on-road fuel economy, for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetRatedGallons(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.RatedGallons.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the energy use in units native for a specific fuel type for either the specified vehicle class (if wishing to
        /// obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value
        /// aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class
        /// must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The energy use in units native for a specific fuel type for the specified vehicle/regulatory class and fuel
        ///   type.</returns>
        public double GetNativeUnits(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.NativeUnits.GetValue(vehClass, regClass, fuelType);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the energy use in BTU.</summary>
        public EffectsValue BTU { get; private set; }
        /// <summary>Gets the fuel consumption in liguid gallons or gasoline equivalent gallons (GEG).</summary>
        public EffectsValue Gallons { get; private set; }
        /// <summary>Gets the fuel consumption in liguid gallons or gasoline equivalent gallons (GEG), using the rated vehicle
        ///   fuel economy instead of the on-road fuel economy.</summary>
        public EffectsValue RatedGallons { get; private set; }
        /// <summary>Gets the energy use in units native for a specific fuel type.</summary>
        public EffectsValue NativeUnits { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides modeling effects for upstream and tailpipe vehicle emissions.
    /// </summary>
    [Serializable]
    public sealed class EmissionEffects : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        EmissionEffects(EmissionEffects prototype)
        {
            // upstream emissions
            this.UpstreamCO           = prototype.UpstreamCO          .Clone(); // emissions for effects modeling
            this.UpstreamVOC          = prototype.UpstreamVOC         .Clone();
            this.UpstreamNOx          = prototype.UpstreamNOx         .Clone();
            this.UpstreamSO2          = prototype.UpstreamSO2         .Clone();
            this.UpstreamPM           = prototype.UpstreamPM          .Clone();
            this.UpstreamCO2          = prototype.UpstreamCO2         .Clone();
            this.UpstreamCH4          = prototype.UpstreamCH4         .Clone();
            this.UpstreamN2O          = prototype.UpstreamN2O         .Clone();
            this.UpstreamAcetaldehyde = prototype.UpstreamAcetaldehyde.Clone(); // additional emissions for EIS modeling
            this.UpstreamAcrolein     = prototype.UpstreamAcrolein    .Clone();
            this.UpstreamBenzene      = prototype.UpstreamBenzene     .Clone();
            this.UpstreamButadiene    = prototype.UpstreamButadiene   .Clone();
            this.UpstreamFormaldehyde = prototype.UpstreamFormaldehyde.Clone();
            this.UpstreamDPM10        = prototype.UpstreamDPM10       .Clone();
            this.UpstreamMTBE         = prototype.UpstreamMTBE        .Clone();
            // tailpipe emissions
            this.TailpipeCO           = prototype.TailpipeCO          .Clone(); // emissions for effects modeling
            this.TailpipeVOC          = prototype.TailpipeVOC         .Clone();
            this.TailpipeNOx          = prototype.TailpipeNOx         .Clone();
            this.TailpipeSO2          = prototype.TailpipeSO2         .Clone();
            this.TailpipePM           = prototype.TailpipePM          .Clone();
            this.TailpipeCO2          = prototype.TailpipeCO2         .Clone();
            this.TailpipeCH4          = prototype.TailpipeCH4         .Clone();
            this.TailpipeN2O          = prototype.TailpipeN2O         .Clone();
            this.TailpipeAcetaldehyde = prototype.TailpipeAcetaldehyde.Clone(); // additional emissions for EIS modeling
            this.TailpipeAcrolein     = prototype.TailpipeAcrolein    .Clone();
            this.TailpipeBenzene      = prototype.TailpipeBenzene     .Clone();
            this.TailpipeButadiene    = prototype.TailpipeButadiene   .Clone();
            this.TailpipeFormaldehyde = prototype.TailpipeFormaldehyde.Clone();
            this.TailpipeDPM10        = prototype.TailpipeDPM10       .Clone();
            this.TailpipeMTBE         = prototype.TailpipeMTBE        .Clone();
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="EmissionEffects"/> class.
        /// </summary>
        public EmissionEffects()
        {
            // upstream emissions
            this.UpstreamCO           = new EffectsValue();     // emissions for effects modeling
            this.UpstreamVOC          = new EffectsValue();
            this.UpstreamNOx          = new EffectsValue();
            this.UpstreamSO2          = new EffectsValue();
            this.UpstreamPM           = new EffectsValue();
            this.UpstreamCO2          = new EffectsValue();
            this.UpstreamCH4          = new EffectsValue();
            this.UpstreamN2O          = new EffectsValue();
            this.UpstreamAcetaldehyde = new EffectsValue();     // additional emissions for EIS modeling
            this.UpstreamAcrolein     = new EffectsValue();
            this.UpstreamBenzene      = new EffectsValue();
            this.UpstreamButadiene    = new EffectsValue();
            this.UpstreamFormaldehyde = new EffectsValue();
            this.UpstreamDPM10        = new EffectsValue();
            this.UpstreamMTBE         = new EffectsValue();
            // tailpipe emissions
            this.TailpipeCO           = new EffectsValue();     // emissions for effects modeling
            this.TailpipeVOC          = new EffectsValue();
            this.TailpipeNOx          = new EffectsValue();
            this.TailpipeSO2          = new EffectsValue();
            this.TailpipePM           = new EffectsValue();
            this.TailpipeCO2          = new EffectsValue();
            this.TailpipeCH4          = new EffectsValue();
            this.TailpipeN2O          = new EffectsValue();
            this.TailpipeAcetaldehyde = new EffectsValue();     // additional emissions for EIS modeling
            this.TailpipeAcrolein     = new EffectsValue();
            this.TailpipeBenzene      = new EffectsValue();
            this.TailpipeButadiene    = new EffectsValue();
            this.TailpipeFormaldehyde = new EffectsValue();
            this.TailpipeDPM10        = new EffectsValue();
            this.TailpipeMTBE         = new EffectsValue();
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EmissionEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EmissionEffects"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EmissionEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EmissionEffects"/>.</returns>
        public EmissionEffects Clone()
        {
            return new EmissionEffects(this);
        }

        #endregion

        /// <summary>
        /// Appends all emission values in the specified <see cref="EmissionEffects"/> instance to this instance. The new value
        /// will be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        public void AppendValue(EmissionEffects value)
        {
            // upstream emissions
            this.UpstreamCO          .AppendValue(value.UpstreamCO          );
            this.UpstreamVOC         .AppendValue(value.UpstreamVOC         );
            this.UpstreamNOx         .AppendValue(value.UpstreamNOx         );
            this.UpstreamSO2         .AppendValue(value.UpstreamSO2         );
            this.UpstreamPM          .AppendValue(value.UpstreamPM          );
            this.UpstreamCO2         .AppendValue(value.UpstreamCO2         );
            this.UpstreamCH4         .AppendValue(value.UpstreamCH4         );
            this.UpstreamN2O         .AppendValue(value.UpstreamN2O         );
            this.UpstreamAcetaldehyde.AppendValue(value.UpstreamAcetaldehyde);
            this.UpstreamAcrolein    .AppendValue(value.UpstreamAcrolein    );
            this.UpstreamBenzene     .AppendValue(value.UpstreamBenzene     );
            this.UpstreamButadiene   .AppendValue(value.UpstreamButadiene   );
            this.UpstreamFormaldehyde.AppendValue(value.UpstreamFormaldehyde);
            this.UpstreamDPM10       .AppendValue(value.UpstreamDPM10       );
            this.UpstreamMTBE        .AppendValue(value.UpstreamMTBE        );
            // tailpipe emissions
            this.TailpipeCO          .AppendValue(value.TailpipeCO          );
            this.TailpipeVOC         .AppendValue(value.TailpipeVOC         );
            this.TailpipeNOx         .AppendValue(value.TailpipeNOx         );
            this.TailpipeSO2         .AppendValue(value.TailpipeSO2         );
            this.TailpipePM          .AppendValue(value.TailpipePM          );
            this.TailpipeCO2         .AppendValue(value.TailpipeCO2         );
            this.TailpipeCH4         .AppendValue(value.TailpipeCH4         );
            this.TailpipeN2O         .AppendValue(value.TailpipeN2O         );
            this.TailpipeAcetaldehyde.AppendValue(value.TailpipeAcetaldehyde);
            this.TailpipeAcrolein    .AppendValue(value.TailpipeAcrolein    );
            this.TailpipeBenzene     .AppendValue(value.TailpipeBenzene     );
            this.TailpipeButadiene   .AppendValue(value.TailpipeButadiene   );
            this.TailpipeFormaldehyde.AppendValue(value.TailpipeFormaldehyde);
            this.TailpipeDPM10       .AppendValue(value.TailpipeDPM10       );
            this.TailpipeMTBE        .AppendValue(value.TailpipeMTBE        );
        }

        /// <summary>
        /// Returns the upstream emissions (from all stages of fuel production and distribution) for a specified pollutant, for
        /// either the specified vehicle class (if wishing to obtain the value aggregated by vehicle class) or the specified
        /// regulatory class (if wishing to obtain the value aggregated by regulatory class), as well as the specified fuel type.
        /// Only one of vehicle class or regulatory class must be used for disaggregation, with the other being "none" (either
        /// <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <param name="pollutant">The pollutant for which to obtain the upstream emissions.</param>
        /// <returns>The upstream emissions (from all stages of fuel production and distribution) for the specified pollutant,
        ///   vehicle/regulatory class, and fuel type.</returns>
        public double GetUpstreamEmissions(VehClass vehClass, RegClass regClass, FuelType fuelType, Pollutant pollutant)
        {
            return this.GetUpstreamEmissions(pollutant).GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the tailpipe emissions (from vehicle operation) for a specified pollutant, for either the specified vehicle
        /// class (if wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to
        /// obtain the value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or
        /// regulatory class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/>
        /// or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <param name="pollutant">The pollutant for which to obtain the tailpipe emissions.</param>
        /// <returns>The tailpipe emissions (from vehicle operation) for the specified pollutant, vehicle/regulatory class, and
        ///   fuel type.</returns>
        public double GetTailpipeEmissions(VehClass vehClass, RegClass regClass, FuelType fuelType, Pollutant pollutant)
        {
            return this.GetTailpipeEmissions(pollutant).GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the total emissions (from all stages of fuel production and distribution as well as from vehicle operation)
        /// for the specified pollutant, for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with
        /// the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <param name="pollutant">The pollutant for which to obtain the total emissions.</param>
        /// <returns>The total emissions (from all stages of fuel production and distribution as well as from vehicle operation)
        ///   for the specified pollutant, vehicle/regulatory class, and fuel type.</returns>
        public double GetTotalEmissions(VehClass vehClass, RegClass regClass, FuelType fuelType, Pollutant pollutant)
        {
            EffectsValue upstream, tailpipe;
            this.GetEmissions(pollutant, out upstream, out tailpipe);
            return upstream.GetValue(vehClass, regClass, fuelType) +
                   tailpipe.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the total emissions (from all stages of fuel production and distribution as well as from vehicle operation)
        /// for the specified pollutant, for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type and vehicle age. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="age">The vehicle age to use for disaggregation.</param>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <param name="pollutant">The pollutant for which to obtain the total emissions.</param>
        /// <returns>The total emissions (from all stages of fuel production and distribution as well as from vehicle operation)
        ///   for the specified pollutant, vehicle/regulatory class, fuel type, and vehicle age.</returns>
        public double GetTotalEmissions(int age, VehClass vehClass, RegClass regClass, FuelType fuelType, Pollutant pollutant)
        {
            EffectsValue upstream, tailpipe;
            this.GetEmissions(pollutant, out upstream, out tailpipe);
            //
            return upstream.GetAnnualValue(age, vehClass, regClass, fuelType) +
                   tailpipe.GetAnnualValue(age, vehClass, regClass, fuelType);
        }

        /// <summary>
        /// Returns the upstream emissions (from all stages of fuel production and distribution) for a specified pollutant.
        /// </summary>
        /// <param name="pollutant">The pollutant for which to obtain the upstream emissions.</param>
        /// <returns>The upstream emissions (from all stages of fuel production and distribution) for the specified pollutant.</returns>
        public EffectsValue GetUpstreamEmissions(Pollutant pollutant)
        {
            EffectsValue upstream, tailpipe;
            this.GetEmissions(pollutant, out upstream, out tailpipe);
            return upstream;
        }
        /// <summary>
        /// Returns the tailpipe emissions (from vehicle operation) for a specified pollutant.
        /// </summary>
        /// <param name="pollutant">The pollutant for which to obtain the tailpipe emissions.</param>
        /// <returns>The tailpipe emissions (from vehicle operation) for the specified pollutant.</returns>
        public EffectsValue GetTailpipeEmissions(Pollutant pollutant)
        {
            EffectsValue upstream, tailpipe;
            this.GetEmissions(pollutant, out upstream, out tailpipe);
            return tailpipe;
        }
        void GetEmissions(Pollutant pollutant, out EffectsValue upstream, out EffectsValue tailpipe)
        {
            switch (pollutant)
            {
                case Pollutant.CO          : upstream = UpstreamCO          ; tailpipe = TailpipeCO          ; break;
                case Pollutant.VOC         : upstream = UpstreamVOC         ; tailpipe = TailpipeVOC         ; break;
                case Pollutant.NOx         : upstream = UpstreamNOx         ; tailpipe = TailpipeNOx         ; break;
                case Pollutant.SO2         : upstream = UpstreamSO2         ; tailpipe = TailpipeSO2         ; break;
                case Pollutant.PM          : upstream = UpstreamPM          ; tailpipe = TailpipePM          ; break;
                case Pollutant.CO2         : upstream = UpstreamCO2         ; tailpipe = TailpipeCO2         ; break;
                case Pollutant.CH4         : upstream = UpstreamCH4         ; tailpipe = TailpipeCH4         ; break;
                case Pollutant.N2O         : upstream = UpstreamN2O         ; tailpipe = TailpipeN2O         ; break;
                case Pollutant.Acetaldehyde: upstream = UpstreamAcetaldehyde; tailpipe = TailpipeAcetaldehyde; break;
                case Pollutant.Acrolein    : upstream = UpstreamAcrolein    ; tailpipe = TailpipeAcrolein    ; break;
                case Pollutant.Benzene     : upstream = UpstreamBenzene     ; tailpipe = TailpipeBenzene     ; break;
                case Pollutant.Butadiene   : upstream = UpstreamButadiene   ; tailpipe = TailpipeButadiene   ; break;
                case Pollutant.Formaldehyde: upstream = UpstreamFormaldehyde; tailpipe = TailpipeFormaldehyde; break;
                case Pollutant.DPM10       : upstream = UpstreamDPM10       ; tailpipe = TailpipeDPM10       ; break;
                case Pollutant.MTBE        : upstream = UpstreamMTBE        ; tailpipe = TailpipeMTBE        ; break;
                //
                default: throw new ArgumentOutOfRangeException("pollutant", "The specified pollutant is not supported.");
            }
        }

        #endregion

        #region /*** Properties ***/

        //----- upstream emissions -----
        /// <summary>Gets the Carbon Monoxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamCO { get; private set; }
        /// <summary>Gets the Volatile Organic Compound upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamVOC { get; private set; }
        /// <summary>Gets the Nitrogen Oxides upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamNOx { get; private set; }
        /// <summary>Gets the Sulfur Dioxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamSO2 { get; private set; }
        /// <summary>Gets the Particulate Matter (diameter of ~2.5 micrometers) upstream emissions (from all stages of fuel
        ///   production and distribution).</summary>
        public EffectsValue UpstreamPM { get; private set; }
        /// <summary>Gets the Carbon Dioxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamCO2 { get; private set; }
        /// <summary>Gets the Methane upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamCH4 { get; private set; }
        /// <summary>Gets the Nitrous Oxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamN2O { get; private set; }

        /// <summary>Gets the Acetaldehyde upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamAcetaldehyde { get; private set; }
        /// <summary>Gets the Acrolein upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamAcrolein { get; private set; }
        /// <summary>Gets the Benzene upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamBenzene { get; private set; }
        /// <summary>Gets the 1,3-Butadiene upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamButadiene { get; private set; }
        /// <summary>Gets the Formaldehyde upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamFormaldehyde { get; private set; }
        /// <summary>Gets the Diesel Particulate Matter (diameter of ~10 micrometers) upstream emissions (from all stages of fuel
        ///   production and distribution).</summary>
        public EffectsValue UpstreamDPM10 { get; private set; }
        /// <summary>Gets the Methyl Tertiary Butyl Ether upstream emissions (from all stages of fuel production and distribution).</summary>
        public EffectsValue UpstreamMTBE { get; private set; }

        //----- tailpipe emissions -----
        /// <summary>Gets the Carbon Monoxide tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeCO { get; private set; }
        /// <summary>Gets the Volatile Organic Compound tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeVOC { get; private set; }
        /// <summary>Gets the Nitrogen Oxides tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeNOx { get; private set; }
        /// <summary>Gets the Sulfur Dioxide tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeSO2 { get; private set; }
        /// <summary>Gets the Particulate Matter (diameter of ~2.5 micrometers) tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipePM { get; private set; }
        /// <summary>Gets the Carbon Dioxide tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeCO2 { get; private set; }
        /// <summary>Gets the Methane tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeCH4 { get; private set; }
        /// <summary>Gets the Nitrous Oxide tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeN2O { get; private set; }

        /// <summary>Gets the Acetaldehyde tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeAcetaldehyde { get; private set; }
        /// <summary>Gets the Acrolein tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeAcrolein { get; private set; }
        /// <summary>Gets the Benzene tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeBenzene { get; private set; }
        /// <summary>Gets the 1,3-Butadiene tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeButadiene { get; private set; }
        /// <summary>Gets the Formaldehyde tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeFormaldehyde { get; private set; }
        /// <summary>Gets the Diesel Particulate Matter (diameter of ~10 micrometers) tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeDPM10 { get; private set; }
        /// <summary>Gets the Methyl Tertiary Butyl Ether tailpipe emissions (from vehicle operation).</summary>
        public EffectsValue TailpipeMTBE { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides modeling effects for changes in safety due to reduction in vehicle weight.
    /// </summary>
    [Serializable]
    public sealed class SafetyEffects : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        SafetyEffects(SafetyEffects prototype)
        {
            this.Fatalities = prototype.Fatalities.Clone();
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="SafetyEffects"/> class.
        /// </summary>
        public SafetyEffects()
        {
            this.Fatalities = new EffectsValue();
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="SafetyEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="SafetyEffects"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="SafetyEffects"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="SafetyEffects"/>.</returns>
        public SafetyEffects Clone()
        {
            return new SafetyEffects(this);
        }

        #endregion

        /// <summary>
        /// Appends all safety values in the specified <see cref="SafetyEffects"/> instance to this instance. The new value will
        /// be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        public void AppendValue(SafetyEffects value)
        {
            this.Fatalities.AppendValue(value.Fatalities);
        }

        /// <summary>
        /// Returns the changes in fatalities due to reduction in vehicle curb weight for either the specified vehicle class (if
        /// wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the
        /// value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory
        /// class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The changes in fatalities due to reduction in vehicle curb weight for the specified vehicle/regulatory class
        ///   and fuel type.</returns>
        public double GetFatalities(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.Fatalities.GetValue(vehClass, regClass, fuelType);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the changes in fatalities due to reduction in vehicle curb weight.</summary>
        public EffectsValue Fatalities { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides owner and societal costs arrising from the various modeling effects.
    /// </summary>
    [Serializable]
    public sealed class OwnerAndSocietalCosts : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        OwnerAndSocietalCosts(OwnerAndSocietalCosts prototype)
        {
            this.DiscountRate       = prototype.DiscountRate;
            //
            this.PreTaxFuelCosts    = prototype.PreTaxFuelCosts   .Clone();
            this.FuelTaxCosts       = prototype.FuelTaxCosts      .Clone();
            this.RetailFuelCosts    = prototype.RetailFuelCosts   .Clone();
            this.DriveSurplus       = prototype.DriveSurplus      .Clone();
            this.RefuelSurplus      = prototype.RefuelSurplus     .Clone();
            this.EconomicCosts      = prototype.EconomicCosts     .Clone();
            this.CongestionCosts    = prototype.CongestionCosts   .Clone();
            this.AccidentCosts      = prototype.AccidentCosts     .Clone();
            this.NoiseCosts         = prototype.NoiseCosts        .Clone();
            this.FatalityCosts      = prototype.FatalityCosts     .Clone();
            //
            this.EmissionsCostsCO   = prototype.EmissionsCostsCO  .Clone();
            this.EmissionsCostsVOC  = prototype.EmissionsCostsVOC .Clone();
            this.EmissionsCostsNOx  = prototype.EmissionsCostsNOx .Clone();
            this.EmissionsCostsSO2  = prototype.EmissionsCostsSO2 .Clone();
            this.EmissionsCostsPM   = prototype.EmissionsCostsPM  .Clone();
            this.EmissionsCostsCO2  = prototype.EmissionsCostsCO2 .Clone();
            this.EmissionsCostsCH4  = prototype.EmissionsCostsCH4 .Clone();
            this.EmissionsCostsN2O  = prototype.EmissionsCostsN2O .Clone();
            //
            this.TotalConsumerCosts = prototype.TotalConsumerCosts.Clone();
            this.TotalSocialCosts   = prototype.TotalSocialCosts  .Clone();
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="OwnerAndSocietalCosts"/> class.
        /// </summary>
        /// <param name="discRate">The discount rate at which the owner and societal costs are discounted, or 0, if the costs are
        ///   undiscounted.</param>
        public OwnerAndSocietalCosts(VCValue<double> discRate)
        {
            this.DiscountRate = discRate;
            //
            this.PreTaxFuelCosts    = new EffectsValue();
            this.FuelTaxCosts       = new EffectsValue();
            this.RetailFuelCosts    = new EffectsValue();
            this.DriveSurplus       = new EffectsValue();
            this.RefuelSurplus      = new EffectsValue();
            this.EconomicCosts      = new EffectsValue();
            this.CongestionCosts    = new EffectsValue();
            this.AccidentCosts      = new EffectsValue();
            this.NoiseCosts         = new EffectsValue();
            this.FatalityCosts      = new EffectsValue();
            //
            this.EmissionsCostsCO   = new EffectsValue();
            this.EmissionsCostsVOC  = new EffectsValue();
            this.EmissionsCostsNOx  = new EffectsValue();
            this.EmissionsCostsSO2  = new EffectsValue();
            this.EmissionsCostsPM   = new EffectsValue();
            this.EmissionsCostsCO2  = new EffectsValue();
            this.EmissionsCostsCH4  = new EffectsValue();
            this.EmissionsCostsN2O  = new EffectsValue();
            //
            this.TotalConsumerCosts = new EffectsValue();
            this.TotalSocialCosts   = new EffectsValue();
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OwnerAndSocietalCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OwnerAndSocietalCosts"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OwnerAndSocietalCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OwnerAndSocietalCosts"/>.</returns>
        public OwnerAndSocietalCosts Clone()
        {
            return new OwnerAndSocietalCosts(this);
        }

        #endregion

        /// <summary>
        /// Appends all costs values in the specified <see cref="OwnerAndSocietalCosts"/> instance to this instance. The new
        /// value will be added to the existing one.
        /// </summary>
        /// <param name="value">The value to append.</param>
        public void AppendValue(OwnerAndSocietalCosts value)
        {
            // energy and safety costs
            this.PreTaxFuelCosts   .AppendValue(value.PreTaxFuelCosts   );
            this.FuelTaxCosts      .AppendValue(value.FuelTaxCosts      );
            this.RetailFuelCosts   .AppendValue(value.RetailFuelCosts   );
            this.DriveSurplus      .AppendValue(value.DriveSurplus      );
            this.RefuelSurplus     .AppendValue(value.RefuelSurplus     );
            this.EconomicCosts     .AppendValue(value.EconomicCosts     );
            this.CongestionCosts   .AppendValue(value.CongestionCosts   );
            this.AccidentCosts     .AppendValue(value.AccidentCosts     );
            this.NoiseCosts        .AppendValue(value.NoiseCosts        );
            this.FatalityCosts     .AppendValue(value.FatalityCosts     );
            // emissions costs
            this.EmissionsCostsCO  .AppendValue(value.EmissionsCostsCO  );
            this.EmissionsCostsVOC .AppendValue(value.EmissionsCostsVOC );
            this.EmissionsCostsNOx .AppendValue(value.EmissionsCostsNOx );
            this.EmissionsCostsSO2 .AppendValue(value.EmissionsCostsSO2 );
            this.EmissionsCostsPM  .AppendValue(value.EmissionsCostsPM  );
            this.EmissionsCostsCO2 .AppendValue(value.EmissionsCostsCO2 );
            this.EmissionsCostsCH4 .AppendValue(value.EmissionsCostsCH4 );
            this.EmissionsCostsN2O .AppendValue(value.EmissionsCostsN2O );
            // "totals"
            this.TotalConsumerCosts.AppendValue(value.TotalConsumerCosts);
            this.TotalSocialCosts  .AppendValue(value.TotalSocialCosts  );
        }

        /// <summary>
        /// Returns the discount rate at which the owner and societal costs are discounted, or 0, if the costs are undiscounted.
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <returns>The discount rate at which the owner and societal costs are discounted, or 0, if the costs are undiscounted.</returns>
        public double GetDiscountRate(VehClass vehClass, RegClass regClass)
        {
            // only return actual disc-rate when a specific vehicle class is used
            if (vehClass != VehClass.None && vehClass != VehicleClass.All) { return this.DiscountRate[vehClass]; }
            // for all other queries, calculate average value
            double dr = 0;
            foreach (VehicleClass vc in VCValue<object>.Classes)
            {
                dr += this.DiscountRate[vc];
            }
            return dr / VCValue<object>.Classes.Length;
        }

        /// <summary>
        /// Returns the pre-tax fuel expenditures for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The pre-tax fuel expenditures for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetPreTaxFuelCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.PreTaxFuelCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the fuel tax revenues for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with
        /// the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The fuel tax revenues for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetFuelTaxCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.FuelTaxCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the retail fuel expenditures for either the specified vehicle class (if wishing to obtain the value aggregated
        /// by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class),
        /// as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation,
        /// with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The retail fuel expenditures for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetRetailFuelCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.RetailFuelCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the vehicle travel value for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with
        /// the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The vehicle travel value for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetDriveSurplus(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.DriveSurplus.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the refueling time value for either the specified vehicle class (if wishing to obtain the value aggregated by
        /// vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by regulatory class), as
        /// well as the specified fuel type. Only one of vehicle class or regulatory class must be used for disaggregation, with
        /// the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The refueling time value for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetRefuelSurplus(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.RefuelSurplus.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the petroleum market externalities from oil imports for either the specified vehicle class (if wishing to
        /// obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value
        /// aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class
        /// must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The petroleum market externalities from oil imports for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetEconomicCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.EconomicCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the congestion costs from additional vehicle use due to rebound effect for either the specified vehicle class
        /// (if wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to obtain
        /// the value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory
        /// class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The congestion costs from additional vehicle use due to rebound effect for the specified vehicle/regulatory
        ///   class and fuel type.</returns>
        public double GetCongestionCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.CongestionCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the accident costs from additional vehicle use due to rebound effect for either the specified vehicle class
        /// (if wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to obtain
        /// the value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory
        /// class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The accident costs from additional vehicle use due to rebound effect for the specified vehicle/regulatory
        ///   class and fuel type.</returns>
        public double GetAccidentCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.AccidentCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the noise costs from additional vehicle use due to rebound effect for either the specified vehicle class (if
        /// wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the
        /// value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory
        /// class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or
        /// <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The noise costs from additional vehicle use due to rebound effect for the specified vehicle/regulatory class
        ///   and fuel type.</returns>
        public double GetNoiseCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.NoiseCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the additional fatality costs resulting from reduction in vehicle weight for either the specified vehicle
        /// class (if wishing to obtain the value aggregated by vehicle class) or the specified regulatory class (if wishing to
        /// obtain the value aggregated by regulatory class), as well as the specified fuel type. Only one of vehicle class or
        /// regulatory class must be used for disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/>
        /// or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The additional fatality costs resulting from reduction in vehicle weight for the specified vehicle/regulatory
        ///   class and fuel type.</returns>
        public double GetFatalityCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.FatalityCosts.GetValue(vehClass, regClass, fuelType);
        }

        /// <summary>
        /// Returns the owner and societal costs arising from damage caused by emissions from the specified pollutant, for either
        /// the specified vehicle class (if wishing to obtain the value aggregated by vehicle class) or the specified regulatory
        /// class (if wishing to obtain the value aggregated by regulatory class), as well as the specified fuel type. Only one
        /// of vehicle class or regulatory class must be used for disaggregation, with the other being "none" (either
        /// <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <param name="pollutant">The pollutant for which to obtain the emission costs.</param>
        /// <returns>The owner and societal costs arising from damage caused by emissions from the specified pollutant, for the
        ///   specified vehicle/regulatory class and fuel type.</returns>
        public double GetEmissionsCosts(VehClass vehClass, RegClass regClass, FuelType fuelType, Pollutant pollutant)
        {
            return this.GetEmissionsCosts(pollutant).GetValue(vehClass, regClass, fuelType);
        }
        EffectsValue GetEmissionsCosts(Pollutant pollutant)
        {
            switch (pollutant)
            {
                case Pollutant.CO : return EmissionsCostsCO ;
                case Pollutant.VOC: return EmissionsCostsVOC;
                case Pollutant.NOx: return EmissionsCostsNOx;
                case Pollutant.SO2: return EmissionsCostsSO2;
                case Pollutant.PM : return EmissionsCostsPM ;
                case Pollutant.CO2: return EmissionsCostsCO2;
                case Pollutant.CH4: return EmissionsCostsCH4;
                case Pollutant.N2O: return EmissionsCostsN2O;
                //
                default: throw new ArgumentOutOfRangeException("pollutant", "The specified pollutant is not supported.");
            }
        }

        /// <summary>
        /// Returns the total sum of all consumer costs for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The total sum of all consumer costs for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetTotalConsumerCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.TotalConsumerCosts.GetValue(vehClass, regClass, fuelType);
        }
        /// <summary>
        /// Returns the total sum of all social costs for either the specified vehicle class (if wishing to obtain the value
        /// aggregated by vehicle class) or the specified regulatory class (if wishing to obtain the value aggregated by
        /// regulatory class), as well as the specified fuel type. Only one of vehicle class or regulatory class must be used for
        /// disaggregation, with the other being "none" (either <see cref="VehicleClass.None"/> or <see cref="RegulatoryClass.None"/>).
        /// </summary>
        /// <param name="vehClass">The vehicle class to use for disaggregation, or <see cref="VehicleClass.None"/> if not applicable.</param>
        /// <param name="regClass">The regulatory class to use for disaggregation, or <see cref="RegulatoryClass.None"/> if not applicable.</param>
        /// <param name="fuelType">The fuel type to use for disaggregation.</param>
        /// <returns>The total sum of all social costs for the specified vehicle/regulatory class and fuel type.</returns>
        public double GetTotalSocialCosts(VehClass vehClass, RegClass regClass, FuelType fuelType)
        {
            return this.TotalSocialCosts.GetValue(vehClass, regClass, fuelType);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the discount rate at which the owner and societal costs are discounted, or 0, if the costs are
        ///   undiscounted.</summary>
        public VCValue<double> DiscountRate { get; private set; }

        /// <summary>Gets the pre-tax fuel expenditures.</summary>
        public EffectsValue PreTaxFuelCosts { get; private set; }
        /// <summary>Gets the fuel tax revenues.</summary>
        public EffectsValue FuelTaxCosts { get; private set; }
        /// <summary>Gets the retail fuel expenditures.</summary>
        public EffectsValue RetailFuelCosts { get; private set; }
        /// <summary>Gets the vehicle travel value.</summary>
        public EffectsValue DriveSurplus { get; private set; }
        /// <summary>Gets the refueling time value.</summary>
        public EffectsValue RefuelSurplus { get; private set; }
        /// <summary>Gets the petroleum market externalities from oil imports.</summary>
        public EffectsValue EconomicCosts { get; private set; }
        /// <summary>Gets the congestion costs from additional vehicle use due to rebound effect.</summary>
        public EffectsValue CongestionCosts { get; private set; }
        /// <summary>Gets the accident costs from additional vehicle use due to rebound effect.</summary>
        public EffectsValue AccidentCosts { get; private set; }
        /// <summary>Gets the noise costs from additional vehicle use due to rebound effect.</summary>
        public EffectsValue NoiseCosts { get; private set; }
        /// <summary>Gets the additional fatality costs resulting from reduction in vehicle weight.</summary>
        public EffectsValue FatalityCosts { get; private set; }
        //
        /// <summary>Gets the owner and societal costs arising from damage caused by Carbon Monoxide emissions.</summary>
        public EffectsValue EmissionsCostsCO { get; private set; }
        /// <summary>Gets the owner and societal costs arising from damage caused by Volatile Organic Compounds emissions.</summary>
        public EffectsValue EmissionsCostsVOC { get; private set; }
        /// <summary>Gets the owner and societal costs arising from damage caused by Nitrogen Oxides emissions.</summary>
        public EffectsValue EmissionsCostsNOx { get; private set; }
        /// <summary>Gets the owner and societal costs arising from damage caused by Sulfur Dioxide emissions.</summary>
        public EffectsValue EmissionsCostsSO2 { get; private set; }
        /// <summary>Gets the owner and societal costs arising from damage caused by Particulate Matter (diameter of ~2.5 micrometers) emissions.</summary>
        public EffectsValue EmissionsCostsPM { get; private set; }
        /// <summary>Gets the owner and societal costs arising from damage caused by Carbon Dioxide emissions.</summary>
        public EffectsValue EmissionsCostsCO2 { get; private set; }
        /// <summary>Gets the owner and societal costs arising from damage caused by Methane emissions.</summary>
        public EffectsValue EmissionsCostsCH4 { get; private set; }
        /// <summary>Gets the owner and societal costs arising from damage caused by Nitrous Oxide emissions.</summary>
        public EffectsValue EmissionsCostsN2O { get; private set; }

        //----- totals -----
        /// <summary>Gets the total sum of all consumer costs.</summary>
        public EffectsValue TotalConsumerCosts { get; private set; }
        /// <summary>Gets the total sum of all social costs.</summary>
        public EffectsValue TotalSocialCosts { get; private set; }

        #endregion

    }

    #endregion

    /// <summary>
    /// Provides modeling effects data for a single model year.
    /// </summary>
    [Serializable]
    public sealed class EffectsData : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        EffectsData() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="EffectsData"/> class.
        /// </summary>
        /// <param name="modelYear">The model year for which the effects data are calculated.</param>
        /// <param name="initCosts">true, to initialize owner and societal costs variable; false, otherwise. Owner and societal
        ///   costs should be initialized for all model years examined as part of the compliance modeling; however, they should
        ///   not be initialized for extended model years used to obtain historic calendar year effects.</param>
        /// <param name="discRate">The social discount rate at which the owner and societal costs are discounted.</param>
        public EffectsData(int modelYear, bool initCosts, VCValue<double> discRate)
        {
            this.ModelYear       = modelYear;
            this.GeneralEffects  = new GeneralEffects();
            this.EnergyEffects   = new EnergyConsumptionEffects();
            this.EmissionEffects = new EmissionEffects();
            this.SafetyEffects   = new SafetyEffects();
            if (initCosts)
            {
                this.UndiscountedCosts = new OwnerAndSocietalCosts(new VCValue<double>(0, 0, 0));
                this.DiscountedCosts   = new OwnerAndSocietalCosts(discRate);
            }
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EffectsData"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EffectsData"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EffectsData"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EffectsData"/>.</returns>
        public EffectsData Clone()
        {
            EffectsData value = new EffectsData();
            //
            value.ModelYear         = this.ModelYear;
            value.GeneralEffects    = this.GeneralEffects   .Clone();
            value.EnergyEffects     = this.EnergyEffects    .Clone();
            value.EmissionEffects   = this.EmissionEffects  .Clone();
            value.SafetyEffects     = this.SafetyEffects    .Clone();
            value.UndiscountedCosts = this.UndiscountedCosts.Clone();
            value.DiscountedCosts   = this.DiscountedCosts  .Clone();
            //
            return value;
        }

        #endregion

        #endregion

        #region /*** Properties ***/

        //----- general data -----
        /// <summary>Gets the model year for which the effects data are calculated.</summary>
        public int ModelYear { get; private set; }

        //----- modeling effects -----
        /// <summary>Gets the various vehicle characteristic data, average fuel economy, on-road fleet, and vehicle miles
        ///   traveled, which serve as the basis for the modeling effects calculations.</summary>
        public GeneralEffects GeneralEffects { get; private set; }
        /// <summary>Gets the modeling effects for energy consumption from vehicle operation.</summary>
        public EnergyConsumptionEffects EnergyEffects { get; private set; }
        /// <summary>Gets the modeling effects for upstream and tailpipe vehicle emissions.</summary>
        public EmissionEffects EmissionEffects { get; private set; }
        /// <summary>Gets the modeling effects for changes in safety due to reduction in vehicle weight.</summary>
        public SafetyEffects SafetyEffects { get; private set; }

        //----- owner & societal costs -----
        /// <summary>Gets the undiscounted owner and societal costs arrising from the various modeling effects.</summary>
        public OwnerAndSocietalCosts UndiscountedCosts { get; private set; }
        /// <summary>Gets the discounted owner and societal costs arrising from the various modeling effects.</summary>
        public OwnerAndSocietalCosts DiscountedCosts { get; private set; }

        #endregion

    }
}
