using System;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Generic;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Provides the fuel price for the remaining lifetime of the vehicle.
    /// </summary>
    [Serializable]
    public sealed class KMYType
    {

        #region /*** Constructors ***/

        // Private constructor for internal usage.
        KMYType() { }

        /// <summary>
        /// Initializes a new instance of the KMYType class, calculating the remaining lifetime of the vehicle for the specified
        /// model year, using the provided parameters, fuel price estimates, and the manufacturer-specific discount rate obtained
        /// from the given manufacturer.
        /// </summary>
        /// <param name="year">The model year for which to calculate the KMY value.</param>
        /// <param name="parameters">The modeling parameters to use in calculating the KMY value.</param>
        /// <param name="priceEstimates">The low, average, or high fuel price estimates to use.</param>
        /// <param name="mfr">The manufacturer from which to obtain the manufacturer-specific discount rate.</param>
        /// <param name="useOCPayback">Specifies whether to use the regular payback period, or the payback period to apply for
        ///   over-compliance.</param>
        KMYType(ModelYear year, Parameters parameters, Estimates priceEstimates, Manufacturer mfr, bool useOCPayback)
        {
            // initialize values
            this.ModelYear = year;
            this.Values    = new VCXValue<FuelValue>();
            //
            this.CalculateKMY(year, parameters, priceEstimates, mfr, useOCPayback, VehicleClass.LDV   , VehicleStyle.None        , HEVType.None);
            this.CalculateKMY(year, parameters, priceEstimates, mfr, useOCPayback, VehicleClass.LDT12a, VehicleStyle.Van         , HEVType.None);
            this.CalculateKMY(year, parameters, priceEstimates, mfr, useOCPayback, VehicleClass.LDT12a, VehicleStyle.SportUtility, HEVType.None);
            this.CalculateKMY(year, parameters, priceEstimates, mfr, useOCPayback, VehicleClass.LDT12a, VehicleStyle.Pickup      , HEVType.None);
            this.CalculateKMY(year, parameters, priceEstimates, mfr, useOCPayback, VehicleClass.LDT12a, VehicleStyle.None        , HEVType.None);
            this.CalculateKMY(year, parameters, priceEstimates, mfr, useOCPayback, VehicleClass.LDV   , VehicleStyle.None        , HEVType.PureElectric);
            this.CalculateKMY(year, parameters, priceEstimates, mfr, useOCPayback, VehicleClass.LDT2b3, VehicleStyle.None        , HEVType.None);
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current KMYType instance.
        /// </summary>
        /// <returns>A new object that is a copy of this KMYType.</returns>
        internal KMYType Clone()
        {
            KMYType kmy = new KMYType();
            //
            kmy.ModelYear = new ModelYear(this.ModelYear.Year);
            kmy.Values    = this.Values.Clone();
            //kmy._cars     = this._cars;
            //kmy._vans     = this._vans;
            //kmy._suvs     = this._suvs;
            //kmy._pickups  = this._pickups;
            //kmy._class12a = this._class12a;
            //kmy._class2b3 = this._class2b3;
            //kmy._class456 = this._class456;
            //
            return kmy;
        }

        /// <summary>
        /// Returns the string representation of this <see cref="KMYType"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="KMYType"/> instance.</returns>
        public override string ToString()
        {
            return "{ModelYear=" + this.ModelYear.ToString() +
                   ", Values="   + this.Values   .ToString() + "}";
        }

        void CalculateKMY(ModelYear year, Parameters parameters, Estimates priceEstimates, Manufacturer mfr, bool useOCPayback,
            VehicleClass vehClass, VehicleStyle vehStyle, HEVType hevType)
        {
            // calculate kmy value
            VehicleAgeData milesDriven   = parameters.MilesDriven;
            VehicleAgeData survivalRates = parameters.SurvivalRates;
            FuelPrices     fuelPrices    = parameters.FuelPrices;
            EconomicValues econData      = parameters.EconomicValues;

            double discRate      = mfr.Description.DiscountRate[vehClass, vehStyle, hevType];
            double paybackPeriod = (useOCPayback) ? mfr.Description.PaybackPeriod_OC[vehClass, vehStyle, hevType] :
                                                    mfr.Description.PaybackPeriod   [vehClass, vehStyle, hevType];
            int    payback       = (int)Math.Floor(paybackPeriod);
            double fract         = paybackPeriod - payback;         // stores the fraction of a year
            int    yrYear        = year.Year;

            // compute energy density ratio to convert from $/kwh or $/scf to $/gallon (GEG)
            FuelValue energyDensity = parameters.FuelProperties.EnergyDensity;
            double elcRatio = energyDensity.Gasoline / energyDensity.Electricity;
            double hgnRatio = energyDensity.Gasoline / energyDensity.Hydrogen;
            double cngRatio = energyDensity.Gasoline / energyDensity.CNG;

            // calculate kmy for whole years
            FuelValue fuelCost = new FuelValue();
            //
            for (int j = 0; j <= payback; j++)
            {
                if (j == payback && fract == 0) { continue; }   // this line prevents extra calcs if there is no partial year

                // get survival
                double surv = survivalRates.GetVehicleAgeData(vehClass, vehStyle, hevType)[j];
                // get miles and apply growth rates
                double vmtGrowth = econData.VMTGrowthRate[vehClass].GetVMTGrowthFactor(priceEstimates, yrYear, j);
                double miles = milesDriven.GetVehicleAgeData(vehClass, vehStyle, hevType)[j] * vmtGrowth;
                double multiplier = surv * miles / Math.Pow(1 + discRate, j);
                // multiply by fuel price in each fuel
                foreach (FuelType fuel in FTValue<object>.Classes)
                {
                    double fuelPrice = fuelPrices.GetFuelPrice(priceEstimates, fuel, year.Year);
                    // apply special logic for non-liquid fuel types (to convert to $/gallon (GEG))
                    if      (fuel == FuelType.Electricity) { fuelCost[fuel] = fuelPrice * multiplier * elcRatio; }
                    else if (fuel == FuelType.Hydrogen   ) { fuelCost[fuel] = fuelPrice * multiplier * hgnRatio; }
                    else if (fuel == FuelType.CNG        ) { fuelCost[fuel] = fuelPrice * multiplier * cngRatio; }
                    else                                   { fuelCost[fuel] = fuelPrice * multiplier; }
                }

                // add in fuel cost to the running sum (for partial years, account for fraction of fuel cost)
                this[vehClass, vehStyle, hevType] += ((j == payback) ? fuelCost * fract : fuelCost);
            }
        }

        /// <summary>
        /// Returns the calculated KMY values for all available model years, using the provided parameters, fuel price estimates,
        /// and the manufacturer-specific discount rate obtained from the given manufacturer.
        /// </summary>
        /// <param name="minYear">The minimum year available for modeling.</param>
        /// <param name="maxYear">The maximum year available for modeling.</param>
        /// <param name="parameters">The modeling parameters to use in calculating the KMY value.</param>
        /// <param name="priceEstimates">The low, average, or high fuel price estimates to use.</param>
        /// <param name="mfr">The manufacturer from which to obtain the number of available years and the manufacturer-specific
        ///   discount rate.</param>
        /// <param name="useOCPayback">Specifies whether to use the regular payback period, or the payback period to apply for
        ///   over-compliance.</param>
        /// <returns>The calculated KMY values for all available model years.</returns>
        internal static KMYType[] GetKmyValues(int minYear, int maxYear,
            Parameters parameters, Estimates priceEstimates, Manufacturer mfr, bool useOCPayback)
        {
            int numYears = maxYear - ModelYear.MinYear + 1;
            KMYType[] kmy = new KMYType[numYears];
            for (int i = 0; i < numYears; i++)
            {
                kmy[i] = new KMYType(ModelYear.NewModelYearFromIndex(i), parameters, priceEstimates, mfr, useOCPayback);
            }
            return kmy;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the price of fuel for the remaining lifetime of the vehicle, for the specified <see cref="VehicleClass"/>
        ///   and <see cref="VehicleStyle"/>.</summary>
        public FuelValue this[VehicleClass vehClass, VehicleStyle vehStyle, HEVType hevType]
        {
            get  { return this.Values[vehClass, vehStyle, hevType]; }
            private set { this.Values[vehClass, vehStyle, hevType] = value; }
        }

        /// <summary>Gets the model year associated with this instance.</summary>
        public ModelYear ModelYear { get; private set; }
        /// <summary>Gets the values associated with this instance.</summary>
        public VCXValue<FuelValue> Values { get; private set; }

        #endregion


        #region /*** Variables ***/

        //FuelValue _cars;
        //FuelValue _vans;
        //FuelValue _suvs;
        //FuelValue _pickups;
        //FuelValue _class12a;
        //FuelValue _class2b3;
        //FuelValue _class456;

        #endregion

    }
}
