using System;
using System.Collections.Generic;
using Volpe.Cafe;
using Volpe.Cafe.Utils;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents a vehicle transmission.
    /// </summary>
    [Serializable]
    public sealed class Transmission : Component
    {

        #region /*** NestedTypes ***/

        /// <summary>
        /// Provides a description of engineering characteristics for a <see cref="Transmission"/>.
        /// </summary>
        [Serializable]
        public sealed class CDescription
        {

            #region /*** Constructors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Transmission.CDescription"/> class.
            /// </summary>
            internal CDescription() : base() { }

            #endregion


            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Transmission.CDescription"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Transmission.CDescription"/>.</returns>
            internal CDescription Clone()
            {
                CDescription td = new CDescription();
                //
                td.Code            = this.Code;
                td.Manufacturer    = this.Manufacturer;
                td.Family          = this.Family;
                td.Type            = this.Type;
                td.NumForwardGears = this.NumForwardGears;
                //
                td.UsedTechnologies      = (int[])Interaction.CloneArray(this.UsedTechnologies     , typeof(int));
                td.AvailableTechnologies = (int[])Interaction.CloneArray(this.AvailableTechnologies, typeof(int));
                //
                return td;
            }

            /// <summary>
            /// Returns the string representation of this <see cref="Transmission.CDescription"/> instance.
            /// </summary>
            /// <returns>The string representation of the <see cref="Transmission.CDescription"/> instance.</returns>
            public override string ToString()
            {
                return
                    (this.Type == "A" || this.Type == "M") ? this.Type + this.NumForwardGears :
                    (this.Type == "S" || this.Type == "DCT" || this.Type == "AMT") ? "S" + this.NumForwardGears :
                    (this.Type.StartsWith("CVT")) ? "CVT" :
                    (this.Type.StartsWith("HEV")) ? "HEV" : this.Type;
            }

            #endregion


            #region /*** Variables ***/

            /// <summary>Represents the internal manufacturer specific transmission code.</summary>
            public int Code;
            /// <summary>Specifies the manufacturer name that uses the transmission.</summary>
            public string Manufacturer;
            /// <summary>Specifies the family that the transmission belongs to.</summary>
            public string Family;
            /// <summary>Represents the type of the transmission (A, M, AMT, or CVT).</summary>
            public string Type;
            /// <summary>Represents the number of forward gears the transmission has.  On CVT transmissions, this value should be 0.</summary>
            public int NumForwardGears;

            /// <summary>Represents the indexes of technologies initially installed on the baseline transmission.</summary>
            public int[] UsedTechnologies;
            /// <summary>Represents the indexes of technologies initially available for applicability on the baseline transmission.</summary>
            public int[] AvailableTechnologies;

            #endregion

        }

        #endregion


        #region /*** Constructors ***/

        // Private constructor used for internal usage (such as cloning).
        Transmission() : base() { }

        /// <summary>
        /// Initializes a new instance of the <see cref="Transmission"/> class, using the specified transmission <see cref="Description"/>.
        /// </summary>
        /// <param name="description">A description of engineering characteristics for this <see cref="Transmission"/>.</param>
        internal Transmission(Transmission.CDescription description) : base()
        {
            this._description = description;
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Transmission"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Transmission"/>.</returns>
        internal Transmission Clone()
        {
            Transmission trn = new Transmission();
            trn._description = this._description.Clone();
            this.CopyTo(trn);
            return trn;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>
        /// Gets an array of technology indexes that are applicable to <see cref="Transmission"/>s.
        /// </summary>
        /// <returns>An array of technology indexes that are applicable to <see cref="Transmission"/>s.</returns>
        public override int[] TechnologyList { get { return TechnologyIndexes.TransmissionLevel; } }

        /// <summary>Gets a description of engineering characteristics for this <see cref="Transmission"/>.</summary>
        public Transmission.CDescription Description { get { return this._description; } }

        #endregion


        #region /*** Variables ***/

        Transmission.CDescription _description;

        #endregion

    }
}
