﻿#region << Using Directives >>
using System;
using System.Collections.Generic;
using Volpe.Cafe.Data;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Generic;
//
using TI = Volpe.Cafe.TechnologyIndexes;
#endregion

namespace Volpe.Cafe.IO.InputParsers
{
    /// <summary>
    /// Provides an input parser for loading and parsing Excel Market-Data files.
    /// </summary>
    sealed class XlMarketDataParser : XlParser
    {

        #region /*** Ctors ***/

        internal XlMarketDataParser(string path, string password) : base(path, password, true) { }

        #endregion


        #region /*** Methods ***/

        protected override void ParseInternal()
        {
            // init data ...
            this._ind = null;

            // begin parsing ...
            string[] sheets  = this.SheetNames;
            string[] names   = {"Manufacturers", "Vehicles", "Engines", "Transmissions"};
            int   [] indexes = {-1, -1, -1, -1};

            // scan each worksheet name, looking for a match
            for (int i = 0; i < sheets.Length; i++)
            {
                for (int j = 0; j < names.Length; j++)
                {
                    if (Interaction.StringCompare(sheets[i], names[j], false)) { indexes[j] = i; break; }
                }
            }
            // check if all sheets were found
            if (!this.VerifyIndexes("Market Data workbook", indexes, names)) { return; }

            // ------------------------------------------
            // nothing is missing--proceed to parse data 
            // ------------------------------------------

            // parse worksheets
            List<Manufacturer> mc = this.LoadManufacturers(indexes[0]);
            // early termination if errors were encountered
            if (this.HasErrors) { return; }
            //
            List<Vehicle     >[] vc = this.LoadVehicles     (indexes[1], mc);
            List<Engine      >[] ec = this.LoadEngines      (indexes[2], mc);
            List<Transmission>[] tc = this.LoadTransmissions(indexes[3], mc);
            // early termination if errors were encountered
            if (this.HasErrors) { return; }

            // initialize and validate each manufacturer
            int indMinYear = int.MaxValue, indMaxYear = -1;
            for (int i = 0; i < mc.Count; i++)
            {
                int minYear, maxYear;
                mc[i].Initialize(vc[i], ec[i], tc[i], out minYear, out maxYear);
                //
                if (mc[i].VehicleCount == 0)
                {
                    this.LogError("Manufacturer Code " + mc[i].Description.Code + " does not contain any valid vehicles.");
                    return;
                }
                else
                {   // this mfr has vehicles --
                    //
                    // * first, check that each vehicle's fuel-econ and fuel-share matches the engine's fuel type
                    // * then, add it to the mfrs list
                    //
                    // check vehs and eng FE
                    for (int j = 0, vehCount = vc[i].Count; j < vehCount; j++)
                    {
                        Vehicle veh = vc[i][j];
                        Vehicle.CDescription vd = veh.Description;
                        FuelType engFuel = (veh.Engine == null) ? FuelType.None : veh.Engine.Description.Fuel;
                        string hdr = "Vehicle (Code:" + vd.Code + ",Mfr:" + vd.Manufacturer + "):  ";
                        //
                        if (engFuel != FuelType.None)
                        {   // engine specifies a fuel type -- check if vehicle has FS and FE at that fuel
                            if (vd.FuelShare  [engFuel] <= 0) { this.LogError(hdr + "Engine is listed as using " + engFuel + " Fuel, but Vehicle " + engFuel + " Fuel Share is not specified or not valid."  ); }
                            if (vd.FuelEconomy[engFuel] <= 0) { this.LogError(hdr + "Engine is listed as using " + engFuel + " Fuel, but Vehicle " + engFuel + " Fuel Economy is not specified or not valid."); }
                        }
                        else
                        {   // vehicle does not list an engine -- ensure the following primary fuels are
                            // not in use for FE and FS:
                            //  - gasoline, diesel, cng, lng, lpg
                            FuelType[] engOnlyFuels = new FuelType[] { FuelType.Gasoline, FuelType.Diesel, FuelType.CNG };
                            foreach (FuelType fuel in engOnlyFuels)
                            {
                                if (vd.FuelShare  [fuel] > 0) { this.LogError(hdr + "Vehicle specifies a " + fuel + " Fuel Share, but the Engine is not listed as using "   + fuel + " Fuel."); }
                                if (vd.FuelEconomy[fuel] > 0) { this.LogError(hdr + "Vehicle specifies a " + fuel + " Fuel Economy, but the Engine is not listed as using " + fuel + " Fuel."); }
                            }
                        }

                        //foreach (FuelType fuel in FTValue<object>.Classes)
                        //{
                        //    if ((engFuel & fuel) == fuel)
                        //    {   // engine uses specific fuel
                        //        if (vd.FuelShare  [fuel] <= 0) { this.LogError(hdr + "Engine is listed as using " + fuel + " Fuel, but Vehicle " + fuel + " Fuel Share is not specified or not valid."  ); }
                        //        if (vd.FuelEconomy[fuel] <= 0) { this.LogError(hdr + "Engine is listed as using " + fuel + " Fuel, but Vehicle " + fuel + " Fuel Economy is not specified or not valid."); }
                        //    }
                        //    else if (fuel != FuelType.Electricity && fuel != FuelType.Hydrogen && (engFuel & fuel) != fuel)
                        //    {   // engine does NOT use specific fuel
                        //        if (vd.FuelShare  [fuel] > 0) { this.LogError(hdr + "Vehicle specifies a " + fuel + " Fuel Share, but the Engine is not listed as using "   + fuel + " Fuel."); }
                        //        if (vd.FuelEconomy[fuel] > 0) { this.LogError(hdr + "Vehicle specifies a " + fuel + " Fuel Economy, but the Engine is not listed as using " + fuel + " Fuel."); }
                        //    }
                        //}
                    }
                    // early termination if errors were encountered
                    if (this.HasErrors) { return; }

                    // if no errors, add mfr to the list
                    mc[i].Index = i;
                    indMinYear = Math.Min(minYear, indMinYear);
                    indMaxYear = Math.Max(maxYear, indMaxYear);
                }
            }

            // populate the industry
            this._ind = new Industry(mc, indMinYear, indMaxYear);
            this._ind.UpdateMinMaxFpCw();
        }

        // ----- Methods for parsing the manufacturers worksheet -----
        List<Manufacturer> LoadManufacturers(int index)
        {
            // activate the worksheet
            this.ActivateWorksheet(index);

            //
            // ----- detect column indexes -----
            //
            int rows = this.Rows;
            int cols = this.Columns;
            string[] mainNames = {  // data cols
                                    "Manufacturer Code" ,   //  0
                                    "Manufacturer Name" ,   //  1
                                    "Prefer Fines"      ,   //  2
                                    "Discount Rate"     ,   //  3
                                    "Payback Period"    ,   //  4
                                    "Payback Period (OC)",  //  5
                                    "CA+S177 Sales (%)" ,   //  6
                                    "CA+S177 ZEV (%)"       //  7
                                };
            string[] bcNames = {
                                    "PC-2010" , "PC-2011" , "PC-2012" , "PC-2013" , "PC-2014",
                                    "LT-2010" , "LT-2011" , "LT-2012" , "LT-2013" , "LT-2014",
                                    "2B3-2010", "2B3-2011", "2B3-2012", "2B3-2013", "2B3-2014"
                               };
            string[] ffvNames = {
                                    "PC-2015" , "PC-2016" , "PC-2017" , "PC-2018" , "PC-2019",
                                    "LT-2015" , "LT-2016" , "LT-2017" , "LT-2018" , "LT-2019",
                                    "2B3-2015", "2B3-2016", "2B3-2017", "2B3-2018", "2B3-2019"
                                };
            int[] mainIndexes = new int[mainNames.Length];
            int[] bcIndexes   = new int[bcNames  .Length];
            int[] ffvIndexes  = new int[ffvNames .Length];
            for (int i = 0; i < mainIndexes.Length; i++) { mainIndexes[i] = -1; }
            for (int i = 0; i < bcIndexes  .Length; i++) { bcIndexes  [i] = -1; }
            for (int i = 0; i < ffvIndexes .Length; i++) { ffvIndexes [i] = -1; }

            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column0 = this.GetString(0, i);
                string column1 = this.GetString(1, i);

                if      (Interaction.StringCompare(column1, mainNames[0], true)) { mainIndexes[0] = i; }
                else if (Interaction.StringCompare(column1, mainNames[1], true)) { mainIndexes[1] = i; }
                else if (Interaction.StringCompare(column1, mainNames[2], true)) { mainIndexes[2] = i; }
                else if (Interaction.StringCompare(column0, mainNames[3], true)) { mainIndexes[3] = i; }
                else if (Interaction.StringCompare(column0, mainNames[4], true)) { mainIndexes[4] = i; }
                else if (Interaction.StringCompare(column0, mainNames[5], true)) { mainIndexes[5] = i; }
                else if (Interaction.StringCompare(column1, mainNames[6], true)) { mainIndexes[6] = i; }
                else if (Interaction.StringCompare(column1, mainNames[7], true)) { mainIndexes[7] = i; }
                else
                {
                    for (int j = 0; j < bcNames.Length; j++)
                    {
                        if (Interaction.StringCompare(column1, bcNames[j], true)) { bcIndexes[j] = i; break; }
                    }
                    for (int j = 0; j < ffvNames.Length; j++)
                    {
                        if (Interaction.StringCompare(column1, ffvNames[j], true)) { ffvIndexes[j] = i; break; }
                    }
                }
            }

            // check if all required columns were found
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", mainIndexes, mainNames)) { return null; }
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet (Banked Credits)", bcIndexes  , bcNames  )) { return null; }
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet (FFV Credits)", ffvIndexes, ffvNames)) { return null; }
            // check disc rate and payback periods (ensure all classes are present)
            for (int i = 0; i < 3; i++)
            {
                for (int j = 0; j < VCXValue<object>.Names.Length; j++)
                {
                    int colIndex = mainIndexes[3 + i] + j; // get DR or PB column index
                    if (colIndex >= cols) { this.LogError("On the Manufacturers tab, the " + mainNames[3 + i] + " section is not set up correctly."); }
                    //
                    string column  = this.GetString(1, mainIndexes[3 + i] + j);
                    string vcxName = VCXValue<object>.Names[j];
                    if (!Interaction.StringCompare(column, vcxName, true))
                    {
                        this.LogError("On the Manufacturers tab, the " + mainNames[3 + i] + " section is missing '" + vcxName + "' entry.");
                    }
                }
            }

            //
            // ----- load the manufacturer data -----
            //
            // initialize manufacturer list
            List<Manufacturer> mc = new List<Manufacturer>(32);
            //
            // load the manufacturer data -- the first two rows are headers
            for (int i = 2; i < rows; i++)
            {
                Manufacturer.CDescription md = this.LoadManufacturer(i, mainIndexes, bcIndexes, ffvIndexes);
                if (md != null) { mc.Add(new Manufacturer(md)); }
            }
            //
            return mc;
        }
        Manufacturer.CDescription LoadManufacturer(int row, int[] mainIndexes, int[] bcIndexes, int[] ffvIndexes)
        {
            // ignore lines where the mfr-code or mfr-name are not valid
            int    mfrCode = this.GetInt32 (row, mainIndexes[0]);
            string mfrName = this.GetString(row, mainIndexes[1]);
            if (mfrCode <= 0 || mfrName == string.Empty) { return null; }

            // initialize ManufacturerDescription instance
            Manufacturer.CDescription md = new Manufacturer.CDescription();
            md.DiscountRate     = new VCXValue<double>();
            md.PaybackPeriod    = new VCXValue<double>();
            md.PaybackPeriod_OC = new VCXValue<double>();
            //
            // load the core data columns
            md.Code = mfrCode;
            md.Name = mfrName;
            md.ProperName = Interaction.GetTitleCase(mfrName, 4);
            //
            // load non-essential columns
            md.PreferFines = this.GetString(row, mainIndexes[ 2]).ToUpper().Equals("Y");
            for (int i = 0; i < VCXValue<object>.Names.Length; i++)
            {
                md.DiscountRate    .Items[i] = this.GetDouble(row, mainIndexes[3] + i);
                md.PaybackPeriod   .Items[i] = this.GetDouble(row, mainIndexes[4] + i);
                md.PaybackPeriod_OC.Items[i] = this.GetDouble(row, mainIndexes[5] + i);
            }
            //
            // load banked and FFV credits sections
            md.BankedCreditsMinYear = 2010;
            md.BankedCreditsMaxYear = 2014;
            md.BankedCredits = new RCObject<double[]>(new double[5], new double[5], new double[5]);
            for (int i = 0; i < md.BankedCredits.Items.Length; i++)
            {
                for (int j = 0; j < 5; j++)
                {
                    md.BankedCredits.Items[i][j] = this.GetDouble(row, bcIndexes[i * 5 + j]);
                }
            }
            md.FFVCreditsMinYear = 2015;
            md.FFVCreditsMaxYear = 2019;
            md.FFVCredits = new RCObject<double[]>(new double[5], new double[5], new double[5]);
            for (int i = 0; i < md.FFVCredits.Items.Length; i++)
            {
                for (int j = 0; j < 5; j++)
                {
                    md.FFVCredits.Items[i][j] = this.GetDouble(row, ffvIndexes[i * 5 + j]);
                }
            }
            // load ZEV credits
            md.ZEVSalesShare  = this.GetDouble(row, mainIndexes[6]);
            md.ZEVCreditShare = this.GetDouble(row, mainIndexes[7]);
            //
            // return the mfr description
            return md;
        }

        // ----- Methods for parsing the vehicles worksheet -----
        List<Vehicle>[] LoadVehicles(int index, List<Manufacturer> mfrs)
        {
            // activate the worksheet
            this.ActivateWorksheet(index);

            //
            // ----- detect column indexes -----
            //
            int rows = this.Rows;
            int cols = this.Columns;
            string[] namesMain = new string[] {
                // data cols
                "Vehicle Code"                  ,   //  0
                "Manufacturer"                  ,   //  1
                "Brand"                         ,   //  2
                "Model"                         ,   //  3
                "Nameplate"                     ,   //  4
                "Platform"                      ,   //  5
                "Engine Code"                   ,   //  6
                "Transmission Code"             ,   //  7
                "Origin"                        ,   //  8
                "MSRP"                          ,   //  9
                "Regulatory Class"              ,   // 10
                "Technology Class"              ,   // 11
                "Engine Technology Class"       ,   // 12
                "Safety Class"                  ,   // 13
                "Style"                         ,   // 14
                "Structure"                     ,   // 15
                "Drive"                         ,   // 16
                "Wheelbase"                     ,   // 17
                "Footprint"                     ,   // 18
                //"Work Factor"                 ,   // --
                "Curb Weight"                   ,   // 19
                "GVWR"                          ,   // 20
                "GCWR"                          ,   // 21
                //"Test Weight"                 ,   // --
                "Fuel Capacity"                 ,   // 22
                "Vehicle Power"                 ,   // 23
                "Vehicle Torque"                ,   // 24
                "Electric Range"                ,   // 25
                "Type of Hybrid/Electric Vehicle",  // 26
                "Refresh Years"                 ,   // 27
                "Redesign Years"                ,   // 28
                "Employment Hours per Vehicle"  ,   // 29
                "Max GVWR/CW"                   ,   // 30
                "Max GCWR/GVWR"                 ,   // 31
                "ZEV Candidate"                     // 32
            };
            string[] namesFuel = new string[] {
                // fuel economy
                "Fuel Economy (G)"              ,   //  0
                "Fuel Economy (E85)"            ,   //  1
                "Fuel Economy (D)"              ,   //  2
                "Fuel Economy (E)"              ,   //  3
                "Fuel Economy (H)"              ,   //  4
                "Fuel Economy (CNG)"            ,   //  5
                // fuel share
                "Fuel Share (G)"                ,   //  6
                "Fuel Share (E85)"              ,   //  7
                "Fuel Share (D)"                ,   //  8
                "Fuel Share (E)"                ,   //  9
                "Fuel Share (H)"                ,   // 10
                "Fuel Share (CNG)"                  // 11
            };
            string[] namesProd = new string[] { "Sales" };
            string[] namesTech = new string[TI.TechNamesVehicle.Length + TI.TechNamesPlatform.Length];
            Array.Copy(TI.TechNamesVehicle , 0, namesTech, 0                         , TI.TechNamesVehicle .Length);
            Array.Copy(TI.TechNamesPlatform, 0, namesTech, TI.TechNamesVehicle.Length, TI.TechNamesPlatform.Length);
            //
            int[] indexesMain = new int[namesMain.Length];
            int[] indexesFuel = new int[namesFuel.Length];
            int[] indexesProd = new int[namesProd.Length];
            int[] indexesTech = new int[namesTech.Length];
            for (int i = 0; i < indexesMain.Length; i++) { indexesMain[i] = -1; }
            for (int i = 0; i < indexesFuel.Length; i++) { indexesFuel[i] = -1; }
            for (int i = 0; i < indexesProd.Length; i++) { indexesProd[i] = -1; }
            for (int i = 0; i < indexesTech.Length; i++) { indexesTech[i] = -1; }

            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column  = this.GetString(1, i);
                string prodHC = this.GetString(0, i); // production header column
                for (int j = 0; j < namesMain.Length; j++)
                {
                    if (indexesMain[j] == -1 && Interaction.StringCompare(column , namesMain[j], true)) { indexesMain[j] = i; break; }
                }
                for (int j = 0; j < namesFuel.Length; j++)
                {
                    if (indexesFuel[j] == -1 && Interaction.StringCompare(column, namesFuel[j], true)) { indexesFuel[j] = i; break; }
                }
                for (int j = 0; j < namesProd.Length; j++)
                {
                    if (indexesProd[j] == -1 && Interaction.StringCompare(prodHC, namesProd[j], true)) { indexesProd[j] = i; break; }
                }
                for (int j = 0; j < namesTech.Length; j++)
                {
                    if (indexesTech[j] == -1 && Interaction.StringCompare(column, namesTech[j], true)) { indexesTech[j] = i; break; }
                }
            }

            // check if all required columns were found
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesMain, namesMain)) { return null; }
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesFuel, namesFuel)) { return null; }
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesProd, namesProd)) { return null; }
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesTech, namesTech)) { return null; }

            // check sales, msrp, and price columns
            int minYear, maxYear;
            this.GetMinMaxYears(1, indexesProd[0], out minYear, out maxYear);

            //
            // ----- load the vehicle data -----
            //
            // initialize vehicle collections (one per mfr)
            List<Vehicle>[] vc = new List<Vehicle>[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                vc[i] = new List<Vehicle>(32);
            }
            //
            // load the vehicle data -- (the first two rows are headers)
            for (int i = 2; i < rows; i++)
            {   // get the vehicle data at the current row
                int mfrIndex;
                Vehicle.CDescription vd = this.LoadVehicle(i, indexesMain, indexesFuel, indexesProd, indexesTech, namesTech,
                    minYear, maxYear, mfrs, out mfrIndex);
                // add the vehicle to the list for the mfr (if mfr-index is found)
                if (vd != null && mfrIndex != -1) { vc[mfrIndex].Add(new Vehicle(vd)); }
            }
            //
            return vc;
        }
        Vehicle.CDescription LoadVehicle(int row, int[] indexesMain, int[] indexesFuel, int[] indexesProd, int[] indexesTech,
            string[] namesTech, int minYear, int maxYear, List<Manufacturer> mfrs, out int mfrIndex)
        {
            // initialize VehicleDescription instance
            mfrIndex = -1;
            Vehicle.CDescription vd = new Vehicle.CDescription();
            //
            // brief checks of the data
            string vehCode = this.GetString(row, indexesMain[0]);
            string mfrName = this.GetString(row, indexesMain[1]);
            string brand   = this.GetString(row, indexesMain[2]);
            string model   = this.GetString(row, indexesMain[3]);
            // if vehicle code, mfr, and model columns are blank, assume the row is empty
            if (vehCode == string.Empty && mfrName == string.Empty && model == string.Empty) { return null; }
            // if the vehicle code is X-ed out, assume we want to skip it from applicability
            if (vehCode == "X") { return null; }
            //
            // load the data columns
            vd.Code                              = Interaction.GetInt32(vehCode);//[0]
            vd.Manufacturer                      = mfrName;                     //[ 1]
            vd.Brand                             = brand;                       //[ 2]
            vd.Model                             = model;                       //[ 2]
            vd.Nameplate                         = this.GetString(row, indexesMain[ 4]);
            vd.Platform                          = this.GetString(row, indexesMain[ 5]);
            vd.EngineCode                        = this.GetInt32 (row, indexesMain[ 6]);
            vd.TransmissionCode                  = this.GetInt32 (row, indexesMain[ 7]);
            vd.Origin                            = this.GetChar  (row, indexesMain[ 8]);
            vd.Msrp                              = this.GetDouble(row, indexesMain[ 9]);
            //
            vd.RegulatoryIndicator               = this.GetString(row, indexesMain[10]);
            vd.TechnologyClass                   = this.GetString(row, indexesMain[11]);
            vd.EngineTechnologyClass             = this.GetString(row, indexesMain[12]);
            vd.SafetyClass                       = this.GetString(row, indexesMain[13]);
            vd.Style                             = this.GetString(row, indexesMain[14]);
            vd.Structure                         = this.GetString(row, indexesMain[15]);
            vd.Drive                             = this.GetChar  (row, indexesMain[16]);
            vd.Wheelbase                         = this.GetDouble(row, indexesMain[17]);
            vd.Footprint                         = this.GetDouble(row, indexesMain[18]);
            vd.CurbWeight                        = this.GetDouble(row, indexesMain[19]);
            vd.GVWR                              = this.GetDouble(row, indexesMain[20]);
            vd.GCWR                              = this.GetDouble(row, indexesMain[21]);
            vd.FuelCapacity                      = this.GetDouble(row, indexesMain[22]);
            vd.VehiclePower                      = this.GetDouble(row, indexesMain[23]);
            vd.VehicleTorque                     = this.GetDouble(row, indexesMain[24]);
            vd.VehicleRange                      = this.GetDouble(row, indexesMain[25]);
            vd.HybridType                        = this.GetString(row, indexesMain[26]);
            vd.RefreshYears                      = this.LoadRefreshRedesignYears(this.GetString(row, indexesMain[27]));
            vd.RedesignYears                     = this.LoadRefreshRedesignYears(this.GetString(row, indexesMain[28]));
            vd.EmploymentHours                   = this.GetDouble(row, indexesMain[29]);
            vd.MaxGVWRToCW                       = this.GetDouble(row, indexesMain[30]);
            vd.MaxGCWRToGVWR                     = this.GetDouble(row, indexesMain[31]);
            //
            string zevCandidate                  = this.GetString(row, indexesMain[32]);
            vd.ZEVCandidate                      = (zevCandidate == "PHEV") ? HEVType.PlugInHybrid :
                                                   (zevCandidate == "EV"  ) ? HEVType.PureElectric : HEVType.None;
            //
            // load fuel economy and fuel share data
            for (int i = 0, fuelCount = FTValue<object>.Classes.Length; i < fuelCount; i++)
            {
                FuelType fuel = FTValue<object>.Classes[i];
                vd.FuelEconomy[fuel] = this.GetDouble(row, indexesFuel[i            ]);
                vd.FuelShare  [fuel] = this.GetDouble(row, indexesFuel[i + fuelCount]);
            }
            vd.BaseFE = vd.FuelEconomy;   // copy over fuel economy into base-FE
            //
            // load sales, msrp, and price
            vd.Sales = this.LoadVehicleSMP(row, indexesProd[0], minYear, maxYear);
            //
            // load technology overrides
            this.LoadTechApplicability(row, namesTech, indexesTech, out vd.UsedTechnologies, out vd.AvailableTechnologies);
            //
            // validate ...
            //  - strong/plug-in hybrid vehicles should not have a transmission present
            //  - electricity-only or hydrogen-only vehicles should not have an engine or transmission present
            bool skipTrn = false;
            bool skipEng = false;
            for (int i = 0; i < vd.UsedTechnologies.Length; i++)
            {
                //if (TI.IsConversionToSHEV(vd.UsedTechnologies[i])) { isSHEV = true; }
                if (TI.IsConversionToPHEV(vd.UsedTechnologies[i])) { skipTrn = true; }
                if (TI.IsConversionToEV  (vd.UsedTechnologies[i]) ||
                    TI.IsConversionToFCV (vd.UsedTechnologies[i])) { skipTrn = skipEng = true; }
            }
            //
            string hdr = "Vehicle (R:" + (row + 1).ToString("0000") + "):  ";
            mfrIndex = this.GetManufacturerIndex(mfrs, vd.Manufacturer);
            if (vd.Code < 1) { this.LogError(hdr + "Vehicle Code is not valid."); }
            if (mfrIndex == -1) { this.LogError(hdr + "Manufacturer is not specified or not valid."); }
            if (vd.Brand == string.Empty) { this.LogError(hdr + "Brand is not specified."); }
            if (vd.Model == string.Empty) { this.LogError(hdr + "Model is not specified."); }
            if (vd.Nameplate == string.Empty) { vd.Nameplate = vd.Model; }
            if (vd.Nameplate == string.Empty) { this.LogError(hdr + "Nameplate is not specified."); }
            if (vd.EngineCode < 1 && !skipEng) { this.LogError(hdr + "Engine Code is not valid."); }
            if (vd.EngineCode > 0 &&  skipEng) { this.LogError(hdr + "Engine Code cannot be specified for pure-Electric or Hydrogen-based vehicles."); }
            if (vd.TransmissionCode < 1 && !skipTrn) { this.LogError(hdr + "Transmission Code is not valid."); }
            if (vd.TransmissionCode > 0 &&  skipTrn) { this.LogError(hdr + "Transmission Code cannot be specified for Plug-In Hybrid, Electric, or Fuel Cell vehicles."); }
            if (vd.Origin == '\0') { vd.Origin = 'D'; }
            if (vd.Origin != 'I' && vd.Origin != 'D') { this.LogError(hdr + "Origin is not valid."); }
            //
            if (!vd.FuelEconomy.IsValid) { this.LogError(hdr + "Fuel Economy is not valid (the Fuel Economy value cannot be negative)."); }
            if ( vd.FuelEconomy.IsEmpty) { this.LogError(hdr + "Fuel Economy is not valid (at least one Fuel Economy value must be specified)."); }
            if (!vd.FuelShare  .IsValid) { this.LogError(hdr + "Fuel Share is not valid (the Fuel Share value cannot be negative)."); }
            if ( vd.FuelShare  .IsEmpty) { this.LogError(hdr + "Fuel Share is not valid (at least one Fuel Share value must be specified)."); }
            if ( vd.FuelShare  .Total != 1) { this.LogError(hdr + "Fuel Share is not valid (Fuel Shares for all fuel types must add up to 100%)."); }
            foreach (FuelType fuelType in FTValue<object>.Classes)
            {
                if ((vd.FuelEconomy[fuelType] == 0 && vd.FuelShare[fuelType] != 0) ||
                    (vd.FuelEconomy[fuelType] != 0 && vd.FuelShare[fuelType] == 0))
                {
                    this.LogError(hdr + fuelType + " Fuel Economy and Fuel Share do not correspond.");
                }
            }
            //
            if (vd.RegulatoryIndicator == "2B3") { vd.RegulatoryIndicator = "LT2B3"; }
            if (!Interaction.StringCompareAny(vd.RegulatoryIndicator, new string[] {"PC", "LT", "LT2B3"}, false)) { this.LogError(hdr + "Regulatory Indicator is not valid."); }
            if (!Interaction.StringCompareAny(vd.TechnologyClass, TechnologyClasses.ClassNames, true)) { this.LogError(hdr + "Technology Class is not valid."); }
            if (!skipEng &&
                !Interaction.StringCompareAny(vd.EngineTechnologyClass, TechnologyClasses.EngineClassNames, true)) { this.LogError(hdr + "Engine Technology Class is not valid."); }
            //if (vd.AutoNewsMarketSegment < 1) { this.LogError(hdr + "Market Segment (Auto News) is not specified or not valid."); }
            if (!Interaction.StringCompareAny(vd.SafetyClass, new string[] { "PC", "LT", "CM" }, true)) { this.LogError(hdr + "Safety Class is not valid."); }
            if (!Interaction.StringCompareAny(zevCandidate, new string[] { "PHEV", "EV", "" }, false)) { this.LogError(hdr + "ZEV Candidate is not valid."); }
            if (vd.Style == string.Empty) { this.LogError(hdr + "Style is not specified."); }
            else if (!Interaction.StringCompareAny(vd.Style, new string[] { "Convertible", "Coupe", "Hatchback", "Sedan", "Wagon", "Sport Utility", "Minivan", "Van", "Passenger Van", "Cargo Van", "Pickup", "Large Pickup", "Chassis Cab", "Cutaway" }, false)) { this.LogError(hdr + "Style is not valid."); }
            if (vd.Structure == string.Empty) { this.LogError(hdr + "Structure is not specified."); }
            if (!Interaction.StringCompareAny(vd.Drive.ToString(), new string[] { "F", "R", "2", "4", "A", "FWD", "RWD", "2WD", "4WD", "AWD" }, false)) { this.LogError(hdr + "Drive is not specified."); }
            if (vd.RegulatoryIndicator != "LT2B3" && vd.Wheelbase < 1) { this.LogError(hdr + "Wheelbase is not valid."); }
            if (vd.RegulatoryIndicator != "LT2B3" && vd.Footprint < 1) { this.LogError(hdr + "Footprint is not valid."); }
            //if (vd.WorkFactor < 1) { this.LogError(hdr + "Work Factor is not valid."); }
            if (vd.CurbWeight < 1) { this.LogError(hdr + "Curb Weight is not valid."); }
            //if (vd.TestWeight < 0) { this.LogError(hdr + "Test Weight is not valid."); }
            if (Interaction.StringCompareAny(vd.RegulatoryIndicator, new string[] {"LT", "LT2B3"}, false) && vd.GVWR < 1) { this.LogError(hdr + "GVWR is not valid."); }
            if (vd.RegulatoryIndicator == "LT2B3" && vd.GCWR < 1) { this.LogError(hdr + "GCWR is not valid."); }
            if (vd.FuelCapacity < 1 && (vd.FuelShare.Gasoline > 0 || vd.FuelShare.Diesel > 0 || vd.FuelShare.Ethanol85 > 0)) { this.LogError(hdr + "Fuel Capacity is not valid."); }
            if (vd.VehiclePower < 1) { this.LogError(hdr + "Vehicle Power is not valid."); }
            //if (vd.VehicleTorque < 1) { this.LogError(hdr + "Vehicle Torque is not valid."); }
            // vehicle range is optional for all fuel types except electric/FCV
            if ((vd.FuelShare.Electricity > 0 || vd.FuelShare.Electricity > 0) &&
                vd.VehicleRange < 1) { this.LogError(hdr + "Vehicle Range has to be specified for electric vehicles."); }
            // "Type of Hybrid/Electric Vehicle" column may be blank
            // "Refresh Years" column may be blank
            // "Redesign Years" column may be blank
            // "Percent of 2 DR Cars" column may be blank
            //if (vd.EmploymentHours < 0) { this.LogError(hdr + "Employment Hours per Vehicle is not valid."); }
            // ensure vehicle has sales in at least one model year
            bool hasSales = false;
            for (int i = 0; i < vd.Sales.Length; i++)
            {
                if (vd.Sales[i] > 0) { hasSales = true; break; }
            }
            if (!hasSales)
            {
                //this.LogError(hdr + "Vehicle does not have sales in any model year.");
                return null;
            }
            //
            // return the vehicle description
            return vd;
        }
        double[] LoadVehicleSMP(int row, int column, int minYear, int maxYear)
        {
            double[] values = new double[maxYear - ModelYear.MinYear + 1];
            int years = maxYear - minYear + 1;
            int offset = minYear - ModelYear.MinYear;
            for (int i = 0; i < years; i++)
            {
                values[i + offset] = this.GetDouble(row, i + column);
            }
            //
            return values;
        }

        // ----- Methods for parsing the engines worksheet -----
        List<Engine>[] LoadEngines(int index, List<Manufacturer> mfrs)
        {
            // activate the worksheet
            this.ActivateWorksheet(index);

            //
            // ----- detect column indexes -----
            //
            int rows = this.Rows;
            int cols = this.Columns;
            string[] namesMain = new string[] {
                "Engine Code"            ,      //  0
                "Manufacturer"           ,      //  1
                "Configuration"          ,      //  2
                "Fuel"                   ,      //  3
                "Engine Oil Viscosity"   ,      //  4
                "Cycle"                  ,      //  5
                "Fuel Delivery System"   ,      //  6
                "Aspiration"             ,      //  7
                "Valvetrain Design"      ,      //  8
                "Valve Actuation/Timing" ,      //  9
                "Valve Lift"             ,      // 10
                "Cylinders"              ,      // 11
                "Valves/Cylinder"        ,      // 12
                "Deactivation"           ,      // 13
                "Displacement"           ,      // 14
                "Engine Size"            };     // 15
            string[] namesTech = TI.TechNamesEngine;
            int[] indexesMain = new int[namesMain.Length];
            int[] indexesTech = new int[namesTech.Length];
            for (int i = 0; i < namesMain.Length; i++) { indexesMain[i] = -1; }
            for (int i = 0; i < namesTech.Length; i++) { indexesTech[i] = -1; }

            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column = this.GetString(1, i);
                for (int j = 0; j < namesMain.Length; j++)
                {
                    if (indexesMain[j] == -1 && Interaction.StringCompare(column, namesMain[j], true)) { indexesMain[j] = i; break; }
                }
                for (int j = 0; j < namesTech.Length; j++)
                {
                    if (indexesTech[j] == -1 && Interaction.StringCompare(column, namesTech[j], true)) { indexesTech[j] = i; break; }
                }
            }

            // check if all required columns were found
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesMain, namesMain)) { return null; }
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesTech, namesTech)) { return null; }

            //
            // ----- load the engine data -----
            //
            // initialize engine collections (one per mfr)
            List<Engine>[] ec = new List<Engine>[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                ec[i] = new List<Engine>(32);
            }
            //
            // load the engine data -- (the first two rows are headers)
            for (int i = 2; i < rows; i++)
            {   // get the engine data at the current row
                int mfrIndex;
                Engine.CDescription ed = this.LoadEngine(i, indexesMain, indexesTech, namesTech, mfrs, out mfrIndex);
                // add the engine to the list for the mfr (if mfr-index is found)
                if (ed != null && mfrIndex != -1) { ec[mfrIndex].Add(new Engine(ed)); }
            }
            //
            return ec;
        }
        Engine.CDescription LoadEngine(int row, int[] indexesMain, int[] indexesTech, string[] namesTech, List<Manufacturer> mfrs,
            out int mfrIndex)
        {
            mfrIndex = -1;
            Engine.CDescription ed = new Engine.CDescription();
            //
            // brief checks of the data
            string engCode = this.GetString(row, indexesMain[0]);
            string mfrName = this.GetString(row, indexesMain[1]);
            // if engine code and mfr columns are blank, assume the row is empty
            if (engCode == string.Empty && mfrName == string.Empty) { return null; }
            // if the engine code is X-ed out, assume we want to skip it from applicability
            if (engCode == "X") { return null; }
            //
            // load the data columns
            ed.Code                 = Interaction.GetInt32(engCode);//[0]
            ed.Manufacturer         = mfrName;                      //[1]
            ed.Configuration        = this.GetString(row, indexesMain[ 2]);
            string fuel             = this.GetString(row, indexesMain[ 3]);
            ed.Fuel                 = (fuel == "G"    ) ? FuelType.Gasoline :
                                      (fuel == "G+E85") ? FuelType.Gasoline : // | FuelType.Ethanol85 :
                                      (fuel == "D"    ) ? FuelType.Diesel :
                                      (fuel == "D+B20") ? FuelType.Diesel : // | FuelType.Biodiesel20 :
                                      (fuel == "CNG"  ) ? FuelType.CNG :
                                                          FuelType.None;
            ed.EngineOilViscosity   = this.GetString(row, indexesMain[ 4]);
            ed.Cycle                = this.GetChar  (row, indexesMain[ 5]);
            ed.FuelSystem           = this.GetString(row, indexesMain[ 6]);
            ed.Aspiration           = this.GetString(row, indexesMain[ 7]);
            ed.ValvetrainDesign     = this.GetString(row, indexesMain[ 8]);
            ed.ValveActuationTiming = this.GetString(row, indexesMain[ 9]);
            ed.ValveLift            = this.GetString(row, indexesMain[10]);
            ed.Cylinders            = this.GetInt32 (row, indexesMain[11]);
            ed.ValvesPerCylinder    = this.GetInt32 (row, indexesMain[12]);
            ed.Deactivation         = this.GetDouble(row, indexesMain[13]); if (ed.Deactivation == 0 && this.GetString(row, indexesMain[14]) == "Y") { ed.Deactivation = 0.3; }
            ed.Displacement         = this.GetDouble(row, indexesMain[14]);
            string engSize          = this.GetString(row, indexesMain[15]);
            ed.EngineSize           = (engSize == "SMALL"  || engSize == "SD") ? EngineSize.Small  :
                                      (engSize == "MEDIUM" || engSize == "MD") ? EngineSize.Medium :
                                      (engSize == "LARGE"  || engSize == "LD") ? EngineSize.Large  :
                                                                                 EngineSize.None;
            //
            // load technology applicability
            this.LoadTechApplicability(row, namesTech, indexesTech, out ed.UsedTechnologies, out ed.AvailableTechnologies);
            //
            // validate ...
            string hdr = "Engine (R:" + (row + 1).ToString("0000") + "):  ";
            mfrIndex = this.GetManufacturerIndex(mfrs, ed.Manufacturer);
            if (ed.Code < 1) { this.LogError(hdr + "Engine Code is not valid."); }
            if (mfrIndex == -1) { this.LogError(hdr + "Manufacturer is not specified or not valid."); }
            if (ed.Configuration == string.Empty) { this.LogError(hdr + "Configuration is not specified."); }
            if (ed.Fuel == FuelType.None) { this.LogError(hdr + "Engine Fuel is not specified or not valid."); }
            //if (ed.EngineOilViscosity == string.Empty) { this.LogError(hdr + "Engine Oil Viscosity is not specified."); }
            if (ed.Cycle == '\0') { this.LogError(hdr + "Cycle is not specified."); }
            if (ed.FuelSystem == string.Empty) { this.LogError(hdr + "Fuel Delivery System is not specified."); }
            if (!Interaction.StringCompareAny(ed.Aspiration, new string[] { "NA", "S", "T", "T2", "T4", "ST", "TS" }, false)) { this.LogError(hdr + "Aspiration is not valid."); }
            if (ed.Configuration != "R")
            {
                if (!Interaction.StringCompareAny(ed.ValvetrainDesign, new string[] { "OHV", "SOHC", "DOHC" }, false)) { this.LogError(hdr + "Valvetrain Design is not valid."); }
                if (ed.ValveActuationTiming == string.Empty) { this.LogError(hdr + "Valve Actuation/Timing is not specified."); }
                if (ed.ValveLift == string.Empty) { this.LogError(hdr + "Valve Lift is not specified."); }
                if (ed.Cylinders < 1) { this.LogError(hdr + "Number of Cylinders is too small."); }
                if (ed.ValvesPerCylinder < 1) { this.LogError(hdr + "Number of Valves/Cylinders is too small."); }
            }
            if (ed.Deactivation < 0) { this.LogError(hdr + "Deactivation is not valid."); }
            if (ed.Displacement < 0) { this.LogError(hdr + "Displacement is not valid."); }
            if (ed.EngineSize == EngineSize.None) { this.LogError(hdr + "Engine Size is not specified or not valid."); }
            //
            // return the engine description
            return ed;
        }
        double GetCompressionRatio(string value)
        {
            double ratio = 0;
            if (value != "")
            {
                int index = value.IndexOf(":");
                if (index != -1)
                {   // compression ratio specified as a fraction (e.g. 12.5:1)
                    ratio = Interaction.GetDouble(value.Substring(0, index)) / Interaction.GetDouble(value.Substring(index + 1));
                }
                else
                {   // compression ratio specified as a decimal (e.g. 12.5)
                    ratio = Interaction.GetDouble(value);
                }
            }
            return ratio;
        }

        // ----- Methods for parsing the transmissions worksheet -----
        List<Transmission>[] LoadTransmissions(int index, List<Manufacturer> mfrs)
        {
            // activate the worksheet
            this.ActivateWorksheet(index);

            //
            // ----- detect column indexes -----
            //
            int rows = this.Rows;
            int cols = this.Columns;
            string[] namesMain = new string[] {
                "Transmission Code"      ,      //  0
                "Manufacturer"           ,      //  1
                "Type"                   ,      //  2
                "Number of Forward Gears"};     //  3
            string[] namesTech = TI.TechNamesTransmission;
            int[] indexesMain = new int[namesMain.Length];
            int[] indexesTech = new int[namesTech.Length];
            for (int i = 0; i < namesMain.Length; i++) { indexesMain[i] = -1; }
            for (int i = 0; i < namesTech.Length; i++) { indexesTech[i] = -1; }

            // scan each column to find the appropriate column names
            for (int i = 0; i < cols; i++)
            {
                string column = this.GetString(1, i);
                for (int j = 0; j < namesMain.Length; j++)
                {
                    if (indexesMain[j] == -1 && Interaction.StringCompare(column, namesMain[j], true)) { indexesMain[j] = i; break; }
                }
                for (int j = 0; j < namesTech.Length; j++)
                {
                    if (indexesTech[j] == -1 && Interaction.StringCompare(column, namesTech[j], true)) { indexesTech[j] = i; break; }
                }
            }

            // check if all required columns were found
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesMain, namesMain)) { return null; }
            if (!this.VerifyIndexes(this.SheetNames[index] + " worksheet", indexesTech, namesTech)) { return null; }

            //
            // ----- load the transmission data -----
            //
            // initialize transmission collections (one per mfr)
            List<Transmission>[] tc = new List<Transmission>[mfrs.Count];
            for (int i = 0; i < mfrs.Count; i++)
            {
                tc[i] = new List<Transmission>(32);
            }
            //
            // load the transmission data -- (the first two rows are headers)
            for (int i = 2; i < rows; i++)
            {   // get the transmission data at the current row
                int mfrIndex;
                Transmission.CDescription td = this.LoadTransmission(i, indexesMain, indexesTech, namesTech, mfrs, out mfrIndex);
                // add the transmission to the list for the mfr (if mfr-index is found)
                if (td != null && mfrIndex != -1) { tc[mfrIndex].Add(new Transmission(td)); }
            }
            //
            return tc;
        }
        Transmission.CDescription LoadTransmission(int row, int[] indexesMain, int[] indexesTech, string[] namesTech,
            List<Manufacturer> mfrs, out int mfrIndex)
        {
            mfrIndex = -1;
            Transmission.CDescription td = new Transmission.CDescription();
            //
            // brief checks of the data
            string trnCode = this.GetString(row, indexesMain[0]);
            string mfrName = this.GetString(row, indexesMain[1]);
            // if transmission code and mfr columns are blank, assume the row is empty
            if (trnCode == string.Empty && mfrName == string.Empty) { return null; }
            // if the transmission code is X-ed out, assume we want to skip it from applicability
            if (trnCode == "X") { return null; }
            //
            // load the data columns
            td.Code            = Interaction.GetInt32(trnCode);//[0]
            td.Manufacturer    = mfrName;                      //[1]
            td.Type            = this.GetString(row, indexesMain[ 2]);
            td.NumForwardGears = this.GetInt32 (row, indexesMain[ 3]);
            if ((td.Type == "CVT" || td.Type == "HEV") && td.NumForwardGears == 0) { td.NumForwardGears = 1; }
            //
            // load technology applicability
            this.LoadTechApplicability(row, namesTech, indexesTech, out td.UsedTechnologies, out td.AvailableTechnologies);
            //
            // validate ...
            string hdr = "Transmission (R:" + (row + 1).ToString("0000") + "):  ";
            mfrIndex = this.GetManufacturerIndex(mfrs, td.Manufacturer);
            if (td.Code < 1) { this.LogError(hdr + "Transmission Code is not valid."); }
            if (mfrIndex == -1) { this.LogError(hdr + "Manufacturer is not specified or not valid."); }
            if (!Interaction.StringCompareAny(td.Type, new string[] { "A", "M", "CVT", "HEV", "AMT", "DCT" }, false)) { this.LogError(hdr + "Type is not valid."); }
            if (td.Type != "CVT" && td.Type != "HEV" && td.NumForwardGears < 3) { this.LogError(hdr + "Number of Forward Gears is too small."); }
            //
            // return the transmission description
            return td;
        }

        // ----- helper methods for loading the tech-applicability section, which is shared among all tabs (mfrs, vehs, engs, trns) -----
        void LoadTechApplicability(
            int row, string[] namesTech, int[] indexesTech, out int[] usedTechnologies, out int[] availableTechnologies)
        {
            int   techUsedCount = 0, techAvailCount = 0;
            int   techCount = namesTech.Length;
            int[] techAp = new int[techCount];
            //
            for (int i = 0; i < techCount; i++)
            {
                techAp[i] = this.ParseTechApplicability(this.GetString(row, indexesTech[i]));
                if      (techAp[i] == 1) { techUsedCount++; }
                else if (techAp[i] == 0) { techAvailCount++; }
            }
            //
            usedTechnologies      = new int[techUsedCount];
            availableTechnologies = new int[techAvailCount];
            for (int i = 0, ut = 0, at = 0; i < techCount; i++)
            {
                int techIndex = TI.GetIndex(namesTech[i]);
                //
                if      (techAp[i] == 1) { usedTechnologies     [ut++] = techIndex; }
                else if (techAp[i] == 0) { availableTechnologies[at++] = techIndex; }
            }
        }
        int ParseTechApplicability(string value)
        {
            return (value == "SKIP" || value == "_SKIP") ? -1 : (value == "USED" || value == "_USED") ? 1 : 0;
        }

        // ----- helper methods for loading refresh/redeisgn years -----
        int[] LoadRefreshRedesignYears(string yearList)
        {
            return this.LoadRefreshRedesignYears(yearList.Split(',', ';'));
        }
        int[] LoadRefreshRedesignYears(string[] years)
        {
            int count = years.Length;
            List<int> parsedYears = new List<int>(count);
            // check the years array
            for (int i = 0; i < count; i++)
            {
                int year = Interaction.GetInt32(years[i]);
                if (year > 1900 && !parsedYears.Contains(year))
                {
                    parsedYears.Add(year);
                }
            }
            parsedYears.Sort();
            return parsedYears.ToArray();
        }

        // ----- helper method for obtaining mfr index based on mfr name -----
        int GetManufacturerIndex(List<Manufacturer> mfrs, string mfrName)
        {
            // find the manufacturer index with the specified name
            for (int i = 0; i < mfrs.Count; i++)
            {
                if (Interaction.StringCompare(mfrs[i].Description.Name, mfrName, false)) { return i; }
            }
            return -1;
        }

        #endregion


        #region /*** Variables ***/

        internal Industry _ind;

        #endregion

    }
}
