#region << Using Directives >>
using System;
using System.Collections;
using System.Collections.Generic;
using Volpe.Cafe;
using Volpe.Cafe.Data;
using Volpe.Cafe.Generic;
using Volpe.Cafe.Settings;
using RC = Volpe.Cafe.RegulatoryClass;
using TI = Volpe.Cafe.TechnologyIndexes;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides static methods to assist the compliance modeling process in performing regulatory calculations, such as
    /// standard, target, and regulatory class, as well as compliance calculations, such as manufacturer CAFE, credits, and fines.
    /// </summary>
    public sealed class Standards
    {

        #region /*** Methods ***/

        #region /* Regulatory calculations (standard, target, reg-class) */

        /// <summary>
        /// Returns the calculated CAFE fuel economy standard, in miles per gallon (mpg), for the entire industry for all
        /// regulatory class assignments of vehicles, operating under the given scenario and model year.
        /// </summary>
        /// <param name="data">The industry for which to calculate the CAFE fuel economy standard.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <returns>The calculated CAFE fuel economy standard, in miles per gallon (mpg), for the entire industry for all
        ///   regulatory class assignments of vehicles, operating under the given scenario and model year.</returns>
        public static RCDouble GetIndustryStandard(Industry data, Scenario scen, ModelYear year)
        {
            RCDouble standard = new RCDouble();
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                standard[regClass] = Standards.GetIndustryStandard(data, scen, year, regClass);
            }
            return standard;
        }
        /// <summary>
        /// Returns the calculated CAFE fuel economy standard, in miles per gallon (mpg), for the entire industry, based on the
        /// given regulatory class assignment of vehicles, operating under the given scenario and model year.
        /// </summary>
        /// <param name="data">The industry for which to calculate the CAFE fuel economy standard.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the CAFE fuel economy
        ///   standard.</param>
        /// <returns>The calculated CAFE fuel economy standard, in miles per gallon (mpg), for the entire industry, based on the
        ///   given regulatory class assignment of vehicles, operating under the given scenario and model year.</returns>
        public static double GetIndustryStandard(Industry data, Scenario scen, ModelYear year, RC regClass)
        {
            List<Manufacturer> mfrs = data.Manufacturers;
            double standardSum = 0, salesSum    = 0;
            for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
            {   // calc manufacturer sales and standard
                Manufacturer manufacturer = mfrs[i];
                double mfrSales = Standards.GetSales   (manufacturer,       year, regClass);
                double mfrStnd  = Standards.GetStandard(manufacturer, scen, year, regClass, null, 0);
                // add in inudstry components
                standardSum += (mfrSales == 0 || mfrStnd == 0) ? 0 : mfrSales / mfrStnd;
                salesSum    += mfrSales;
            }
            return (standardSum == 0) ? 0 : salesSum / standardSum;
        }

        /// <summary>
        /// Calculates the CAFE fuel economy standard, in miles per gallon (mpg), for the specified manufacturer for all
        /// regulatory class assignments of vehicles, operating under the given scenario and model year.  When this function
        /// completes, the manufacturer's CAFE fuel economy standard will be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the CAFE fuel economy standard.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <remarks>
        /// This function referes to the "current" vehicle attributes, including the vehicle's regulatory class assignment, which
        /// is assumed to be correct.  Therefore, finding the fuel economy target that would apply to a pending "test" vehicle
        /// requires that the caller temporarily substitute the relevant attributes for those in the "current" vehicle's data
        /// structure.
        /// </remarks>
        public static void CalcStandard(Manufacturer mfr, Scenario scen, ModelYear year)
        {
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (mfr.ModelData.Sales[regClass] != 0)
                {
                    Standards.CalcStandard(mfr, scen, year, regClass);
                }
            }
        }
        /// <summary>
        /// Calculates the CAFE fuel economy standard, in miles per gallon (mpg), for the specified manufacturer, based on the
        /// given regulatory class assignment of vehicles, operating under the given scenario and model year.  When this function
        /// completes, the manufacturer's CAFE fuel economy standard will be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the CAFE fuel economy standard.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the CAFE fuel economy
        ///   standard.</param>
        /// <remarks>
        /// This function referes to the "current" vehicle attributes, including the vehicle's regulatory class assignment, which
        /// is assumed to be correct.  Therefore, finding the fuel economy target that would apply to a pending "test" vehicle
        /// requires that the caller temporarily substitute the relevant attributes for those in the "current" vehicle's data
        /// structure.
        /// </remarks>
        public static void CalcStandard(Manufacturer mfr, Scenario scen, ModelYear year, RC regClass)
        {
            mfr.ModelData.Standard[regClass] = Standards.GetStandard(mfr, scen, year, regClass, null, 0);
        }
        /// <summary>
        /// Returns the calculated CAFE fuel economy standard, in miles per gallon (mpg), for the specified manufacturer, based
        /// on the given regulatory class assignment of vehicles, operating under the given scenario and model year, using the
        /// optional test vehicles information.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the CAFE fuel economy standard.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the CAFE fuel economy
        ///   standard.</param>
        /// <param name="techImpacts">An array of technology impacts and vehicles to examine when evaluating a functional
        ///   standard.  Use null to ignore this parameter.</param>
        /// <param name="techImpactCount">The number of technology impacts and vehicles to consider from techImpacts when
        ///   evaluating a functional standard.  Use less than or equal to 0 to ignore this parameter.</param>
        /// <returns>The calculated CAFE fuel economy standard, in miles per gallon (mpg), for the specified manufacturer, based
        ///   on the given regulatory class assignment of vehicles, operating under the given scenario and model year.</returns>
        /// <exception cref="System.ArgumentNullException">The testCount > 0 while one or more of the test arrays is null.</exception>
        /// <remarks>
        /// This function referes to the "current" vehicle attributes, including the vehicle's regulatory class assignment, which
        /// is assumed to be correct.  Therefore, finding the fuel economy target that would apply to a pending "test" vehicle
        /// requires that the caller temporarily substitute the relevant attributes for those in the "current" vehicle's data
        /// structure.
        /// </remarks>
        public static double GetStandard(Manufacturer mfr, Scenario scen, ModelYear year, RC regClass, TechImpact[] techImpacts,
            int techImpactCount)
        {
            if (techImpactCount > 0 && (techImpacts == null || techImpacts.Length < techImpactCount))
            {   // array is null or out of bounds
                throw new ArgumentOutOfRangeException("techImpacts", "techImpacts is null or less than techImpactCount.");
            }

            Manufacturer.CModelData mmd = mfr.ModelData;
            double standard = 0;
            int yrIndex = year.Index;

            // determine the functional type and coefficients
            int      fncType;
            double[] fncCoef;
            int numCoeff = Standards.GetCoefficients(regClass, scen, year, out fncType, out fncCoef);

            if (fncType == 0 || fncCoef == null || regClass == RC.None)
            {   // unregulated vehicles are assigned a standard of 0
                standard = 0;
            }
            else if (fncType == 1)
            {   // flat standard functional type
                standard = fncCoef[0];
            }
            else if (fncType == 99)
            {   // manufacturer-specific flat standard functional type
                standard = mmd.DefaultStandard[yrIndex][regClass];
            }
            else
            {   // non flat functional standard applies -- calculate the standard using a sales-weighted harmonic
                // average by evaluating vehicle specific targets
                double prevStandard = mmd.PreliminaryStandard[regClass];

                // recalculate the standard if:
                //  - a previously calculated standard does not exist
                //  - there are no test-vehicles
                //  - there are test-vehicles, but one of the vehicle's attributes has changed
                bool recalcStandard = (prevStandard == 0 || double.IsNaN(prevStandard) || techImpactCount == 0);
                if (!recalcStandard)
                {
                    if (FunctionInformation.IsAreaBasedFunction(fncType))
                    {   // area system (vehicle footprint does not change -- thus, this check is commented out)
                        for (int i = 0; i < techImpactCount; i++)
                        {
                            if (techImpacts[i].HasImpact          [yrIndex] &&
                                techImpacts[i].HasFootprintChanged(yrIndex)) { recalcStandard = true; break; }
                        }
                    }
                    else if (FunctionInformation.IsWeightBasedFunction(fncType))
                    {   // weight system
                        for (int i = 0; i < techImpactCount; i++)
                        {
                            if (techImpacts[i].HasImpact       [yrIndex] &&
                                techImpacts[i].HasWeightChanged(yrIndex)) { recalcStandard = true; break; }
                        }
                    }
                    else if (FunctionInformation.IsWorkFactorBasedFunction(fncType))
                    {   // work-factor system
                        for (int i = 0; i < techImpactCount; i++)
                        {
                            if ( techImpacts[i].HasImpact         [yrIndex] &&
                                (techImpacts[i].HasFuelTypeChanged(yrIndex) ||
                                 techImpacts[i].HasWeightChanged  (yrIndex) ||
                                 techImpacts[i].HasGVWRChanged    (yrIndex) ||
                                 techImpacts[i].HasGCWRChanged    (yrIndex))) { recalcStandard = true; break; }
                        }
                    }
                }

                if (recalcStandard)
                {   // test vehicle's area or weight has changed--recalculate the standard
                    //string merging = scenario.RegulatoryMerging[yrIndex].ToUpper();
                    double sales  = 0;      // the total sales for all vehicles
                    double stdSum = 0;      // the total (sales * target) for all vehicles

                    // get the model year vehicles
                    List<Vehicle> vehs = mfr.Vehicles;

                    // iterate through each vehicle to obtain its sales and sales * target
                    for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
                    {
                        Vehicle              veh      = vehs[i];
                        Vehicle.CDescription vd       = veh.Description;
                        double               vehSales = vd.Sales[yrIndex];
                        //
                        if (vehSales != 0 && veh.RegClass == regClass)
                        {
                            double target = 0;
                            // check if this vehicle is on the "test-list"
                            int idx = -1;
                            for (int j = 0; j < techImpactCount; j++)
                            {
                                if (veh == techImpacts[j].Vehicle[yrIndex]) { idx = j; break; }
                            }
                            target = Standards.GetTarget(veh, scen, year,
                                (idx == -1) ? FuelType.None : techImpacts[idx].NewFuelShare[yrIndex].FuelType,
                                (idx == -1) ? 0             : vd.Footprint,
                                (idx == -1) ? 0             : vd.CurbWeight + techImpacts[idx].DeltaCurbWeight[yrIndex],
                                (idx == -1) ? 0             : vd.GVWR       + techImpacts[idx].DeltaGVWR      [yrIndex],
                                (idx == -1) ? 0             : vd.GCWR       + techImpacts[idx].DeltaGCWR      [yrIndex],
                                (idx == -1) ? '\0'          : vd.Drive,
                                fncType, fncCoef);
                            sales  += vehSales;
                            stdSum += vehSales * target;
                        } // end if (veh has sales; exam desired regClass)
                    } // next i (veh)
                    // calculate the standard
                    standard = (sales == 0) ? 0 : sales / stdSum;
                }
                else
                {   // test vehicle's area or weight did not changed--use the previously calculated standard
                    standard = prevStandard;
                }
            }

            if (techImpactCount == 0)
            {   // save the preliminary standard to the manufacturer, before adjusting for "alternative minimum standards"
                mmd.PreliminaryStandard[regClass] = RoundFE(standard, regClass);
            }

            // check if the scenario uses "alternative minimum standards"
            if (standard < scen.ScenInfo[regClass].MinStndMpg[yrIndex]) { standard = scen.ScenInfo[regClass].MinStndMpg[yrIndex]; }
            double pctIndStnd = scen.ScenInfo[regClass].MinStndPct[yrIndex] * mmd.AverageStandard[regClass];
            if (standard < pctIndStnd) { standard = pctIndStnd; }

            // round and return the calculated standard
            return RoundFE(standard, regClass);
        }

        /// <summary>
        /// Returns the fuel economy target, in gallons per mile (gpm), for the specified vehicle, scenario, and model year.
        /// </summary>
        /// <param name="veh">The vehicle for which to compute the target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <returns>The fuel economy target, in gallons per mile (gpm), for the specified vehicle, scenario, and model year.</returns>
        /// <remarks>
        /// This function referes to the "current" vehicle attributes, including the vehicle's regulatory class assignment, which
        /// is assumed to be correct.  Therefore, finding the fuel economy target that would apply to a pending "test" vehicle
        /// requires that the caller temporarily substitute the relevant attributes for those in the "current" vehicle's data
        /// structure.
        /// </remarks>
        public static double GetTarget(Vehicle veh, Scenario scen, ModelYear year)
        {
            return Standards.GetTarget(veh, scen, year, FuelType.None, 0, 0, 0, 0, '\0', 0, null);
        }
        /// <summary>
        /// Returns the fuel economy target, in gallons per mile (gpm), for the specified vehicle, scenario, and model year,
        /// using the specified vehicle attributes, as well as the given function type and functional coefficients.
        /// </summary>
        /// <param name="veh">The vehicle for which to compute the target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="fuel">The optional "test" fuel type to use in obtaining the vehicle target. Not applicable if function
        ///   type is not workfactor-based. Use <see cref="FuelType.None"/> to get default vehicle fuel.</param>
        /// <param name="area">The optional "test" area (footprint) to use in obtaining the vehicle target. Not applicable if
        ///   function type is not area-based. Use 0 to get default vehicle area.</param>
        /// <param name="weight">The optional "test" curb weight to use in obtaining the vehicle target. Not applicable if
        ///   function type is not weight-based or workfactor-based. Use 0 to get default vehicle weight.</param>
        /// <param name="gvwr">The optional "test" GVWR to use in obtaining the vehicle target. Not applicable if function type
        ///   is not workfactor-based. Use 0 to get default vehicle GVWR.</param>
        /// <param name="gcwr">The optional "test" GCWR to use in obtaining the vehicle target. Not applicable if function type
        ///   is not workfactor-based. Use 0 to get default vehicle GCWR.</param>
        /// <param name="drive">The optional "test" drive type to use in obtaining the vehicle target. Not applicable if function
        ///   type is not workfactor-based. Use the null character ('\0') to get default vehicle drive.</param>
        /// <param name="fncType">The target function type to use in obtaining the vehicle target.
        ///   If specifying function type and functional coefficients, the fncType and fncCoef parameters must correlate. Use 0
        ///   for fncType or null for fncCoef to calculate these values based on vehicle and scenario.</param>
        /// <param name="fncCoef">The functional coefficients to use in obtaining the vehicle target.
        ///   If specifying function type and functional coefficients, the fncType and fncCoef parameters must correlate. Use 0
        ///   for fncType or null for fncCoef to calculate these values based on vehicle and scenario.</param>
        /// <returns>The fuel economy target, in gallons per mile (gpm), for the specified vehicle, scenario, and model year.</returns>
        public static double GetTarget(Vehicle veh, Scenario scen, ModelYear year,
            FuelType fuel, double area, double weight, double gvwr, double gcwr, char drive,
            int fncType, double[] fncCoef)
        {
            // check if valid vehicle attributes were supplied
            if (fuel    == FuelType.None) { fuel   = veh.Description.FuelShare.FuelType; }
            if (area    <= 0            ) { area   = veh.Description.Footprint ; }
            if (weight  <= 0            ) { weight = veh.Description.CurbWeight; }
            if (gvwr    <= 0            ) { gvwr   = veh.Description.GVWR      ; }
            if (gcwr    <= 0            ) { gcwr   = veh.Description.GCWR      ; }
            if (drive   == '\0'         ) { drive  = veh.Description.Drive     ; }
            if (fncType == 0 || fncCoef == null)
            {   // calculate functional type and coefs
                Standards.GetCoefficients(veh.RegClass, scen, year, out fncType, out fncCoef);
            }
            // get A through J coefficients and return the target
            return Standards.GetTarget(veh, scen, year, fuel, area, weight, gvwr, gcwr, drive, fncType, fncCoef[0], fncCoef[1],
                fncCoef[2], fncCoef[3], fncCoef[4], fncCoef[5], fncCoef[6], fncCoef[7], fncCoef[8], fncCoef[9]);
        }
        /// <summary>
        /// Returns the fuel economy target, in gallons per mile (gpm), for the specified vehicle, scenario, and model year,
        /// using the specified vehicle attributes, as well as the given function type and functional coefficients.
        /// </summary>
        /// <param name="veh">The vehicle for which to compute the target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="fuel">The vehicle fuel type to use in obtaining the vehicle target. Not applicable if function type is
        ///   not workfactor-based.</param>
        /// <param name="area">The vehicle area (footprint) to use in obtaining the vehicle target. Not applicable if function
        ///   type is not area-based.</param>
        /// <param name="weight">The vehicle curb weight to use in obtaining the vehicle target. Not applicable if function type
        ///   is not weight-based or workfactor-based.</param>
        /// <param name="gvwr">The vehicle GVWR to use in obtaining the vehicle target. Not applicable if function type is not
        ///   workfactor-based.</param>
        /// <param name="gcwr">The vehicle GCWR to use in obtaining the vehicle target. Not applicable if function type is not
        ///   workfactor-based.</param>
        /// <param name="drive">The vehicle drive type to use in obtaining the vehicle target. Not applicable if function type
        ///   is not workfactor-based.</param>
        /// <param name="fncType">The target function type to use in obtaining the vehicle target.</param>
        /// <param name="A">The functional A coefficient to use in obtaining the vehicle target.</param>
        /// <param name="B">The functional B coefficient to use in obtaining the vehicle target.</param>
        /// <param name="C">The functional C coefficient to use in obtaining the vehicle target.</param>
        /// <param name="D">The functional D coefficient to use in obtaining the vehicle target.</param>
        /// <param name="E">The functional E coefficient to use in obtaining the vehicle target.</param>
        /// <param name="F">The functional F coefficient to use in obtaining the vehicle target.</param>
        /// <param name="G">The functional G coefficient to use in obtaining the vehicle target.</param>
        /// <param name="H">The functional H coefficient to use in obtaining the vehicle target.</param>
        /// <param name="I">The functional I coefficient to use in obtaining the vehicle target.</param>
        /// <param name="J">The functional J coefficient to use in obtaining the vehicle target.</param>
        /// <returns>The fuel economy target, in gallons per mile (gpm), for the specified vehicle, scenario, and model year.</returns>
        /// <remarks>
        /// This GetTarget method supports the following function types:
        /// <list type="table">
        ///   <listheader>
        ///     <th width="0%">Function</th>
        ///     <th width="50%">Description</th>
        ///     <th width="50%">Specification</th>
        ///   </listheader>
        ///   <item>
        ///     <description>1</description>
        ///     <description>Flat standard.<br/><br/>
        ///                  <b>A</b>:  mpg</description>
        ///     <description><b>Target</b> = <i>1 / <b>A</b></i></description>
        ///   </item>
        ///   <item>
        ///     <description>2</description>
        ///     <description>Logistic area-based function.<br/><br/>
        ///                  <b>A</b>:  mpg ("ceiling")<br/>
        ///                  <b>B</b>:  mpg ("floor")<br/>
        ///                  <b>C</b>:  square feet ("midpoint")<br/>
        ///                  <b>D</b>:  square feet ("width")</description>
        ///     <description><b>Target</b> = <i>omitted</i></description>
        ///   </item>
        ///   <item>
        ///     <description>3</description>
        ///     <description>Logistic weight-based function.<br/><br/>
        ///                  <b>A</b>:  mpg ("ceiling")<br/>
        ///                  <b>B</b>:  mpg ("floor")<br/>
        ///                  <b>C</b>:  pounds ("midpoint")<br/>
        ///                  <b>D</b>:  pounds ("width")</description>
        ///     <description><b>Target</b> = <i>omitted</i></description>
        ///   </item>
        ///   <item>
        ///     <description>4</description>
        ///     <description>Exponential area-based function.<br/><br/>
        ///                  <b>A</b>:  mpg ("ceiling")<br/>
        ///                  <b>B</b>:  mpg (should be > A)<br/>
        ///                  <b>C</b>:  square feet (determines "height")<br/>
        ///                  <i>Note:  if AREAmin is the lowest possible area, C must not exceed AREAmin/(1-ln(B/A)).</i></description>
        ///     <description><b>Target</b> = <i>1 / <b>A</b> - EXP(1 - Area / <b>C</b>) / <b>B</b></i></description>
        ///   </item>
        ///   <item>
        ///     <description>5</description>
        ///     <description>Exponential weight-based function.<br/><br/>
        ///                  <b>A</b>:  mpg ("ceiling")<br/>
        ///                  <b>B</b>:  mpg (should be > A)<br/>
        ///                  <b>C</b>:  pounds (determines "height")<br/>
        ///                  <i>Note:  if CWmin is the lowest possible weight, C must not exceed CWmin/(1-ln(B/A)).</i></description>
        ///     <description><b>Target</b> = <i>1 / <b>A</b> - EXP(1 - CW / <b>C</b>) / <b>B</b></i></description>
        ///   </item>
        ///   <item>
        ///     <description>6</description>
        ///     <description>Linear area-based function.<br/><br/>
        ///                  <b>A</b>:  mpg ("ceiling")<br/>
        ///                  <b>B</b>:  mpg ("floor")<br/>
        ///                  <b>C</b>:  change in gpm / change in square feet ("slope" of the function)<br/>
        ///                  <b>D</b>:  gpm ("y-intercept")</description>
        ///     <description><b>Target</b> = <i>MAX(<b>1 / A</b>, MIN(<b>1 / B</b>, <b>C</b> * Area + <b>D</b>))</i></description>
        ///   </item>
        ///   <item>
        ///     <description>7</description>
        ///     <description>Linear weight-based function.<br/><br/>
        ///                  <b>A</b>:  mpg ("ceiling")<br/>
        ///                  <b>B</b>:  mpg ("floor")<br/>
        ///                  <b>C</b>:  change in gpm / change in pounds ("slope" of the function)<br/>
        ///                  <b>D</b>:  gpm ("y-intercept")</description>
        ///     <description><b>Target</b> = <i>MAX(<b>1 / A</b>, MIN(<b>1 / B</b>, <b>C</b> * CW + <b>D</b>))</i></description>
        ///   </item>
        ///   <item>
        ///     <description>8</description>
        ///     <description>Linear work-factor-based function.<br/><br/>
        ///                  <b><i>General coefficients</i></b><br/>
        ///                  - <b>A</b>:  'xwd' coefficient; additional offset, in lbs, for 4-wheel drive vehicles<br/>
        ///                  - <b>B</b>:  weighting multiplier for payload vs. towing capacity<br/>
        ///                  <b><i>Coefficients for gasoline vehicles</i></b><br/>
        ///                  - <b>C</b>:  change in gpm / change in work-factor ("slope" of the function)<br/>
        ///                  - <b>D</b>:  gallons per 100-miles ("y-intercept")<br/>
        ///                  <b><i>Coefficients for diesel vehicles</i></b><br/>
        ///                  - <b>E</b>:  change in gpm / change in work-factor ("slope" of the function)<br/>
        ///                  - <b>F</b>:  gallons per 100-miles ("y-intercept")<br/>
        ///                  <b><i>Coefficients for CNG vehicles</i></b><br/>
        ///                  - <b>G</b>:  change in gpm / change in work-factor ("slope" of the function)<br/>
        ///                  - <b>H</b>:  gallons per 100-miles ("y-intercept")</description>
        ///     <description><b>Target</b> = <i>omitted</i></description>
        ///   </item>
        ///   <item>
        ///     <description>206</description>
        ///     <description>Dual linear area-based function.<br/><br/>
        ///                  <b><i>Primary function coefficients</i></b><br/>
        ///                  - <b>A</b>:  mpg (ceiling; lower bound asymptote)<br/>
        ///                  - <b>B</b>:  mpg (floor; upper bound asymptote)<br/>
        ///                  - <b>C</b>:  change in gpm / change in square feet ("slope" of the function)<br/>
        ///                  - <b>D</b>:  gpm ("y-intercept")<br/>
        ///                  <b><i>Secondary function coefficients</i></b><br/>
        ///                  - <b>E</b>:  mpg (ceiling; lower bound asymptote); for the secondary function<br/>
        ///                  - <b>F</b>:  mpg (floor; upper bound asymptote); for the secondary function<br/>
        ///                  - <b>G</b>:  change in gpm / change in square feet ("slope" of the function); for the secondary function<br/>
        ///                  - <b>H</b>:  gpm ("y-intercept"); for the secondary function</description>
        ///     <description><b>Target</b> = <i>MIN(MAX(<b>1 / A</b>, MIN(<b>1 / B</b>, <b>C</b> * Area + <b>D</b>)), MAX(<b>1 / E</b>, MIN(<b>1 / F</b>, <b>G</b> * Area + <b>H</b>)))</i></description>
        ///   </item>
        ///   <item>
        ///     <description>207</description>
        ///     <description>Dual linear weight-based function.<br/><br/>
        ///                  <b><i>Primary function coefficients</i></b><br/>
        ///                  - <b>A</b>:  mpg (ceiling; lower bound asymptote)<br/>
        ///                  - <b>B</b>:  mpg (floor; upper bound asymptote)<br/>
        ///                  - <b>C</b>:  change in gpm / change in pounds ("slope" of the function)<br/>
        ///                  - <b>D</b>:  gpm ("y-intercept")<br/>
        ///                  <b><i>Secondary function coefficients</i></b><br/>
        ///                  - <b>E</b>:  mpg (ceiling; lower bound asymptote); for the secondary function<br/>
        ///                  - <b>F</b>:  mpg (floor; upper bound asymptote); for the secondary function<br/>
        ///                  - <b>G</b>:  change in gpm / change in square feet ("slope" of the function); for the secondary function<br/>
        ///                  - <b>H</b>:  gpm ("y-intercept"); for the secondary function</description>
        ///     <description><b>Target</b> = <i>MIN(MAX(<b>1 / A</b>, MIN(<b>1 / B</b>, <b>C</b> * Area + <b>D</b>)), MAX(<b>1 / E</b>, MIN(<b>1 / F</b>, <b>G</b> * Area + <b>H</b>)))</i></description>
        ///   </item>
        ///   <item>
        ///     <description>208</description>
        ///     <description>Dual linear work-factor-based function.<br/><br/>
        ///                  <b><i>General coefficients</i></b><br/>
        ///                  - <b>A</b>:  'xwd' coefficient; additional offset, in lbs, for 4-wheel drive vehicles<br/>
        ///                  - <b>B</b>:  weighting multiplier for payload vs. towing capacity<br/>
        ///                  <b><i>Coefficients for gasoline vehicles</i></b><br/>
        ///                  - <b>C</b>:  change in gpm / change in work-factor ("slope" of the function)<br/>
        ///                  - <b>D</b>:  gallons per 100-miles ("y-intercept")<br/>
        ///                  <b><i>Coefficients for diesel vehicles</i></b><br/>
        ///                  - <b>E</b>:  change in gpm / change in work-factor ("slope" of the function)<br/>
        ///                  - <b>F</b>:  gallons per 100-miles ("y-intercept")<br/>
        ///                  <b><i>Coefficients for CNG vehicles</i></b><br/>
        ///                  - <b>G</b>:  change in gpm / change in work-factor ("slope" of the function)<br/>
        ///                  - <b>H</b>:  gallons per 100-miles ("y-intercept")<br/>
        ///                  <b><i>Secondary function coefficients</i></b><br/>
        ///                  - <b>I</b>:  the model year whose function serves as the "floor" for this function</description>
        ///     <description><b>Target</b> = <i>omitted</i></description>
        ///   </item>
        /// </list>
        /// </remarks>
        public static double GetTarget(Vehicle veh, Scenario scen, ModelYear year,
            FuelType fuel, double area, double weight, double gvwr, double gcwr, char drive,
            int fncType, double A, double B, double C, double D, double E, double F, double G, double H, double I, double J)
        {
            // determine the vehicle target
            double target = 0.0, invA = 1.0 / A, invB = 1.0 / B;
            double exp;
            double t1, t2;
            //
            switch (fncType)
            {
                //
                // Flat standard function
                //
                // A:  mpg
                //
                case 1:
                    target = invA;
                    break;

                //
                // Logistic area-/weight-based function
                //
                // A:  mpg ("ceiling")
                // B:  mpg ("floor")
                // C:  square feet or pounds ("midpoint")
                // D:  square feet or pounds ("width")
                //
                case 2:     // area-based
                case 3:     // weight-based
                    exp    = Math.Exp((((fncType == 2) ? area : weight) - C) / D);
                    target = invA + (invB - invA) * (exp / (1.0 + exp));
                    break;

                //
                // Exponential area-/weight-based function
                //
                // A:  mpg ("ceiling")
                // B:  mpg (should be > A)
                // C:  square feet or pounds (determines "height")
                //
                case 4:     // area-based
                case 5:     // weight-based
                    exp    = Math.Exp(1.0 - ((fncType == 4) ? area : weight) / C);
                    target = invA - exp * invB;
                    break;

                //
                // Linear area-/weight-based function
                //
                // A:  mpg ("ceiling")
                // B:  mpg ("floor")
                // C:  change in gpm / change in square feet or pounds ("slope" of the function)
                // D:  gpm ("y-intercept")
                //
                case 6:     // area-based
                case 7:     // weight-based
                    target = C * ((fncType == 6) ? area : weight) + D;                  // calculate target
                    target = (target < invA) ? invA : (target > invB) ? invB : target;  // trim bounds and convert to gpm as needed
                    break;

                //
                // Linear work-factor-based function
                //
                // General coefficients:
                //   A:  'xwd' coefficient; additional offset, in lbs, for 4-wheel drive vehicles
                //   B:  weighting multiplier for payload vs. towing capacity
                // Coefficients for gasoline vehicles:
                //   C:  change in gpm / change in work-factor ("slope" of the function)
                //   D:  gallons per 100-miles ("y-intercept")
                // Coefficients for diesel vehicles:
                //   E:  change in gpm / change in work-factor ("slope" of the function)
                //   F:  gallons per 100-miles ("y-intercept")
                // Coefficients for CNG vehicles:
                //   G:  change in gpm / change in work-factor ("slope" of the function)
                //   H:  gallons per 100-miles ("y-intercept")
                //
                case 8:     // work-factor-based
                    bool isDSL = ((fuel & FuelType.Diesel) == FuelType.Diesel);
                    bool isCNG = ((fuel & FuelType.CNG   ) == FuelType.CNG   );
                    //
                    double workFactor = Standards.GetWorkFactor(veh, scen, year, weight, gvwr, gcwr, drive, A, B);
                    target = ((isCNG) ? (G * workFactor + H) :                          // calculate target (CNG)
                              (isDSL) ? (E * workFactor + F) :                          // calculate target (diesel)
                                        (C * workFactor + D)                            // calculate target (gasoline)
                             ) / 100;                                                   // divide by 100 to convert to gpm
                    break;

                //
                // Dual linear area-/weight-based function
                //
                // Primary function coefficients:
                //   A:  mpg ("ceiling")
                //   B:  mpg ("floor")
                //   C:  change in gpm / change in square feet or pounds ("slope" of the function)
                //   D:  gpm ("y-intercept")
                // Secondary function coefficients:
                //   E:  mpg ("ceiling")
                //   F:  mpg ("floor")
                //   G:  change in gpm / change in square feet or pounds ("slope" of the function)
                //   H:  gpm ("y-intercept")
                //
                case 206:   // area-based
                case 207:   // weight-based
                    t1 = C * ((fncType == 206) ? area : weight) + D;                    // calculate target for the primary function
                    t1 = (t1 < invA) ? invA : (t1 > invB) ? invB : t1;                  // trim bounds and convert to gpm as needed
                    double invE = 1.0 / E, invF = 1.0 / F;
                    t2 = G * ((fncType == 206) ? area : weight) + H;                    // calculate target for the 2nd function
                    t2 = (t2 < invE) ? invE : (t2 > invF) ? invF : t2;                  // trim bounds and convert to gpm as needed
                    // pick the smaller of the two targets
                    target = (t1 < t2) ? t1 : t2;
                    break;

                //
                // Dual linear work-factor-based function
                //
                // General coefficients:
                //   A:  'xwd' coefficient; additional offset, in lbs, for 4-wheel drive vehicles
                //   B:  weighting multiplier for payload vs. towing capacity
                // Coefficients for gasoline vehicles:
                //   C:  change in gpm / change in work-factor ("slope" of the function)
                //   D:  gallons per 100-miles ("y-intercept")
                // Coefficients for diesel vehicles:
                //   E:  change in gpm / change in work-factor ("slope" of the function)
                //   F:  gallons per 100-miles ("y-intercept")
                // Coefficients for CNG vehicles:
                //   G:  change in gpm / change in work-factor ("slope" of the function)
                //   H:  gallons per 100-miles ("y-intercept")
                // Secondary function coefficients:
                //   I:  the model year whose function serves as the "floor" for this function
                //
                case 208:   // work-factor-based
                    // calculate target for the primary function
                    t1 = Standards.GetTarget(veh, scen, year, fuel, area, weight, gvwr, gcwr, drive, 8, A, B, C, D, E, F, G, H, I, J);
                    // calculate target for the secondary function
                    t2 = Standards.GetTarget(veh, scen, new ModelYear((int)I), fuel, area, weight, gvwr, gcwr, drive, 0, null);
                    // pick the smaller of the two targets
                    target = (t1 < t2) ? t1 : t2;
                    break;
            }

            // return the calculated target
            return target;
        }

        /// <summary>
        /// Returns the regulatory class assignment for the specified vehicle.
        /// </summary>
        /// <param name="vehicle">The vehicle for which to determine the regulatory class assignment.</param>
        /// <returns>The regulatory class assignment for the specified vehicle based on the given scenario and model year.</returns>
        public static RC GetRegClass(Vehicle vehicle)
        {
            // determine reg-class
            switch (vehicle.Description.RegulatoryIndicator)
            {
                case "PC"   : return RC.PassengerCar;
                case "LT"   : return RC.LightTruck;
                case "LT2B3": return RC.LightTruck2b3;
                default     : return RC.None;
            }
        }

        /// <summary>
        /// Returns the work-factor for the specified vehicle, scenario, and model year.
        /// </summary>
        /// <param name="veh">The vehicle for which to calculate the work-factor.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <returns>The work-factor for the specified vehicle, scenario, and model year.</returns>
        public static double GetWorkFactor(Vehicle veh, Scenario scen, ModelYear year)
        {
            int      fncType;
            double[] fncCoef;
            Standards.GetCoefficients(veh.RegClass, scen, year, out fncType, out fncCoef);
            if (FunctionInformation.IsWorkFactorBasedFunction(fncType))
            {
                double xwdCoef = fncCoef[0];
                double mltCoef = fncCoef[1];
                Vehicle.CDescription vd = veh.Description;
                return GetWorkFactor(veh, scen, year, vd.CurbWeight, vd.GVWR, vd.GCWR, vd.Drive, xwdCoef, mltCoef);
            }
            else { return 0; }
        }
        /// <summary>
        /// Returns the work-factor for the specified vehicle, scenario, and model year, using the specified vehicle attributes,
        /// as well as the given 'xwdCoef' and 'mltCoef' values.
        /// </summary>
        /// <param name="veh">The vehicle for which to calculate the work-factor.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="curbWeight">The curb weight of the vehicle.</param>
        /// <param name="gvwr">The gvwr of the vehicle.</param>
        /// <param name="gcwr">The gcwr of the vehicle.</param>
        /// <param name="drive">The drive type of the vehicle (4WD, AWD, 2WD/FWD/RWD).</param>
        /// <param name="xwdCoef">The additional offset, in lbs, for 4-wheel drive vehicles.</param>
        /// <param name="mltCoef">The weighting multiplier for payload vs. towing capacity.</param>
        /// <returns>The work-factor for the specified vehicle, scenario, and model year.</returns>
        static double GetWorkFactor(Vehicle veh, Scenario scen, ModelYear year,
            double curbWeight, double gvwr, double gcwr, char drive, double xwdCoef, double mltCoef)
        {
            double xwd = (drive == '4') ? xwdCoef : 0;
            return (gvwr - curbWeight + xwd) * mltCoef + (gcwr - gvwr) * (1 - mltCoef);
        }

        /// <summary>
        /// Returns the test weight for the specified vehicle, scenario, and model year.
        /// </summary>
        /// <param name="veh">The vehicle for which to calculate the test weight.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <returns>The test weight for the specified vehicle, scenario, and model year.</returns>
        public static double GetTestWeight(Vehicle veh, Scenario scen, ModelYear year)
        {
            return Standards.GetTestWeight(veh, scen, year, 0, 0, 0, 0);
        }
        /// <summary>
        /// Returns the test weight for the specified vehicle, scenario, and model year, using the specified vehicle attributes,
        /// as well as the given test weight function type.
        /// </summary>
        /// <param name="veh">The vehicle for which to calculate the test weight.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="weight">The optional "test" curb weight to use in obtaining the vehicle test weight. Use 0 to get
        ///   default vehicle weight.</param>
        /// <param name="gvwr">The optional "test" GVWR to use in obtaining the vehicle test weight. Use 0 to get default vehicle
        ///   GVWR.</param>
        /// <param name="gcwr">The optional "test" GCWR to use in obtaining the vehicle test weight. Use 0 to get default vehicle
        ///   GCWR.</param>
        /// <param name="twFunction">The test weight function type to use in obtaining the vehicle test weight. Use 0 to calculate
        ///   default value based on vehicle and scenario.</param>
        /// <returns>The test weight for the specified vehicle, scenario, and model year.</returns>
        public static double GetTestWeight(Vehicle veh, Scenario scen, ModelYear year,
            double weight, double gvwr, double gcwr,
            int twFunction)
        {
            if (weight <= 0) { weight = veh.Description.CurbWeight; }
            if (gvwr   <= 0) { gvwr   = veh.Description.GVWR      ; }
            if (gcwr   <= 0) { gcwr   = veh.Description.GCWR      ; }
            //
            if (twFunction <= 0)
            {
                RegulatoryClass regClass = (veh.RegClass == RegulatoryClass.None) ? Standards.GetRegClass(veh) : veh.RegClass;
                if (regClass == RegulatoryClass.None) { return 0; }
                twFunction = scen.ScenInfo[regClass].TWFunction[year.Index];
                if (twFunction == 0)
                {
                    twFunction =
                        (regClass == RegulatoryClass.PassengerCar ) ? 1 :
                        (regClass == RegulatoryClass.LightTruck   ) ? 2 :
                        (regClass == RegulatoryClass.LightTruck2b3) ? 3 : 0;
                }
            }
            //
            // calculate the vehicle test weight
            double testWeight = 0;
            //
            switch (twFunction)
            {
                //
                // 1) Default LVW-based function with a lower bound cap of 1000 lbs. and an upper bound cap of 5500 lbs.
                // 2) Default LVW-based function with a lower bound cap of 1000 lbs.
                // 3) Default ALVW-based function with a lower bound cap of 1000 lbs.
                //
                case 1:
                case 2:
                case 3:
                    double w = (twFunction == 3) ? (weight + gvwr) / 2 : weight + 300;
                    double r = (w > 5500) ? 500 : (w > 4000) ? 250 : 125;
                    testWeight = Math.Max(1000, Math.Ceiling(w / r - 0.5) * r);
                    if (twFunction == 1 && testWeight > 5500) { testWeight = 5500; }
                    break;
            }
            //
            // return the calculated test weight
            return testWeight;
        }

        #endregion

        #region /* Sales calculations */

        /// <summary>
        /// Calculates and returns the sum of vehicle sales, for all vehicles produced by the given manufactuer, for the given
        /// model year and all regulatory classes.  (The sum of sales will be aggregated separately for each regulatory class.)
        /// </summary>
        /// <param name="mfr">The manufacturer for which to sum vehicle sales, per regulatory class.</param>
        /// <param name="year">The model year for which to sum vehicle sales, per regulatory class.</param>
        /// <returns>The sum of vehicle sales for the specified manufacturer, aggregated per regulatory class.</returns>
        public static RCDouble GetSales(Manufacturer mfr, ModelYear year)
        {
            RCDouble sales = new RCDouble();
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                sales[regClass] = Standards.GetSales(mfr, year, regClass);
            }
            return sales;
        }
        /// <summary>
        /// Calculates and returns the sum of vehicle sales, for all vehicles produced by the given manufactuer, for the given
        /// model year and regulatory class.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to sum vehicle sales.</param>
        /// <param name="year">The model year for which to sum vehicle sales.</param>
        /// <param name="regClass">The regulatory class for which to sum vehicle sales.</param>
        /// <returns>The sum of vehicle sales for the specified manufacturer.</returns>
        public static double GetSales(Manufacturer mfr, ModelYear year, RC regClass)
        {
            return Standards.GetSales(mfr.Vehicles, year, regClass);
        }
        /// <summary>
        /// Calculates and returns the sum of vehicle sales, for all vehicles in the specified list, for the given model year and
        /// regulatory class.
        /// </summary>
        /// <param name="vehs">A list of vehicles for which to calculate the sum of sales.</param>
        /// <param name="year">The model year for which to sum vehicle sales.</param>
        /// <param name="regClass">The regulatory class for which to sum vehicle sales.</param>
        /// <returns>The sum of vehicle sales in the specified list.</returns>
        public static double GetSales(List<Vehicle> vehs, ModelYear year, RC regClass)
        {
            double sales = 0;
            int yrIndex = year.Index;
            // get model year vehicles
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {
                if (vehs[i].RegClass == regClass) { sales += vehs[i].Description.Sales[yrIndex]; }
            }
            // return the sum of all vehicle sales for the given reg class
            return sales;
        }

        #endregion

        #region /* Fines calculations */

        /// <summary>
        /// Calculates the fines for the specified manufacturer, for all regulatory classes, using the given modeling parameters.
        /// When this function completes, the manufacturer's fines value will be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the fines, per regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        public static void CalcFines(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            // calculate the fines for multiple regulatory classes, one at a time
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (mfr.ModelData.Sales[regClass] != 0)
                {
                    Standards.CalcFines(mfr, scen, year, settings, regClass);
                }
            }
        }
        /// <summary>
        /// Calculates the fines for the specified manufacturer, based on the given regulatory class assignment of vehicles,
        /// using the given modeling parameters.  When this function completes, the manufacturer's fines value will be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the fines for the given regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the manufacturer
        ///   fines.</param>
        public static void CalcFines(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings, RC regClass)
        {
            // get the runtime data for the manufacturer
            Manufacturer.CModelData mmd = mfr.ModelData;
            double credits     = mmd.Credits    [regClass];     // get the credits
            double tCreditsIn  = mmd.TCreditsIn [regClass];     // get the transferable credits, transferred into the mfr
            double tCreditsOut = mmd.TCreditsOut[regClass];     // get the transferable credits, transferred out of the mfr
            double netCredits  = credits + tCreditsIn - tCreditsOut;
            double fineRate    = scen.ScenInfo[regClass].FineRate[year.Index];  // get the fine rate
            // calculate the fines for the manufacturer / reg class -- (if credits is NaN, set fines to 0)
            mmd.Fines[regClass] = (double.IsNaN(credits)) ? 0 : -fineRate * Math.Min(netCredits, 0);
        }

        /// <summary>
        /// Calculates and returns the fines for the specified manufacturer, for all regulatory classes, using the given modeling
        /// parameters, as well as the specified test vehicles and vehicle test attributes.  This function implicitly calculates
        /// the test manufacturer credits.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the fines, per regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="newCafe">The new calculated test CAFE value to use in determining the manufacturer fines.</param>
        /// <param name="techImpacts">An array of technology impacts and vehicles to examine when evaluating a functional
        ///   standard and calculating fines.  Use null to ignore this parameter.</param>
        /// <param name="techImpactCount">The number of technology impacts and vehicles to consider from techImpacts when
        ///   evaluating a functional standard and calculating fines.  Use less than or equal to 0 to ignore this parameter.</param>
        /// <param name="useRounding">Specifies whether to round the test CAFE value before computing test credits.</param>
        /// <returns>The fines for the specified manufacturer, for all regulatory classes.</returns>
        public static RCDouble TestFines(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings,
            RCDouble newCafe, TechImpact[] techImpacts, int techImpactCount, bool useRounding)
        {
            // calculate the fines for multiple regulatory classes, one at a time
            RCDouble fines = new RCDouble();
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (mfr.ModelData.Sales[regClass] != 0)
                {
                    fines[regClass] = Standards.TestFines(mfr, scen, year, settings, regClass, newCafe[regClass], techImpacts, techImpactCount, useRounding);
                }
            }
            return fines;
        }
        /// <summary>
        /// Calculates and returns the fines for the specified manufacturer, based on the given regulatory class assignment of
        /// vehicles, using the given modeling parameters, as well as the specified test vehicles and vehicle test attributes.
        /// This function implicitly calculates the test manufacturer credits.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the fines for the given regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the manufacturer
        ///   fines.</param>
        /// <param name="newCafe">The new calculated test CAFE value to use in determining the manufacturer fines.</param>
        /// <param name="techImpacts">An array of technology impacts and vehicles to examine when evaluating a functional
        ///   standard and calculating fines.  Use null to ignore this parameter.</param>
        /// <param name="techImpactCount">The number of technology impacts and vehicles to consider from techImpacts when
        ///   evaluating a functional standard and calculating fines.  Use less than or equal to 0 to ignore this parameter.</param>
        /// <param name="useRounding">Specifies whether to round the test CAFE value before computing test credits.</param>
        /// <returns>The fines for the specified manufacturer, based on the given regulatory class assignment of vehicles.</returns>
        public static double TestFines(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings, RC regClass,
            double newCafe, TechImpact[] techImpacts, int techImpactCount, bool useRounding)
        {
            // calculate the test credits
            double credits = Standards.TestCredits(mfr, scen, year, settings, regClass, newCafe, techImpacts, techImpactCount, useRounding);
            // get the fine rate
            double fineRate = scen.ScenInfo[regClass].FineRate[year.Index];
            // calculate and return the fines for the manufacturer and reg class
            return -fineRate * Math.Min(credits, 0);
        }

        #endregion

        #region /* Credits calculations */

        /// <summary>
        /// Calculates the credits for the specified manufacturer, for all regulatory classes, using the previously calculated
        /// manufacturer sales and standard values.  When this function completes, the manufacturer's credits earned value will
        /// be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the credits, per regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        public static void CalcCredits(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            // calculate the credits for multiple regulatory classes, one at a time
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (mfr.ModelData.Sales[regClass] != 0)
                {
                    Standards.CalcCredits(mfr, scen, year, settings, regClass);
                }
            }
        }
        /// <summary>
        /// Calculates the credits for the specified manufacturer, based on the given regulatory class assignment of vehicles,
        /// operating under the given scenario and model year.  When this function completes, the manufacturer's credits earned
        /// value will be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the credits for the given regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the manufacturer
        ///   credits.</param>
        public static void CalcCredits(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings, RC regClass)
        {
            // get the runtime data for the manufacturer, and get cafe, sales, and standard
            Manufacturer.CModelData mmd = mfr.ModelData;
            double sales     = mmd.Sales   [regClass];
            double standard  = mmd.Standard[regClass];
            double cafe      = mmd.CAFE    [regClass];

            // if sales, standard, or cafe is NaN, set credits to 0
            if (double.IsNaN(sales) || double.IsNaN(standard) || double.IsNaN(cafe)) { mmd.Credits[regClass] = 0; }
            else
            {   // adjust CAFE value for FFV credits
                cafe += mfr.Description.GetFFVCredits(regClass, year.Year);
                // calculate and save the credits for the manufacturer / reg class
                if (regClass == RegulatoryClass.LightTruck2b3)
                {
                    mmd.Credits[regClass] = Math.Round((100 / standard - Math.Round(100 / cafe, 2)) * sales * 100);
                }
                else
                {
                    mmd.Credits[regClass] = Math.Round((Math.Round(cafe, 1) - standard) * sales * 10);
                }
            }
        }

        /// <summary>
        /// Calculates and returns the credits for the specified manufacturer, for all regulatory classes, operating under the
        /// given scenario and model year, using the specified test vehicles and vehicle test attributes.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the credits, per regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="techImpacts">An array of technology impacts and vehicles to examine when evaluating a functional
        ///   standard and calculating credits.  Use null to ignore this parameter.</param>
        /// <param name="techImpactCount">The number of technology impacts and vehicles to consider from techImpacts when
        ///   evaluating a functional standard and calculating credits.  Use less than or equal to 0 to ignore this parameter.</param>
        /// <param name="newCafe">The new calculated test CAFE value to use in determining the manufacturer credits.</param>
        /// <param name="useRounding">Specifies whether to round the test CAFE value before computing test credits.</param>
        /// <returns>The credits for the specified manufacturer, for all regulatory classes, operating under the given scenario
        ///   and model year.</returns>
        /// <exception cref="System.ArgumentNullException">The techImpactCount > 0 while one or more of the test arrays is null.</exception>
        public static RCDouble TestCredits(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings,
            RCDouble newCafe, TechImpact[] techImpacts, int techImpactCount, bool useRounding)
        {
            // calculate the credits for multiple regulatory classes, one at a time
            RCDouble credits = new RCDouble();
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (mfr.ModelData.Sales[regClass] != 0)
                {
                    credits[regClass] = Standards.TestCredits(mfr, scen, year, settings, regClass, newCafe[regClass], techImpacts, techImpactCount, useRounding);
                }
            }
            return credits;
        }
        /// <summary>
        /// Calculates and returns the credits for the specified manufacturer, based on the given regulatory class assignment of
        /// vehicles, operating under the given scenario and model year, using the specified test vehicles and vehicle test
        /// attributes.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the credits for the given regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the manufacturer
        ///   credits.</param>
        /// <param name="techImpacts">An array of technology impacts and vehicles to examine when evaluating a functional
        ///   standard and calculating credits.  Use null to ignore this parameter.</param>
        /// <param name="techImpactCount">The number of technology impacts and vehicles to consider from techImpacts when
        ///   evaluating a functional standard and calculating credits.  Use less than or equal to 0 to ignore this parameter.</param>
        /// <param name="newCafe">The new calculated test CAFE value to use in determining the manufacturer credits.</param>
        /// <param name="useRounding">Specifies whether to round the test CAFE value before computing test credits.</param>
        /// <returns>The credits for the specified manufacturer, based on the given regulatory class assignment of vehicles,
        ///   operating under the given scenario and model year.</returns>
        /// <exception cref="System.ArgumentNullException">The techImpactCount > 0 while one or more of the test arrays is null.</exception>
        public static double TestCredits(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings,
            RC regClass, double newCafe, TechImpact[] techImpacts, int techImpactCount, bool useRounding)
        {
            // get the runtime data for the manufacturer
            Manufacturer.CModelData mmd = mfr.ModelData;
            // get the sales and the test standard
            double sales    = mmd.Sales[regClass];
            double standard = Standards.GetStandard(mfr, scen, year, regClass, techImpacts, techImpactCount);
            // adjust CAFE value for FFV credits
            newCafe += mfr.Description.GetFFVCredits(regClass, year.Year);
            // calculate and return the credits for the manufacturer / reg class
            if (regClass == RegulatoryClass.LightTruck2b3)
            {
                return (useRounding) ?
                    Math.Round((100 / standard - Math.Round(100 / newCafe, 2)) * sales * 100) :
                    (100 / standard - 100 / newCafe) * sales * 100;
            }
            else
            {
                return (useRounding) ?
                    Math.Round((Math.Round(newCafe, 1) - standard) * sales * 10) :
                    (newCafe - standard) * sales * 10;
            }
        }

        #endregion

        #region /* CAFE calculations */

        /// <summary>
        /// Calculates the CAFE value for the specified manufacturer, for all regulatory classes, in the given model year. When
        /// this function completes, the manufacturer's sales, sales/FE, and CAFE values will be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the CAFE value, per regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        public static void CalcCAFE(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            // calculate the cafe levels for multiple regulatory classes, one at a time
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (mfr.ModelData.Sales[regClass] != 0)
                {
                    Standards.CalcCAFE(mfr, scen, year, settings, regClass);
                }
            }
        }
        /// <summary>
        /// Calculates the CAFE value for the specified manufacturer, for a specific regulatory class assignment of vehicles, in
        /// the given model year.  When this function completes, the manufacturer's sales, sales/FE, and CAFE values will be updated.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the CAFE value for the given regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles for which to calculate the manufacturer CAFE value.</param>
        public static void CalcCAFE(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings, RC regClass)
        {
            double sales    = mfr.ModelData.Sales[regClass];
            double sofe     = 0;
            double sofe2Bag = 0;

            // get the model year vehicles
            List<Vehicle> vehs = mfr.Vehicles;

            // scan each vehicle to sum the total sales and sales over fuel-economy
            for (int i = 0, vehCount = vehs.Count; i < vehCount; i++)
            {
                Vehicle              veh      = vehs[i];
                Vehicle.CDescription vd       = veh.Description;
                double               vehSales = vd.Sales[year.Index];
                //
                if (vehSales != 0 && veh.RegClass == regClass)
                {
                    double vehFE       = RoundFE(Standards.GetAverageFuelEconomy(veh, scen, year, settings, true, true , true ), regClass);
                    double vehFE2Bag   = RoundFE(Standards.GetAverageFuelEconomy(veh, scen, year, settings, true, false, false), regClass);
                    double vehSOFE     = vehSales / vehFE;
                    double vehSOFE2Bag = vehSales / vehFE2Bag;
                    // update mfr-level aggregation values
                    //sales    += vehSales;
                    sofe     += vehSOFE;
                    sofe2Bag += vehSOFE2Bag;
                    // save sales/FE for each vehicle (for use in TestCAFE)
                    veh.ModelData.SalesOverFE = vehSOFE;
                } // end if (veh has sales; exam desired regClass)
            } // next i (veh)

            // save the sales, sales/FE, and cafe levels for the manufacturer / reg class
            Manufacturer.CModelData mmd = mfr.ModelData;
            //mmd.Sales      [regClass] =  sales;
            mmd.SalesOverFE[regClass] =  sofe;
            mmd.CAFE       [regClass] = (sales == 0) ? 0 : sales / sofe;
            mmd.CAFE_2Bag  [regClass] = (sales == 0) ? 0 : sales / sofe2Bag;
        }

        /// <summary>
        /// Calculates and returns the CAFE value for the specified manufacturer, for all regulatory classes, operating under the
        /// given model year, using the specified test vehicles and technology impacts values.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the CAFE value, per regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="techImpacts">An array of technology impacts and vehicles to examine when evaluating a functional
        ///   standard and calculating CAFE.  Use null to ignore this parameter.</param>
        /// <param name="techImpactCount">The number of technology impacts and vehicles to consider from techImpacts when
        ///   evaluating a functional standard and calculating CAFE.  Use less than or equal to 0 to ignore this parameter.</param>
        /// <param name="useRounding">Specifies whether to round the sum of vehicle's FE and delta-FE when calculating the CAFE
        ///   value for the manufacturer.  Note that this parameter only applies to rounding the vehicle's FE value; the returned
        ///   CAFE value would still be unrounded.</param>
        /// <returns>The CAFE value for the specified manufacturer, for all regulatory classes, operating under the given model
        ///   year.</returns>
        /// <exception cref="System.ArgumentNullException">The testCount > 0 while one or more of the test arrays is null.</exception>
        /// <seealso cref="Standards.TestCAFE(Manufacturer, Scenario, ModelYear, ModelingSettings, RC, TechImpact[], int, bool)"/>
        public static RCDouble TestCAFE(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings,
            TechImpact[] techImpacts, int techImpactCount, bool useRounding)
        {
            // calculate the cafe levels for multiple regulatory classes, one at a time
            RCDouble cafe = new RCDouble();
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (mfr.ModelData.Sales[regClass] != 0)
                {
                    cafe[regClass] = Standards.TestCAFE(mfr, scen, year, settings, regClass, techImpacts, techImpactCount, useRounding);
                }
            }
            return cafe;
        }
        /// <summary>
        /// Calculates and returns the CAFE value for the specified manufacturer, based on the given regulatory class assignment
        /// of vehicles, operating under the given model year, using the specified test vehicles and technology impacts values.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the CAFE value for the given regulatory class.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="regClass">The regulatory class assignment of vehicles, based on which to calculate the manufacturer CAFE
        ///   value.</param>
        /// <param name="techImpacts">An array of technology impacts and vehicles to examine when evaluating a functional
        ///   standard and calculating CAFE.  Use null to ignore this parameter.</param>
        /// <param name="techImpactCount">The number of technology impacts and vehicles to consider from techImpacts when
        ///   evaluating a functional standard and calculating CAFE.  Use less than or equal to 0 to ignore this parameter.</param>
        /// <param name="useRounding">Specifies whether to round the sum of vehicle's FE and delta-FE when calculating the CAFE
        ///   value for the manufacturer.  Note that this parameter only applies to rounding the vehicle's FE value; the returned
        ///   CAFE value would still be unrounded.</param>
        /// <returns>The CAFE value for the specified manufacturer, based on the given regulatory class assignment of vehicles,
        ///   operating under the given model year</returns>
        /// <exception cref="ArgumentNullException">The testCount > 0 while one or more of the test arrays is null.</exception>
        /// <seealso cref="Standards.TestCAFE(Manufacturer, Scenario, ModelYear, ModelingSettings, TechImpact[], int, bool)"/>
        public static double TestCAFE(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings, RC regClass,
            TechImpact[] techImpacts, int techImpactCount, bool useRounding)
        {
            if (techImpactCount > 0 && (techImpacts == null || techImpacts.Length < techImpactCount))
            {   // array is null or out of bounds
                throw new ArgumentOutOfRangeException("techImpacts", "techImpacts is null or less than techImpactCount.");
            }

            // get the runtime data for the manufacturer
            Manufacturer.CModelData mmd = mfr.ModelData;

            // get the previously calculated values
            double sales = mmd.Sales      [regClass];
            double sofe  = mmd.SalesOverFE[regClass];

            // scan each test vehicle to update the "sofe" value
            for (int i = 0; i < techImpactCount; i++)
            {
                if (!techImpacts[i].HasImpact[year.Index]) { continue; }
                //
                Vehicle veh = techImpacts[i].Vehicle[year.Index];
                if (veh.RegClass == regClass)
                {
                    Vehicle.CDescription vd = veh.Description;
                    double vehSales = vd.Sales[year.Index];
                    if (vehSales != 0)
                    {
                        FuelValue adjFE    = Standards.GetAdjustedFuelEconomy(veh, techImpacts[i], scen, year, settings, true, true, true);
                        double    vehFE    = Standards.GetAverageFuelEconomy(adjFE, techImpacts[i].NewFuelShare[year.Index], regClass, scen, year);
                        double    vehSOFE  = veh.ModelData.SalesOverFE;
                        double    sofeDiff = vehSales / ((useRounding) ? RoundFE(vehFE, regClass) : vehFE) - vehSOFE;
                        //
                        sofe += sofeDiff;
                    }
                }
            }

            // calculate and return the updated cafe level for the manufacturer / reg-class
            return (sales == 0) ? 0 : sales / sofe;
        }

        #endregion

        #region /* Vehicle Fuel Economy calculations (CO-2 adjustments, harmonic averages for multi-fuel vehicles) */

        /// <summary>
        /// Computes the harmonically averaged fuel economy value from the specified fuel economy and fuel share values.
        /// </summary>
        /// <param name="fuelEconomy">The vehicle's multi-fuel fuel economy value, from which obtain the harmonically averaged
        ///   fuel economy value.</param>
        /// <param name="fuelShare">The vehicle's multi-fuel fuel share values to use for obtaining the harmonically averaged
        ///   fuel economy value.</param>
        /// <param name="regClass">The regulatory class assignment of a vehicle for which to obtain the harmonically averaged
        ///   fuel economy value.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <returns>The harmonically averaged fuel economy value based on the specified multi-fuel vehicle's fuel economy and
        ///   fuel share values.</returns>
        public static double GetAverageFuelEconomy(FuelValue fuelEconomy, FuelValue fuelShare, RegulatoryClass regClass,
            Scenario scen, ModelYear year)
        {
            if (fuelShare.IsMultiFuel)
            {   // the vehicle uses multiple fuels
                //--------------------------------------------------------------------------------------------------------------//
                // The multi-fuel mode is interpreted by the model as follows:                                                  //
                //  0:  only primary fuel economy value is considered (primary fuel share is assumed to be 100%)                //
                //  1:  for Gasoline/E85 vehicles, only the gasoline fuel economy value is considered (gasoline fuel share is   //
                //      assumed to be 100%); for Gasoline-or-Diesel/Electricity vehicles (PHEVs), both fuel economy values are  //
                //      considered (Gasoline or Diesel is always assumed to be the primary fuel, and the primary fuel share is  //
                //      assumed to be 100% - elctricity share)                                                                  //
                //  2:  for Gasoline/E85 and Gasoline-or-Diesel/Electricity vehicles, both fuel economy values are considered   //
                //      (Gasoline or Diesel is always assumed to be the primary fuel, and the primary fuel share is assumed to  //
                //      be 100% - secondary fuel share)                                                                         //
                //--------------------------------------------------------------------------------------------------------------//
                int mode = scen.ScenInfo[regClass].MultiFuel[year.Index];
                //
                if (mode == 0 || (mode == 1 && fuelShare.Electricity == 0))
                {
                    return fuelEconomy.PrimaryValue;
                }
                else if ((mode == 1 || mode == 2) && fuelShare.Electricity > 0)
                {
                    double elcFS = scen.ScenInfo[regClass].MultiFuelPHEVShare[year.Index];
                    if (elcFS == 0) { elcFS = fuelShare.Electricity; }
                    return 1 / ((1 - elcFS) / fuelEconomy.PrimaryValue + elcFS / fuelEconomy.Electricity);
                }
                else if (mode == 2 && fuelShare.Ethanol85 > 0)
                {
                    double e85FS = scen.ScenInfo[regClass].MultiFuelFFVShare[year.Index];
                    if (e85FS == 0) { e85FS = fuelShare.Ethanol85; }
                    return 1 / ((1 - e85FS) / fuelEconomy.Gasoline + e85FS / fuelEconomy.Ethanol85);
                }
                //
                throw new ArgumentException("The vehicle's fuel economy and/or fuel share values are not valid; or, incorrect multi-fuel mode specified in the scenario.");
            }
            else
            {   // only one fuel value is used
                return fuelEconomy.PrimaryValue;
            }
        }
        /// <summary>
        /// Computes the harmonically averaged fuel economy value for the specified vehicle, adjusting for the petroleum
        /// equivalency factor, the improvements in air conditioning, and off-cycle credits, where applicable.
        /// </summary>
        /// <param name="veh">The vehicle for which to obtain the harmonically averaged fuel economy value.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="applyPEFAdjustment">Whether to consider applying the petroleum equivalency factor to the vehicle for
        ///   some alternative fuels.</param>
        /// <param name="applyACAdjustment">Whether to consider applying adjustment for improvements in air conditioning to the
        ///   vehicle.  A vehicle's fuel economy will be adjusted for AC only if this parameter is true AND scenario settings
        ///   permit AC adjustments.</param>
        /// <param name="applyOffCycleCredit">Whether to consider applying off-cycle credits to the vehicle.  A vehicle's fuel
        ///   economy will be adjusted for off-cycle credits only if this parameter is true AND scenario settings permit off-cycle
        ///   credits.</param>
        /// <returns>The vehicle's harmonically averaged fuel economy value, adjusted for improvements in air conditioning and
        ///   off-cycle credits, where applicable..</returns>
        public static double GetAverageFuelEconomy(Vehicle veh, Scenario scen, ModelYear year, ModelingSettings settings,
            bool applyPEFAdjustment, bool applyACAdjustment, bool applyOffCycleCredit)
        {
            // adjust for PEF, improvements in air conditioning, and off-cycle credits
            FuelValue fuelEconomy = Standards.GetAdjustedFuelEconomy(veh, scen, year, settings, applyPEFAdjustment,
                applyACAdjustment, applyOffCycleCredit);
            // compute harmonic average for the fuel economy
            return Standards.GetAverageFuelEconomy(fuelEconomy, veh.Description.FuelShare, veh.RegClass, scen, year);
        }
        /// <summary>
        /// Computes vehicle's fuel economy, adjusted for the petroleum equivalency factor, the improvements in air conditioning,
        /// and off-cycle credits, where applicable. If applyPEFAdjustment, appylACAdjustment, and applyOffCycleCredit parameters
        /// are false, or if scenario settings do not permit fuel economy adjustments, an unadjusted fuel economy value will be
        /// returned.
        /// </summary>
        /// <param name="veh">The vehicle for which to obtain the adjusted fuel economy.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="applyPEFAdjustment">Whether to consider applying the petroleum equivalency factor to the vehicle for
        ///   some alternative fuels.</param>
        /// <param name="applyACAdjustment">Whether to consider applying adjustment for improvements in air conditioning to the
        ///   vehicle.  A vehicle's fuel economy will be adjusted for AC only if this parameter is true AND scenario settings
        ///   permit AC adjustments.</param>
        /// <param name="applyOffCycleCredit">Whether to consider applying off-cycle credits to the vehicle.  A vehicle's fuel
        ///   economy will be adjusted for off-cycle credits only if this parameter is true AND scenario settings permit off-cycle
        ///   credits.</param>
        /// <returns>The vehicle's fuel economy value, adjusted for improvements in air conditioning and off-cycle credits,
        ///   where applicable.</returns>
        public static FuelValue GetAdjustedFuelEconomy(Vehicle veh, Scenario scen, ModelYear year, ModelingSettings settings,
            bool applyPEFAdjustment, bool applyACAdjustment, bool applyOffCycleCredit)
        {
            return Standards.GetAdjustedFuelEconomy(veh, null, scen, year, settings, applyPEFAdjustment, applyACAdjustment,
                applyOffCycleCredit);
        }
        static FuelValue GetAdjustedFuelEconomy(Vehicle veh, TechImpact techImpact, Scenario scen, ModelYear year,
            ModelingSettings settings, bool applyPEFAdjustment, bool applyACAdjustment, bool applyOffCycleCredit)
        {
            // get unadjusted vehicle FE
            FuelValue fuelEconomy = (techImpact == null) ? veh.Description.FuelEconomy : techImpact.NewFuelEconomy[year.Index];

            // apply PEF adjustment
            if (applyPEFAdjustment && veh.RegClass != RegulatoryClass.LightTruck2b3)
            {
                FuelValue ed = settings.Parameters.FuelProperties.EnergyDensity;
                //
                fuelEconomy[FuelType.Ethanol85  ] = fuelEconomy.Ethanol85   / 0.15;
                fuelEconomy[FuelType.Electricity] = fuelEconomy.Electricity * 82.049 * (ed.Electricity / ed.Gasoline);
                //fuelEconomy[FuelType.Hydrogen   ] = fuelEconomy.Hydrogen    / 0.15;
                fuelEconomy[FuelType.CNG        ] = fuelEconomy.CNG         / 0.15;
            }

            // compute total CO-2 adjustment
            RegulatoryClass regClass = veh.RegClass;
            double gramsCO2 = 0;
            if (applyACAdjustment && scen.ScenInfo[regClass].IncludeAC[year.Index])
            {
                gramsCO2 += scen.ScenInfo[regClass].ACAdjustment[year.Index];
            }
            if (applyOffCycleCredit)
            {
                double cap = scen.ScenInfo[regClass].OffCycleCap[year.Index];
                double occ = veh.ModelData.OffCycleCredits;
                if (techImpact != null) { occ += techImpact.OffCycleCredit[year.Index]; }
                gramsCO2 += Math.Min(occ, cap);
            }
            if (gramsCO2 == 0) { return fuelEconomy; }

            // apply CO-2 adjustment to vehicle's fuel economy
            FuelValue fuelShare = (techImpact == null) ? veh.Description.FuelShare : techImpact.NewFuelShare[year.Index];
            FuelValue fuelCarbon = Standards.GetFuelCarbonContent(settings, true);
            //
            foreach (FuelType fuel in FTValue<object>.Classes)
            {
                if (fuelShare[fuel] > 0) { fuelEconomy[fuel] = fuelCarbon[fuel] / (fuelCarbon[fuel] / fuelEconomy[fuel] - gramsCO2); }
            }
            //
            return fuelEconomy;
        }

        /// <summary>
        /// Calculates the fuel carbon content value that can be applied to a vehicle's fuel economy when coverting between
        /// grams/mile of CO-2 and mpg.
        /// </summary>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="useGasolineForGEG">true, to use gasoline carbon content and mass density values for fuel types whose fuel
        ///   economy is specified in gasoline-equivalent gallons (GEG); false, to use "native" carbon content and mass density
        ///   values for all fuel types.</param>
        /// <returns>The fuel carbon content value that can be applied to a vehicle's fuel economy when coverting between
        ///   grams/mile of CO-2 and mpg.</returns>
        static FuelValue GetFuelCarbonContent(ModelingSettings settings, bool useGasolineForGEG)
        {
            //
            // useGasolineForGEG -- for fuel types whose FE is specified in GEG, determines whether
            //                      to use gasoline's fuel carbon content
            //
            FuelValue fuelCarbon = new FuelValue();
            //
            FuelValue co2Emissions = settings.Parameters.FuelProperties.CO2Emissions;
            //
            foreach (FuelType fuel in FTValue<object>.Classes)
            {
                if (useGasolineForGEG && (fuel == FuelType.Electricity || fuel == FuelType.Hydrogen || fuel == FuelType.CNG))
                {   // electricity, hydrogen, and CNG FE is specified in GEG -- hence, use gasoline's carbon content for AC adjustments
                    fuelCarbon[fuel] = co2Emissions.Gasoline;
                }
                else
                {
                    fuelCarbon[fuel] = co2Emissions[fuel];
                }
            }
            //
            return fuelCarbon;
        }

        #endregion

        #region /* Required and achieved CO-2 calculations */

        /// <summary>
        /// Computes the average grams/mile of CO-2 for the specified vehicle.
        /// </summary>
        /// <param name="veh">The vehicle for which to obtain the grams/mile of CO-2.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <param name="applyPEFAdjustment">Whether to consider applying the petroleum equivalency factor to the vehicle for
        ///   some alternative fuels.</param>
        /// <param name="applyACAdjustment">Whether to consider applying adjustment for improvements in air conditioning to the
        ///   vehicle.  A vehicle's fuel economy will be adjusted for AC only if this parameter is true AND scenario settings
        ///   permit AC adjustments.</param>
        /// <param name="applyOffCycleCredit">Whether to consider applying off-cycle credits to the vehicle.  A vehicle's fuel
        ///   economy will be adjusted for off-cycle credits only if this parameter is true AND scenario settings permit off-cycle
        ///   credits.</param>
        /// <returns>The average grams/mile of CO-2 for the specified vehicle.</returns>
        public static double GetVehicleCO2Rating(Vehicle veh, Scenario scen, ModelYear year, ModelingSettings settings,
            bool applyPEFAdjustment, bool applyACAdjustment, bool applyOffCycleCredit)
        {
            FuelValue       fuelEconomy = Standards.GetAdjustedFuelEconomy(veh, scen, year, settings, applyPEFAdjustment,
                                                                           applyACAdjustment, applyOffCycleCredit);
            FuelValue       fuelShare   = veh.Description.FuelShare;
            RegulatoryClass regClass    = veh.RegClass;
            //
            return Standards.GetVehicleCO2Rating(fuelEconomy, fuelShare, regClass, scen, year, settings);
        }
        /// <summary>
        /// Computes the average grams/mile of CO-2 from the specified fuel economy and fuel share values.
        /// </summary>
        /// <param name="fuelEconomy">The vehicle's multi-fuel fuel economy value, from which obtain the average grams/mile of CO-2.</param>
        /// <param name="fuelShare">The vehicle's multi-fuel fuel share values to use for obtaining the average grams/mile of CO-2.</param>
        /// <param name="regClass">The regulatory class assignment of a vehicle for which to obtain the average grams/mile of CO-2.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The average grams/mile of CO-2 from the specified fuel economy and fuel share values.</returns>
        public static double GetVehicleCO2Rating(FuelValue fuelEconomy, FuelValue fuelShare, RegulatoryClass regClass, Scenario scen,
            ModelYear year, ModelingSettings settings)
        {
            // compute fuel CO2 for each fuel type
            FuelValue fuelCarbon = Standards.GetFuelCarbonContent(settings, true);
            FuelValue fuelCO2    = new FuelValue();
            foreach (FuelType fuel in FTValue<object>.Classes)
            {
                if (fuelShare[fuel] > 0) { fuelCO2[fuel] = (fuelCarbon[fuel] / fuelEconomy[fuel]); }
            }

            // mix multi-fuel vehicles
            if (fuelShare.IsMultiFuel)
            {   // the vehicle uses multiple fuels
                //--------------------------------------------------------------------------------------------------------------//
                // The multi-fuel mode is interpreted by the model as follows:                                                  //
                //  0:  only primary fuel economy value is considered (primary fuel share is assumed to be 100%)                //
                //  1:  for Gasoline/E85 vehicles, only the gasoline fuel economy value is considered (gasoline fuel share is   //
                //      assumed to be 100%); for Gasoline-or-Diesel/Electricity vehicles (PHEVs), both fuel economy values are  //
                //      considered (Gasoline or Diesel is always assumed to be the primary fuel, and the primary fuel share is  //
                //      assumed to be 100% - elctricity share)                                                                  //
                //  2:  for Gasoline/E85 and Gasoline-or-Diesel/Electricity vehicles, both fuel economy values are considered   //
                //      (Gasoline or Diesel is always assumed to be the primary fuel, and the primary fuel share is assumed to  //
                //      be 100% - secondary fuel share)                                                                         //
                //--------------------------------------------------------------------------------------------------------------//
                int mode = scen.ScenInfo[regClass].MultiFuel[year.Index];
                //
                if (mode == 0 || (mode == 1 && fuelShare.Electricity == 0))
                {
                    return fuelCO2.PrimaryValue;
                }
                else if ((mode == 1 || mode == 2) && fuelShare.Electricity > 0)
                {
                    double elcFS = scen.ScenInfo[regClass].MultiFuelPHEVShare[year.Index];
                    if (elcFS == 0) { elcFS = fuelShare.Electricity; }
                    return (1 - elcFS) * fuelCO2.PrimaryValue + elcFS * fuelCO2.Electricity;
                }
                else if (mode == 2 && fuelShare.Ethanol85 > 0)
                {
                    double e85FS = scen.ScenInfo[regClass].MultiFuelFFVShare[year.Index];
                    if (e85FS == 0) { e85FS = fuelShare.Ethanol85; }
                    return (1 - e85FS) * fuelCO2.Gasoline + e85FS * fuelCO2.Ethanol85;
                }
                //
                throw new ArgumentException("The vehicle's fuel economy and/or fuel share values are not valid; or, incorrect multi-fuel mode specified in the scenario.");
            }
            else
            {   // only one fuel value is used
                return fuelCO2.PrimaryValue;
            }
        }
        /// <summary>
        /// Returns the CO-2 target, in grams per mile, for the specified vehicle, scenario, and model year.
        /// </summary>
        /// <param name="veh">The vehicle for which to compute the CO-2 target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The CO-2 target, in grams per mile, for the specified vehicle, scenario, and model year.</returns>
        public static double GetVehicleCO2Target(Vehicle veh, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            double    feTarget   = Standards.GetTarget(veh, scen, year);
            FuelValue fuelCarbon = Standards.GetFuelCarbonContent(settings, true);
            FuelType  fuel       = veh.Description.FuelShare.FuelType;
            bool      isDSL      = ((fuel & FuelType.Diesel) == FuelType.Diesel);
            bool      isCNG      = ((fuel & FuelType.CNG   ) == FuelType.CNG   );
            //
            return ((isDSL) ? fuelCarbon.Diesel   :
                    (isCNG) ? fuelCarbon.CNG      :
                              fuelCarbon.Gasoline) * feTarget;
        }

        /// <summary>
        /// Computes the average grams/mile of CO-2 for the specified array of vehicles.
        /// </summary>
        /// <param name="vehs">The array of vehicles for which to obtain the average grams/mile of CO-2.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The average grams/mile of CO-2 for the specified array of vehicles.</returns>
        public static RCDouble GetVehicleCO2Rating(Vehicle[] vehs, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            return Standards.CalcAggregateCO2(vehs, scen, year, settings, false, true, true, true);
        }
        /// <summary>
        /// Computes the average CO-2 target, in grams per mile, for the specified array of vehicles.
        /// </summary>
        /// <param name="vehs">The array of vehicles for which to compute the average CO-2 target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The CO-2 target, in grams per mile, for the specified array of vehicles.</returns>
        public static RCDouble GetVehicleCO2Target(Vehicle[] vehs, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            return Standards.CalcAggregateCO2(vehs, scen, year, settings, true, true, true, true);
        }
        static RCDouble CalcAggregateCO2(Vehicle[] vehs, Scenario scen, ModelYear year, ModelingSettings settings, bool calcTarget,
            bool applyPEF, bool applyAC, bool applyOffCycle)
        {
            RCDouble salesSum = RCDouble.Zero;
            RCDouble co2Sum   = RCDouble.Zero;
            foreach (Vehicle veh in vehs)
            {
                double   co2   = (calcTarget) ? Standards.GetVehicleCO2Target(veh, scen, year, settings) :
                                                Standards.GetVehicleCO2Rating(veh, scen, year, settings, applyPEF, applyAC, applyOffCycle);
                double   sales = veh.Description.Sales[year.Index];
                RC       rc    = veh.RegClass;
                co2Sum  [rc]  += sales * co2;
                salesSum[rc]  += sales;
            }
            foreach (RegulatoryClass regClass in RCValue<object>.Classes)
            {
                if (salesSum[regClass] != 0) { co2Sum[regClass] /= salesSum[regClass]; }
            }
            return co2Sum;
        }

        /// <summary>
        /// Computes the average grams/mile of CO-2 for the specified manufacturer.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to obtain the grams/mile of CO-2.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The average grams/mile of CO-2 for the specified manufacturer.</returns>
        public static RCDouble GetManufacturerCO2Rating(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            return Standards.GetVehicleCO2Rating(mfr.Vehicles.ToArray(), scen, year, settings);
        }
        /// <summary>
        /// Computes the CO-2 target, in grams per mile, for the specified manufacturer.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to compute the CO-2 target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The CO-2 target, in grams per mile, for the specified manufacturer.</returns>
        public static RCDouble GetManufacturerCO2Target(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            return Standards.GetVehicleCO2Target(mfr.Vehicles.ToArray(), scen, year, settings);
        }
        /// <summary>
        /// Computes the average grams/mile of CO-2 for the industry.
        /// </summary>
        /// <param name="ind">The industry for which to obtain the grams/mile of CO-2.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The average grams/mile of CO-2 for the industry.</returns>
        public static RCDouble GetIndustryCO2Rating(Industry ind, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            return Standards.GetVehicleCO2Rating(ind.Vehicles.ToArray(), scen, year, settings);
        }
        /// <summary>
        /// Computes the CO-2 target, in grams per mile, for the industry.
        /// </summary>
        /// <param name="ind">The industry for which to compute the CO-2 target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The CO-2 target, in grams per mile, for the industry.</returns>
        public static RCDouble GetIndustryCO2Target(Industry ind, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            return Standards.GetVehicleCO2Target(ind.Vehicles.ToArray(), scen, year, settings);
        }

        #endregion

        #region /* ZEV credit calculations */

        /// <summary>
        /// Calculates the ZEV target for the industry.
        /// </summary>
        /// <param name="ind">The industry for which to calculate the ZEV target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The ZEV target for the industry.</returns>
        public static double GetIndustryZEVTarget(Industry ind, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            double d = 0;
            foreach (Manufacturer mfr in ind.Manufacturers)
            {
                d += GetManufacturerZEVTarget(mfr, scen, year, settings);
            }
            return d;
        }
        /// <summary>
        /// Calculates the ZEV credits accumulated by the industry.
        /// </summary>
        /// <param name="ind">The industry for which to calculate the ZEV credits.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The ZEV credits accumulated by the industry.</returns>
        public static double GetIndustryZEVCredits(Industry ind, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            double d = 0;
            foreach (Manufacturer mfr in ind.Manufacturers)
            {
                d += GetManufacturerZEVCredits(mfr, scen, year, settings);
            }
            return d;
        }
        /// <summary>
        /// Calculates the ZEV target for the specified manufacturer.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the ZEV target.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The ZEV target for the manufacturer.</returns>
        public static double GetManufacturerZEVTarget(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            var    md          = mfr.Description;
            var    mmd         = mfr.ModelData;
            double mfrSales    = mmd.Sales[RC.PassengerCar] + mmd.Sales[RC.LightTruck];
            double caS177Sales = mfrSales * md.ZEVSalesShare;
            double zevReqPct   = settings.Parameters.ZEVCreditValues.GetZEVRequirement(year.Year);
            double zevTarget   = caS177Sales * zevReqPct;
            return zevTarget;
        }
        /// <summary>
        /// Calculates the ZEV credits accumulated by the specified manufacturer.
        /// </summary>
        /// <param name="mfr">The manufacturer for which to calculate the ZEV credits.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="settings">The modeling settings used by the compliance model.</param>
        /// <returns>The ZEV credits accumulated by the manufacturer.</returns>
        public static double GetManufacturerZEVCredits(Manufacturer mfr, Scenario scen, ModelYear year, ModelingSettings settings)
        {
            var    techList    = settings.Technologies.TechnologyList;
            var    zevParams   = settings.Parameters.ZEVCreditValues;
            var    md          = mfr.Description;
            var    mmd         = mfr.ModelData;
            double mfrSales    = mmd.Sales[RC.PassengerCar] + mmd.Sales[RC.LightTruck];
            double caS177Sales = mfrSales * md.ZEVSalesShare;
            double phevCap     = caS177Sales * zevParams.GetPHEVCap(year.Year);
            double phevCredits = 0;
            double evCredits   = 0;
            // aggregate phev/ev/fcv credits
            for (int i = 0; i < mfr.Vehicles.Count; i++)
            {
                Vehicle veh = mfr.Vehicles[i];
                if (veh.RegClass == RC.LightTruck2b3) { continue; }
                //
                double  zev = veh.ModelData.ZEVCredits * veh.Description.Sales[year.Index];
                if      (veh.HEVType == HEVType.PlugInHybrid) { phevCredits += zev; }
                else if (veh.HEVType == HEVType.FuelCell ||
                         veh.HEVType == HEVType.PureElectric) { evCredits   += zev; }
            }
            double zevCredits = Math.Min(phevCap, phevCredits * md.ZEVCreditShare) + evCredits * md.ZEVCreditShare;
            return zevCredits;
        }

        #endregion

        #region /* Regulatory Incentives calculations (tax credits, etc.) */

        /// <summary>
        /// For all vehicles within the specified industry, computes the amount of tax credit realized by the buyer for purchasing
        /// a specific vehicle.
        /// </summary>
        /// <param name="data">The industry data for which to compute vehicle tax credit.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        public static void CalcVehicleTaxCredit(Industry data, Scenario scen, ModelYear year)
        {
            List<Manufacturer> mfrs = data.Manufacturers;
            for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
            {
                List<Vehicle> vehs = mfrs[i].Vehicles;
                for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                {
                    Vehicle veh = vehs[j];
                    veh.ModelData.TaxCredit = Standards.GetVehicleTaxCredit(veh, scen, year);
                }
            }
        }
        /// <summary>
        /// Computes the amount of tax credit realized by the buyer for purchasing the specified vehicle.
        /// </summary>
        /// <param name="veh">The vehicle for which to obtain the amount of tax credit realized by the buyer after purchase.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <returns>The amount of tax credit realized by the buyer for purchasing the specified vehicle.</returns>
        public static double GetVehicleTaxCredit(Vehicle veh, Scenario scen, ModelYear year)
        {
            ScenarioInfo scenInfo = scen.ScenInfo[veh.RegClass];
            return (veh.HEVType == HEVType.PureElectric) ? scenInfo.EVTaxCredit  [year.Index] :
                   (veh.HEVType == HEVType.PlugInHybrid) ? scenInfo.PHEVTaxCredit[year.Index] : 0;
        }

        #endregion

        #region /* Other, misc, and support methods */

        /// <summary>
        /// Computes the harmonically averaged vehicle fuel cost value from the specified multi-fuel vehicle's fuel economy and fuel
        /// share values, given the specified onroad fuel economy gap and the fuel prices for the remaining lifetime of the vehicle.
        /// </summary>
        /// <param name="fuelEconomy">The vehicle's multi-fuel fuel economy value, from which obtain the harmonically averaged
        ///   vehicle fuel cost value.</param>
        /// <param name="fuelShare">The vehicle's multi-fuel fuel share values to use for obtaining the harmonically averaged
        ///   vehicle fuel cost value.</param>
        /// <param name="regClass">The regulatory class assignment of a vehicle for which to obtain the fuel cost.</param>
        /// <param name="scen">The scenario currently being analyzed.</param>
        /// <param name="year">The model year currently being analyzed.</param>
        /// <param name="gap">The onroad fuel economy gap.</param>
        /// <param name="fuelCost">The fuel prices for the remaining lifetime of the vehicle.</param>
        /// <returns>The harmonically averaged vehicle fuel cost value from the specified multi-fuel vehicle's fuel economy and
        ///   fuel share values, given the specified onroad fuel economy gap and the fuel prices for the remaining lifetime of
        ///   the vehicle.</returns>
        public static double GetVehicleFuelCost(FuelValue fuelEconomy, FuelValue fuelShare, RegulatoryClass regClass,
            Scenario scen, ModelYear year, FuelValue gap, FuelValue fuelCost)
        {
            FuelType primaryFuel = fuelEconomy.PrimaryFuel;
            //
            if (fuelShare.IsMultiFuel)
            {   // the vehicle uses multiple fuels
                //--------------------------------------------------------------------------------------------------------------//
                // The multi-fuel mode is interpreted by the model as follows:                                                  //
                //  0:  only primary fuel economy value is considered (primary fuel share is assumed to be 100%)                //
                //  1:  for Gasoline/E85 vehicles, only the gasoline fuel economy value is considered (gasoline fuel share is   //
                //      assumed to be 100%); for Gasoline-or-Diesel/Electricity vehicles (PHEVs), both fuel economy values are  //
                //      considered (Gasoline or Diesel is always assumed to be the primary fuel, and the primary fuel share is  //
                //      assumed to be 100% - elctricity share)                                                                  //
                //  2:  for Gasoline/E85 and Gasoline-or-Diesel/Electricity vehicles, both fuel economy values are considered   //
                //      (Gasoline or Diesel is always assumed to be the primary fuel, and the primary fuel share is assumed to  //
                //      be 100% - secondary fuel share)                                                                         //
                //--------------------------------------------------------------------------------------------------------------//
                int mode = scen.ScenInfo[regClass].MultiFuel[year.Index];
                //
                if (mode == 0 || (mode == 1 && fuelShare.Electricity == 0))
                {
                    return fuelCost[primaryFuel] / (fuelEconomy.PrimaryValue * (1 - gap[primaryFuel]));
                }
                else if ((mode == 1 || mode == 2) && fuelShare.Electricity > 0)
                {
                    double elcFS = scen.ScenInfo[regClass].MultiFuelPHEVShare[year.Index];
                    if (elcFS == 0) { elcFS = fuelShare.Electricity; }
                    return fuelCost[primaryFuel] / (fuelEconomy[primaryFuel] * (1 - gap[primaryFuel])) * (1 - elcFS) +
                           fuelCost.Electricity  / (fuelEconomy.Electricity  * (1 - gap.Electricity )) *      elcFS;
                }
                else if (mode == 2 && fuelShare.Ethanol85 > 0)
                {
                    double e85FS = scen.ScenInfo[regClass].MultiFuelFFVShare[year.Index];
                    if (e85FS == 0) { e85FS = fuelShare.Ethanol85; }
                    return fuelCost[primaryFuel] / (fuelEconomy[primaryFuel] * (1 - gap[primaryFuel])) * (1 - e85FS) +
                           fuelCost.Ethanol85    / (fuelEconomy.Ethanol85    * (1 - gap.Ethanol85   )) *      e85FS;
                }
                //
                throw new ArgumentException("The vehicle's fuel economy and/or fuel share values are not valid; or, incorrect multi-fuel mode specified in the scenario.");
            }
            else
            {   // only one fuel value is used
                return fuelCost[primaryFuel] / (fuelEconomy.PrimaryValue * (1 - gap[primaryFuel]));
            }
        }

        /// <summary>
        /// Obtains the fucntional type and coefficients for the specified regulatroy class, based on the given scenario and
        /// model year.  The results are stored in fncType and fncCoeff parameters.  The return value of this function represents
        /// the number of coefficients available.
        /// </summary>
        /// <param name="regClass">The regulatroy class from which to obtain functional coefficients.</param>
        /// <param name="scen">The scenario currently being modeled.</param>
        /// <param name="year">The model year currently being modeled.</param>
        /// <param name="fncType">When the function returns, contains the determined functional type for the specified vehicle,
        ///   based on the given scenario and model year.</param>
        /// <param name="fncCoeff">When the function returns, contains the determined functional coefficients for the specified
        ///   vehicle, based on the given scenario and model year.</param>
        /// <returns>The number of coefficients in the scenario for the model year.</returns>
        public static int GetCoefficients(RC regClass, Scenario scen, ModelYear year, out int fncType, out double[] fncCoeff)
        {
            fncType  = scen.ScenInfo[regClass].Function    [year.Index];
            fncCoeff = scen.ScenInfo[regClass].Coefficients[year.Index];
            // return number of coefs
            return fncCoeff.Length;
        }

        #endregion

        static double RoundFE(double value, RC regClass)
        {
            return (regClass == RC.LightTruck2b3) ? 100 / Math.Round(100 / value, 2) : Math.Round(value, 1);
        }

        #endregion

    }
}
