using System;
using System.Globalization;
using Volpe.Cafe.Utils;

namespace Volpe.Cafe
{
    /// <summary>
    /// Provides an object for converting model year values to the <see cref="MinYear"/>-based indexes and vice-versa.
    /// </summary>
    /// <remarks>
    /// The static <see cref="ModelYear.ToIndex(int)"/> function may be used to convert any year to the equivalent index, and the
    /// static <see cref="ModelYear.FromIndex(int)"/> function may be used to convert an index to the equivalent model year.
    /// </remarks>
    [Serializable]
    public class ModelYear : IComparable
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="ModelYear"/> class using the specified year value.
        /// </summary>
        /// <param name="year">The model year value to assign to the instance.</param>
        /// <exception cref="ArgumentOutOfRangeException">The specified model year is less than <see cref="MinYear"/>.</exception>
        public ModelYear(int year)
        {
            if (year < ModelYear.MinYear) { throw new ArgumentOutOfRangeException("year", year, ErrorStrings.ModelYearTooSmall); }
            this.Year  = year;
            this.Index = year - ModelYear.MinYear;
        }

        #endregion


        #region /*** Methods ***/

        #region /* IComparable Members */

        /// <summary>
        /// Compares the current <see cref="ModelYear"/> instance with the specified numeric or <see cref="ModelYear"/> value.
        /// </summary>
        /// <param name="obj">An object to compare with this instance.</param>
        /// <returns>A 32-bit signed integer that indicates the relative order of the comparands.
        /// The return value has these meanings:
        /// <list type="table">
        ///   <listheader>
        ///     <term>Value</term>
        ///     <description>Meaning</description>
        ///   </listheader>
        ///   <item>
        ///     <term>Less than zero</term>
        ///     <description>This <see cref="ModelYear"/> instance is less than obj.</description>
        ///   </item>
        ///   <item>
        ///     <term>Zero</term>
        ///     <description>This <see cref="ModelYear"/> instance is equal to obj.</description>
        ///   </item>
        ///   <item>
        ///     <term>Greater than zero</term>
        ///     <description>This <see cref="ModelYear"/> instance is greater than obj.</description>
        ///   </item>
        /// </list>
        /// </returns>
        /// <exception cref="System.ArgumentException">obj is not numeric or a <see cref="ModelYear"/> instance.</exception>
        public int CompareTo(object obj)
        {
            int year = 0;
            //
            if      (obj is ModelYear          ) { year = ((ModelYear)obj).Year; }
            else if (Interaction.IsNumeric(obj)) { year = Interaction.GetInt32(obj); }
            else                                 { throw new ArgumentException(ErrorStrings.ObjectIsNotModelYearOrNumeric); }
            //
            return this.Year.CompareTo(year);
        }

        #endregion

        #region /* Overriden from object */

        /// <summary>
        /// Returns the string representation of the current <see cref="ModelYear"/> instance.
        /// </summary>
        /// <returns>The string representation of the current <see cref="ModelYear"/> instance.</returns>
        public override string ToString()
        {
            return this.Year.ToString();
        }

        /// <summary>
        /// Serves as a hash function for a particular type, suitable for use in hashing algorithms and data structures like a
        /// hash table.
        /// </summary>
        /// <returns>A hash code for the current <see cref="ModelYear"/> instance.</returns>
        public override int GetHashCode()
        {
            return this.Year.GetHashCode();
        }

        /// <summary>
        /// Determines whether the specified <see cref="object"/> is equal to the current <see cref="ModelYear"/> instance.
        /// </summary>
        /// <param name="obj">The <see cref="object"/> to compare with the current <see cref="ModelYear"/>.</param>
        /// <returns>true, if the specified <see cref="object"/> is equal to the current <see cref="ModelYear"/>; false,
        ///   otherwise.</returns>
        /// <exception cref="System.ArgumentNullException">obj is null.</exception>
        /// <exception cref="System.FormatException">obj is not a <see cref="ModelYear"/> instance or a numeric value.</exception>
        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                throw new ArgumentNullException("obj");
            }
            return (this.CompareTo(obj) == 0);
        }
        /// <summary>
        /// Determines whether the specified year is equal to the current <see cref="ModelYear"/> instance.
        /// </summary>
        /// <param name="year">The year to compare with the current <see cref="ModelYear"/>.</param>
        /// <returns>true, if the specified year is equal to the current <see cref="ModelYear"/>; false, otherwise.</returns>
        public bool Equals(ModelYear year)
        {
            return (this.Year == year.Year);
        }
        /// <summary>
        /// Determines whether the specified year is equal to the current <see cref="ModelYear"/> instance.
        /// </summary>
        /// <param name="year">The year to compare with the current <see cref="ModelYear"/>.</param>
        /// <returns>true, if the specified year is equal to the current <see cref="ModelYear"/>; false, otherwise.</returns>
        public bool Equals(int year)
        {
            return (this.Year == year);
        }
        /// <summary>
        /// Determines whether the specified index of the year is equal to the current <see cref="ModelYear"/> instance.
        /// </summary>
        /// <param name="index">The index of the year to compare with the current <see cref="ModelYear"/>.</param>
        /// <returns>true, if the specified index of the year is equal to the current <see cref="ModelYear"/>; false, otherwise.</returns>
        public bool EqualsIndex(int index)
        {
            return (this.Index == index);
        }

        #endregion

        // ----- static methods -----
        /// <summary>
        /// Returns the <see cref="MinYear"/>-based index of the given model year.
        /// </summary>
        /// <param name="year">The model year to convert.</param>
        /// <returns>The <see cref="MinYear"/>-based index of the given model year.</returns>
        /// <exception cref="ArgumentOutOfRangeException">The specified year is less than <see cref="MinYear"/>.</exception>
        public static int ToIndex(int year)
        {
            if (year < ModelYear.MinYear) { throw new ArgumentOutOfRangeException("year", year, ErrorStrings.ModelYearTooSmall); }
            return year - ModelYear.MinYear;
        }
        /// <summary>
        /// Returns the <see cref="MinYear"/>-based equivalent model year of the given index.
        /// </summary>
        /// <param name="index">The model year index to convert.</param>
        /// <returns>The <see cref="MinYear"/>-based equivalent model year of the given index.</returns>
        public static int FromIndex(int index)
        {
            return index + ModelYear.MinYear;
        }
        /// <summary>
        /// Determines whether the specified model year is valid, that is, if it is greater than or equal to <see cref="MinYear"/>.
        /// </summary>
        /// <param name="year">The model year to test for validity.</param>
        /// <returns>true, if year is valid; false, otherwise.</returns>
        public static bool IsValid(int year)
        {
            return (year >= ModelYear.MinYear);
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="ModelYear"/> class using the specified index value.
        /// </summary>
        /// <param name="index">The index of the model year to assign to the instance.</param>
        /// <returns>A new <see cref="ModelYear"/> instance.</returns>
        /// <exception cref="ArgumentOutOfRangeException">The specified index is less than 0.</exception>
        public static ModelYear NewModelYearFromIndex(int index)
        {
            return new ModelYear(index + ModelYear.MinYear);
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="ModelYear"/> class using the specified year value.
        /// </summary>
        /// <param name="year">The model year value to assign to the instance.</param>
        /// <returns>A new <see cref="ModelYear"/> instance.</returns>
        /// <exception cref="ArgumentOutOfRangeException">The specified model year is less than <see cref="MinYear"/>.</exception>
        public static ModelYear NewModelYearFromYear(int year)
        {
            return new ModelYear(year);
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the <see cref="MinYear"/>-based index of the <see cref="ModelYear"/> instance.</summary>
        public int Index { get; private set; }
        /// <summary>Gets the actual year of the <see cref="ModelYear"/> instance.</summary>
        public int Year { get; private set; }

        #endregion


        #region /*** Variables ***/

        // ----- constants and readonly varaibles -----
        /// <summary>Represents the minimum allowable modeling year value.</summary>
        public const int MinYear = 2000;
        /// <summary>Represents the <see cref="ModelYear"/> instance based on the minimum allowable modeling year value.</summary>
        public static readonly ModelYear MinModelYear = new ModelYear(ModelYear.MinYear);

        #endregion

    }
}
