﻿#region << Using Directives >>
using System;
using Volpe.Cafe.Data;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Generic;
#endregion

namespace Volpe.Cafe.Settings
{
    #region /*** Helper Classes ***/

    /// <summary>
    /// Provides economic values for benefits computations.
    /// </summary>
    [Serializable]
    public sealed class EconomicValues
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        EconomicValues() { }
        internal EconomicValues(
            VCValue<double> reboundEffect, VCValue<double> discRate, VCValue<int> drBaseYear,
            VCObject<VMTGrowthRate> vmtGrowthRate,
            VCValue<FuelValue> onRoadGap, VCValue<FuelValue> refuelTime, VCValue<double> refuelTank, VCValue<double> vehTravelTimeValue,
            EconomicCosts economicCosts, VCObject<ExternalCosts> externalCosts, VCObject<OperatingCosts> operatingCosts)
        {
            this.ReboundEffect          = reboundEffect;
            this.DiscountRate           = discRate;
            this.DRBaseYear             = drBaseYear;
            //
            this.VMTGrowthRate          = vmtGrowthRate;
            this.OnRoadGap              = onRoadGap;
            this.RefuelTime             = refuelTime;
            this.RefuelTankVolume       = refuelTank;
            this.VehicleTravelTimeValue = vehTravelTimeValue;
            //
            this.EconomicCosts          = economicCosts;
            this.ExternalCosts          = externalCosts;
            this.OperatingCosts         = operatingCosts;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EconomicValues"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EconomicValues"/>.</returns>
        internal EconomicValues Clone()
        {
            EconomicValues value = new EconomicValues();
            //
            value.ReboundEffect          = this.ReboundEffect         .Clone();
            value.DiscountRate           = this.DiscountRate          .Clone();
            value.DRBaseYear             = this.DRBaseYear            .Clone();
            //
            value.VMTGrowthRate          = this.VMTGrowthRate         .Clone();
            value.OnRoadGap              = this.OnRoadGap             .Clone();
            value.RefuelTime             = this.RefuelTime            .Clone();
            value.RefuelTankVolume       = this.RefuelTankVolume      .Clone();
            value.VehicleTravelTimeValue = this.VehicleTravelTimeValue.Clone();
            //
            value.EconomicCosts          = this.EconomicCosts         .Clone();
            value.ExternalCosts          = this.ExternalCosts         .Clone();
            value.OperatingCosts         = this.OperatingCosts        .Clone();
            //
            return value;
        }

        /// <summary>
        /// Updates the parameters with the specified values for a new Monte-Carlo trial.
        /// </summary>
        internal void UpdateForMonteCarlo(double reboundEffect, double gapGas, double gapDsl)
        {
            this.ReboundEffect = new VCValue<double>();
            for (int i = 0; i < this.ReboundEffect.Items.Length; i++)
            {
                this.ReboundEffect.Items[i]                       = reboundEffect;
                this.OnRoadGap    .Items[i][FuelType.Gasoline   ] = gapGas;
                this.OnRoadGap    .Items[i][FuelType.Ethanol85  ] = gapGas;
                this.OnRoadGap    .Items[i][FuelType.Diesel     ] = gapDsl;
            }
            //this.ReboundEffect = reboundEffect;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the percent change in average annual VMT per vehicle resulting from a percent change in fuel cost per
        ///   mile driven.</summary>
        public VCValue<double> ReboundEffect { get; private set; }
        /// <summary>Gets the percent rate by which the dollar value of a benefit or cost is reduced when its receipt or payment
        ///   is postponed by one additional year into the future.</summary>
        public VCValue<double> DiscountRate { get; private set; }
        /// <summary>Gets the calendar year to use for "present year" discounting.
        ///   If a base year value is used, social discounting is assumed, with all costs and benefits being discounted to that
        ///   year. If 0 is used, private discounting is implied, with all costs and benefits being discounted to the model year
        ///   being analyzed.</summary>
        public VCValue<int> DRBaseYear { get; private set; }

        /// <summary>Gets the annual growth rate for average VMT per vehicle, assuming low, average, or high fuel prices are used.</summary>
        public VCObject<VMTGrowthRate> VMTGrowthRate { get; private set; }
        /// <summary>Gets the difference between a vehicle's EPA fuel economy rating and its actual on-road fuel economy, expressed
        ///   as percent.</summary>
        public VCValue<FuelValue> OnRoadGap { get; private set; }
        /// <summary>Gets the average refueling time a spent by a consumer refueling the vehicle tank or recharging the vehicle battery.</summary>
        public VCValue<FuelValue> RefuelTime { get; private set; }
        /// <summary>Gets the average tank volume refilled during a refueling stop.</summary>
        public VCValue<double> RefuelTankVolume { get; private set; }
        /// <summary>Gets the amount that the driver of a vehicle would be willing to pay to reduce the time required to make a trip.</summary>
        public VCValue<double> VehicleTravelTimeValue { get; private set; }

        /// <summary>Gets the economic costs of oil imports.</summary>
        public EconomicCosts EconomicCosts { get; private set; }
        /// <summary>Gets the external costs from additional vehicle use due to "rebound" effect.</summary>
        public VCObject<ExternalCosts> ExternalCosts { get; private set; }
        /// <summary>Gets the ownership and operating costs associated with purchase of new vehicles.</summary>
        public VCObject<OperatingCosts> OperatingCosts { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides the annual growth rate for average VMT per vehicle, assuming low, average, or high fuel prices are used.
    /// </summary>
    [Serializable]
    public sealed class VMTGrowthRate : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        VMTGrowthRate() { }
        internal VMTGrowthRate(int baseYear, double low, double average, double high)
        {
            this.BaseYear = baseYear;
            this.Low      = low;
            this.Average  = average;
            this.High     = high;
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="VMTGrowthRate"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="VMTGrowthRate"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }

        #endregion

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="VMTGrowthRate"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="VMTGrowthRate"/>.</returns>
        internal VMTGrowthRate Clone()
        {
            VMTGrowthRate value = new VMTGrowthRate();
            //
            value.BaseYear = this.BaseYear;
            value.Low      = this.Low;
            value.Average  = this.Average;
            value.High     = this.High;
            //
            return value;
        }

        /// <summary>
        /// Calculates the VMT growth factor assuming the specified fuel price estimates are used, in the specified model year,
        /// at the given vehicle age.
        /// </summary>
        /// <param name="estimates">The low, average, or high fuel price estimates to use.</param>
        /// <param name="year">The model year for which to obtain the VMT growth factor.</param>
        /// <param name="age">The vehicle age for which to obtain the VMT growth factor.</param>
        /// <returns>The VMT growth factor for the specified fuel price estimates, in the specified model year, at the given
        ///   vehicle age.</returns>
        public double GetVMTGrowthFactor(Estimates estimates, int year, int age)
        {
            double growthRate = (estimates == Estimates.VeryHigh || estimates == Estimates.High) ? this.High :
                (estimates == Estimates.Low) ? this.Low : this.Average;
            //
            return Math.Pow(1 + growthRate, year + age - this.BaseYear);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the base year for annual growth rate for average VMT per vehicle.</summary>
        public int BaseYear { get; private set; }
        /// <summary>Gets the annual growth rate for average VMT per vehicle, assuming low fuel prices.</summary>
        public double Low { get; private set; }
        /// <summary>Gets the annual growth rate for average VMT per vehicle, assuming average fuel prices.</summary>
        public double Average { get; private set; }
        /// <summary>Gets the annual growth rate for average VMT per vehicle, assuming high fuel prices.</summary>
        public double High { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides the economic costs of oil imports.
    /// </summary>
    [Serializable]
    public sealed class EconomicCosts : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        EconomicCosts() { }
        internal EconomicCosts(int minCY, int maxCY, double[] monopsony, double[] priceShock, double[] milSecurity)
        {
            this.MinCY            = minCY;
            this.MaxCY            = maxCY;
            //
            this.Monopsony        = monopsony;
            this.PriceShock       = priceShock;
            this.MilitarySecurity = milSecurity;
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EconomicCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EconomicCosts"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }

        #endregion

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EconomicCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EconomicCosts"/>.</returns>
        internal EconomicCosts Clone()
        {
            EconomicCosts value = new EconomicCosts();
            //
            value.MinCY            = this.MinCY;
            value.MaxCY            = this.MaxCY;
            //
            value.Monopsony        = this.Monopsony       .CloneArray();
            value.PriceShock       = this.PriceShock      .CloneArray();
            value.MilitarySecurity = this.MilitarySecurity.CloneArray();
            //
            return value;
        }

        /// <summary>
        /// Updates the parameters with the specified values for a new Monte-Carlo trial.
        /// </summary>
        internal void UpdateForMonteCarlo(FuelPrices oldFP, FuelPrices newFP, double priceShockAlpha)
        {
            for (int i = 0; i < this.PriceShock.Length; i++)
            {
                int year = this.MinCY + i;
                double ratio = newFP.GetFuelPrice(Estimates.Average, FuelType.Gasoline, year) /
                               oldFP.GetFuelPrice(Estimates.Average, FuelType.Gasoline, year);
                this.PriceShock[i] *= ratio * priceShockAlpha;
            }
        }

        /// <summary>
        /// Returns the total economic costs of oil imports, comprising of monopsony, price shock, and military security costs
        /// for the specified calendar year.
        /// </summary>
        /// <param name="year">The calendar year for which to obtain the total economic costs of oil imports.</param>
        /// <returns>The total economic costs of oil imports in the specified calendar year.</returns>
        public double GetTotalEconomicCosts(int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxCY) - this.MinCY, 0);
            return (this.Monopsony[index] + this.PriceShock[index] + this.MilitarySecurity[index]);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest calendar year for which economic cost data is available.</summary>
        public int MinCY { get; private set; }
        /// <summary>Gets the highest calendar year for which economic cost data is available.</summary>
        public int MaxCY { get; private set; }

        /// <summary>Gets the "monopsony" component of economic costs of oil imports, specified in $/gallon.</summary>
        public double[] Monopsony { get; private set; }
        /// <summary>Gets the price shock component of economic costs of oil imports, specified in $/gallon.</summary>
        public double[] PriceShock { get; private set; }
        /// <summary>Gets the military security component of economic costs of oil imports, specified in $/gallon.</summary>
        public double[] MilitarySecurity { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides the external costs from additional vehicle use due to "rebound" effect.
    /// </summary>
    [Serializable]
    public sealed class ExternalCosts : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        ExternalCosts() { }
        internal ExternalCosts(double congestion, double accident, double noise)
        {
            this.Congestion = congestion;
            this.Accident   = accident;
            this.Noise      = noise;
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="ExternalCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="ExternalCosts"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }

        #endregion

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="ExternalCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="ExternalCosts"/>.</returns>
        internal ExternalCosts Clone()
        {
            ExternalCosts value = new ExternalCosts();
            //
            value.Congestion = this.Congestion;
            value.Accident   = this.Accident;
            value.Noise      = this.Noise;
            //
            return value;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the congestion component of external costs from additional vehicle use due to "rebound" effect, specified
        ///   in $/vehicle-mile.</summary>
        public double Congestion { get; private set; }
        /// <summary>Gets the accident component of external costs from additional vehicle use due to "rebound" effect, specified
        ///   in $/vehicle-mile.</summary>
        public double Accident { get; private set; }
        /// <summary>Gets the noise component of external costs from additional vehicle use due to "rebound" effect, specified in
        ///   $/vehicle-mile.</summary>
        public double Noise { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides the ownership and operating costs associated with purchase of new vehicles.
    /// </summary>
    [Serializable]
    public sealed class OperatingCosts : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        OperatingCosts() { }
        internal OperatingCosts(double taxesAndFees, double financing, double insurance, double relValueLoss, double resaleValue)
        {
            this.TaxesAndFees      = taxesAndFees;
            this.Financing         = financing;
            this.Insurance         = insurance;
            this.RelativeValueLoss = relValueLoss;
            this.ResaleValue       = resaleValue;
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OperatingCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OperatingCosts"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }

        #endregion

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OperatingCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OperatingCosts"/>.</returns>
        internal OperatingCosts Clone()
        {
            OperatingCosts value = new OperatingCosts();
            //
            value.TaxesAndFees      = this.TaxesAndFees;
            value.Financing         = this.Financing;
            value.Insurance         = this.Insurance;
            value.RelativeValueLoss = this.RelativeValueLoss;
            value.ResaleValue       = this.ResaleValue;
            //
            return value;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the average percentage of the vehicle's final MSRP the consumer pays in taxes and fees when purchasing a
        ///   new vehicle.</summary>
        public double TaxesAndFees { get; private set; }
        /// <summary>Gets the average percentage of the vehicle's final MSRP the consumer would pay for financing a new vehicle.</summary>
        public double Financing { get; private set; }
        /// <summary>Gets the average percentage of the vehicle's final MSRP the consumer would pay for insuring a new vehicle.</summary>
        public double Insurance { get; private set; }
        /// <summary>Gets the average percentage of the vehicle's final MSRP, which translates into relative value loss to consumer
        ///   due to decreased operating life of pure electic vehicles.</summary>
        public double RelativeValueLoss { get; private set; }
        /// <summary>Gets the average percentage of the vehicle's final MSRP the consumer recoups after selling the vehicle.</summary>
        public double ResaleValue { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides vehicle age data for cars, vans, SUVs, and pickups.
    /// </summary>
    [Serializable]
    public sealed class VehicleAgeData : VCXObject<double[]>
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        VehicleAgeData() : base() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="VehicleAgeData"/> class.
        /// </summary>
        internal VehicleAgeData(
            double[] cars, double[] vans, double[] suvs, double[] pickups, double[] class12a, double[] zevs, double[] class2b3)
            : base(cars, vans, suvs, pickups, class12a, zevs, class2b3) { }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="VehicleAgeData"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="VehicleAgeData"/>.</returns>
        internal new VehicleAgeData Clone()
        {
            VehicleAgeData value = new VehicleAgeData();
            this.CopyTo(value);
            return value;
        }

        /// <summary>
        /// Updates the parameters with the specified values for a new Monte-Carlo trial.
        /// </summary>
        internal void UpdateForMonteCarlo(VCValue<double> miles0, VCValue<double> milesCap)
        {
            foreach (VehicleClass vehClass in VCValue<object>.Classes)
            {
                this.UpdateVMTSchedule(miles0[vehClass], this[vehClass, VehicleStyle.None, HEVType.None], milesCap[vehClass]);
            }
            // also update vehicle age data for vans, suvs, and pickups to be the same as class-1/2a trucks
            this[VehicleClass.LDT12a, VehicleStyle.Van         , HEVType.None] =
            this[VehicleClass.LDT12a, VehicleStyle.SportUtility, HEVType.None] =
            this[VehicleClass.LDT12a, VehicleStyle.Pickup      , HEVType.None] = this[VehicleClass.LDT12a, VehicleStyle.None, HEVType.None];
        }
        void UpdateVMTSchedule(double newMiles0, double[] miles, double milesCap)
        {
            double ratio = newMiles0 / miles[0];
            double sum   = newMiles0;
            miles[0]     = newMiles0;
            for (int i = 1; i < miles.Length; i++)
            {
                miles[i] *= ratio;
                sum += miles[i];
                if (sum >= milesCap) { Array.Clear(miles, i + 1, miles.Length - i - 1); break; }
            }
        }

        /// <summary>
        /// Returns the vehicle age data for the specified vehicle.
        /// </summary>
        /// <param name="veh">The vehicle for which to obtain the vehicle age data.</param>
        /// <returns>The vehicle age data for the specified vehicle.</returns>
        public double[] GetVehicleAgeData(Vehicle veh)
        {
            return this.GetVehicleAgeData(veh.VehicleClass, veh.VehicleStyle, veh.HEVType);
        }
        /// <summary>
        /// Returns the vehicle age data for the specified vehicle class and vehicle style.
        /// </summary>
        /// <param name="vehClass">The class of the vehicle for which to obtain the vehicle age data.</param>
        /// <param name="vehStyle">The style of the vehicle for which to obtain the vehicle age data.</param>
        /// <param name="hevType">The hybrid/electric type of the vehicle for which to obtain the vehicle age data.</param>
        /// <returns>The vehicle age data for the specified vehicle class and vehicle style.</returns>
        public double[] GetVehicleAgeData(VehicleClass vehClass, VehicleStyle vehStyle, HEVType hevType)
        {
            return this[vehClass, vehStyle, hevType];
        }

        #endregion

    }
    /// <summary>
    /// Provides a forecast of fuel prices per calendar year for all supported fuel types.
    /// </summary>
    [Serializable]
    public sealed class FuelPrices
    {

        #region /*** Ctors ***/
        
        // Private constructor (for cloning).
        FuelPrices() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="FuelPrices"/> class.
        /// </summary>
        internal FuelPrices(int minCY, int maxCY,
            FTObject<double[]> fuelTax, FTObject<double[]> fpLow, FTObject<double[]> fpAvg, FTObject<double[]> fpHigh)
        {
            this.MinCY            = minCY;
            this.MaxCY            = maxCY;
            //
            this.FuelTax          = fuelTax;
            this.FuelPriceLow     = fpLow;
            this.FuelPriceAverage = fpAvg;
            this.FuelPriceHigh    = fpHigh;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="FuelPrices"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="FuelPrices"/>.</returns>
        internal FuelPrices Clone()
        {
            FuelPrices value = new FuelPrices();
            //
            value.MinCY    = this.MinCY;
            value.MaxCY    = this.MaxCY;
            //
            value.FuelTax          = this.FuelTax         .Clone();
            value.FuelPriceLow     = this.FuelPriceLow    .Clone();
            value.FuelPriceAverage = this.FuelPriceAverage.Clone();
            value.FuelPriceHigh    = this.FuelPriceHigh   .Clone();
            //
            return value;
        }

        /// <summary>
        /// Updates the parameters with the specified values for a new Monte-Carlo trial.
        /// </summary>
        internal void UpdateForMonteCarlo(double fpScaleFactor, int fpStartYear)
        {
            for (int i = fpStartYear; i <= this.MaxCY; i++)
            {
                int yr = i - this.MinCY;
                foreach (FuelType fuel in FTValue<object>.Classes)
                {
                    double[] fpH = this.FuelPriceHigh   [fuel];
                    double[] fpA = this.FuelPriceAverage[fuel];
                    fpA[yr] = (1 + fpScaleFactor * (fpH[yr] / fpH[yr - 1] - 1)) * fpA[yr - 1];
                }
            }
        }

        /// <summary>
        /// Returns forecast data for fuel taxes for the specified fuel type, in the specified calendar year.
        /// </summary>
        /// <param name="fuelType">The fuel type for which to obtain the fuel taxes.</param>
        /// <param name="year">The calendar year for which to obtain the fuel taxes.</param>
        /// <returns>Forecast data for fuel taxes for the specified fuel type, in the specified calendar year.</returns>
        public double GetFuelTax(FuelType fuelType, int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxCY) - this.MinCY, 0);
            return this.FuelTax[fuelType][index];
        }
        /// <summary>
        /// Returns forecast data for fuel taxes for the specified fuel type, in the specified calendar year.
        /// </summary>
        /// <param name="fuelType">The fuel type for which to obtain the fuel taxes.</param>
        /// <param name="year">The calendar year for which to obtain the fuel taxes.</param>
        /// <param name="fuelProperties">The physical and chemical properties of various fuel types as well as fuel import
        ///   assumptions.</param>
        /// <returns>Forecast data for fuel taxes for the specified fuel type, in the specified calendar year.</returns>
        public double GetFuelTax(FuelType fuelType, int year, FuelProperties fuelProperties)
        {
            return this.ConvertToGGE(this.GetFuelTax(fuelType, year), fuelType, fuelProperties);
        }
        /// <summary>
        /// Returns forecast data for low, average, or high retail fuel price for the specified price estimates and fuel type, in
        /// the specified calendar year.
        /// </summary>
        /// <param name="priceEstimates">The low, average, or high fuel price estimates to use.</param>
        /// <param name="fuelType">The fuel type for which to obtain the retail fuel prices.</param>
        /// <param name="year">The calendar year for which to obtain the retail fuel prices.</param>
        /// <returns>Forecast data for low, average, or high retail fuel price for the specified price estimates and fuel type,
        ///   in the specified calendar year.</returns>
        public double GetFuelPrice(Estimates priceEstimates, FuelType fuelType, int year)
        {
            FTObject<double[]> fp = (priceEstimates == Estimates.Low ) ? this.FuelPriceLow  :
                                    (priceEstimates == Estimates.High) ? this.FuelPriceHigh : this.FuelPriceAverage;
            int index = Math.Max(Math.Min(year, this.MaxCY) - this.MinCY, 0);
            return fp[fuelType][index];
        }
        /// <summary>
        /// Returns forecast data for low, average, or high retail fuel price for the specified price estimates and fuel type, in
        /// the specified calendar year.
        /// </summary>
        /// <param name="priceEstimates">The low, average, or high fuel price estimates to use.</param>
        /// <param name="fuelType">The fuel type for which to obtain the retail fuel prices.</param>
        /// <param name="year">The calendar year for which to obtain the retail fuel prices.</param>
        /// <param name="fuelProperties">The physical and chemical properties of various fuel types as well as fuel import
        ///   assumptions.</param>
        /// <returns>Forecast data for low, average, or high retail fuel price for the specified price estimates and fuel type,
        ///   in the specified calendar year.</returns>
        public double GetFuelPrice(Estimates priceEstimates, FuelType fuelType, int year, FuelProperties fuelProperties)
        {
            return this.ConvertToGGE(this.GetFuelPrice(priceEstimates, fuelType, year), fuelType, fuelProperties);
        }
        double ConvertToGGE(double value, FuelType fuelType, FuelProperties fuelProperties)
        {
            FuelValue ed = fuelProperties.EnergyDensity;
            return (fuelType == FuelType.Electricity) ? value * ed.Gasoline / ed.Electricity :
                   (fuelType == FuelType.Hydrogen   ) ? value * ed.Gasoline / ed.Hydrogen    :
                   (fuelType == FuelType.CNG        ) ? value * ed.Gasoline / ed.CNG         : value;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest calendar year for which forecast fuel price data is available.</summary>
        public int MinCY { get; private set; }
        /// <summary>Gets the highest calendar year for which forecast fuel price data is available.</summary>
        public int MaxCY { get; private set; }

        /// <summary>Gets the forecast data for fuel taxes per calendar year.</summary>
        public FTObject<double[]> FuelTax { get; private set; }
        /// <summary>Gets the forecast data for low retail fuel prices per calendar year.</summary>
        public FTObject<double[]> FuelPriceLow { get; private set; }
        /// <summary>Gets the forecast data for average retail fuel prices per calendar year.</summary>
        public FTObject<double[]> FuelPriceAverage { get; private set; }
        /// <summary>Gets the forecast data for high retail fuel prices per calendar year.</summary>
        public FTObject<double[]> FuelPriceHigh { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides historic and projected fuel economy values and their associated fuel shares by model year.
    /// </summary>
    [Serializable]
    public sealed class FuelEconomyData
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        FuelEconomyData() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="FuelEconomyData"/> class.
        /// </summary>
        internal FuelEconomyData(int minMY, int maxMY, FTObject<VCObject<double[]>> fuelEconomy,
            FTObject<VCObject<double[]>> fuelShare)
        {
            this.MinMY       = minMY;
            this.MaxMY       = maxMY;
            //
            this.FuelEconomy = fuelEconomy;
            this.FuelShare   = fuelShare;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="FuelEconomyData"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="FuelEconomyData"/>.</returns>
        internal FuelEconomyData Clone()
        {
            FuelEconomyData value = new FuelEconomyData();
            //
            value.MinMY       = this.MinMY;
            value.MaxMY       = this.MaxMY;
            //
            value.FuelEconomy = this.FuelEconomy.Clone();
            value.FuelShare   = this.FuelShare  .Clone();
            //
            return value;
        }

        /// <summary>
        /// Returns the average fuel economy value for the specified fuel type and vehicle class, in the specified model year.
        /// </summary>
        /// <param name="fuelType">The fuel type for which to obtain the average fuel economy value.</param>
        /// <param name="vehClass">The vehicle class for which to obtain the average fuel economy value.</param>
        /// <param name="year">The model year for which to obtain the average fuel economy value.</param>
        /// <returns>The average fuel economy value for the specified fuel type and vehicle class, in the specified model year.</returns>
        public double GetFuelEconomy(FuelType fuelType, VehicleClass vehClass, int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxMY) - this.MinMY, 0);
            return this.FuelEconomy[fuelType][vehClass][index];
        }
        /// <summary>
        /// Returns the average fuel economy value for the specified vehicle class, in the specified model year.
        /// </summary>
        /// <param name="vehClass">The vehicle class for which to obtain the average fuel economy value.</param>
        /// <param name="year">The model year for which to obtain the average fuel economy value.</param>
        /// <returns>The average fuel economy value for the specified vehicle class, in the specified model year.</returns>
        public FuelValue GetFuelEconomy(VehicleClass vehClass, int year)
        {
            FuelValue value = new FuelValue();
            foreach (FuelType fuelType in FTValue<object>.Classes)
            {
                value[fuelType] = this.GetFuelEconomy(fuelType, vehClass, year);
            }
            return value;
        }
        /// <summary>
        /// Returns the average fuel share value for the specified fuel type and vehicle class, in the specified model year.
        /// </summary>
        /// <param name="fuelType">The fuel type for which to obtain the average fuel share value.</param>
        /// <param name="vehClass">The vehicle class for which to obtain the average fuel share value.</param>
        /// <param name="year">The model year for which to obtain the average fuel share value.</param>
        /// <returns>The average fuel share value for the specified fuel type and vehicle class, in the specified model year.</returns>
        public double GetFuelShare(FuelType fuelType, VehicleClass vehClass, int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxMY) - this.MinMY, 0);
            return this.FuelShare[fuelType][vehClass][index];
        }
        /// <summary>
        /// Returns the average fuel share value for the specified vehicle class, in the specified model year.
        /// </summary>
        /// <param name="vehClass">The vehicle class for which to obtain the average fuel share value.</param>
        /// <param name="year">The model year for which to obtain the average fuel share value.</param>
        /// <returns>The average fuel share value for the specified vehicle class, in the specified model year.</returns>
        public FuelValue GetFuelShare(VehicleClass vehClass, int year)
        {
            FuelValue value = new FuelValue();
            foreach (FuelType fuelType in FTValue<object>.Classes)
            {
                value[fuelType] = this.GetFuelShare(fuelType, vehClass, year);
            }
            return value;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest model year for which fuel economy data is available.</summary>
        public int MinMY { get; private set; }
        /// <summary>Gets the highest model year for which fuel economy data is available.</summary>
        public int MaxMY { get; private set; }

        /// <summary>Gets the historic and projected fuel economy data by model year.</summary>
        FTObject<VCObject<double[]>> FuelEconomy { get; /*private*/ set; }
        /// <summary>Gets the historic and projected fuel economy shares by model year.</summary>
        FTObject<VCObject<double[]>> FuelShare { get; /*private*/ set; }

        #endregion

    }
    /// <summary>
    /// Provides fine tuning parameters for using fleet analysis in the modeling system.
    /// </summary>
    [Serializable]
    public sealed class FleetAnalysisValues
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        FleetAnalysisValues() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="FleetAnalysisValues"/> class.
        /// </summary>
        internal FleetAnalysisValues(int minMY, int maxMY, VCValue<double> baseCAFEGrowthRates, VCValue<double> scenCAFEGrowthRates,
            VCValue<double> cafeStartYear, VCObject<double[]> salesForecast)
        {
            this.MinMY                   = minMY;
            this.MaxMY                   = maxMY;
            //
            this.BaselineCAFEGrowthRates = baseCAFEGrowthRates;
            this.ScenarioCAFEGrowthRates = scenCAFEGrowthRates;
            this.CAFEStartYear           = cafeStartYear;
            this.SalesForecast           = salesForecast;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="FleetAnalysisValues"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="FleetAnalysisValues"/>.</returns>
        internal FleetAnalysisValues Clone()
        {
            FleetAnalysisValues value = new FleetAnalysisValues();
            //
            value.MinMY                   = this.MinMY;
            value.MaxMY                   = this.MaxMY;
            //
            value.BaselineCAFEGrowthRates = this.BaselineCAFEGrowthRates.Clone();
            value.ScenarioCAFEGrowthRates = this.ScenarioCAFEGrowthRates.Clone();
            value.CAFEStartYear           = this.CAFEStartYear          .Clone();
            value.SalesForecast           = this.SalesForecast          .Clone();
            //
            return value;
        }

        /// <summary>
        /// Returns the forecast of sales for the specified vehicle class, in the specified model year.
        /// </summary>
        /// <param name="vehClass">The vehicle class for which to obtain the forecast of sales.</param>
        /// <param name="year">The model year for which to obtain the forecast of sales.</param>
        /// <returns>The forecast of sales for the specified vehicle class, in the specified model year.</returns>
        public double GetSalesForecast(VehicleClass vehClass, int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxMY) - this.MinMY, 0);
            return this.SalesForecast[vehClass][index];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest model year for which forecast of sales data is available.</summary>
        public int MinMY { get; private set; }
        /// <summary>Gets the highest model year for which forecast of sales data is available.</summary>
        public int MaxMY { get; private set; }

        /// <summary>Gets the fuel economy growth rates to apply to the baseline scenario while performing Fleet Analysis
        ///   calculations whenever the voluntary overcompliance mode is enabled. When used, growth rates take effect after the
        ///   last compliance model year.</summary>
        public VCValue<double> BaselineCAFEGrowthRates { get; private set; }
        /// <summary>Gets the fuel economy growth rates to apply to the alternative scenarios while performing Fleet Analysis
        ///   calculations whenever the voluntary overcompliance mode is enabled. When used, growth rates take effect after the
        ///   last compliance model year.</summary>
        public VCValue<double> ScenarioCAFEGrowthRates { get; private set; }
        /// <summary>Gets the model year when Fuel Economy regulations were first introduced. This value is used when evaluating
        ///   the fuel use and environmental effects assuming the absence of CAFE standards (aka, "No-CAFE" alternative).</summary>
        public VCValue<double> CAFEStartYear { get; private set; }
        /// <summary>Gets the forecast of total industry sales by model year. Forecast of Sales are used to scale individual
        ///   vehicle sales, after the last compliance model year, in order to evaluate the fuel use and environmental effects of
        ///   future years during Fleet Analysis. Additionally, these values, in conjunction with the
        ///   <see cref="HistoricFleetData"/>, are also used when evaluating the fuel use and environmental effects assuming the
        ///   absence of CAFE standards.</summary>
        public VCObject<double[]> SalesForecast { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides historic fleet data by vehicle class.
    /// </summary>
    [Serializable]
    public sealed class HistoricFleetData
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        HistoricFleetData() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="HistoricFleetData"/> class.
        /// </summary>
        internal HistoricFleetData(int minMY, int maxMY, VCObject<double[][]> fleetData)
        {
            this.MinMY     = minMY;
            this.MaxMY     = maxMY;
            //
            this.FleetData = fleetData;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="HistoricFleetData"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="HistoricFleetData"/>.</returns>
        internal HistoricFleetData Clone()
        {
            HistoricFleetData value = new HistoricFleetData();
            //
            value.MinMY     = this.MinMY;
            value.MaxMY     = this.MaxMY;
            //
            value.FleetData = this.Clone(this.FleetData);
            //
            return value;
        }
        VCObject<double[][]> Clone(VCObject<double[][]> value)
        {
            VCObject<double[][]> newValue = new VCObject<double[][]>();
            foreach (VehicleClass vehClass in VCValue<object>.Classes)
            {
                newValue[vehClass] = value[vehClass].CloneArray();
            }
            return newValue;
        }

        /// <summary>
        /// Returns the fleet data for the specified vehicle class, in the specified model year, at the given vehicle age.
        /// </summary>
        /// <param name="vehClass">The vehicle class for which to obtain the fleet data.</param>
        /// <param name="year">The model year for which to obtain the fleet data.</param>
        /// <param name="age">The vehicle age for which to obtain the fleet data.</param>
        /// <returns>The fleet data for the specified vehicle class, in the specified model year, at the given vehicle age.</returns>
        public double GetFleetData(VehicleClass vehClass, int year, int age)
        {
            int index = Math.Max(Math.Min(year, this.MaxMY) - this.MinMY, 0);
            return this.FleetData[vehClass][index][age];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest model year for which historic fleet data is available.</summary>
        public int MinMY { get; private set; }
        /// <summary>Gets the highest model year for which historic fleet data is available.</summary>
        public int MaxMY { get; private set; }

        /// <summary>Gets the historic fleet data.</summary>
        public VCObject<double[][]> FleetData { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides values neccessary for safety calculations.
    /// </summary>
    [Serializable]
    public sealed class SafetyValues
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        SafetyValues() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="SafetyValues"/> class.
        /// </summary>
        internal SafetyValues(SCValue<double> threshold,
            SCValue<double> smallEffect, SCValue<double> smallBase, SCValue<double> smallFMVSS,
            SCValue<double> largeEffect, SCValue<double> largeBase, SCValue<double> largeFMVSS,
            double fatalityCosts, double growthRate, int baseYear)
        {
            this.Threshold       = threshold;
            this.SmallEffect     = smallEffect;
            this.SmallBase       = smallBase;
            this.SmallFMVSS      = smallFMVSS;
            this.SmallMultiplier = this.CalcMultiplier(smallEffect, smallBase, smallFMVSS);
            this.LargeEffect     = largeEffect;
            this.LargeBase       = largeBase;
            this.LargeFMVSS      = largeFMVSS;
            this.LargeMultiplier = this.CalcMultiplier(largeEffect, largeBase, largeFMVSS);
            this.FatalityCosts   = fatalityCosts;
            this.GrowthRate      = growthRate;
            this.BaseYear        = baseYear;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="SafetyValues"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="SafetyValues"/>.</returns>
        internal SafetyValues Clone()
        {
            SafetyValues value = new SafetyValues();
            //
            value.Threshold       = this.Threshold      .Clone();
            value.SmallEffect     = this.SmallEffect    .Clone();
            value.SmallBase       = this.SmallBase      .Clone();
            value.SmallFMVSS      = this.SmallFMVSS     .Clone();
            value.SmallMultiplier = this.SmallMultiplier.Clone();
            value.LargeEffect     = this.LargeEffect    .Clone();
            value.LargeBase       = this.LargeBase      .Clone();
            value.LargeFMVSS      = this.LargeFMVSS     .Clone();
            value.LargeMultiplier = this.LargeMultiplier.Clone();
            value.FatalityCosts   = this.FatalityCosts;
            value.GrowthRate      = this.GrowthRate;
            value.BaseYear        = this.BaseYear;
            //
            return value;
        }

        SCValue<double> CalcMultiplier(SCValue<double> effect, SCValue<double> baseFatals, SCValue<double> fmvss)
        {
            SCValue<double> multiplier = new SCValue<double>();
            for (int i = 0; i < multiplier.Items.Length; i++)
            {
                multiplier.Items[i] = effect.Items[i] * baseFatals.Items[i] * fmvss.Items[i];
            }
            return multiplier;
        }

        /// <summary>
        /// Returns the effect of weight reduction (the change in fatalities per 100 lb reduction in curb weight) for the
        /// specified vehicle safety class and curb weight.
        /// </summary>
        /// <param name="safetyClass">The safety class for which to obtain the effect of weight reduction.</param>
        /// <param name="curbWeight">The curb weight for which to obtain the effect of weight reduction.</param>
        /// <returns>The the effect of weight reduction for the specified vehicle safety class and curb weight.</returns>
        public double GetEffect(SafetyClass safetyClass, double curbWeight)
        {
            return (curbWeight < this.Threshold[safetyClass]) ? this.SmallEffect[safetyClass] : this.LargeEffect[safetyClass];
        }
        /// <summary>
        /// Returns the base fatalities per billion miles for the specified vehicle safety class and curb weight.
        /// </summary>
        /// <param name="safetyClass">The safety class for which to obtain the base fatalities per billion miles.</param>
        /// <param name="curbWeight">The curb weight for which to obtain the base fatalities per billion miles.</param>
        /// <returns>The base fatalities per billion miles for the specified vehicle safety class and curb weight.</returns>
        public double GetBase(SafetyClass safetyClass, double curbWeight)
        {
            return (curbWeight < this.Threshold[safetyClass]) ? this.SmallBase[safetyClass] : this.LargeBase[safetyClass];
        }
        /// <summary>
        /// Returns the adjustment for new FMVSS for the specified vehicle safety class and curb weight.
        /// </summary>
        /// <param name="safetyClass">The safety class for which to obtain the adjustment for new FMVSS.</param>
        /// <param name="curbWeight">The curb weight for which to obtain the adjustment for new FMVSS.</param>
        /// <returns>The adjustment for new FMVSS for the specified vehicle safety class and curb weight.</returns>
        public double GetFMVSS(SafetyClass safetyClass, double curbWeight)
        {
            return (curbWeight < this.Threshold[safetyClass]) ? this.SmallFMVSS[safetyClass] : this.LargeFMVSS[safetyClass];
        }
        /// <summary>
        /// Returns the combined safety multiplier (effect * base * FMVSS) to use when obtaining vehicle fatalities.
        /// </summary>
        /// <param name="safetyClass">The safety class for which to obtain the safety multipler.</param>
        /// <param name="curbWeight">The curb weight for which to obtain the safety multipler.</param>
        /// <returns>The combined safety multiplier (effect * base * FMVSS) to use when obtaining vehicle fatalities.</returns>
        public double GetMultiplier(SafetyClass safetyClass, double curbWeight)
        {
            return (curbWeight < this.Threshold[safetyClass]) ? this.SmallMultiplier[safetyClass] : this.LargeMultiplier[safetyClass];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the boundary, in lbs., between small and large weigth effects.</summary>
        public SCValue<double> Threshold { get; private set; }

        /// <summary>Gets the effect of weight reduction (the change in fatalities per 100 lb reduction in curb weight) for
        ///   vehicles below the weight threshold.</summary>
        public SCValue<double> SmallEffect { get; private set; }
        /// <summary>Gets the base fatalities per billion miles for vehicles below the weight threshold.</summary>
        public SCValue<double> SmallBase { get; private set; }
        /// <summary>Gets the adjustment for new FMVSS for vehicles below the weight threshold.</summary>
        public SCValue<double> SmallFMVSS { get; private set; }
        /// <summary>Gets the combined safety multiplier (effect * base * FMVSS) to use when obtaining fatalities for vehicles
        ///   below the weight threshold.</summary>
        public SCValue<double> SmallMultiplier { get; private set; }

        /// <summary>Gets the effect of weight reduction (the change in fatalities per 100 lb reduction in curb weight) for
        ///   vehicles above the weight threshold.</summary>
        public SCValue<double> LargeEffect { get; private set; }
        /// <summary>Gets the base fatalities per billion miles for vehicles above the weight threshold.</summary>
        public SCValue<double> LargeBase { get; private set; }
        /// <summary>Gets the adjustment for new FMVSS for vehicles above the weight threshold.</summary>
        public SCValue<double> LargeFMVSS { get; private set; }
        /// <summary>Gets the combined safety multiplier (effect * base * FMVSS) to use when obtaining fatalities for vehicles
        ///   above the weight threshold.</summary>
        public SCValue<double> LargeMultiplier { get; private set; }

        /// <summary>Gets the social costs arising from additional vehicle fatalities.</summary>
        public double FatalityCosts { get; private set; }
        /// <summary>Gets the annual growth rate for fatality costs per vehicle.</summary>
        public double GrowthRate { get; private set; }
        /// <summary>Gets the base year for annual growth rate for fatality costs per vehicle.</summary>
        public int BaseYear { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides fine tuning parameters for using credit trading in the modeling system.
    /// </summary>
    [Serializable]
    public sealed class CreditTradingValues
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        CreditTradingValues() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="CreditTradingValues"/> class.
        /// </summary>
        internal CreditTradingValues(
            bool allowTrading, bool allowTransfers, bool allowCarryFwd, int carryFwdYears, bool allowCarryBwd, int carryBwdYears,
            int minMY, RCObject<double[]> transferCaps, RCObject<double[]> adjVMT, CreditAdjustmentMode adjMode,
            // additional runtime options
            int maxExpiringYears)
        {
            this.AllowMfrTrading     = allowTrading;
            this.AllowFleetTransfers = allowTransfers;
            this.AllowCarryForward   = allowCarryFwd;
            this.CarryForwardYears   = carryFwdYears;
            this.AllowCarryBackward  = allowCarryBwd;
            this.CarryBackwardYears  = carryBwdYears;
            //
            this.MinMY               = minMY;
            this.TransferCaps        = transferCaps;
            this.AdjustmentVMT       = adjVMT;
            this.AdjustmentMode      = adjMode;
            //
            this.MaxExpiringYears    = maxExpiringYears;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="CreditTradingValues"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="CreditTradingValues"/>.</returns>
        internal CreditTradingValues Clone()
        {
            CreditTradingValues value = new CreditTradingValues();
            //
            value.AllowMfrTrading     = this.AllowMfrTrading;
            value.AllowFleetTransfers = this.AllowFleetTransfers;
            value.AllowCarryForward   = this.AllowCarryForward;
            value.CarryForwardYears   = this.CarryForwardYears;
            value.AllowCarryBackward  = this.AllowCarryBackward;
            value.CarryBackwardYears  = this.CarryBackwardYears;
            //
            value.MinMY               = this.MinMY;
            value.TransferCaps        = this.TransferCaps .Clone();
            value.AdjustmentVMT       = this.AdjustmentVMT.Clone();
            value.AdjustmentMode      = this.AdjustmentMode;
            //
            value.MaxExpiringYears    = this.MaxExpiringYears;
            //
            return value;
        }

        /// <summary>
        /// Returns the transfer caps for the specified regulatory class, in the specified model year.
        /// </summary>
        /// <param name="regClass">The regulatory class for which to obtain the adjustment VMT.</param>
        /// <param name="year">The model year for which to obtain the transfer caps.</param>
        /// <returns>The transfer caps for the specified regulatory class, in the specified model year.</returns>
        public double GetTransferCaps(RegulatoryClass regClass, int year)
        {
            if (year < this.MinMY) { return 0; }
            int index = Math.Min(year - this.MinMY, this.TransferCaps[regClass].Length - 1);
            return this.TransferCaps[regClass][index];
        }
        /// <summary>
        /// Returns the adjustment VMT -- the assumed lifetime VMT to use when credits are transferred between compliance
        /// categories -- for the specified regulatory class, in the specified model year.
        /// </summary>
        /// <param name="regClass">The regulatory class for which to obtain the adjustment VMT.</param>
        /// <param name="year">The model year for which to obtain the adjustment VMT.</param>
        /// <returns>The adjustment VMT for the specified regulatory class, in the specified model year.</returns>
        public double GetAdjustmentVMT(RegulatoryClass regClass, int year)
        {
            if (year < this.MinMY) { return 0; }
            int index = Math.Min(year - this.MinMY, this.TransferCaps[regClass].Length - 1);
            return this.AdjustmentVMT[regClass][index];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets whether to consider credit trading between manufacturers within the same regulatory class and model year.</summary>
        public bool AllowMfrTrading { get; private set; }
        /// <summary>Gets whether to consider credit transfers between regulatory class within the same manufacturer and model year.</summary>
        public bool AllowFleetTransfers { get; private set; }
        /// <summary>Gets whether to carry credits forward into the analysis year from earlier model years within the same
        ///   manufacturer and regulatory class.</summary>
        public bool AllowCarryForward { get; private set; }
        /// <summary>Gets the maximum number of model years to look forward.</summary>
        public int CarryForwardYears { get; private set; }
        /// <summary>Gets whether to carry credits backward into the analysis year from future model years within the same
        ///   manufacturer and regulatory class.</summary>
        public bool AllowCarryBackward { get; private set; }
        /// <summary>Gets the maximum number of model years to look backward.</summary>
        public int CarryBackwardYears { get; private set; }

        /// <summary>Gets the first model year when credits may be traded, transferred, or carried forward/backward.</summary>
        public int MinMY { get; private set; }
        /// <summary>Gets the transfer caps, specified in mpg, corresponding to the maximum amount of credits that may be
        ///   transferred into a regulatory class for each model year. The cap from the latest model year is carried forward for
        ///   all subsequent years.</summary>
        public RCObject<double[]> TransferCaps { get; private set; }
        /// <summary>Gets the assumed lifetime VMT to use when credits are transferred between compliance categories. The assumed
        ///   lifetime VMT from the latest year is carried forward for all subsequent years. This parameter is ignored if the
        ///   <see cref="AdjustmentMode"/> is <see cref="CreditAdjustmentMode.Fixed"/>.</summary>
        public RCObject<double[]> AdjustmentVMT { get; private set; }
        /// <summary>Gets the adjustment mode to use when credits are transferred between compliance categories.</summary>
        public CreditAdjustmentMode AdjustmentMode { get; private set; }

        // --- additional runtime settings ---
        /// <summary>Gets the maximum number of model years to consider when using expiring credits.</summary>
        public int MaxExpiringYears { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides fine tuning parameters for using ZEV credits in the modeling system.
    /// </summary>
    [Serializable]
    public sealed class ZEVCreditValues
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        ZEVCreditValues() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="ZEVCreditValues"/> class.
        /// </summary>
        internal ZEVCreditValues(int minMY, int maxMY, double[] zevRequirement, double[] phevCap)
        {
            this.MinMY          = minMY;
            this.MaxMY          = maxMY;
            this.ZEVRequirement = zevRequirement;
            this.PHEVCap        = phevCap;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="ZEVCreditValues"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="ZEVCreditValues"/>.</returns>
        internal ZEVCreditValues Clone()
        {
            ZEVCreditValues value = new ZEVCreditValues();
            //
            value.MinMY          = this.MinMY;
            value.MaxMY          = this.MaxMY;
            value.ZEVRequirement = this.ZEVRequirement.CloneArray();
            value.PHEVCap        = this.PHEVCap       .CloneArray();
            //
            return value;
        }

        /// <summary>
        /// Returns the minimum percentage of ZEV credits that a manufacturer must generate in order to meet the ZEV requirement
        /// in the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to obtain the ZEV requirement.</param>
        /// <returns>The ZEV requirement for the specified model year.</returns>
        public double GetZEVRequirement(int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxMY) - this.MinMY, 0);
            return this.ZEVRequirement[index];
        }
        /// <summary>
        /// Returns the maximum percentage of ZEV credits that a manufacturer may generate from PHEVs in order to meet the ZEV
        /// requirement in the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to obtain the PHEV allowance.</param>
        /// <returns>The PHEV allowance for the specified model year.</returns>
        public double GetPHEVCap(int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxMY) - this.MinMY, 0);
            return this.PHEVCap[index];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest model year for which ZEV credit values are available.</summary>
        public int MinMY { get; private set; }
        /// <summary>Gets the highest model year for which ZEV credit values are available.</summary>
        public int MaxMY { get; private set; }

        /// <summary>Gets the minimum percentage of zero emission vehicle (ZEV) credits that a manufacturer must generate in order
        ///   to meet the ZEV requirement in each specified model year.</summary>
        public double[] ZEVRequirement { get; private set; }
        /// <summary>Gets the maximum percentage of ZEV credits that a manufacturer may generate from PHEVs in order to meet the
        ///   ZEV requirement in each specified model year.</summary>
        public double[] PHEVCap { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides fine tuning parameters for the Dynamic Fleet Share model.
    /// </summary>
    [Serializable]
    public sealed class DFSModelValues
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        DFSModelValues() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="DFSModelValues"/> class.
        /// </summary>
        internal DFSModelValues(double initialLDVShare, VCValue<double> initialFE, double epsilon)
        {
            this.InitialLDVShare = initialLDVShare;
            this.InitialFE       = initialFE;
            this.Epsilon         = epsilon;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="DFSModelValues"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="DFSModelValues"/>.</returns>
        internal DFSModelValues Clone()
        {
            DFSModelValues value = new DFSModelValues();
            //
            value.InitialLDVShare = this.InitialLDVShare;
            value.InitialFE       = this.InitialFE.Clone();
            value.Epsilon         = this.Epsilon;
            //
            return value;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the fleet share of the light duty passenger vehicles for the model year immediately preceding the first
        ///   analysis year.</summary>
        public double InitialLDVShare { get; private set; }
        /// <summary>Gets the average fuel economy for each fleet for the model year immediately preceding the first analysis year.</summary>
        public VCValue<double> InitialFE { get; private set; }
        /// <summary>Gets the additional constant that serves as a "randomizer" when dynamically adjusting the PC/LT fleet share.
        ///   This value should typically be zero.</summary>
        public double Epsilon { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides physical and chemical properties of various fuel types as well as fuel import assumptions.
    /// </summary>
    [Serializable]
    public sealed class FuelProperties
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        FuelProperties() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="FuelProperties"/> class.
        /// </summary>
        internal FuelProperties(
            FuelValue energyDensity, FuelValue massDensity, FuelValue carbonContent, FuelValue co2Emissions, FuelValue so2Emissions,
            FuelValue fsLowImports, FuelValue fsReducedRef, FuelValue reducedDomCrude, FuelValue reducedImpCrude)
        {
            this.EnergyDensity                       = energyDensity;
            this.MassDensity                         = massDensity;
            this.CarbonContent                       = carbonContent;
            this.CO2Emissions                        = co2Emissions;
            this.SO2Emissions                        = so2Emissions;
            this.FuelSavingsLeadingToLowerImports    = fsLowImports;
            this.FuelSavingsLeadingToReducedRefining = fsReducedRef;
            this.ReducedRefiningFromDomesticCrude    = reducedDomCrude;
            this.ReducedRefiningFromImportedCrude    = reducedImpCrude;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="FuelProperties"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="FuelProperties"/>.</returns>
        internal FuelProperties Clone()
        {
            FuelProperties value = new FuelProperties();
            //
            value.EnergyDensity                       = this.EnergyDensity;
            value.MassDensity                         = this.MassDensity;
            value.CarbonContent                       = this.CarbonContent;
            value.CO2Emissions                        = this.CO2Emissions;
            value.SO2Emissions                        = this.SO2Emissions;
            value.FuelSavingsLeadingToLowerImports    = this.FuelSavingsLeadingToLowerImports;
            value.FuelSavingsLeadingToReducedRefining = this.FuelSavingsLeadingToReducedRefining;
            value.ReducedRefiningFromDomesticCrude    = this.ReducedRefiningFromDomesticCrude;
            value.ReducedRefiningFromImportedCrude    = this.ReducedRefiningFromImportedCrude;
            //
            return value;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the energy density of various fuel types (specified in BTU/gallon, BTU/kW-h, or BTU/scf).</summary>
        public FuelValue EnergyDensity { get; private set; }
        /// <summary>Gets the mass density of various fuel types (specified in grams/gallon, grams/kW-h, or grams/scf).</summary>
        public FuelValue MassDensity { get; private set; }
        /// <summary>Gets the carbon content of various fuel types (specified in % by weight).</summary>
        public FuelValue CarbonContent { get; private set; }
        /// <summary>Gets the Carbon Dioxide upstream emissions for various fuel types (specified in grams/gallon, grams/kW-h, or
        ///   grams/scf).</summary>
        public FuelValue CO2Emissions { get; private set; }
        /// <summary>Gets the Sulfur Dioxide upstream emissions for various fuel types (specified in grams/gallon, grams/kW-h, or
        ///   grams/scf).</summary>
        public FuelValue SO2Emissions { get; private set; }

        /// <summary>Gets an assumed value for share of fuel savings leading to lower fuel imports.</summary>
        public FuelValue FuelSavingsLeadingToLowerImports { get; private set; }
        /// <summary>Gets an assumed value for share of fuel savings leading to reduced domestic fuel refining.</summary>
        public FuelValue FuelSavingsLeadingToReducedRefining { get; private set; }
        /// <summary>Gets an assumed value for share of reduced domestic refining from domestic crude.</summary>
        public FuelValue ReducedRefiningFromDomesticCrude { get; private set; }
        /// <summary>Gets an assumed value for share of reduced domestic refining from imported crude.</summary>
        public FuelValue ReducedRefiningFromImportedCrude { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides costs arising from emission damage.
    /// </summary>
    [Serializable]
    public sealed class EmissionCosts
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        EmissionCosts() { }
        internal EmissionCosts(double co, double voc, double nox, double pm, double so2, double ch4Scalar, double n2oScalar,
            CO2EmissionCosts co2)
        {
            this.CO        = co;
            this.VOC       = voc;
            this.NOX       = nox;
            this.PM        = pm;
            this.SO2       = so2;
            this.CH4Scalar = ch4Scalar;
            this.N2OScalar = n2oScalar;
            this.CO2       = co2;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="EmissionCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="EmissionCosts"/>.</returns>
        internal EmissionCosts Clone()
        {
            EmissionCosts value = new EmissionCosts();
            //
            value.CO        = this.CO;
            value.VOC       = this.VOC;
            value.NOX       = this.NOX;
            value.PM        = this.PM;
            value.SO2       = this.SO2;
            value.CH4Scalar = this.CH4Scalar;
            value.N2OScalar = this.N2OScalar;
            value.CO2       = this.CO2.Clone();
            //
            return value;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the economic costs arising from Carbon Monoxide damage, specified in $/metric ton.</summary>
        public double CO { get; private set; }
        /// <summary>Gets the economic costs arising from Volatile Organic Compounds damage, specified in $/metric ton.</summary>
        public double VOC { get; private set; }
        /// <summary>Gets the economic costs arising from Nitrous Oxides damage, specified in $/metric ton.</summary>
        public double NOX { get; private set; }
        /// <summary>Gets the economic costs arising from Particulate Matter damage, specified in $/metric ton.</summary>
        public double PM { get; private set; }
        /// <summary>Gets the economic costs arising from Sulfur Dioxide damage, specified in $/metric ton.</summary>
        public double SO2 { get; private set; }
        /// <summary>Gets the GWP scalar to apply to Methane emissions before computing the economic costs arising from Methane
        ///   damage.</summary>
        public double CH4Scalar { get; private set; }
        /// <summary>Gets the GWP scalar to apply to Nitrous Oxide emissions before computing the economic costs arising from
        ///   Nitrous Oxide damage.</summary>
        public double N2OScalar { get; private set; }
        /// <summary>Gets the economic costs arising from Carbon Dioxide damage, specified in $/metric ton.</summary>
        public CO2EmissionCosts CO2 { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides costs arising from Carbon Dioxide emission damage.
    /// </summary>
    [Serializable]
    public sealed class CO2EmissionCosts
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        CO2EmissionCosts() { }
        internal CO2EmissionCosts(int minCY, int maxCY,
            double discRateLow, double[] costsLow, double discRateAvg, double[] costsAvg,
            double discRateHigh, double[] costsHigh, double discRateVeryHigh, double[] costsVeryHigh)
        {
            this.MinCY = minCY;
            this.MaxCY = maxCY;
            //
            this._discRate = new double[]   { discRateLow, discRateAvg, discRateHigh, discRateVeryHigh };
            this._costs    = new double[][] { costsLow   , costsAvg   , costsHigh   , costsVeryHigh    };
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="CO2EmissionCosts"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="CO2EmissionCosts"/>.</returns>
        internal CO2EmissionCosts Clone()
        {
            CO2EmissionCosts value = new CO2EmissionCosts();
            //
            value.MinCY     = this.MinCY;
            value.MaxCY     = this.MaxCY;
            //
            value._discRate = this._discRate.CloneArray();
            value._costs    = this._costs   .CloneArray();
            //
            return value;
        }

        /// <summary>
        /// Returns low, average, high, or very high CO2 discount rate for the specified CO2 estimates.
        /// </summary>
        /// <param name="co2Estimates">The low, average, high, or very high CO2 estimates to use.</param>
        /// <returns>Low, average, high, or very high CO2 discount rate for the specified CO2 estimates.</returns>
        public double GetCO2DiscountRate(Estimates co2Estimates)
        {
            return this._discRate[(int)co2Estimates];
        }
        /// <summary>
        /// Returns low, average, high, or very high CO2 cost for the specified CO2 estimates, in the specified calendar year.
        /// </summary>
        /// <param name="co2Estimates">The low, average, high, or very high CO2 estimates to use.</param>
        /// <param name="year">The calendar year for which to obtain the CO2 costs.</param>
        /// <returns>Low, average, high, or very high CO2 costs for the specified CO2 estimates, in the specified calendar year.</returns>
        public double GetCO2Costs(Estimates co2Estimates, int year)
        {
            int index = Math.Max(Math.Min(year, this.MaxCY) - this.MinCY, 0);
            return this._costs[(int)co2Estimates][index];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest calendar year for which CO-2 price data is available.</summary>
        public int MinCY { get; private set; }
        /// <summary>Gets the highest calendar year for which CO-2 price data is available.</summary>
        public int MaxCY { get; private set; }

        #endregion

        #region /*** Variables ***/

        /// <summary>Specifies the discount rate to apply to costs arising from Carbon Dioxide damage.</summary>
        double[] _discRate;
        /// <summary>Specifies the economic costs arising from Carbon Dioxide damage, specified for each calendar year in $/metric ton.</summary>
        double[][] _costs;

        #endregion

    }
    /// <summary>
    /// Provides upstream emissions, in grams/million BTU, for various fuel types (from all stages of fuel production and distribution).
    /// </summary>
    [Serializable]
    public sealed class UpstreamEmissions
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        UpstreamEmissions() { }
        internal UpstreamEmissions(
            FuelValue co, FuelValue voc, FuelValue nox, FuelValue so2, FuelValue pm, FuelValue co2, FuelValue ch4, FuelValue n2o,
            FuelValue acetaldehyde, FuelValue acrolein, FuelValue benzene, FuelValue butadiene, FuelValue formaldehyde,
            FuelValue dpm10, FuelValue mtbe)
        {
            this.CO           = co;                 // emissions for effects modeling
            this.VOC          = voc;
            this.NOx          = nox;
            this.SO2          = so2;
            this.PM           = pm;
            this.CO2          = co2;
            this.CH4          = ch4;
            this.N2O          = n2o;
            this.Acetaldehyde = acetaldehyde;       // additional emissions for EIS modeling
            this.Acrolein     = acrolein;
            this.Benzene      = benzene;
            this.Butadiene    = butadiene;
            this.Formaldehyde = formaldehyde;
            this.DPM10        = dpm10;
            this.MTBE         = mtbe;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="UpstreamEmissions"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="UpstreamEmissions"/>.</returns>
        internal UpstreamEmissions Clone()
        {
            UpstreamEmissions value = new UpstreamEmissions();
            //
            value.CO           = this.CO;           // emissions for effects modeling
            value.VOC          = this.VOC;
            value.NOx          = this.NOx;
            value.SO2          = this.SO2;
            value.PM           = this.PM;
            value.CO2          = this.CO2;
            value.CH4          = this.CH4;
            value.N2O          = this.N2O;
            value.Acetaldehyde = this.Acetaldehyde; // additional emissions for EIS modeling
            value.Acrolein     = this.Acrolein;
            value.Benzene      = this.Benzene;
            value.Butadiene    = this.Butadiene;
            value.Formaldehyde = this.Formaldehyde;
            value.DPM10        = this.DPM10;
            value.MTBE         = this.MTBE;
            //
            return value;
        }

        /// <summary>
        /// Returns the upstream emissions for the specified pollutant and fuel type.
        /// </summary>
        /// <param name="pollutant">The pollutant for which to obtain the upstream emissions.</param>
        /// <param name="fuelType">The fuel type for which to obtain the upstream emissions.</param>
        /// <returns>The upstream emissions for the specified pollutant and fuel type.</returns>
        public double GetUpstreamEmissions(Pollutant pollutant, FuelType fuelType)
        {
            return this[pollutant][fuelType];
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the tailpipe emissions for a specific <see cref="Pollutant"/>.</summary>
        /// <param name="pollutant">The pollutant for which to obtain the tailpipe emissions.</param>
        /// <returns>The tailpipe emissions for a specific <see cref="Pollutant"/>.</returns>
        public FuelValue this[Pollutant pollutant]
        {
            get
            {
                switch (pollutant)
                {
                    // emissions for effects modeling
                    case Pollutant.CO          : return this.CO;
                    case Pollutant.VOC         : return this.VOC;
                    case Pollutant.NOx         : return this.NOx;
                    case Pollutant.SO2         : return this.SO2;
                    case Pollutant.PM          : return this.PM;
                    case Pollutant.CO2         : return this.CO2;
                    case Pollutant.CH4         : return this.CH4;
                    case Pollutant.N2O         : return this.N2O;

                    // additional emissions for EIS modeling
                    case Pollutant.Acetaldehyde: return this.Acetaldehyde;
                    case Pollutant.Acrolein    : return this.Acrolein;
                    case Pollutant.Benzene     : return this.Benzene;
                    case Pollutant.Butadiene   : return this.Butadiene;
                    case Pollutant.Formaldehyde: return this.Formaldehyde;
                    case Pollutant.DPM10       : return this.DPM10;
                    case Pollutant.MTBE        : return this.MTBE;

                    // invalid pollutant specified
                    default:
                        throw new ArgumentOutOfRangeException("pollutant", "The specified pollutant value is not supported.");
                }
            }
        }

        //----- emissions for effects modeling -----
        /// <summary>Gets the Carbon Monoxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue CO { get; private set; }
        /// <summary>Gets the Volatile Organic Compound upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue VOC { get; private set; }
        /// <summary>Gets the Nitrogen Oxides upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue NOx { get; private set; }
        /// <summary>Gets the Sulfur Dioxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue SO2 { get; private set; }
        /// <summary>Gets the Particulate Matter (diameter of ~2.5 micrometers) upstream emissions (from all stages of fuel
        ///   production and distribution).</summary>
        public FuelValue PM { get; private set; }
        /// <summary>Gets the Carbon Dioxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue CO2 { get; private set; }
        /// <summary>Gets the Methane upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue CH4 { get; private set; }
        /// <summary>Gets the Nitrous Oxide upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue N2O { get; private set; }

        //----- additional emissions for EIS modeling -----
        /// <summary>Gets the Acetaldehyde upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue Acetaldehyde { get; private set; }
        /// <summary>Gets the Acrolein upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue Acrolein { get; private set; }
        /// <summary>Gets the Benzene upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue Benzene { get; private set; }
        /// <summary>Gets the 1,3-Butadiene upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue Butadiene { get; private set; }
        /// <summary>Gets the Formaldehyde upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue Formaldehyde { get; private set; }
        /// <summary>Gets the Diesel Particulate Matter (diameter of ~10 micrometers) upstream emissions (from all stages of fuel
        ///   production and distribution).</summary>
        public FuelValue DPM10 { get; private set; }
        /// <summary>Gets the Methyl Tertiary Butyl Ether upstream emissions (from all stages of fuel production and distribution).</summary>
        public FuelValue MTBE { get; private set; }

        #endregion

    }
    /// <summary>
    /// Provides tailpipe emissions, in grams/mile, for various vehicle classes.
    /// </summary>
    [Serializable]
    public sealed class TailpipeEmissions
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        TailpipeEmissions() { }
        internal TailpipeEmissions(
            int minMY, int maxMY,
            // emissions for effects modeling
            VCObject<double[][]> co , VCObject<double[][]> voc, VCObject<double[][]> nox,
            VCObject<double[][]> so2, VCObject<double[][]> pm , VCObject<double[][]> co2,
            VCObject<double[][]> ch4, VCObject<double[][]> n2o,
            // additional emissions for EIS modeling
            VCObject<double[][]> acetaldehyde, VCObject<double[][]> acrolein,
            VCObject<double[][]> benzene     , VCObject<double[][]> butadiene,
            VCObject<double[][]> formaldehyde, VCObject<double[][]> dpm10, VCObject<double[][]> mtbe)
        {
            this.MinMY        = minMY;
            this.MaxMY        = maxMY;
            // emissions for effects modeling
            this.CO           = co;
            this.VOC          = voc;
            this.NOx          = nox;
            this.SO2          = so2;
            this.PM           = pm;
            this.CO2          = co2;
            this.CH4          = ch4;
            this.N2O          = n2o;
            // additional emissions for EIS modeling
            this.Acetaldehyde = acetaldehyde;
            this.Acrolein     = acrolein;
            this.Benzene      = benzene;
            this.Butadiene    = butadiene;
            this.Formaldehyde = formaldehyde;
            this.DPM10        = dpm10;
            this.MTBE         = mtbe;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="TailpipeEmissions"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="TailpipeEmissions"/>.</returns>
        internal TailpipeEmissions Clone()
        {
            TailpipeEmissions value = new TailpipeEmissions();
            //
            value.MinMY        = this.MinMY;
            value.MaxMY        = this.MaxMY;
            //
            value.CO           = this.Clone(this.CO);           // emissions for effects modeling
            value.VOC          = this.Clone(this.VOC);
            value.NOx          = this.Clone(this.NOx);
            value.SO2          = this.Clone(this.SO2);
            value.PM           = this.Clone(this.PM);
            value.CO2          = this.Clone(this.CO2);
            value.CH4          = this.Clone(this.CH4);
            value.N2O          = this.Clone(this.N2O);
            value.Acetaldehyde = this.Clone(this.Acetaldehyde); // additional emissions for EIS modeling
            value.Acrolein     = this.Clone(this.Acrolein);
            value.Benzene      = this.Clone(this.Benzene);
            value.Butadiene    = this.Clone(this.Butadiene);
            value.Formaldehyde = this.Clone(this.Formaldehyde);
            value.DPM10        = this.Clone(this.DPM10);
            value.MTBE         = this.Clone(this.MTBE);
            //
            return value;
        }
        VCObject<double[][]> Clone(VCObject<double[][]> value)
        {
            if (value == null) { return null; }
            //
            VCObject<double[][]> newValue = new VCObject<double[][]>();
            foreach (VehicleClass vehClass in VCValue<object>.Classes)
            {
                newValue[vehClass] = value[vehClass].CloneArray();
            }
            return newValue;
        }

        /// <summary>
        /// Returns the tailpipe emissions for the specified pollutant and vehicle class, in the specified model year,
        /// at the given vehicle age.
        /// </summary>
        /// <param name="pollutant">The pollutant for which to obtain the tailpipe emissions.</param>
        /// <param name="vehClass">The vehicle class for which to obtain the tailpipe emissions.</param>
        /// <param name="year">The model year for which to obtain the tailpipe emissions.</param>
        /// <param name="age">The vehicle age for which to obtain the tailpipe emissions.</param>
        /// <returns>The tailpipe emissions for the specified pollutant, fuel type, and vehicle class, in the specified model
        ///   year, at the given vehicle age.</returns>
        public double GetTailpipeEmissions(Pollutant pollutant, VehicleClass vehClass, int year, int age)
        {
            var te = this[pollutant];
            if (te == null) { return 0; }
            int index = Math.Max(Math.Min(year, this.MaxMY) - this.MinMY, 0);
            return te[vehClass][index][age];
        }

        /// <summary>
        /// Determines whether the rate tables for tailpipe emissions for the specified pollutant have been loaded.
        /// </summary>
        /// <param name="pollutant">The pollutant to check.</param>
        /// <returns>true, if the rate tables for tailpipe emissions for the specified pollutant exist; false, otherwise.</returns>
        public bool HasEmissions(Pollutant pollutant)
        {
            return (this[pollutant] != null);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the lowest model year for which tailpipe emissions data is available.</summary>
        public int MinMY { get; private set; }
        /// <summary>Gets the highest model year for which tailpipe emissions data is available.</summary>
        public int MaxMY { get; private set; }

        /// <summary>Gets the tailpipe emissions for a specific <see cref="Pollutant"/>.</summary>
        /// <param name="pollutant">The pollutant for which to obtain the tailpipe emissions.</param>
        /// <returns>The tailpipe emissions for a specific <see cref="Pollutant"/>.</returns>
        public VCObject<double[][]> this[Pollutant pollutant]
        {
            get
            {
                switch (pollutant)
                {
                    // emissions for effects modeling
                    case Pollutant.CO          : return this.CO;
                    case Pollutant.VOC         : return this.VOC;
                    case Pollutant.NOx         : return this.NOx;
                    case Pollutant.SO2         : return this.SO2;
                    case Pollutant.PM          : return this.PM;
                    case Pollutant.CO2         : return this.CO2;
                    case Pollutant.CH4         : return this.CH4;
                    case Pollutant.N2O         : return this.N2O;

                    // additional emissions for EIS modeling
                    case Pollutant.Acetaldehyde: return this.Acetaldehyde;
                    case Pollutant.Acrolein    : return this.Acrolein;
                    case Pollutant.Benzene     : return this.Benzene;
                    case Pollutant.Butadiene   : return this.Butadiene;
                    case Pollutant.Formaldehyde: return this.Formaldehyde;
                    case Pollutant.DPM10       : return this.DPM10;
                    case Pollutant.MTBE        : return this.MTBE;

                    // invalid pollutant specified
                    default:
                        throw new ArgumentOutOfRangeException("pollutant", "The specified pollutant value is not supported.");
                }
            }
        }

        //----- emissions for effects modeling -----
        /// <summary>Gets the Carbon Monoxide tailpipe emissions.</summary>
        public VCObject<double[][]> CO { get; private set; }
        /// <summary>Gets the Volatile Organic Compound tailpipe emissions.</summary>
        public VCObject<double[][]> VOC { get; private set; }
        /// <summary>Gets the Nitrogen Oxides tailpipe emissions.</summary>
        public VCObject<double[][]> NOx { get; private set; }
        /// <summary>Gets the Sulfur Dioxide tailpipe emissions.</summary>
        public VCObject<double[][]> SO2 { get; private set; }
        /// <summary>Gets the Particulate Matter (diameter of ~2.5 micrometers) tailpipe emissions.</summary>
        public VCObject<double[][]> PM { get; private set; }
        /// <summary>Gets the Carbon Dioxide tailpipe emissions.</summary>
        public VCObject<double[][]> CO2 { get; private set; }
        /// <summary>Gets the Methane tailpipe emissions.</summary>
        public VCObject<double[][]> CH4 { get; private set; }
        /// <summary>Gets the Nitrous Oxide tailpipe emissions.</summary>
        public VCObject<double[][]> N2O { get; private set; }

        //----- additional emissions for EIS modeling -----
        /// <summary>Gets the Acetaldehyde tailpipe emissions.</summary>
        public VCObject<double[][]> Acetaldehyde { get; private set; }
        /// <summary>Gets the Acrolein tailpipe emissions.</summary>
        public VCObject<double[][]> Acrolein { get; private set; }
        /// <summary>Gets the Benzene tailpipe emissions.</summary>
        public VCObject<double[][]> Benzene { get; private set; }
        /// <summary>Gets the 1,3-Butadiene tailpipe emissions.</summary>
        public VCObject<double[][]> Butadiene { get; private set; }
        /// <summary>Gets the Formaldehyde tailpipe emissions.</summary>
        public VCObject<double[][]> Formaldehyde { get; private set; }
        /// <summary>Gets the Diesel Particulate Matter (diameter of ~10 micrometers) tailpipe emissions.</summary>
        public VCObject<double[][]> DPM10 { get; private set; }
        /// <summary>Gets the Methyl Tertiary Butyl Ether tailpipe emissions.</summary>
        public VCObject<double[][]> MTBE { get; private set; }

        #endregion

    }

    #endregion

    /// <summary>
    /// Provides compliance and effects parameters for the modeling system.
    /// </summary>
    [Serializable]
    public class Parameters : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        Parameters() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="Parameters"/> class.
        /// </summary>
        public Parameters(EconomicValues economicValues, VehicleAgeData survivalRates, VehicleAgeData milesDriven,
            FuelPrices fuelPrices, FuelEconomyData fuelEconomyData, FleetAnalysisValues fleetAnalysisValues,
            HistoricFleetData historicFleetData, SafetyValues safetyValues, CreditTradingValues creditTradingValues,
            ZEVCreditValues zevCreditValues, DFSModelValues dfsModelValues, FuelProperties fuelProperties,
            EmissionCosts emissionCosts, UpstreamEmissions upstreamEmissions, TailpipeEmissions tailpipeGasoline,
            TailpipeEmissions tailpipeDiesel)
        {
            this.EconomicValues      = economicValues;
            this.SurvivalRates       = survivalRates;
            this.MilesDriven         = milesDriven;
            this.FuelPrices          = fuelPrices;
            this.FuelEconomyData     = fuelEconomyData;
            this.FleetAnalysisValues = fleetAnalysisValues;
            this.HistoricFleetData   = historicFleetData;
            this.SafetyValues        = safetyValues;
            this.CreditTradingValues = creditTradingValues;
            this.ZEVCreditValues     = zevCreditValues;
            this.DFSModelValues      = dfsModelValues;
            this.FuelProperties      = fuelProperties;
            this.EmissionCosts       = emissionCosts;
            this.UpstreamEmissions   = upstreamEmissions;
            this.TailpipeGasoline    = tailpipeGasoline;
            this.TailpipeDiesel      = tailpipeDiesel;
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Parameters"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Parameters"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Parameters"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Parameters"/>.</returns>
        public Parameters Clone()
        {
            Parameters value = new Parameters();
            //
            value.EconomicValues      = this.EconomicValues     .Clone();
            value.SurvivalRates       = this.SurvivalRates      .Clone();
            value.MilesDriven         = this.MilesDriven        .Clone();
            value.FuelPrices          = this.FuelPrices         .Clone();
            value.FuelEconomyData     = this.FuelEconomyData    .Clone();
            value.FleetAnalysisValues = this.FleetAnalysisValues.Clone();
            value.HistoricFleetData   = this.HistoricFleetData  .Clone();
            value.SafetyValues        = this.SafetyValues       .Clone();
            value.CreditTradingValues = this.CreditTradingValues.Clone();
            value.ZEVCreditValues     = this.ZEVCreditValues    .Clone();
            value.DFSModelValues      = this.DFSModelValues     .Clone();
            value.FuelProperties      = this.FuelProperties     .Clone();
            value.EmissionCosts       = this.EmissionCosts      .Clone();
            value.UpstreamEmissions   = this.UpstreamEmissions  .Clone();
            value.TailpipeGasoline    = this.TailpipeGasoline   .Clone();
            value.TailpipeDiesel      = this.TailpipeDiesel     .Clone();
            //
            return value;
        }

        #endregion

        /// <summary>
        /// Updates the parameters with the specified values for a new Monte-Carlo trial.
        /// </summary>
        public void UpdateForMonteCarlo(double fuelPriceScaleFactor, int fuelPriceStartYear, double priceShockAlpha,
            double reboundEffect, double onRoadGap_Gasoline, double onRoadGap_Diesel)
        {
            // save old fuel prices
            FuelPrices oldFP = this.FuelPrices.Clone();
            // update new fuel price forecast
            this.FuelPrices                  .UpdateForMonteCarlo(fuelPriceScaleFactor, fuelPriceStartYear);
            this.EconomicValues              .UpdateForMonteCarlo(reboundEffect, onRoadGap_Gasoline, onRoadGap_Diesel);
            this.EconomicValues.EconomicCosts.UpdateForMonteCarlo(oldFP, this.FuelPrices, priceShockAlpha);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the economic values for benefits computations.</summary>
        public EconomicValues EconomicValues { get; private set; }
        /// <summary>Gets the proportion of original vehicles sales surviving at each vehicle age.</summary>
        public VehicleAgeData SurvivalRates { get; private set; }
        /// <summary>Gets the annual miles driven by vehicles at each vehicle age.</summary>
        public VehicleAgeData MilesDriven { get; private set; }
        /// <summary>Gets a forecast of fuel prices per calendar year for all supported fuel types.</summary>
        public FuelPrices FuelPrices { get; private set; }
        /// <summary>Gets the historic and projected fuel economy values and their associated fuel shares by calendar year.</summary>
        public FuelEconomyData FuelEconomyData { get; private set; }
        /// <summary>Gets the fine tuning parameters for using fleet analysis in the modeling system.</summary>
        public FleetAnalysisValues FleetAnalysisValues { get; private set; }
        /// <summary>Gets the historic fleet data by vehicle class.</summary>
        public HistoricFleetData HistoricFleetData { get; private set; }
        /// <summary>Gets the values neccessary for safety calculations.</summary>
        public SafetyValues SafetyValues { get; private set; }
        /// <summary>Gets the fine tuning parameters for using credit trading in the modeling system.</summary>
        public CreditTradingValues CreditTradingValues { get; private set; }
        /// <summary>Gets the fine tuning parameters for using ZEV credits in the modeling system.</summary>
        public ZEVCreditValues ZEVCreditValues { get; private set; }
        /// <summary>Gets the fine tuning parameters for the Dynamic Fleet Share model.</summary>
        public DFSModelValues DFSModelValues { get; private set; }
        /// <summary>Gets the physical and chemical properties of various fuel types as well as fuel import assumptions.</summary>
        public FuelProperties FuelProperties { get; private set; }
        /// <summary>Gets the costs arising from emission damage.</summary>
        public EmissionCosts EmissionCosts { get; private set; }
        /// <summary>Gets the upstream emissions, in grams/million BTU, for various fuel types (from all stages of fuel production
        ///   and distribution).</summary>
        public UpstreamEmissions UpstreamEmissions { get; private set; }
        /// <summary>Gets the tailpipe emissions, in grams/mile, arising from gasoline operation.</summary>
        public TailpipeEmissions TailpipeGasoline { get; private set; }
        /// <summary>Gets the tailpipe emissions, in grams/mile, arising from diesel operation.</summary>
        public TailpipeEmissions TailpipeDiesel { get; private set; }

        #endregion

    }
}
