using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Windows.Forms;
using Volpe.Cafe.Utils;
using Volpe.XlLib;
using Volpe.Cafe.UI.Properties;
using Volpe.Cafe.HelperClasses;

/*
 * The main namespace within cafegui.exe assembly, which serves as the main entry point for the application and
 * contains the static void Main() method.
 */
namespace Volpe.Cafe.UI
{
    /// <summary>
    /// Provides the entry point for the application.  The Program class contains the static void Main() method and
    /// serves as the start up object for the CAFE Model application.
    /// </summary>
    public sealed class Program
    {

        #region /*** Constructors ***/

        // Hide the constructor, since we want to prevent users from instantiating this class.
        /**
         * <summary>
         * Initializes a new instance of the Program class for executing the CAFE Model application.
         * </summary>
         */
        private Program()
        {
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// The Main() method serves as the main entry point for the application.
        /// </summary>
        /// <param name="args">The runtime arguments passed in to the model.</param>
        [STAThread]
        static void Main(string[] args)
        {
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);

            // add support for embeded assemblies
            AppDomain.CurrentDomain.AssemblyResolve += new ResolveEventHandler(CurrentDomain_AssemblyResolve);

            // start the model
            Console.WriteLine("Starting run at:  {0}", DateTime.Now.ToString());
            Program.Run(args);
            Console.WriteLine("Exiting model at: {0}", DateTime.Now.ToString());
        }
        static void Run(string[] args)
        {
            try
            {
                // display warnings & notice (as required)
                if (Program.ShowWarningsScreen(true) == DialogResult.OK)
                {
                    // check excel version (skip checking if NoWarn option is selected)
                    int xlCheck = XlUtilities.CompatibilityCheck();
                    string message = null;
                    if (xlCheck == -1)
                    {   // excel could not be loaded or not installed
                        message = "A version of Excel could not be found on the current system.  The model will not be " +
                            "able to load input files or generate output reports.\n\nPress OK to continue running the model.";
                    }
                    else if (xlCheck == 0)
                    {   // excel version is not supported
                        message = "The version of Excel installed on the current system is not supported by the model.  " +
                            "Errors may occur when loading input files or generating output reports.\n\n" +
                            "Press OK to continue running the model.";
                    }
                    //
                    if (message != null)
                    {
                        RtfDialog.ShowDialog("Compatibility Warning", message,
                            new Font(Cafe2d.DefaultFont.FontFamily, 10, FontStyle.Regular), Color.DarkRed, PromptOption.Ok, true, false);
                    }

                    // launch the user interface
                    Application.Run(new Cafe2d());
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "CAFE Model Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                Console.WriteLine(ex.ToString());
            }
        }

        static Assembly CurrentDomain_AssemblyResolve(object sender, ResolveEventArgs args)
        {
            string resource = "Volpe.Cafe.UI.Resources.DLL." + (new AssemblyName(args.Name)).Name + ".dll";
            Stream stream = Assembly.GetExecutingAssembly().GetManifestResourceStream(resource);
            byte[] data = new byte[stream.Length];
            //
            stream.Read(data, 0, data.Length);
            return Assembly.Load(data);
        }

        internal static DialogResult ShowWarningsScreen(bool showCancelButton)
        {
            PromptOption promptOption = (showCancelButton) ? PromptOption.OkCancel : PromptOption.Ok;
            return ShowScreenHelper("Volpe.Cafe.UI.Resources.warnings.rtf", "CAFE Model Warnings", promptOption, false, false);
        }
        internal static DialogResult ShowNoticeScreen()
        {
            return ShowScreenHelper("Volpe.Cafe.UI.Resources.notice.rtf", "CAFE Model Notice", PromptOption.Ok, false, true);
        }
        internal static DialogResult ShowAboutScreen()
        {
            return ShowScreenHelper("Volpe.Cafe.UI.Resources.about.rtf", "About CAFE Model", PromptOption.Ok, true, false);
        }
        static DialogResult ShowScreenHelper(string resourceName,
            string caption, PromptOption promptOption, bool showBuildInfo, bool showLicenseButton)
        {
            DialogResult dr = DialogResult.OK;
            using (Stream stream = GetAssemblyResource(resourceName))
            {
                dr = RtfDialog.ShowDialogFromStream(caption, stream, promptOption, showBuildInfo, showLicenseButton);
            }
            return dr;
        }

        internal static Stream GetAssemblyResource(string resourceName)
        {
            Assembly assembly = Assembly.GetExecutingAssembly();
            return assembly.GetManifestResourceStream(resourceName);
        }

        #endregion

    }
}
