using System;
using System.Collections.Generic;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents a vehicle platfrom, which is a group of vehicles produced by the same manufacturer and which share the same body type.
    /// </summary>
    [Serializable]
    public sealed class Platform : Component
    {

        #region /*** Constructors ***/

        // Private constructor used for internal usage (such as cloning).
        Platform() : base() { }

        /// <summary>
        /// Initializes a new instance of the <see cref="Platform"/> class, using the specified platform name.
        /// </summary>
        /// <param name="name">The name to assign to this <see cref="Platform"/>.</param>
        internal Platform(string name) : base()
        {
            this._name     = name.Trim();
            this._year     = 0;
            this._revision = 0;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Platform"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Platform"/>.</returns>
        internal Platform Clone()
        {
            Platform value  = new Platform();
            value._name     = this._name;
            value._year     = this._year;
            value._revision = this._revision;
            this.CopyTo(value);
            return value;
        }

        /// <summary>
        /// Selects the platform leader for the vehicle platform from among the specified vehicles (note: all input vehicles must
        /// belong to this platform). A platform leader is assigned by filtering for vehicles with the highest level of technology
        /// utilization, then picking a vehicle with the lowest sales. In the event of a tie (if multiple vehicles have the same
        /// sales), the first vehicle, in chronological order, with the highest MSRP is selected as a leader.
        /// </summary>
        /// <param name="vehicles">The vehicles from which to compute the platform leader.</param>
        /// <returns>The platform leader of the component.</returns>
        protected override Vehicle SelectPlatformLeader(List<Vehicle> vehicles)
        {
            // scan for vehicle with highest technology utilization
            int techCount = 0;
            foreach (Vehicle veh in vehicles)
            {
                if (veh.Description.ZEVCandidate != HEVType.None) { continue; } // ignore zev-candidate vehicles
                //
                int vehTechCount = this.GetVehicleTechCount(veh);
                if (vehTechCount > techCount) { techCount = vehTechCount; }
            }
            // scan for and save all vehicles with technology utilization equal to techCount
            List<Vehicle> filteredVehicles = new List<Vehicle>();
            foreach (Vehicle veh in vehicles)
            {
                if (veh.Description.ZEVCandidate != HEVType.None) { continue; } // ignore zev-candidate vehicles
                //
                int vehTechCount = this.GetVehicleTechCount(veh);
                if (vehTechCount == techCount) { filteredVehicles.Add(veh); }
            }
            // select leader from the filtered list
            return base.SelectPlatformLeader(filteredVehicles);
        }
        int GetVehicleTechCount(Vehicle veh)
        {
            int[] techList = this.TechnologyList;
            int vehTechCount = 0;
            for (int i = 0; i < techList.Length; i++)
            {
                if (veh.ModelData.TechUsed[techList[i]] && !veh.ModelData.TechSuperseded[techList[i]]) { vehTechCount++; }
            }
            return vehTechCount;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>
        /// Gets an array of technology indexes that are applicable to <see cref="Platform"/>s.
        /// </summary>
        /// <returns>An array of technology indexes that are applicable to <see cref="Platform"/>s.</returns>
        public override int[] TechnologyList { get { return TechnologyIndexes.PlatformLevel; } }

        /// <summary>Gets the name of the <see cref="Platform"/>.</summary>
        public string Name
        {
            get
            {
                string name = this._name;
                if (this._year > 0)
                {
                    name += " (v." + this._year;
                    if (this._revision > 0)
                    {
                        name += "-R" + this._revision;
                    }
                    name += ")";
                }
                return name;
            }
        }

        #endregion

        #region /*** Variables ***/

        string _name;
        int _year;
        int _revision;

        #endregion

    }
}
