using System;
using Volpe.Cafe.Generic;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents an object that stores compliance modeling data, as double precision numeric values, for each regulatory class.
    /// </summary>
    [Serializable]
    public class RCDouble : RCValue<double>, ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="RCDouble"/> class.
        /// </summary>
        public RCDouble() : base() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCDouble"/> class using the specified initial value for all members.
        /// </summary>
        public RCDouble(double initialValue) : base(initialValue, initialValue, initialValue) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCDouble"/> class using the specified values.
        /// </summary>
        public RCDouble(double passengerCar, double lightTruck, double lightTruck2b3) : base(passengerCar, lightTruck, lightTruck2b3) { }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the sum of the two specified <see cref="RCDouble"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the two specified <see cref="RCDouble"/> values.</returns>
        public static RCDouble operator +(RCDouble value1, RCDouble value2)
        {
            RCDouble result = new RCDouble();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] + value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the sum of the specified <see cref="RCDouble"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="RCDouble"/> value and a <see cref="Double"/> value.</returns>
        public static RCDouble operator +(RCDouble value1, double value2)
        {
            return value1 + new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the sum of the specified <see cref="RCDouble"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="RCDouble"/> value and a <see cref="Double"/> value.</returns>
        public static RCDouble operator +(double value1, RCDouble value2)
        {
            return new RCDouble(value1) + value2;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the sum of the specified <see cref="RCDouble"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="RCDouble"/> value and an <see cref="Int32"/> value.</returns>
        public static RCDouble operator +(RCDouble value1, int value2)
        {
            return value1 + new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the sum of the specified <see cref="RCDouble"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="RCDouble"/> value and an <see cref="Int32"/> value.</returns>
        public static RCDouble operator +(int value1, RCDouble value2)
        {
            return new RCDouble(value1) + value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCDouble operator -(RCDouble value1, RCDouble value2)
        {
            RCDouble result = new RCDouble();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] - value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCDouble operator -(RCDouble value1, double value2)
        {
            return value1 - new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCDouble operator -(double value1, RCDouble value2)
        {
            return new RCDouble(value1) - value2;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCDouble operator -(RCDouble value1, int value2)
        {
            return value1 - new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCDouble operator -(int value1, RCDouble value2)
        {
            return new RCDouble(value1) - value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the product of the two specified <see cref="RCDouble"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the two specified <see cref="RCDouble"/> values.</returns>
        public static RCDouble operator *(RCDouble value1, RCDouble value2)
        {
            RCDouble result = new RCDouble();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] * value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the product of the specified <see cref="RCDouble"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="RCDouble"/> value and a <see cref="Double"/> value.</returns>
        public static RCDouble operator *(RCDouble value1, double value2)
        {
            return value1 * new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the product of the specified <see cref="RCDouble"/> value and a
        /// <see cref="Double"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="RCDouble"/> value and a <see cref="Double"/> value.</returns>
        public static RCDouble operator *(double value1, RCDouble value2)
        {
            return new RCDouble(value1) * value2;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the product of the specified <see cref="RCDouble"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="RCDouble"/> value and an <see cref="Int32"/> value.</returns>
        public static RCDouble operator *(RCDouble value1, int value2)
        {
            return value1 * new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the product of the specified <see cref="RCDouble"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="RCDouble"/> value and an <see cref="Int32"/> value.</returns>
        public static RCDouble operator *(int value1, RCDouble value2)
        {
            return new RCDouble(value1) * value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCDouble operator /(RCDouble value1, RCDouble value2)
        {
            RCDouble result = new RCDouble();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] / value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCDouble operator /(RCDouble value1, double value2)
        {
            return value1 / new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCDouble operator /(double value1, RCDouble value2)
        {
            return new RCDouble(value1) / value2;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCDouble operator /(RCDouble value1, int value2)
        {
            return value1 / new RCDouble(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCDouble operator /(int value1, RCDouble value2)
        {
            return new RCDouble(value1) / value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value, which is the copy of the specified <see cref="RCDouble"/> value.
        /// </summary>
        /// <param name="value">The value to copy.</param>
        /// <returns>A copy of the specified <see cref="RCDouble"/> value.</returns>
        public static RCDouble operator +(RCDouble value)
        {
            return RCDouble.Zero + value;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value, which is the negative of the specified <see cref="RCDouble"/> value.
        /// </summary>
        /// <param name="value">The value to negate.</param>
        /// <returns>The specified <see cref="RCDouble"/> value negated.</returns>
        public static RCDouble operator -(RCDouble value)
        {
            return RCDouble.Zero - value;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value, which is one more than the specified value (increment).
        /// </summary>
        /// <param name="value">The value to increment by one.</param>
        /// <returns>The specified value, incremented by one.</returns>
        public static RCDouble operator ++(RCDouble value)
        {
            return value + RCDouble.One;
        }
        /// <summary>
        /// Returns a new <see cref="RCDouble"/> value, which is one less than the specified value (decrement).
        /// </summary>
        /// <param name="value">The value to decrement by one.</param>
        /// <returns>The specified value, decremented by one.</returns>
        public static RCDouble operator --(RCDouble value)
        {
            return value - RCDouble.One;
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="RCDouble"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCDouble"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a copy of the current <see cref="RCDouble"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCDouble"/>.</returns>
        public new RCDouble Clone()
        {
            RCDouble value = new RCDouble();
            this.CopyTo(value);
            return value;
        }

        #endregion

        /// <summary>
        /// Returns the string representation of this <see cref="RCDouble"/> instance, using the specified format for all elements.
        /// </summary>
        /// <param name="format">A format string.</param>
        /// <returns>The string representation of the <see cref="RCDouble"/> instance, using the specified format.</returns>
        public string ToString(string format)
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString(format));
            }
            return "{" + s + "}";
        }
        /// <summary>
        /// Returns the string representation of this <see cref="RCDouble"/> instance, whose elements are rounded with the specified precision.
        /// </summary>
        /// <param name="digits">The rounding precision; e.g. number of digits after the period.</param>
        /// <returns>The string representation of the <see cref="RCDouble"/> instance.</returns>
        public string ToString(int digits)
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + Math.Round(this.Items[i], digits).ToString());
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Returns the compliance modeling data value for the specified regulatory class, or the <see cref="Total"/> of all
        /// regulatory classes if <see cref="RegulatoryClass.All"/> is specified.
        /// </summary>
        /// <param name="regClass">The regulatory class for which to obtain the value.</param>
        /// <returns>The compliance modeling data value for the specified regulatory class.</returns>
        public double GetValue(RegulatoryClass regClass)
        {
            return (regClass == RegulatoryClass.All) ? this.Total : this[regClass];
        }

        /// <summary>
        /// Returns a copy of this <see cref="RCDouble"/> instance, whose elements are rounded with the specified precision.
        /// </summary>
        /// <param name="digits">The rounding precision; e.g. number of digits after the period.</param>
        /// <returns>A copy of this <see cref="RCDouble"/> instance, whose elements are rounded with the specified precision.</returns>
        public RCDouble CreateRounded(int digits)
        {
            RCDouble result = new RCDouble();
            for (int i = 0; i < this.Items.Length; i++)
            {
                result.Items[i] = Math.Round(this.Items[i], digits);
            }
            return result;
        }

        //----- methods for computing averages -----
        /// <summary>
        /// Computes the arithmetic mean of this <see cref="RCDouble"/> value.
        /// </summary>
        public double ArithmeticMean()
        {
            return this.ArithmeticMean(RCDouble.One);
        }
        /// <summary>
        /// Computes the weighted arithmetic mean of this <see cref="RCDouble"/> value.
        /// </summary>
        /// <param name="weight">The weight value to use for averaging.</param>
        public double ArithmeticMean(RCDouble weight)
        {
            double result = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                result += weight.Items[i] * this.Items[i];
            }
            return result / weight.Total;
            //return (this * weight).Total / weight.Total;
        }
        /// <summary>
        /// Computes the geometric mean of this <see cref="RCDouble"/> value.
        /// </summary>
        public double GeometricMean()
        {
            return this.GeometricMean(RCDouble.One);
        }
        /// <summary>
        /// Computes the weighted geometric mean of this <see cref="RCDouble"/> value.
        /// </summary>
        /// <param name="weight">The weight value to use for averaging.</param>
        public double GeometricMean(RCDouble weight)
        {
            double result = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (this.Items[i] != 0)
                {
                    result += weight.Items[i] * Math.Log(this.Items[i]);
                }
            }
            return Math.Exp(result / weight.Total);
        }
        /// <summary>
        /// Computes the harmonic mean of this <see cref="RCDouble"/> value.
        /// </summary>
        public double HarmonicMean()
        {
            return this.HarmonicMean(RCDouble.One);
        }
        /// <summary>
        /// Computes the weighted harmonic mean of this <see cref="RCDouble"/> value.
        /// </summary>
        /// <param name="weight">The weight value to use for averaging.</param>
        public double HarmonicMean(RCDouble weight)
        {
            double result = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (this.Items[i] != 0)
                {
                    result += weight.Items[i] / this.Items[i];
                }
            }
            return (result == 0) ? 0 : weight.Total / result;
        }

        #endregion

        #region /*** Properties ***/

        //----- static values -----
        /// <summary>Gets the <see cref="RCDouble"/> value whose elements are all 0.</summary>
        public static RCDouble Zero { get { return new RCDouble(0); } }
        /// <summary>Gets the <see cref="RCDouble"/> value whose elements are all 1.</summary>
        public static RCDouble One { get { return new RCDouble(1); } }

        //----- instance values -----
        /// <summary>Gets the sum of the compliance modeling data of all regulatory classes.</summary>
        public double Total
        {
            get
            {
                double total = 0;
                for (int i = 0; i < this.Items.Length; i++)
                {
                    total += this.Items[i];
                }
                return total;
            }
        }

        #endregion

    }
}
