using System;
using System.Collections.Generic;
using System.IO;
using Volpe.Cafe;
using Volpe.Cafe.Data;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;

namespace Volpe.Cafe.IO
{
    /// <summary>
    /// Provides output file streams for logging errors, warnings, summary, and compliance findings of the Compliance models.
    /// </summary>
    [Serializable]
    public class LogWriter
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="LogWriter"/> class that is empty.
        /// </summary>
        LogWriter()
        {
            this._isEmpty = true;
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="LogWriter"/> class using the specified base path for all log files.
        /// </summary>
        /// <param name="basePath">The path where all log files will be stored.</param>
        public LogWriter(string basePath) : this(basePath, true, false) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="LogWriter"/> class using the specified base path for all log files, and
        /// whether to write compliance finding and/or extendend compliance finding logs.
        /// </summary>
        /// <param name="basePath">The path where all log files will be stored.</param>
        /// <param name="writeCFs">true, to write compliance finding logs; false, otherwise.</param>
        /// <param name="writeExtCFs">true, to write extended compliance finding logs; false, otherwise.</param>
        public LogWriter(string basePath, bool writeCFs, bool writeExtCFs)
        {
            if (!Directory.Exists(basePath)) { Directory.CreateDirectory(basePath); }
            //
            this._basePath    = basePath;
            this._writeCFs    = writeCFs;
            this._writeExtCFs = writeExtCFs;
            this._isEmpty     = false;
        }
        /// <summary>
        /// Cleans up the <see cref="LogWriter"/>, closing any opened streams, before it is reclaimed by the garbage collector.
        /// </summary>
        ~LogWriter()
        {
            this.Close();
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Generates and returns a log-header for the "compliance findings" log file.
        /// </summary>
        string GenerateComplianceLogHeader()
        {
            const string header =
                "LN\tCF\tScen\tMfr\tApplied\tEff\tTotal Delta RPE\tTotal Delta Fines\tTotal Delta Credits\tTotal FC Savings\tTotal Affected Sales\t" +
                "Reg-Class\tStandard\tCAFE\tNew CAFE\tFines\tNew Fines\tVeh\tEng\tTrn\tTech-Class\tTechs\tYear\t" +
                "FE\tShare\tNew FE\tNew Share\t" +
                "Sales\tCost\tPenalty Cost\tFC\tFC2\tFC-Adj\tFC-PwrAdj\tDelta FE\tDelta Off-Cycle\tDelta HP\tDelta CW\tDelta GVWR\tDelta GCWR";
            //
            return header;
        }

        void WriteLog(TextWriter tw, ref string value)
        {
            tw.Write(value);
        }

        // ----- Write methods -----
        /// <summary>
        /// Writes the specified value to the <see cref="Errors"/> stream.
        /// </summary>
        /// <param name="value">The value to write.</param>
        public void WriteError(string value)
        {
            if (this._errors != null) { this.WriteLog(this._errors, ref value); }
        }
        /// <summary>
        /// Writes the specified value to the <see cref="Warnings"/> stream.
        /// </summary>
        /// <param name="value">The value to write.</param>
        public void WriteWarning(string value)
        {
            if (this._warnings != null) { this.WriteLog(this._warnings, ref value); }
        }
        /// <summary>
        /// Writes the specified value to the <see cref="Summary"/> stream.
        /// </summary>
        /// <param name="value">The value to write.</param>
        public void WriteSummary(string value)
        {
            if (this._summary != null) { this.WriteLog(this._summary, ref value); }
        }

        // ----- WriteLine methods -----
        /// <summary>
        /// Writes the specified value, followed by a line terminator, to the <see cref="Errors"/> stream.
        /// </summary>
        /// <param name="value">The value to write.</param>
        public void WriteErrorLine(string value)
        {
            this.WriteError(value); this.WriteError(NewLine);
        }
        /// <summary>
        /// Writes the specified value, followed by a line terminator, to the <see cref="Warnings"/> stream.
        /// </summary>
        /// <param name="value">The value to write.</param>
        public void WriteWarningLine(string value)
        {
            this.WriteWarning(value); this.WriteWarning(NewLine);
        }
        /// <summary>
        /// Writes the specified value, followed by a line terminator, to the <see cref="Summary"/> stream.
        /// </summary>
        /// <param name="value">The value to write.</param>
        public void WriteSummaryLine(string value)
        {
            this.WriteSummary(value); this.WriteSummary(NewLine);
        }

        // ----- Open/Close methods -----
        /// <summary>
        /// Opens the log files for writing and initializes the parameters relevant for the compliance logging.
        /// </summary>
        /// <param name="append">true, to append the existing log files; false, to overwrite.</param>
        /// <param name="scenCount">The number of scenarios available for modeling.</param>
        public void Open(bool append, int scenCount)
        {
            if (this._basePath == null) { return; }
            this._errors   = new StreamWriter(this._basePath + "\\Errors.txt"  , append);
            this._warnings = new StreamWriter(this._basePath + "\\Warnings.txt", append);
            this._summary  = new StreamWriter(this._basePath + "\\Summary.txt" , append);
            //
            this.OpenCF(append, scenCount);
            this.OpenOther(append, scenCount);
            this._isOpen = true;
        }
        void OpenCF(bool append, int scenCount)
        {
            if (!this._writeCFs) { return; }

            //  open cf logs
            string header = this.GenerateComplianceLogHeader();
            this._cfs = new ComplianceWriter[scenCount];
            for (int i = 0; i < scenCount; i++)
            {
                this._cfs[i] = new ComplianceWriter(this._basePath + "\\cf_trace_sn" + i + ".txt", append);
                this._cfs[i].WriteLine(header);
            }

            //  open ext-cf logs
            if (this._writeExtCFs)
            {
                this._extCFs = new ComplianceWriter[scenCount];
                for (int i = 0; i < scenCount; i++)
                {
                    this._extCFs[i] = new ComplianceWriter(this._basePath + "\\ext-cf_trace_sn" + i + ".txt", append);
                    this._extCFs[i].WriteLine(header);
                }
            }
        }
        /// <summary>
        /// Allows inherited classes to open additional log files.
        /// </summary>
        /// <param name="append">true, to append the existing log files; false, to overwrite.</param>
        /// <param name="scenCount">The number of scenarios available for modeling.</param>
        protected virtual void OpenOther(bool append, int scenCount) { }

        /// <summary>
        /// Closes all log files.
        /// </summary>
        public void Close()
        {
            if (this._errors   != null) { this.Errors  .Close(); this._errors   = null; }
            if (this._warnings != null) { this.Warnings.Close(); this._warnings = null; }
            if (this._summary  != null) { this.Summary .Close(); this._summary  = null; }
            //
            this.CloseComplianceWriters(this._cfs);
            this.CloseComplianceWriters(this._extCFs);
            this.CloseOther();
            this._isOpen = false;
        }
        /// <summary>
        /// Closes all <see cref="ComplianceWriter"/>s in the specified array.
        /// </summary>
        /// <param name="writers">An array of <see cref="ComplianceWriter"/>s to close.</param>
        protected void CloseComplianceWriters(ComplianceWriter[] writers)
        {
            if (writers != null)
            {
                for (int i = 0; i < writers.Length; i++)
                {
                    if (writers[i] != null) { writers[i].Close(); writers[i] = null; }
                }
            }
        }
        /// <summary>
        /// Allows inherited classes to close additional log files.
        /// </summary>
        protected virtual void CloseOther() { }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets or sets the path where all log files are located.</summary>
        /// <exception cref="InvalidOperationException">When setting, LogWriter is empty or opened.</exception>
        public string LogPath
        {
            get { return this._basePath; }
            set
            {
                if (this._isEmpty) { throw new InvalidOperationException("Cannot modify an empty LogWriter--it is read-only."); }
                if (this._isOpen)  { throw new InvalidOperationException("The LogWriter is opened; path cannot be modified."); }
                this._basePath = value;
            }
        }

        /// <summary>Gets whether this <see cref="LogWriter"/> writes compliance finding logs.</summary>
        public bool WriteCFs { get { return this._writeCFs; } }

        /// <summary>Gets whether this <see cref="LogWriter"/> writes extended compliance finding logs.</summary>
        public bool WriteExtCFs { get { return this._writeExtCFs; } }

        /// <summary>Gets whether the <see cref="LogWriter"/> is empty.</summary>
        public bool IsEmpty { get { return this._isEmpty; } }

        /// <summary>Gets whether the logs of this <see cref="LogWriter"/> are opened for writing.</summary>
        public bool IsOpen { get { return this._isOpen; } }

        /// <summary>Gets the errors.txt file stream to use for the compliance modeling process.  This value may be null.</summary>
        public TextWriter Errors { get { return this._errors;   } }
        /// <summary>Gets the warnings.txt file stream to use for the compliance modeling process.  This value may be null.</summary>
        public TextWriter Warnings { get { return this._warnings; } }
        /// <summary>Gets the summary.txt file stream to use for the compliance modeling process.  This value may be null.</summary>
        public TextWriter Summary { get { return this._summary;  } }

        /// <summary>Gets an array of file streams (one per scenario) to use for logging of compliance findings.
        ///   This value may be null.</summary>
        public ComplianceWriter[] CFs { get { return this._cfs; } }
        /// <summary>Gets an array of file streams (one per scenario) to use for extended logging of compliance findings.
        ///   This value may be null.</summary>
        public ComplianceWriter[] ExtCFs { get { return this._extCFs; } }

        #endregion


        #region /*** Variables ***/

        // ----- constant/readonly variables -----
        /// <summary>Represents an empty <see cref="LogWriter"/> instance.  This value is read-only.</summary>
        public static readonly LogWriter Empty = new LogWriter();

        /// <summary>Represents an empty line.  This value is read-only.</summary>
        public static readonly string NewLine = "\n";

        // ----- property and runtime variables -----
        string _basePath;
        bool   _writeCFs;
        bool   _writeExtCFs;
        bool   _isEmpty;
        bool   _isOpen;

        TextWriter _errors;
        TextWriter _warnings;
        TextWriter _summary;

        ComplianceWriter[] _cfs;
        ComplianceWriter[] _extCFs;

        #endregion

    }
}
