﻿#region << Using Directives >>
using System;
using System.IO;
using Volpe.Cafe.Data;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
using System.Collections.Generic;
#endregion

namespace Volpe.Cafe.IO.Reporting.CSV
{
    /// <summary>
    /// Provides the default CSV report generator for the compliance model.
    /// </summary>
    [Serializable]
    public class CsvReportGenerator
    {

        #region /*** Ctors ***/

        CsvReportGenerator()
        {
            this._isEmpty = true;
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="CsvReportGenerator"/> class.
        /// </summary>
        public CsvReportGenerator(ICompliance compliance)
        {
            this._compliance = compliance;
            this._compliance.ScenarioCompleted += new ModelingProgressEventHandler(this.Compliance_ScenarioCompleted);
            this._compliance.ModelingCompleted += new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
            this._compliance.ModelingStopped   += new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
            this._isEmpty = false;
        }

        #endregion

        #region /*** Methods ***/

        void Compliance_ModelingStoppedOrCompleted(object sender, ModelingEventArgs e)
        {
            // de-register events
            this._compliance.ScenarioCompleted -= new ModelingProgressEventHandler(this.Compliance_ScenarioCompleted);
            this._compliance.ModelingCompleted -= new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
            this._compliance.ModelingStopped   -= new ModelingEventHandler(this.Compliance_ModelingStoppedOrCompleted);
        }
        void Compliance_ScenarioCompleted(object sender, ModelingProgressEventArgs e)
        {
            this.GenerateReportsForScenario((e.Progress == null) ? null : e.Progress.Scenario);
        }
        /// <summary>
        /// Begins generating reports for the specified <see cref="Scenario"/>.
        /// </summary>
        /// <param name="scen">The compliance scenario for which to generate modeling reports.</param>
        protected void GenerateReportsForScenario(Scenario scen)
        {
            if (scen == null || this.IsModelStopped()) { return; }
            //
            try
            {
                this._reporting = true;
                this._reportingProgress = "Initializing reports ...";

                // open reports
                this._csvReports = this.OpenReports(scen.Index);

                // parse data
                if (!this.IsModelStopped())
                {
                    for (int i = 0; i < this._compliance.ModelYears.Length; i++)
                    {
                        if (this.IsModelStopped()) { break; }
                        //
                        ModelYear year = this._compliance.ModelYears[i];
                        this.WriteReportData(scen, year);
                    } // next i (model year)
                }
            }
            catch (OutputException ex) { Console.WriteLine(ex.ToString()); throw ex; }
            catch (Exception       ex) { Console.WriteLine(ex.ToString()); throw new OutputException("<unknown file>", null, ex); }
            finally
            {
                this.CloseReports(scen.Index);  // save & close the reports
                //
                this._reportingProgress = "Reporting for scenario " + scen.Index + " completed.";
                this._reporting = false;
            }
        }

        /// <summary>
        /// Opens modeling reports.
        /// </summary>
        /// <param name="snIndex">The index of the scenario being reported.</param>
        /// <returns>A list of opened CSV reports.</returns>
        protected virtual List<CsvReportingBase> OpenReports(int snIndex)
        {
            return this.OpenReports(snIndex, this.OutputPath);
        }
        /// <summary>
        /// Opens modeling reports, using the specified path as the base output path.
        /// </summary>
        /// <param name="snIndex">The index of the scenario being reported.</param>
        /// <param name="path">The output path where CSV reports are written to.</param>
        /// <returns>A list of opened CSV reports.</returns>
        protected virtual List<CsvReportingBase> OpenReports(int snIndex, string path)
        {
            this.UpdateProgress("Opening CSV reports ...");
            if (!Directory.Exists(path)) { Directory.CreateDirectory(path); }
            //
            List<CsvReportingBase> csvReports = new List<CsvReportingBase>(10);
            //
            csvReports.Add(new CsvComplianceReport               (path + "\\compliance_report.csv"             , (snIndex > 0)));
            csvReports.Add(new CsvTechnologyUtilizationReport    (path + "\\technology_utilization_report.csv" , (snIndex > 0)));
            csvReports.Add(new CsvVehiclesReport                 (path + "\\vehicles_report.csv"               , (snIndex > 0)));
            csvReports.Add(new CsvSocietalEffectsByVehClassReport(path + "\\societal_effects_report_vc.csv"    , (snIndex > 0)));
            csvReports.Add(new CsvSocietalEffectsByRegClassReport(path + "\\societal_effects_report_rc.csv"    , (snIndex > 0)));
            csvReports.Add(new CsvAnnualSocietalEffectsReport    (path + "\\annual_societal_effects_report.csv", (snIndex > 0)));
            csvReports.Add(new CsvSocietalCostsByVehClassReport  (path + "\\societal_costs_report_vc.csv"      , (snIndex > 0)));
            csvReports.Add(new CsvSocietalCostsByRegClassReport  (path + "\\societal_costs_report_rc.csv"      , (snIndex > 0)));
            csvReports.Add(new CsvAnnualSocietalCostsReport      (path + "\\annual_societal_costs_report.csv"  , (snIndex > 0)));
            //
            return csvReports;
        }
        /// <summary>
        /// Writes report data.
        /// </summary>
        /// <param name="scen">The scenario to report on.</param>
        /// <param name="year">The model year to report on.</param>
        /*protected virtual*/ void WriteReportData(Scenario scen, ModelYear year)
        {
            if (this._csvReports != null)
            {
                for (int i = 0; i < this._csvReports.Count; i++)
                {
                    this.WriteReportData(this._csvReports[i], scen, year);
                }
            }
        }
        /// <summary>
        /// Writes report data for the specied <see cref="CsvReportingBase"/> report.
        /// </summary>
        /// <param name="report">The CSV report to write.</param>
        /// <param name="scen">The scenario to report on.</param>
        /// <param name="year">The model year to report on.</param>
        /*protected*/ void WriteReportData(CsvReportingBase report, Scenario scen, ModelYear year)
        {
            if (report != null && !report.Disposed)
            {
                this.UpdateProgress("Writing model year " + year.Year + " data for the " + report.ReportName + ".");
                report.ParseData(scen, year, this._compliance);
            }
        }
        /// <summary>
        /// Saves and closes any open modeling reports.
        /// </summary>
        /*protected virtual*/ void CloseReports(int snIndex)
        {
            if (this._csvReports != null)
            {
                for (int i = 0; i < this._csvReports.Count; i++)
                {
                    this.CloseReport(this._csvReports[i]);
                    this._csvReports[i] = null;
                }
            }
        }
        /// <summary>
        /// Saves and closes the specified report.  If modeling was stopped, the report will contain data that was written thus far.
        /// </summary>
        /// <param name="report">The CSV report to close.</param>
        /*protected*/ void CloseReport(CsvReportingBase report)
        {
            if (report != null && !report.Disposed)
            {
                this.UpdateProgress("Saving " + report.ReportName + ".");
                //bool saveReport = !this.IsModelStopped();
                report.Close();
                //if (!saveReport) { File.Delete(report.Path); }
            }
        }

        /// <summary>
        /// Updates the reporting progress with the specified progress string.
        /// </summary>
        /*protected*/ void UpdateProgress(string progress)
        {
            this._reportingProgress = progress;
        }
        /// <summary>
        /// Returns true, if model state is stopped or stopping; false, otherwise.
        /// </summary>
        protected bool IsModelStopped()
        {
            return (this._compliance.Stopped || this._compliance.State == ModelingState.StopRequested);
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets whether the generator is currently parsing or writing reports.</summary>
        public bool Reporting { get { return !this._isEmpty && this._reporting; } }
        /// <summary>Gets the report generator's progress.</summary>
        public string ReportingProgress { get { return this._reportingProgress; } }

        /// <summary>Gets the compliance model with which this <see cref="CsvReportGenerator"/> is associated.</summary>
        protected ICompliance Compliance { get { return this._compliance; } }

        /// <summary>Gets the default directory where output files are saved to.</summary>
        protected string OutputPath { get { return this._compliance.Settings.OutputSettings.OutputPath + "\\reports-csv"; } }

        #endregion

        #region /*** Variables ***/

        // ----- constant/readonly variables -----
        /// <summary>Represents an empty <see cref="CsvReportGenerator"/> instance.  This value is read-only.</summary>
        public static readonly CsvReportGenerator Empty = new CsvReportGenerator();

        // ----- compliance & reporting variables -----
        ICompliance _compliance;

        // model reports
        List<CsvReportingBase> _csvReports;

        // ----- runtime variables -----
        bool   _isEmpty;
        bool   _reporting;
        string _reportingProgress;

        #endregion

    }
}
