﻿#region << Using Directives >>
using System;
using System.Collections.Generic;
using System.Threading;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.IO.Reporting.CSV;
using Volpe.Cafe.IO.Reporting.XL;
using Volpe.Cafe.Settings;
using Volpe.Cafe.UI;
using Volpe.Cafe.Utils;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides an abstract base class for the compliance model implementation.
    /// </summary>
    /// <remarks>
    /// The <see cref="ComplianceBase"/> class provides only the basic implementation of the <see cref="ICompliance"/> interface
    /// and can be used to simplify the implementation for extending classes.  When the
    /// <see cref="Start(Industry, ModelingSettings)"/> method is called, the <see cref="ComplianceBase"/> class spawns a new
    /// thread on which the modeling process will run.
    /// 
    /// The example below provides a sample <see cref="ICompliance"/> implementation by extending this class.
    /// </remarks>
    /// <example>
    /// <code>
    /// TODO: ... insert example code here ...
    /// </code>
    /// </example>
    [Serializable]
    [ModelDescription("ComplianceBase", "Provides an abstract base class for the compliance model implementation.", 1.2F)]
    public abstract class ComplianceBase : ICompliance
    {

        #region /*** Events ***/

        #region /* ICompliance Members */

        /// <summary>Occurs whenever the compliance model displays a prompt, which requires user action.</summary>
        public event PromptEventHandler Prompt;

        /// <summary>Occurs after the compliance model has started.</summary>
        public event ModelingEventHandler ModelingStarted;
        /// <summary>Occurs after the compliance model has stopped, but did not complete successfully.  This typically indicates
        ///   that the user has aborted the modeling process or an exception in the code has occured.</summary>
        public event ModelingEventHandler ModelingStopped;
        /// <summary>Occurs after the compliance model has completed successfully.</summary>
        public event ModelingEventHandler ModelingCompleted;
        /// <summary>Occurs after the compliance model has started, stopped, or completed.  This event is generated along side
        ///   <see cref="ModelingStarted"/>, <see cref="ModelingStopped"/>, and <see cref="ModelingCompleted"/> events.  The
        ///   <see cref="ModelingChanged"/> event also occurs when a model start or stop has been requested by the user.</summary>
        public event ModelingEventHandler ModelingChanged;

        /// <summary>Occurs before the processing of a single scenario (along with any pre-processing) has started.</summary>
        public event ModelingProgressEventHandler ScenarioStarted;
        /// <summary>Occurs after the processing of a single scenario (along with any post-processing) has completed.</summary>
        public event ModelingProgressEventHandler ScenarioCompleted;
        /// <summary>Occurs before the processing of a single model year (along with any pre-processing) has started.</summary>
        public event ModelingProgressEventHandler ModelYearStarted;
        /// <summary>Occurs after the processing of a single model year (along with any post-processing) has completed.</summary>
        public event ModelingProgressEventHandler ModelYearCompleted;
        /// <summary>Occurs before the processing of a single manufacturer (along with any pre-processing) has started.</summary>
        public event ModelingProgressEventHandler ManufacturerStarted;
        /// <summary>Occurs after the processing of a single manufacturer (along with any post-processing) has completed.</summary>
        public event ModelingProgressEventHandler ManufacturerCompleted;

        #endregion

        #endregion


        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="ComplianceBase"/> class.
        /// </summary>
        public ComplianceBase()
        {
            this._state = ModelingState.Unstarted;
            this._abortRequested = false;
        }

        /// <summary>
        /// Cleans up any resources being used by this <see cref="ComplianceBase"/> instance before it is reclaimed by the garbage collector.
        /// </summary>
        ~ComplianceBase()
        {
            // terminate the compliance process if it is still running
            this.Abort(false);

            // clear all registered events
            this.Prompt                = null;
            //
            this.ModelingStarted       = null;
            this.ModelingStopped       = null;
            this.ModelingCompleted     = null;
            this.ModelingChanged       = null;
            //
            this.ScenarioStarted       = null;
            this.ScenarioCompleted     = null;
            this.ModelYearStarted      = null;
            this.ModelYearCompleted    = null;
            this.ManufacturerStarted   = null;
            this.ManufacturerCompleted = null;
        }

        #endregion


        #region /*** Methods ***/

        #region /* Methods for raising events */

        /// <summary>
        /// Raises the <see cref="ComplianceBase.Prompt"/> event.
        /// </summary>
        /// <param name="e">The <see cref="PromptEventArgs"/> to pass to the event.</param>
        protected virtual void OnPrompt(PromptEventArgs e)
        {
            if (this.Prompt != null)
            {
                this.Prompt(this, e);
            }
        }

        /// <summary>
        /// Raises the <see cref="ComplianceBase.ModelingStarted"/> event.
        /// </summary>
        protected virtual void OnModelingStarted()
        {
            if (this.ModelingStarted != null)
            {
                this.ModelingStarted(this, new ModelingEventArgs(this._state));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ModelingStopped"/> event.
        /// </summary>
        protected virtual void OnModelingStopped()
        {
            if (this.ModelingStopped != null)
            {
                this.ModelingStopped(this, new ModelingEventArgs(this._state));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ModelingCompleted"/> event.
        /// </summary>
        protected virtual void OnModelingCompleted()
        {
            if (this.ModelingCompleted != null)
            {
                this.ModelingCompleted(this, new ModelingEventArgs(this._state));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ModelingChanged"/> event.
        /// </summary>
        protected virtual void OnModelingChanged()
        {
            if (this.ModelingChanged != null)
            {
                this.ModelingChanged(this, new ModelingEventArgs(this._state));
            }
        }

        /// <summary>
        /// Raises the <see cref="ComplianceBase.ScenarioStarted"/> event.
        /// </summary>
        /// <remarks>
        /// The event data will include the contents of the current <see cref="Progress"/> property.
        /// </remarks>
        protected virtual void OnScenarioStarted()
        {
            if (this.ScenarioStarted != null)
            {
                this.ScenarioStarted(this, new ModelingProgressEventArgs(this.Progress));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ScenarioCompleted"/> event.
        /// </summary>
        /// <remarks>
        /// The event data will include the contents of the current <see cref="Progress"/> property.
        /// </remarks>
        protected virtual void OnScenarioCompleted()
        {
            if (this.ScenarioCompleted != null)
            {
                this.ScenarioCompleted(this, new ModelingProgressEventArgs(this.Progress));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ModelYearStarted"/> event.
        /// </summary>
        /// <remarks>
        /// The event data will include the contents of the current <see cref="Progress"/> property.
        /// </remarks>
        protected virtual void OnModelYearStarted()
        {
            if (this.ModelYearStarted != null)
            {
                this.ModelYearStarted(this, new ModelingProgressEventArgs(this.Progress));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ModelYearCompleted"/> event.
        /// </summary>
        /// <remarks>
        /// The event data will include the contents of the current <see cref="Progress"/> property.
        /// </remarks>
        protected virtual void OnModelYearCompleted()
        {
            if (this.ModelYearCompleted != null)
            {
                this.ModelYearCompleted(this, new ModelingProgressEventArgs(this.Progress));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ManufacturerStarted"/> event.
        /// </summary>
        /// <remarks>
        /// The event data will include the contents of the current <see cref="Progress"/> property.
        /// </remarks>
        protected virtual void OnManufacturerStarted()
        {
            if (this.ManufacturerStarted != null)
            {
                this.ManufacturerStarted(this, new ModelingProgressEventArgs(this.Progress));
            }
        }
        /// <summary>
        /// Raises the <see cref="ComplianceBase.ManufacturerCompleted"/> event.
        /// </summary>
        /// <remarks>
        /// The event data will include the contents of the current <see cref="Progress"/> property.
        /// </remarks>
        protected virtual void OnManufacturerCompleted()
        {
            if (this.ManufacturerCompleted != null)
            {
                this.ManufacturerCompleted(this, new ModelingProgressEventArgs(this.Progress));
            }
        }

        #endregion

        /// <summary>
        /// Creates a new instance with the same underlying type as this <see cref="ICompliance"/>.
        /// </summary>
        /// <returns>A new <see cref="ICompliance"/> object with the same underlying type as the current instance.</returns>
        public abstract ICompliance CreateNew();

        /// <summary>
        /// Starts compliance modeling in a new thread, using the specified modeling data and settings, as well as the specified
        /// <see cref="LogWriter"/> object.
        /// </summary>
        /// <param name="data">The modeling data to use for compliance modeling.</param>
        /// <param name="settings">The modeling settings to use for compliance modeling.</param>
        /// <remarks>
        /// A compliance model should always be started in a separate thread.
        /// </remarks>
        public void Start(Industry data, ModelingSettings settings)
        {
            if (this.Running || this._state == ModelingState.StartRequested)
            {   // the model is already running -- exit
                string errString = "The compliance modeling process has already started at: " +
                    this._startTime.ToString("M/dd/yyyy h:mm:ss tt") + " -- exiting ...";
                this._logWriter.WriteErrorLine(errString);
                this.OnPrompt(new PromptEventArgs("CAFE Model Error", errString, PromptOption.Ok));
                return;
            }

            // reset start time
            this._startTime = new DateTime(0);

            //--------------------------------------//
            // begin pre-modeling preparations ...  //
            //--------------------------------------//
            this.UpdateState(ModelingState.StartRequested);     // set the modeling state as starting (start-requested)

            // make user changes to the settings
            this.OverrideUserSettings(settings);

            // set references
            this._data      = data;
            this._settings  = settings;

            // pre-calculate some values
            if (this._settings != null)
            {   // get number of scenarios
                if (this._settings.Scenarios != null) { this._scenCount = this._settings.Scenarios.Count; }
            }
            if (this._data != null)
            {
                this._minYear      = this._data.MinYear.Year;
                this._maxYear      = this._data.MaxYear.Year;
                this._minYearIndex = this._data.MinYear.Index;
                this._maxYearIndex = this._data.MaxYear.Index;
                this._yearCount    = this._maxYear - this._minYear + 1;
                this._vehCount     = this._data.VehicleCount;
                this._mfrCount     = this._data.ManufacturerCount;
                this._mfrVehCount  = new int   [this._mfrCount];
                this._mfrNames     = new string[this._mfrCount];
                //
                for (int i = 0; i < this._mfrCount; i++)
                {
                    Manufacturer mfr = this._data.Manufacturers[i];
                    this._mfrVehCount[i] = mfr.VehicleCount;
                    this._mfrNames   [i] = Interaction.GetTitleCase(mfr.Description.Name, 4);
                }
                //
                this._modelYears = new ModelYear[this._yearCount];
                for (int i = this._minYear; i <= this._maxYear; i++) { this._modelYears[i - this._minYear] = new ModelYear(i); }
                //
                // initialize the model data for baseline and alternative scenario
                this._baseData = new Industry[this._maxYearIndex + 1];
                this._scenData = new Industry[this._maxYearIndex + 1];

                // initialize the effects data for baseline and alternative scenario
                this._minEffectsYear = 1;
                this._maxEffectsYear = 0;
                if (this._settings != null)
                {
                    if (this._settings.OperatingModes.FleetAnalysis)
                    {
                        this._minEffectsYear = this._settings.Parameters.FuelEconomyData.MinMY;
                        this._maxEffectsYear = this._settings.Parameters.FuelEconomyData.MaxMY;
                    }
                    else
                    {
                        this._minEffectsYear = this._minYear;
                        this._maxEffectsYear = this._maxYear;
                    }
                }
                int effectsYears = this._maxEffectsYear - this._minEffectsYear + 1;
                this._baseEffects = new EffectsData[effectsYears];
                this._scenEffects = new EffectsData[effectsYears];
            }

            // --- initialize logs and reports ---
            // initialize log writer
            if (settings.OutputSettings.DisableLogWriter) { this._logWriter = LogWriter.Empty; }
            else { this._logWriter = this.CreateLogWriter(); }
            // initialize XL report generator
            if (!settings.OutputSettings.GenerateXLReports) { this._xlReportGenerator = XlReportGenerator.Empty; }
            else { this._xlReportGenerator = this.CreateXlReportGenerator(); }
            // initialize CSV report generator
            if (!settings.OutputSettings.GenerateCSVReports) { this._csvReportGenerator = CsvReportGenerator.Empty; }
            else { this._csvReportGenerator = this.CreateCsvReportGenerator(); }

            // validate the model data and other inputs
            string validatedString = this.Validate();
            this._valid = (validatedString == string.Empty);
            if (!this._valid)
            {   // the model is not valid -- log, prompt, and exit
                this._logWriter.WriteErrorLine(new string('-', 25));
                this._logWriter.WriteErrorLine(validatedString);
                this._logWriter.WriteErrorLine(new string('-', 25));
                this.OnPrompt(new PromptEventArgs("CAFE Model Error", validatedString, PromptOption.Ok));
                this.UpdateState(ModelingState.Stopped);    // set state as stopped
                return;
            }

            //--------------------------------------//
            // start modeling in a new thread ...   //
            //--------------------------------------//
            this.Start();
        }
        void Start()
        {
            if (!this._logWriter.IsOpen)
            {
                this._logWriter.Open(false, this._scenCount);
            }

            // start the compliance modeling process in a new thread
            this._complianceThread = new Thread(new ThreadStart(this.StartMainComplianceThread));
            this._complianceThread.Name = "MainComplianceThread";
            this._startTime = DateTime.Now;
            this._abortRequested = false;
            this.UpdateState(ModelingState.Running);                            // set the modeling state as running
            this._complianceThread.Start();
        }
        void StartMainComplianceThread()
        {
            ModelingState endState = ModelingState.Unstarted;   // the end state of the model (stopped or completed)
            string modelType = "<UNNAMED-MODEL>";
            string modelDescription = "N/A";
            try
            {   // get the title from a description attribute of the current compliance model
                ModelDescriptionAttribute mda = (ModelDescriptionAttribute)Attribute.GetCustomAttribute(this.GetType(),
                    typeof(ModelDescriptionAttribute));
                modelType        = mda.Title;
                modelDescription = mda.Description;

                // start the compliance modeling process
                this._lassErrorMessage = null;
                this._logWriter.WriteSummaryLine(modelType + " started at " + DateTime.Now + ".");
                this.StartInternal();
                endState = (this._abortRequested) ? ModelingState.Stopped : ModelingState.Completed;
            }
            catch (Exception ex)
            {   // an exception has occured -- set the state as stopped
                this._lassErrorMessage = ex.Message;
                this._logWriter.WriteErrorLine(ex.ToString());
                endState = ModelingState.Stopped;
            }
            finally
            {
                this._logWriter.WriteSummaryLine(new string('=', 50));
                // write out machine and use name
                this._logWriter.WriteSummaryLine("---------- User Settings ----------");
                this._logWriter.WriteSummaryLine("  Machine Name: " + SystemInfo.MachineName);
                this._logWriter.WriteSummaryLine("  OS Version:   " + SystemInfo.OSVersion);
                this._logWriter.WriteSummaryLine("  CLR Version:  " + SystemInfo.CLRVersion);
                this._logWriter.WriteSummaryLine("  CAFE Version: " + ApplicationInfo.ApplicationVersion + " (" + ApplicationInfo.ApplicationTimestamp + ")");
                this._logWriter.WriteSummaryLine("  User Name:    " + System.Environment.UserName);
                this._logWriter.WriteSummaryLine(new string('=', 50));
                // write out run-specific settings
                this._logWriter.WriteSummaryLine("---------- General Settings ----------");
                this._logWriter.WriteSummaryLine("  ModelType="        + modelType);
                this._logWriter.WriteSummaryLine("  ModelDescription=" + modelDescription);
                this._logWriter.WriteSummaryLine("  UserNotes="        + this.UserNotes);
                this._logWriter.WriteSummaryLine("  UserKeywords="     + this.UserKeywords);
                // write out modeling settings
                if (this._settings != null)
                {
                    this._settings.WriteSummaryLog(this._logWriter);
                }
                if (this._data != null)
                {
                    this._data.WriteSummaryLog(this._logWriter);
                }
                this._logWriter.WriteSummaryLine(new string('=', 50));
                this._logWriter.WriteSummaryLine(modelType + " " + endState + " at " + DateTime.Now + " in " +
                    (this.Runtime / 1000) + " sec.");
                this.UpdateState(endState);
            }
        }
        /// <summary>
        /// This method is called by the <see cref="Start(Industry, ModelingSettings)"/> method and must be overriden by a
        /// subclass.  This method always begins executing in a new thread.  When implemented by a subclass, starts the actual
        /// compliance modeling process.
        /// </summary>
        protected abstract void StartInternal();

        /// <summary>
        /// Aborts all threads spawned by this <see cref="ICompliance"/> instance, and terminates compliance modeling immediately.
        /// </summary>
        public void Abort()
        {
            this.Abort(false);
        }
        /// <summary>
        /// Aborts all threads spawned by this <see cref="ICompliance"/> instance, and terminates compliance modeling, optionally
        /// waiting for the pending operations to complete.
        /// </summary>
        /// <param name="abortWhenReady">If true, specifies that the pending operations will be completed prior to aborting the
        ///   modeling process; otherwise, modeling will be terminated immediately.</param>
        public void Abort(bool abortWhenReady)
        {
            if (this._state == ModelingState.Running || (!abortWhenReady && this._state == ModelingState.StopRequested))
            {   // set modeling state
                if (this._state != ModelingState.StopRequested) { this.UpdateState(ModelingState.StopRequested); }
                this._abortRequested = true;
                // begin stopping the compliance modeling process
                this.AbortInternal(abortWhenReady);
            }
        }
        /// <summary>
        /// This method is called by the <see cref="Abort(bool)"/> method and can be implemented by a subclass.  When implemented
        /// by a subclass, aborts all compliance threads and terminates the compliance modeling process, optionally waiting for
        /// the pending operations to complete.  When overriding, sub-classes should still call base.AbortInternal(abortWhenReady)
        /// to ensure desired behavior.
        /// </summary>
        /// <param name="abortWhenReady">If true, specifies that the pending operations will be completed prior to aborting the
        ///   modeling process; otherwise, modeling will be terminated immediately.</param>
        protected virtual void AbortInternal(bool abortWhenReady)
        {
            // begin stopping
            if (!abortWhenReady)
            {   // kill the thread
                try
                {
                    if (this._complianceThread != null && this._complianceThread.IsAlive)
                    {
                        this._complianceThread.Abort();
                        this._complianceThread = null;
                    }
                }
                catch (Exception ex)
                {
                    this._logWriter.WriteErrorLine("Failed to terminate thread: " +
                        ((this._complianceThread == null) ? "null" : this._complianceThread.Name) + "\nException: " + ex.Message);
                }

                // set the modeling state as stopped
                this.UpdateState(ModelingState.Stopped);
            }
        }

        /// <summary>
        /// Creates a new <see cref="LogWriter"/> instance to use for saving the log files generated during compliance modeling.
        /// Sub-classes may override this method to return a custom log writer.  Custom log writers must inherit from the
        /// <see cref="LogWriter"/> class.
        /// </summary>
        /// <returns>A new <see cref="LogWriter"/> instance to use for saving the log files generated during compliance modeling.</returns>
        protected virtual LogWriter CreateLogWriter()
        {
            return new LogWriter(
                this._settings.OutputSettings.OutputPath + "\\logs",
                this._settings.OutputSettings.WriteLogFiles,
                this._settings.OutputSettings.WriteExtendedLogFiles);
        }
        /// <summary>
        /// Creates a new <see cref="XlReportGenerator"/> instance to use for generating compliance model reports.  Sub-classes
        /// may override this method to return a custom report generator.  Custom report generators must inherit from the
        /// <see cref="XlReportGenerator"/> class.
        /// </summary>
        /// <returns>A new <see cref="XlReportGenerator"/> instance to use for generating compliance model reports.</returns>
        protected virtual XlReportGenerator CreateXlReportGenerator()
        {
            return XlReportGenerator.Empty;
        }
        /// <summary>
        /// Creates a new <see cref="CsvReportGenerator"/> instance to use for generating compliance model reports.  Sub-classes
        /// may override this method to return a custom report generator.  Custom report generators must inherit from the
        /// <see cref="CsvReportGenerator"/> class.
        /// </summary>
        /// <returns>A new <see cref="CsvReportGenerator"/> instance to use for generating compliance model reports.</returns>
        protected virtual CsvReportGenerator CreateCsvReportGenerator()
        {
            return new CsvReportGenerator(this);
        }

        /// <summary>
        /// Returns an array of "working" or "final" compliance modeling data for the specified scenario.
        /// </summary>
        /// <param name="scen">The scenario for which to obtain the array of compliance modeling data.</param>
        /// <returns>An array of compliance modeling data for the specified scenario.</returns>
        /// <remarks>
        /// The array returned by this method should be initialized to a length capable of storing the greatest model year index.
        /// If, however, the compliance model has not yet been run, the return value may be null.  Once modeling begins, if the
        /// compliance model has not started processing the specific scenario, the return value should be an array where each
        /// element is null.  If the compliance model has not processed a specific model year of the scenario, the return value
        /// should be an array where only the unprocessed model years are null.
        /// </remarks>
        public Industry[] GetData(Scenario scen)
        {
            if (scen.IsBaseline) { return this._baseData; }
            //
            if (this._scen == null) { return null; }
            if (this._scen.Index != scen.Index)
            {
                // TODO:  obtain from cache
                return null;
            }
            return this._scenData;
        }
        /// <summary>
        /// Returns the "working" or "final" compliance modeling data for the specified scenario and model year.
        /// </summary>
        /// <param name="scen">The scenario for which to obtain the compliance modeling data.</param>
        /// <param name="year">The model year for which to obtain the compliance modeling data.</param>
        /// <returns>The compliance modeling data for the specified scenario and model year.</returns>
        /// <remarks>
        /// If the compliance model has not processed a specific scenario and/or model year, the return value should be null.
        /// </remarks>
        public Industry GetData(Scenario scen, ModelYear year)
        {
            Industry[] scenData = this.GetData(scen);
            return (scenData != null && year.Index - this._minYearIndex < this._yearCount) ? scenData[year.Index] : null;
        }
        /// <summary>
        /// Sets the "working" compliance modeling data object for the current scenario and the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to set the compliance modeling data.</param>
        /// <param name="data">The modeling data to set for the current scenario and the specified model year.</param>
        protected void SetActiveScenarioData(ModelYear year, Industry data)
        {
            if (this._scen != null)
            {
                this._scenData[year.Index] = data;
                if (this._scen.IsBaseline) { this._baseData[year.Index] = data; }
            }
        }
        /// <summary>
        /// Resets the "working" modeling data object for the current scenario, clearing the data for each model year.
        /// </summary>
        protected void ResetActiveScenarioData()
        {
            if (this._scen != null)
            {
                Array.Clear(this._scenData, 0, this._scenData.Length);
                if (this._scen.IsBaseline) { Array.Clear(this._baseData, 0, this._baseData.Length); }
            }
        }

        /// <summary>
        /// Returns the effects modeling data for the entire industry for the specified scenario and model year.
        /// </summary>
        /// <param name="scen">The scenario for which to obtain the effects modeling data.</param>
        /// <param name="year">The model year for which to obtain the effects modeling data.</param>
        /// <returns>The effects modeling data for the entire industry for the specified scenario and model year.</returns>
        public EffectsData GetEffectsData(Scenario scen, int year)
        {
            EffectsData[] ed;
            if      (null       == this._scen      ) { ed = null;              } // no active scenario
            else if (scen.Index == 0               ) { ed = this._baseEffects; } // requested scenario is baseline
            else if (scen.Index == this._scen.Index) { ed = this._scenEffects; } // requested scenario is current
            else                                     { ed = null;              } // TODO: obtain from cache
            //
            if (ed == null || year < this._minEffectsYear || year > this._maxEffectsYear) { return null; }
            return ed[year - this._minEffectsYear];
        }
        /// <summary>
        /// Sets the effects modeling data for the entire industry for the current scenario and the specified model year.
        /// </summary>
        /// <param name="year">The model year for which to set the effects modeling data.</param>
        /// <param name="data">The effects data to set.</param>
        protected void SetEffectsData(int year, EffectsData data)
        {
            if (this._scen != null)
            {
                int index = year - this._minEffectsYear;
                this._scenEffects[index] = data;
                if (this._scen.IsBaseline) { this._baseEffects[index] = data; }
            }
        }

        /// <summary>
        /// Updates the modeling <see cref="State"/> to the specified value and raises the necessary events.
        /// </summary>
        /// <param name="newState">The new state of the modeling process.</param>
        protected void UpdateState(ModelingState newState)
        {
            this._state = newState;
            if (this.StoppedOrCompleted)
            {   // set modeling stop time, if model was started
                if (this._startTime.Ticks != 0) { this._stopTime = DateTime.Now; }
                this._logWriter.Close();            // close log writer
                this._xlReportGenerator  = null;    // clear out XL report generator
                this._csvReportGenerator = null;    // clear out CSV report generator
            }

            // raise started, stopped, or completed events
            if      (this._state == ModelingState.Running  ) { this.OnModelingStarted();   }
            else if (this._state == ModelingState.Stopped  ) { this.OnModelingStopped();   }
            else if (this._state == ModelingState.Completed) { this.OnModelingCompleted(); }

            // raise ModelingChanged event as well
            this.OnModelingChanged();
        }

        /// <summary>
        /// Validates the modeling data and settings and returns the string containing validation errors.  If there were no
        /// validation errors, the return value will be an empty string.  Subclasses may override this method for customized
        /// validation.
        /// </summary>
        /// <returns>An error string containing any validation errors, or <see cref="string.Empty"/>, if no errors were encountered.</returns>
        /// <remarks>
        /// The <see cref="Validate"/> method provides automated validation of modeling data and settings, taking into account
        /// that some elements (e.g., tailpipe emission rates, see: <see cref="Volpe.Cafe.Model.RequiredModelInputs.Scenarios"/>)
        /// may not be necessary to run the compliance model.  In this case, the unncessary settings are ignored.<br></br><br></br>
        /// To minimize redundancy and avoid runtime errors, when overriding, sub-classes should still call
        /// "base.Validate(RequiredModelInputs)".
        /// </remarks>
        protected virtual string Validate()
        {
            string message = string.Empty;
            //
            if ((this.RequiredModelInputs & RequiredModelInputs.MarketData) == RequiredModelInputs.MarketData)
            {
                if (this._data == null)
                {
                    message += "\n    The modeling data was not specified.";
                }
                else if (this._data.ManufacturerCount == 0 || this._data.VehicleCount == 0)
                {
                    message += "\n    The specified modeling data does not contain any valid manufacturers or vehicles.";
                }
            }
            if (this._settings == null)
            {
                message += "\n    The modeling settings were not specified.";
            }
            else
            {
                string[] missingSets = new string[4];
                int missingCntr = 0;
                if ((this.RequiredModelInputs & RequiredModelInputs.Technologies) == RequiredModelInputs.Technologies &&
                    (this._settings.Technologies == null || this._settings.Technologies.TechnologyList == null ||
                     this._settings.Technologies.TechnologyList.Length == 0)) { missingSets[missingCntr++] = "Technologies"; }
                if ((this.RequiredModelInputs & RequiredModelInputs.Parameters) == RequiredModelInputs.Parameters &&
                    this._settings.Parameters == null) { missingSets[missingCntr++] = "Parameters"; }
                if ((this.RequiredModelInputs & RequiredModelInputs.Scenarios) == RequiredModelInputs.Scenarios &&
                    (this._settings.Scenarios == null || this._settings.Scenarios.Count == 0)) { missingSets[missingCntr++] = "Scenarios"; }

                if (missingCntr > 0)
                {
                    string messageSets = "";
                    for (int i = 0; i < missingCntr - 1; i++)
                    {
                        messageSets += missingSets[i] + ", ";
                    }
                    if (missingCntr == 1) { messageSets = missingSets[0]; }
                    else
                    {
                        if (missingCntr == 2) { messageSets = messageSets.Remove(messageSets.Length - 2, 2) + " "; }
                        messageSets += "and " + missingSets[missingCntr - 1];
                    }
                    message += "\n    The " + messageSets + " are missing from the modeling settings.";
                }
            }
            //
            if (message != string.Empty)
            {
                message = "The following errors have occured during initialization:\n" + message + "\n\nThe model will not run.";
            }
            return message;
        }

        /// <summary>
        /// Allows the extending classes to override certain user settigns that may be incompatible with specific implementations,
        /// before beginning any of the modeling.
        /// </summary>
        /// <param name="settings">The modeling settings to use for compliance modeling.</param>
        protected virtual void OverrideUserSettings(ModelingSettings settings) { }

        #endregion


        #region /*** Properties ***/

        #region /* ICompliance Members */

        /// <summary>Gets whether the <see cref="ICompliance"/> instance is valid to begin modeling.</summary>
        /// <remarks>Generally, the <see cref="ICompliance"/> will be invalid if its <see cref="Data"/> or <see cref="Settings"/>
        ///   are missing or not valid.</remarks>
        public bool IsValid { get { return this._valid; } }

        /// <summary>Gets the default model inputs that are required for modeling.</summary>
        public virtual RequiredModelInputs RequiredModelInputs { get { return RequiredModelInputs.All; } }

        /// <summary>Gets the product plan modeling data associated with this <see cref="ICompliance"/> instance, which is the
        ///   input data being used for compliance modeling.</summary>
        public Industry Data { get { return this._data; } }

        /// <summary>Gest an array of "working" or "final" modeling data objects for the baseline scenario, by model year.</summary>
        public Industry[] BaselineData { get { return this._baseData; } }
        /// <summary>Gest an array of "working" or "final" modeling data objects for the current scenario, by model year.  This
        ///   value should be reset before starting a new scenario.  At the end of the scenario, this value should contain the
        ///   final state of the industry for each model year.</summary>
        public Industry[] ScenarioData { get { return this._scenData; } }

        /// <summary>Gets the modeling settings associated with this <see cref="ICompliance"/> instance, which are the settings
        ///   being used for compliance modeling.</summary>
        public ModelingSettings Settings { get { return this._settings; } }

        /// <summary>Gets the <see cref="LogWriter"/> to use for writing log files generated during compliance modeling.</summary>
        public LogWriter LogWriter { get { return this._logWriter; } }
        /// <summary>Gets the <see cref="XlReportGenerator"/> to use for generating Excel reports for the compliance model.</summary>
        public XlReportGenerator XlReportGenerator { get { return this._xlReportGenerator; } }
        /// <summary>Gets the <see cref="CsvReportGenerator"/> to use for generating CSV reports for the compliance model.</summary>
        public CsvReportGenerator CsvReportGenerator { get { return this._csvReportGenerator; } }

        /// <summary>Gets the current state of the modeling process.</summary>
        public ModelingState State { get { return this._state; } }
        /// <summary>Gets the state of the modeling process, whether the compliance model is currently running.</summary>
        public bool Running { get { return (this._state == ModelingState.Running || this._state == ModelingState.StopRequested); } }
        /// <summary>Gets the state of the modeling process, whether the compliance model has been stopped.</summary>
        public bool Stopped { get { return (this._state == ModelingState.Stopped); } }
        /// <summary>Gets the state of the modeling process, whether the compliance model has completed.</summary>
        public bool Completed { get { return (this._state == ModelingState.Completed); } }
        /// <summary>Gets the state of the modeling process, whether the compliance model has stopped or completed.</summary>
        public bool StoppedOrCompleted { get { return (this._state == ModelingState.Stopped || this._state == ModelingState.Completed); } }

        /// <summary>Gets the current modeling progress, or null if compliance modeling has not started.</summary>
        public abstract IModelingProgress Progress { get; }
        /// <summary>Gets the timestamp of when the modeling process was started (or restarted).</summary>
        public DateTime StartTime { get { return this._startTime; } }
        /// <summary>Gets the timestamp of when the modeling process has stopped or completed.</summary>
        public DateTime StopTime { get { return (this.StoppedOrCompleted) ? this._stopTime : DateTime.MinValue; } }
        /// <summary>Gets the amount of time, in milliseconds, that the compliance model has been running thus far or since it
        ///   was last restarted.</summary>
        public long Runtime { get { return (this._startTime.Ticks == 0) ? 0 :
            (((this.StoppedOrCompleted) ? this._stopTime.Ticks : DateTime.Now.Ticks) - this._startTime.Ticks) / 10000; } }

        /// <summary>Gets an array of <see cref="ModelYear"/>s avaiable for modeling.  The 0-th index in the array represents the
        ///   first <see cref="ModelYear"/> available.</summary>
        public ModelYear[] ModelYears { get { return this._modelYears; } }

        #endregion

        /// <summary>Gets whether the caller has requested to abort the modeling process.</summary>
        protected bool AbortRequested { get { return this._abortRequested; } }

        /// <summary>Gets the total number of scenarios available for modeling.</summary>
        public int ScenCount { get { return this._scenCount; } }
        /// <summary>Gets the total number of model years available for modeling.</summary>
        public int YearCount { get { return this._yearCount; } }
        /// <summary>Gets the minimum year avaiable for modeling.</summary>
        public int MinYear { get { return this._minYear; } }
        /// <summary>Gets the maximum year avaiable for modeling.</summary>
        public int MaxYear { get { return this._maxYear; } }
        /// <summary>Gets the index of the minimum year, relative to <see cref="ModelYear.MinYear"/>, avaiable for modeling.</summary>
        public int MinYearIndex { get { return this._minYearIndex; } }
        /// <summary>Gets the index of the maximum year, relative to <see cref="ModelYear.MinYear"/>, avaiable for modeling.</summary>
        public int MaxYearIndex { get { return this._maxYearIndex; } }

        /// <summary>Gets the minimum year avaiable for effects modeling.</summary>
        public int MinEffectsYear { get { return this._minEffectsYear; } }
        /// <summary>Gets the maximum year avaiable for effects modeling.</summary>
        public int MaxEffectsYear { get { return this._maxEffectsYear; } }

        /// <summary>Gets the total number of vehicles contained within the industry.</summary>
        public int VehCount { get { return this._vehCount; } }
        /// <summary>Gets the total number of manufacturers contained within the industry.</summary>
        public int MfrCount { get { return this._mfrCount; } }
        /// <summary>Gets an array of the total number of vehicles produced by each manufacturer within the industry.</summary>
        public int[] MfrVehCount { get { return this._mfrVehCount; } }
        /// <summary>Gets an array of formatted manufacturer names (in "Title Case" notation) for each manufacturer contained
        ///   within the industry.</summary>
        public string[] MfrNames { get { return this._mfrNames; } }

        /// <summary>Gets the last error message generated by the compliance model.</summary>
        public string LastErrorMessage { get { return this._lassErrorMessage; } }

        /// <summary>Gets or sets the additional user specified notes associated with the compliance model.</summary>
        public string UserNotes { get; set; }
        /// <summary>Gets or sets the user specified keywords (or tags) associated with the compliance model.</summary>
        public string UserKeywords { get; set; }

        #endregion


        #region /*** Variables ***/

        // ----- property variables -----
        bool               _valid;
        LogWriter          _logWriter;
        XlReportGenerator  _xlReportGenerator;
        CsvReportGenerator _csvReportGenerator;
        Industry           _data;
        ModelingSettings   _settings;
        ModelingState      _state;
        DateTime           _startTime;
        DateTime           _stopTime;
        string             _lassErrorMessage;   // if an error was generated during modeling

        // ----- support & runtime variables -----
        [NonSerialized] Thread _complianceThread;
        [NonSerialized] bool   _abortRequested;

        //Industry[][] _modelData;    // model data indexed as:  (scenario)(model year)
        Industry [] _baseData;      // model data indexed by model year
        Industry [] _scenData;      // model data indexed by model year
        ModelYear[] _modelYears;    // array of years avaiable for modeling; 0-th index in the array represents the first year available

        EffectsData[] _baseEffects; // effects data indexed by model year
        EffectsData[] _scenEffects; // effects data indexed by model year

        int      _scenCount;
        int      _yearCount;
        int      _vehCount;
        int      _mfrCount;
        int   [] _mfrVehCount;
        string[] _mfrNames;
        int      _minYear;
        int      _maxYear;
        int      _minYearIndex;
        int      _maxYearIndex;

        int      _minEffectsYear;
        int      _maxEffectsYear;

        // ----- progress variables -----
        /// <summary>Specifies the scenario currently being analyzed, or null, if the scenario loop hasn't started yet.</summary>
        protected Scenario _scen;
        /// <summary>Specifies the model year currently being analyzed, or null, if the model year loop hasn't started yet.</summary>
        protected ModelYear _year;
        /// <summary>Specifies the effects year currently being analyzed, or -1, if the model year loop hasn't started yet.</summary>
        protected int _effectsYear;
        /// <summary>Specifies the manufacturer currently being analyzed, or null, if the manufacturer loop hasn't started yet.</summary>
        protected Manufacturer _mfr;

        #endregion

    }
}
