﻿#region << Using Directives >>
using System;
using System.Collections.Generic;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Generic;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides methods to support the Dynamic Fleet Share mode for the compliance model.
    /// </summary>
    [Serializable]
    public class DynamicFleetShare
    {

        #region /*** Ctors ***/

        ///// <summary>
        ///// Initializes a new instance of the <see cref="DynamicFleetShare"/> class.
        ///// </summary>
        //public DynamicFleetShare()
        //{
        //}

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Adjusts the PC/LT fleet share in the specified model data, scenario, and model year, using the provided modeling
        /// settings and previous year's model data.
        /// </summary>
        /// <param name="data">The current year's model data whose PC/LT fleet shares to adjust.</param>
        /// <param name="prevData">The previous year's model data based on which to adjust the PC/LT fleet shares.  If the first
        ///   model year is being analyzed, prevData should be null.</param>
        /// <param name="settings">The modeling settings used for modeling.</param>
        /// <param name="scen">The current scenario being analyzed.</param>
        /// <param name="year">The current model year being analyzed.</param>
        public static void AdjustFleetShare(Industry data, Industry prevData, ModelingSettings settings, Scenario scen, ModelYear year)
        {
            //--- compute new PC share ---
            DFSModelValues dfsValues = settings.Parameters.DFSModelValues;
            double pcShare;
            VCValue<double> cafe;
            if (year.Year == data.MinYear.Year)
            {   // for the first model year, use input data
                pcShare = dfsValues.InitialLDVShare;
                cafe    = dfsValues.InitialFE;
            }
            else
            {   // for all model years after the first year, use previous year's data
                GetPreviousShare(prevData, scen, year, out pcShare, out cafe);
            }
            double gasPrice   = settings.Parameters.FuelPrices.GetFuelPrice(settings.OperatingModes.FuelPriceEstimates, FuelType.Gasoline, year.Year);
            double epsilon    = dfsValues.Epsilon;
            double newPCShare = 0.259694 +
                                0.222462 * pcShare +
                                0.005969 * cafe[VehicleClass.LDV   ] -
                                0.004116 * cafe[VehicleClass.LDT12a] +
                                0.018314 * gasPrice + epsilon;

            //--- adjust PC/LT fleet shares ---
            List<Manufacturer> mfrs = data.Manufacturers;
            VCValue<double> sales = new VCValue<double>();
            //
            for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
            {   // aggregate current PC, LT, and total sales
                List<Vehicle> vehs = mfrs[i].Vehicles;
                for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                {
                    VehicleClass vehClass = vehs[j].VehicleClass;
                    double vehSales = vehs[j].Description.Sales[year.Index];
                    //
                    sales[vehClass] += vehSales;
                }
            }
            // get new PC/LT sales based on computed new PC share
            VCValue<double> newSales = new VCValue<double>(
                (sales[VehicleClass.LDT12a] + sales[VehicleClass.LDV]) *      newPCShare ,
                (sales[VehicleClass.LDT12a] + sales[VehicleClass.LDV]) * (1 - newPCShare),
                 sales[VehicleClass.LDT2b3]);
            // adjust each veh's sales
            for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
            {
                List<Vehicle> vehs = mfrs[i].Vehicles;
                for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                {
                    VehicleClass vehClass = vehs[j].VehicleClass;
                    if (vehClass == VehicleClass.LDV || vehClass == VehicleClass.LDT1 || vehClass == VehicleClass.LDT2a)
                    {
                        double ratio = (sales[vehClass] == 0) ? 1 : newSales[vehClass] / sales[vehClass];
                        vehs[j].Description.Sales[year.Index] *= ratio;
                    }
                }
            } // next i (mfr)
        }
        static void GetPreviousShare(Industry prevData, Scenario scen, ModelYear year, out double pcShare, out VCValue<double> cafe)
        {
            VCValue<double> sales = new VCValue<double>();
            cafe = new VCValue<double>();
            //
            List<Manufacturer> mfrs = prevData.Manufacturers;
            for (int i = 0, mfrCount = mfrs.Count; i < mfrCount; i++)
            {
                List<Vehicle> vehs = mfrs[i].Vehicles;
                for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                {
                    Vehicle.CDescription  vd = vehs[j].Description;
                    RegulatoryClass regClass = vehs[j].RegClass;
                    VehicleClass    vehClass = vehs[j].VehicleClass;
                    double          vehSales = vehs[j].Description.Sales[year.Index];
                    double             vehFE = Standards.GetAverageFuelEconomy(vd.FuelEconomy, vd.FuelShare, regClass, scen, year);
                    //
                    sales[vehClass] += vehSales;
                    cafe [vehClass] += vehSales / vehFE;
                } // next j (veh)
            } // next i (mfr)
            //
            // calc new PC/LT share and finalize calculation of average FE
            pcShare = (sales[VehicleClass.LDV] == 0) ? 0 : sales[VehicleClass.LDV] / (sales[VehicleClass.LDV] + sales[VehicleClass.LDT12a]);
            foreach (VehicleClass vehClass in VCValue<object>.Classes)
            {
                cafe[vehClass] = (cafe[vehClass] == 0) ? 0 : sales[vehClass] / cafe[vehClass];
            }
        }

        #endregion

    }
}
