using System;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Generic;

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides a default implementation of the <see cref="IModelingProgress"/> interface.
    /// </summary>
    /// <seealso cref="IModelingProgress"/>
    [Serializable]
    public class ModelingProgress : IModelingProgress
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="ModelingProgress"/> class using the specified parameters.
        /// </summary>
        /// <param name="scen">The scenario currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="year">The model year currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="mfr">The manufacturer currently being analyzed by the modeling system.  This value may be null.</param>
        public ModelingProgress(Scenario scen, ModelYear year, Manufacturer mfr) : this(scen, year, mfr, null) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="ModelingProgress"/> class using the specified parameters, as well as the
        /// additional modeling progress information.
        /// </summary>
        /// <param name="scen">The scenario currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="year">The model year currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="mfr">The manufacturer currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="additionalInfo">Additional modeling progress information.  This value may be null.</param>
        public ModelingProgress(Scenario scen, ModelYear year, Manufacturer mfr, object additionalInfo) : this(null, scen, year, mfr, additionalInfo) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="ModelingProgress"/> class using the specified parameters.
        /// </summary>
        /// <param name="progressString">The progress string to display in the UI.  This value may be null.</param>
        public ModelingProgress(string progressString) : this(progressString, null) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="ModelingProgress"/> class using the specified parameters.
        /// </summary>
        /// <param name="progressString">The progress string to display in the UI.  This value may be null.</param>
        /// <param name="additionalInfo">Additional modeling progress information.  This value may be null.</param>
        public ModelingProgress(string progressString, object additionalInfo) : this(progressString, null, null, null, additionalInfo) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="ModelingProgress"/> class using the specified parameters.
        /// </summary>
        /// <param name="progressString">The progress string to display in the UI.  This value may be null.</param>
        /// <param name="scen">The scenario currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="year">The model year currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="mfr">The manufacturer currently being analyzed by the modeling system.  This value may be null.</param>
        /// <param name="additionalInfo">Additional modeling progress information.  This value may be null.</param>
        protected ModelingProgress(string progressString, Scenario scen, ModelYear year, Manufacturer mfr, object additionalInfo)
        {
            this._progressString = progressString;
            this._scen           = scen;
            this._year           = year;
            this._mfr            = mfr;
            this._additionalInfo = additionalInfo;
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Returns a string that represents the current <see cref="ModelingProgress"/> instance.
        /// </summary>
        /// <returns>A string that represents the current <see cref="ModelingProgress"/> instance.</returns>
        public override string ToString()
        {
            if (this._progressString != null)
            {
                return this._progressString;
            }
            else if (this._scen == null)
            {
                return "No progress information available.";
            }
            else
            {   // build the mfr progress string
                string mfrProgress = null;
                if (this._mfr != null)
                {
                    Manufacturer.CModelData mmd = this._mfr.ModelData;
                    mfrProgress = "\n    Standard: "   + mmd.Standard.ToString(4) +
                                  "\n    CAFE: "       + mmd.CAFE    .ToString(4) +
                                  "\n    Credits: "    + mmd.Credits .ToString(0) +
                                  "\n    Fines: "      + mmd.Fines   .ToString(0) +
                                  "\n    Tech Costs: " + mmd.TechCost.ToString(0);
                }
                // add base progress and return
                return
                    "Scenario: "       + ((this._scen == null) ? "N/A" : this._scen.ToString()) +
                    "\nModel Year: "   + ((this._year == null) ? "N/A" : this._year.ToString()) +
                    "\nManufacturer: " + ((this._mfr  == null) ? "N/A" : this._mfr .Description.Name) + ((mfrProgress == null) ? "" : mfrProgress);
            }
        }

        #endregion


        #region /*** Properties ***/

        #region /* IModelingProgress Members */

        /// <summary>Gets the progress string to display in the UI.  This value may be null.</summary>
        public virtual string ProgressString { get { return this._progressString; } }
        /// <summary>Gets the scenario currently being analyzed by the modeling system.  This value may be null.</summary>
        public virtual Scenario Scenario { get { return this._scen; } }
        /// <summary>Gets the model year currently being analyzed by the modeling system.  This value may be null.</summary>
        public virtual ModelYear ModelYear { get { return this._year; } }
        /// <summary>Gets the manufacturer currently being analyzed by the modeling system.  This value may be null.</summary>
        public virtual Manufacturer Manufacturer { get { return this._mfr; } }
        /// <summary>Gets or sets additional modeling progress information associated with the <see cref="IModelingProgress"/>
        ///   implementation.  This value may be null.</summary>
        public virtual object AdditionalInfo { get { return this._additionalInfo; } set { this._additionalInfo = value; } }

        #endregion

        #endregion


        #region /*** Variables ***/

        /// <summary>Specifies the progress string to display in the UI.  This value may be null.</summary>
        protected string _progressString;
        /// <summary>Specifies the scenario currently being analyzed by the modeling system.  This value may be null.</summary>
        protected Scenario _scen;
        /// <summary>Specifies the model year currently being analyzed by the modeling system.  This value may be null.</summary>
        protected ModelYear _year;
        /// <summary>Specifies the manufacturer currently being analyzed by the modeling system.  This value may be null.</summary>
        protected Manufacturer _mfr;
        /// <summary>Represents additional modeling progress information associated with the <see cref="IModelingProgress"/>
        ///   implementation.  This value may be null.</summary>
        protected object _additionalInfo;

        #endregion

    }
}
