using System;
using Volpe.Cafe.IO;

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Provides input configuration settings for the modeling system.
    /// </summary>
    [Serializable]
    public class InputSettings : ISettings, ICloneable
    {

        #region /*** Methods ***/

        #region /* ICloneable/ISettings Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="InputSettings"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="InputSettings"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="InputSettings"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="InputSettings"/>.</returns>
        ISettings ISettings.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="InputSettings"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="InputSettings"/>.</returns>
        InputSettings Clone()
        {
            InputSettings value = new InputSettings();
            Clone(this, value);
            return value;
        }
        static void Clone(InputSettings src, InputSettings dst)
        {
            dst.EncryptInput               = src.EncryptInput;
            dst.InputPassword              = src.InputPassword;
            //
            dst.MarketDataFile             = src.MarketDataFile;
            dst.TechnologiesFile           = src.TechnologiesFile;
            dst.ParametersFile             = src.ParametersFile;
            dst.ScenariosFile              = src.ScenariosFile;
            //
            dst.MarketDataTimestamp        = src.MarketDataTimestamp;
            dst.TechnologiesTimestamp      = src.TechnologiesTimestamp;
            dst.ParametersTimestamp        = src.ParametersTimestamp;
            dst.ScenariosTimestamp         = src.ScenariosTimestamp;
        }

        #endregion

        #region /* ISettings Members */

        /// <summary>
        /// Sets the initial state for the <see cref="ISettings"/> object to the default values.
        /// </summary>
        void ISettings.SetDefaultSettings()
        {
            this.EncryptInput          = false;
            this.InputPassword         = string.Empty;
            //
            this.MarketDataFile        = string.Empty;
            this.TechnologiesFile      = string.Empty;
            this.ParametersFile        = string.Empty;
            this.ScenariosFile         = string.Empty;
        }

        /// <summary>
        /// Updates this <see cref="InputSettings"/> object with values from the specified new settings.
        /// </summary>
        /// <param name="newSettings">An <see cref="InputSettings"/> object containing new runtime settings for this instance.</param>
        void ISettings.UpdateSettings(ISettings newSettings)
        {
            this.UpdateSettings((InputSettings)newSettings);
        }
        /// <summary>
        /// Updates this <see cref="InputSettings"/> object with values from the specified new settings.
        /// </summary>
        /// <param name="newSettings">An <see cref="InputSettings"/> object containing new runtime settings for this instance.</param>
        public void UpdateSettings(InputSettings newSettings)
        {
            Clone(newSettings, this);
        }

        /// <summary>
        /// Writes settings data to the summary.txt log file.
        /// </summary>
        void ISettings.WriteSummaryLog(LogWriter writer)
        {
            string format = "M/dd/yy H:mm:ss";
            if (writer.Summary == null) { return; }
            string outStr = "--- Input Settings ---" +
                "\n  EncryptInput="          + this.EncryptInput +
                "\n    InputPassword="       + this.MaskInputPassword('*') +
                "\n  MarketDataFile="        + this.MarketDataFile         + " (" + this.MarketDataTimestamp       .ToString(format) + ")" +
                "\n  TechnologiesFile="      + this.TechnologiesFile       + " (" + this.TechnologiesTimestamp     .ToString(format) + ")" +
                "\n  ParametersFile="        + this.ParametersFile         + " (" + this.ParametersTimestamp       .ToString(format) + ")" +
                "\n  ScenariosFile="         + this.ScenariosFile          + " (" + this.ScenariosTimestamp        .ToString(format) + ")";
            writer.Summary.WriteLine(outStr + "\n");
        }
        string MaskInputPassword(char pwMask)
        {
            return (this.InputPassword == null || this.InputPassword == string.Empty) ? "" : new string(pwMask, 8);
        }

        #endregion

        #endregion


        #region /*** Variables ***/

        /// <summary>Specifies whether the input data files are password protected.</summary>
        public bool EncryptInput;
        /// <summary>Specifies the password to use when reading password protected files.</summary>
        public string InputPassword;

        /// <summary>Specifies the location of the market data file.</summary>
        public string MarketDataFile;
        /// <summary>Specifies the location of the technologies file.</summary>
        public string TechnologiesFile;
        /// <summary>Specifies the location of the parameters file.</summary>
        public string ParametersFile;
        /// <summary>Specifies the location of the scenarios file.</summary>
        public string ScenariosFile;

        /// <summary>Specifies when the market data file was last modified, at the time it was loaded by the model.</summary>
        public DateTime MarketDataTimestamp;
        /// <summary>Specifies when the technologies file was last modified, at the time it was loaded by the model.</summary>
        public DateTime TechnologiesTimestamp;
        /// <summary>Specifies when the parameters file was last modified, at the time it was loaded by the model.</summary>
        public DateTime ParametersTimestamp;
        /// <summary>Specifies when the scenarios file was last modified, at the time it was loaded by the model.</summary>
        public DateTime ScenariosTimestamp;

        #endregion

    }
}
