﻿#region << Using Directives >>
using System;
using System.Globalization;
#endregion

namespace Volpe.Cafe.Utils
{
    /// <summary>
    /// Provides various helper methods, such as:
    /// methods for parsing and converting objects from one type to another (such as object to string, object to double),
    /// utility methods for swapping two objects (int, double, etc.),
    /// string manipulation methods for case-insensitive string comparison and conversion to "Title Case",
    /// methods for array cloning and resizing,
    /// as well as other utility methods.
    /// </summary>
    public class Interaction
    {

        #region /*** Methods ***/

        #region /* 'Swap' methods */

        /// <summary>
        /// Swaps the two <see cref="Int32"/> values.
        /// </summary>
        /// <param name="x">A reference to an <see cref="Int32"/> whose value to swap.</param>
        /// <param name="y">A reference to an <see cref="Int32"/> whose value to swap.</param>
        public static void Swap(ref int x, ref int y)
        {
            if (x != y) { int tmp = x; x = y; y = tmp; }
        }
        /// <summary>
        /// Swaps the two <see cref="Double"/> values.
        /// </summary>
        /// <param name="x">A reference to a <see cref="Double"/> whose value to swap.</param>
        /// <param name="y">A reference to a <see cref="Double"/> whose value to swap.</param>
        public static void Swap(ref double x, ref double y)
        {
            if (x != y) { double tmp = x; x = y; y = tmp; }
        }
        /// <summary>
        /// Swaps the two objects.
        /// </summary>
        /// <param name="x">A reference to an <see cref="Object"/> whose value to swap.</param>
        /// <param name="y">A reference to an <see cref="Object"/> whose value to swap.</param>
        public static void Swap(ref object x, ref object y)
        {
            if (!x.Equals(y)) { object tmp = x; x = y; y = tmp; }
        }

        #endregion

        #region /* Methods for parsing data (such as string, int, double, etc.) from objects */

        /// <summary>
        /// Determines whether the specified object is a numeric value.
        /// </summary>
        /// <param name="value">The value to check.</param>
        /// <returns>true, if the specified object is a numeric value; false, otherwise.</returns>
        public static bool IsNumeric(object value)
        {
            double result;
            return double.TryParse(GetString(value), NumberStyles.Any, NumberFormatInfo.InvariantInfo, out result);
        }
        /// <summary>
        /// Determines whether the specified object is a 32-bit integer value.
        /// </summary>
        /// <param name="value">The value to check.</param>
        /// <returns>true, if the specified object is a 32-bit integer value; false, otherwise.</returns>
        public static bool IsInteger(object value)
        {
            int result;
            return int.TryParse(GetString(value), NumberStyles.Any, NumberFormatInfo.InvariantInfo, out result);
        }

        /// <summary>
        /// Converts the specified value to its boolean equivalent.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The boolean equivalent of the specified value.</returns>
        public static bool GetBool(object value)
        {
            return GetString(value).Equals("TRUE");
        }
        /// <summary>
        /// Converts the specified value to its Byte equivalent.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The Byte equivalent of the specified value.</returns>
        public static byte GetByte(object value)
        {
            return (byte)GetDouble(value, false);
        }
        /// <summary>
        /// Converts the specified value to its Int16 equivalent.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The Int16 equivalent of the specified value.</returns>
        public static short GetInt16(object value)
        {
            return (short)GetDouble(value, false);
        }
        /// <summary>
        /// Converts the specified value to its Int32 equivalent.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The Int32 equivalent of the specified value.</returns>
        public static int GetInt32(object value)
        {
            return (int)GetDouble(value, false);
        }
        /// <summary>
        /// Converts the specified value to its Int64 equivalent.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The Int64 equivalent of the specified value.</returns>
        public static long GetInt64(object value)
        {
            return (long)GetDouble(value, false);
        }
        /// <summary>
        /// Converts the specified value to its double equivalent.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The double equivalent of the specified value.</returns>
        public static double GetDouble(object value)
        {
            return GetDouble(value, true);
        }
        static double GetDouble(object value, bool checkNaN)
        {
            string valueStr = GetString(value);
            if (checkNaN && valueStr == "NAN") { return double.NaN; }
            //
            double result = 0;
            double.TryParse(valueStr, NumberStyles.Any, NumberFormatInfo.InvariantInfo, out result);
            return result;
        }
        /// <summary>
        /// Converts the specified value to its char equivalent.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The string equivalent of the specified value.</returns>
        public static char GetChar(object value)
        {
            string str = GetString(value);
            return str.Equals(string.Empty) ? '\0' : str[0];
        }
        /// <summary>
        /// Converts the specified value to its upper-case string equivalent, with all whitespaces at the beginning and
        /// end of the object trimmed.
        /// </summary>
        /// <param name="value">The value to convert.</param>
        /// <returns>The string equivalent of the specified value.</returns>
        public static string GetString(object value)
        {
            return (value == null) ? string.Empty : value.ToString().Trim().ToUpper();
        }

        #endregion

        #region /* Array manipulation -- cloning/resizing */

        /// <summary>
        /// Returns a shallow copy of the specified array.  If the array is null, the return value will be null;
        /// otherwise, the return value is the copy of the specified array.
        /// </summary>
        /// <param name="value">The array, whose elements to copy.</param>
        /// <param name="elementType">The <see cref="Type"/> of each element in the new array.</param>
        /// <returns>A shallow copy of the specified array.</returns>
        public static Array CloneArray(Array value, Type elementType)
        {
            if (value == null) { return null; }
            else
            {
                Array destination;
                if (value.Rank == 1) { destination = Array.CreateInstance(elementType, value.Length); }
                else
                {   // multi-dimensional array
                    int[] lens = new int[value.Rank];
                    for (int i = 0, lenCount = value.Rank; i < lenCount; i++)
                    {
                        lens[i] = value.GetLength(i);
                    }
                    destination = Array.CreateInstance(elementType, lens);
                }
                Array.Copy(value, destination, value.Length);
                return destination;
            }
        }

        /// <summary>
        /// Resizes the specified array, optionally preserving its contents.
        /// </summary>
        /// <param name="arr">The array whose elements to resize.  This value cannot be null.</param>
        /// <param name="type">The <see cref="Type"/> of elements contained in the array.</param>
        /// <param name="newLength">The new length of the array.  This value must be >= 0.</param>
        /// <param name="preserve">true, to preserve the elements in the array during resize; false, otherwise.</param>
        /// <returns>An array of the specified <see cref="Type"/> and new length, with elements from arr optionally preserved.</returns>
        public static Array ResizeArray(Array arr, Type type, int newLength, bool preserve)
        {
            Array newArr = Array.CreateInstance(type, newLength);
            if (preserve)
            {   // copy elements
                int arrLen = arr.Length;
                if (arrLen > newLength) { arrLen = newLength; }
                Array.Copy(arr, newArr, arrLen);
            }
            return newArr;
        }

        #endregion

        #region /* Methods for case-insensitive string comparison */

        /// <summary>
        /// Compares the two values using straight forward or fuzzy logic.  The comparison is always case-insensitive, even if
        /// fuzzy logic is not used.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <param name="fuzzy">Specifies whether to use fuzzy logic (removes spaces, punctuations, and all special chars).</param>
        /// <returns>true, if the two values are equal; false, otherwise.</returns>
        /// <remarks>
        /// Using fuzzy logic significantly decreases the performance of comparison, therefore, its use is generally not
        /// recommended.
        /// </remarks>
        public static bool StringCompare(string value1, string value2, bool fuzzy)
        {
            if (!fuzzy)
            {   // perform initial "quick checks" if not using fuzzy logic
                if (value1.Length != value2.Length) { return false; }
                if (string.Equals(value1, value2)) { return true; }
            }
            //
            // examine the strings closer ...
            //
            // if fuzzy logic, remove spaces, new-lines, tabs, punctuations, and special characters
            if (fuzzy) { StringPrepareFuzzy(ref value1, ref value2); }
            //
            // compare modified strings
            return (string.Compare(value1, value2, true) == 0);
        }
        /// <summary>
        /// Compares the specified value with the provided array of comparands using simple or fuzzy logic.  The comparison is
        /// always case-insensitive, even if fuzzy logic is not used.
        /// </summary>
        /// <param name="value">The value to compare.</param>
        /// <param name="comparands">An array of comparands to compare against.</param>
        /// <param name="fuzzy">Specifies whether to use fuzzy logic (removes spaces, punctuations, and all special chars).</param>
        /// <returns>true, if the value matches any of the comparands; false, otherwise.</returns>
        /// <remarks>
        /// Using fuzzy logic significantly decreases the performance of comparison, therefore, its use is generally not
        /// recommended.
        /// </remarks>
        public static bool StringCompareAny(string value, string[] comparands, bool fuzzy)
        {
            int index;
            return StringCompareAny(value, comparands, fuzzy, out index);
        }
        /// <summary>
        /// Compares the specified value with the provided array of comparands using simple or fuzzy logic.  The comparison is
        /// always case-insensitive, even if fuzzy logic is not used.
        /// </summary>
        /// <param name="value">The value to compare.</param>
        /// <param name="comparands">An array of comparands to compare against.</param>
        /// <param name="fuzzy">Specifies whether to use fuzzy logic (removes spaces, punctuations, and all special chars).</param>
        /// <param name="index">When this method returns, contains the index of the first element in comparands that match the
        ///   specified value, or -1, if no match was found.</param>
        /// <returns>true, if the value matches any of the comparands; false, otherwise.</returns>
        /// <remarks>
        /// Using fuzzy logic significantly decreases the performance of comparison, therefore, its use is generally not
        /// recommended.
        /// </remarks>
        public static bool StringCompareAny(string value, string[] comparands, bool fuzzy, out int index)
        {
            index = -1;
            // return false if there are no comparands
            if (comparands == null || comparands.Length == 0) { return false; }
            // examine each comparand
            for (int i = 0; i < comparands.Length; i++)
            {   // if at least one matches, return true
                if (StringCompare(value, comparands[i], fuzzy)) { index = i; return true; }
            }
            // the value didn't match any of the comparands
            return false;
        }

        /// <summary>
        /// Checks whether the specified substring is part of the given value.  The comparison is always case-insensitive, even
        /// if fuzzy logic is not used.
        /// </summary>
        /// <param name="value">The string value to examine.</param>
        /// <param name="substring">The substring to look for in value.</param>
        /// <param name="fuzzy">Specifies whether to use fuzzy logic (removes spaces, punctuations, and all special chars).</param>
        /// <returns>true, if substring is contained in value; false, otherwise.</returns>
        /// <remarks>
        /// Using fuzzy logic significantly decreases the performance of comparison, therefore, its use is generally not
        /// recommended.
        /// </remarks>
        public static bool StringContains(string value, string substring, bool fuzzy)
        {
            // if fuzzy logic, remove spaces, new-lines, tabs, punctuations, and special characters
            if (fuzzy) { StringPrepareFuzzy(ref value, ref substring); }
            //
            // look for substring, first using "quick checks"
            if (value.Length < substring.Length) { return false; }  // substring is longer than actual -- thus, false
            if (string.Equals(value, substring)) { return true; }  // equality test is extremely fast; if two are equal, substring is guaranted true
            // convert to upper case and look for substring in value
            value = value.ToUpper();
            substring = substring.ToUpper();
            return (value.IndexOf(substring) != -1);
        }

        /// <summary>
        /// Checks whether the specified value starts with the given substring.  The comparison is always case-insensitive, even
        /// if fuzzy logic is not used.
        /// </summary>
        /// <param name="value">The string value to examine.</param>
        /// <param name="substring">The substring to look for in value.</param>
        /// <param name="fuzzy">Specifies whether to use fuzzy logic (removes spaces, punctuations, and all special chars).</param>
        /// <returns>true, if substring is contained in value; false, otherwise.</returns>
        /// <remarks>
        /// Using fuzzy logic significantly decreases the performance of comparison, therefore, its use is generally not
        /// recommended.
        /// </remarks>
        public static bool StringStartsWith(string value, string substring, bool fuzzy)
        {
            // if fuzzy logic, remove spaces, new-lines, tabs, punctuations, and special characters
            if (fuzzy) { StringPrepareFuzzy(ref value, ref substring); }
            //
            // look for substring, first using "quick checks"
            if (value.Length < substring.Length) { return false; }  // substring is longer than actual -- thus, false
            if (string.Equals(value, substring)) { return true ; }  // equality test is extremely fast; if two are equal, value is guaranted to start with substring
            // convert to upper case and check if value starts with substring
            value = value.ToUpper();
            substring = substring.ToUpper();
            return value.StartsWith(substring);
        }

        /// <summary>
        /// Prepares the specified values for fuzzy logic comparison (removes spaces, punctuations, and all special chars).
        /// </summary>
        static void StringPrepareFuzzy(ref string value1, ref string value2)
        {
            // remove spaces, new-lines, tabs, punctuations, and special characters
            value1 = StringRemoveWhitespacePunctuationAndSpecialChars(value1);
            value2 = StringRemoveWhitespacePunctuationAndSpecialChars(value2);
        }

        /// <summary>
        /// Removes all special special characters, which cannot be handled by the windows file systems, from the
        /// specified string and replaces them with the given character.
        /// </summary>
        /// <param name="value">The string from which to remove special characters.</param>
        /// <param name="c">The character with which to replace the special characters.</param>
        /// <returns>The modified string.</returns>
        public static string RemoveSpecialChars(string value, char c)
        {
            int spCharCount = SpecialChars.Length;
            for (int i = 0; i < spCharCount; i++)
            {
                if (value.IndexOf(SpecialChars[i]) != -1)
                {
                    value = value.Replace(SpecialChars[i], c);
                }
            }
            return value;
        }

        #endregion

        #region /* Methods for conversion to "Title Case" */

        /// <summary>
        /// Converts the specified string to Title Case.
        /// </summary>
        /// <param name="value">The string to convert to Title Case.</param>
        /// <returns>The specified string converted to Title Case.</returns>
        public static string GetTitleCase(string value)
        {
            char[] chars = value.ToCharArray();
            for (int i = 0, count = value.Length; i < count; i++)
            {
                bool isTitleChar = (i == 0 || ((chars[i - 1] < 48 || chars[i - 1] > 57) &&
                    (chars[i - 1] < 65 || chars[i - 1] > 90) && (chars[i - 1] < 97 || chars[i - 1] > 122)));
                int c = chars[i];
                if (isTitleChar) { if (c >= 97 && c <= 122) { chars[i] = (char)(c - 32); } }
                else if (c >= 65 && c <= 90) { chars[i] = (char)(c + 32); }
            }
            return new string(chars);
        }
        /// <summary>
        /// Converts the specified string to Title Case.
        /// </summary>
        /// <param name="value">The string to convert to Title Case.</param>
        /// <param name="minLength">The minimum length the string needs to be before being converted.</param>
        /// <returns>The specified string converted to Title Case.</returns>
        public static string GetTitleCase(string value, int minLength)
        {
            return (value.Length >= minLength) ? GetTitleCase(value) : value;
        }

        static bool IsLetterOrNumber(char c)
        {
            return (c >= 48 && c <= 57) || (c >= 65 && c <= 90) || (c >= 97 && c <= 122);
        }
        static bool IsNotLetterOrNumber(char c)
        {
            return (c < 48 || c > 57) && (c < 65 || c > 90) && (c < 97 || c > 122);
        }

        #endregion

        #region /* Methods for converting time-span to string:  "DAY.HH:MM:SS" */

        /// <summary>
        /// Converts the given value, specified in milliseconds, to an equivalent time string in the following format:
        /// "[D.]HH:MM:SS".
        /// </summary>
        /// <param name="value">The value, specified in milliseconds, to convert.</param>
        /// <returns>A string representing the time equivalent to the provided value.</returns>
        public static string GetTimeString(long value)
        {
            long sec = value / 1000;
            long s = sec % 60,
                 m = sec / 60 % 60,
                 h = sec / 3600 % 24,
                 d = sec / 86400;

            return ((d != 0) ? d.ToString() + "." : "") + h.ToString("00:") + m.ToString("00:") + s.ToString("00");
        }
        /// <summary>
        /// Converts the given <see cref="TimeSpan"/> value to an equivalent time string in the following format:
        /// "[D.]HH:MM:SS".
        /// </summary>
        /// <param name="value">The <see cref="TimeSpan"/> value to convert.</param>
        /// <returns>A string representing the time equivalent to the provided value.</returns>
        public static string GetTimeString(TimeSpan value)
        {
            return GetTimeString(value.Ticks / 10000);
        }

        #endregion

        #region /* Internal Helper Methods */

        /// <summary>
        /// Removes the following characters the specified string:  " " (space), "\t", "\r", "\n", "-", "_", ".", ",", "!", "?",
        /// "@", "#", "$", "%", "^", "*", "(", ")", "{", "}", "[", "]", "'", "\"", "/", "\\", "|", and the 'Ampersand' character.
        /// </summary>
        /// <param name="value">The value to modify.</param>
        /// <returns>value, with whitespace, punctuation, and special characters removed.</returns>
        public static string StringRemoveWhitespacePunctuationAndSpecialChars(string value)
        {
            return StringRemove(StringRemove(value, BaseRemove), ExtRemove);
        }
        static string StringRemove(string value, string[] chars)
        {
            for (int i = 0; i < chars.Length; i++)
            {
                value = value.Replace(chars[i], "");
            }
            return value;
        }

        #endregion

        #endregion


        #region /*** Variables ***/

        /// <summary>Specifies the number of calendar years to consider for effects calculations.  This value is hard-coded at 40.</summary>
        public const int CY = 40;


        /// <summary>Contains the following characters: " ", "\t", "\r", "\n", "-", "_".</summary>
        static readonly string[] BaseRemove = new string[] { " ", "\t", "\r", "\n", "-", "_" };
        /// <summary>Contains the following characters: ".", ",", "!", "?", "@", "#", "$", "%", "^", "*", "(", ")", "{", "}",
        ///   "[", "]", "'", "\"", "/", "\\", "|", and the 'Ampersand' character.</summary>
        static readonly string[] ExtRemove = new string[] {".", ",", "!", "?", "@", "#", "$", "%", "^", "&", "*", "(", ")",
            "{", "}", "[", "]", "'", "\"", "/", "\\", "|" };
        /// <summary>Represents the special characters which cannot be handled by the windows file systems.</summary>
        static readonly char[] SpecialChars = new char[] { '/', '\\', ':', '*', '?', '"', '<', '>', '|' };

        #endregion

    }
}
