﻿#region << Using Directives >>
using System;
using System.Collections.Generic;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Data;
#endregion

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Aggregates information relating to vehicle technologies and adjustment factors for all technology classes.
    /// </summary>
    [Serializable]
    public sealed class Technologies
    {

        #region /*** Ctors ***/

        // Private ctor
        Technologies() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="Technologies"/> class.
        /// </summary>
        internal Technologies(
            TechnologyInfo[] techList, Dictionary<ulong, double[]> fcAdjFactors, Dictionary<ulong, double[]> costAdjFactors)
        {
            this.TechnologyList = techList;
            this.FCAdjFactors   = fcAdjFactors;
            this.CostAdjFactors = costAdjFactors;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Retrieves the primary fuel consumption improvement for a technology with the specified index, applicable to the given
        /// vehicle.
        /// </summary>
        /// <returns>The primary fuel consumption improvement of a technology.</returns>
        public double GetFC(int techIndex, Vehicle veh)
        {
            TechnologyAttributes attr = this.TechnologyList[techIndex].GetAttributes(veh);
            return (attr.Applicable) ? attr.FC : 0;
        }

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Technologies"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Technologies"/>.</returns>
        internal Technologies Clone()
        {
            Technologies techs = new Technologies();
            //
            techs.TechnologyList = this.TechnologyList.CloneArray(x => x.Clone());
            techs.FCAdjFactors   = this.CloneDictionary(this.FCAdjFactors  );
            techs.CostAdjFactors = this.CloneDictionary(this.CostAdjFactors);
            //
            return techs;
        }
        Dictionary<ulong, double[]> CloneDictionary(Dictionary<ulong, double[]> value)
        {
            var newValue = new Dictionary<ulong, double[]>();
            foreach (var item in value)
            {
                newValue.Add(item.Key, item.Value.CloneArray());
            }
            return newValue;
        }

        /// <summary>
        /// Gets the fuel consumption adjustment factor associated with the specified key, for the specified technology class.
        /// </summary>
        /// <param name="key">The key for which to obtain the adjustment value.</param>
        /// <param name="techClass">The technology class for which to obtain the adjustment value.</param>
        /// <returns>The fuel consumption adjustment factor associated with the specified key and technology class. If an
        ///   adjustment factor for the specified key does not exist, a default value will be returned.</returns>
        public double GetFCAdjFactors(ulong key, int techClass)
        {
            return DictionaryTryGet(this.FCAdjFactors, key, techClass, 1.0);
        }
        /// <summary>
        /// Gets the cost adjustment factor associated with the specified key, for the specified technology class.
        /// </summary>
        /// <param name="key">The key for which to obtain the adjustment value.</param>
        /// <param name="techClass">The technology class for which to obtain the adjustment value.</param>
        /// <returns>The cost adjustment factor associated with the specified key and technology class. If an adjustment factor
        ///   for the specified key does not exist, a default value will be returned.</returns>
        public double GetCostAdjFactors(ulong key, int techClass)
        {
            return DictionaryTryGet(this.CostAdjFactors, key, techClass, 0.0);
        }
        double DictionaryTryGet(Dictionary<ulong, double[]> value, ulong key, int techClass, double defaultValue)
        {
            double[] d;
            return (value.TryGetValue(key, out d)) ? d[techClass] : defaultValue;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets a list of vehicle technologies for all technology classes.</summary>
        public TechnologyInfo[] TechnologyList { get; private set; }
        /// <summary>Gets a table of technology fuel consumption adjustment factors for all technology classes.</summary>
        public Dictionary<ulong, double[]> FCAdjFactors { get; private set; }
        /// <summary>Gets a table of technology cost adjustment factors for all technology classes.</summary>
        public Dictionary<ulong, double[]> CostAdjFactors { get; private set; }

        #endregion

    }
}
