using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Threading;
using System.Windows.Forms;
using Volpe.Cafe.Data;
//using Volpe.Cafe.EIS.Settings;
//using Volpe.Cafe.EIS.IO;
//using Volpe.Cafe.EIS.IO.InputParsers;
using Volpe.Cafe.IO;
using Volpe.Cafe.IO.InputParsers;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;

namespace Volpe.Cafe.UI.Panels
{
    /// <summary>
    /// Provides a panel for configuring input configuration settings for the compliance model.
    /// </summary>
	public class InputPanel : CafeUserControl
	{

        #region /*** Events ***/

        /// <summary>Occurs when the <see cref="Parsing"/> property has changed.</summary>
        public event EventHandler ParsingChanged;

        #endregion


        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="InputPanel"/> class.
        /// </summary>
		public InputPanel()
		{
			this.InitializeComponent();
            this.PostInitialize();
        }

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose(bool disposing)
		{
			base.Dispose(disposing);
		}

        #endregion


        #region /*** Methods ***/

		#region /* Designer generated code */

		void InitializeComponent()
		{
            this.ux_inputFilesLabel = new System.Windows.Forms.Label();
            this.ux_inputFilesPanel = new System.Windows.Forms.Panel();
            this.ux_reloadAllFromDisk = new System.Windows.Forms.CheckBox();
            this.ux_scenariosFileBrowse = new System.Windows.Forms.Button();
            this.ux_scenariosFile = new System.Windows.Forms.TextBox();
            this.ux_parametersFileBrowse = new System.Windows.Forms.Button();
            this.ux_parametersFile = new System.Windows.Forms.TextBox();
            this.ux_technologiesFileBrowse = new System.Windows.Forms.Button();
            this.ux_technologiesFile = new System.Windows.Forms.TextBox();
            this.ux_marketDataFileBrowse = new System.Windows.Forms.Button();
            this.ux_marketDataFile = new System.Windows.Forms.TextBox();
            this.ux_marketDataFileOutdatedLabel = new System.Windows.Forms.Label();
            this.ux_scenariosFileOutdatedLabel = new System.Windows.Forms.Label();
            this.ux_parametersFileOutdatedLabel = new System.Windows.Forms.Label();
            this.ux_technologiesFileOutdatedLabel = new System.Windows.Forms.Label();
            this.ux_technologiesFileLabel = new System.Windows.Forms.Label();
            this.ux_parametersFileLabel = new System.Windows.Forms.Label();
            this.ux_scenariosFileLabel = new System.Windows.Forms.Label();
            this.ux_marketDataFileLabel = new System.Windows.Forms.Label();
            this.ux_inputFilesPanel.SuspendLayout();
            this.SuspendLayout();
            // 
            // ux_inputFilesLabel
            // 
            this.ux_inputFilesLabel.AutoSize = true;
            this.ux_inputFilesLabel.Location = new System.Drawing.Point(8, 8);
            this.ux_inputFilesLabel.Name = "ux_inputFilesLabel";
            this.ux_inputFilesLabel.Size = new System.Drawing.Size(316, 13);
            this.ux_inputFilesLabel.TabIndex = 2;
            this.ux_inputFilesLabel.Text = "Please specify the input files you would like to use with the model:";
            // 
            // ux_inputFilesPanel
            // 
            this.ux_inputFilesPanel.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_inputFilesPanel.Controls.Add(this.ux_reloadAllFromDisk);
            this.ux_inputFilesPanel.Controls.Add(this.ux_scenariosFileBrowse);
            this.ux_inputFilesPanel.Controls.Add(this.ux_scenariosFile);
            this.ux_inputFilesPanel.Controls.Add(this.ux_parametersFileBrowse);
            this.ux_inputFilesPanel.Controls.Add(this.ux_parametersFile);
            this.ux_inputFilesPanel.Controls.Add(this.ux_technologiesFileBrowse);
            this.ux_inputFilesPanel.Controls.Add(this.ux_technologiesFile);
            this.ux_inputFilesPanel.Controls.Add(this.ux_marketDataFileBrowse);
            this.ux_inputFilesPanel.Controls.Add(this.ux_marketDataFile);
            this.ux_inputFilesPanel.Controls.Add(this.ux_marketDataFileOutdatedLabel);
            this.ux_inputFilesPanel.Controls.Add(this.ux_scenariosFileOutdatedLabel);
            this.ux_inputFilesPanel.Controls.Add(this.ux_parametersFileOutdatedLabel);
            this.ux_inputFilesPanel.Controls.Add(this.ux_technologiesFileOutdatedLabel);
            this.ux_inputFilesPanel.Controls.Add(this.ux_technologiesFileLabel);
            this.ux_inputFilesPanel.Controls.Add(this.ux_parametersFileLabel);
            this.ux_inputFilesPanel.Controls.Add(this.ux_scenariosFileLabel);
            this.ux_inputFilesPanel.Controls.Add(this.ux_marketDataFileLabel);
            this.ux_inputFilesPanel.Location = new System.Drawing.Point(8, 16);
            this.ux_inputFilesPanel.Name = "ux_inputFilesPanel";
            this.ux_inputFilesPanel.Size = new System.Drawing.Size(424, 192);
            this.ux_inputFilesPanel.TabIndex = 3;
            // 
            // ux_reloadAllFromDisk
            // 
            this.ux_reloadAllFromDisk.AutoSize = true;
            this.ux_reloadAllFromDisk.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
            this.ux_reloadAllFromDisk.Location = new System.Drawing.Point(8, 168);
            this.ux_reloadAllFromDisk.Name = "ux_reloadAllFromDisk";
            this.ux_reloadAllFromDisk.Size = new System.Drawing.Size(183, 17);
            this.ux_reloadAllFromDisk.TabIndex = 99;
            this.ux_reloadAllFromDisk.Text = "Reload all selected files from disk.";
            // 
            // ux_scenariosFileBrowse
            // 
            this.ux_scenariosFileBrowse.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_scenariosFileBrowse.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
            this.ux_scenariosFileBrowse.Location = new System.Drawing.Point(344, 144);
            this.ux_scenariosFileBrowse.Name = "ux_scenariosFileBrowse";
            this.ux_scenariosFileBrowse.Size = new System.Drawing.Size(80, 21);
            this.ux_scenariosFileBrowse.TabIndex = 19;
            this.ux_scenariosFileBrowse.Text = "Browse";
            // 
            // ux_scenariosFile
            // 
            this.ux_scenariosFile.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_scenariosFile.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_scenariosFile.Location = new System.Drawing.Point(8, 144);
            this.ux_scenariosFile.Name = "ux_scenariosFile";
            this.ux_scenariosFile.Size = new System.Drawing.Size(335, 20);
            this.ux_scenariosFile.TabIndex = 18;
            // 
            // ux_parametersFileBrowse
            // 
            this.ux_parametersFileBrowse.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_parametersFileBrowse.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
            this.ux_parametersFileBrowse.Location = new System.Drawing.Point(344, 104);
            this.ux_parametersFileBrowse.Name = "ux_parametersFileBrowse";
            this.ux_parametersFileBrowse.Size = new System.Drawing.Size(80, 21);
            this.ux_parametersFileBrowse.TabIndex = 11;
            this.ux_parametersFileBrowse.Text = "Browse";
            // 
            // ux_parametersFile
            // 
            this.ux_parametersFile.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_parametersFile.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_parametersFile.Location = new System.Drawing.Point(8, 104);
            this.ux_parametersFile.Name = "ux_parametersFile";
            this.ux_parametersFile.Size = new System.Drawing.Size(335, 20);
            this.ux_parametersFile.TabIndex = 10;
            // 
            // ux_technologiesFileBrowse
            // 
            this.ux_technologiesFileBrowse.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_technologiesFileBrowse.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
            this.ux_technologiesFileBrowse.Location = new System.Drawing.Point(344, 64);
            this.ux_technologiesFileBrowse.Name = "ux_technologiesFileBrowse";
            this.ux_technologiesFileBrowse.Size = new System.Drawing.Size(80, 21);
            this.ux_technologiesFileBrowse.TabIndex = 7;
            this.ux_technologiesFileBrowse.Text = "Browse";
            // 
            // ux_technologiesFile
            // 
            this.ux_technologiesFile.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_technologiesFile.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_technologiesFile.Location = new System.Drawing.Point(8, 64);
            this.ux_technologiesFile.Name = "ux_technologiesFile";
            this.ux_technologiesFile.Size = new System.Drawing.Size(335, 20);
            this.ux_technologiesFile.TabIndex = 6;
            // 
            // ux_marketDataFileBrowse
            // 
            this.ux_marketDataFileBrowse.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_marketDataFileBrowse.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
            this.ux_marketDataFileBrowse.Location = new System.Drawing.Point(344, 24);
            this.ux_marketDataFileBrowse.Name = "ux_marketDataFileBrowse";
            this.ux_marketDataFileBrowse.Size = new System.Drawing.Size(80, 21);
            this.ux_marketDataFileBrowse.TabIndex = 3;
            this.ux_marketDataFileBrowse.Text = "Browse";
            // 
            // ux_marketDataFile
            // 
            this.ux_marketDataFile.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.ux_marketDataFile.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.ux_marketDataFile.Location = new System.Drawing.Point(8, 24);
            this.ux_marketDataFile.Name = "ux_marketDataFile";
            this.ux_marketDataFile.Size = new System.Drawing.Size(335, 20);
            this.ux_marketDataFile.TabIndex = 2;
            // 
            // ux_marketDataFileOutdatedLabel
            // 
            this.ux_marketDataFileOutdatedLabel.AutoEllipsis = true;
            this.ux_marketDataFileOutdatedLabel.AutoSize = true;
            this.ux_marketDataFileOutdatedLabel.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Italic, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.ux_marketDataFileOutdatedLabel.ForeColor = System.Drawing.Color.Gray;
            this.ux_marketDataFileOutdatedLabel.Location = new System.Drawing.Point(96, 8);
            this.ux_marketDataFileOutdatedLabel.Name = "ux_marketDataFileOutdatedLabel";
            this.ux_marketDataFileOutdatedLabel.Size = new System.Drawing.Size(228, 13);
            this.ux_marketDataFileOutdatedLabel.TabIndex = 1;
            this.ux_marketDataFileOutdatedLabel.Text = "(The file on disk is newer than data in memory.)";
            this.ux_marketDataFileOutdatedLabel.Visible = false;
            // 
            // ux_scenariosFileOutdatedLabel
            // 
            this.ux_scenariosFileOutdatedLabel.AutoEllipsis = true;
            this.ux_scenariosFileOutdatedLabel.AutoSize = true;
            this.ux_scenariosFileOutdatedLabel.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Italic, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.ux_scenariosFileOutdatedLabel.ForeColor = System.Drawing.Color.Gray;
            this.ux_scenariosFileOutdatedLabel.Location = new System.Drawing.Point(80, 128);
            this.ux_scenariosFileOutdatedLabel.Name = "ux_scenariosFileOutdatedLabel";
            this.ux_scenariosFileOutdatedLabel.Size = new System.Drawing.Size(228, 13);
            this.ux_scenariosFileOutdatedLabel.TabIndex = 17;
            this.ux_scenariosFileOutdatedLabel.Text = "(The file on disk is newer than data in memory.)";
            this.ux_scenariosFileOutdatedLabel.Visible = false;
            // 
            // ux_parametersFileOutdatedLabel
            // 
            this.ux_parametersFileOutdatedLabel.AutoEllipsis = true;
            this.ux_parametersFileOutdatedLabel.AutoSize = true;
            this.ux_parametersFileOutdatedLabel.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Italic, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.ux_parametersFileOutdatedLabel.ForeColor = System.Drawing.Color.Gray;
            this.ux_parametersFileOutdatedLabel.Location = new System.Drawing.Point(88, 88);
            this.ux_parametersFileOutdatedLabel.Name = "ux_parametersFileOutdatedLabel";
            this.ux_parametersFileOutdatedLabel.Size = new System.Drawing.Size(228, 13);
            this.ux_parametersFileOutdatedLabel.TabIndex = 9;
            this.ux_parametersFileOutdatedLabel.Text = "(The file on disk is newer than data in memory.)";
            this.ux_parametersFileOutdatedLabel.Visible = false;
            // 
            // ux_technologiesFileOutdatedLabel
            // 
            this.ux_technologiesFileOutdatedLabel.AutoEllipsis = true;
            this.ux_technologiesFileOutdatedLabel.AutoSize = true;
            this.ux_technologiesFileOutdatedLabel.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Italic, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.ux_technologiesFileOutdatedLabel.ForeColor = System.Drawing.Color.Gray;
            this.ux_technologiesFileOutdatedLabel.Location = new System.Drawing.Point(96, 48);
            this.ux_technologiesFileOutdatedLabel.Name = "ux_technologiesFileOutdatedLabel";
            this.ux_technologiesFileOutdatedLabel.Size = new System.Drawing.Size(228, 13);
            this.ux_technologiesFileOutdatedLabel.TabIndex = 5;
            this.ux_technologiesFileOutdatedLabel.Text = "(The file on disk is newer than data in memory.)";
            this.ux_technologiesFileOutdatedLabel.Visible = false;
            // 
            // ux_technologiesFileLabel
            // 
            this.ux_technologiesFileLabel.AutoSize = true;
            this.ux_technologiesFileLabel.Location = new System.Drawing.Point(8, 48);
            this.ux_technologiesFileLabel.Name = "ux_technologiesFileLabel";
            this.ux_technologiesFileLabel.Size = new System.Drawing.Size(90, 13);
            this.ux_technologiesFileLabel.TabIndex = 4;
            this.ux_technologiesFileLabel.Text = "Technologies file:";
            // 
            // ux_parametersFileLabel
            // 
            this.ux_parametersFileLabel.AutoSize = true;
            this.ux_parametersFileLabel.Location = new System.Drawing.Point(8, 88);
            this.ux_parametersFileLabel.Name = "ux_parametersFileLabel";
            this.ux_parametersFileLabel.Size = new System.Drawing.Size(79, 13);
            this.ux_parametersFileLabel.TabIndex = 8;
            this.ux_parametersFileLabel.Text = "Parameters file:";
            // 
            // ux_scenariosFileLabel
            // 
            this.ux_scenariosFileLabel.AutoSize = true;
            this.ux_scenariosFileLabel.Location = new System.Drawing.Point(8, 128);
            this.ux_scenariosFileLabel.Name = "ux_scenariosFileLabel";
            this.ux_scenariosFileLabel.Size = new System.Drawing.Size(73, 13);
            this.ux_scenariosFileLabel.TabIndex = 16;
            this.ux_scenariosFileLabel.Text = "Scenarios file:";
            // 
            // ux_marketDataFileLabel
            // 
            this.ux_marketDataFileLabel.AutoSize = true;
            this.ux_marketDataFileLabel.Location = new System.Drawing.Point(8, 8);
            this.ux_marketDataFileLabel.Name = "ux_marketDataFileLabel";
            this.ux_marketDataFileLabel.Size = new System.Drawing.Size(85, 13);
            this.ux_marketDataFileLabel.TabIndex = 0;
            this.ux_marketDataFileLabel.Text = "Market-Data file:";
            // 
            // InputPanel
            // 
            this.Controls.Add(this.ux_inputFilesLabel);
            this.Controls.Add(this.ux_inputFilesPanel);
            this.MinimumSize = new System.Drawing.Size(440, 0);
            this.Name = "InputPanel";
            this.Size = new System.Drawing.Size(440, 220);
            this.ux_inputFilesPanel.ResumeLayout(false);
            this.ux_inputFilesPanel.PerformLayout();
            this.ResumeLayout(false);
            this.PerformLayout();

        }

		#endregion

        #region /* Methods for raising events */

        /// <summary>
        /// Raises the <see cref="ParsingChanged"/> event.
        /// </summary>
        /// <param name="e">Event data to pass to the event</param>
        protected virtual void OnParsingChanged(EventArgs e)
        {
            if (this.ParsingChanged != null)
            {
                this.ParsingChanged(this, e);
            }
        }

        #endregion

        void PostInitialize()
        {
            // add support for dragging and dropping for the input panel
            this.InitializeInputPanelDragDrop();
            //
            // initialize check-changes
            this.AutoInitializeCheckChangesDetector(this.Controls);
            this.DeInitializeCheckChangesDetector(this.ux_reloadAllFromDisk);
            // add event handlers for certain controls
            this.ux_marketDataFile        .TextChanged    += new EventHandler(this.InputFile_TextChanged);
            this.ux_technologiesFile      .TextChanged    += new EventHandler(this.InputFile_TextChanged);
            this.ux_parametersFile        .TextChanged    += new EventHandler(this.InputFile_TextChanged);
            this.ux_scenariosFile         .TextChanged    += new EventHandler(this.InputFile_TextChanged);
            //
            this.ux_marketDataFile        .VisibleChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            this.ux_technologiesFile      .VisibleChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            this.ux_parametersFile        .VisibleChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            this.ux_scenariosFile         .VisibleChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            //
            this.ux_marketDataFile        .EnabledChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            this.ux_technologiesFile      .EnabledChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            this.ux_parametersFile        .EnabledChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            this.ux_scenariosFile         .EnabledChanged += new EventHandler(this.InputFile_VisibleOrEnabledChanged);
            //
            this.ux_marketDataFileBrowse  .Click          += new EventHandler(this.InputFileBrowse_Click);
            this.ux_technologiesFileBrowse.Click          += new EventHandler(this.InputFileBrowse_Click);
            this.ux_parametersFileBrowse  .Click          += new EventHandler(this.InputFileBrowse_Click);
            this.ux_scenariosFileBrowse   .Click          += new EventHandler(this.InputFileBrowse_Click);
            //
            // set initial value for certain controls to trigger events
            this.ux_marketDataFile        .Text = " ";
            this.ux_technologiesFile      .Text = " ";
            this.ux_parametersFile        .Text = " ";
            this.ux_scenariosFile         .Text = " ";
            //
            // initialize internal "working" arrays and other variables
            this._data          = new object[FileCount];
            this._verifiedFiles = new bool  [FileCount];
            this._visibleFiles  = new bool  [FileCount];
            this._loadFailed    = new bool  [FileCount];
            // assume all input file textboxes are initially visible
            for (int i = 0; i < FileCount; i++) { this._visibleFiles[i] = true; }
            //
            this._parsing = 0;
            this._finalizeParsing = false;
            this._saveThread = null;
        }
        void InputFile_TextChanged(object sender, EventArgs e)
        {
            TextBox tb             = (TextBox)sender;
            bool    isMarketData   = (tb == this.ux_marketDataFile);
            bool    isTechnologies = (tb == this.ux_technologiesFile);
            bool    isParameters   = (tb == this.ux_parametersFile);
            bool    isScenarios    = (tb == this.ux_scenariosFile);
            //
            int     fileID =
                (isMarketData  ) ? MarketDataID :
                (isTechnologies) ? TechnologiesID :
                (isParameters  ) ? ParametersID :
                (isScenarios   ) ? ScenariosID : -1;
            Label   outdatedLbl =
                (isMarketData  ) ? this.ux_marketDataFileOutdatedLabel :
                (isTechnologies) ? this.ux_technologiesFileOutdatedLabel :
                (isParameters  ) ? this.ux_parametersFileOutdatedLabel :
                (isScenarios   ) ? this.ux_scenariosFileOutdatedLabel : null;
            //
            if (fileID == -1 || outdatedLbl == null) { return; }
            //
            string filePath = tb.Text.Trim();
            if (filePath != string.Empty && File.Exists(filePath))
            {
                bool inputFilePathChanged =
                    (isMarketData   && !this.CompareFilePaths(this._inputSettings.MarketDataFile  , filePath)) ||
                    (isTechnologies && !this.CompareFilePaths(this._inputSettings.TechnologiesFile, filePath)) ||
                    (isParameters   && !this.CompareFilePaths(this._inputSettings.ParametersFile  , filePath)) ||
                    (isScenarios    && !this.CompareFilePaths(this._inputSettings.ScenariosFile   , filePath));
                tb.ForeColor = (this._visualTracking && inputFilePathChanged) ? this._visualTrackingColor : Control.DefaultForeColor;
                tb.BackColor = Color.Empty;
                this._messages.RemoveMessage(tb);
                //
                if (inputFilePathChanged)
                {   // the file path has changed -- check to see if the new path is a valid excel file
                    this.BeginParse();
                    if (this.VerifyFile(tb, fileID, this.GetCurrentPassword()))
                    {
                        tb.ForeColor = (this._visualTracking) ? this._visualTrackingColor : Control.DefaultForeColor;
                        tb.BackColor = Color.Empty;
                    }
                    this.EndParse();
                    //
                    outdatedLbl.Visible = false;
                } // end if (inputFilePathChanged)
            }
            else
            {
                if (filePath == string.Empty)
                {
                    this._messages.AddMessage(
                        "Please select a " + FriendlyTitles[fileID] + " file you would like to use with the model.", null, Color.Red, tb);
                }
                else
                {
                    this._messages.AddMessage(
                        "The specified " + FriendlyTitles[fileID] + " file does not exist on disk.", null, Color.Red, tb);
                }
                tb.BackColor = Color.Honeydew;
                tb.ForeColor = Color.Red;
            }
        }
        void InputFile_VisibleOrEnabledChanged(object sender, EventArgs e)
        {
            if (this._finalizeParsing) { return; }
            TextBox tb = (TextBox)sender;
            if (tb.Enabled && tb.Visible) { this.InputFile_TextChanged(sender, e); }
            else                          { this._messages.RemoveMessage(sender); tb.BackColor = Color.Empty; }
        }
        void InputFileBrowse_Click(object sender, EventArgs e)
        {
            TextBox tb = null;
            int id = -1;
            if      (sender == this.ux_marketDataFileBrowse  ) { id = MarketDataID  ; tb = this.ux_marketDataFile  ; }
            else if (sender == this.ux_technologiesFileBrowse) { id = TechnologiesID; tb = this.ux_technologiesFile; }
            else if (sender == this.ux_parametersFileBrowse  ) { id = ParametersID  ; tb = this.ux_parametersFile  ; }
            else if (sender == this.ux_scenariosFileBrowse   ) { id = ScenariosID   ; tb = this.ux_scenariosFile   ; }
            //
            OpenFileDialog dlg = new OpenFileDialog();
            if (id != -1) { dlg.Title = "Please select the " + FriendlyTitles[id] + " file you would like to use with the model:"; }
            if (File.Exists(tb.Text)) { dlg.FileName = tb.Text; }
            dlg.Multiselect = false;
            dlg.Filter = "Microsoft Excel Workbook Files (*.xls)|*.xls";
            if (dlg.ShowDialog() == DialogResult.OK) { tb.Text = dlg.FileName; }
        }
        bool CompareFilePaths(string path1, string path2)
        {
            if (File.Exists(path1)) { path1 = Path.GetFullPath(path1); }
            if (File.Exists(path2)) { path2 = Path.GetFullPath(path2); }
            return Interaction.StringCompare(path1, path2, false);
        }

        // ----- support for drag & drop -----
        void InitializeInputPanelDragDrop()
        {
            this.ux_inputFilesPanel.AllowDrop = true;
            this.ux_inputFilesPanel.DragEnter += new DragEventHandler(this.InputFilePanel_DragEnter);
            this.ux_inputFilesPanel.DragDrop  += new DragEventHandler(this.InputFilePanel_DragDrop);
        }
        void InputFilePanel_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop, false))
            {
                e.Effect = DragDropEffects.All;
            }
        }
        void InputFilePanel_DragDrop(object sender, DragEventArgs e)
        {
            // get the files and directories from the drag-drop and parse them
            string[] files = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            this.ParseFiles(files, this.GetCurrentPassword(), false);
        }

        // ----- begin/end parsing methods -----
        void BeginParse()
        {
            this._parsing++;
            if (this._parsing == 1)
            {
                this.ChangeInputState(this, EventArgs.Empty);
                this.OnParsingChanged(EventArgs.Empty);
            }
        }
        void EndParse()
        {
            if (this._parsing > 0)
            {
                this._parsing--;
                if (this._parsing == 0)
                {
                    this._finalizeParsing = true;
                    this.ChangeInputState(this, EventArgs.Empty);
                    this.OnParsingChanged(EventArgs.Empty);
                    this._finalizeParsing = false;
                }
            }
        }
        void ChangeInputState(object sender, EventArgs e)
        {
            if (this.Visible && this.IsHandleCreated)
            {
                if (this.InvokeRequired)
                {
                    this.Invoke(new EventHandler(this.ChangeInputState));
                }
                else
                {
                    bool enabled = (this._parsing == 0);
                    //
                    this.AllowDrop                           =  enabled;
                    //
                    this.ux_inputFilesPanel       .AllowDrop =  enabled;
                    //
                    this.ux_marketDataFile        .ReadOnly  = !enabled;
                    this.ux_technologiesFile      .ReadOnly  = !enabled;
                    this.ux_parametersFile        .ReadOnly  = !enabled;
                    this.ux_scenariosFile         .ReadOnly  = !enabled;
                    //
                    this.ux_marketDataFileBrowse  .Enabled   =  enabled;
                    this.ux_technologiesFileBrowse.Enabled   =  enabled;
                    this.ux_parametersFileBrowse  .Enabled   =  enabled;
                    this.ux_scenariosFileBrowse   .Enabled   =  enabled;
                    //
                    if (enabled)
                    {
                        this.ux_reloadAllFromDisk.Checked = false;
                    }
                } // end if (invoke-required)
            } // end if (visible & handle created)
        }

        // ----- file parsing, verification and loading methods -----
        void ParseFile(string path, string password, bool keepExisting)
        {
            this.ParseFiles(new string[] {path}, password, keepExisting);
        }
        void ParseFiles(string[] paths, string password, bool keepExisting)
        {
            if (paths == null || paths.Length == 0) { return; }
            //
            this.BeginParse();

            // first, split paths into files and directories
            StringCollection scFiles = new StringCollection();
            StringCollection scDirs  = new StringCollection();
            for (int i = 0; i < paths.Length; i++)
            {
                if      (Directory.Exists(paths[i])) { scDirs .Add(paths[i]); }
                else if (File     .Exists(paths[i])) { scFiles.Add(paths[i]); }
            }
            string[] files = new string[scFiles.Count]; scFiles.CopyTo(files, 0);
            string[] dirs  = new string[scDirs .Count]; scDirs .CopyTo(dirs , 0);

            // next, initialize the "found" array
            bool[] found = new bool[FileCount];
            if (keepExisting)
            {
                if (File.Exists(this.ux_marketDataFile  .Text)) { found[MarketDataID  ] = true; }
                if (File.Exists(this.ux_technologiesFile.Text)) { found[TechnologiesID] = true; }
                if (File.Exists(this.ux_parametersFile  .Text)) { found[ParametersID  ] = true; }
                if (File.Exists(this.ux_scenariosFile   .Text)) { found[ScenariosID   ] = true; }
            }

            // next, parse files, then dirs
            this.ParseFilesHelper(files, dirs, found, password, 3);

            this.EndParse();
        }
        void ParseFilesHelper(string[] files, string[] dirs, bool[] found, string password, int recurseDepth)
        {
            // count up the number of files that have been found already
            int foundCount = 0;
            for (int i = 0; i < found.Length; i++)
            {
                if (found[i]) { foundCount++; }
            }

            // scan any available files first
            for (int i = 0; i < files.Length; i++)
            {
                // check to see if all files have been found already
                if (foundCount == FileCount) { return; }
                //
                // continue with parse ...
                this._messages.AddMessage("Checking input file:  \"" + files[i] + "\"", null, Color.Blue, files[i]);
                Application.DoEvents();

                string wbTitle;
                try
                {   // try opening the input file to obtain the workbook title
                    Input input = new Input(files[i], password);
                    input.Close();
                    wbTitle = input.Title;
                } catch { continue; }

                for (int j = 0; j < FileCount; j++)
                {
                    if (this._visibleFiles[j] && !found[j] && Interaction.StringCompareAny(wbTitle, DesiredTitles[j], true))
                    {
                        found[j] = true;
                        foundCount++;
                        // update file-path in the text-box
                        if      (j == MarketDataID  ) { this.ux_marketDataFile  .Text = files[i]; }
                        else if (j == TechnologiesID) { this.ux_technologiesFile.Text = files[i]; }
                        else if (j == ParametersID  ) { this.ux_parametersFile  .Text = files[i]; }
                        else if (j == ScenariosID   ) { this.ux_scenariosFile   .Text = files[i]; }
                        //
                        break;
                    }
                }

                this._messages.RemoveMessage(files[i]);
                Application.DoEvents();
            } // next i (file)

            // recurse into directories
            for (int i = 0; i < dirs.Length; i++)
            {
                string[] recurseFiles = Directory.GetFiles(dirs[i], "*.xls");
                string[] recurseDirs  = (recurseDepth > 0) ? Directory.GetDirectories(dirs[i]) : new string[0];
                this.ParseFilesHelper(recurseFiles, recurseDirs, found, password, recurseDepth - 1);
            }
        }
        void VerifyFiles()
        {
            this.BeginParse();
            string password = this.GetCurrentPassword();
            //
            this.VerifyFile(this.ux_marketDataFile  , MarketDataID  , password);
            this.VerifyFile(this.ux_technologiesFile, TechnologiesID, password);
            this.VerifyFile(this.ux_parametersFile  , ParametersID  , password);
            this.VerifyFile(this.ux_scenariosFile   , ScenariosID   , password);
            //
            this.EndParse();
        }
        bool VerifyFile(TextBox tb, int fileID, string password)
        {
            this._messages.AddMessage("Checking " + FriendlyTitles[fileID] + " input file ...", null, Color.Blue, tb);
            Application.DoEvents();
            //
            bool verified = false;
            try
            {
                Input inputFile = new Input(tb.Text, password);
                inputFile.Close();
                // compare the title of the input file to one of the desired titles
                verified = Interaction.StringCompareAny(inputFile.Title, DesiredTitles[fileID], true);
            }
            catch { verified = false; }
            //
            this._messages.RemoveMessage(tb);
            Application.DoEvents();
            //
            if (!verified)
            {
                this._messages.AddMessage("The selected " + FriendlyTitles[fileID] + " file could not be opened.", null, Color.Red, tb);
                tb.BackColor = Color.Honeydew;
                tb.ForeColor = Color.Red;
                Application.DoEvents();
            }
            //
            return verified;
        }
        void LoadFiles()
        {
            this.BeginParse();
            string password = this.GetCurrentPassword();
            //
            this.LoadFile(MarketDataID  , this.ux_marketDataFile  , password, ref this._inputSettings.MarketDataFile  , ref this._inputSettings.MarketDataTimestamp  );
            this.LoadFile(TechnologiesID, this.ux_technologiesFile, password, ref this._inputSettings.TechnologiesFile, ref this._inputSettings.TechnologiesTimestamp);
            this.LoadFile(ParametersID  , this.ux_parametersFile  , password, ref this._inputSettings.ParametersFile  , ref this._inputSettings.ParametersTimestamp  );
            this.LoadFile(ScenariosID   , this.ux_scenariosFile   , password, ref this._inputSettings.ScenariosFile   , ref this._inputSettings.ScenariosTimestamp   );
            //
            this.EndParse();
        }
        bool LoadFile(int fileID, TextBox tb, string password, ref string file, ref DateTime timestamp)
        {
            // do not load files from hidden text-boxes
            if (!this._visibleFiles[fileID]) { return false; }

            // check if the file needs reloading -- (if not, return false)
            //   if "reload-from-disk" option is selected, always reload
            //   otherwise, reload if current file from input settings is not the same as the one in the textbox
            if (!(this.ux_reloadAllFromDisk.Checked || !this.CompareFilePaths(file, tb.Text))) { return false; }

            // -------------------------------------------------- \\
            // continue with file loading
            this._messages.AddMessage("Loading " + FriendlyTitles[fileID] + " input file ...", null, Color.Blue, tb);
            Application.DoEvents();
            //
            bool fileLoaded = false;
            string exMessage = string.Empty;
            try
            {
                if      (fileID == MarketDataID  ) { this._data[fileID] = XlParser.ParseMarketData  (tb.Text, password); }
                else if (fileID == TechnologiesID) { this._data[fileID] = XlParser.ParseTechnologies(tb.Text, password); }
                else if (fileID == ParametersID  ) { this._data[fileID] = XlParser.ParseParameters  (tb.Text, password); }
                else if (fileID == ScenariosID   ) { this._data[fileID] = XlParser.ParseScenarios   (tb.Text, password); }
                //
                file       = tb.Text.Trim();
                timestamp  = File.GetLastWriteTime(file);
                fileLoaded = true;
            }
            catch (Exception ex) { exMessage = ex.Message; fileLoaded = false; }
            //
            this._messages.RemoveMessage(tb);
            Application.DoEvents();
            //
            if (!fileLoaded)
            {
                this._data[fileID] = null;
                this._messages.AddMessage("The selected " + FriendlyTitles[fileID] + " file could not be opened.", exMessage, Color.Red, tb);
                tb.BackColor = Color.Honeydew;
                tb.ForeColor = Color.Red;
            }
            //
            this._loadFailed[fileID] = !fileLoaded;
            return fileLoaded;
        }

        string GetCurrentPassword()
        {
            return string.Empty;
        }
        void ShowHideFilePaths()
        {
            if (!this.Visible) { return; }

            // calc start positions
            int[] pt = new int[FileCount + 1];
            pt[0] = 8;
            for (int i = 0; i < FileCount + 1; i++) { pt[i] = 8 + i * 40; }
            bool showAny = false;
            for (int i = 1; i < FileCount + 1; i++)
            {
                pt[i] = (this._visibleFiles[i - 1]) ? pt[i - 1] + 40 : pt[i - 1];
                if (this._visibleFiles[i - 1]) { showAny = true; }
            }

            // show/hide prompts and adjust positions as necessary
            if (!this._visibleFiles[MarketDataID]) { this.ux_marketDataFileOutdatedLabel.Visible = false; }
            this.ux_marketDataFileLabel          .Visible = this._visibleFiles[MarketDataID];
            this.ux_marketDataFile               .Visible = this._visibleFiles[MarketDataID];
            this.ux_marketDataFileBrowse         .Visible = this._visibleFiles[MarketDataID];
            //
            if (!this._visibleFiles[TechnologiesID]) { this.ux_technologiesFileOutdatedLabel.Visible = false; }
            this.ux_technologiesFileLabel        .Visible = this._visibleFiles[TechnologiesID];
            this.ux_technologiesFile             .Visible = this._visibleFiles[TechnologiesID];
            this.ux_technologiesFileBrowse       .Visible = this._visibleFiles[TechnologiesID];
            this.ux_technologiesFileLabel        .Top     = pt[1];
            this.ux_technologiesFileOutdatedLabel.Top     = pt[1];
            this.ux_technologiesFile             .Top     = pt[1] + 16;
            this.ux_technologiesFileBrowse       .Top     = pt[1] + 16;
            //
            if (!this._visibleFiles[ParametersID]) { this.ux_parametersFileOutdatedLabel.Visible = false; }
            this.ux_parametersFileLabel          .Visible = this._visibleFiles[ParametersID];
            this.ux_parametersFile               .Visible = this._visibleFiles[ParametersID];
            this.ux_parametersFileBrowse         .Visible = this._visibleFiles[ParametersID];
            this.ux_parametersFileLabel          .Top     = pt[2];
            this.ux_parametersFileOutdatedLabel  .Top     = pt[2];
            this.ux_parametersFile               .Top     = pt[2] + 16;
            this.ux_parametersFileBrowse         .Top     = pt[2] + 16;
            //
            if (!this._visibleFiles[ScenariosID]) { this.ux_scenariosFileOutdatedLabel.Visible = false; }
            this.ux_scenariosFileLabel           .Visible = this._visibleFiles[ScenariosID];
            this.ux_scenariosFile                .Visible = this._visibleFiles[ScenariosID];
            this.ux_scenariosFileBrowse          .Visible = this._visibleFiles[ScenariosID];
            this.ux_scenariosFileLabel           .Top     = pt[3];
            this.ux_scenariosFileOutdatedLabel   .Top     = pt[3];
            this.ux_scenariosFile                .Top     = pt[3] + 16;
            this.ux_scenariosFileBrowse          .Top     = pt[3] + 16;

            //
            this.ux_reloadAllFromDisk            .Visible = showAny;
            this.ux_reloadAllFromDisk            .Top     = pt[4];
            //
            this.ux_inputFilesPanel              .Height  = pt[4] + 24;
        }

        /// <summary>
        /// Sets up the initial states of the controls for the <see cref="InputPanel"/> instance based on the specified input
        /// settings.
        /// </summary>
        /// <param name="inputSettings">The input settings for the modeling system.</param>
        public void SetInitialState(InputSettings inputSettings)
        {
            // copy input settings
            this._inputSettings = inputSettings;

            // set initial states
            bool forceReload = false;
            //
            this.SetInitialState_ProcessFile(inputSettings.MarketDataFile  , inputSettings.MarketDataTimestamp  , MarketDataID  , this.ux_marketDataFile  , this.ux_marketDataFileOutdatedLabel  , ref forceReload);
            this.SetInitialState_ProcessFile(inputSettings.TechnologiesFile, inputSettings.TechnologiesTimestamp, TechnologiesID, this.ux_technologiesFile, this.ux_technologiesFileOutdatedLabel, ref forceReload);
            this.SetInitialState_ProcessFile(inputSettings.ParametersFile  , inputSettings.ParametersTimestamp  , ParametersID  , this.ux_parametersFile  , this.ux_parametersFileOutdatedLabel  , ref forceReload);
            this.SetInitialState_ProcessFile(inputSettings.ScenariosFile   , inputSettings.ScenariosTimestamp   , ScenariosID   , this.ux_scenariosFile   , this.ux_scenariosFileOutdatedLabel   , ref forceReload);

            // set whether reloading of files is required
            this.ux_reloadAllFromDisk.Checked = forceReload;

            // refresh file path visible states
            this.ShowHideFilePaths();

            // call save in base to reset initial "changes"
            base.SaveChanges();
            //this.PostSave();
        }
        void SetInitialState_ProcessFile(string file, DateTime timestamp, int fileID, TextBox tb, Label outdatedLbl, ref bool forceReload)
        {
            // do not check files from hidden text-boxes
            if (!this._visibleFiles[fileID]) { return; }

            tb.Text = file;
            Application.DoEvents();
            //
            // check timestamps -- if file on disk is newer than the one loaded in memory, generate warning message
            if (File.Exists(file))
            {
                DateTime dt = File.GetLastWriteTime(file);
                if (timestamp < dt)
                {
                    outdatedLbl.Visible = true;
                    forceReload = true;
                }
            }
        }

        /// <summary>
        /// Commits the most recently made changes to the internal <see cref="InputSettings"/> structure.
        /// </summary>
        public override void SaveChanges()
        {
            if (this.SettingsChanged)
            {   // some output chaged -- continue with save
                // load files in a new thread
                this._saveThread = new Thread(new ThreadStart(this.SaveChangesThread));
                this._saveThread.Name = "InputPanelSaveThread";
                this._saveThread.Start();
            }
        }
        void SaveChangesThread()
        {
            this.LoadFiles();       // load input files
            base.SaveChanges();     // call "save-changes" from base class to reset "visual tracking changes"
            this.PostSave();        // additional processing for cetain text controls
        }
        /// <summary>
        /// Clears visual tracking for certain text controls.
        /// </summary>
        void PostSave()
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new EmptyEventHandler(this.PostSave));
            }
            else if (this._visualTracking)
            {   // clear visual tracking
                if (!this._loadFailed[MarketDataID])
                {
                    this.ux_marketDataFile.ForeColor = Control.DefaultForeColor;
                    this.ux_marketDataFile.BackColor = Color  .Empty;
                }
                if (!this._loadFailed[TechnologiesID])
                {
                    this.ux_technologiesFile.ForeColor = Control.DefaultForeColor;
                    this.ux_technologiesFile.BackColor = Color  .Empty;
                }
                if (!this._loadFailed[ParametersID])
                {
                    this.ux_parametersFile.ForeColor = Control.DefaultForeColor;
                    this.ux_parametersFile.BackColor = Color  .Empty;
                }
                if (!this._loadFailed[ScenariosID])
                {
                    this.ux_scenariosFile.ForeColor = Control.DefaultForeColor;
                    this.ux_scenariosFile.BackColor = Color  .Empty;
                }

                //
                this.ux_marketDataFileOutdatedLabel  .Visible = false;
                this.ux_technologiesFileOutdatedLabel.Visible = false;
                this.ux_parametersFileOutdatedLabel  .Visible = false;
                this.ux_scenariosFileOutdatedLabel   .Visible = false;
            }
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets or sets whether to display the Market-Data file prompt.</summary>
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Market-Data file prompt.")]
        public bool ShowMarketData
        {
            get { return this._visibleFiles[MarketDataID]; }
            set { this._visibleFiles[MarketDataID] = value; this.ShowHideFilePaths(); }
        }
        /// <summary>Gets or sets whether to display the Technologies file prompt.</summary>
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Technologies file prompt.")]
        public bool ShowTechnologies
        {
            get { return this._visibleFiles[TechnologiesID]; }
            set { this._visibleFiles[TechnologiesID] = value; this.ShowHideFilePaths(); }
        }
        /// <summary>Gets or sets whether to display the Parameters file prompt.</summary>
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Parameters file prompt.")]
        public bool ShowParameters
        {
            get { return this._visibleFiles[ParametersID]; }
            set { this._visibleFiles[ParametersID] = value; this.ShowHideFilePaths(); }
        }
        /// <summary>Gets or sets whether to display the Scenarios file prompt.</summary>
        [Browsable(true)]
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Specifies whether to display the Scenarios file prompt.")]
        public bool ShowScenarios
        {
            get { return this._visibleFiles[ScenariosID]; }
            set { this._visibleFiles[ScenariosID] = value; this.ShowHideFilePaths(); }
        }

        /// <summary>Gets the updated data loaded from the market-data input file, or null, if nothing was loaded.</summary>
        [Browsable(false)]
        public Industry MarketData
        {
            get { return (this._data[MarketDataID] == null) ? null : (Industry)this._data[MarketDataID]; }
        }
        /// <summary>Gets the updated data loaded from the technologies input file, or null, if nothing was loaded.</summary>
        [Browsable(false)]
        public Technologies Technologies
        {
            get { return (this._data[TechnologiesID] == null) ? null : (Technologies)this._data[TechnologiesID]; }
        }
        /// <summary>Gets the updated data loaded from the parameters input file, or null, if nothing was loaded.</summary>
        [Browsable(false)]
        public Parameters Parameters
        {
            get { return (this._data[ParametersID] == null) ? null : (Parameters)this._data[ParametersID]; }
        }
        /// <summary>Gets the updated data loaded from the scenarios input file, or null, if nothing was loaded.</summary>
        [Browsable(false)]
        public List<Scenario> Scenarios
        {
            get { return (this._data[ScenariosID] == null) ? null : (List<Scenario>)this._data[ScenariosID]; }
        }


        /// <summary>Gets whether the <see cref="InputPanel"/> is currently parsing, verifying, or loading input files.</summary>
        [Browsable(false)]
        public bool Parsing { get { return this._parsing > 0; } }

        /// <summary>Gets whether any of the settings have changed since the control has been loaded or since the changes have
        ///   been last saved.</summary>
        [Browsable(false)]
        public override bool SettingsChanged
        {
            get
            {
                if (this.Parsing) { return false; }     // cannot save if parsing in progress
                //
                // first, perform some error checking (e.g., missing or incorrect text fields)
                if ((this._visibleFiles[MarketDataID  ] && !File.Exists(this.ux_marketDataFile  .Text)) ||
                    (this._visibleFiles[TechnologiesID] && !File.Exists(this.ux_technologiesFile.Text)) ||
                    (this._visibleFiles[ParametersID  ] && !File.Exists(this.ux_parametersFile  .Text)) ||
                    (this._visibleFiles[ScenariosID   ] && !File.Exists(this.ux_scenariosFile   .Text))) { return false; }
                //
                // then, check to see if things actually changed compared to the input-settings struct
                return
                    base.SettingsChanged ||
                    this.ux_reloadAllFromDisk.Checked ||
                    (this._visibleFiles[MarketDataID  ] && !this.CompareFilePaths(this._inputSettings.MarketDataFile  , this.ux_marketDataFile  .Text)) ||
                    (this._visibleFiles[TechnologiesID] && !this.CompareFilePaths(this._inputSettings.TechnologiesFile, this.ux_technologiesFile.Text)) ||
                    (this._visibleFiles[ParametersID  ] && !this.CompareFilePaths(this._inputSettings.ParametersFile  , this.ux_parametersFile  .Text)) ||
                    (this._visibleFiles[ScenariosID   ] && !this.CompareFilePaths(this._inputSettings.ScenariosFile   , this.ux_scenariosFile   .Text));
            }
        }

        #endregion


        #region /*** Variables ***/

        #region /* UI Variables */

        //
        Label   ux_inputFilesLabel;
        Panel   ux_inputFilesPanel;
        // market data
        Label   ux_marketDataFileLabel;
        Label   ux_marketDataFileOutdatedLabel;
        TextBox ux_marketDataFile;
        Button  ux_marketDataFileBrowse;
        // technologies
        Label   ux_technologiesFileLabel;
        Label   ux_technologiesFileOutdatedLabel;
        TextBox ux_technologiesFile;
        Button  ux_technologiesFileBrowse;
        // parameters
        Label   ux_parametersFileLabel;
        Label   ux_parametersFileOutdatedLabel;
        TextBox ux_parametersFile;
        Button  ux_parametersFileBrowse;
        // scenarios
        Label   ux_scenariosFileLabel;
        Label   ux_scenariosFileOutdatedLabel;
        TextBox ux_scenariosFile;
        Button  ux_scenariosFileBrowse;
        //
        CheckBox ux_reloadAllFromDisk;

        #endregion

        static readonly string[][] DesiredTitles =
            new string[][] {
                               new string[] {"CAFE Market Data"},
                               new string[] {"CAFE Technologies"},
                               new string[] {"CAFE Parameters"},
                               new string[] {"CAFE Scenarios"}
                           };

        static readonly string[] FriendlyTitles =
            new string[] {
                             "Market-Data",
                             "Technologies",
                             "Parameters",
                             "Scenarios"
                         };

        const int FileCount              = 4;   // total number of files
        const int MarketDataID           = 0;
        const int TechnologiesID         = 1;
        const int ParametersID           = 2;
        const int ScenariosID            = 3;

        int    _parsing;                        // indicates whether the panel is parsing, verifying, or loading input files
        bool   _finalizeParsing;
        Thread _saveThread;                     // thread used for loading input data

        bool  [] _visibleFiles;                 // indicates whether the text-box for a specific input file is visible to the user
        bool  [] _verifiedFiles;                // indicates that the path entered in the input text-box is valid
        bool  [] _loadFailed;                   // indicates whether the file has failed to load (used for post-save when clearing visual tracking)
        object[] _data;                         // the input data objects loaded from file into memory (populated during save)

        /// <summary>Represents the input configuration settings for the modeling system.</summary>
        InputSettings _inputSettings;

        #endregion

    }
}
