using System;
using Volpe.Cafe.Generic;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents an object that stores compliance modeling data, as integer values, for each regulatory class.
    /// </summary>
    [Serializable]
    public class RCInt32 : RCValue<int>, ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="RCInt32"/> class.
        /// </summary>
        public RCInt32() : base() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCInt32"/> class using the specified initial value for all members.
        /// </summary>
        public RCInt32(int initialValue) : base(initialValue, initialValue, initialValue) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="RCInt32"/> class using the specified values.
        /// </summary>
        public RCInt32(int passengerCar, int lightTruck, int lightTruck2b3) : base(passengerCar, lightTruck, lightTruck2b3) { }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the sum of the two specified <see cref="RCInt32"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the two specified <see cref="RCInt32"/> values.</returns>
        public static RCInt32 operator +(RCInt32 value1, RCInt32 value2)
        {
            RCInt32 result = new RCInt32();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] + value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the sum of the specified <see cref="RCInt32"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="RCInt32"/> value and an <see cref="Int32"/> value.</returns>
        public static RCInt32 operator +(RCInt32 value1, int value2)
        {
            return value1 + new RCInt32(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the sum of the specified <see cref="RCInt32"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the sum.</param>
        /// <param name="value2">The second value to include in the sum.</param>
        /// <returns>The sum of the specified <see cref="RCInt32"/> value and an <see cref="Int32"/> value.</returns>
        public static RCInt32 operator +(int value1, RCInt32 value2)
        {
            return new RCInt32(value1) + value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCInt32 operator -(RCInt32 value1, RCInt32 value2)
        {
            RCInt32 result = new RCInt32();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] - value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCInt32 operator -(RCInt32 value1, int value2)
        {
            return value1 - new RCInt32(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the difference of the second value subtracted from the first value.
        /// </summary>
        /// <param name="value1">The value to subtract from.</param>
        /// <param name="value2">The value to subtract.</param>
        /// <returns>The difference of the second value subtracted from the first value.</returns>
        public static RCInt32 operator -(int value1, RCInt32 value2)
        {
            return new RCInt32(value1) - value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the product of the two specified <see cref="RCInt32"/> values.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the two specified <see cref="RCInt32"/> values.</returns>
        public static RCInt32 operator *(RCInt32 value1, RCInt32 value2)
        {
            RCInt32 result = new RCInt32();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] * value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the product of the specified <see cref="RCInt32"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="RCInt32"/> value and an <see cref="Int32"/> value.</returns>
        public static RCInt32 operator *(RCInt32 value1, int value2)
        {
            return value1 * new RCInt32(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the product of the specified <see cref="RCInt32"/> value and an
        /// <see cref="Int32"/> value.
        /// </summary>
        /// <param name="value1">The first value to include in the product.</param>
        /// <param name="value2">The second value to include in the product</param>
        /// <returns>The product of the specified <see cref="RCInt32"/> value and an <see cref="Int32"/> value.</returns>
        public static RCInt32 operator *(int value1, RCInt32 value2)
        {
            return new RCInt32(value1) * value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCInt32 operator /(RCInt32 value1, RCInt32 value2)
        {
            RCInt32 result = new RCInt32();
            for (int i = 0; i < result.Items.Length; i++)
            {
                result.Items[i] = value1.Items[i] / value2.Items[i];
            }
            return result;
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCInt32 operator /(RCInt32 value1, int value2)
        {
            return value1 / new RCInt32(value2);
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value that is the first value divided by the second value.
        /// </summary>
        /// <param name="value1">The value of the numerator.</param>
        /// <param name="value2">The value of the denominator.</param>
        /// <returns>The first value divided by the second value.</returns>
        public static RCInt32 operator /(int value1, RCInt32 value2)
        {
            return new RCInt32(value1) / value2;
        }

        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value, which is the copy of the specified <see cref="RCInt32"/> value.
        /// </summary>
        /// <param name="value">The value to copy.</param>
        /// <returns>A copy of the specified <see cref="RCInt32"/> value.</returns>
        public static RCInt32 operator +(RCInt32 value)
        {
            return RCInt32.Zero + value;
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value, which is the negative of the specified <see cref="RCInt32"/> value.
        /// </summary>
        /// <param name="value">The value to negate.</param>
        /// <returns>The specified <see cref="RCInt32"/> value negated.</returns>
        public static RCInt32 operator -(RCInt32 value)
        {
            return RCInt32.Zero - value;
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value, which is one more than the specified value (increment).
        /// </summary>
        /// <param name="value">The value to increment by one.</param>
        /// <returns>The specified value, incremented by one.</returns>
        public static RCInt32 operator ++(RCInt32 value)
        {
            return value + RCInt32.One;
        }
        /// <summary>
        /// Returns a new <see cref="RCInt32"/> value, which is one less than the specified value (decrement).
        /// </summary>
        /// <param name="value">The value to decrement by one.</param>
        /// <returns>The specified value, decremented by one.</returns>
        public static RCInt32 operator --(RCInt32 value)
        {
            return value - RCInt32.One;
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="RCInt32"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCInt32"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a copy of the current <see cref="RCInt32"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="RCInt32"/>.</returns>
        public new RCInt32 Clone()
        {
            RCInt32 value = new RCInt32();
            this.CopyTo(value);
            return value;
        }

        #endregion

        /// <summary>
        /// Returns the string representation of this <see cref="RCInt32"/> instance, using the specified format for all elements.
        /// </summary>
        /// <param name="format">A format string.</param>
        /// <returns>The string representation of the <see cref="RCInt32"/> instance, using the specified format.</returns>
        public string ToString(string format)
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString(format));
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Returns the compliance modeling data value for the specified regulatory class, or the <see cref="Total"/> of all
        /// regulatory classes if <see cref="RegulatoryClass.All"/> is specified.
        /// </summary>
        /// <param name="regClass">The regulatory class for which to obtain the value.</param>
        /// <returns>The compliance modeling data value for the specified regulatory class.</returns>
        public int GetValue(RegulatoryClass regClass)
        {
            return (regClass == RegulatoryClass.All) ? this.Total : this[regClass];
        }

        #endregion

        #region /*** Properties ***/

        //----- static values -----
        /// <summary>Gets the <see cref="RCInt32"/> value whose elements are all 0.</summary>
        public static RCInt32 Zero { get { return new RCInt32(0); } }
        /// <summary>Gets the <see cref="RCInt32"/> value whose elements are all 1.</summary>
        public static RCInt32 One { get { return new RCInt32(1); } }

        //----- instance values -----
        /// <summary>Gets the sum of the compliance modeling data of all regulatory classes.</summary>
        public int Total
        {
            get
            {
                int total = 0;
                for (int i = 0; i < this.Items.Length; i++)
                {
                    total += this.Items[i];
                }
                return total;
            }
        }

        #endregion

    }
}
