using System;

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides information about functional form types supported by the CAFE Model.
    /// </summary>
    public sealed class FunctionInformation
    {

        #region /*** Variables ***/

        static int[] SupportedFunctions = { 1, 2, 3, 4, 5, 6, 7, 8, 99, 206, 207, 208 };
        static int[] CoefficientCount   = { 1, 4, 4, 3, 3, 4, 4, 8,   8,   8,   9 };
        static string[][] CoefficientNames =
            {
                new string[] {"A"},                                         //   1
                new string[] {"A", "B", "C", "D"},                          //   2
                new string[] {"A", "B", "C", "D"},                          //   3
                new string[] {"A", "B", "C"},                               //   4
                new string[] {"A", "B", "C"},                               //   5
                new string[] {"A", "B", "C", "D"},                          //   6
                new string[] {"A", "B", "C", "D"},                          //   7
                new string[] {"A", "B", "C", "D", "E", "F", "G", "H"},      //   8
                new string[] {"A", "B", "C", "D", "E", "F", "G", "H"},      // 206
                new string[] {"A", "B", "C", "D", "E", "F", "G", "H"},      // 207
                new string[] {"A", "B", "C", "D", "E", "F", "G", "H", "I"}  // 208
            };
        //
        static int[] FlatStandard    = { 1, 99 };
        static int[] Logistic        = { 2, 3 };
        static int[] Exponential     = { 4, 5 };
        static int[] Linear          = { 6, 7, 8, 206, 207, 208 };
        //
        static int[] AreaBased       = { 2, 4, 6, 206 };
        static int[] WeightBased     = { 3, 5, 7, 207 };
        static int[] WorkFactorBased = {       8, 208 };

        #endregion

        #region /*** Methods ***/

        static int FindFunction(int fncType, params int[] values)
        {
            for (int i = 0; i < values.Length; i++)
            {
                if (fncType == values[i]) { return i; }
            }
            return -1;
        }
        static bool HasFunction(int fncType, params int[] values)
        {
            return (FindFunction(fncType, values) != -1);
        }

        /// <summary>
        /// Determines whether the function with the specified type code is supported by the modeling system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type is supported by the model; false, otherwise.</returns>
        public static bool IsSupportedFunction(int fncType)
        {
            return HasFunction(fncType, SupportedFunctions);
        }

        /// <summary>
        /// Determines whether the function with the specified type code represents a flat standard system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type represents a flat standard system; false, otherwise.</returns>
        public static bool IsFlatStandard(int fncType)
        {
            return HasFunction(fncType, FlatStandard);
        }
        /// <summary>
        /// Determines whether the function with the specified type code represents a logistic functional system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type represents a logistic functional system; false, otherwise.</returns>
        public static bool IsLogisticFunction(int fncType)
        {
            return HasFunction(fncType, Logistic);
        }
        /// <summary>
        /// Determines whether the function with the specified type code represents an exponential functional system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type represents an exponential functional system; false, otherwise.</returns>
        public static bool IsExponentialFunction(int fncType)
        {
            return HasFunction(fncType, Exponential);
        }
        /// <summary>
        /// Determines whether the function with the specified type code represents a linear functional system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type represents a linear functional system; false, otherwise.</returns>
        public static bool IsLinearFunction(int fncType)
        {
            return HasFunction(fncType, Linear);
        }

        /// <summary>
        /// Determines whether the function with the specified type code represents an area-based functional system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type represents an area-based functional system; false, otherwise.</returns>
        public static bool IsAreaBasedFunction(int fncType)
        {
            return HasFunction(fncType, AreaBased);
        }
        /// <summary>
        /// Determines whether the function with the specified type code represents a weight-based functional system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type represents a weight-based functional system; false, otherwise.</returns>
        public static bool IsWeightBasedFunction(int fncType)
        {
            return HasFunction(fncType, WeightBased);
        }
        /// <summary>
        /// Determines whether the function with the specified type code represents a work-factor-based functional system.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>true, if the specified function type represents a work-factor-based functional system; false, otherwise.</returns>
        public static bool IsWorkFactorBasedFunction(int fncType)
        {
            return HasFunction(fncType, WorkFactorBased);
        }

        /// <summary>
        /// Returns the number of functional coefficients in the function with the specified type code, or -1 if the function
        /// with the specified type code is not supported.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>The number of functional coefficients in the function with the specified type code, or -1 if the function
        ///   with the specified type code is not supported.</returns>
        public static int GetCoefficientCount(int fncType)
        {
            int fncIndex = FindFunction(fncType, SupportedFunctions);
            return (fncIndex == -1) ? -1 : CoefficientCount[fncIndex];
        }

        /// <summary>
        /// Returns an array of strings representing the coefficient names for the function with the specified type code, or null
        /// if the function with the specified type code is not supported.
        /// </summary>
        /// <param name="fncType">The type of the function to check.</param>
        /// <returns>An array of strings representing the coefficient names for the function with the specified type code, or
        ///   null if the function with the specified type code is not supported.</returns>
        public static string[] GetCoefficientNames(int fncType)
        {
            int fncIndex = FindFunction(fncType, SupportedFunctions);
            return (fncIndex == -1) ? null : CoefficientNames[fncIndex];
        }

        #endregion

    }
}
