using System;
using Volpe.Cafe.IO;

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Provides operating modes settings for the modeling system.
    /// </summary>
    [Serializable]
    public class OperatingModes : ISettings, ICloneable
    {

        #region /*** Methods ***/

        #region /* ICloneable/ISettings Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OperatingModes"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OperatingModes"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OperatingModes"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OperatingModes"/>.</returns>
        ISettings ISettings.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="OperatingModes"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="OperatingModes"/>.</returns>
        OperatingModes Clone()
        {
            OperatingModes value = new OperatingModes();
            Clone(this, value);
            return value;
        }
        static void Clone(OperatingModes src, OperatingModes dst)
        {
            dst.NoFines                           = src.NoFines;
            dst.Overcomply                        = src.Overcomply;
            dst.MultiYearModeling                 = src.MultiYearModeling;
            dst.MultiYearStartYear                = src.MultiYearStartYear;
            dst.AllowCreditTrading                = src.AllowCreditTrading;
            dst.DynamicFleetShare                 = src.DynamicFleetShare;
            dst.FleetAnalysis                     = src.FleetAnalysis;
            //
            dst.Backfill                          = src.Backfill;
            dst.BackfillMissingBEP                = src.BackfillMissingBEP;
            //
            dst.IgnoreYearAvailable               = src.IgnoreYearAvailable;
            dst.IgnorePhaseIn                     = src.IgnorePhaseIn;
            dst.IgnoreRefreshRedesign             = src.IgnoreRefreshRedesign;
            //
            dst.FuelPriceEstimates                = src.FuelPriceEstimates;
            dst.CO2Estimates                      = src.CO2Estimates;
            dst.ConsumerBenefitsScale             = src.ConsumerBenefitsScale;
        }

        #endregion

        #region /* ISettings Members */

        /// <summary>
        /// Sets the initial state for the <see cref="ISettings"/> object to the default values.
        /// </summary>
        void ISettings.SetDefaultSettings()
        {
            // compliance modeling settings
            this.NoFines               = false;
            this.Overcomply            = true;
            this.MultiYearModeling     = true;
            this.MultiYearStartYear    = 2016;
            this.AllowCreditTrading    = true;
            this.DynamicFleetShare     = false;
            this.FleetAnalysis         = false;
            // technology settings
            this.Backfill              = true;
            this.BackfillMissingBEP    = true;
            // supplemental settings
            this.IgnoreYearAvailable   = false;
            this.IgnorePhaseIn         = false;
            this.IgnoreRefreshRedesign = false;
            // hidden settings
            this.FuelPriceEstimates    = Estimates.Average;
            this.CO2Estimates          = Estimates.Average;
            this.ConsumerBenefitsScale = 1.0;
        }

        /// <summary>
        /// Updates this <see cref="OperatingModes"/> object with values from the specified new settings.
        /// </summary>
        /// <param name="newSettings">An <see cref="OperatingModes"/> object containing new runtime settings for this instance.</param>
        void ISettings.UpdateSettings(ISettings newSettings)
        {
            this.UpdateSettings((OperatingModes)newSettings);
        }
        /// <summary>
        /// Updates this <see cref="OperatingModes"/> object with values from the specified new settings.
        /// </summary>
        /// <param name="newSettings">An <see cref="OperatingModes"/> object containing new runtime settings for this instance.</param>
        public void UpdateSettings(OperatingModes newSettings)
        {
            Clone(newSettings, this);
        }

        /// <summary>
        /// Writes settings data to the summary.txt log file.
        /// </summary>
        public void WriteSummaryLog(LogWriter writer)
        {
            if (writer.Summary == null) { return; }
            string outStr = "--- Operating Modes ---" +
                "\n  NoFines="                        + this.NoFines                     +
                "\n  Overcomply="                     + this.Overcomply                  +
                "\n  MultiYearModeling="              + this.MultiYearModeling           +
                                                        ((this.MultiYearModeling) ? " (" + this.MultiYearStartYear + ")" : "") +
                "\n  AllowCreditTrading="             + this.AllowCreditTrading          +
                "\n  DynamicFleetShare="              + this.DynamicFleetShare           +
                "\n  FleetAnalysis="                  + this.FleetAnalysis               +
                "\n  Backfill="                       + this.Backfill                    +
                "\n  BackfillMissingBEP="             + this.BackfillMissingBEP          +
                "\n  IgnoreYearAvailable="            + this.IgnoreYearAvailable         +
                "\n  IgnorePhaseIn="                  + this.IgnorePhaseIn               +
                "\n  IgnoreRefreshRedesign="          + this.IgnoreRefreshRedesign       +
                "\n  FuelPriceEstimates="             + this.FuelPriceEstimates          +
                "\n  CO2Estimates="                   + this.CO2Estimates                +
                "\n  ConsumerBenefitsScale="          + this.ConsumerBenefitsScale;
            writer.Summary.WriteLine(outStr + "\n");
        }

        #endregion

        #endregion


        #region /*** Variables ***/

        #region /* Compliance Modeling Settings */

        /// <summary>Specifies that the model should continue to apply technologies until there are no more fines remaining.
        ///   This setting overriddes the manufacturer specific willingness to pay fines.</summary>
        public bool NoFines;

        /// <summary>Specifies that the model should continue to apply technologies after reaching compliance during a given
        ///   model year.</summary>
        public bool Overcomply;

        /// <summary>Specifies whether the modeling system operates on a multi-year basis, redesign-tied technologies which are
        ///   unused in a redesign year to remain available for use in subsequent years.  When used, the tech-cost is back-dated
        ///   to the redesign year but the FC improvements remain from the present going forward.</summary>
        public bool MultiYearModeling;
        /// <summary>When multi-year modeling is enabled, specifies the first year that multi-year applies.  Vehicles with
        ///   redesign years below this value will not be considered during multi-year lingering.</summary>
        public int MultiYearStartYear;

        /// <summary>Specifies whether the modeling system should allow manufacturers to trade or transfer credits between fleets,
        ///   or to carry credits forward or back between model years.</summary>
        public bool AllowCreditTrading;

        /// <summary>Specifies whether the model should dynamically adjust the present year's PC/LT fleet share based on achieved
        ///   CAFE levels from the previous year, the PC share from the previous year, and the present year's fuel prices.</summary>
        public bool DynamicFleetShare;

        /// <summary>Specifies whether the model should perform fleet analysis, evaluating modeling effects for historic and
        ///   forcast model years (before the first compliance model year as well as after the last compliance model year).</summary>
        public bool FleetAnalysis;

        #endregion

        #region /* Technology Settings */

        /// <summary>Specifies whether the model should backfill the preceding technologies.</summary>
        public bool Backfill;
        /// <summary>Specifies whether the model should backfill missing technologies on the basic engine path.</summary>
        public bool BackfillMissingBEP;

        #endregion

        #region /* Supplemental Settings */

        /// <summary>Specifies that the model should ignore the year availability setting when considering technologies for
        ///   applicability.</summary>
        public bool IgnoreYearAvailable;
        /// <summary>Specifies that the model should ignore technology specific phase-in caps when considering a technology for
        ///   applicability.</summary>
        public bool IgnorePhaseIn;
        /// <summary>Specifies that the model should ignore vehicle-specific refresh and redesign schedules.</summary>
        public bool IgnoreRefreshRedesign;

        #endregion

        #region /* Hidden Settings */

        /// <summary>Specifies whether to use low, average, or high retail fuel price and pre-tax fuel price estimates.</summary>
        public Estimates FuelPriceEstimates;

        /// <summary>Specifies whether to use low, average, high, or very high estimates for economic costs arising from Carbon
        ///   Dioxide damage.</summary>
        public Estimates CO2Estimates;

        /// <summary>Specifies the percentage by which to scale the private consumer benefits for the effects calculations.</summary>
        public double ConsumerBenefitsScale;

        #endregion

        #endregion

    }
}
