#region << Using Directives >>
using System;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Generic;
#endregion

namespace Volpe.Cafe.Settings
{
    /// <summary>
    /// Provides information relating to a single modeling scenario.
    /// </summary>
    [Serializable]
    public sealed class Scenario : ICloneable
    {

        #region /*** Ctors ***/

        // Private constructor (for cloning).
        Scenario() { }
        /// <summary>
        /// Initializes a new instance of the <see cref="Scenario"/> class.
        /// </summary>
        internal Scenario(int index, string description, RCObject<ScenarioInfo> scenInfo)
        {
            this.Index       = index;
            this.IsBaseline  = (index == 0);
            this.Description = description;
            this.ScenInfo    = scenInfo;
        }

        #endregion

        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Scenario"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Scenario"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Scenario"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Scenario"/>.</returns>
        public Scenario Clone()
        {
            Scenario scen = new Scenario();
            //
            scen.Index       = this.Index;
            scen.IsBaseline  = this.IsBaseline;
            scen.Description = this.Description;
            scen.ScenInfo    = this.ScenInfo.Clone();
            //
            return scen;
        }

        #endregion

        /// <summary>
        /// Returns the string representation of this <see cref="Scenario"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="Scenario"/> instance.</returns>
        public override string ToString()
        {
            return this.Index + ". " + Interaction.GetTitleCase(this.Description);
        }
        /// <summary>
        /// Returns the string representation of this <see cref="Scenario"/> instance.
        /// </summary>
        /// <param name="descrLength">Specifies the maximum number of characters to return from the description field.</param>
        /// <returns>The string representation of the <see cref="Scenario"/> instance.</returns>
        public string ToString(int descrLength)
        {
            int len = this.Description.Length;
            bool outOfBounds = (descrLength < len);
            descrLength = (outOfBounds) ? descrLength - 3 : len;
            string descr = Interaction.GetTitleCase(this.Description.Substring(0, descrLength));
            if (outOfBounds) { descr += " ..."; }
            //
            return this.Index + ". " + descr;
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets the index of the compliance scenario.</summary>
        public int Index { get; private set; }
        /// <summary>Gets whether the compliance scenario is the baseline.</summary>
        public bool IsBaseline { get; private set; }
        /// <summary>Gets the description of the compliance scenario.</summary>
        public string Description { get; private set; }
        /// <summary>Gets the compliance scenario attributes for each regulatory class.</summary>
        public RCObject<ScenarioInfo> ScenInfo { get; private set; }

        #endregion

    }
}
