using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using Volpe.Cafe.Utils;

namespace Volpe.Cafe
{
    /// <summary>
    /// Provides infromation for the various known vehicle technologies.
    /// </summary>
    [Serializable]
    public sealed class TechnologyIndexes
    {

        #region /*** Nested Types ***/

        /// <summary>
        /// Provides configuration flags for a technology.
        /// </summary>
        [Serializable]
        [Flags]
        internal enum TechnologyFlags : ulong
        {
            /// <summary>Specifies a technology does not have any flags. This is the initial state of an "empty" technology,
            ///   however, it is not a valid for technologies selected for modeling.</summary>
            None = 0,
            // app-level
            BaselineOnly                = 1UL <<  0,
            VehicleLevel                = 1UL <<  1,
            PlatformLevel               = 1UL <<  2,
          //PowertrainLevel             = 1UL <<  3,
            EngineLevel                 = 1UL <<  4,
            TransmissionLevel           = 1UL <<  5,
            // app-schedule
            RefreshBased                = 1UL <<  6,
            RedesignBased               = 1UL <<  7,
            // cost-basis
            CostPerPound                = 1UL <<  8,
            CostPerCylinder             = 1UL <<  9,
            CostPerBank                 = 1UL << 10,
            // technology paths
            BasicEnginePath             = 1UL << 21,
            TurboEnginePath             = 1UL << 22,
            AdvancedEnginePath          = 1UL << 23,
            DieselEnginePath            = 1UL << 24,
            MTransmissionPath           = 1UL << 25,
            ATransmissionPath           = 1UL << 26,
            ElectrificationPath         = 1UL << 27,
            HybridElectricPath          = 1UL << 28,
            AdvancedHybridElectricPath  = 1UL << 29,
            DLRPath                     = 1UL << 30,
            ROLLPath                    = 1UL << 31,
            MRPath                      = 1UL << 32,
            AEROPath                    = 1UL << 33,
            // powertrain-conversion
            C2Diesel                    = 1UL << 41,
            C2CNG                       = 1UL << 42,
            C2LNG                       = 1UL << 43,
            C2LPG                       = 1UL << 44,
            C2MHEV                      = 1UL << 45,
            C2SHEV                      = 1UL << 46,
            C2PHEV                      = 1UL << 47,
            C2EV                        = 1UL << 48,
            C2FCV                       = 1UL << 49,
            // other/misc flags
            /// <summary>Indicates whether a technology should be automatically applied to a vehicle as soon as it becomes available.</summary>
            ForcedApplication           = 1UL << 51,
            /// <summary>Indicates whether the fuel consumption improvement of a technology should increase as a function of time.</summary>
            FCTimeBased                 = 1UL << 52,
            /// <summary>Indicates whether a technology, when it is used on a vehicle, modifies vehicle-specific FCAdjKey.</summary>
            FCAdjKey                    = 1UL << 53
        }

        /// <summary>
        /// Describes a predefined technology.
        /// </summary>
        [Serializable]
        public struct TechnologyDefinition
        {
            /// <summary>
            /// Initializes a new <see cref="TechnologyDefinition"/> instance.
            /// </summary>
            internal TechnologyDefinition(string name, int index, TechnologyFlags flags) : this()
            {
                this.Name  = name;
                this.Index = index;
                this.Flags = flags;
            }

            // methods
            /// <summary>
            /// Determines whether the specified technology flag is set on this technology definition.
            /// </summary>
            /// <param name="flag">The technology flag to check.</param>
            /// <returns>true, if the specified technology flag is set on this technology definition; false, otherwise.</returns>
            internal bool IsFlagSet(TechnologyFlags flag)
            {
                return ((this.Flags & flag) == flag);
            }

            // properties
            /// <summary>Gets the name of a technology.</summary>
            public string Name { get; private set; }
            /// <summary>Gets the unique zero-based index of a technology.</summary>
            public int Index { get; private set; }
            /// <summary>Gets the descriptive flags of a technology.</summary>
            TechnologyFlags Flags { get; set; }
        }

        #endregion

        #region /*** Ctors ***/

        static TechnologyIndexes()
        {
            // initialize technology definitions and bind to known variables
            TechnologyDefinitions   = InitializeTechnologyDefinitions();
            TechnologyCount         = BindTechnologyIndexes();
            if (TechnologyCount != TechnologyDefinitions.Count) { ThrowHelper.InternalModelError(); }

            // assign pre-defined arrays
            // app-level
            EngineLevel             = FilterTechnologies(TechnologyFlags.EngineLevel);
            TransmissionLevel       = FilterTechnologies(TechnologyFlags.TransmissionLevel);
            PlatformLevel           = FilterTechnologies(TechnologyFlags.PlatformLevel);
            VehicleLevel            = FilterTechnologies(TechnologyFlags.VehicleLevel);

            // build a list of tech names
            TechNames               = BuildNamesList(out TechNamesCSV);
            TechNamesEngine         = FilterTechnologyNames(TechnologyFlags.EngineLevel);
            TechNamesTransmission   = FilterTechnologyNames(TechnologyFlags.TransmissionLevel);
            TechNamesPlatform       = FilterTechnologyNames(TechnologyFlags.PlatformLevel);
            TechNamesVehicle        = FilterTechnologyNames(TechnologyFlags.VehicleLevel);
        }

        static List<TechnologyDefinition> InitializeTechnologyDefinitions()
        {
            List<TechnologyDefinition> techs = new List<TechnologyDefinition>();
            int index = 0;  // indexes start at 0
            //
            techs.Add(new TechnologyDefinition("SOHC",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.BaselineOnly | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("DOHC",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.BaselineOnly | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("OHV",       index++, TechnologyFlags.EngineLevel | TechnologyFlags.BaselineOnly | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("TEFRI",     index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.FCTimeBased));
            techs.Add(new TechnologyDefinition("LUBEFR1",   index++, TechnologyFlags.EngineLevel | TechnologyFlags.RefreshBased | TechnologyFlags.CostPerCylinder | TechnologyFlags.BasicEnginePath));
            techs.Add(new TechnologyDefinition("LUBEFR2",   index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerCylinder | TechnologyFlags.BasicEnginePath));
            techs.Add(new TechnologyDefinition("LUBEFR3",   index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerCylinder | TechnologyFlags.BasicEnginePath));
            techs.Add(new TechnologyDefinition("VVT",       index++, TechnologyFlags.EngineLevel | TechnologyFlags.RefreshBased | TechnologyFlags.CostPerBank | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("VVL",       index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerCylinder | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("SGDI",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerCylinder | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("DEAC",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("HCR",       index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.BasicEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("HCRP",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.BasicEnginePath));
            techs.Add(new TechnologyDefinition("TURBO1",    index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.TurboEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("SEGR",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.TurboEnginePath));
            techs.Add(new TechnologyDefinition("DWSP",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.TurboEnginePath));
            techs.Add(new TechnologyDefinition("TURBO2",    index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.TurboEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("CEGR1",     index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.TurboEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("CEGR1P",    index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.TurboEnginePath));
            techs.Add(new TechnologyDefinition("CEGR2",     index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.TurboEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("HCR2",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.AdvancedEnginePath));
            techs.Add(new TechnologyDefinition("CNG",       index++, TechnologyFlags.EngineLevel | TechnologyFlags.BaselineOnly | TechnologyFlags.C2CNG | TechnologyFlags.AdvancedEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("ADSL",      index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2Diesel | TechnologyFlags.DieselEnginePath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("TURBODSL",  index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2Diesel | TechnologyFlags.DieselEnginePath));
            techs.Add(new TechnologyDefinition("DWSPDSL",   index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2Diesel | TechnologyFlags.DieselEnginePath));
            techs.Add(new TechnologyDefinition("EFRDSL",    index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2Diesel | TechnologyFlags.DieselEnginePath));
            techs.Add(new TechnologyDefinition("CLCDSL",    index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2Diesel | TechnologyFlags.DieselEnginePath));
            techs.Add(new TechnologyDefinition("LPEGRDSL",  index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2Diesel | TechnologyFlags.DieselEnginePath));
            techs.Add(new TechnologyDefinition("DSIZEDSL",  index++, TechnologyFlags.EngineLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2Diesel | TechnologyFlags.DieselEnginePath));
            techs.Add(new TechnologyDefinition("MT5",       index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.BaselineOnly | TechnologyFlags.MTransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("MT6",       index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.MTransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("MT7",       index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.MTransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("TATI",      index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RefreshBased | TechnologyFlags.FCTimeBased));
            techs.Add(new TechnologyDefinition("AT5",       index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.BaselineOnly | TechnologyFlags.ATransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("AT6",       index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.ATransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("AT6P",      index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.ATransmissionPath));
            techs.Add(new TechnologyDefinition("AT8",       index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.ATransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("AT8P",      index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.ATransmissionPath));
            techs.Add(new TechnologyDefinition("DCT6",      index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.ATransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("DCT8",      index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.ATransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("CVT",       index++, TechnologyFlags.TransmissionLevel | TechnologyFlags.RedesignBased | TechnologyFlags.ATransmissionPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("EPS",       index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.ElectrificationPath));
            techs.Add(new TechnologyDefinition("IACC1",     index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.ElectrificationPath));
            techs.Add(new TechnologyDefinition("IACC2",     index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.ElectrificationPath));
            techs.Add(new TechnologyDefinition("SS12V",     index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.C2MHEV | TechnologyFlags.ElectrificationPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("BISG",      index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2MHEV | TechnologyFlags.ElectrificationPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("CISG",      index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2MHEV | TechnologyFlags.ElectrificationPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("SHEVP2",    index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2SHEV | TechnologyFlags.HybridElectricPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("SHEVPS",    index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2SHEV | TechnologyFlags.HybridElectricPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("PHEV30",    index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2PHEV | TechnologyFlags.AdvancedHybridElectricPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("PHEV50",    index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2PHEV | TechnologyFlags.AdvancedHybridElectricPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("BEV200",    index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2EV | TechnologyFlags.AdvancedHybridElectricPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("FCV",       index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RedesignBased | TechnologyFlags.C2FCV | TechnologyFlags.AdvancedHybridElectricPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("LDB",       index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.DLRPath));
            techs.Add(new TechnologyDefinition("SAX",       index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.DLRPath));
            techs.Add(new TechnologyDefinition("ROLL10",    index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.ROLLPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("ROLL20",    index++, TechnologyFlags.VehicleLevel | TechnologyFlags.RefreshBased | TechnologyFlags.ROLLPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("MR1",       index++, TechnologyFlags.PlatformLevel | TechnologyFlags.RefreshBased | TechnologyFlags.CostPerPound | TechnologyFlags.MRPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("MR2",       index++, TechnologyFlags.PlatformLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerPound | TechnologyFlags.MRPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("MR3",       index++, TechnologyFlags.PlatformLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerPound | TechnologyFlags.MRPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("MR4",       index++, TechnologyFlags.PlatformLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerPound | TechnologyFlags.MRPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("MR5",       index++, TechnologyFlags.PlatformLevel | TechnologyFlags.RedesignBased | TechnologyFlags.CostPerPound | TechnologyFlags.MRPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("AERO10",    index++, TechnologyFlags.PlatformLevel | TechnologyFlags.RefreshBased | TechnologyFlags.AEROPath | TechnologyFlags.FCAdjKey));
            techs.Add(new TechnologyDefinition("AERO20",    index++, TechnologyFlags.PlatformLevel | TechnologyFlags.RedesignBased | TechnologyFlags.AEROPath | TechnologyFlags.FCAdjKey));
            //
            return techs;
        }
        static int BindTechnologyIndexes()
        {
            int techCount = 0;
            FieldInfo[] fields = typeof(TechnologyIndexes).GetFields();
            for (int i = 0; i < fields.Length; i++)
            {
                FieldInfo f = fields[i];
                int index = FindTechnologyDefinition(f.Name.TrimStart('_'));
                if (index != -1)
                {
                    f.SetValue(null, index);
                    techCount++;
                }
            }
            return techCount;
        }
        static int[] FilterTechnologies(TechnologyFlags flag)
        {
            int   techCount = TechnologyDefinitions.Count;
            List<int> techs = new List<int>();
            //
            for (int i = 0; i < techCount; i++)
            {
                TechnologyDefinition t = TechnologyDefinitions[i];
                if (t.IsFlagSet(flag))
                {
                    techs.Add(t.Index);
                }
            }
            return techs.ToArray();
        }
        static string[] FilterTechnologyNames(TechnologyFlags flag)
        {
            int techCount = TechnologyDefinitions.Count;
            List<string> techs = new List<string>();
            //
            for (int i = 0; i < techCount; i++)
            {
                TechnologyDefinition t = TechnologyDefinitions[i];
                if (t.IsFlagSet(flag))
                {
                    techs.Add(t.Name);
                }
            }
            return techs.ToArray();
        }
        static string[] BuildNamesList(out string csv)
        {
            int    techCount = TechnologyDefinitions.Count;
            string[]   names = new string[techCount];
            StringBuilder sb = new StringBuilder(techCount * 5);
            //
            for (int i = 0; i < techCount; i++)
            {
                names[i] = TechnologyDefinitions[i].Name;
                sb.Append(names[i]);
                sb.Append(',');
            }
            csv = sb.ToString(0, sb.Length - 1);
            return names;
        }
        static int FindTechnologyDefinition(string techName)
        {
            for (int i = 0; i < TechnologyDefinitions.Count; i++)
            {
                if (TechnologyDefinitions[i].Name.Equals(techName, StringComparison.InvariantCultureIgnoreCase))
                {
                    return TechnologyDefinitions[i].Index;
                }
            }
            return -1;
        }

        #endregion

        #region /*** Methods ***/

        /// <summary>
        /// Returns the index of a technology based on the specified name.
        /// </summary>
        /// <param name="name">The name of a technology.</param>
        /// <returns>The index of the technology with the specified abbreviation, or -1 if not found.</returns>
        public static int GetIndex(string name)
        {
            for (int i = 0; i < TechnologyCount ; i++)
            {
                if (TechNames[i] == name) { return i; }
            }
            return -1;
        }
        /// <summary>
        /// Returns the name of a given technology based on the specified index.
        /// </summary>
        /// <param name="index">The index of a technology.</param>
        /// <returns>Technology name at index, or an empty string if index is out of range.</returns>
        public static string GetName(int index)
        {
            if ((index < 0) || (index >= TechnologyCount)) { return string.Empty; }
            return TechNames[index];
        }

        // baseline-vs-runtime-level technologies
        /// <summary>Returns whether the technology specified by the given index represents a baseline-only technology.</summary>
        public static bool IsBaselineOnly           (int index) { return CheckTechFlags(index, TechnologyFlags.BaselineOnly); }

        // engine-/transmission-/platform-level technologies
        /// <summary>Returns whether the technology specified by the given index represents an engine-level technology.</summary>
        public static bool IsEngineLevel            (int index) { return CheckTechFlags(index, TechnologyFlags.EngineLevel); }
        /// <summary>Returns whether the technology specified by the given index represents a transmission-level technology.</summary>
        public static bool IsTransmissionLevel      (int index) { return CheckTechFlags(index, TechnologyFlags.TransmissionLevel); }
        /// <summary>Returns whether the technology specified by the given index represents a platform-level technology.</summary>
        public static bool IsPlatformLevel          (int index) { return CheckTechFlags(index, TechnologyFlags.PlatformLevel); }
        /// <summary>Returns whether the technology specified by the given index represents a vehicle-level technology.</summary>
        public static bool IsVehicleLevel           (int index) { return CheckTechFlags(index, TechnologyFlags.VehicleLevel); }

        // vehicle-conversion technologies
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle's engine to run a diesel.</summary>
        public static bool IsConversionToDiesel     (int index) { return CheckTechFlags(index, TechnologyFlags.C2Diesel); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle's engine to run a compressed natural gas. </summary>
        public static bool IsConversionToCNG        (int index) { return CheckTechFlags(index, TechnologyFlags.C2CNG); }
        /// <summary> Returns whether the technology specified by the given index may convert a vehicle's engine to run a liquefied natural gas.</summary>
        public static bool IsConversionToLNG        (int index) { return CheckTechFlags(index, TechnologyFlags.C2LNG); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle's engine to run a liquefied petroleum gas.</summary>
        public static bool IsConversionToLPG        (int index) { return CheckTechFlags(index, TechnologyFlags.C2LPG); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle to a micro hybrid.</summary>
        public static bool IsConversionToMHEV       (int index) { return CheckTechFlags(index, TechnologyFlags.C2MHEV); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle to a strong hybrid.</summary>
        public static bool IsConversionToSHEV       (int index) { return CheckTechFlags(index, TechnologyFlags.C2SHEV); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle to a plug-in hybrid.</summary>
        public static bool IsConversionToPHEV       (int index) { return CheckTechFlags(index, TechnologyFlags.C2PHEV); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle to an electric vehicle.</summary>
        public static bool IsConversionToEV         (int index) { return CheckTechFlags(index, TechnologyFlags.C2EV); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle to a fuel cell vehicle.</summary>
        public static bool IsConversionToFCV        (int index) { return CheckTechFlags(index, TechnologyFlags.C2FCV); }
        /// <summary>Returns whether the technology specified by the given index may convert a vehicle to a zero-emissions vehicle.</summary>
        public static bool IsConversionToZEV        (int index) { return CheckTechFlags(index, TechnologyFlags.C2PHEV) ||
                                                                         CheckTechFlags(index, TechnologyFlags.C2EV  ) ||
                                                                         CheckTechFlags(index, TechnologyFlags.C2FCV ); }

        // technologies with cost-basis
        /// <summary>Returns whether the specified technology index represents a material substitution technology.</summary>
        public static bool IsCostPerPound           (int index) { return CheckTechFlags(index, TechnologyFlags.CostPerPound); }
        /// <summary>Returns whether the specified technology index represents a technology whose costs are based on engine configuration.</summary>
        public static bool IsCostPerBank            (int index) { return CheckTechFlags(index, TechnologyFlags.CostPerBank); }
        /// <summary>Returns whether the specified technology index represents a technology whose costs are based on the number of engine cylinders.</summary>
        public static bool IsCostPerCylinder        (int index) { return CheckTechFlags(index, TechnologyFlags.CostPerCylinder); }

        // tied to redesign/refresh
        /// <summary>Returns whether the specified technology index is tied to vehicle's redesign schedule.</summary>
        public static bool IsTiedToRedesign         (int index) { return CheckTechFlags(index, TechnologyFlags.RedesignBased); }
        /// <summary>Returns whether the specified technology index is tied to vehicle's refresh schedule.</summary>
        public static bool IsTiedToRefresh          (int index) { return CheckTechFlags(index, TechnologyFlags.RefreshBased); }

        // technology path checks
        /// <summary>Returns whether the specified technology index is within the Basic Engine technology path.</summary>
        public static bool IsBasicEnginePath        (int index) { return CheckTechFlags(index, TechnologyFlags.BasicEnginePath); }
        /// <summary>Returns whether the specified technology index is within the Turbocharged Engine technology path.</summary>
        public static bool IsTurboEnginePath        (int index) { return CheckTechFlags(index, TechnologyFlags.TurboEnginePath); }
        /// <summary>Returns whether the specified technology index is within the Advanced Engine technology path.</summary>
        public static bool IsAdvancedEnginePath     (int index) { return CheckTechFlags(index, TechnologyFlags.AdvancedEnginePath); }
        /// <summary>Returns whether the specified technology index is within the Diesel Engine technology path.</summary>
        public static bool IsDieselEnginePath       (int index) { return CheckTechFlags(index, TechnologyFlags.DieselEnginePath); }
        /// <summary>Returns whether the specified technology index is within the Manual Transmission technology path.</summary>
        public static bool IsManualTransmissionPath (int index) { return CheckTechFlags(index, TechnologyFlags.MTransmissionPath); }
        /// <summary>Returns whether the specified technology index is within the Automatic Transmission technology path.</summary>
        public static bool IsAutoTransmissionPath   (int index) { return CheckTechFlags(index, TechnologyFlags.ATransmissionPath); }
        /// <summary>Returns whether the specified technology index is within the Electrification technology path.</summary>
        public static bool IsElectrificationPath    (int index) { return CheckTechFlags(index, TechnologyFlags.ElectrificationPath); }
        /// <summary>Returns whether the specified technology index is within the Hybrid/Electric technology path.</summary>
        public static bool IsHybridElectricPath     (int index) { return CheckTechFlags(index, TechnologyFlags.HybridElectricPath); }
        /// <summary>Returns whether the specified technology index is within the Advanced Hybrid/Electric technology path.</summary>
        public static bool IsAdvancedHybridElecPath (int index) { return CheckTechFlags(index, TechnologyFlags.AdvancedHybridElectricPath); }
        /// <summary>Returns whether the specified technology index is within the Dynamic Load Reduction technology path.</summary>
        public static bool IsDLRPath                (int index) { return CheckTechFlags(index, TechnologyFlags.DLRPath); }
        /// <summary>Returns whether the specified technology index is within the Low Rolling Resistance Tires technology path.</summary>
        public static bool IsROLLPath               (int index) { return CheckTechFlags(index, TechnologyFlags.ROLLPath); }
        /// <summary>Returns whether the specified technology index is within the Mass Reduction technology path.</summary>
        public static bool IsMassReductionPath      (int index) { return CheckTechFlags(index, TechnologyFlags.MRPath); }
        /// <summary>Returns whether the specified technology index is within the Aerodynamic Improvements technology path.</summary>
        public static bool IsAEROPath               (int index) { return CheckTechFlags(index, TechnologyFlags.AEROPath); }

        // technologies with other/misc flags
        /// <summary>Returns whether the specified technology index is used with FC adjustment factors calculations.</summary>
        public static bool IsFCAdjKey               (int index) { return CheckTechFlags(index, TechnologyFlags.FCAdjKey); }
        /// <summary>Returns whether the specified technology index uses a time-based FC function.</summary>
        public static bool IsFCTimeBased            (int index) { return CheckTechFlags(index, TechnologyFlags.FCTimeBased); }
        /// <summary>Returns whether the specified technology index should be automatically applied to a vehicle.</summary>
        public static bool IsForcedApplication      (int index) { return CheckTechFlags(index, TechnologyFlags.ForcedApplication); }

        /// <summary>
        /// Returns an array of technology indexes that share phase-in caps with the specified technology index.  If the
        /// technology does not share phase-in caps with any other technology, a value of null is returned.
        /// </summary>
        /// <param name="index">The index of the technology to search for.</param>
        /// <returns>An array of technology indexes that share phase-in caps with the specified technology index; or null, if the
        ///   technology does not share phase-in caps with any other technology.</returns>
        public static int[] GetSharedPhaseInTechs(int index)
        {
            return null;
        }

        // ----- helper method -- checks if a specific tech index is part of the specified tech list -----
        /// <summary>
        /// Determines whether the specified technology flag is set for a technology with the given index.
        /// </summary>
        /// <param name="techIndex">The index of a technology to check.</param>
        /// <param name="flag">The technology flag to check.</param>
        /// <returns>true, if the specified technology flag is set for a technology with the given index; false, otherwise.</returns>
        static bool CheckTechFlags(int techIndex, TechnologyFlags flag)
        {
            return TechnologyDefinitions[techIndex].IsFlagSet(flag);
        }

        #endregion

        #region /*** Variables ***/

        static readonly List<TechnologyDefinition> TechnologyDefinitions;

        /// <summary>Represents the total count of technologies available to the modeling system.</summary>
        public static readonly int TechnologyCount;

        /// <summary>Represents an array of technology names.</summary>
        public static readonly string[] TechNames;
        /// <summary>Represents a string of technology names in Comma Separated Values (CSV) format.</summary>
        public static readonly string TechNamesCSV;

        #region /* Tech indexes and names for each application-level (eng, trn, plt, veh) */

        // engine-/transmission-/platform-/vehicle-level technologies
        /// <summary>Represents an array of technology indexes that are applicable to engines.</summary>
        public static readonly int[] EngineLevel;
        /// <summary>Represents an array of technology indexes that are applicable to transmissions.</summary>
        public static readonly int[] TransmissionLevel;
        /// <summary>Represents an array of technology indexes that are applicable to platforms.</summary>
        public static readonly int[] PlatformLevel;
        /// <summary>Represents an array of technology indexes that are applicable to vehicles.</summary>
        public static readonly int[] VehicleLevel;

        /// <summary>Represents an array of engine-level technology names.</summary>
        public static readonly string[] TechNamesEngine;
        /// <summary>Represents an array of transmission-level technology names.</summary>
        public static readonly string[] TechNamesTransmission;
        /// <summary>Represents an array of platform-level technology names.</summary>
        public static readonly string[] TechNamesPlatform;
        /// <summary>Represents an array of vehicle-level technology names.</summary>
        public static readonly string[] TechNamesVehicle;

        #endregion

        /// <summary>Represents an engine-level technology on a Basic Engine technology path.</summary>
        public static readonly int
            SOHC,
            DOHC,
            OHV,
            TEFRI,
            LUBEFR1,
            LUBEFR2,
            LUBEFR3,
            VVT,
            VVL,
            SGDI,
            DEAC,
            HCR,
            HCRP;
        /// <summary>Represents an engine-level technology on a Turbocharged Engine technology path.</summary>
        public static readonly int
            TURBO1,
            SEGR,
            DWSP,
            TURBO2,
            CEGR1,
            CEGR1P,
            CEGR2;
        /// <summary>Represents an engine-level technology on an Advanced Engine technology path.</summary>
        public static readonly int
            HCR2,
            CNG;
        /// <summary>Represents an engine-level technology on an Diesel Engine technology path.</summary>
        public static readonly int
            ADSL,
            TURBODSL,
            DWSPDSL,
            EFRDSL,
            CLCDSL,
            LPEGRDSL,
            DSIZEDSL;
        /// <summary>Represents a transmission-level technology on a Manual Transmission technology path.</summary>
        public static readonly int
            MT5,
            MT6,
            MT7;
        /// <summary>Represents a transmission-level technology on an Automatic Transmission technology path.</summary>
        public static readonly int
            TATI,
            AT5,
            AT6,
            AT6P,
            AT8,
            AT8P,
            DCT6,
            DCT8,
            CVT;
        /// <summary>Represents a vehicle-level technology on an Electrification technology path.</summary>
        public static readonly int
            EPS,
            IACC1,
            IACC2,
            SS12V,
            BISG,
            CISG;
        /// <summary>Represents a vehicle-level technology on a Hybrid/Electric technology path.</summary>
        public static readonly int
            SHEVP2,
            SHEVPS,
            PHEV30,
            PHEV50,
            BEV200,
            FCV;
        /// <summary>Represents a platform-level technology on a Dynamic Load Reduction technology path.</summary>
        public static readonly int
            LDB,
            SAX;
        /// <summary>Represents a platform-level technology on a Low Rolling Resistance Tires technology path.</summary>
        public static readonly int
            ROLL10,
            ROLL20;
        /// <summary>Represents a platform-level technology on a Mass Reduction technology path.</summary>
        public static readonly int
            MR1,
            MR2,
            MR3,
            MR4,
            MR5;
        /// <summary>Represents a platform-level technology on a Aerodynamic Improvements technology path.</summary>
        public static readonly int
            AERO10,
            AERO20;

        #endregion

    }
}
