using System;
using System.Collections.Generic;
using Volpe.Cafe;
using Volpe.Cafe.Utils;

namespace Volpe.Cafe.Data
{
    /// <summary>
    /// Represents a vehicle engine.
    /// </summary>
    [Serializable]
    public sealed class Engine : Component
    {

        #region /*** NestedTypes ***/

        /// <summary>
        /// Provides a description of engineering characteristics for an <see cref="Engine"/>.
        /// </summary>
        [Serializable]
        public sealed class CDescription
        {

            #region /*** Constructors ***/

            /// <summary>
            /// Initializes a new instance of the <see cref="Engine.CDescription"/> class.
            /// </summary>
            internal CDescription() : base() { }

            #endregion


            #region /*** Methods ***/

            /// <summary>
            /// Creates a new object that is a copy of the current <see cref="Engine.CDescription"/> instance.
            /// </summary>
            /// <returns>A new object that is a copy of this <see cref="Engine.CDescription"/>.</returns>
            internal CDescription Clone()
            {
                CDescription ed = new CDescription();
                //
                ed.Code                 = this.Code;
                ed.Manufacturer         = this.Manufacturer;
                ed.Family               = this.Family;
                ed.Configuration        = this.Configuration;
                ed.Fuel                 = this.Fuel;
                ed.EngineOilViscosity   = this.EngineOilViscosity;
                ed.Cycle                = this.Cycle;
                ed.FuelSystem           = this.FuelSystem;
                ed.Aspiration           = this.Aspiration;
                ed.ValvetrainDesign     = this.ValvetrainDesign;
                ed.ValveActuationTiming = this.ValveActuationTiming;
                ed.ValveLift            = this.ValveLift;
                ed.Cylinders            = this.Cylinders;
                ed.ValvesPerCylinder    = this.ValvesPerCylinder;
                ed.Deactivation         = this.Deactivation;
                ed.Displacement         = this.Displacement;
                ed.EngineSize           = this.EngineSize;
                //
                ed.UsedTechnologies      = (int[])Interaction.CloneArray(this.UsedTechnologies     , typeof(int));
                ed.AvailableTechnologies = (int[])Interaction.CloneArray(this.AvailableTechnologies, typeof(int));
                //
                return ed;
            }

            /// <summary>
            /// Returns the string representation of this <see cref="Engine.CDescription"/> instance.
            /// </summary>
            /// <returns>The string representation of the <see cref="Engine.CDescription"/> instance.</returns>
            public override string ToString()
            {
                return this.Displacement.ToString("0.0") + "L " +
                       this.Configuration + this.Cylinders + " (" +
                       this.Aspiration + ")";
            }

            #endregion


            #region /*** Variables ***/

            /// <summary>Represents the internal manufacturer specific engine code.</summary>
            public int Code;
            /// <summary>Specifies the manufacturer name that uses the engine.</summary>
            public string Manufacturer;
            /// <summary>Specifies the family that the engine belongs to.</summary>
            public string Family;
            /// <summary>Represents the engine configuration.</summary>
            public string Configuration;
            /// <summary>Represents the fuel type of the engine.</summary>
            public FuelType Fuel;
            /// <summary>Represents the engine's oil viscocity.</summary>
            public string EngineOilViscosity;
            /// <summary>Represents the cycle of the engine.</summary>
            public char Cycle;
            /// <summary>Represents the fuel system of the engine.</summary>
            public string FuelSystem;
            /// <summary>Represents the aspiration of the engine.</summary>
            public string Aspiration;
            /// <summary>Represents the valvetrain design of the engine.</summary>
            public string ValvetrainDesign;
            /// <summary>Represents the valve actuation/timing of the engine.</summary>
            public string ValveActuationTiming;
            /// <summary>Represents the valve lift of the engine.</summary>
            public string ValveLift;
            /// <summary>Represents the number of cylinders the engine has.</summary>
            public int Cylinders;
            /// <summary>Represents the number of valves per cylinder the engine has.</summary>
            public int ValvesPerCylinder;
            /// <summary>Represents the deactivation of the engine.</summary>
            public double Deactivation;
            /// <summary>Represents the engine displacement.</summary>
            public double Displacement;
            /// <summary>Represents the size of the engine.</summary>
            public EngineSize EngineSize;

            /// <summary>Represents the indexes of technologies initially installed on the baseline engine.</summary>
            public int[] UsedTechnologies;
            /// <summary>Represents the indexes of technologies initially available for applicability on the baseline engine.</summary>
            public int[] AvailableTechnologies;

            #endregion

        }

        #endregion


        #region /*** Constructors ***/

        // Private constructor used for internal usage (such as cloning).
        Engine() : base() { }

        /// <summary>
        /// Initializes a new instance of the <see cref="Engine"/> class, using the specified engine <see cref="Description"/>.
        /// </summary>
        /// <param name="description">A description of engineering characteristics for this <see cref="Engine"/>.</param>
        internal Engine(Engine.CDescription description) : base()
        {
            this._description = description;
        }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="Engine"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="Engine"/>.</returns>
        internal Engine Clone()
        {
            Engine eng = new Engine();
            eng._description = this._description.Clone();
            this.CopyTo(eng);
            return eng;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>
        /// Gets an array of technology indexes that are applicable to <see cref="Engine"/>s.
        /// </summary>
        /// <returns>An array of technology indexes that are applicable to <see cref="Engine"/>s.</returns>
        public override int[] TechnologyList { get { return TechnologyIndexes.EngineLevel; } }

        /// <summary>Gets a description of engineering characteristics for this <see cref="Engine"/>.</summary>
        public Engine.CDescription Description { get { return this._description; } }

        #endregion


        #region /*** Variables ***/

        Engine.CDescription _description;

        #endregion

    }
}
