using System;
using System.IO;
using Volpe.Cafe.Utils;
using Volpe.XlLib;

namespace Volpe.Cafe.IO
{
    /// <summary>
    /// Provides wrapper methods for opening and parsing files in Excel by using the <see cref="XlUtilities"/> class.
    /// </summary>
    /// <remarks>
    /// This classes uses an internal <see cref="XlUtilities"/> instance for underlying file access.
    /// </remarks>
    /// <seealso cref="XlUtilities"/>
    [Serializable]
    public class Input
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="Input"/> class, using the specified path for the location of the Excel
        /// workbook to open.
        /// </summary>
        /// <param name="path">The location of the Excel workbook to open.</param>
        public Input(string path) : this(path, null) { }
        /// <summary>
        /// Initializes a new instance of the <see cref="Input"/> class, using the specified path for the location of the Excel
        /// workbook to open, and the specified password.
        /// </summary>
        /// <param name="path">The location of the Excel workbook to load.</param>
        /// <param name="password">The input password to use for opening the workbook.</param>
        public Input(string path, string password)
        {
            if (!File.Exists(path)) { throw new FileNotFoundException(ErrorStrings.XlOpenFileNotFound, path); }
            //
            this.XlUtils    = new XlUtilities(path, true, password);
            this.SheetNames = this.XlUtils.GetWorksheetNames();
            this.ParseTitle();
        }

        #endregion


        #region /*** Methods ***/

        // Obtains title and version information from the the workbook title.
        void ParseTitle()
        {
            string[] title = this.XlUtils.WorkbookTitle.Split(',');
            if (title != null)
            {
                if (title.Length >= 1) { this.Title   = title[0]; }
                if (title.Length >= 2) { this.Version = (float)Interaction.GetDouble(title[1]); }
            }
        }

        /// <summary>
        /// Closes the workbook associated with this <see cref="Input"/> instance.
        /// </summary>
        public void Close()
        {
            this.XlUtils.Close(false);
        }

        /// <summary>
        /// Activates the worksheet specified by the given name, by obtaining the value of the worksheet into an internal buffer
        /// object, between the first cell and the last cell.
        /// </summary>
        /// <param name="name">The name of the worksheet to activate within the active workbook.</param>
        public void ActivateWorksheet(string name)
        {
            this.ActivateWorksheetInternal(name, XlCell.Empty, XlCell.Empty);
        }
        /// <summary>
        /// Activates the worksheet specified by the given index, by obtaining the value of the worksheet into an internal buffer
        /// object, between the first cell and the last cell.
        /// </summary>
        /// <param name="index">The zero-based index of the worksheet to activate within the active workbook.</param>
        public void ActivateWorksheet(int index)
        {
            this.ActivateWorksheetInternal(index + 1, XlCell.Empty, XlCell.Empty);
        }
        /// <summary>
        /// Activates the worksheet specified by the given name, by obtaining the value of the worksheet into an internal buffer
        /// object, between the specified first and last cells.
        /// </summary>
        /// <param name="name">The name of the worksheet to activate within the active workbook.</param>
        /// <param name="firstCell">The location of the first cell to look at when obtaining the value of the
        ///   worksheet.</param>
        /// <param name="lastCell">The location of the last cell to look at when obtaining the value of the worksheet.
        ///   </param>
        public void ActivateWorksheet(string name, XlCell firstCell, XlCell lastCell)
        {
            this.ActivateWorksheetInternal(name, firstCell, lastCell);
        }
        /// <summary>
        /// Activates the worksheet specified by the given index, by obtaining the value of the worksheet into an internal buffer
        /// object, between the specified first and last cells.
        /// </summary>
        /// <param name="index">The zero-based index of the worksheet to activate within the active workbook.</param>
        /// <param name="firstCell">The location of the first cell to look at when obtaining the value of the
        ///   worksheet.</param>
        /// <param name="lastCell">The location of the last cell to look at when obtaining the value of the worksheet.
        ///   </param>
        public void ActivateWorksheet(int index, XlCell firstCell, XlCell lastCell)
        {
            this.ActivateWorksheetInternal(index + 1, firstCell, lastCell);
        }
        void ActivateWorksheetInternal(object index, XlCell firstCell, XlCell lastCell)
        {
            // obtain raw buffer from Excel (Excel uses 1-based indexing)
            object[,] buf = this.XlUtils.GetData(index, firstCell, lastCell);
            // convert to 0-based indexing
            this.Rows    = (buf == null) ? 0 : buf.GetUpperBound(0);
            this.Columns = (buf == null) ? 0 : buf.GetUpperBound(1);
            this.Buffer  = new object[this.Rows, this.Columns];
            for (int i = 0; i < this.Rows; i++)
            {
                for (int j = 0; j < this.Columns; j++)
                {
                    this.Buffer[i, j] = buf[i + 1, j + 1];
                }
            }
        }


        #region /* Methods for parsing a cell into bool, int, double, string, etc. */

        /// <summary>
        /// Converts the value in the buffer at the specified row and column to its boolean equivalent.
        /// </summary>
        /// <param name="row">The row location of the value in the buffer to convert.</param>
        /// <param name="column">The column location of the value in the buffer to convert.</param>
        /// <returns>The boolean equivalent of the value in the buffer at the specified row and column.</returns>
        public bool GetBool(int row, int column)
        {
            return Interaction.GetBool(this.Buffer[row, column]);
        }
        /// <summary>
        /// Converts the value in the buffer at the specified row and column to its <see cref="Int32"/> equivalent.
        /// </summary>
        /// <param name="row">The row location of the value in the buffer to convert.</param>
        /// <param name="column">The column location of the value in the buffer to convert.</param>
        /// <returns>The Int32 equivalent of the value in the buffer at the specified row and column.</returns>
        public int GetInt32(int row, int column)
        {
            return Interaction.GetInt32(this.Buffer[row, column]);
        }
        /// <summary>
        /// Converts the value in the buffer at the specified row and column to its double equivalent.
        /// </summary>
        /// <param name="row">The row location of the value in the buffer to convert.</param>
        /// <param name="column">The column location of the value in the buffer to convert.</param>
        /// <returns>The double equivalent of the value in the buffer at the specified row and column.</returns>
        public double GetDouble(int row, int column)
        {
            return Interaction.GetDouble(this.Buffer[row, column]);
        }
        /// <summary>
        /// Converts the value in the buffer at the specified row and column to its char equivalent.
        /// </summary>
        /// <param name="row">The row location of the value in the buffer to convert.</param>
        /// <param name="column">The column location of the value in the buffer to convert.</param>
        /// <returns>The char equivalent of the value in the buffer at the specified row and column.</returns>
        public char GetChar(int row, int column)
        {
            return Interaction.GetChar(this.Buffer[row, column]);
        }
        /// <summary>
        /// Converts the value in the buffer at the specified row and column to its string equivalent.
        /// </summary>
        /// <param name="row">The row location of the value in the buffer to convert.</param>
        /// <param name="column">The column location of the value in the buffer to convert.</param>
        /// <returns>The string equivalent of the value in the buffer at the specified row and column.</returns>
        public string GetString(int row, int column)
        {
            return Interaction.GetString(this.Buffer[row, column]);
        }

        #endregion

        /// <summary>
        /// Scans the specified row for minimum and maximum year ranges, starting at the specified column.  The results are
        /// computed and stored in the minYear and maxYear parameters.
        /// </summary>
        /// <param name="row">The row to scan for minimum and maximum years.</param>
        /// <param name="startColumn">The column from which to start scanning.</param>
        /// <param name="minYear">Contains the resulting minimum year.</param>
        /// <param name="maxYear">Contains the resulting maximum year.</param>
        /// <exception cref="InputException">The minYear is less than <see cref="ModelYear.MinYear"/>.</exception>
        public void GetMinMaxYears(int row, int startColumn, out int minYear, out int maxYear)
        {
            // initialize to default values
            minYear = 0;
            maxYear = 0;

            if (row == -1)
            {
                throw new ArgumentException("The value of row must be greater than or equal to 0.", "row");
            }
            else if (startColumn == -1)
            {   // this section has not been found, thus we can't parse it
                return;
            }

            int cols = this.Columns;
            int numYears = 0;
            minYear = this.GetInt32(row, startColumn);
            if (minYear == 0)
            {   // could not parse the column hearder
                // try again, looking for the MY#### format
                minYear = Interaction.GetInt32(this.GetString(row, startColumn).Replace("MY", ""));
            }
            // check for valid min model year
            if (minYear < ModelYear.MinYear)
            {
                throw new InputException(this.XlUtils.Path, "Only model years " + ModelYear.MinYear + " and above are allowed.");
            }
            int prev = minYear - 1, current = minYear;
            // scan the columns
            while (prev + 1 == current)
            {
                numYears++;
                prev = current;
                if (startColumn + numYears >= cols)
                {   // reached the end -- there are no more columns
                    break;
                }
                current = this.GetInt32(row, startColumn + numYears);
                if (current == 0)
                {
                    current = Interaction.GetInt32(this.GetString(row, startColumn + numYears).Replace("MY", ""));
                }
            }
            maxYear = prev;
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the underlying <see cref="XlUtilities"/> object used to load the current Excel workbook.</summary>
        public XlUtilities XlUtils { get; private set; }

        /// <summary>Gets the workbook title of the Excel workbook associated with this Input instance.</summary>
        public string Title { get; private set; }
        /// <summary>Gets the workbook version of the Excel workbook associated with this Input instance.</summary>
        public float Version { get; private set; }

        /// <summary>Gets an array of worksheet names located within the current workbook.</summary>
        public string[] SheetNames { get; private set; }

        /// <summary>Gets the data of the active worksheet located within the current workbook.</summary>
        public object[,] Buffer { get; private set; }

        /// <summary>Gets the number of rows in the buffer of the active worksheet within the current workbook.</summary>
        public int Rows { get; private set; }
        /// <summary>Gets the number of columns in the buffer of the active worksheet within the current workbook.</summary>
        public int Columns { get; private set; }

        #endregion

    }
}
