﻿#region << Using Directives >>
using System;
using System.Collections.Generic;
using System.Text;
using Volpe.Cafe.Data;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
using Volpe.Cafe.Utils;
using Volpe.Cafe.Generic;
//
using TI = Volpe.Cafe.TechnologyIndexes;
#endregion

namespace Volpe.Cafe.IO.Reporting.CSV
{
    /// <summary>
    /// Creates and writes a Vehicles CSV Report into a plain text file.
    /// </summary>
    [Serializable]
    public sealed class CsvVehiclesReport : CsvReportingBase
    {

        #region /*** Ctors ***/

        static CsvVehiclesReport()
        {
            Header =
                "Scenario,Scenario Name,Model Year,Manufacturer,Veh Index,Veh Code,Brand,Model,Name Plate," +
                "Platform,Plt Version," +
                "Powertrain,Veh Power Initial,Veh Power," +
                "Eng Code,Eng Fuel Initial,Eng Type Initial,Eng Version,Eng Fuel,Eng Type," +
                "Trn Code,Trn Type Initial,Trn Version,Trn Type," +
                // fuel economy/fuel share -- initial, final rated, final compliance
                "FE Primary Initial,FE Secondary Initial,FE Initial,Fuel Initial,FS Initial," +
                "FE Primary Rated,FE Secondary Rated,FE Rated," +
                "FE Primary Compliance,FE Secondary Compliance,FE Compliance,Fuel,Fuel Share," +
                // regulatory, compliance, and veh characteristics
                "Veh Class,Reg Class,Tech Class,Safety Class,Redesign State,Refresh State,Platform Leader," +
                "Sales Initial,Sales,MSRP Initial,MSRP,k.Labor Hours," +
                "CW Initial,CW,TW Initial,TW,GVWR Initial,GVWR,GCWR Initial,GCWR,Footprint,Work Factor,FE Target,CO2 Target,CO2 Rating,ZEV Credits," +
                // vehicle costs
                "Tech Cost,Price Increase,Tax Credit,Consumer Valuation,Rel. Value Loss,Maint Cost,Repair Cost,Taxes/Fees,Financing,Insurance," +
                // technologies
                TechnologyIndexes.TechNamesCSV;
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="CsvVehiclesReport"/> class.
        /// </summary>
        /// <param name="path">The output path for the report.</param>
        /// <param name="append">Determines whether data is to be appended to the file.  If the file exists and append is false,
        ///   the file is overwritten. If the file exists and append is true, the data is appended to the file. Otherwise, a new
        ///   file is created.</param>
        public CsvVehiclesReport(string path, bool append) : base(path, append) { }

        #endregion


        #region /*** Methods ***/

        /// <summary>
        /// Returns the header used for the report.
        /// </summary>
        protected override string GetHeader()
        {
            return Header;
        }

        /// <summary>
        /// Parses the data into the Vehicles CSV Report.
        /// </summary>
        /// <param name="scen">The scenario to report on.</param>
        /// <param name="year">The model year to report on.</param>
        /// <param name="compliance">The compliance model that invoked the report generator.</param>
        protected override void ParseData_Internal(Scenario scen, ModelYear year, ICompliance compliance)
        {
            // get product plan and scenario data
            Industry         ppData   = compliance.Data;
            Industry         scenData = compliance.GetData(scen, year);
            ModelingSettings settings = compliance.Settings;

            List<Vehicle> ppVehs  = ppData  .Vehicles;
            List<Vehicle> snVehs  = scenData.Vehicles;
            int           yrIndex = year.Index;

            for (int i = 0, vehCount = ppVehs.Count; i < vehCount; i++)
            {
                Vehicle      pVeh = ppVehs[i];
                var          pVD  = pVeh.Description;
                Engine       pEng = pVeh.Engine;
                Transmission pTrn = pVeh.Transmission;
                Platform     pPlt = pVeh.Platform;
                //
                Vehicle      veh  = snVehs[i];
                var          vd   = veh.Description;
                var          vmd  = veh.ModelData;
                Engine       eng  = veh.Engine;
                Transmission trn  = veh.Transmission;
                Platform     plt  = veh.Platform;
                var          mmd  = veh.Manufacturer.ModelData;
                //
                RegulatoryClass vehRC    = veh.RegClass;
                double          vehSales = vd.Sales[yrIndex];
                string          mfrName  = Interaction.GetTitleCase(vd.Manufacturer, 4);

                this.Write(scen.Index);
                this.Write(Interaction.GetTitleCase(scen.Description));
                this.Write(year.Year);
                this.Write(mfrName);
                this.Write(i);
                this.Write(vd.Code);
                this.Write(Interaction.GetTitleCase(vd.Brand));
                this.Write(Interaction.GetTitleCase(vd.Model));
                this.Write(Interaction.GetTitleCase(vd.Nameplate));
                // platform info
                this.Write(pPlt.Name);
                if (plt != null)
                {
                    string pltVer = plt.GetComponentVersion(veh);
                    this.Write(pltVer);
                }
                // powertrain info
                this.Write(veh.HEVTypeString);
                this.Write(pVD.VehiclePower);
                this.Write(vd .VehiclePower);
                // engine info
                if (pEng == null) { this.WriteEmpty(3); }
                else
                {
                    this.Write(pEng.Description.Code);
                    this.Write(this.GetFuelTypeString(pEng.Description.Fuel, true));
                    this.Write(pEng.Description.ToString());
                }
                if (eng == null) { this.WriteEmpty(3); }
                else
                {
                    string engVer = eng.Description.Code + " (" + eng.GetComponentVersion(veh) + ")";
                    this.Write(engVer);
                    this.Write(this.GetFuelTypeString(eng.Description.Fuel, true));
                    this.Write(this.GetVehicleEngineType(vmd.TechUsed));
                }
                // transmission info
                if (pTrn == null) { this.WriteEmpty(2); }
                else
                {
                    this.Write(pTrn.Description.Code);
                    this.Write(pTrn.Description.ToString());
                }
                if (trn == null) { this.WriteEmpty(2); }
                else
                {
                    string trnVer = trn.Description.Code + " (" + trn.GetComponentVersion(veh) + ")";
                    this.Write(trnVer);
                    this.Write(trn.Description.ToString());
                }
                // fuel economy initial
                FuelValue initFE    = pVD.FuelEconomy;
                double    avgInitFE = Standards.GetAverageFuelEconomy(initFE, pVD.FuelShare, vehRC, scen, year);
                //
                this.Write(initFE       );
                this.Write(avgInitFE    );
                this.Write(this.GetFuelTypeString(pVD.FuelEconomy.FuelType, false));
                this.Write(pVD.FuelShare.ToString("0.00", true, true));
                // fuel economy new
                FuelValue ratedFE    = vd.FuelEconomy;
                FuelValue complFE    = Standards.GetAdjustedFuelEconomy(veh, scen, year, settings, true, true, true);
                double    avgRatedFE = Standards.GetAverageFuelEconomy (ratedFE, vd.FuelShare, vehRC, scen, year);
                double    avgComplFE = Standards.GetAverageFuelEconomy (complFE, vd.FuelShare, vehRC, scen, year);
                //
                this.Write(ratedFE     );
                this.Write(avgRatedFE  );
                this.Write(complFE     );
                this.Write(avgComplFE  );
                this.Write(this.GetFuelTypeString(vd.FuelEconomy.FuelType, false));
                this.Write(vd.FuelShare.ToString("0.00", true, true));
                // regulatory classification
                this.Write(veh.VehicleClass   .ToString());
                this.Write(veh.RegClass       .ToString());
                this.Write(veh.TechClassString           );
                this.Write(veh.SafetyClass    .ToString());
                // redesign/refresh
                this.Write((veh.IsAtRedesign(year)) ? "At Redesign" : null);
                this.Write((veh.IsAtRefresh (year)) ? "At Refresh"  : null);
                this.Write(((eng != null && eng.GetPlatformLeader() == veh) ? "E" : string.Empty) +
                           ((trn != null && trn.GetPlatformLeader() == veh) ? "T" : string.Empty) +
                           ((plt != null && plt.GetPlatformLeader() == veh) ? "P" : string.Empty));
                // sales/msrp
                this.Write(pVD.Sales[yrIndex]);
                if (vehSales == 0)
                {
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                }
                else
                {
                    this.Write(vehSales);
                    this.Write(pVD.Msrp);
                    this.Write(vd .Msrp + vmd.RegCost);
                    this.Write(veh.CalcLaborHours(year));
                }
                // cw/area/work-factor
                this.Write(pVD.CurbWeight);
                this.Write(vd .CurbWeight);
                this.Write(Standards.GetTestWeight(pVeh, scen, year));
                this.Write(Standards.GetTestWeight(veh , scen, year));
                this.Write(pVD.GVWR     );
                this.Write(vd .GVWR     );
                this.Write(pVD.GCWR     );
                this.Write(vd .GCWR     );
                this.Write(vd .Footprint);
                this.Write(Standards.GetWorkFactor      (veh, scen, year));
                this.Write(Standards.GetTarget          (veh, scen, year));
                this.Write(Standards.GetVehicleCO2Target(veh, scen, year, settings));
                this.Write(Standards.GetVehicleCO2Rating(veh, scen, year, settings, true, true, true));
                this.Write(veh.ModelData.ZEVCredits);
                // unit costs
                if (vehSales == 0)
                {
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                    this.Write(0);
                }
                else
                {
                    this.Write(vmd.TechCost           );
                    this.Write(vmd.RegCost            );
                    this.Write(vmd.TaxCredit          );
                    this.Write(vmd.ConsumerValuation  );
                    this.Write(vmd.RelativeLossOfValue);
                    this.Write(vmd.MaintenanceCost    );
                    this.Write(vmd.RepairCost         );
                    this.Write(vmd.TaxesAndFees       );
                    this.Write(vmd.FinancingCost      );
                    this.Write(vmd.InsuranceCost      );
                }
                // technology utilization/applicability
                for (int j = 0; j < TechnologyIndexes.TechnologyCount; j++)
                {
                    string techUsedStr = null;
                    // parse tech applicability state
                    techUsedStr =
                        ( vmd.TechUsed     [j] && !vmd.TechApplied  [j]) ? "U" :
                        ( vmd.TechUsed     [j] &&  vmd.TechInherited[j]) ? "I" :
                        ( vmd.TechUsed     [j] &&  vmd.TechApplied  [j]) ? "A" :
                        (!vmd.TechEnabled  [j]                         ) ? "X" :
                        ( mmd.TechExhausted[j]                         ) ? "P" : null;
                    // adjust techUsedStr if tech was superseded
                    if (vmd.TechSuperseded[j]) { techUsedStr += "S"; }
                    //
                    this.Write(techUsedStr);
                } // next j (technology)
                this.NewRow();
            } // next i (vehicle)
        }

        void Write(FuelValue value)
        {
            if (value.IsMultiFuel)
            {   // multi-fuel vehicle
                this.Write(value.PrimaryValue);
                // only electricity and E85 may be secondary fuel types
                // in case of tertiary fuel vehicles (G+E85+E), only electricity is used
                if      (value.Electricity != 0) { this.Write(value.Electricity); }
                else if (value.Ethanol85   != 0) { this.Write(value.Ethanol85  ); }
            }
            else
            {
                this.Write(value.PrimaryValue);
                this.WriteEmpty(1);
            }
        }
        string GetFuelTypeString(FuelType fuelType, bool engineFuelOnly)
        {
            StringBuilder sb = new StringBuilder(8);
            //
            for (int i = 0, fuelCount = FTValue<object>.Classes.Length; i < fuelCount; i++)
            {
                FuelType fuel = FTValue<object>.Classes[i];
                if (engineFuelOnly && (fuel == FuelType.Hydrogen || fuel == FuelType.Electricity)) { continue; }
                if ((fuelType & fuel) == fuel)
                {
                    sb.Append(FTValue<object>.Acronyms[i]);
                    sb.Append("+");
                }
            }
            //
            return sb.ToString(0, sb.Length - 1);
        }
        string GetVehicleEngineType(bool[] techUsed)
        {
            if (techUsed[TI.TURBO1] || techUsed[TI.TURBO2] ||
                techUsed[TI.  SEGR] || techUsed[TI.  DWSP] ||
                techUsed[TI. CEGR1] || techUsed[TI. CEGR2]) { return "T"; }
            //
            return "NA";
        }

        #endregion


        #region /*** Properties ***/

        /// <summary>Gets the friendly name of the report.</summary>
        public override string ReportName { get { return CsvReportName; } }

        #endregion


        #region /*** Variables ***/

        /// <summary>Represents the friendly name of the report.</summary>
        public const string CsvReportName = "Vehicles CSV Report";

        static readonly string Header;

        #endregion

    }
}
