using System;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;

namespace Volpe.Cafe.Model
{
    //
    // Priority for credit trading:
    //
    //  Stage 1.  carry forward  :  look to the past N years (5 by default) and bring forward unused credits
    //  Stage 2.  fleet transfers:  transfer credits between domestic PC, imported PC, and LT fleets (within the same mfr)
    //  Stage 3.  carry backward :  look to the future N years (3 by default) and borrow unused credits
    //  Stage 4.  mfr trading    :  trade credits between different mfrs
    //
    // Stage 4 (mfr trading) can only be implemented as part of post-analysis
    //  - Stage 4 is evaluated after the completion of the entire year, in order to avoid bias for or against mfrs that are
    //    evaluated later, due to ascending order sorting
    //
    sealed class CreditTrading
    {

        /// <summary>
        /// Determines whether there are existing credits that can be carried forward from earlier years for the specified regulatory class.
        /// </summary>
        public static bool CanCarryForward(RegulatoryClass rc, Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex,
            ModelingSettings settings)
        {
            CreditTradingValues ct = settings.Parameters.CreditTradingValues;
            if (!settings.OperatingModes.AllowCreditTrading || !ct.AllowCarryForward) { return false; }

            // do not transfer if:
            //  - the fleet has positive credits (thus, in compliance) for this reg-class
            Manufacturer.CModelData mmd = modelData[year.Index].Manufacturers[mfrIndex].ModelData;
            double credits = ComputeNetCredits(mmd, rc);
            if (credits >= 0) { return false; }

            // scan each year to see if credits can be carried forward
            int minYearIndex = modelData[year.Index].MinYear.Index;
            for (int i = minYearIndex; i < year.Index; i++)
            {   // get expiration of credits for this scenario and model year
                int carryFwdYears = scen.ScenInfo[rc].CreditCarryFwd[i];
                if (carryFwdYears == 0)
                {   // no scenario/year specific value defined -- use global setting
                    carryFwdYears = ct.CarryForwardYears;
                }
                if (i + carryFwdYears >= year.Index)
                {   // credits from i-th year have not yet expired at current year -- check if any are available
                    Manufacturer.CModelData pMMD = modelData[i].Manufacturers[mfrIndex].ModelData;
                    // **note:  when computing available credits from previous years, TCreditsIn should not be considered,
                    // since there is no way of tracking where they came from or their expiration
                    double eCredits = Math.Max(0, pMMD.Credits[rc] - pMMD.TCreditsOut[rc]);
                    if (eCredits > 0) { return true; }
                }
            }
            return false;
        }
        /// <summary>
        /// Determines whether there are existing credits that can be transferred into the specified regulatory class from other regulatory classes.
        /// </summary>
        public static bool CanFleetTransfer(RegulatoryClass rc, Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex,
            ModelingSettings settings)
        {
            CreditTradingValues ct = settings.Parameters.CreditTradingValues;
            if (!settings.OperatingModes.AllowCreditTrading || !ct.AllowFleetTransfers) { return false; }

            // do not transfer if:
            //  - the amount of credits transferred into the reg-class has already reached or exceeded the cap -or-
            //  - the fleet has positive credits (thus, in compliance) for this reg-class
            Manufacturer.CModelData mmd = modelData[year.Index].Manufacturers[mfrIndex].ModelData;
            double credits = ComputeNetCredits(mmd, rc);
            double cap = ComputeTransferCap(mmd, rc, year, ct);
            if (cap <= 0 || credits >= 0) { return false; }

            // for passenger-car reg-class, also need to determine whether achieved CAFE is less than min STND
            if (rc == RegulatoryClass.PassengerCar)
            {   // we're not allowed to use credits (other than carry-forward AND carry-backward) to offset min-STND shortfall
                if (mmd.PreliminaryStandard[rc] < mmd.CAFE[rc]) { return false; }
            }

            RegulatoryClass eRC = CT_FleetTransfers_GetNextRC(rc);
            if (eRC != RegulatoryClass.None)
            {
                double eRC_creditBalance = ComputeNetCredits(mmd, eRC);

                // check if there are credits in other reg-classes for the current year
                if (eRC_creditBalance > 0) { return true; }

                // if carry-fwd allowed:
                //  * check if there are credits in other reg-class from the previous years
                //  * but, ignore this eRC, if the present year has a negative balance (i.e., paying fines)
                if (ct.AllowCarryForward &&
                    eRC_creditBalance >= 0 &&
                    CanCarryForward(eRC, modelData, scen, year, mfrIndex, settings)) { return true; }
            }

            // no credits can be generated
            return false;
        }

        //
        // Examines credits for trading stages 1, 2, and 3 (carry forward, fleet transfers, and carry backward).  If necessary,
        // credits will be transferred from one compliance category to another.
        //
        public static void ExamineCredits(Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex, ModelingSettings settings)
        {
            CT_CarryForward  (modelData, scen, year, mfrIndex, settings, false, 0); // transfer credits from carry-forward years
            CT_FleetTransfers(modelData, scen, year, mfrIndex, settings, false, 0); // transfer credits between reg-classes
            //CT_CarryBackward (modelData, scen, year, mfrIndex, settings          ); // transfer credits to carry-backward years
        }

        public static void CT_CarryForward(Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex, ModelingSettings settings,
            bool expiringOnly, int expiringYrIndex)
        {
            CT_CarryForward(RegulatoryClass.PassengerCar , modelData, scen, year, mfrIndex, settings, expiringOnly, expiringYrIndex);
            CT_CarryForward(RegulatoryClass.LightTruck   , modelData, scen, year, mfrIndex, settings, expiringOnly, expiringYrIndex);
            CT_CarryForward(RegulatoryClass.LightTruck2b3, modelData, scen, year, mfrIndex, settings, expiringOnly, expiringYrIndex);
        }
        public static void CT_CarryForward(RegulatoryClass rc, Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex,
            ModelingSettings settings, bool expiringOnly, int expiringYrIndex)
        {
            CreditTradingValues ct = settings.Parameters.CreditTradingValues;
            if (!settings.OperatingModes.AllowCreditTrading || !ct.AllowCarryForward) { return; }

            Manufacturer mfr     = modelData[year.Index].Manufacturers[mfrIndex];
            string       mfrName = mfr.Description.Name;
            double       credits = ComputeNetCredits(mfr.ModelData, rc);

            // the fleet has positive credits (is in compliance) for this reg-class
            if (credits >= 0) { return; }

            // scan each year to see if credits can be carried forward
            int minComplianceYearIndex = modelData[year.Index].MinYear.Index;
            int minBankedCredYearIndex = mfr.Description.BankedCreditsMinYear - ModelYear.MinYear;
            int minYearIndex = Math.Min(minComplianceYearIndex, minBankedCredYearIndex);
            //
            for (int i = minYearIndex; i < year.Index; i++)
            {   // get carry forward years
                int carryFwdYears = 0;
                if (i >= minComplianceYearIndex)
                {   // get expiration of credits for this scenario and model year
                    carryFwdYears = scen.ScenInfo[rc].CreditCarryFwd[i];
                }
                if (carryFwdYears == 0)
                {   // no scenario/year specific value defined -- use global setting
                    carryFwdYears = ct.CarryForwardYears;
                }

                // if only expiring credits should be used, skip all other years
                if (expiringOnly && (i > expiringYrIndex - carryFwdYears)) { continue; }

                // if credits already expired, skip year
                if (i + carryFwdYears < year.Index) { continue; }

                Manufacturer pMfr = null;
                double eCredits = 0;
                bool useBank = false;
                if (i >= minComplianceYearIndex)
                {   // checking one of analysis years -- check for credits generated during compliance
                    pMfr = modelData[i].Manufacturers[mfrIndex];
                    // **note:  when computing available credits from previous years, TCreditsIn should not be considered,
                    // since there is no way of tracking where they came from or their expiration
                    eCredits = Math.Max(0, pMfr.ModelData.Credits[rc] - pMfr.ModelData.TCreditsOut[rc]);
                }
                else
                {   // checking a year prior to start of modeling -- check for banked credits
                    pMfr = mfr;
                    eCredits = mfr.Description.GetBankedCredits(rc, i + ModelYear.MinYear);
                    useBank = true;
                }

                // continue to the next year if did not earn any credits in the "i-th" year
                if (eCredits <= 0) { continue; }

                // perform a credit transfer
                CreditAdjustmentMode cfAdjMode = CreditAdjustmentMode.Fixed;
                DoTransfer(mfrName, eCredits, -credits, cfAdjMode, ct, rc, ModelYear.NewModelYearFromIndex(i), pMfr, useBank, rc, year, mfr);

                // update credits; if compliance achieved, return
                credits = ComputeNetCredits(mfr.ModelData, rc);
                if (credits >= 0) { return; }
            } // next i (carry forward credit year)
        }

        public static void CT_FleetTransfers(Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex, ModelingSettings settings,
            bool expiringOnly, int expiringYrIndex)
        {
            CT_FleetTransfers(RegulatoryClass.PassengerCar , modelData, scen, year, mfrIndex, settings, expiringOnly, expiringYrIndex);
            CT_FleetTransfers(RegulatoryClass.LightTruck   , modelData, scen, year, mfrIndex, settings, expiringOnly, expiringYrIndex);
            CT_FleetTransfers(RegulatoryClass.LightTruck2b3, modelData, scen, year, mfrIndex, settings, expiringOnly, expiringYrIndex);
        }
        public static void CT_FleetTransfers(RegulatoryClass rc, Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex,
            ModelingSettings settings, bool expiringOnly, int expiringYrIndex)
        {
            CreditTradingValues ct = settings.Parameters.CreditTradingValues;
            if (!settings.OperatingModes.AllowCreditTrading || !ct.AllowFleetTransfers) { return; }

            Manufacturer mfr     = modelData[year.Index].Manufacturers[mfrIndex];
            var          mmd     = mfr.ModelData;
            string       mfrName = mfr.Description.Name;
            double       credits = ComputeNetCredits (mmd, rc);
            double       cap     = ComputeTransferCap(mmd, rc, year, ct);

            // the amount of credits transferred into the reg-class has already reached or exceeded the cap -or-
            // the fleet has positive credits (thus, in compliance) for this reg-class
            if (cap <= 0 || credits >= 0) { return; }

            // for passenger-car reg-class, also need to determine whether achieved CAFE is less than min STND
            if (rc == RegulatoryClass.PassengerCar)
            {   // we're not allowed to use credits (other than carry-forward AND carry-backward) to offset min-STND shortfall
                if (mmd.PreliminaryStandard[rc] < mmd.CAFE[rc]) { return; }
            }

            // obtain maximum credits that can be transferred into this reg-class, taking the cap into account
            double rCredits = Math.Min(cap, -credits);

            //------------------
            // Transfer Credits
            //
            RegulatoryClass eRC = CT_FleetTransfers_GetNextRC(rc);
            if (eRC == RegulatoryClass.None) { return; }

            // first, transfer credits from other reg-class, from the previous years (if settings allow)
            // but, skip this eRC if the present year has a negative balance of credits (i.e., paying fines)
            if (ct.AllowCarryForward && ComputeNetCredits(mmd, eRC) >= 0)
            {
                // scan each year
                int minComplianceYearIndex = modelData[year.Index].MinYear.Index;
                int minBankedCredYearIndex = mfr.Description.BankedCreditsMinYear - ModelYear.MinYear;
                int minYearIndex = Math.Min(minComplianceYearIndex, minBankedCredYearIndex);
                //
                for (int j = minYearIndex; j < year.Index; j++)
                {   // get carry forward years
                    int carryFwdYears = 0;
                    if (j >= minComplianceYearIndex)
                    {   // get expiration of credits for this scenario and model year
                        carryFwdYears = scen.ScenInfo[rc].CreditCarryFwd[j];
                    }
                    if (carryFwdYears == 0)
                    {   // no scenario/year specific value defined -- use global setting
                        carryFwdYears = ct.CarryForwardYears;
                    }

                    // if only expiring credits should be used, skip all other years
                    if (expiringOnly && (j > expiringYrIndex - carryFwdYears)) { continue; }

                    // if credits already expired, skip year
                    if (j + carryFwdYears < year.Index) { continue; }

                    Manufacturer pMfr = null;
                    double eCredits = 0;
                    bool useBank = false;
                    if (j >= minComplianceYearIndex)
                    {   // checking one of analysis years -- check for credits generated during compliance
                        pMfr = modelData[j].Manufacturers[mfrIndex];
                        eCredits = Math.Max(0, pMfr.ModelData.Credits[eRC] - pMfr.ModelData.TCreditsOut[eRC]); // see note in "CT_CarryForward(...)"
                    }
                    else
                    {   // checking a year prior to start of modeling -- check for banked credits
                        pMfr = mfr;
                        eCredits = mfr.Description.GetBankedCredits(rc, j + ModelYear.MinYear);
                        useBank = true;
                    }

                    // continue to the next year if did not earn any credits in the "i-th" year
                    if (eCredits <= 0) { continue; }

                    // determine amount of credits that can be transferred
                    double outECredits, outRCredits;
                    RequestCredits(eCredits, rCredits, ct.AdjustmentMode, ct, eRC, year, mfr, rc, year, mfr, out outECredits, out outRCredits);

                    // carry forward credits within the reg-class
                    CreditAdjustmentMode cfAdjMode = CreditAdjustmentMode.Fixed;
                    DoTransfer(mfrName, outECredits, double.PositiveInfinity, cfAdjMode, ct, eRC, ModelYear.NewModelYearFromIndex(j), pMfr, useBank, eRC, year, mfr);
                    // transfer credits between reg-classes
                    eCredits = ComputeNetCredits(mmd, eRC);
                    DoTransfer(mfrName, eCredits, outRCredits, ct.AdjustmentMode, ct, eRC, year, mfr, false, rc, year, mfr);

                    // update credits and cap
                    credits = ComputeNetCredits (mmd, rc);
                    cap     = ComputeTransferCap(mmd, rc, year, ct);
                    if (cap <= 0 || credits >= 0) { return; }   // if compliance achieved or cap reached, exit
                    rCredits = Math.Min(cap, -credits);         // update maximum credits that can be transferred
                }
            }

            // second, exhaust credits from other reg-classes for the current year
            if (expiringOnly) { return; }   // do not transfer if only expiring credits should be used
            //
            // examine credits ...
            if (ComputeNetCredits(mmd, eRC) > 0)
            {
                double eCredits = ComputeNetCredits(mmd, eRC);

                // perform a credit transfer
                DoTransfer(mfrName, eCredits, rCredits, ct.AdjustmentMode, ct, eRC, year, mfr, false, rc, year, mfr);

                // update credits and cap
                credits = ComputeNetCredits(mmd, rc);
                cap = ComputeTransferCap(mmd, rc, year, ct);
                if (cap <= 0 || credits >= 0) { return; }   // if compliance achieved or cap reached, exit
                rCredits = Math.Min(cap, -credits);         // update maximum credits that can be transferred
            }
        }
        static RegulatoryClass CT_FleetTransfers_GetNextRC(RegulatoryClass uRC)
        {
            if (uRC == RegulatoryClass.PassengerCar) { return RegulatoryClass.LightTruck; }
            if (uRC == RegulatoryClass.LightTruck  ) { return RegulatoryClass.PassengerCar; }
            return RegulatoryClass.None;
        }

        static void CT_CarryBackward(Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex, ModelingSettings settings)
        {
            CT_CarryBackward(RegulatoryClass.PassengerCar , modelData, scen, year, mfrIndex, settings);
            CT_CarryBackward(RegulatoryClass.LightTruck   , modelData, scen, year, mfrIndex, settings);
            CT_CarryBackward(RegulatoryClass.LightTruck2b3, modelData, scen, year, mfrIndex, settings);
        }
        static void CT_CarryBackward(RegulatoryClass rc, Industry[] modelData, Scenario scen, ModelYear year, int mfrIndex,
            ModelingSettings settings)
        {
            CreditTradingValues ct = settings.Parameters.CreditTradingValues;
            if (!settings.OperatingModes.AllowCreditTrading || !ct.AllowCarryBackward || ct.CarryBackwardYears < 1) { return; }

            Manufacturer mfr      = modelData[year.Index].Manufacturers[mfrIndex];
            string       mfrName  = mfr.Description.Name;
            double       eCredits = ComputeNetCredits(mfr.ModelData, rc);

            // the fleet does not have available credits to transfer from this reg-class
            if (eCredits <= 0) { return; }

            // compute carry-backward start year index (cb s yr index)
            int cbSYrIndex = ComputeCarryBwdStartYr(modelData, year, ct);

            // scan each year to see if credits can be carried backward
            int minYearIndex = modelData[year.Index].MinYear.Index;
            for (int i = minYearIndex; i < year.Index; i++)
            {
                Manufacturer pMfr    = modelData[i].Manufacturers[mfrIndex];
                double       credits = ComputeNetCredits(pMfr.ModelData, rc);

                // continue to the next year if the mfr has positive credits (is in compliance)
                if (credits >= 0) { continue; }

                // perform a credit transfer
                CreditAdjustmentMode cfAdjMode = CreditAdjustmentMode.Fixed;
                DoTransfer(mfrName, eCredits, -credits, cfAdjMode, ct, rc, year, mfr, false, rc, ModelYear.NewModelYearFromIndex(i), pMfr);

                // update earned credits; if no more remaining, return
                eCredits = ComputeNetCredits(mfr.ModelData, rc);
                if (eCredits <= 0) { return; }
            } // next i (carry backwrd credit year)
        }

        [Obsolete("cannot use ComputeCarryFwdStartYr anymore since credit expiration changes by origin year", true)]
        static int ComputeCarryFwdStartYr(Industry[] modelData, ModelYear year, CreditTradingValues ct)
        {
            return Math.Max(modelData[year.Index].MinYear.Index, year.Index - ct.CarryForwardYears);
        }
        static int ComputeCarryBwdStartYr(Industry[] modelData, ModelYear year, CreditTradingValues ct)
        {
            return Math.Max(modelData[year.Index].MinYear.Index, year.Index - ct.CarryBackwardYears);
        }

        static double ComputeNetCredits(Manufacturer.CModelData mmd, RegulatoryClass rc)
        {
            return mmd.Credits[rc] + mmd.TCreditsIn[rc] - mmd.TCreditsOut[rc];
        }
        static double ComputeTransferCap(Manufacturer.CModelData mmd, RegulatoryClass rc, ModelYear year, CreditTradingValues ct)
        {
            double cap = ct.GetTransferCaps(rc, year.Year);
            return Math.Floor(cap * mmd.Sales[rc]) - mmd.TCreditsInCapped[rc];
        }
        static double ComputeVMT(CreditTradingValues ct, RegulatoryClass rc, ModelYear year)
        {
            return ct.GetAdjustmentVMT(rc, year.Year);
        }


        static void DoTransfer(string mfrName, double eCredits, double rCredits, CreditAdjustmentMode adjMode, CreditTradingValues ct,
            RegulatoryClass eRC, ModelYear eYear, Manufacturer eMfr, bool eBank, // params from the "earned" compliance category
            RegulatoryClass uRC, ModelYear uYear, Manufacturer uMfr)             // params from the "used" compliance category
        {
            //
            // eCredits:  available credits from another compliance category (such reg-class and/or model year);
            //            if this value is infinity, maximum "out" earned credits will be returned
            // rCredits:  requested credits that can be transferred into the current compliance category (reg-class)
            //            if this value is infinity, maximum "out" required credits will be returned
            //
            // eCredits & rCredits may not both be infinity
            //
            // eBank:     if set to true, using banked credits generated prior to start of analysis
            //
            double outECredits, outRCredits;
            RequestCredits(eCredits, rCredits, adjMode, ct, eRC, eYear, eMfr, uRC, uYear, uMfr, out outECredits, out outRCredits);

            // adjust credits in / credits out
            if (eBank)
            {
                eMfr.Description.BankedCredits[eRC][eYear.Year - eMfr.Description.BankedCreditsMinYear] -= outECredits;
            }
            else
            {
                eMfr.ModelData.TCreditsOut[eRC] += outECredits;
            }
            uMfr.ModelData.TCreditsIn [uRC] += outRCredits;
            // also adjust "capped" credits in, if different reg-classes (dif rc's occur when transferring between fleets)
            if (eRC != uRC) { uMfr.ModelData.TCreditsInCapped[uRC] += outRCredits; }

            // debug logging
            //Console.WriteLine(mfrName + "," +
            //    eCredits + "," + rCredits + "," + eRC + "," + eYrIndex + "," + outECredits + "," + uRC + "," + uYrIndex + "," + outRCredits);
        }
        static void RequestCredits(double eCredits, double rCredits, CreditAdjustmentMode adjMode, CreditTradingValues ct,
            RegulatoryClass eRC, ModelYear eYear, Manufacturer eMfr, // params from the "earned" compliance category
            RegulatoryClass uRC, ModelYear uYear, Manufacturer uMfr, // params from the "used" compliance category
            out double outECredits, out double outRCredits)
        {
            //
            // This function computes the maximum amount of credits that can be transferred, given the available
            // eCredits and the requested rCredits.  The resultant out params (outECredits and outRCredits) will
            // not exceed the maximum of eCredits and rCredits.
            //
            //
            // eCredits:  available credits from another compliance category (such reg-class and/or model year);
            //            if this value is infinity, maximum "out" earned credits will be returned
            // rCredits:  requested credits that can be transferred into the current compliance category (reg-class)
            //            if this value is infinity, maximum "out" required credits will be returned
            //
            // eCredits & rCredits may not both be infinity
            //

            // compute vmt levels first (if the adjustment factor is not 1:1)
            double eVMT = 0, uVMT = 0;
            if (adjMode == CreditAdjustmentMode.Variable)
            {
                eVMT = ComputeVMT(ct, eRC, eYear);
                uVMT = ComputeVMT(ct, uRC, uYear);
            }

            // compute credits
            double eCAFE = (eMfr == null) ? 0 : eMfr.ModelData.CAFE    [eRC];
            double eSTND = (eMfr == null) ? 0 : eMfr.ModelData.Standard[eRC];
            double uCAFE = (uMfr == null) ? 0 : uMfr.ModelData.CAFE    [uRC];
            double uSTND = (uMfr == null) ? 0 : uMfr.ModelData.Standard[uRC];
            //
            RequestCredits(eCredits, rCredits, adjMode, eVMT, eCAFE, eSTND, uVMT, uCAFE, uSTND, out outECredits, out outRCredits);
        }
        static void RequestCredits(double eCredits, double rCredits, CreditAdjustmentMode adjMode,
            double eVMT, double eCAFE, double eSTND, double uVMT, double uCAFE, double uSTND,
            out double outECredits, out double outRCredits)
        {
            //
            // This function computes the maximum amount of credits that can be transferred, given the available
            // eCredits and the requested rCredits.  The resultant out params (outECredits and outRCredits) will
            // not exceed the maximum of eCredits and rCredits.
            //
            // eCredits:  available credits from another compliance category (such reg-class and/or model year);
            //            if this value is infinity, maximum "out" earned credits will be returned
            // rCredits:  requested credits that can be transferred into the current compliance category (reg-class)
            //            if this value is infinity, maximum "out" required credits will be returned
            //
            // eCredits & rCredits may not both be infinity
            //

            if (double.IsInfinity(rCredits) && double.IsInfinity(eCredits)) { outECredits = outRCredits = 0; return; }

            // obtain the adjustment factor
            double adjFactor = GetCreditAdjFactor(adjMode, eVMT, eCAFE, eSTND, uVMT, uCAFE, uSTND);

            // determine max tCredits that can be obtained from available eCredits
            if (!double.IsInfinity(eCredits))
            {
                outRCredits = Math.Floor(eCredits / adjFactor);
                if (double.IsInfinity(rCredits) || outRCredits <= rCredits)
                {   // if outRCredits <= rCredits, that means that avail eCredits is not too much
                    outECredits = Math.Ceiling(outRCredits * adjFactor);
                    return;
                }
            }

            // this step is reached if avail eCredits produced too much transferable credits ... in this case, determine
            // amount of eCredits necessary using requested rCredits
            outECredits = Math.Ceiling(rCredits    * adjFactor);
            outRCredits = Math.Floor  (outECredits / adjFactor);
        }


        #region /* Methods for adjusting credits */

        /// <summary>
        /// Returns the credit adjustment factor between two fleets specified by their VMT, CAFE, and Standard.
        /// </summary>
        /// <param name="adjMode">The credit adjustement mode to use.</param>
        /// <param name="eVMT">The lifetime vehicle miles traveled for the manufacturer, model year, and compliance category in which the credit was earned.</param>
        /// <param name="eCAFE">The achieved CAFE value for the manufacturer, compliance category, and model year in which the credit was earned.</param>
        /// <param name="eSTND">The required CAFE standard for the manufacturer, compliance category, and model year in which the credit was earned.</param>
        /// <param name="uVMT">The lifetime vehicle miles traveled for the manufacturer, model year, and compliance category in which the credit is used for compliance.</param>
        /// <param name="uCAFE">The achieved CAFE value for the manufacturer, compliance category, and model year in which the credit is used for compliance.</param>
        /// <param name="uSTND">The required CAFE standard for the manufacturer, compliance category, and model year in which the credit is used for compliance.</param>
        /// <returns>The credit adjustment factor between two fleets specified by their VMT, CAFE, and Standard.</returns>
        static double GetCreditAdjFactor(CreditAdjustmentMode adjMode, double eVMT, double eCAFE, double eSTND, double uVMT, double uCAFE, double uSTND)
        {
            switch (adjMode)
            {
                case CreditAdjustmentMode.Fixed   : return 1;
                case CreditAdjustmentMode.Variable: return (uVMT * eCAFE * eSTND) / (eVMT * uCAFE * uSTND);
                    //
                default: throw new ArgumentException("The Credit Adjustment Factor is not valid.");
            }
        }
        /// <summary>
        /// Returns the credit adjustment factor between two regulatory classes within the same model year.
        /// </summary>
        public static double GetCreditAdjFactorFT(RegulatoryClass eRC, RegulatoryClass uRC, Manufacturer mfr, ModelYear year, CreditTradingValues ct)
        {
            double eVMT = 0, uVMT = 0;
            if (ct.AdjustmentMode == CreditAdjustmentMode.Variable)
            {
                eVMT = ComputeVMT(ct, eRC, year);
                uVMT = ComputeVMT(ct, uRC, year);
            }
            return GetCreditAdjFactor(ct.AdjustmentMode, eVMT, mfr.ModelData.CAFE[eRC], mfr.ModelData.Standard[eRC],
                                                         uVMT, mfr.ModelData.CAFE[uRC], mfr.ModelData.Standard[uRC]);
        }

        #endregion

    }
}
