#region << Using Directives >>
using System;
using Volpe.Cafe.Data;
#endregion

namespace Volpe.Cafe.Model
{
    /// <summary>
    /// Provides information regarding the impact of one or more technologies on a vehicle.
    /// </summary>
    /// <remarks>
    /// The TechImpact structure is used in conjunction with the <see cref="ComplianceFinding"/> class, to assist in calculating
    /// the efficiency of a compliance finding and the associated cost and improvement.  The technologies applied to generate the
    /// specified cost, fuel improvement, and changes in curb weight and fuel economy, as well as the vehicle to which the
    /// technologies were applied, can be found in the <see cref="ComplianceFinding"/> instance that houses this TechImpact
    /// structure.
    /// </remarks>
    public class TechImpact : ICloneable
    {

        #region /*** Constructors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="TechImpact"/> class.
        /// </summary>
        public TechImpact(int len)
        {
            this.HasImpact         = new bool     [len];
            this.Vehicle           = new Vehicle  [len];
            this.Cost              = new double   [len];
            this.ERPCost           = new double   [len];
            this.MaintenanceCost   = new double   [len];
            this.RepairCost        = new double   [len];
            this.ConsumerValuation = new double   [len];
            this.OffCycleCredit    = new double   [len];
            this.ZEVCredit         = new double   [len];
            this.FC                = new double   [len];
            this.FCSecondary       = new double   [len];
            this.FCAdjustment      = new double   [len];
            this.FCPowerAdjustment = new double   [len];
            this.NewFuelEconomy    = new FuelValue[len];
            this.NewFuelShare      = new FuelValue[len];
            this.DeltaFuelEconomy  = new FuelValue[len];
            this.DeltaPower        = new double   [len];
            this.DeltaCurbWeight   = new double   [len];
            this.DeltaGVWR         = new double   [len];
            this.DeltaGCWR         = new double   [len];
        }

        #endregion


        #region /*** Methods ***/

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="TechImpact"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="TechImpact"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a new object that is a copy of the current <see cref="TechImpact"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="TechImpact"/>.</returns>
        public TechImpact Clone()
        {
            TechImpact impact = new TechImpact(this.HasImpact.Length);
            this.CopyTo(impact);
            return impact;
        }
        /// <summary>
        /// Copies the data in this <see cref="TechImpact"/> instance to the specified value.
        /// </summary>
        /// <param name="impact">The <see cref="TechImpact"/> where to save the data copied from this instance.</param>
        public void CopyTo(TechImpact impact)
        {
            int len = this.HasImpact.Length;
            //
            Array.Copy(this.HasImpact        , impact.HasImpact        , len);
            Array.Copy(this.Vehicle          , impact.Vehicle          , len);
            Array.Copy(this.Cost             , impact.Cost             , len);
            Array.Copy(this.ERPCost          , impact.ERPCost          , len);
            Array.Copy(this.MaintenanceCost  , impact.MaintenanceCost  , len);
            Array.Copy(this.RepairCost       , impact.RepairCost       , len);
            Array.Copy(this.ConsumerValuation, impact.ConsumerValuation, len);
            Array.Copy(this.OffCycleCredit   , impact.OffCycleCredit   , len);
            Array.Copy(this.ZEVCredit        , impact.ZEVCredit        , len);
            Array.Copy(this.FC               , impact.FC               , len);
            Array.Copy(this.FCSecondary      , impact.FCSecondary      , len);
            Array.Copy(this.FCAdjustment     , impact.FCAdjustment     , len);
            Array.Copy(this.FCPowerAdjustment, impact.FCPowerAdjustment, len);
            Array.Copy(this.NewFuelEconomy   , impact.NewFuelEconomy   , len);
            Array.Copy(this.NewFuelShare     , impact.NewFuelShare     , len);
            Array.Copy(this.DeltaFuelEconomy , impact.DeltaFuelEconomy , len);
            Array.Copy(this.DeltaPower       , impact.DeltaPower       , len);
            Array.Copy(this.DeltaCurbWeight  , impact.DeltaCurbWeight  , len);
            Array.Copy(this.DeltaGVWR        , impact.DeltaGVWR        , len);
            Array.Copy(this.DeltaGCWR        , impact.DeltaGCWR        , len);
        }

        #endregion

        /// <summary>
        /// Returns the string representation of this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="TechImpact"/> instance.</returns>
        public override string ToString()
        {
            return "{" + this.ToStringInternal() + "}";
        }
        /// <summary>
        /// Returns the string representation of this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <param name="fullString">true, to return the string including all values from this <see cref="TechImpact"/>; false,
        ///   to omit vehicle code and ignore flag.</param>
        /// <returns>The string representation of the <see cref="TechImpact"/> instance.</returns>
        public string ToString(bool fullString)
        {
            if (fullString)
            {
                return "{Vehicle Code=" + this.Vehicle[this.Vehicle.Length - 1].Description.Code + ", " + this.ToStringInternal() + "}";
            }
            else { return this.ToString(); }
        }
        string ToStringInternal()
        {
            int lastYear  = this.HasImpact.Length - 1;
            int firstYear = 0;
            for (int i = 0; i < this.HasImpact.Length; i++)
            {   // if there are more than one impact, we'll bracket each set with "{" and "}"
                if (this.HasImpact[i]) { firstYear = i; break; }
            }
            bool multiple = (firstYear != lastYear);
            string impactString = string.Empty;
            for (int i = lastYear; i >= firstYear; i--)
            {   // do a year at a time (in non-multiyear, only one year will be converted)
                if (this.HasImpact[i])
                {
                    if (multiple) { impactString += "{"; }
                    impactString +=
                        "Model Year="         + (ModelYear.MinYear +   i)                     .ToString(        ) +
                        ", Veh Code="         + this.Vehicle          [i].Description.Code    .ToString(        ) +
                        ", Sales="            + this.Vehicle          [i].Description.Sales[i].ToString(        ) +
                        ", Cost="             + this.Cost             [i]                     .ToString("0.####") +
                        ", ERPCost="          + this.ERPCost          [i]                     .ToString("0.####") +
                        ", MaintenanceCost="  + this.MaintenanceCost  [i]                     .ToString("0.####") +
                        ", RepairCost="       + this.RepairCost       [i]                     .ToString("0.####") +
                        ", Consumer Val="     + this.ConsumerValuation[i]                     .ToString("0.####") +
                        ", Off-Cycle Credit=" + this.OffCycleCredit   [i]                     .ToString("0.####") +
                        ", ZEV Credit="       + this.ZEVCredit        [i]                     .ToString("0.####") +
                        ", FC="               + this.FC               [i]                     .ToString("0.####") +
                        ", FC Secondary="     + this.FCSecondary      [i]                     .ToString("0.####") +
                        ", FC Adjustment="    + this.FCAdjustment     [i]                     .ToString("0.####") +
                        ", FC Power Adj="     + this.FCPowerAdjustment[i]                     .ToString("0.####") +
                        ", New FE="           + this.NewFuelEconomy   [i]                     .ToString("0.####") +
                        ", New Share="        + this.NewFuelShare     [i]                     .ToString("0.####") +
                        ", Delta FE="         + this.DeltaFuelEconomy [i]                     .ToString("0.####") +
                        ", Delta HP="         + this.DeltaPower       [i]                     .ToString("0.####") +
                        ", Delta CW="         + this.DeltaCurbWeight  [i]                     .ToString("0.####") +
                        ", Delta GVWR="       + this.DeltaGVWR        [i]                     .ToString("0.####") +
                        ", Delta GCWR="       + this.DeltaGCWR        [i]                     .ToString("0.####");
                    if (multiple) { impactString += ((i == firstYear) ? "}" : "}, "); }
                }
            }
            return impactString;
        }

        /// <summary>
        /// Resets the runtime variables within this <see cref="TechImpact"/> instance.
        /// </summary>
        public void Reset()
        {
            int len = this.HasImpact.Length;
            Array.Clear(this.HasImpact        , 0, len);
            Array.Clear(this.Vehicle          , 0, len);
            Array.Clear(this.Cost             , 0, len);
            Array.Clear(this.ERPCost          , 0, len);
            Array.Clear(this.MaintenanceCost  , 0, len);
            Array.Clear(this.RepairCost       , 0, len);
            Array.Clear(this.ConsumerValuation, 0, len);
            Array.Clear(this.OffCycleCredit   , 0, len);
            Array.Clear(this.ZEVCredit        , 0, len);
            Array.Clear(this.FC               , 0, len);
            Array.Clear(this.FCSecondary      , 0, len);
            Array.Clear(this.FCAdjustment     , 0, len);
            Array.Clear(this.FCPowerAdjustment, 0, len);
            Array.Clear(this.NewFuelEconomy   , 0, len);
            Array.Clear(this.NewFuelShare     , 0, len);
            Array.Clear(this.DeltaFuelEconomy , 0, len);
            Array.Clear(this.DeltaPower       , 0, len);
            Array.Clear(this.DeltaCurbWeight  , 0, len);
            Array.Clear(this.DeltaGVWR        , 0, len);
            Array.Clear(this.DeltaGCWR        , 0, len);
        }

        //----- support methods to test vehicle attribute changes -----
        /// <summary>
        /// Tests whether the fuel type of the vehicle changes in the specified year as a result of applying technology described
        /// by this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <param name="yrIndex">The index of the year to test.</param>
        /// <returns>true, if the fuel type of the vehicle changes; false, otherwise.</returns>
        public bool HasFuelTypeChanged(int yrIndex)
        {
            return (this.NewFuelShare[yrIndex].FuelType != this.Vehicle[yrIndex].Description.FuelShare.FuelType);
        }
        /// <summary>
        /// Tests whether the horsepower of the vehicle changes in the specified year as a result of applying technology
        /// described by this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <param name="yrIndex">The index of the year to test.</param>
        /// <returns>true, if the horsepower of the vehicle changes; false, otherwise.</returns>
        public bool HasPowerChanged(int yrIndex)
        {
            return (this.DeltaPower[yrIndex] != 0);
        }
        /// <summary>
        /// Tests whether the curb weight of the vehicle changes in the specified year as a result of applying technology
        /// described by this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <param name="yrIndex">The index of the year to test.</param>
        /// <returns>true, if the curb weight of the vehicle changes; false, otherwise.</returns>
        public bool HasWeightChanged(int yrIndex)
        {
            return (this.DeltaCurbWeight[yrIndex] != 0);
        }
        /// <summary>
        /// Tests whether the GVWR of the vehicle changes in the specified year as a result of applying technology described by
        /// this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <param name="yrIndex">The index of the year to test.</param>
        /// <returns>true, if the GVWR of the vehicle changes; false, otherwise.</returns>
        public bool HasGVWRChanged(int yrIndex)
        {
            return (this.DeltaGVWR[yrIndex] != 0);
        }
        /// <summary>
        /// Tests whether the GCWR of the vehicle changes in the specified year as a result of applying technology described by
        /// this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <param name="yrIndex">The index of the year to test.</param>
        /// <returns>true, if the GCWR of the vehicle changes; false, otherwise.</returns>
        public bool HasGCWRChanged(int yrIndex)
        {
            return (this.DeltaGCWR[yrIndex] != 0);
        }
        /// <summary>
        /// Tests whether the footprint of the vehicle changes in the specified year as a result of applying technology described
        /// by this <see cref="TechImpact"/> instance.
        /// </summary>
        /// <param name="yrIndex">The index of the year to test.</param>
        /// <returns>true, if the footprint of the vehicle changes; false, otherwise.</returns>
        public bool HasFootprintChanged(int yrIndex)
        {
            return false;
        }

        #endregion


        #region /*** Variables ***/

        /// <summary>Specifies whether this <see cref="TechImpact"/> instance has valid data for a specific year.</summary>
        public bool[] HasImpact;

        /// <summary>Represents the <see cref="Vehicle"/> associated with this <see cref="TechImpact"/> instance.</summary>
        public Vehicle[] Vehicle;
        /// <summary>Specifies the accumulated technology cost that a vehicle will accrue as a result of applying a set of
        ///   technologies.</summary>
        public double[] Cost;
        /// <summary>Specifies the accumulated technology early-replacement penalty cost that a vehicle will accrue as a result
        ///   of applying a set of technologies.</summary>
        public double[] ERPCost;
        /// <summary>Specifies the accumulated technology maintenance cost that a vehicle will accrue as a result of applying a
        ///   set of technologies.</summary>
        public double[] MaintenanceCost;
        /// <summary>Specifies the accumulated technology repair cost that a vehicle will accrue as a result of applying a set of
        ///   technologies.</summary>
        public double[] RepairCost;
        /// <summary>Specifies the accumulated changes in consumer valuation that a vehicle will accrue as a result of applying a
        ///   set of technologies.</summary>
        public double[] ConsumerValuation;
        /// <summary>Specifies the accumulated off-cycle credit that a vehicle will accrue as a result of applying a set of
        ///   technologies.</summary>
        public double[] OffCycleCredit;
        /// <summary>Specifies the accumulated ZEV credit that a vehicle will accrue as a result of applying a set of technologies.</summary>
        public double[] ZEVCredit;
        /// <summary>Specifies the accumulated fuel consumption improvement that a vehicle will accrue as a result of applying a
        ///   set of technologies.</summary>
        public double[] FC;
        /// <summary>Specifies the accumulated fuel consumption improvement that a vehicle will accrue as a result of applying a
        ///   set of technologies, when operating on secondary fuel (if applicable).</summary>
        public double[] FCSecondary;
        /// <summary>Specifies the accumulated fuel consumption adjustment factor that a vehicle will accrue as a result of
        ///   applying a set of technologies.</summary>
        public double[] FCAdjustment;
        /// <summary>Specifies the accumulated fuel consumption adjustment factor, due to differences between simulated vehicle's
        ///   horsepower and actual vehicle's horsepower, that a vehicle will accrue as a result of applying a set of technologies.</summary>
        public double[] FCPowerAdjustment;
        /// <summary>Specifies what the vehicle's new fuel economy will be, as a result of applying a set of technologies.</summary>
        public FuelValue[] NewFuelEconomy;
        /// <summary>Specifies what the vehicle's new fuel share will be, as a result of applying a set of technologies.</summary>
        public FuelValue[] NewFuelShare;
        /// <summary>Specifies what the change in vehicle's fuel economy will be, as a result of applying a set of technologies.</summary>
        public FuelValue[] DeltaFuelEconomy;
        /// <summary>Specifies what the change in vehicle's power will be, as a result of applying a set of technologies.</summary>
        public double[] DeltaPower;
        /// <summary>Specifies what the change in vehicle's curb weight will be, as a result of applying a set of technologies.</summary>
        public double[] DeltaCurbWeight;
        /// <summary>Specifies what the change in vehicle's GVWR will be, as a result of applying a set of technologies.</summary>
        public double[] DeltaGVWR;
        /// <summary>Specifies what the change in vehicle's GCWR will be, as a result of applying a set of technologies.</summary>
        public double[] DeltaGCWR;

        #endregion

    }
}
