#region << Using Directives >>
using System;
using System.IO;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.Settings;
using Volpe.Utils;
#endregion
namespace Volpe.Cafe.Model.EIS
{
    public class EisModel : IEISModel
    {
        #region 
        public EisModel()
        {
        }
        #endregion
        #region 
        void OpenFiles(string eisParametersFile, string eisTailpipeEmissionsFile)
        {
            this._params   = new EisParameters       (eisParametersFile       );
            this._tailpipe = new EisTailpipeEmissions(eisTailpipeEmissionsFile);
        }
        public void Analyze(EISSettings settings, ICompliance compliance, int scenCount, int minYear, int maxYear)
        {
            this.OpenFiles(settings.EISParametersFile, settings.EISTailpipeEmissionsFile);
            this._eisMinYear      = this._params.MinYear;
            this._eisMaxYear      = this._params.MaxYear;
            this._yearCount       = this._eisMaxYear - this._eisMinYear + 1;
            this._carFEFlatCAFE   = new EisParameters.FuelEconomy[scenCount];
            this._truckFEFlatCAFE = new EisParameters.FuelEconomy[scenCount];
            this._carFEGrowCAFE   = new EisParameters.FuelEconomy[scenCount];
            this._truckFEGrowCAFE = new EisParameters.FuelEconomy[scenCount];
            this._fuelPriceEstimates = compliance.Settings.OperatingModes.FuelPriceEstimates;
            this._paramsOverrides    = compliance.Settings.ParametersOverrides;
            if (this._paramsOverrides.OverrideReboundEffect)
            {
                this._params.BaseParameters.ReboundEffect = -this._paramsOverrides.ReboundEffect;
                this._params.BaseParameters.Elasticity    =  this._paramsOverrides.ReboundEffect;
            }
            this._scenDataFlatCAFE = new EisData[scenCount];
            this._scenDataGrowCAFE = new EisData[scenCount];
            for (int i = 0; i < scenCount; i++)
            {
                this._scenDataFlatCAFE[i] = new EisData(this._eisMinYear, this._eisMaxYear);
                this._scenDataGrowCAFE[i] = new EisData(this._eisMinYear, this._eisMaxYear);
                this.UpdateFEProjections(compliance, i, minYear, maxYear);
                this.CalcEisData(this._scenDataFlatCAFE[i], false, i, this._carFEFlatCAFE, this._truckFEFlatCAFE);
                this.CalcEisData(this._scenDataGrowCAFE[i], false, i, this._carFEGrowCAFE, this._truckFEGrowCAFE);
            }
            this._noCafeDataFlatCAFE = new EisData(this._eisMinYear, this._eisMaxYear);
            this._noCafeDataGrowCAFE = new EisData(this._eisMinYear, this._eisMaxYear);
            this.CalcEisData(this._noCafeDataFlatCAFE, true, -1, this._carFEFlatCAFE, this._truckFEFlatCAFE);
            this.CalcEisData(this._noCafeDataGrowCAFE, true, -1, this._carFEGrowCAFE, this._truckFEGrowCAFE);
        }
        void UpdateFEProjections(ICompliance compliance, int scenIndex, int minYear, int maxYear)
        {
            Scenario.EISModelData eisScenData = compliance.Settings.Scenarios[scenIndex].EISData;
            EisParameters.FuelEconomy carFeFlatCAFE   = this._params.CarFuelEconomy  .Clone();
            EisParameters.FuelEconomy truckFeFlatCAFE = this._params.TruckFuelEconomy.Clone();
            EisParameters.FuelEconomy carFeGrowCAFE   = this._params.CarFuelEconomy  .Clone();
            EisParameters.FuelEconomy truckFeGrowCAFE = this._params.TruckFuelEconomy.Clone();
            for (int i = minYear; i <= maxYear; i++)
            {   
                Industry data = compliance.GetData(scenIndex, i - ModelYear.MinYear);
                AggregateEffectsData ed   = data.ModelingData.EffectsData;
                M6Double salesOverFE      = ed.Sales / ed.FuelEconomy;
                if (double.IsNaN(salesOverFE.Ldgv  )) { salesOverFE.Ldgv   = 0; }
                if (double.IsNaN(salesOverFE.Lddv  )) { salesOverFE.Lddv   = 0; }
                if (double.IsNaN(salesOverFE.Ldgt1 )) { salesOverFE.Ldgt1  = 0; }
                if (double.IsNaN(salesOverFE.Lddt1 )) { salesOverFE.Lddt1  = 0; }
                if (double.IsNaN(salesOverFE.Ldgt2 )) { salesOverFE.Ldgt2  = 0; }
                if (double.IsNaN(salesOverFE.Lddt2 )) { salesOverFE.Lddt2  = 0; }
                if (double.IsNaN(salesOverFE.Hdgv2b)) { salesOverFE.Hdgv2b = 0; }
                if (double.IsNaN(salesOverFE.Hddv2b)) { salesOverFE.Hddv2b = 0; }
                double carGSales   = ed.Sales   .Ldgv;
                double carDSales   = ed.Sales   .Lddv;
                double carGSOFE    = salesOverFE.Ldgv;
                double carDSOFE    = salesOverFE.Lddv;
                double truckGSales = ed.Sales   .Ldgt1 + ed.Sales   .Ldgt2 + ed.Sales   .Hdgv2b;
                double truckDSales = ed.Sales   .Lddt1 + ed.Sales   .Lddt2 + ed.Sales   .Hddv2b;
                double truckGSOFE  = salesOverFE.Ldgt1 + salesOverFE.Ldgt2 + salesOverFE.Hdgv2b;
                double truckDSOFE  = salesOverFE.Lddt1 + salesOverFE.Lddt2 + salesOverFE.Hddv2b;
                double carGasolineFE   = (carGSOFE                  > 0) ? carGSales   / (carGSOFE                 ) : 0;
                double carDieselFE     = (carDSOFE                  > 0) ? carDSales   / (carDSOFE                 ) : 0;
                double carDslShare     = (carGSales   + carDSales   > 0) ? carDSales   / (carGSales   + carDSales  ) : 0;
                double truckGasolineFE = (truckGSOFE                > 0) ? truckGSales / (truckGSOFE               ) : 0;
                double truckDieselFE   = (truckDSOFE                > 0) ? truckDSales / (truckDSOFE               ) : 0;
                double truckDslShare   = (truckGSales + truckDSales > 0) ? truckDSales / (truckGSales + truckDSales) : 0;
                carFeFlatCAFE  .GasolineCAFE   [i - this._eisMinYear] = carGasolineFE;
                carFeFlatCAFE  .DieselCAFE     [i - this._eisMinYear] = carDieselFE;
                carFeFlatCAFE  .DieselShareCAFE[i - this._eisMinYear] = carDslShare;
                truckFeFlatCAFE.GasolineCAFE   [i - this._eisMinYear] = truckGasolineFE;
                truckFeFlatCAFE.DieselCAFE     [i - this._eisMinYear] = truckDieselFE;
                truckFeFlatCAFE.DieselShareCAFE[i - this._eisMinYear] = truckDslShare;
                carFeGrowCAFE  .GasolineCAFE   [i - this._eisMinYear] = carGasolineFE;
                carFeGrowCAFE  .DieselCAFE     [i - this._eisMinYear] = carDieselFE;
                carFeGrowCAFE  .DieselShareCAFE[i - this._eisMinYear] = carDslShare;
                truckFeGrowCAFE.GasolineCAFE   [i - this._eisMinYear] = truckGasolineFE;
                truckFeGrowCAFE.DieselCAFE     [i - this._eisMinYear] = truckDieselFE;
                truckFeGrowCAFE.DieselShareCAFE[i - this._eisMinYear] = truckDslShare;
                if (i == maxYear)
                {
                    for (int j = i + 1; j <= this._eisMaxYear; j++)
                    {
                        carFeFlatCAFE  .GasolineCAFE   [j - this._eisMinYear] = carFeFlatCAFE  .GasolineCAFE[j - this._eisMinYear - 1];
                        carFeFlatCAFE  .DieselCAFE     [j - this._eisMinYear] = carFeFlatCAFE  .DieselCAFE  [j - this._eisMinYear - 1];
                        carFeFlatCAFE  .DieselShareCAFE[j - this._eisMinYear] = carDslShare;
                        truckFeFlatCAFE.GasolineCAFE   [j - this._eisMinYear] = truckFeFlatCAFE.GasolineCAFE[j - this._eisMinYear - 1];
                        truckFeFlatCAFE.DieselCAFE     [j - this._eisMinYear] = truckFeFlatCAFE.DieselCAFE  [j - this._eisMinYear - 1];
                        truckFeFlatCAFE.DieselShareCAFE[j - this._eisMinYear] = truckDslShare;
                        double carGrowth = 1 + ((j > 2030) ? 0 : (j <= 2020) ? eisScenData.AutoCAFEGrowth1 : eisScenData.AutoCAFEGrowth2);
                        double trkGrowth = 1 + ((j > 2030) ? 0 : (j <= 2020) ? eisScenData.LtCAFEGrowth1   : eisScenData.LtCAFEGrowth2  );
                        carFeGrowCAFE  .GasolineCAFE   [j - this._eisMinYear] = (carFeGrowCAFE  .GasolineCAFE[j - this._eisMinYear - 1] * carGrowth);
                        carFeGrowCAFE  .DieselCAFE     [j - this._eisMinYear] = (carFeGrowCAFE  .DieselCAFE  [j - this._eisMinYear - 1] * carGrowth);
                        carFeGrowCAFE  .DieselShareCAFE[j - this._eisMinYear] =  carDslShare;
                        truckFeGrowCAFE.GasolineCAFE   [j - this._eisMinYear] = (truckFeGrowCAFE.GasolineCAFE[j - this._eisMinYear - 1] * trkGrowth);
                        truckFeGrowCAFE.DieselCAFE     [j - this._eisMinYear] = (truckFeGrowCAFE.DieselCAFE  [j - this._eisMinYear - 1] * trkGrowth);
                        truckFeGrowCAFE.DieselShareCAFE[j - this._eisMinYear] =  truckDslShare;
                    }
                } 
            } 
            this._carFEFlatCAFE  [scenIndex] = carFeFlatCAFE  .Clone();
            this._truckFEFlatCAFE[scenIndex] = truckFeFlatCAFE.Clone();
            this._carFEGrowCAFE  [scenIndex] = carFeGrowCAFE  .Clone();
            this._truckFEGrowCAFE[scenIndex] = truckFeGrowCAFE.Clone();
        }
        void CalcEisData(EisData data, bool isNoCafe, int scenIndex,
            EisParameters.FuelEconomy[] carFE, EisParameters.FuelEconomy[] truckFE)
        {
            double[,]            carGasVmt       = data.CarData.GasolineVMT;
            double[,]            carDslVmt       = data.CarData.DieselVMT;
            double[,]            carGasGallons   = data.CarData.GasolineGallons;
            double[,]            carDslGallons   = data.CarData.DieselGallons;
            EisData.EmissionData carAcetaldehyde = data.CarData.Acetaldehyde;
            EisData.EmissionData carAcrolein     = data.CarData.Acrolein;
            EisData.EmissionData carBenzene      = data.CarData.Benzene;
            EisData.EmissionData carButadiene    = data.CarData.Butadiene;
            EisData.EmissionData carCH4          = data.CarData.CH4;
            EisData.EmissionData carCO           = data.CarData.CO;
            EisData.EmissionData carCO2          = data.CarData.CO2;
            EisData.EmissionData carDieselPM     = data.CarData.DieselPM;
            EisData.EmissionData carFormaldehyde = data.CarData.Formaldehyde;
            EisData.EmissionData carMTBE         = data.CarData.MTBE;
            EisData.EmissionData carN2O          = data.CarData.N2O;
            EisData.EmissionData carNOx          = data.CarData.NOx;
            EisData.EmissionData carPM           = data.CarData.PM;
            EisData.EmissionData carSO2          = data.CarData.SO2;
            EisData.EmissionData carVOC          = data.CarData.VOC;
            double[,]            truckGasVmt       = data.TruckData.GasolineVMT;
            double[,]            truckDslVmt       = data.TruckData.DieselVMT;
            double[,]            truckGasGallons   = data.TruckData.GasolineGallons;
            double[,]            truckDslGallons   = data.TruckData.DieselGallons;
            EisData.EmissionData truckAcetaldehyde = data.TruckData.Acetaldehyde;
            EisData.EmissionData truckAcrolein     = data.TruckData.Acrolein;
            EisData.EmissionData truckBenzene      = data.TruckData.Benzene;
            EisData.EmissionData truckButadiene    = data.TruckData.Butadiene;
            EisData.EmissionData truckCH4          = data.TruckData.CH4;
            EisData.EmissionData truckCO           = data.TruckData.CO;
            EisData.EmissionData truckCO2          = data.TruckData.CO2;
            EisData.EmissionData truckDieselPM     = data.TruckData.DieselPM;
            EisData.EmissionData truckFormaldehyde = data.TruckData.Formaldehyde;
            EisData.EmissionData truckMTBE         = data.TruckData.MTBE;
            EisData.EmissionData truckN2O          = data.TruckData.N2O;
            EisData.EmissionData truckNOx          = data.TruckData.NOx;
            EisData.EmissionData truckPM           = data.TruckData.PM;
            EisData.EmissionData truckSO2          = data.TruckData.SO2;
            EisData.EmissionData truckVOC          = data.TruckData.VOC;
            int       baseUsageYear    = this._params.BaseParameters.BaseUsageYear - this._eisMinYear;
            double    carGrowthRate    = this._params.BaseParameters.AutoGrowthRate;
            double    truckGrowthRate  = this._params.BaseParameters.LtGrowthRate;
            double    reboundEffect    = this._params.BaseParameters.ReboundEffect;
            double    elasticity       = this._params.BaseParameters.Elasticity;
            double    onRoadGap        = this._params.BaseParameters.TestVsOnRoadGap;
            double[]  gasPrice         = null;
            double[]  dslPrice         = null;
            if (this._fuelPriceEstimates == Estimates.Low)
            {
                gasPrice = this._params.LowGasolinePrice;
                dslPrice = this._params.LowDieselPrice;
            }
            else if (this._fuelPriceEstimates == Estimates.High)
            {
                gasPrice = this._params.HighGasolinePrice;
                dslPrice = this._params.HighDieselPrice;
            }
            else 
            {
                gasPrice = this._params.RefGasolinePrice;
                dslPrice = this._params.RefDieselPrice;
            }
            double[]  carBaseGasCAFE   = carFE[0].GasolineCAFE;
            double[]  carBaseDslCAFE   = carFE[0].DieselCAFE;
            double[]  carGasCAFE       = (isNoCafe) ? this._params.CarFuelEconomy.GasolineNoCAFE    : carFE[scenIndex].GasolineCAFE;
            double[]  carDslCAFE       = (isNoCafe) ? this._params.CarFuelEconomy.DieselNoCAFE      : carFE[scenIndex].DieselCAFE;
            double[]  carDslShare      = (isNoCafe) ? this._params.CarFuelEconomy.DieselShareNoCAFE : carFE[scenIndex].DieselShareCAFE;
            double[]  carMilesDriven   =
                (this._fuelPriceEstimates == Estimates.Low ) ? this._params.LowCarMilesDriven  :
                (this._fuelPriceEstimates == Estimates.High) ? this._params.HighCarMilesDriven : this._params.RefCarMilesDriven;
            double[,] carFleet         = this._params.CarFleet;
            double[]  truckBaseGasCAFE = truckFE[0].GasolineCAFE;
            double[]  truckBaseDslCAFE = truckFE[0].DieselCAFE;
            double[]  truckGasCAFE     = (isNoCafe) ? this._params.TruckFuelEconomy.GasolineNoCAFE    : truckFE[scenIndex].GasolineCAFE;
            double[]  truckDslCAFE     = (isNoCafe) ? this._params.TruckFuelEconomy.DieselNoCAFE      : truckFE[scenIndex].DieselCAFE;
            double[]  truckDslShare    = (isNoCafe) ? this._params.TruckFuelEconomy.DieselShareNoCAFE : truckFE[scenIndex].DieselShareCAFE;
            double[]  truckMilesDriven =
                (this._fuelPriceEstimates == Estimates.Low ) ? this._params.LowTruckMilesDriven  :
                (this._fuelPriceEstimates == Estimates.High) ? this._params.HighTruckMilesDriven : this._params.RefTruckMilesDriven;
            double[,] truckFleet       = this._params.TruckFleet;
            EisTailpipeEmissions.Mobile6Type acetaldehyde = this._tailpipe.Acetaldehyde;
            EisTailpipeEmissions.Mobile6Type acrolein     = this._tailpipe.Acrolein;
            EisTailpipeEmissions.Mobile6Type benzene      = this._tailpipe.Benzene;
            EisTailpipeEmissions.Mobile6Type butadiene    = this._tailpipe.Butadiene;
            EisTailpipeEmissions.Mobile6Type ch4          = this._tailpipe.CH4;
            EisTailpipeEmissions.Mobile6Type co           = this._tailpipe.CO;
            double                           co2Gas       = this._params.GasolineTailpipeCO2;
            double                           co2Dsl       = this._params.DieselTailpipeCO2;
            EisTailpipeEmissions.Mobile6Type dieselPM     = this._tailpipe.DieselPM;
            EisTailpipeEmissions.Mobile6Type formaldehyde = this._tailpipe.Formaldehyde;
            EisTailpipeEmissions.Mobile6Type mtbe         = this._tailpipe.MTBE;
            EisTailpipeEmissions.Mobile6Type n2o          = this._tailpipe.N2O;
            EisTailpipeEmissions.Mobile6Type nox          = this._tailpipe.NOx;
            EisTailpipeEmissions.Mobile6Type pm           = this._tailpipe.PM;
            double                           so2Gas       = this._params.GasolineTailpipeSO2;
            double                           so2Dsl       = this._params.DieselTailpipeSO2;
            EisTailpipeEmissions.Mobile6Type voc          = this._tailpipe.VOC;
            EisParameters.UpstreamEmissions gasUpstream   = this._params.GasolineUpstream;
            EisParameters.UpstreamEmissions dslUpstream   = this._params.DieselUpstream;
            for (int i = 0; i < this._yearCount; i++)       
            {
                for (int j = i; j < this._yearCount; j++)   
                {
                    if (carGasCAFE[i] > 0)
                    {
                        carGasVmt[i, j] =
                            carFleet[i, j] * (1 - carDslShare[i]) * carMilesDriven[Math.Min(j - i, carMilesDriven.Length - 1)] *
                            Math.Pow(1 + carGrowthRate, j - baseUsageYear) *
                            (1 + elasticity * (carBaseGasCAFE[i] / carGasCAFE[i] - 1)) * 1e-9;
                    }
                    if (carDslCAFE[i] > 0)
                    {
                        carDslVmt[i, j] =
                            carFleet[i, j] * (carDslShare[i]) * carMilesDriven[Math.Min(j - i, carMilesDriven.Length - 1)] *
                            Math.Pow(1 + carGrowthRate, j - baseUsageYear) *
                            (1 + elasticity * (carBaseDslCAFE[i] / carDslCAFE[i] - 1)) * 1e-9;
                    }
                    if (truckGasCAFE[i] > 0)
                    {
                        truckGasVmt[i, j] =
                            truckFleet[i, j] * (1 - truckDslShare[i]) * truckMilesDriven[Math.Min(j - i, truckMilesDriven.Length - 1)] *
                            Math.Pow(1 + truckGrowthRate, j - baseUsageYear) *
                            (1 + elasticity * (truckBaseGasCAFE[i] / truckGasCAFE[i] - 1)) * 1e-9;
                    }
                    if (truckDslCAFE[i] > 0)
                    {
                        truckDslVmt[i, j] =
                            truckFleet[i, j] * (truckDslShare[i]) * truckMilesDriven[Math.Min(j - i, truckMilesDriven.Length - 1)] *
                            Math.Pow(1 + truckGrowthRate, j - baseUsageYear) *
                            (1 + elasticity * (truckBaseGasCAFE[i] / truckDslCAFE[i] - 1)) * 1e-9;
                    }
                    if (carGasCAFE  [i] > 0) { carGasGallons  [i, j] = carGasVmt  [i, j] / (carGasCAFE  [i] * (1 - onRoadGap)); }
                    if (carDslCAFE  [i] > 0) { carDslGallons  [i, j] = carDslVmt  [i, j] / (carDslCAFE  [i] * (1 - onRoadGap)); }
                    if (truckGasCAFE[i] > 0) { truckGasGallons[i, j] = truckGasVmt[i, j] / (truckGasCAFE[i] * (1 - onRoadGap)); }
                    if (truckDslCAFE[i] > 0) { truckDslGallons[i, j] = truckDslVmt[i, j] / (truckDslCAFE[i] * (1 - onRoadGap)); }
                    carAcetaldehyde.TailpipeEmissions[i, j] = (carGasVmt    [i, j] * acetaldehyde.LDGV[i, j] + carDslVmt    [i, j] * acetaldehyde.LDDV[i, j]) * EmFract;
                    carAcrolein    .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * acrolein    .LDGV[i, j] + carDslVmt    [i, j] * acrolein    .LDDV[i, j]) * EmFract;
                    carBenzene     .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * benzene     .LDGV[i, j] + carDslVmt    [i, j] * benzene     .LDDV[i, j]) * EmFract;
                    carButadiene   .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * butadiene   .LDGV[i, j] + carDslVmt    [i, j] * butadiene   .LDDV[i, j]) * EmFract;
                    carCH4         .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * ch4         .LDGV[i, j] + carDslVmt    [i, j] * ch4         .LDDV[i, j]) * EmFract;
                    carCO          .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * co          .LDGV[i, j] + carDslVmt    [i, j] * co          .LDDV[i, j]) * EmFract;
                    carCO2         .TailpipeEmissions[i, j] = (carGasGallons[i, j] * co2Gas                  + carDslGallons[i, j] * co2Dsl                 ) * EmFract;
                    carDieselPM    .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * dieselPM    .LDGV[i, j] + carDslVmt    [i, j] * dieselPM    .LDDV[i, j]) * EmFract;
                    carFormaldehyde.TailpipeEmissions[i, j] = (carGasVmt    [i, j] * formaldehyde.LDGV[i, j] + carDslVmt    [i, j] * formaldehyde.LDDV[i, j]) * EmFract;
                    carMTBE        .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * mtbe        .LDGV[i, j] + carDslVmt    [i, j] * mtbe        .LDDV[i, j]) * EmFract;
                    carN2O         .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * n2o         .LDGV[i, j] + carDslVmt    [i, j] * n2o         .LDDV[i, j]) * EmFract;
                    carNOx         .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * nox         .LDGV[i, j] + carDslVmt    [i, j] * nox         .LDDV[i, j]) * EmFract;
                    carPM          .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * pm          .LDGV[i, j] + carDslVmt    [i, j] * pm          .LDDV[i, j]) * EmFract;
                    carSO2         .TailpipeEmissions[i, j] = (carGasGallons[i, j] * so2Gas                  + carDslGallons[i, j] * so2Dsl                 ) * EmFract;
                    carVOC         .TailpipeEmissions[i, j] = (carGasVmt    [i, j] * voc         .LDGV[i, j] + carDslVmt    [i, j] * voc         .LDDV[i, j]) * EmFract;
                    carAcetaldehyde.UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.Acetaldehyde + carDslGallons[i, j] * dslUpstream.Acetaldehyde) * EmFract;
                    carAcrolein    .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.Acrolein     + carDslGallons[i, j] * dslUpstream.Acrolein    ) * EmFract;
                    carBenzene     .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.Benzene      + carDslGallons[i, j] * dslUpstream.Benzene     ) * EmFract;
                    carButadiene   .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.Butadiene    + carDslGallons[i, j] * dslUpstream.Butadiene   ) * EmFract;
                    carCH4         .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.CH4          + carDslGallons[i, j] * dslUpstream.CH4         ) * EmFract;
                    carCO          .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.CO           + carDslGallons[i, j] * dslUpstream.CO          ) * EmFract;
                    carCO2         .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.CO2          + carDslGallons[i, j] * dslUpstream.CO2         ) * EmFract;
                    carDieselPM    .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.DieselPM     + carDslGallons[i, j] * dslUpstream.DieselPM    ) * EmFract;
                    carFormaldehyde.UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.Formaldehyde + carDslGallons[i, j] * dslUpstream.Formaldehyde) * EmFract;
                    carMTBE        .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.MTBE         + carDslGallons[i, j] * dslUpstream.MTBE        ) * EmFract;
                    carN2O         .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.N2O          + carDslGallons[i, j] * dslUpstream.N2O         ) * EmFract;
                    carNOx         .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.NOx          + carDslGallons[i, j] * dslUpstream.NOx         ) * EmFract;
                    carPM          .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.PM           + carDslGallons[i, j] * dslUpstream.PM          ) * EmFract;
                    carSO2         .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.SOx          + carDslGallons[i, j] * dslUpstream.SOx         ) * EmFract;
                    carVOC         .UpstreamEmissions[i, j] = (carGasGallons[i, j] * gasUpstream.VOC          + carDslGallons[i, j] * dslUpstream.VOC         ) * EmFract;
                    carAcetaldehyde.TotalEmissions[i, j] = carAcetaldehyde.TailpipeEmissions[i, j] + carAcetaldehyde.UpstreamEmissions[i, j];
                    carAcrolein    .TotalEmissions[i, j] = carAcrolein    .TailpipeEmissions[i, j] + carAcrolein    .UpstreamEmissions[i, j];
                    carBenzene     .TotalEmissions[i, j] = carBenzene     .TailpipeEmissions[i, j] + carBenzene     .UpstreamEmissions[i, j];
                    carButadiene   .TotalEmissions[i, j] = carButadiene   .TailpipeEmissions[i, j] + carButadiene   .UpstreamEmissions[i, j];
                    carCH4         .TotalEmissions[i, j] = carCH4         .TailpipeEmissions[i, j] + carCH4         .UpstreamEmissions[i, j];
                    carCO          .TotalEmissions[i, j] = carCO          .TailpipeEmissions[i, j] + carCO          .UpstreamEmissions[i, j];
                    carCO2         .TotalEmissions[i, j] = carCO2         .TailpipeEmissions[i, j] + carCO2         .UpstreamEmissions[i, j];
                    carDieselPM    .TotalEmissions[i, j] = carDieselPM    .TailpipeEmissions[i, j] + carDieselPM    .UpstreamEmissions[i, j];
                    carFormaldehyde.TotalEmissions[i, j] = carFormaldehyde.TailpipeEmissions[i, j] + carFormaldehyde.UpstreamEmissions[i, j];
                    carMTBE        .TotalEmissions[i, j] = carMTBE        .TailpipeEmissions[i, j] + carMTBE        .UpstreamEmissions[i, j];
                    carN2O         .TotalEmissions[i, j] = carN2O         .TailpipeEmissions[i, j] + carN2O         .UpstreamEmissions[i, j];
                    carNOx         .TotalEmissions[i, j] = carNOx         .TailpipeEmissions[i, j] + carNOx         .UpstreamEmissions[i, j];
                    carPM          .TotalEmissions[i, j] = carPM          .TailpipeEmissions[i, j] + carPM          .UpstreamEmissions[i, j];
                    carSO2         .TotalEmissions[i, j] = carSO2         .TailpipeEmissions[i, j] + carSO2         .UpstreamEmissions[i, j];
                    carVOC         .TotalEmissions[i, j] = carVOC         .TailpipeEmissions[i, j] + carVOC         .UpstreamEmissions[i, j];
                    truckAcetaldehyde.TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * acetaldehyde.LDGT[i, j] + truckDslVmt    [i, j] * acetaldehyde.LDDT[i, j]) * EmFract;
                    truckAcrolein    .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * acrolein    .LDGT[i, j] + truckDslVmt    [i, j] * acrolein    .LDDT[i, j]) * EmFract;
                    truckBenzene     .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * benzene     .LDGT[i, j] + truckDslVmt    [i, j] * benzene     .LDDT[i, j]) * EmFract;
                    truckButadiene   .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * butadiene   .LDGT[i, j] + truckDslVmt    [i, j] * butadiene   .LDDT[i, j]) * EmFract;
                    truckCH4         .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * ch4         .LDGT[i, j] + truckDslVmt    [i, j] * ch4         .LDDT[i, j]) * EmFract;
                    truckCO          .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * co          .LDGT[i, j] + truckDslVmt    [i, j] * co          .LDDT[i, j]) * EmFract;
                    truckCO2         .TailpipeEmissions[i, j] = (truckGasGallons[i, j] * co2Gas                  + truckDslGallons[i, j] * co2Dsl                 ) * EmFract;
                    truckDieselPM    .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * dieselPM    .LDGT[i, j] + truckDslVmt    [i, j] * dieselPM    .LDDT[i, j]) * EmFract;
                    truckFormaldehyde.TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * formaldehyde.LDGT[i, j] + truckDslVmt    [i, j] * formaldehyde.LDDT[i, j]) * EmFract;
                    truckMTBE        .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * mtbe        .LDGT[i, j] + truckDslVmt    [i, j] * mtbe        .LDDT[i, j]) * EmFract;
                    truckN2O         .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * n2o         .LDGT[i, j] + truckDslVmt    [i, j] * n2o         .LDDT[i, j]) * EmFract;
                    truckNOx         .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * nox         .LDGT[i, j] + truckDslVmt    [i, j] * nox         .LDDT[i, j]) * EmFract;
                    truckPM          .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * pm          .LDGT[i, j] + truckDslVmt    [i, j] * pm          .LDDT[i, j]) * EmFract;
                    truckSO2         .TailpipeEmissions[i, j] = (truckGasGallons[i, j] * so2Gas                  + truckDslGallons[i, j] * so2Dsl                 ) * EmFract;
                    truckVOC         .TailpipeEmissions[i, j] = (truckGasVmt    [i, j] * voc         .LDGT[i, j] + truckDslVmt    [i, j] * voc         .LDDT[i, j]) * EmFract;
                    truckAcetaldehyde.UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.Acetaldehyde + truckDslGallons[i, j] * dslUpstream.Acetaldehyde) * EmFract;
                    truckAcrolein    .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.Acrolein     + truckDslGallons[i, j] * dslUpstream.Acrolein    ) * EmFract;
                    truckBenzene     .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.Benzene      + truckDslGallons[i, j] * dslUpstream.Benzene     ) * EmFract;
                    truckButadiene   .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.Butadiene    + truckDslGallons[i, j] * dslUpstream.Butadiene   ) * EmFract;
                    truckCH4         .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.CH4          + truckDslGallons[i, j] * dslUpstream.CH4         ) * EmFract;
                    truckCO          .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.CO           + truckDslGallons[i, j] * dslUpstream.CO          ) * EmFract;
                    truckCO2         .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.CO2          + truckDslGallons[i, j] * dslUpstream.CO2         ) * EmFract;
                    truckDieselPM    .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.DieselPM     + truckDslGallons[i, j] * dslUpstream.DieselPM    ) * EmFract;
                    truckFormaldehyde.UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.Formaldehyde + truckDslGallons[i, j] * dslUpstream.Formaldehyde) * EmFract;
                    truckMTBE        .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.MTBE         + truckDslGallons[i, j] * dslUpstream.MTBE        ) * EmFract;
                    truckN2O         .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.N2O          + truckDslGallons[i, j] * dslUpstream.N2O         ) * EmFract;
                    truckNOx         .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.NOx          + truckDslGallons[i, j] * dslUpstream.NOx         ) * EmFract;
                    truckPM          .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.PM           + truckDslGallons[i, j] * dslUpstream.PM          ) * EmFract;
                    truckSO2         .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.SOx          + truckDslGallons[i, j] * dslUpstream.SOx         ) * EmFract;
                    truckVOC         .UpstreamEmissions[i, j] = (truckGasGallons[i, j] * gasUpstream.VOC          + truckDslGallons[i, j] * dslUpstream.VOC         ) * EmFract;
                    truckAcetaldehyde.TotalEmissions[i, j] = truckAcetaldehyde.TailpipeEmissions[i, j] + truckAcetaldehyde.UpstreamEmissions[i, j];
                    truckAcrolein    .TotalEmissions[i, j] = truckAcrolein    .TailpipeEmissions[i, j] + truckAcrolein    .UpstreamEmissions[i, j];
                    truckBenzene     .TotalEmissions[i, j] = truckBenzene     .TailpipeEmissions[i, j] + truckBenzene     .UpstreamEmissions[i, j];
                    truckButadiene   .TotalEmissions[i, j] = truckButadiene   .TailpipeEmissions[i, j] + truckButadiene   .UpstreamEmissions[i, j];
                    truckCH4         .TotalEmissions[i, j] = truckCH4         .TailpipeEmissions[i, j] + truckCH4         .UpstreamEmissions[i, j];
                    truckCO          .TotalEmissions[i, j] = truckCO          .TailpipeEmissions[i, j] + truckCO          .UpstreamEmissions[i, j];
                    truckCO2         .TotalEmissions[i, j] = truckCO2         .TailpipeEmissions[i, j] + truckCO2         .UpstreamEmissions[i, j];
                    truckDieselPM    .TotalEmissions[i, j] = truckDieselPM    .TailpipeEmissions[i, j] + truckDieselPM    .UpstreamEmissions[i, j];
                    truckFormaldehyde.TotalEmissions[i, j] = truckFormaldehyde.TailpipeEmissions[i, j] + truckFormaldehyde.UpstreamEmissions[i, j];
                    truckMTBE        .TotalEmissions[i, j] = truckMTBE        .TailpipeEmissions[i, j] + truckMTBE        .UpstreamEmissions[i, j];
                    truckN2O         .TotalEmissions[i, j] = truckN2O         .TailpipeEmissions[i, j] + truckN2O         .UpstreamEmissions[i, j];
                    truckNOx         .TotalEmissions[i, j] = truckNOx         .TailpipeEmissions[i, j] + truckNOx         .UpstreamEmissions[i, j];
                    truckPM          .TotalEmissions[i, j] = truckPM          .TailpipeEmissions[i, j] + truckPM          .UpstreamEmissions[i, j];
                    truckSO2         .TotalEmissions[i, j] = truckSO2         .TailpipeEmissions[i, j] + truckSO2         .UpstreamEmissions[i, j];
                    truckVOC         .TotalEmissions[i, j] = truckVOC         .TailpipeEmissions[i, j] + truckVOC         .UpstreamEmissions[i, j];
                } 
            } 
        }
        public void GenerateReports(string path, EISSettings settings, Input template)
        {
            if (!Directory.Exists(path)) { throw new IOException("The directory provided for EIS reports does not exist."); }
            if (settings.RunEISModelFlatCAFE) { this.GenerateReports_Internal(path + "\\EIS" , template, this._carFEFlatCAFE, this._truckFEFlatCAFE, this._noCafeDataFlatCAFE, this._scenDataFlatCAFE); }
            if (settings.RunEISModelGrowCAFE) { this.GenerateReports_Internal(path + "\\EIS2", template, this._carFEGrowCAFE, this._truckFEGrowCAFE, this._noCafeDataGrowCAFE, this._scenDataGrowCAFE); }
        }
        void GenerateReports_Internal(string path, Input template,
            EisParameters.FuelEconomy[] carFE, EisParameters.FuelEconomy[] truckFE, EisData noCafeData, EisData[] scenData)
        {
            Directory.CreateDirectory(path);
            this.GenerateVmtGallonsCAFEReports(path, template, carFE, truckFE, noCafeData, scenData);
            this.GenerateEmissionsReports     (path, template, carFE, truckFE, noCafeData, scenData);
        }
        void GenerateVmtGallonsCAFEReports(string path, Input template,
            EisParameters.FuelEconomy[] carFE, EisParameters.FuelEconomy[] truckFE, EisData noCafeData, EisData[] scenData)
        {
            Output vmt     = this.OpenReport(path, "VMT"    , TemplateType.EIS_VMT);
            Output gallons = this.OpenReport(path, "Gallons", TemplateType.EIS_Gallons);
            Output cafe    = this.OpenReport(path, "CAFE"   , TemplateType.EIS_CAFE);
            int eisMinYear = noCafeData.MinYear;
            int eisMaxYear = noCafeData.MaxYear;
            this.GenerateCAFEReport(cafe, "Cars", this._params.CarFuelEconomy, carFE);
            this.GenerateVmtGallonsReport(vmt    , noCafeData.CarData.GasolineVMT    , noCafeData.CarData.DieselVMT    , true, true, -1, "VMT"    );
            this.GenerateVmtGallonsReport(gallons, noCafeData.CarData.GasolineGallons, noCafeData.CarData.DieselGallons, true, true, -1, "Gallons");
            for (int i = 0; i < scenData.Length; i++)
            {
                this.GenerateVmtGallonsReport(vmt    , scenData[i].CarData.GasolineVMT    , scenData[i].CarData.DieselVMT    , false, true, i, "VMT"    );
                this.GenerateVmtGallonsReport(gallons, scenData[i].CarData.GasolineGallons, scenData[i].CarData.DieselGallons, false, true, i, "Gallons");
            }
            this.GenerateCAFEReport(cafe, "Trucks", this._params.TruckFuelEconomy, truckFE);
            this.GenerateVmtGallonsReport(vmt    , noCafeData.TruckData.GasolineVMT    , noCafeData.TruckData.DieselVMT    , true, false, -1, "VMT"    );
            this.GenerateVmtGallonsReport(gallons, noCafeData.TruckData.GasolineGallons, noCafeData.TruckData.DieselGallons, true, false, -1, "Gallons");
            for (int i = 0; i < scenData.Length; i++)
            {
                this.GenerateVmtGallonsReport(vmt    , scenData[i].TruckData.GasolineVMT    , scenData[i].TruckData.DieselVMT    , false, false, i, "VMT"    );
                this.GenerateVmtGallonsReport(gallons, scenData[i].TruckData.GasolineGallons, scenData[i].TruckData.DieselGallons, false, false, i, "Gallons");
            }
            this.FinalizeCAFEReport(cafe   , template, 1 + scenData.Length);
            this.FinalizeReport    (vmt    , template, false);
            this.FinalizeReport    (gallons, template, false);
        }
        void GenerateCAFEReport(Output report, string name, EisParameters.FuelEconomy noCafe, EisParameters.FuelEconomy[] cafe)
        {
            int rows = this._yearCount + 4;
            int cols = (1 + cafe.Length) * 3 + 1;
            report.AddBuffer(rows, cols);
            report.Buffer.Name = name;
            int row = 0, col = 0;
            int scenCount = cafe.Length;
            report.Buffer[row++, col] = name + " Fuel Economy Projections";
            row++;  
            report.Buffer[row, col++] = "Model\nYear";
            report.Buffer[row, col] = "No CAFE"; col += 3;
            for (int i = 0; i < scenCount; i++)
            {
                report.Buffer[row, col] = "Scenario " + i; col += 3;
            }
            row++; col = 1;
            for (int i = 0; i < scenCount + 1; i++)
            {
                report.Buffer[row, col++] = "Gasoline";
                report.Buffer[row, col++] = "Diesel";
                report.Buffer[row, col++] = "Diesel Share";
            }
            for (int i = 0; i < this._yearCount; i++)
            {
                row++; col = 0;
                report.Buffer[row, col++] = this._eisMinYear + i;
                report.Buffer[row, col++] = noCafe.GasolineNoCAFE   [i];
                report.Buffer[row, col++] = noCafe.DieselNoCAFE     [i];
                report.Buffer[row, col++] = noCafe.DieselShareNoCAFE[i];
                for (int j = 0; j < scenCount; j++)
                {
                    report.Buffer[row, col++] = cafe[j].GasolineCAFE   [i];
                    report.Buffer[row, col++] = cafe[j].DieselCAFE     [i];
                    report.Buffer[row, col++] = cafe[j].DieselShareCAFE[i];
                }
            }
        }
        void GenerateVmtGallonsReport(Output report, double[,] gasoline, double[,] diesel, bool isNoCafe, bool isCar,
            int scenIndex, string vmtGallonsString)
        {
            int rows = 2 * (this._yearCount + 4) + 1;
            int cols = this._yearCount + 2;
            report.AddBuffer(rows, cols);
            report.Buffer.Name = ((isNoCafe) ? "NoCAFE" : "Sn" + scenIndex) + ((isCar) ? "_Cars" : "_Trucks");
            int row = 0, col = 0;
            report.Buffer[row++, col] = ((isCar) ? "Car" : "Truck") + " " + vmtGallonsString + ", " +
                ((isNoCafe) ? "No CAFE" : "CAFE Sn-" + scenIndex);
            row++;  
            report.Buffer[row, col++] = "Model\nYear";
            report.Buffer[row++, col] = "Gasoline -- Calendar Year";
            for (int i = 0; i < this._yearCount; i++)
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
            }
            report.Buffer[row++, col] = "Total";
            col = 0;
            for (int i = 0; i < this._yearCount; i++)       
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
                for (int j = 0; j < this._yearCount; j++)   
                {
                    report.Buffer[row                      , col     ] =                                                                        gasoline[i, j];
                    report.Buffer[row                      , cols - 1] = Global.GetDouble(report.Buffer[row                      , cols - 1]) + gasoline[i, j];
                    report.Buffer[row + this._yearCount - i, col     ] = Global.GetDouble(report.Buffer[row + this._yearCount - i, col     ]) + gasoline[i, j];
                    report.Buffer[row + this._yearCount - i, cols - 1] = Global.GetDouble(report.Buffer[row + this._yearCount - i, cols - 1]) + gasoline[i, j];
                    col++;
                }
                row++;
                col = 0;
            }
            report.Buffer[row++, col] = "Total";
            row++;
            report.Buffer[row, col++] = "Model\nYear";
            report.Buffer[row++, col] = "Diesel -- Calendar Year";
            for (int i = 0; i < this._yearCount; i++)
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
            }
            report.Buffer[row++, col] = "Total";
            col = 0;
            for (int i = 0; i < this._yearCount; i++)       
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
                for (int j = 0; j < this._yearCount; j++)   
                {
                    report.Buffer[row                      , col     ] =                                                                        diesel[i, j];
                    report.Buffer[row                      , cols - 1] = Global.GetDouble(report.Buffer[row                      , cols - 1]) + diesel[i, j];
                    report.Buffer[row + this._yearCount - i, col     ] = Global.GetDouble(report.Buffer[row + this._yearCount - i, col     ]) + diesel[i, j];
                    report.Buffer[row + this._yearCount - i, cols - 1] = Global.GetDouble(report.Buffer[row + this._yearCount - i, cols - 1]) + diesel[i, j];
                    col++;
                }
                row++;
                col = 0;
            }
            report.Buffer[row++, col] = "Total";
        }
        void GenerateEmissionsReports(string path, Input template,
            EisParameters.FuelEconomy[] carFE, EisParameters.FuelEconomy[] truckFE, EisData noCafeData, EisData[] scenData)
        {
            Output acetaldehyde = this.OpenReport(path, "Acetaldehyde", TemplateType.EIS_Emissions);
            Output acrolein     = this.OpenReport(path, "Acrolein"    , TemplateType.EIS_Emissions);
            Output benzene      = this.OpenReport(path, "Benzene"     , TemplateType.EIS_Emissions);
            Output butadiene    = this.OpenReport(path, "Butadiene"   , TemplateType.EIS_Emissions);
            Output ch4          = this.OpenReport(path, "CH4"         , TemplateType.EIS_Emissions);
            Output co           = this.OpenReport(path, "CO"          , TemplateType.EIS_Emissions);
            Output co2          = this.OpenReport(path, "CO2"         , TemplateType.EIS_Emissions);
            Output dieselPM     = this.OpenReport(path, "Diesel PM10" , TemplateType.EIS_Emissions);
            Output formaldehyde = this.OpenReport(path, "Formaldehyde", TemplateType.EIS_Emissions);
            Output mtbe         = this.OpenReport(path, "MTBE"        , TemplateType.EIS_Emissions);
            Output n2o          = this.OpenReport(path, "N2O"         , TemplateType.EIS_Emissions);
            Output nox          = this.OpenReport(path, "NOx"         , TemplateType.EIS_Emissions);
            Output pm           = this.OpenReport(path, "PM"          , TemplateType.EIS_Emissions);
            Output so2          = this.OpenReport(path, "SO2"         , TemplateType.EIS_Emissions);
            Output voc          = this.OpenReport(path, "VOC"         , TemplateType.EIS_Emissions);
            int eisMinYear = noCafeData.MinYear;
            int eisMaxYear = noCafeData.MaxYear;
            this.GenerateEmissionsReport(acetaldehyde, noCafeData.CarData.Acetaldehyde, true, true, -1);
            this.GenerateEmissionsReport(acrolein    , noCafeData.CarData.Acrolein    , true, true, -1);
            this.GenerateEmissionsReport(benzene     , noCafeData.CarData.Benzene     , true, true, -1);
            this.GenerateEmissionsReport(butadiene   , noCafeData.CarData.Butadiene   , true, true, -1);
            this.GenerateEmissionsReport(ch4         , noCafeData.CarData.CH4         , true, true, -1);
            this.GenerateEmissionsReport(co          , noCafeData.CarData.CO          , true, true, -1);
            this.GenerateEmissionsReport(co2         , noCafeData.CarData.CO2         , true, true, -1);
            this.GenerateEmissionsReport(dieselPM    , noCafeData.CarData.DieselPM    , true, true, -1);
            this.GenerateEmissionsReport(formaldehyde, noCafeData.CarData.Formaldehyde, true, true, -1);
            this.GenerateEmissionsReport(mtbe        , noCafeData.CarData.MTBE        , true, true, -1);
            this.GenerateEmissionsReport(n2o         , noCafeData.CarData.N2O         , true, true, -1);
            this.GenerateEmissionsReport(nox         , noCafeData.CarData.NOx         , true, true, -1);
            this.GenerateEmissionsReport(pm          , noCafeData.CarData.PM          , true, true, -1);
            this.GenerateEmissionsReport(so2         , noCafeData.CarData.SO2         , true, true, -1);
            this.GenerateEmissionsReport(voc         , noCafeData.CarData.VOC         , true, true, -1);
            for (int i = 0; i < scenData.Length; i++)
            {
                this.GenerateEmissionsReport(acetaldehyde, scenData[i].CarData.Acetaldehyde, false, true, i);
                this.GenerateEmissionsReport(acrolein    , scenData[i].CarData.Acrolein    , false, true, i);
                this.GenerateEmissionsReport(benzene     , scenData[i].CarData.Benzene     , false, true, i);
                this.GenerateEmissionsReport(butadiene   , scenData[i].CarData.Butadiene   , false, true, i);
                this.GenerateEmissionsReport(ch4         , scenData[i].CarData.CH4         , false, true, i);
                this.GenerateEmissionsReport(co          , scenData[i].CarData.CO          , false, true, i);
                this.GenerateEmissionsReport(co2         , scenData[i].CarData.CO2         , false, true, i);
                this.GenerateEmissionsReport(dieselPM    , scenData[i].CarData.DieselPM    , false, true, i);
                this.GenerateEmissionsReport(formaldehyde, scenData[i].CarData.Formaldehyde, false, true, i);
                this.GenerateEmissionsReport(mtbe        , scenData[i].CarData.MTBE        , false, true, i);
                this.GenerateEmissionsReport(n2o         , scenData[i].CarData.N2O         , false, true, i);
                this.GenerateEmissionsReport(nox         , scenData[i].CarData.NOx         , false, true, i);
                this.GenerateEmissionsReport(pm          , scenData[i].CarData.PM          , false, true, i);
                this.GenerateEmissionsReport(so2         , scenData[i].CarData.SO2         , false, true, i);
                this.GenerateEmissionsReport(voc         , scenData[i].CarData.VOC         , false, true, i);
            }
            this.GenerateEmissionsReport(acetaldehyde, noCafeData.TruckData.Acetaldehyde, true, false, -1);
            this.GenerateEmissionsReport(acrolein    , noCafeData.TruckData.Acrolein    , true, false, -1);
            this.GenerateEmissionsReport(benzene     , noCafeData.TruckData.Benzene     , true, false, -1);
            this.GenerateEmissionsReport(butadiene   , noCafeData.TruckData.Butadiene   , true, false, -1);
            this.GenerateEmissionsReport(ch4         , noCafeData.TruckData.CH4         , true, false, -1);
            this.GenerateEmissionsReport(co          , noCafeData.TruckData.CO          , true, false, -1);
            this.GenerateEmissionsReport(co2         , noCafeData.TruckData.CO2         , true, false, -1);
            this.GenerateEmissionsReport(dieselPM    , noCafeData.TruckData.DieselPM    , true, false, -1);
            this.GenerateEmissionsReport(formaldehyde, noCafeData.TruckData.Formaldehyde, true, false, -1);
            this.GenerateEmissionsReport(mtbe        , noCafeData.TruckData.MTBE        , true, false, -1);
            this.GenerateEmissionsReport(n2o         , noCafeData.TruckData.N2O         , true, false, -1);
            this.GenerateEmissionsReport(nox         , noCafeData.TruckData.NOx         , true, false, -1);
            this.GenerateEmissionsReport(pm          , noCafeData.TruckData.PM          , true, false, -1);
            this.GenerateEmissionsReport(so2         , noCafeData.TruckData.SO2         , true, false, -1);
            this.GenerateEmissionsReport(voc         , noCafeData.TruckData.VOC         , true, false, -1);
            for (int i = 0; i < scenData.Length; i++)
            {
                this.GenerateEmissionsReport(acetaldehyde, scenData[i].TruckData.Acetaldehyde, false, false, i);
                this.GenerateEmissionsReport(acrolein    , scenData[i].TruckData.Acrolein    , false, false, i);
                this.GenerateEmissionsReport(benzene     , scenData[i].TruckData.Benzene     , false, false, i);
                this.GenerateEmissionsReport(butadiene   , scenData[i].TruckData.Butadiene   , false, false, i);
                this.GenerateEmissionsReport(ch4         , scenData[i].TruckData.CH4         , false, false, i);
                this.GenerateEmissionsReport(co          , scenData[i].TruckData.CO          , false, false, i);
                this.GenerateEmissionsReport(co2         , scenData[i].TruckData.CO2         , false, false, i);
                this.GenerateEmissionsReport(dieselPM    , scenData[i].TruckData.DieselPM    , false, false, i);
                this.GenerateEmissionsReport(formaldehyde, scenData[i].TruckData.Formaldehyde, false, false, i);
                this.GenerateEmissionsReport(mtbe        , scenData[i].TruckData.MTBE        , false, false, i);
                this.GenerateEmissionsReport(n2o         , scenData[i].TruckData.N2O         , false, false, i);
                this.GenerateEmissionsReport(nox         , scenData[i].TruckData.NOx         , false, false, i);
                this.GenerateEmissionsReport(pm          , scenData[i].TruckData.PM          , false, false, i);
                this.GenerateEmissionsReport(so2         , scenData[i].TruckData.SO2         , false, false, i);
                this.GenerateEmissionsReport(voc         , scenData[i].TruckData.VOC         , false, false, i);
            }
            this.FinalizeReport(acetaldehyde, template, true);
            this.FinalizeReport(acrolein    , template, true);
            this.FinalizeReport(benzene     , template, true);
            this.FinalizeReport(butadiene   , template, true);
            this.FinalizeReport(ch4         , template, true);
            this.FinalizeReport(co          , template, true);
            this.FinalizeReport(co2         , template, true);
            this.FinalizeReport(dieselPM    , template, true);
            this.FinalizeReport(formaldehyde, template, true);
            this.FinalizeReport(mtbe        , template, true);
            this.FinalizeReport(n2o         , template, true);
            this.FinalizeReport(nox         , template, true);
            this.FinalizeReport(pm          , template, true);
            this.FinalizeReport(so2         , template, true);
            this.FinalizeReport(voc         , template, true);
        }
        void GenerateEmissionsReport(Output report, EisData.EmissionData data, bool isNoCafe, bool isCar, int scenIndex)
        {
            int rows = 3 * (this._yearCount + 4) + 1;
            int cols = this._yearCount + 2;
            report.AddBuffer(rows, cols);
            report.Buffer.Name = ((isNoCafe) ? "NoCAFE" : "Sn" + scenIndex) + ((isCar) ? "_Cars" : "_Trucks");
            int row = 0, col = 0;
            report.Buffer[row++, col] = ((isCar) ? "Car" : "Truck") + " Emissions, " + ((isNoCafe) ? "No CAFE" : "CAFE Sn-" + scenIndex);
            row++;  
            report.Buffer[row, col++] = "Model\nYear";
            report.Buffer[row++, col] = "Tailpipe Emissions -- Calendar Year";
            for (int i = 0; i < this._yearCount; i++)
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
            }
            report.Buffer[row++, col] = "Total";
            col = 0;
            for (int i = 0; i < this._yearCount; i++)       
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
                for (int j = 0; j < this._yearCount; j++)   
                {
                    report.Buffer[row                      , col     ] =                                                                        data.TailpipeEmissions[i, j];
                    report.Buffer[row                      , cols - 1] = Global.GetDouble(report.Buffer[row                      , cols - 1]) + data.TailpipeEmissions[i, j];
                    report.Buffer[row + this._yearCount - i, col     ] = Global.GetDouble(report.Buffer[row + this._yearCount - i, col     ]) + data.TailpipeEmissions[i, j];
                    report.Buffer[row + this._yearCount - i, cols - 1] = Global.GetDouble(report.Buffer[row + this._yearCount - i, cols - 1]) + data.TailpipeEmissions[i, j];
                    col++;
                }
                row++;
                col = 0;
            }
            report.Buffer[row++, col] = "Total";
            row++;
            report.Buffer[row, col++] = "Model\nYear";
            report.Buffer[row++, col] = "Upstream Emissions -- Calendar Year";
            for (int i = 0; i < this._yearCount; i++)
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
            }
            report.Buffer[row++, col] = "Total";
            col = 0;
            for (int i = 0; i < this._yearCount; i++)       
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
                for (int j = 0; j < this._yearCount; j++)   
                {
                    report.Buffer[row                      , col     ] =                                                                        data.UpstreamEmissions[i, j];
                    report.Buffer[row                      , cols - 1] = Global.GetDouble(report.Buffer[row                      , cols - 1]) + data.UpstreamEmissions[i, j];
                    report.Buffer[row + this._yearCount - i, col     ] = Global.GetDouble(report.Buffer[row + this._yearCount - i, col     ]) + data.UpstreamEmissions[i, j];
                    report.Buffer[row + this._yearCount - i, cols - 1] = Global.GetDouble(report.Buffer[row + this._yearCount - i, cols - 1]) + data.UpstreamEmissions[i, j];
                    col++;
                }
                row++;
                col = 0;
            }
            report.Buffer[row++, col] = "Total";
            row++;
            report.Buffer[row, col++] = "Model\nYear";
            report.Buffer[row++, col] = "Total Emissions -- Calendar Year";
            for (int i = 0; i < this._yearCount; i++)
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
            }
            report.Buffer[row++, col] = "Total";
            col = 0;
            for (int i = 0; i < this._yearCount; i++)       
            {
                report.Buffer[row, col++] = (this._eisMinYear + i);
                for (int j = 0; j < this._yearCount; j++)   
                {
                    report.Buffer[row                      , col     ] =                                                                        data.TotalEmissions[i, j];
                    report.Buffer[row                      , cols - 1] = Global.GetDouble(report.Buffer[row                      , cols - 1]) + data.TotalEmissions[i, j];
                    report.Buffer[row + this._yearCount - i, col     ] = Global.GetDouble(report.Buffer[row + this._yearCount - i, col     ]) + data.TotalEmissions[i, j];
                    report.Buffer[row + this._yearCount - i, cols - 1] = Global.GetDouble(report.Buffer[row + this._yearCount - i, cols - 1]) + data.TotalEmissions[i, j];
                    col++;
                }
                row++;
                col = 0;
            }
            report.Buffer[row++, col] = "Total";
        }
        Output OpenReport(string path, string name, TemplateType type)
        {
            Output report = new Output(path + "\\" + name + ".xls", type, new OutputSettings(null), true);
            report.FillColor    = XlColor.Automatic;
            report.Landscape    = true;
            report.Margins      = new XlMargins(0.5);
            report.NumberFormat = "0";
            return report;
        }
        void FinalizeReport(Output report, Input template, bool isEmissions)
        {
            report.WriteBuffers(template, true);
            XlUtilities xlu = report.XlUtils;
            int sheetCount = xlu.WorksheetCount;
            if (template == null)
            {
                int sectCount  = (isEmissions) ? 3 : 2;
                for (int i = 1; i <= sheetCount; i++)
                {
                    XlCell firstCell = new XlCell(1, 1);
                    XlCell lastCell  = xlu.GetLastCell(i);
                    int rows = lastCell.Row;
                    int cols = lastCell.Column;
                    xlu.SetFillColor(i, XlCell.Empty, XlCell.Empty, XlColor.White);
                    xlu.SetFont   (i, firstCell, new XlCell(rows, 1), XlFonts.Arial_10PtBold);
                    xlu.SetFont   (i, firstCell, firstCell          , XlFonts.Arial_12PtBold);
                    xlu.MergeCells(i, firstCell, new XlCell(1, cols));
                    for (int j = 0; j < sectCount; j++)
                    {
                        xlu.MergeCells(i, new XlCell(3 + j * (4 + this._yearCount), 1), new XlCell(4 + j * (4 + this._yearCount),    1));
                        xlu.MergeCells(i, new XlCell(3 + j * (4 + this._yearCount), 2), new XlCell(3 + j * (4 + this._yearCount), cols));
                        xlu.SetFont   (i, new XlCell(3 + j * (4 + this._yearCount), 2), new XlCell(4 + j * (4 + this._yearCount), cols), XlFonts.Arial_10PtBold);
                        xlu.SetBorders(i, new XlCell(4 +                   j * (4 + this._yearCount), 1), new XlCell(4 +                   j * (4 + this._yearCount), cols    ), false, true , false, false, true, true, XlColor.DarkBlue);
                        xlu.SetBorders(i, new XlCell(4 + this._yearCount + j * (4 + this._yearCount), 1), new XlCell(4 + this._yearCount + j * (4 + this._yearCount), cols    ), false, true , false, false, true, true, XlColor.DarkBlue);
                        xlu.SetBorders(i, new XlCell(3 +                   j * (4 + this._yearCount), 2), new XlCell(5 + this._yearCount + j * (4 + this._yearCount), cols - 1), false, false, true , true , true, true, XlColor.DarkBlue);
                    }
                }
            }
            for (int i = 1; i <= sheetCount; i++)
            {
                XlCell firstCell = new XlCell(1, 1);
                XlCell lastCell  = xlu.GetLastCell(i);
                xlu.AutoFitText(i, firstCell, lastCell);
            }
            report.Close(true);
        }
        void FinalizeCAFEReport(Output report, Input template, int sectCount)
        {
            report.WriteBuffers(template, true);
            XlUtilities xlu = report.XlUtils;
            for (int i = 1; i <= 2; i++)
            {
                XlCell firstCell = new XlCell(1, 1);
                XlCell lastCell  = xlu.GetLastCell(i);
                int rows = lastCell.Row;
                int cols = lastCell.Column;
                xlu.SetFillColor(i, XlCell.Empty, XlCell.Empty, XlColor.White);
                xlu.SetFont   (i, firstCell       , new XlCell(rows, 1), XlFonts.Arial_10PtBold);
                xlu.SetFont   (i, new XlCell(3, 1), new XlCell(4, cols), XlFonts.Arial_10PtBold);
                xlu.AlignText (i, new XlCell(3, 2), new XlCell(4, cols), new XlTextAlignment(XlHAlign.Center));
                xlu.SetFont   (i, firstCell       , firstCell          , XlFonts.Arial_12PtBold);
                xlu.MergeCells(i, firstCell, new XlCell(1, cols));
                xlu.MergeCells(i, new XlCell(3, 1), new XlCell(4, 1));
                xlu.SetBorders(i, new XlCell(4, 1), new XlCell(4 , cols), false, true , false, false, true, true, XlColor.DarkBlue);
                for (int j = 0; j < sectCount; j++)
                {
                    xlu.MergeCells     (i, new XlCell(3, 2 + 3 * j), new XlCell(3   , 4 + 3 * j));
                    xlu.SetBorders     (i, new XlCell(3, 2 + 3 * j), new XlCell(rows, 2 + 3 * j), false, false, true, false, true, true, XlColor.DarkBlue);
                    xlu.SetNumberFormat(i, new XlCell(5, 2 + 3 * j), new XlCell(rows, 3 + 3 * j), XlNumberFormats.NumericDecimal1);
                    xlu.SetNumberFormat(i, new XlCell(5, 4 + 3 * j), new XlCell(rows, 4 + 3 * j), XlNumberFormats.PercentDecimal1);
                }
                xlu.AutoFitText(i, firstCell, lastCell);
            }
            report.Close(true);
        }
        #endregion
        #region 
        const double EmFract = 1000D / 0.907185;
        EisParameters        _params;
        EisTailpipeEmissions _tailpipe;
        EisParameters.FuelEconomy[] _carFEFlatCAFE;
        EisParameters.FuelEconomy[] _truckFEFlatCAFE;
        EisParameters.FuelEconomy[] _carFEGrowCAFE;
        EisParameters.FuelEconomy[] _truckFEGrowCAFE;
        Estimates                   _fuelPriceEstimates;
        ParametersOverrides         _paramsOverrides;
        int _eisMinYear;
        int _eisMaxYear;
        int _yearCount;
        EisData[] _scenDataFlatCAFE;
        EisData   _noCafeDataFlatCAFE;
        EisData[] _scenDataGrowCAFE;
        EisData   _noCafeDataGrowCAFE;
        #endregion
    }
}

