#region << Using Directives >>
using System;
using Volpe.Cafe.IO;
#endregion
namespace Volpe.Cafe.Model.EIS
{
    [Serializable]
    class EisParameters : ICloneable
    {
        #region 
        [Serializable]
        public struct Parameters
        {
            public int    BaseUsageYear;
            public double AutoGrowthRate;
            public double LtGrowthRate;
            public double ReboundEffect;
            public double TestVsOnRoadGap;
            public double Elasticity;
        }
        [Serializable]
        public struct UpstreamEmissions
        {
            public double CO2;
            public double CH4;
            public double N2O;
            public double VOC;
            public double CO;
            public double NOx;
            public double PM;
            public double SOx;
            public double Formaldehyde;
            public double Acetaldehyde;
            public double Butadiene;
            public double Benzene;
            public double Acrolein;
            public double MTBE;
            public double DieselPM;
        }
        [Serializable]
        public struct FuelEconomy : ICloneable
        {
            #region 
            public FuelEconomy(int length)
            {
                this.GasolineNoCAFE    = new double[length];
                this.DieselNoCAFE      = new double[length];
                this.DieselShareNoCAFE = new double[length];
                this.GasolineCAFE      = new double[length];
                this.DieselCAFE        = new double[length];
                this.DieselShareCAFE   = new double[length];
            }
            #endregion
            #region 
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public FuelEconomy Clone()
            {
                int         length = this.GasolineNoCAFE.Length;
                FuelEconomy fe     = new FuelEconomy(length);
                Array.Copy(this.GasolineNoCAFE   , 0, fe.GasolineNoCAFE   , 0, length);
                Array.Copy(this.DieselNoCAFE     , 0, fe.DieselNoCAFE     , 0, length);
                Array.Copy(this.DieselShareNoCAFE, 0, fe.DieselShareNoCAFE, 0, length);
                Array.Copy(this.GasolineCAFE     , 0, fe.GasolineCAFE     , 0, length);
                Array.Copy(this.DieselCAFE       , 0, fe.DieselCAFE       , 0, length);
                Array.Copy(this.DieselShareCAFE  , 0, fe.DieselShareCAFE  , 0, length);
                return fe;
            }
            #endregion
            #endregion
            public double[] GasolineNoCAFE;         
            public double[] DieselNoCAFE;           
            public double[] DieselShareNoCAFE;      
            public double[] GasolineCAFE;           
            public double[] DieselCAFE;             
            public double[] DieselShareCAFE;        
        }
        #endregion
        #region 
        private EisParameters()
        {
        }
        public EisParameters(string path)
            : this(path, string.Empty)
        {
        }
        public EisParameters(string path, string password)
            : this()
        {
            Input input = new Input(path, password);
            try
            {   
                this.ParseFile(input);
            }
            catch (Exception ex)
            {   
                throw new InputException("Errors were encountered while parsing file: " + ex.Message, path, ex);
            }
            finally
            {   
                input.Close();
            }
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public EisParameters Clone()
        {
            EisParameters eis = new EisParameters();
            eis.MinYear              = this.MinYear;
            eis.MaxYear              = this.MaxYear;
            eis.BaseParameters       = this.BaseParameters;
            eis.GasolineUpstream     = this.GasolineUpstream;
            eis.DieselUpstream       = this.DieselUpstream;
            eis.GasolineTailpipeCO2  = this.GasolineTailpipeCO2;
            eis.DieselTailpipeCO2    = this.DieselTailpipeCO2;
            eis.GasolineTailpipeSO2  = this.GasolineTailpipeSO2;
            eis.DieselTailpipeSO2    = this.DieselTailpipeSO2;
            eis.LowGasolinePrice     = (double[] )Global.CloneArray(this.LowGasolinePrice    , typeof(double));
            eis.RefGasolinePrice     = (double[] )Global.CloneArray(this.RefGasolinePrice    , typeof(double));
            eis.HighGasolinePrice    = (double[] )Global.CloneArray(this.HighGasolinePrice   , typeof(double));
            eis.LowDieselPrice       = (double[] )Global.CloneArray(this.LowDieselPrice      , typeof(double));
            eis.RefDieselPrice       = (double[] )Global.CloneArray(this.RefDieselPrice      , typeof(double));
            eis.HighDieselPrice      = (double[] )Global.CloneArray(this.HighDieselPrice     , typeof(double));
            eis.LowCarMilesDriven    = (double[] )Global.CloneArray(this.LowCarMilesDriven   , typeof(double));
            eis.RefCarMilesDriven    = (double[] )Global.CloneArray(this.RefCarMilesDriven   , typeof(double));
            eis.HighCarMilesDriven   = (double[] )Global.CloneArray(this.HighCarMilesDriven  , typeof(double));
            eis.LowTruckMilesDriven  = (double[] )Global.CloneArray(this.LowTruckMilesDriven , typeof(double));
            eis.RefTruckMilesDriven  = (double[] )Global.CloneArray(this.RefTruckMilesDriven , typeof(double));
            eis.HighTruckMilesDriven = (double[] )Global.CloneArray(this.HighTruckMilesDriven, typeof(double));
            eis.CarFleet             = (double[,])Global.CloneArray(this.CarFleet            , typeof(double));
            eis.TruckFleet           = (double[,])Global.CloneArray(this.TruckFleet          , typeof(double));
            eis.CarFuelEconomy       = this.CarFuelEconomy  .Clone();
            eis.TruckFuelEconomy     = this.TruckFuelEconomy.Clone();
            return eis;
        }
        #endregion
        void ParseFile(Input input)
        {
            string[] sheets  = input.SheetNames;
            string[] names   = {"Parameters", "Emission Rates", "Fuel Prices", "Forecast Data", "Car Fleet", "Truck Fleet",
                                   "Fuel Economy"};
            int   [] indexes = new int[names.Length];
            for (int i = 0; i < names.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < sheets.Length; i++)
            {
                for (int j = 0; j < names.Length; j++)
                {
                    if (input.Compare(sheets[i], names[j], false)) { indexes[j] = i; break; }
                }
            }
            input.VerifyIndexes(indexes, names);
            this.MinYear = 1975;
            this.MaxYear = 2060;
            int yearCount = this.MaxYear - this.MinYear + 1;
            int ageCount  = 61;     
            this.BaseParameters       = new Parameters();
            this.GasolineUpstream     = new UpstreamEmissions();
            this.DieselUpstream       = new UpstreamEmissions();
            this.LowGasolinePrice     = new double[yearCount];
            this.RefGasolinePrice     = new double[yearCount];
            this.HighGasolinePrice    = new double[yearCount];
            this.LowDieselPrice       = new double[yearCount];
            this.RefDieselPrice       = new double[yearCount];
            this.HighDieselPrice      = new double[yearCount];
            this.LowCarMilesDriven    = new double[ageCount ];
            this.RefCarMilesDriven    = new double[ageCount ];
            this.HighCarMilesDriven   = new double[ageCount ];
            this.LowTruckMilesDriven  = new double[ageCount ];
            this.RefTruckMilesDriven  = new double[ageCount ];
            this.HighTruckMilesDriven = new double[ageCount ];
            this.CarFleet             = new double[yearCount, yearCount];
            this.TruckFleet           = new double[yearCount, yearCount];
            this.CarFuelEconomy       = new FuelEconomy(yearCount);
            this.TruckFuelEconomy     = new FuelEconomy(yearCount);
            input.ActivateWorksheet(names[0]);
            this.BaseParameters.BaseUsageYear      = input.GetInt32 ( 3, 1);
            this.BaseParameters.AutoGrowthRate     = input.GetDouble( 4, 1);
            this.BaseParameters.LtGrowthRate       = input.GetDouble( 5, 1);
            this.BaseParameters.ReboundEffect      = input.GetDouble( 6, 1);
            this.BaseParameters.TestVsOnRoadGap    = input.GetDouble( 7, 1);
            this.BaseParameters.Elasticity         = input.GetDouble( 8, 1);
            input.ActivateWorksheet(names[1]);
            this.GasolineUpstream.CO2          = input.GetDouble(4,  1);
            this.GasolineUpstream.CH4          = input.GetDouble(4,  2);
            this.GasolineUpstream.N2O          = input.GetDouble(4,  3);
            this.GasolineUpstream.VOC          = input.GetDouble(4,  4);
            this.GasolineUpstream.CO           = input.GetDouble(4,  5);
            this.GasolineUpstream.NOx          = input.GetDouble(4,  6);
            this.GasolineUpstream.PM           = input.GetDouble(4,  7);
            this.GasolineUpstream.SOx          = input.GetDouble(4,  8);
            this.GasolineUpstream.Formaldehyde = input.GetDouble(4,  9);
            this.GasolineUpstream.Acetaldehyde = input.GetDouble(4, 10);
            this.GasolineUpstream.Butadiene    = input.GetDouble(4, 11);
            this.GasolineUpstream.Benzene      = input.GetDouble(4, 12);
            this.GasolineUpstream.Acrolein     = input.GetDouble(4, 13);
            this.GasolineUpstream.MTBE         = input.GetDouble(4, 14);
            this.GasolineUpstream.DieselPM     = input.GetDouble(4, 15);
            this.DieselUpstream.CO2            = input.GetDouble(5,  1);
            this.DieselUpstream.CH4            = input.GetDouble(5,  2);
            this.DieselUpstream.N2O            = input.GetDouble(5,  3);
            this.DieselUpstream.VOC            = input.GetDouble(5,  4);
            this.DieselUpstream.CO             = input.GetDouble(5,  5);
            this.DieselUpstream.NOx            = input.GetDouble(5,  6);
            this.DieselUpstream.PM             = input.GetDouble(5,  7);
            this.DieselUpstream.SOx            = input.GetDouble(5,  8);
            this.DieselUpstream.Formaldehyde   = input.GetDouble(5,  9);
            this.DieselUpstream.Acetaldehyde   = input.GetDouble(5, 10);
            this.DieselUpstream.Butadiene      = input.GetDouble(5, 11);
            this.DieselUpstream.Benzene        = input.GetDouble(5, 12);
            this.DieselUpstream.Acrolein       = input.GetDouble(5, 13);
            this.DieselUpstream.MTBE           = input.GetDouble(5, 14);
            this.DieselUpstream.DieselPM       = input.GetDouble(5, 15);
            this.GasolineTailpipeCO2 = input.GetDouble(10, 1);
            this.DieselTailpipeCO2   = input.GetDouble(11, 1);
            this.GasolineTailpipeSO2 = input.GetDouble(10, 2);
            this.DieselTailpipeSO2   = input.GetDouble(11, 2);
            input.ActivateWorksheet(names[2]);
            for (int i = 0; i < yearCount; i++)
            {
                this.LowGasolinePrice [i] = input.GetDouble(5 + i, 1);
                this.RefGasolinePrice [i] = input.GetDouble(5 + i, 2);
                this.HighGasolinePrice[i] = input.GetDouble(5 + i, 3);
                this.LowDieselPrice   [i] = input.GetDouble(5 + i, 4);
                this.RefDieselPrice   [i] = input.GetDouble(5 + i, 5);
                this.HighDieselPrice  [i] = input.GetDouble(5 + i, 6);
            }
            input.ActivateWorksheet(names[3]);
            for (int i = 0; i < ageCount; i++)
            {
                this.LowCarMilesDriven   [i] = input.GetDouble(5 + i,  5);
                this.RefCarMilesDriven   [i] = input.GetDouble(5 + i,  6);
                this.HighCarMilesDriven  [i] = input.GetDouble(5 + i,  7);
                this.LowTruckMilesDriven [i] = input.GetDouble(5 + i,  8);
                this.RefTruckMilesDriven [i] = input.GetDouble(5 + i,  9);
                this.HighTruckMilesDriven[i] = input.GetDouble(5 + i, 10);
            }
            input.ActivateWorksheet(names[4]);
            for (int i = 0; i < yearCount; i++)
            {
                for (int j = 0; j < yearCount; j++)
                {
                    this.CarFleet[i, j] = input.GetDouble(5 + i, 1 + j);
                }
            }
            input.ActivateWorksheet(names[5]);
            for (int i = 0; i < yearCount; i++)
            {
                for (int j = 0; j < yearCount; j++)
                {
                    this.TruckFleet[i, j] = input.GetDouble(5 + i, 1 + j);
                }
            }
            input.ActivateWorksheet(names[6]);
            for (int i = 0; i < yearCount; i++)
            {
                this.CarFuelEconomy.GasolineNoCAFE     [i] = input.GetDouble(5 + i,  1);
                this.CarFuelEconomy.DieselNoCAFE       [i] = input.GetDouble(5 + i,  2);
                this.CarFuelEconomy.DieselShareNoCAFE  [i] = input.GetDouble(5 + i,  3);
                this.CarFuelEconomy.GasolineCAFE       [i] = input.GetDouble(5 + i,  4);
                this.CarFuelEconomy.DieselCAFE         [i] = input.GetDouble(5 + i,  5);
                this.CarFuelEconomy.DieselShareCAFE    [i] = input.GetDouble(5 + i,  6);
                this.TruckFuelEconomy.GasolineNoCAFE   [i] = input.GetDouble(5 + i,  7);
                this.TruckFuelEconomy.DieselNoCAFE     [i] = input.GetDouble(5 + i,  8);
                this.TruckFuelEconomy.DieselShareNoCAFE[i] = input.GetDouble(5 + i,  9);
                this.TruckFuelEconomy.GasolineCAFE     [i] = input.GetDouble(5 + i, 10);
                this.TruckFuelEconomy.DieselCAFE       [i] = input.GetDouble(5 + i, 11);
                this.TruckFuelEconomy.DieselShareCAFE  [i] = input.GetDouble(5 + i, 12);
            }
        }
        #endregion
        #region 
        public int MinYear;     
        public int MaxYear;     
        public EisParameters.Parameters        BaseParameters;
        public EisParameters.UpstreamEmissions GasolineUpstream;
        public EisParameters.UpstreamEmissions DieselUpstream;
        public double                          GasolineTailpipeCO2;
        public double                          DieselTailpipeCO2;
        public double                          GasolineTailpipeSO2;
        public double                          DieselTailpipeSO2;
        public double[]                        LowGasolinePrice;        
        public double[]                        RefGasolinePrice;        
        public double[]                        HighGasolinePrice;       
        public double[]                        LowDieselPrice;          
        public double[]                        RefDieselPrice;          
        public double[]                        HighDieselPrice;         
        public double[]                        LowCarMilesDriven;       
        public double[]                        RefCarMilesDriven;       
        public double[]                        HighCarMilesDriven;      
        public double[]                        LowTruckMilesDriven;     
        public double[]                        RefTruckMilesDriven;     
        public double[]                        HighTruckMilesDriven;    
        public double[,]                       CarFleet;                
        public double[,]                       TruckFleet;              
        public EisParameters.FuelEconomy       CarFuelEconomy;          
        public EisParameters.FuelEconomy       TruckFuelEconomy;        
        #endregion
    }
}

