#region << Using Directives >>
using System;
using System.Collections;
using System.IO;
using System.Reflection;
using System.Threading;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.Model.EIS;
using Volpe.Cafe.Settings;
using Volpe.Ui;
using TI = Volpe.Cafe.TechnologyIndexes;
#endregion
namespace Volpe.Cafe.Model
{
    [Serializable]
    [ModelDescription("Standard Compliance Model", "Runs a regular compliance model that estimates technology costs and " +
         "benefits under scenarios with pre-specified flat or reformed CAFE standards.", 1.0F)]
    public class Compliance : ComplianceBase
    {
        #region 
        public Compliance()
            : base()
        {
        }
        ~Compliance()
        {
        }
        #endregion
        #region 
        #region 
        public override ICompliance CreateNew()
        {
            return new Compliance();
        }
        protected override void StartInternal()
        {
            if (this.PreProcess())
            {   
                this.ScenarioLoop();
                this.PostProcess();
            }
        }
        public override Industry GetData(int scenIndex, int yearIndex)
        {
            if (this._modelData != null && scenIndex < this._scenCount && yearIndex - this._minYearIndex < this._yearCount)
            {
                return this._modelData[scenIndex][yearIndex];
            }
            return null;
        }
        protected void SetData(int scenIndex, int yearIndex, Industry data)
        {
            this._modelData[scenIndex][yearIndex] = data;
        }
        #endregion
        #region 
        protected virtual bool PreProcess()
        {
            if (this.State == ModelingState.Completed) { return false; }
            this._baselineInfo = new BaselineInfo[this._maxYearIndex + 1];
            this._marketSimulation = new MarketSimulation();
            this._finders = new LinearComplianceFinder[this._scenCount];
            for (int i = 0; i < this._mfrCount; i++)
            {
                Manufacturer            mfr      = this._data.Manufacturers[i];
                ManufacturerDescription mfrDescr = mfr.Description;
                if (mfrDescr.DiscountRate < 0)
                {   
                    mfrDescr.DiscountRate = this._settings.Parameters.EconomicValues.DiscountRate;
                }
                mfr.ModelingData.KMY = KMYType.GetKmyValues(this._settings.Parameters,
                    this._settings.OperatingModes.FuelPriceEstimates, mfr);
            }
            this._modelData = new Industry[this._scenCount][];
            for (int i = 0; i < this._scenCount; i++)
            {
                this._modelData[i] = new Industry[this._maxYearIndex + 1];
            }
            this._scen = null;
            this._year = null;
            this._mfr  = null;
            return true;
        }
        protected virtual void PostProcess()
        {
            EISSettings eisSet = this._settings.EISSettings;
            if (eisSet.RunEISModelFlatCAFE || eisSet.RunEISModelGrowCAFE)
            {
                try
                {
                    this._eisModelRunning = true;
                    this._eisModel = new EisModel();
                    this._eisModel.Analyze(eisSet, this, this._scenCount, this._minYear, this._maxYear);
                }
                catch (Exception ex)
                {
                    this.OnPrompt(new PromptEventArgs("Error In EIS Model", ex.ToString(), PromptOption.Ok));
                }
                this._eisModelRunning = false;
            }
        }
        #endregion
        #region 
        #region 
        protected virtual void ScenarioLoop()
        {
            for (int i = 0; i < this._scenCount; i++)
            {   
                Scenario scen = this._settings.Scenarios[i];
                Industry data = this._data.Clone();
                this.RunScenario(scen, data);
                if (this._abortRequested) { break; }
            } 
        }
        protected virtual void RunScenario(Scenario scen, Industry data)
        {
            this.PreProcessScenario (scen, data);
            this.ModelYearLoop      (scen, data);
            this.PostProcessScenario(scen, data);
        }
        protected virtual void PreProcessScenario(Scenario scen, Industry data)
        {
            this._scen = scen;
            this._finders[scen.Index] = new LinearComplianceFinder(scen, this._minYear, this._settings, this._logWriter);
            data.ModelingData.TechUsedSales = new RCDouble[TI.TechnologyCount];
            for (int i = 0; i < this._mfrCount; i++)
            {
                Manufacturer             mfr = data.Manufacturers[i];
                ManufacturerModelingData mmd = mfr.ModelingData;
                mmd.TechUsedSales    = new RCDouble [TI.TechnologyCount];
                mmd.TechAppliedSales = new RCDouble [TI.TechnologyCount];
                mmd.TechExhausted    = new RCBoolean[TI.TechnologyCount];
                VehicleCollection      vc  = mfr.Vehicles;
                EngineCollection       ec  = mfr.Engines;
                TransmissionCollection tc  = mfr.Transmissions;
                NamePlateCollection    npc = mfr.NamePlates;
                for (int j = 0, vcCount = vc.Count; j < vcCount; j++)
                {   
                    Vehicle            veh      = vc[j];
                    VehicleDescription vehDescr = veh.Description;
                    TechnologyApplicability.Initialize(veh, this._settings);
                }
                for (int j = 0, ecCount = ec.Count; j < ecCount; j++)
                {   
                    Engine eng = ec[j];
                    eng.ModelingData.TechUsed = new bool[TI.TechnologyCount];
                }
                for (int j = 0, tcCount = tc.Count; j < tcCount; j++)
                {   
                    Transmission trn = tc[j];
                    trn.ModelingData.TechUsed = new bool[TI.TechnologyCount];
                }
                for (int j = 0, npcCount = npc.Count; j < npcCount; j++)
                {   
                    NamePlate np = npc[j];
                    np.ModelingData.TechUsed = new bool[TI.TechnologyCount];
                }
            }
            if (!scen.IsBaseline)
            {   
                data.ModelingData.BaselineInfo = this._baselineInfo;
            }
        }
        protected virtual void PostProcessScenario(Scenario scen, Industry data)
        {
            this._finders[scen.Index].Clear();
        }
        #endregion
        #region 
        protected virtual void ModelYearLoop(Scenario scen, Industry data)
        {
            this.SetData(scen.Index, this._minYearIndex, data);
            for (int i = this._minYear; i <= this._maxYear; i++)
            {   
                ModelYear year = new ModelYear(i);
                this.RunModelYear(scen, year);
                if (this._abortRequested) { break; }
            } 
        }
        protected virtual void RunModelYear(Scenario scen, ModelYear year)
        {
            this.PreProcessModelYear (scen, year);
            this.ManufacturerLoop    (scen, year);
            this.PostProcessModelYear(scen, year);
        }
        protected virtual void PreProcessModelYear(Scenario scen, ModelYear year)
        {
            this._year = year;
            int yrIndex = year.Index;
            if (yrIndex != this._minYearIndex)
            {   
                Industry prevData = this.GetData(scen.Index, yrIndex - 1);
                this.SetData(scen.Index, yrIndex, prevData.Clone());
            }
            Industry data = this.GetData(scen.Index, yrIndex);
            IndustryModelingData imd = data.ModelingData;
            for (int i = 0; i < TI.TechnologyCount; i++) { imd.TechUsedSales[i].Clear(); }
            TechnologyCollection technologies = this._settings.Technologies;
            for (int i = 0; i < this._mfrCount; i++)
            {   
                Manufacturer             mfr = data.Manufacturers[i];
                ManufacturerModelingData mmd = mfr .ModelingData;
                VehicleCollection mfrVehs = mfr.InitializeModelYearVehicles(year);
                mmd.PreliminaryStandard.Clear();
                mmd.Standard           .Clear();
                mmd.TechCost           .Clear();
                mmd.RegCost            .Clear();
                for (int j = 0; j < TI.TechnologyCount; j++)
                {   
                    mmd.TechUsedSales   [j].Clear();
                    mmd.TechAppliedSales[j].Clear();
                    mmd.TechExhausted   [j].Clear();
                }
                for (int j = 0, mfrVehCount = mfrVehs.Count; j < mfrVehCount; j++)
                {   
                    Vehicle             veh = mfrVehs[j];
                    VehicleDescription  vd  = veh.Description;
                    VehicleModelingData vmd = veh.ModelingData;
                    RegulatoryClass regClass = Standards.GetRegClass(veh, scen, year);
                    veh.RegClass = regClass;
                    TechnologyApplicability.Enable(veh, year, -1, this._settings);
                    PredecessorsAndLegacies.ExamineVehicle(veh, scen, year, data, this._settings, this._minYear, this._logWriter);
                    for (int k = 0; k < TI.TechnologyCount; k++)
                    {
                        if (vmd.TechUsed[k])
                        {   
                            imd.TechUsedSales[k][regClass] += vd.Sales[yrIndex];
                            mmd.TechUsedSales[k][regClass] += vd.Sales[yrIndex];
                            if (vmd.TechApplied[k])
                            {   
                                mmd.TechAppliedSales[k][regClass] += vd.Sales[yrIndex];
                            }
                        }
                    } 
                } 
            } 
            RCDouble indStandard = RCDouble.Zero;
            if (scen.AutoMinStndPct[yrIndex] > 0 || scen.LtMinStndPct[yrIndex] > 0)
            {
                indStandard = Standards.GetIndustryStandard(data, scen, year, false);
            }
            for (int i = 0; i < this._mfrCount; i++)
            {   
                Manufacturer             mfr    = data.Manufacturers[i];
                ManufacturerModelingData mmd    = mfr .ModelingData;
                VehicleCollection       mfrVehs = mfr.ModelYearVehicles;
                for (int j = 0, mfrVehCount = mfrVehs.Count; j < mfrVehCount; j++)
                {   
                    Vehicle             veh       = mfrVehs[j];
                    VehicleDescription  vd        = veh.Description;
                    VehicleModelingData vmd       = veh.ModelingData;
                    double              vehCost   = 0;
                    TechnologyClass     techClass = veh.TechnologyClass;
                    for (int k = 0; k < TI.TechnologyCount; k++)
                    {
                        Technology tech = technologies[k];
                        if (vmd.TechUsed[k] && vmd.TechApplied[k])
                        {   
                            vehCost += TechnologyApplicability.GetCost(tech, veh, data,
                                this._settings.OperatingModes.TechnologyCostEstimates, year, this._minYear, true,
                                this._settings.OperatingModes.ClipTechCosts);;
                        }
                    } 
                    vmd.TechCost = vehCost * vd.Sales[yrIndex];
                    mmd.TechCost[veh.RegClass] += vmd.TechCost;
                } 
                mmd.AverageStandard = indStandard;                      
                mmd.Sales           = Standards.GetSales(mfr, year);    
                for (int j = 0; j < TI.TechnologyCount; j++)
                {
                    TechnologyApplicability.CheckPhaseIn(mmd, RegulatoryClass.DomesticAuto, technologies[j], year);
                    TechnologyApplicability.CheckPhaseIn(mmd, RegulatoryClass.ImportedAuto, technologies[j], year);
                    TechnologyApplicability.CheckPhaseIn(mmd, RegulatoryClass.LightTruck  , technologies[j], year);
                    TechnologyApplicability.CheckPhaseIn(mmd, RegulatoryClass.Unregulated , technologies[j], year);
                } 
            } 
        }
        protected virtual void PostProcessModelYear(Scenario scen, ModelYear year)
        {
            if (this._abortRequested) { return; }
            int      techCount = TI.TechnologyCount;
            int      yrYear    = year.Year;
            int      yrIndex   = year.Index;
            Industry data      = this.GetData(scen.Index, yrIndex);
            this._marketSimulation.AllocateCosts(data, this._settings, year);
            Effects.Process(scen, year, data, this._settings);
            if (scen.IsBaseline)
            {
                int biIdx = yrYear - ModelYear.MinYear;
                this._baselineInfo[biIdx].ModelYear = year;
                this._baselineInfo[biIdx].FinalFuelEconomy      = new double  [this._mfrCount][];
                this._baselineInfo[biIdx].MfrSales              = new RCDouble[this._mfrCount];
                this._baselineInfo[biIdx].MfrTechCosts          = new RCDouble[this._mfrCount];
                this._baselineInfo[biIdx].MfrFines              = new RCDouble[this._mfrCount];
                this._baselineInfo[biIdx].MfrRegCosts           = new RCDouble[this._mfrCount];
                this._baselineInfo[biIdx].MfrFuelConsumption    = new RCDouble[this._mfrCount];
                this._baselineInfo[biIdx].MfrDiscSocialBenefits = new RCDouble[this._mfrCount];
                for (int i = 0; i < this._mfrCount; i++)
                {
                    Manufacturer             mfr = data.Manufacturers[i];
                    ManufacturerModelingData mmd = mfr.ModelingData;
                    VehicleCollection vehs = mfr.ModelYearVehicles;
                    int vehCount = vehs.Count;
                    this._baselineInfo[biIdx].FinalFuelEconomy[i] = new double[vehCount];
                    this._baselineInfo[biIdx].MfrSales    [i] = mmd.Sales;
                    this._baselineInfo[biIdx].MfrTechCosts[i] = mmd.TechCost;
                    this._baselineInfo[biIdx].MfrFines    [i] = mmd.Fines;
                    this._baselineInfo[biIdx].MfrRegCosts [i] = mmd.RegCost;
                    this._baselineInfo[biIdx].MfrFuelConsumption   [i] = mmd.FuelConsumption;
                    this._baselineInfo[biIdx].MfrDiscSocialBenefits[i] = mmd.DiscSocialBenefits;
                    for (int j = 0; j < vehCount; j++)
                    {
                        this._baselineInfo[biIdx].FinalFuelEconomy[i][j] = vehs[j].Description.FuelEconomy;
                    }
                }
            }
            if (this._settings.OperatingModes.MultiYearModeling)
            {   
                if (yrYear > this._minYear)
                {   
                    this.PostProcessModelYear(scen, this._modelYears[yrYear - this._minYear - 1]);
                }
            }
        }
        #endregion
        #region 
        protected virtual void ManufacturerLoop(Scenario scen, ModelYear year)
        {
            for (int i = 0; i < this._mfrCount; i++)
            {   
                this.RunManufacturer(scen, year, i);
                if (this._abortRequested) { break; }
            } 
        }
        protected virtual void RunManufacturer(Scenario scen, ModelYear year, int mfrIndex)
        {
            this.PreProcessManufacturer(scen, year, mfrIndex);
            int                      yrIndex      = year.Index;
            Industry                 data         = this.GetData(scen.Index, year.Index);
            Manufacturer             mfr          = data.Manufacturers[mfrIndex];
            ManufacturerModelingData mmd          = mfr.ModelingData;
            ManufacturerDescription  mfrDescr     = mfr.Description;
            ModelingApproach         approach     = this._settings.OperatingModes.ModelingApproach;
            bool                     reserveTechs = this._settings.OperatingModes.ReserveTechnologiesIfInCompliance;
            bool                     noFines      = this._settings.OperatingModes.NoFines || !mfrDescr.WillingToPayFines[year.Index];
            double                   mfrTechCost  = mmd.TechCost.TotalAll;
            double                   mfrFines     = mmd.Fines   .TotalAll;
            bool                     ignoredReset = false;
            ComplianceFinding finding    = null;
            ComplianceFinding reservedCf = null;
            mmd.CFExhausted = false;
            while (mfrFines > 0)
            {   
                finding = this.ComplianceFindingLoop(scen, year, mfrIndex, noFines, ref approach);
                if (finding != null && finding.IsValid)
                {   
                    if (reserveTechs && finding.InCompliance)
                    {   
                        reservedCf = new ComplianceFinding();
                        reservedCf.CopyFrom(finding);
                        finding.ReserveTechnologies();
                        while (finding != null)
                        {   
                            finding  = this.ComplianceFindingLoop(scen, year, mfrIndex, noFines, ref approach);
                            if (finding != null && finding.IsValid)
                            {   
                                if (finding.InCompliance && finding.TechCost < reservedCf.TechCost)
                                {   
                                    reservedCf.CopyFrom(finding);
                                }
                                finding.ReserveTechnologies();
                            }
                        } 
                        finding = reservedCf;
                    } 
                    finding.ApplyFinding();
                    this.Recomply(scen, year, mfr);
                } 
                else
                {
                    if (finding == null)
                    {   
                        if (ignoredReset)
                        {   
                            mmd.CFExhausted = true;
                            break;
                        }
                        else
                        {   
                            VehicleCollection mfrVehs = mfr.ModelYearVehicles;
                            for (int i = 0, vehCount = mfrVehs.Count; i < vehCount; i++)
                            {
                                VehicleModelingData vmd = mfrVehs[i].ModelingData;
                                for (int j = 0; j < TI.TechnologyCount; j++) { vmd.TechIgnored[j] = false; }
                            }
                            ignoredReset = true;    
                        }
                    }
                    else    
                    {   
                        throw new Exception("The compliance finding is not valid!");
                    }
                }
                mfrTechCost = mmd.TechCost.TotalAll;
                mfrFines    = mmd.Fines   .TotalAll;
                if (this._abortRequested) { break; }
            } 
            this.PostProcessManufacturer(scen, year, mfrIndex);
        }
        protected virtual void PreProcessManufacturer(Scenario scen, ModelYear year, int mfrIndex)
        {
            Manufacturer mfr = this.GetData(scen.Index, year.Index).Manufacturers[mfrIndex];
            this._mfr = mfr;
            this.Recomply(scen, year, mfr);
        }
        protected virtual void PostProcessManufacturer(Scenario scen, ModelYear year, int mfrIndex)
        {
            if (this._abortRequested) { return; }
            int          yrIndex = year.Index;
            Industry     data    = this.GetData(scen.Index, yrIndex);
            Manufacturer mfr     = data.Manufacturers[mfrIndex];
            this.Recomply(scen, year, mfr);
            if (this._settings.OperatingModes.MultiYearModeling &&
                this._settings.OperatingModes.MultiYearStartYear <= year.Year)
            {   
                for (int i = 0, mfrVehCount = mfr.ModelYearVehicles.Count; i < mfrVehCount; i++)
                {   
                    Vehicle             veh = mfr.ModelYearVehicles[i];
                    VehicleDescription  vd  = veh.Description;
                    VehicleModelingData vmd = veh.ModelingData;
                    bool atRedesign, inShadow;
                    veh.GetRedesignState(year, out atRedesign, out inShadow);
                    if (atRedesign)
                    {
                        for (int j = 0; j < TI.TechnologyCount; j++)
                        {
                            if (TI.IsTiedToRedesign(j))
                            {
                                RegulatoryClass regClass = veh.RegClass;
                                bool techValid = (vmd.TechEnabled[j] && !vmd.TechUsed[j] &&
                                    ((!vmd.TechReserved[j] && !vmd.TechIgnored[j] &&
                                    (this._settings.OperatingModes.IgnorePhaseIn || !mfr.ModelingData.TechExhausted[j][regClass]))));
                                if (techValid)
                                {
                                    vmd.TechLingerYear = year;
                                    vmd.TechLingering[j] = true;
                                }
                            } 
                        } 
                    } 
                }
                for (int i = this._minYearIndex; i < yrIndex; i++)
                {
                    Industry prevData = this.GetData(scen.Index, i);
                    Manufacturer lMfr = prevData.Manufacturers[mfrIndex];
                    this.Recomply(scen, this._modelYears[i - this._minYearIndex], lMfr);
                }
            } 
        }
        protected virtual void Recomply(Scenario scen, ModelYear year, Manufacturer mfr)
        {
            ManufacturerModelingData mmd = mfr.ModelingData;
            mmd.Standard = Standards.GetStandard(mfr, scen, year);
            mmd.CAFE     = Standards.GetCAFE    (mfr,       year, out mmd.Sales, out mmd.SalesOverFE);
            mmd.Credits  = Standards.GetCredits (mfr, scen, year, this._settings.Parameters);
            mmd.Fines    = Standards.GetFines   (mfr,             this._settings.Parameters);
        }
        #endregion
        #region 
        protected virtual ComplianceFinding ComplianceFindingLoop(Scenario scen, ModelYear year, int mfrIndex, bool noFines,
            ref ModelingApproach approach)
        {
            int                      snIndex = scen.Index;
            int                      yrIndex = year.Index;
            Industry                 data    = this.GetData(snIndex, yrIndex);
            Manufacturer             mfr     = data.Manufacturers[mfrIndex];
            EngineCollection         mfrEngs = mfr.Engines;
            TransmissionCollection   mfrTrns = mfr.Transmissions;
            NamePlateCollection      mfrNpts = mfr.NamePlates;
            VehicleCollection        mfrVehs = mfr.ModelYearVehicles;
            ManufacturerModelingData mmd     = mfr.ModelingData;
            ComplianceFinding finding = null;
            bool lowCostFirst    = (approach == ModelingApproach.LowCostFirst);
            bool allowSecondBest = !lowCostFirst;   
            bool loop            = true;
            while (loop)
            {   
                finding = this._finders[snIndex].GetNextFinding(year, this._modelData[snIndex], mfrIndex, lowCostFirst,
                    allowSecondBest);
                if (lowCostFirst && (finding == null || !finding.IsEfficient))
                {   
                    if (this._settings.OperatingModes.AutoToggleApproach)
                    {   
                        approach        = ModelingApproach.EconomicEfficiency;
                        lowCostFirst    = false;
                        allowSecondBest = true;
                        continue;
                    }
                    else if (!allowSecondBest)
                    {   
                        allowSecondBest = true;
                        continue;
                    }
                } 
                double mfrTechCost = mmd.TechCost.TotalAll;
                double mfrFines    = mmd.Fines   .TotalAll;
                if (finding == null || !finding.IsValid || noFines || finding.IsEfficient)
                {   
                    loop = false;
                }
                else
                {   
                    finding = null;
                    loop    = false;
                }
            } 
            return finding;
        }
        #endregion
        #endregion
        #endregion
        #region 
        #region 
        public override IEISModel EISModel { get { return this._eisModel; } }
        public override IModelingProgress Progress
        {
            get { return (this.Running) ? new ComplianceProgress(this._scen, this._year, this._mfr,
                      (this._eisModelRunning) ? "Executing the EIS Model ..." : null) : null; }
        }
        #endregion
        #endregion
        #region 
        protected Industry[][] _modelData;
        protected BaselineInfo[] _baselineInfo;
        [NonSerialized] protected MarketSimulation _marketSimulation;
        [NonSerialized] EisModel _eisModel;
        [NonSerialized] bool _eisModelRunning;
        [NonSerialized] internal LinearComplianceFinder[] _finders;
        [NonSerialized] protected Scenario _scen;
        [NonSerialized] protected ModelYear _year;
        [NonSerialized] protected Manufacturer _mfr;
        #endregion
    }
}

