#region << Using Directives >>
using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;
using TI = Volpe.Cafe.TechnologyIndexes;
#endregion
namespace Volpe.Cafe.Model
{
    public sealed class TechnologyApplicability
    {
        #region 
        static TechnologyApplicability()
        {
            TechSupersedes       = new int[TI.TechnologyCount][];
            TechDisallowBackfill = new int[TI.TechnologyCount][];
            TechKeys = new ulong[63];
            for (int i = 0; i < 63; i++) { TechKeys[i] = (ulong)Math.Pow(2, i); }
            TechTypes = new TechnologyType[] {TechnologyType.Engine, TechnologyType.Transmission, TechnologyType.Elec,
                TechnologyType.Hybrid, TechnologyType.MSR, TechnologyType.DLR, TechnologyType.Aerodynamics, TechnologyType.Other};
            TechTypeCount = TechTypes.Length;
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                TechSupersedes      [i] = new int[0];
                TechDisallowBackfill[i] = new int[0];
                switch (i)
                {
                    case TI.DCP:
                        TechSupersedes[i] = new int[] {TI.ICP};
                        break;
                    case TI.DVVLD:
                        TechDisallowBackfill[i] = new int[] {TI.CVVL};
                        break;
                    case TI.CVVL:
                        TechDisallowBackfill[i] = new int[] {TI.DVVLD};
                        break;
                    case TI.DVVLO:
                        TechDisallowBackfill[i] = new int[] {TI.CDOHC};
                        break;
                    case TI.CDOHC:
                        TechSupersedes[i] = new int[] {TI.CCPO};
                        TechDisallowBackfill[i] = new int[] {TI.DVVLO};
                        break;
                    case TI.TRBDS:
                        TechSupersedes[i] = new int[] {TI.DEACS, TI.DEACD, TI.DEACO};
                        TechDisallowBackfill[i] = new int[] {TI.DSLC};
                        break;
                    case TI.EGRB:
                        TechSupersedes[i] = new int[] {TI.DEACS, TI.DEACD, TI.DEACO};
                        TechDisallowBackfill[i] = new int[] {TI.DSLC};
                        break;
                    case TI.DSLT:
                        TechSupersedes[i] = new int[] {TI.LUB, TI.EFR, TI.CCPS, TI.DVVLS, TI.DEACS, TI.ICP, TI.DCP, TI.DVVLD, 
                            TI.CVVL, TI.DEACD, TI.DEACO, TI.CCPO, TI.DVVLO, TI.CDOHC, TI.SGDI, TI.CBRST, TI.TRBDS, TI.EGRB};
                        TechDisallowBackfill[i] = new int[] {TI.DSLC};
                        break;
                    case TI.DSLC:
                        TechSupersedes[i] = new int[] {TI.LUB, TI.EFR, TI.CCPS, TI.DVVLS, TI.DEACS, TI.ICP, TI.DCP, TI.DVVLD, 
                            TI.CVVL, TI.DEACD, TI.DEACO, TI.CCPO, TI.DVVLO, TI.CDOHC, TI.SGDI, TI.CBRST};
                        TechDisallowBackfill[i] = new int[] {TI.TRBDS, TI.EGRB, TI.DSLT};
                        break;
                    case TI.CVT:
                        TechSupersedes[i] = new int[] {TI.IATC};
                        TechDisallowBackfill[i] = new int[] {TI.NAUTO, TI.DCTAM};
                        break;
                    case TI.NAUTO:
                        TechSupersedes[i] = new int[] {TI.IATC};
                        TechDisallowBackfill[i] = new int[] {TI.CVT};
                        break;
                    case TI.DCTAM:
                        TechSupersedes[i] = new int[] {TI.IATC, TI.NAUTO};
                        TechDisallowBackfill[i] = new int[] {TI.CVT};
                        break;
                    case TI.MHEV:
                        TechSupersedes[i] = new int[] {TI.CBRST};
                        break;
                    case TI.BISG:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.MHEV};
                        break;
                    case TI.CISG:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.MHEV, TI.BISG};
                        break;
                    case TI.PSHEV:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.IATC, TI.CVT, TI.NAUTO, TI.DCTAM, TI.MHEV, TI.BISG, TI.CISG};
                        TechDisallowBackfill[i] = new int[] {TI.MHEV2, TI.PHEV};
                        break;
                    case TI.MHEV2:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.IATC, TI.CVT, TI.NAUTO, TI.DCTAM, TI.MHEV, TI.BISG, TI.CISG};
                        TechDisallowBackfill[i] = new int[] {TI.PSHEV, TI.PHEV};
                        break;
                    case TI.PHEV:
                        TechSupersedes[i] = new int[] {TI.CBRST, TI.IATC, TI.CVT, TI.NAUTO, TI.DCTAM, TI.MHEV, TI.BISG, TI.CISG};
                        TechDisallowBackfill[i] = new int[] {TI.PSHEV, TI.MHEV2};
                        break;
                } 
            } 
        }
        #endregion
        #region 
        #region 
        public static void Initialize(Vehicle vehicle, ModelingSettings settings)
        {
            VehicleDescription      vd = vehicle.Description;
            EngineDescription       ed = vehicle.Engine.Description;
            TransmissionDescription td = vehicle.Transmission.Description;
            VehicleModelingData     vmd = vehicle.ModelingData;
            TechnologyClass      techClass    = vehicle.TechnologyClass;
            TechnologyCollection technologies = settings.Technologies;
            vmd.TechAvailable         = new bool[TI.TechnologyCount];
            vmd.TechUsed              = new bool[TI.TechnologyCount];
            vmd.TechApplied           = new bool[TI.TechnologyCount];
            vmd.TechSuperseded        = new bool[TI.TechnologyCount];
            vmd.TechBackfilled        = new bool[TI.TechnologyCount];
            vmd.TechPhaseInBackfilled = new bool[TI.TechnologyCount];
            vmd.TechEnabled           = new bool[TI.TechnologyCount];
            vmd.TechIgnored           = new bool[TI.TechnologyCount];
            vmd.TechReserved          = new bool[TI.TechnologyCount];
            vmd.TechLingering         = new bool[TI.TechnologyCount];
            vmd.TechLingerYear        = null;
            for (int i = 0; i < vd.UsedTechnologies.Length; i++) { vmd.TechUsed[vd.UsedTechnologies[i]] = true; }
            for (int i = 0; i < ed.UsedTechnologies.Length; i++) { vmd.TechUsed[ed.UsedTechnologies[i]] = true; }
            for (int i = 0; i < td.UsedTechnologies.Length; i++) { vmd.TechUsed[td.UsedTechnologies[i]] = true; }
            for (int i = 0; i < vd.AvailableTechnologies.Length; i++)
            {
                int index = vd.AvailableTechnologies[i];
                vmd.TechAvailable[index] = technologies[index].Attributes[techClass].Applicable;
            }
            for (int i = 0; i < ed.AvailableTechnologies.Length; i++)
            {
                int index = ed.AvailableTechnologies[i];
                vmd.TechAvailable[index] = technologies[index].Attributes[techClass].Applicable;
            }
            for (int i = 0; i < td.AvailableTechnologies.Length; i++)
            {
                int index = td.AvailableTechnologies[i];
                vmd.TechAvailable[index] = technologies[index].Attributes[techClass].Applicable;
            }
            if (td.Type == "M")
            {
                TechnologyApplicability.RectifyDependencies_DisableDeacEng(vmd.TechAvailable);
            }
            if (!settings.OperatingModes.IgnoreWeightInMsApplication)
            {
                if (vd.CurbWeight < 5000)
                {   
                    vmd.TechAvailable[TI.MS1] = false;
                    vmd.TechAvailable[TI.MS2] = false;
                    vmd.TechAvailable[TI.MS5] = false;
                }
            }
            TechnologyApplicability.RectifyDependencies(vehicle, null, -1, settings);
        }
        public static void Enable(Vehicle vehicle, ModelYear modelYear, int techIndex, ModelingSettings settings)
        {
            VehicleDescription   vehDescr     = vehicle.Description;
            VehicleModelingData  vmd          = vehicle.ModelingData;
            TechnologyClass      techClass    = vehicle.TechnologyClass;
            TechnologyCollection technologies = settings.Technologies;
            int                  year         = modelYear.Year;
            if (techIndex == -1)
            {   
                for (int i = 0; i < TI.TechnologyCount; i++)
                {
                    vmd.TechEnabled [i] = vmd.TechAvailable[i] &&
                        (settings.OperatingModes.IgnoreYearAvailable || technologies[i].Attributes[techClass].Year <= year);
                    vmd.TechIgnored [i] = false;
                    vmd.TechReserved[i] = false;
                }
            }
            else
            {   
                vmd.TechEnabled[techIndex] = vmd.TechAvailable[techIndex] &&
                    (technologies[techIndex].Attributes[techClass].Year <= year);
            }
            if (!settings.OperatingModes.IgnoreRefreshRedesign)
            {   
                bool atRedesign, inShadowOfRedesign;
                bool atRefresh , inShadowOfRefresh ;
                vehicle.GetRedesignState(modelYear, out atRedesign, out inShadowOfRedesign);
                vehicle.GetRefreshState (modelYear, out atRefresh , out inShadowOfRefresh );
                int sIdx = (techIndex == -1) ? 0                  : techIndex;
                int eIdx = (techIndex == -1) ? TI.TechnologyCount : techIndex + 1;
                for (int i = sIdx; i < eIdx; i++)
                {   
                    if (vmd.TechEnabled[i])
                    {
                        if (inShadowOfRedesign)
                        {   
                            if (TI.IsTiedToRedesign(i))
                            {
                                vmd.TechEnabled[i] = false;
                            }
                            else if (!atRefresh)
                            {   
                                if (TI.IsTiedToRefresh(i)) { vmd.TechEnabled[i] = false; }
                            }
                        }
                        else if (!atRedesign && inShadowOfRefresh)
                        {   
                            if (TI.IsTiedToRefresh(i)) { vmd.TechEnabled[i] = false; }
                        }
                    } 
                } 
            }
            TechnologyApplicability.RectifyDependencies(vehicle, modelYear, techIndex, settings);
        }
        static void RectifyDependencies(Vehicle vehicle, ModelYear year, int techIdx, ModelingSettings settings)
        {
            VehicleModelingData vmd = vehicle.ModelingData;     
            bool updateTechEnabled = (year != null);
            bool updateSuperseded  = (techIdx != -1);
            bool[] techAE   = (updateTechEnabled) ? vmd.TechEnabled : vmd.TechAvailable;
            bool[] techLing = vmd.TechLingering;
            bool[] techUsed = vmd.TechUsed;
            int sIdx = (techIdx == -1) ? 0                  : techIdx;
            int eIdx = (techIdx == -1) ? TI.TechnologyCount : techIdx + 1;
            bool hevRectified = false;  
            for (int i = sIdx; i < eIdx; i++)
            {   
                if (techUsed[i]) { techAE[i] = false; }
                RectifyDependenciesHelper(i, techUsed, techAE, updateTechEnabled, ref hevRectified, vehicle, year, techIdx,
                    settings);
                if (updateTechEnabled)
                {   
                    RectifyDependenciesHelper(i, techUsed, techLing, updateTechEnabled, ref hevRectified, vehicle, year, techIdx,
                        settings);
                }
                if (techUsed[i] && vmd.TechApplied[i] && updateSuperseded)
                {   
                    int[] techSupersedes = TechSupersedes[i];
                    for (int j = 0, techSupCount = techSupersedes.Length; j < techSupCount; j++)
                    {   
                        vmd.TechSuperseded[techSupersedes[j]] = vmd.TechUsed[techSupersedes[j]];
                    }
                }
            } 
        }
        static void RectifyDependenciesHelper(int i, bool[] techUsed, bool[] techArr, bool updateTechEnabled,
            ref bool hevRectified, Vehicle vehicle, ModelYear year, int techIdx, ModelingSettings settings)
        {
            switch (i)
            {
                case TI.LUB:
                case TI.EFR:
                    break;
                case TI.CCPS:
                case TI.DVVLS:
                case TI.DEACS:
                    break;
                case TI.ICP:
                case TI.DEACD:
                    break;
                case TI.DCP:
                    if (techUsed[i]) { techArr[TI.ICP] = false; }
                    break;
                case TI.DVVLD:
                    if (techUsed[i]) { techArr[TI.CVVL] = false; }
                    break;
                case TI.CVVL:
                    if (techUsed[i]) { techArr[TI.DVVLD] = false; }
                    break;
                case TI.DEACO:
                case TI.CCPO:
                    break;
                case TI.DVVLO:
                    if (techUsed[i]) { techArr[TI.CDOHC] = false; }
                    break;
                case TI.CDOHC:
                    if (techUsed[i]) { techArr[TI.DVVLO] = false; }
                    break;
                case TI.SGDI:
                case TI.CBRST:
                    break;
                case TI.TRBDS:
                case TI.EGRB:
                    if (techUsed[i])
                    {   
                        TechnologyApplicability.RectifyDependencies_DisableDeacEng(techArr);
                        techArr[TI.DSLC] = false;
                    }
                    break;
                case TI.DSLT:
                case TI.DSLC:
                    if (techUsed[i]) { TechnologyApplicability.RectifyDependencies_DslUsed(techArr); }
                    break;
                case TI.MAN6:
                    if (techUsed[i])
                    {   
                        TechnologyApplicability.RectifyDependencies_DisableAutoTrn(techArr);
                        TechnologyApplicability.RectifyDependencies_DisableAllHev (techArr);
                        TechnologyApplicability.RectifyDependencies_DisableDeacEng(techArr);
                    }
                    break;
                case TI.IATC:
                case TI.CVT:
                case TI.NAUTO:
                case TI.DCTAM:
                    if (techUsed[i])
                    {   
                        TechnologyApplicability.RectifyDependencies_DisableManualTrn(techArr);
                        techArr[TI.IATC] = false;
                        if (i == TI.CVT)
                        {   
                            techArr[TI.NAUTO] = false;
                            techArr[TI.DCTAM] = false;
                            techArr[TI.MHEV2] = false;
                        }
                        else if (i == TI.NAUTO || i == TI.DCTAM)
                        {   
                            techArr[TI.CVT  ] = false;
                            techArr[TI.NAUTO] = false;
                        }
                    }
                    if (updateTechEnabled && !hevRectified)
                    {
                        RectifyDependencies_EnableHev(techArr, techUsed, vehicle, year, techIdx, settings);
                        hevRectified = true;    
                    }
                    break;
                case TI.EPS:
                case TI.IACC:
                    break;
                case TI.MHEV:
                case TI.BISG:
                case TI.CISG:
                    if (techUsed[i])
                    {   
                        techArr[TI.CBRST] = false;
                        if (i == TI.BISG) { RectifyDependencies_DisableAllElec(techArr, false); }
                        if (i == TI.CISG) { RectifyDependencies_DisableAllElec(techArr, true); }
                    }
                    break;
                case TI.PSHEV:
                case TI.MHEV2:
                case TI.PHEV:
                    if (techUsed[i]) { TechnologyApplicability.RectifyDependencies_HevUsed(techArr, settings.OperatingModes); }
                    break;
                case TI.MS1:
                case TI.MS2:
                case TI.MS5:
                    break;
                case TI.ROLL:
                case TI.LDB:
                case TI.SAXU:
                case TI.SAXL:
                    break;
                case TI.AERO:
                    break;
            } 
        }
        static void RectifyDependencies_DslUsed(bool[] techAE)
        {
            TechnologyApplicability.RectifyDependencies_DisableAllEng(techAE);
        }
        static void RectifyDependencies_HevUsed(bool[] techAE, OperatingModes opModes)
        {
            if (opModes.ShutoffEngineTechsAfterHEV)       { TechnologyApplicability.RectifyDependencies_DisableAllEng (techAE); }
            else                                          { techAE[TI.CBRST] = false; }     
            if (opModes.ShutoffTransmissionTechsAfterHEV) { TechnologyApplicability.RectifyDependencies_DisableAllTrn (techAE); }
            if (opModes.ShutoffElecTechsAfterHEV)         { TechnologyApplicability.RectifyDependencies_DisableAllElec(techAE, true); }
            TechnologyApplicability.RectifyDependencies_DisableAllHev(techAE);
        }
        static void RectifyDependencies_DisableAllEng(bool[] techAE)
        {
            TechnologyApplicability.RectifyDependencies_DisableSohcEng(techAE);
            TechnologyApplicability.RectifyDependencies_DisableDohcEng(techAE);
            TechnologyApplicability.RectifyDependencies_DisableOhvEng (techAE);
            techAE[TI.LUB  ] = false;
            techAE[TI.EFR  ] = false;
            techAE[TI.SGDI ] = false;
            techAE[TI.CBRST] = false;
            techAE[TI.TRBDS] = false;
            techAE[TI.EGRB ] = false;
            techAE[TI.DSLC ] = false;
            techAE[TI.DSLT ] = false;
        }
        static void RectifyDependencies_DisableSohcEng(bool[] techAE)
        {
            techAE[TI.CCPS ] = false;
            techAE[TI.DVVLS] = false;
            techAE[TI.DEACS] = false;
        }
        static void RectifyDependencies_DisableDohcEng(bool[] techAE)
        {
            techAE[TI.ICP  ] = false;
            techAE[TI.DCP  ] = false;
            techAE[TI.DVVLD] = false;
            techAE[TI.CVVL ] = false;
            techAE[TI.DEACD] = false;
        }
        static void RectifyDependencies_DisableOhvEng(bool[] techAE)
        {
            techAE[TI.DEACO] = false;
            techAE[TI.CCPO ] = false;
            techAE[TI.DVVLO] = false;
            techAE[TI.CDOHC] = false;
        }
        static void RectifyDependencies_DisableDeacEng(bool[] techAE)
        {
            techAE[TI.DEACS] = false;
            techAE[TI.DEACD] = false;
            techAE[TI.DEACO] = false;
        }
        static void RectifyDependencies_DisableAllTrn(bool[] techAE)
        {
            TechnologyApplicability.RectifyDependencies_DisableManualTrn(techAE);
            TechnologyApplicability.RectifyDependencies_DisableAutoTrn  (techAE);
        }
        static void RectifyDependencies_DisableManualTrn(bool[] techAE)
        {
            techAE[TI.MAN6] = false;
        }
        static void RectifyDependencies_DisableAutoTrn(bool[] techAE)
        {
            techAE[TI.IATC ] = false;
            techAE[TI.CVT  ] = false;
            techAE[TI.NAUTO] = false;
            techAE[TI.DCTAM] = false;
        }
        static void RectifyDependencies_DisableAllElec(bool[] techAE, bool disableCISG)
        {
            techAE[TI.EPS ] = false;
            techAE[TI.IACC] = false;
            techAE[TI.MHEV] = false;
            techAE[TI.BISG] = false;
            if (disableCISG) { techAE[TI.CISG] = false; }
        }
        static void RectifyDependencies_DisableAllHev(bool[] techAE)
        {
            techAE[TI.PSHEV] = false;
            techAE[TI.MHEV2] = false;
            techAE[TI.PHEV ] = false;
        }
        static void RectifyDependencies_EnableHev(bool[] techEnabled, bool[] techUsed, Vehicle vehicle, ModelYear year,
            int techIdx, ModelingSettings settings)
        {
            bool[] techAvailable = vehicle.ModelingData.TechAvailable;
            bool hevSatisfied =
                (!techAvailable[TI.IATC ] || techUsed[TI.IATC ]) &&
                (!techAvailable[TI.CVT  ] || techUsed[TI.CVT  ]) &&
                (!techAvailable[TI.NAUTO] || techUsed[TI.NAUTO]) &&
                (!techAvailable[TI.DCTAM] || techUsed[TI.DCTAM]);
            if (!hevSatisfied)
            {   
                TechnologyApplicability.RectifyDependencies_DisableAllHev(techEnabled);
            }
            else if (techIdx != -1)
            {   
                TechnologyApplicability.Enable(vehicle, year, TI.PSHEV, settings);
                if (!techUsed[TI.CVT])
                {   
                    TechnologyApplicability.Enable(vehicle, year, TI.MHEV2, settings);
                }
                TechnologyApplicability.Enable(vehicle, year, TI.PHEV, settings);
            }
        }
        #endregion
        #region 
        #region 
        static double GetSynergy(int techIndex, bool[] techUsedOrApplied, bool[] techSuperseded, double[] techSynergies,
            SynergyType[] synergyType, TechnologyCollection techGroup, ulong techKey)
        {
            double synergy = 0.0;
            for (int i = 0; i < TI.TechnologyCount; i++)
            {
                if (techUsedOrApplied[i] &&
                    (synergyType[i] != SynergyType.Physical || !techSuperseded[i])) { synergy += techSynergies[i]; }
            }
            if (techGroup != null && techKey > 0)
            {
                for (int i = 0, techCount = techGroup.Count; i < techCount; i++)
                {   
                    if ((techKey & TechKeys[i]) != 0)
                    {   
                        if (techGroup[i].Index >= techIndex) { break; }
                        synergy += techSynergies[techGroup[i].Index];
                    }
                } 
            }
            return synergy;
        }
        static double GetSynergy(int techIndex, bool[] techApplied, bool[] techSuperseded, double[] techSynergies,
            SynergyType[] synergyType)
        {
            double synergy = 0.0;
            for (int i = 0; i < techIndex; i++)
            {
                if (techApplied[i] &&
                    (synergyType[i] != SynergyType.Physical || !techSuperseded[i])) { synergy += techSynergies[i]; }
            }
            return synergy;
        }
        #endregion
        #region 
        public static double GetImprovement(Technology tech, TechnologyClass techClass, Estimates estimate, bool[] techUsed,
            bool[] techSuperseded, bool clipAtZero)
        {
            return GetImprovement(tech, techClass, estimate, techUsed, techSuperseded, null, 0, clipAtZero);
        }
        public static double GetImprovement(Technology tech, TechnologyClass techClass, Estimates estimate, bool[] techUsed,
            bool[] techSuperseded, TechnologyCollection techGroup, ulong techKey, bool clipAtZero)
        {
            TechnologyAttributes ta = tech.Attributes[techClass];
            double imprv = (estimate == Estimates.Low) ? ta.FcLow :
                (estimate == Estimates.High) ? ta.FcHigh : (ta.FcLow + ta.FcHigh) / 2;
            double synergy = GetSynergy(tech.Index, techUsed, techSuperseded, tech.ImprvSynergies[techClass].Synergies,
                tech.ImprvSynergies[techClass].Type, techGroup, techKey);
            imprv += synergy;
            return (clipAtZero && imprv < 0) ? 0 : imprv;
        }
        #endregion
        #region 
        public static double GetCost(Technology tech, Vehicle veh, Industry data, Estimates estimate, ModelYear year,
            int startYear, bool adjustForTechApplied, bool clipAtZero)
        {
            return GetCost(tech, veh, data, estimate, year, startYear, null, 0, adjustForTechApplied, clipAtZero);
        }
        public static double GetCost(Technology tech, Vehicle veh, Industry data, Estimates estimate, ModelYear year,
            int startYear, TechnologyCollection techGroup, ulong techKey, bool clipAtZero)
        {
            return GetCost(tech, veh, data, estimate, year, startYear, techGroup, techKey, false, clipAtZero);
        }
        static double GetCost(Technology tech, Vehicle veh, Industry data, Estimates estimate, ModelYear year,
            int startYear, TechnologyCollection techGroup, ulong techKey, bool adjustForTechApplied, bool clipAtZero)
        {
			VehicleModelingData vmd = veh.ModelingData;
			double techCost     = GetEstimatedCost(tech, veh, estimate);
			double learningRate = GetLearningRate(tech, veh, data, year, startYear, techCost);
			double synergy      = (adjustForTechApplied) ?
				GetSynergy(tech.Index, vmd.TechApplied, vmd.TechSuperseded, tech.CostSynergies[veh.TechnologyClass].Synergies,
				tech.CostSynergies[veh.TechnologyClass].Type) :
				GetSynergy(tech.Index, vmd.TechApplied, vmd.TechSuperseded, tech.CostSynergies[veh.TechnologyClass].Synergies,
				tech.CostSynergies[veh.TechnologyClass].Type, techGroup, techKey);
			if (techCost < 0 && learningRate != 1.0) 
			{     
				double absCost = Math.Abs(techCost);
				double absDiff = absCost - absCost * learningRate;          
				techCost = techCost - absDiff + synergy;                    
			}
			else 
			{     
				techCost = techCost * learningRate + synergy;
			}
			return (clipAtZero && techCost < 0) ? 0 : techCost;
        }
        static double GetEstimatedCost(Technology tech, Vehicle veh, Estimates estimate)
        {
            TechnologyClass      techClass = veh.TechnologyClass;
            TechnologyAttributes ta        = tech.Attributes[techClass];
            double               techCost  = (estimate == Estimates.Low ) ? ta.CostLow  :
                (estimate == Estimates.High) ? ta.CostHigh : (ta.CostHigh + ta.CostLow) / 2;
            if (TI.IsMaterialSubstitution(tech.Index))
            {   
                techCost *= (veh.Description.BaseWeight * ta.Aux);
            }
            else if (TI.IsCylinderBased(tech.Index))
            {   
                techCost *= veh.Engine.Description.Cylinders;
            }
            else if (TI.IsConfigurationBased(tech.Index))
            {   
                string engConfig = veh.Engine.Description.Configuration.ToUpper();
                int costMult = (engConfig == "R" || engConfig == "I") ? 1 :
                    (engConfig == "F" || engConfig == "H" || engConfig == "V") ? 2 : (engConfig == "W") ? 4 : 0;
                techCost *= costMult;
            }
            return techCost;
        }
        static double GetLearningRate(Technology tech, Vehicle veh, Industry data, ModelYear year,
            int startYear, double techCost)
        {
            TechnologyClass techClass = veh.TechnologyClass;
            TechnologyAttributes   ta = tech.Attributes[techClass];
            if (ta.LearningRate != 0 && ta.LearningStartYear < year.Year)
            {
                if (ta.LearningType == LearningType.Volume && ta.LearningThreshold >= 1)
                {   
                    double          volume      = 0;
                    RCDouble[]      techVolumes = data.ModelingData.TechUsedSales;
                    RegulatoryClass regClass    = veh.RegClass;
                    if (regClass == RegulatoryClass.LightTruck)
                    {   
                        volume = (TI.IsDieselization(tech.Index)) ?
                            techVolumes[TI  .DSLT ][RegulatoryClass.LightTruck] +
                            techVolumes[TI  .DSLC ][RegulatoryClass.LightTruck] :
                            techVolumes[tech.Index][RegulatoryClass.LightTruck];
                    }
                    else if (regClass != RegulatoryClass.Unregulated)
                    {   
                        volume = (TI.IsDieselization(tech.Index)) ?
                            techVolumes[TI  .DSLT ][RegulatoryClass.DomesticAuto] +
                            techVolumes[TI  .DSLC ][RegulatoryClass.DomesticAuto] +
                            techVolumes[TI  .DSLT ][RegulatoryClass.ImportedAuto] +
                            techVolumes[TI  .DSLC ][RegulatoryClass.ImportedAuto] :
                            techVolumes[tech.Index][RegulatoryClass.DomesticAuto] +
                            techVolumes[tech.Index][RegulatoryClass.ImportedAuto];
                    }
                    if (volume >= ta.LearningThreshold)
                    {
                        return GetLearningRate_Volume(volume, ta.LearningThreshold, ta.LearningRate);
                    }
                }
                else if (ta.LearningType == LearningType.Time)
                {   
                    return GetLearningRate_Time(year.Year, startYear, ta.LearningStartYear, ta.LearningRate);
                }
                else if (ta.LearningType == LearningType.Slope)
                {   
                    return GetLearningRate_Slope(year.Year, startYear, ta.LearningStartYear, 
                        ta.LearningThreshold, ta.LearningRate, techCost);
                }
            }
            return 1;
        }
        static double GetLearningRate_Volume(double volume, double threshold, double rate)
        {
            return Math.Pow(1.0 - rate, Math.Min(Math.Floor(volume / threshold), 2));
        }
        static double GetLearningRate_Time(int year, int startYear, int learningStartYear, double rate)
        {
            if (startYear < learningStartYear) { startYear = learningStartYear; }
            return Math.Pow(1.0 - rate, year - startYear);
        }
        static double GetLearningRate_Slope(int year, int startYear, int learningStartYear, double threshold, double rate, double b)
        {
            if (startYear < learningStartYear) {startYear = learningStartYear; }
            double y = Math.Max(rate * (year - startYear) + b, threshold);
            return y/b;        
        }
        #endregion
        #endregion
        public static bool CanBackfill(int techIndex, TechnologyCollection techs, bool[] techUsed)
        {
            for (int i = 0, count = techs.Count; i < count; i++)
            {   
                if (!CanBackfillHelper(techIndex, techs[i].Index)) { return false; }
            }
            for (int i = 0; i < TI.TechnologyCount; i++)
            {   
                if (techUsed[i] && !CanBackfillHelper(techIndex, i)) { return false; }
            }
            return true;
        }
        public static bool CanBackfill(int techIndex, TechnologyCollection techGroup, ulong techKey, bool[] techUsed)
        {
            int trueTechIndex = techGroup[techIndex].Index;
            for (int i = 0, techCount = techGroup.Count; i < techCount; i++)
            {   
                if ((techKey & TechKeys[i]) != 0)
                {   
                    if (!CanBackfillHelper(trueTechIndex, techGroup[i].Index)) { return false; }
                }
            }
            for (int i = 0; i < TI.TechnologyCount; i++)
            {   
                if (techUsed[i] && !CanBackfillHelper(trueTechIndex, i)) { return false; }
            }
            return true;
        }
        static bool CanBackfillHelper(int techIndex, int selTechIndex)
        {
            if (techIndex == selTechIndex) { return false; }
            int[] techDisallowBackfill = TechDisallowBackfill[selTechIndex];
            for (int j = 0, disallowCount = techDisallowBackfill.Length; j < disallowCount; j++)
            {
                if (techIndex == techDisallowBackfill[j])
                {   
                    return false;
                }
            }
            return true;
        }
        public static bool IsTechValid(ManufacturerModelingData mmd, VehicleModelingData vmd, RegulatoryClass regClass,
            int techIdx, bool ignorePhaseIn, bool backfilling, ManufacturerModelingData lMmd, out bool lingering)
        {
            lingering = false;
            bool valid = (vmd.TechEnabled[techIdx] && !vmd.TechUsed[techIdx] &&
                (backfilling || (!vmd.TechReserved[techIdx] && !vmd.TechIgnored[techIdx] &&
                (ignorePhaseIn || !mmd.TechExhausted[techIdx][regClass]))));
            if (valid)
            {
                return true;
            }
            else if (vmd.TechLingering[techIdx]) 
            {   
                lingering = (!vmd.TechUsed[techIdx]) && 
                    (ignorePhaseIn || !lMmd.TechExhausted[techIdx][regClass]);
                return lingering;
            }
            return false;
        }
        public static void InstallTechnologyOnVehicle(Vehicle veh, Technology tech, ModelYear year, ModelingSettings settings,
            bool backfilling, bool isExhausted)
        {
            VehicleModelingData     vmd = veh.ModelingData;
            VehicleDescription      vd  = veh.Description;
            EngineDescription       ed  = veh.Engine.Description;
            TransmissionDescription td  = veh.Transmission.Description;
            vmd.TechApplied[tech.Index] = true;
            vmd.TechUsed   [tech.Index] = true;
            if (backfilling)
            {
                vmd.TechBackfilled[tech.Index] = true;
                if (isExhausted) { vmd.TechPhaseInBackfilled[tech.Index] = true; }
            }
            switch (tech.Index)
            {
                case TI.LUB:
                    ed.EngineOilViscosity = "LUB1";
                    break;
                case TI.CDOHC:
                    ed.ValvetrainDesign = "DOHC";
                    ed.ValvesPerCylinder = 4;
                    break;
                case TI.DVVLS:
                case TI.CVVL:
                    ed.ValveLift = "CV";
                    break;
                case TI.DVVLD:
                case TI.DVVLO:
                    ed.ValveLift = "SVI";
                    break;
                case TI.TRBDS:
                    ed.Aspiration = "T";
                    break;
                case TI.DSLC:
                case TI.DSLT:
                    ed.ValveActuationTiming = "F";
                    ed.ValveLift = "F";
                    ed.Fuel = "D";
                    ed.FuelSystem = "DI";
                    ed.Aspiration = "T";
                    ed.Cycle = 'D';
                    break;
                case TI.MAN6:
                    td.Type            = "M";
                    td.NumForwardGears = 6;
                    break;
                case TI.IATC:
                    td.Type = "A";
                    break;
                case TI.CVT:
                    td.Type = "CVT";
                    break;
                case TI.NAUTO:
                    td.Type            = "A";
                    td.NumForwardGears = 6;
                    break;
                case TI.DCTAM:
                    td.Type = "DCTAM";
                    break;
                case TI.PSHEV:
                case TI.MHEV2:
                case TI.PHEV:
                    vd.HybridType = "E";
                    break;
            }
            TechnologyApplicability.Enable(veh, year, tech.Index, settings);
        }
        public static void CheckPhaseIn(ManufacturerModelingData mmd, RegulatoryClass regClass, Technology tech, ModelYear year)
        {
            if (mmd.TechExhausted[tech.Index][regClass]) { return; }
            int[]  phaseInPairs  = new int[] {-1, -1, -1, -1};
            switch (tech.Index)
            {
                case TI.CCPS: case TI.ICP: case TI.DCP: case TI.CCPO:       
                    phaseInPairs[0] = TI.CCPS; phaseInPairs[1] = TI.ICP; phaseInPairs[2] = TI.DCP; phaseInPairs[3] = TI.CCPO;
                    break;
                case TI.DVVLS: case TI.DVVLD: case TI.CVVL: case TI.DVVLO:  
                    phaseInPairs[0] = TI.DVVLS; phaseInPairs[1] = TI.DVVLD; phaseInPairs[2] = TI.CVVL; phaseInPairs[3] = TI.DVVLO;
                    break;
                case TI.DEACS: case TI.DEACD: case TI.DEACO:                
                    phaseInPairs[0] = TI.DEACS; phaseInPairs[1] = TI.DEACD; phaseInPairs[2] = TI.DEACO;
                    break;
                case TI.DSLC: case TI.DSLT:                                 
                    phaseInPairs[0] = TI.DSLC; phaseInPairs[1] = TI.DSLT;
                    break;
                case TI.PSHEV: case TI.MHEV2: case TI.PHEV:                 
                    phaseInPairs[1] = TI.PSHEV; phaseInPairs[2] = TI.MHEV2; phaseInPairs[3] = TI.PHEV;
                    break;
                case TI.SAXL: case TI.SAXU:                                 
                    phaseInPairs[0] = TI.SAXL; phaseInPairs[1] = TI.SAXU;
                    break;
                default:                                                    
                    phaseInPairs[0] = tech.Index;
                    break;
            }
            double appliedSales = 0D;
            for (int i = 0; i < phaseInPairs.Length; i++)
            {
                if (phaseInPairs[i] != -1) { appliedSales += mmd.TechAppliedSales[phaseInPairs[i]][regClass]; }
            }
            double sales = mmd.Sales[regClass];
            if (sales > 0 && appliedSales >= sales * tech.GetPhaseIn(year.Index))
            {   
                for (int i = 0; i < phaseInPairs.Length; i++)
                {
                    if (phaseInPairs[i] != -1) { mmd.TechExhausted[phaseInPairs[i]][regClass] = true; }
                }
            }
        }
        #endregion
        #region 
        static readonly int[][] TechSupersedes;
        static readonly int[][] TechDisallowBackfill;
        public static readonly ulong[] TechKeys;
        public static TechnologyType[] TechTypes;
        public static int TechTypeCount;
        #endregion
    }
}

