using System;
using System.Drawing;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Data.Optimization;
using Volpe.Cafe.IO;
using Volpe.Cafe.Model;
using Volpe.Cafe.Settings;
using Volpe.Utils;
namespace Volpe.Cafe.Reports.ReportTypes
{
    class OptIndReport : Output
    {
        #region 
        public OptIndReport(string path, OutputSettings outputSettings)
            : base(path, TemplateType.None, outputSettings, true)
        {
            base._autoChart    = true;
            base._autoBorder   = false;
            base._fitToPage    = new Size(-1, -1);
            base._font         = new XlFont("Arial");
            base._landscape    = true;
            base._margins      = new XlMargins(0.5);
            base._numberFormat = "#,##0.0";
            this._initialized = false;
        }
        #endregion
        #region 
        public void ParseData(ModelYear year, ModelingSettings settings, Industry data, IterationsData[] iterData, int optScen,
            bool isOpt)
        {
            if (!this._initialized) { this.Initialize(year, settings, data, iterData, optScen, isOpt); }
            data.UpdateMinMaxFpCw();
            if (data.MinFp < this._minFp) { this._minFp = data.MinFp; }
            if (data.MaxFp > this._maxFp) { this._maxFp = data.MaxFp; }
            if (data.MinCw < this._minCw) { this._minCw = data.MinCw; }
            if (data.MaxCw > this._maxCw) { this._maxCw = data.MaxCw; }
            this.WriteData(year, data, iterData);
        }
        void Initialize(ModelYear year, ModelingSettings settings, Industry data, IterationsData[] iterData, int optScen, bool isOpt)
        {
            this._isOpt          = isOpt;
            this._optSettings    = settings.OptimizationSettings;
            this._scen           = settings.Scenarios[optScen];
            this._minYear   = data.MinYear.Year;
            this._maxYear   = data.MaxYear.Year;
            this._yearCount = this._maxYear - this._minYear + 1;
            int aboveOpt     = this._optSettings.AboveOptimum;
            int belowOpt     = this._optSettings.BelowOptimum;
            this._iterCount  = aboveOpt + belowOpt + 1;
            this._minOptYear = this._optSettings.StartYear;
            if (this._minOptYear < this._minYear) { this._minOptYear = this._minYear; }
            this._iterOffset = 0;
            if (this._iterCount > 250)
            {
                int avgOptIndex = 0, minOptIndex, maxOptIndex;
                for (int i = 0; i < this._yearCount; i++)
                {
                    avgOptIndex += iterData[i].Optimized;
                }
                avgOptIndex /= this._yearCount;
                minOptIndex = avgOptIndex - 125;
                maxOptIndex = avgOptIndex + 125;
                if (minOptIndex < 0              ) { minOptIndex  = 0; }
                if (maxOptIndex > this._iterCount) { minOptIndex += (this._iterCount - maxOptIndex); }
                this._iterOffset = minOptIndex;
                this._iterCount  = 250;
            }
            int mfrCount = data.Manufacturers.Count;
            string[] mfrNames = new string[mfrCount];
            this._mfrCount = 0;
            for (int i = 0; i < mfrCount; i++)
            {
                if (data.Manufacturers[i].Description.Optimize)
                {
                    string mfrName = data.Manufacturers[i].Description.Name;
                    mfrName = (mfrName.Length == 3) ? mfrName : Global.GetTitleCase(mfrName);
                    mfrNames[this._mfrCount++] = mfrName;
                }
            }
            this._mfrNames = new string[this._mfrCount];
            Array.Copy(mfrNames, 0, this._mfrNames, 0, this._mfrCount);
            this._optType = (isOpt) ? this._optSettings.Type : OptimizationType.Default;
            this._ffMode = FunctionalFitMode.None;
            if (this._optType == OptimizationType.Default)
            {
                bool typeDetermined = false;
                for (int i = 0; i < this._mfrCount; i++)
                {   
                    if (data.Manufacturers[i].Description.Optimize)
                    {
                        VehicleCollection vehs = data.Manufacturers[i].Vehicles;
                        for (int j = 0, vehCount = vehs.Count; j < vehCount; j++)
                        {
                            RegulatoryClass regClass = vehs[j].RegClass;
                            if (regClass == RegulatoryClass.LightTruck)
                            {
                                this._optType = OptimizationType.LightTruck;
                                typeDetermined = true;
                            }
                            else if (regClass != RegulatoryClass.Unregulated)
                            {
                                this._optType = OptimizationType.PassengerAuto;
                                typeDetermined = true;
                            }
                            if (typeDetermined) { break; }
                        } 
                        if (typeDetermined) { break; }
                    } 
                } 
            }
            this._fncType = new int[this._yearCount];
            int[] fncType = (this._optType == OptimizationType.PassengerAuto) ? this._scen.AutoFunctionType :
                this._scen.LtFunctionType;
            for (int i = 0; i < this._yearCount; i++)
            {
                this._fncType[i] = fncType[this._minYear - ModelYear.MinYear + i];
            }
            this._minFp      = this._minCw = 99999;
            this._maxFp      = this._maxCw = -1;
            this._minBCoef   = 99999;
            this._maxACoef   =    -1;
            this._optIdx     = new int[this._yearCount];
            this._smoothing  = true;
            this._minStnd    = 99999;
            this._maxStnd    =    -1;
            this._bcMinStnd  = 99999;
            this._bcMaxStnd  =    -1;
            this._mbcMinStnd = 99999;
            this._mbcMaxStnd =    -1;
            this.SetupBuffers();
            this._initialized = true;;
        }
        void SetupBuffers()
        {
            int rows = Sections.Length * (this._yearCount + 3) - 5;
            int cols = Math.Max(this._iterCount + 1, 7);    
            this._cafeSRow    = SectionCAFE       * (this._yearCount + 3);
            this._nbSRow      = SectionNB         * (this._yearCount + 3) - 1;
            this._co2_mmtSRow = this._nbSRow      + (this._yearCount + 3);
            this._co2_dlrSRow = this._co2_mmtSRow + (this._yearCount + 3);
            this._fsSRow      = this._co2_dlrSRow + (this._yearCount + 3);
            this._bcSRow      = this._fsSRow      + (this._yearCount + 3);
            this._mbcSRow     = this._bcSRow      + (this._yearCount + 2);
            for (int i = 0; i < this._mfrCount; i++)
            {
                this.SetupBuffer(this._mfrNames[i], this._mfrNames[i] + " Manufacturer-Level Summary", rows, cols);
            }
            this.SetupBuffer("Industry", "Entire Industry Summary", rows, cols);
        }
        void SetupBuffer(string name, string title, int rows, int cols)
        {
            this.AddBuffer(rows, cols);
            this.Buffer.Header      = new XlCell(0, 1);
            this.Buffer.FreezePanes = new XlCell(0, 1);
            this.Buffer.Name        = name;
            this.Buffer.Title       = title;
            int row = 0;
            for (int i = 0; i < SectionCount; i++)
            {
                if (i == SectionOpt)
                {
                    this.Buffer[row, 1] = "A";
                    this.Buffer[row, 2] = "B";
                    this.Buffer[row, 3] = "C";
                    this.Buffer[row, 4] = "D";
                    this.Buffer[row, 5] = "Stnd";
                    this.Buffer[row, 6] = "Index";
                }
                else if (i == SectionStnd)
                {   
                    for (int j = 0; j < this._iterCount; j++)
                    {
                        this.Buffer[row, j + 1] = j + this._iterOffset;
                    }
                }
                this.Buffer[row++, 0] = Sections[i];
                this.SetupBuffer_WriteYears(ref row, (i >= SectionMainStart && i <= SectionMainEnd));
            }
        }
        void SetupBuffer_WriteYears(ref int row, bool writeTotal)
        {
            for (int i = 0; i < this._yearCount; i++)
            {
                this.Buffer[row++, 0] = (this._minYear + i);
            }
            if (writeTotal) { this.Buffer[row++, 0] = "Total"; }
            row++;
        }
        void WriteData(ModelYear year, Industry data, IterationsData[] iterData)
        {
            for (int i = 0, j = 0, mfrCount = data.ManufacturerCount; i < mfrCount; i++)
            {
                if (data.Manufacturers[i].Description.Optimize)
                {
                    this.WriteData(j++, year, iterData);
                }
            }
            this.WriteData(this._mfrCount, year, iterData);
        }
        void WriteData(int mfrIdx, ModelYear year, IterationsData[] iterData)
        {
            bool isInd    = (mfrIdx == this._mfrCount);
            bool isLt     = (this._optType == OptimizationType.LightTruck);
            int  yrIndex  = year.Index;
            int  yrOffset = year.Year - this._minYear;
            int  row, col;
            IterationsData iters = iterData[yrOffset];
            OutputBuffer buffer = this.Buffers[mfrIdx];
            row = 1 + yrOffset;
            col = 1;
            if (year.Year < this._minOptYear)
            {
                double[,] fncCoefs = (this._optType == OptimizationType.PassengerAuto) ? this._scen.AutoCoefficients :
                    this._scen.LtCoefficients;
                for (int i = 0; i < 4; i++)
                {
                    buffer[row, col++] = this.WriteValue(fncCoefs[i, yrIndex]);
                    if (i == 1 && fncCoefs[i, yrIndex] < this._minBCoef) { this._minBCoef = fncCoefs[i, yrIndex]; }
                    if (i == 0 && fncCoefs[i, yrIndex] > this._maxACoef) { this._maxACoef = fncCoefs[i, yrIndex]; }
                }
                return;
            }
            double[] coefs = iters.GetCoefficients();
            if (coefs != null)
            {
                for (int i = 0; i < coefs.Length; i++)
                {
                    buffer[row, col++] = this.WriteValue(coefs[i]);
                    if (i == 1 && coefs[i] < this._minBCoef) { this._minBCoef = coefs[i]; }
                    if (i == 0 && coefs[i] > this._maxACoef) { this._maxACoef = coefs[i]; }
                }
            }
            buffer[row, col++] = (isInd) ? this.WriteValue(iters.GetStandard()) :
                iters.Iterations[iters.Optimized].AdditionalIterationData[mfrIdx].Standard;
            buffer[row, col++] = this.WriteValue(iters.Optimized);
            if (isInd) { this._optIdx[yrOffset] = iters.Optimized; }
            col = 1;
            for (int i = 1; i <= this._iterCount; i++)
            {
                row = 3 + yrOffset + this._yearCount;
                IterationData id = iters.Iterations[i - 1 + this._iterOffset];
                if (!id.Valid) { continue; }
                double bcr, mbcr, stnd;
                if (isInd)
                {
                    bcr  = id.BCRatio;
                    mbcr = id.MBCRatio;
                    stnd = id.Standard;
                    if (stnd < this._minStnd) { this._minStnd = stnd; }
                    if (stnd > this._maxStnd) { this._maxStnd = stnd; }
                    if (!double.IsNaN(bcr) && !double.IsInfinity(bcr))
                    {
                        if (stnd < this._bcMinStnd) { this._bcMinStnd = stnd; }
                        if (stnd > this._bcMaxStnd) { this._bcMaxStnd = stnd; }
                    }
                    if (i > 1 && (!this._smoothing || i < this._iterCount) && !double.IsNaN(mbcr) && !double.IsInfinity(mbcr))
                    {
                        if (stnd < this._mbcMinStnd) { this._mbcMinStnd = stnd; }
                        if (stnd > this._mbcMaxStnd) { this._mbcMaxStnd = stnd; }
                    }
                }
                else
                {   
                    bcr  = id.AdditionalIterationData[mfrIdx].BCRatio;
                    mbcr = id.AdditionalIterationData[mfrIdx].MBCRatio;
                    stnd = id.AdditionalIterationData[mfrIdx].Standard;
                }
                double cafe                = (isInd) ? id.CAFE                : id.AdditionalIterationData[mfrIdx].CAFE;
                double techCost            = (isInd) ? id.DeltaTechCost       : id.AdditionalIterationData[mfrIdx].DeltaTechCost;
                double fines               = (isInd) ? id.Fines               : id.AdditionalIterationData[mfrIdx].Fines;
                double benefits            = (isInd) ? id.DeltaBenefits       : id.AdditionalIterationData[mfrIdx].DeltaBenefits;
                double discBenefits        = (isInd) ? id.DeltaDiscBenefits   : id.AdditionalIterationData[mfrIdx].DeltaDiscBenefits;
                double netBenefits         = (isInd) ? id.NetBenefits         : id.AdditionalIterationData[mfrIdx].NetBenefits;
                double co2EmissionSavings  = (isInd) ? id.CO2EmissionSavings  : id.AdditionalIterationData[mfrIdx].CO2EmissionSavings;
                double co2MonetizedSavings = (isInd) ? id.CO2MonetizedSavings : id.AdditionalIterationData[mfrIdx].CO2MonetizedSavings;
                double fuelSavings         = (isInd) ? id.FuelSavings         : id.AdditionalIterationData[mfrIdx].FuelSavings;
                buffer[row, col] = this.WriteValue(stnd); row += 2 + this._yearCount;
                buffer[row, col] = this.WriteValue(cafe); row += 2 + this._yearCount;
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, techCost);
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, fines);
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, benefits);
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, discBenefits);
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, netBenefits);
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, co2EmissionSavings);
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, co2MonetizedSavings);
                this.WriteSummedValue(buffer, ref row, col, row + this._yearCount - yrOffset, fuelSavings);
                buffer[row, col] = this.WriteValue(bcr ); row += 2 + this._yearCount;
                buffer[row, col] = this.WriteValue(mbcr);
                col++;
            }
        }
        void WriteSummedValue(OutputBuffer buffer, ref int row, int col, int sumRow, double value)
        {
            buffer[row   , col] = this.WriteValue(value);
            buffer[sumRow, col] = this.AddValue  (value, buffer[sumRow, col]);
            row += 3 + this._yearCount;
        }
        object WriteValue(double value)
        {
            return (double.IsInfinity(value) || double.IsNaN(value)) ? (object)"=NA()" : (object)value;
        }
        object AddValue(double value, object addTo)
        {
            return (!(addTo == null || addTo.ToString() == "" || Global.IsNumeric(addTo)) || double.IsInfinity(value) ||
                double.IsNaN(value)) ? (object)"=NA()" : (object)(Global.GetDouble(addTo) + value);
        }
        void SetupCharts()
        {
            this.SetupCharts_NBChartsByMY();
            this.SetupCharts_IndustryLevel(this.Buffers[this._mfrCount], this._maxStnd, this._minStnd, this._bcMaxStnd,
                this._bcMinStnd, this._mbcMaxStnd, this._mbcMinStnd, 4 + this._yearCount, this._cafeSRow, this._nbSRow,
                this._co2_mmtSRow, this._co2_dlrSRow, this._fsSRow, this._bcSRow, this._mbcSRow, this._minYear, this._iterCount,
                this._yearCount, this._smoothing);
            this.SetupCharts_DrawShape();
        }
        void SetupCharts_NBChartsByMY()
        {
            OutputBuffer buffer = Buffers[this._mfrCount];
            for (int i = 0; i < this._yearCount; i++)
            {    
                int    year    = this._minYear + i;
                string yearStr = year.ToString();
                string nbChartName = buffer.Name + "-NBGraph " + year;
                double maxStnd = Math.Floor  (this._maxStnd) + 1;
                double minStnd = Math.Ceiling(this._minStnd) - 1;
                XlChartFormat nbChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, nbChartName,
                    nbChartName, XlColor.White, new XlSeriesFormat[0], null, null);
                XlTickLabelsFormat ticksFormat = new XlTickLabelsFormat(XlTickLabelsFormat.DefaultFont, "#,##0.0");
                XlGridlinesFormat gridlinesFormat = new XlGridlinesFormat(true, true, false, XlColor.Gray25);
                nbChartFormat.ValuesAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                    double.NaN, double.NaN, double.NaN, false, false, gridlinesFormat, ticksFormat);
                nbChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                    double.NaN, maxStnd, minStnd, false, false, gridlinesFormat, ticksFormat);
                XlRange xValues = new XlRange(new XlCell(4 + this._yearCount + i, 2),
                    new XlCell(4 + this._yearCount + i, this._iterCount + 1));
                for (int j = 0; j < this._mfrCount; j++)
                {
                    int vR1 = this._nbSRow + i, vC1 = 2;
                    int vR2 = this._nbSRow + i, vC2 = this._iterCount + 1;
                    int xR1 = xValues.Cell1.Row, xC1 = xValues.Cell1.Column;
                    int xR2 = xValues.Cell2.Row, xC2 = xValues.Cell2.Column;
                    string mfrValues  = "='" + this._mfrNames[j] + "'!R" + vR1 + "C" + vC1 + ":R" + vR2 + "C" + vC2;
                    string mfrXValues = "='" + buffer.Name       + "'!R" + xR1 + "C" + xC1 + ":R" + xR2 + "C" + xC2;
                    nbChartFormat.AddSeries(new XlSeriesFormat(this._mfrNames[j], mfrValues, mfrXValues));
                }
                nbChartFormat.AddSeries(new XlSeriesFormat(yearStr, new XlRange(new XlCell(this._nbSRow + i, 2),
                    new XlCell(this._nbSRow + i, this._iterCount + 1)), xValues));
                buffer.AddChartInfo(nbChartFormat);
            }
        }
        void SetupCharts_IndustryLevel(OutputBuffer buffer, double maxStnd, double minStnd, double bcMaxStnd,
            double bcMinStnd, double mbcMaxStnd, double mbcMinStnd, int stndSRow, int cafeSRow, int nbSRow, int co2_mmtSRow,
            int co2_dlrSRow, int fsSRow, int bcSRow, int mbcSRow, int minYear, int iterCount, int yearCount, bool smoothing)
        {
            string bufName = buffer.Name;
            string nbChartName      = bufName + "-NBGraph";
            string cafeChartName    = bufName + "-CAFEGraph";
            string co2_mmtChartName = bufName + "-CO2_MMTGraph";
            string co2_dlrChartName = bufName + "-CO2_DollarGraph";
            string fsChartName      = bufName + "-FSGraph";
            string bcChartName      = bufName + "-BCGraph";
            string mbcChartName     = bufName + "-MBCGraph";
            maxStnd    = Math.Floor  (maxStnd   ) + 1;
            minStnd    = Math.Ceiling(minStnd   ) - 1;
            bcMaxStnd  = Math.Floor  (bcMaxStnd ) + 1;
            bcMinStnd  = Math.Ceiling(bcMinStnd ) - 1;
            mbcMaxStnd = Math.Floor  (mbcMaxStnd) + 1;
            mbcMinStnd = Math.Ceiling(mbcMinStnd) - 1;
            XlChartFormat nbChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, nbChartName,
                nbChartName, XlColor.White, new XlSeriesFormat[0], null, null);
            XlChartFormat cafeChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, cafeChartName,
                cafeChartName, XlColor.White, new XlSeriesFormat[0], null, null);
            XlChartFormat co2_mmtChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, co2_mmtChartName,
                co2_mmtChartName, XlColor.White, new XlSeriesFormat[0], null, null);
            XlChartFormat co2_dlrChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, co2_dlrChartName,
                co2_dlrChartName, XlColor.White, new XlSeriesFormat[0], null, null);
            XlChartFormat fsChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, fsChartName,
                fsChartName, XlColor.White, new XlSeriesFormat[0], null, null);
            XlChartFormat bcChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, bcChartName,
                bcChartName, XlColor.White, new XlSeriesFormat[0], null, null);
            XlChartFormat mbcChartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, mbcChartName,
                mbcChartName, XlColor.White, new XlSeriesFormat[0], null, null);
            XlTickLabelsFormat ticksFormat = new XlTickLabelsFormat(XlTickLabelsFormat.DefaultFont, "#,##0.0");
            XlGridlinesFormat gridlinesFormat = new XlGridlinesFormat(true, true, false, XlColor.Gray25);
            XlAxisFormat valuesAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, 0.5, 0.1, 2, 0,
                false, false, gridlinesFormat, ticksFormat);
            nbChartFormat.ValuesAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, double.NaN, double.NaN, false, false, gridlinesFormat, ticksFormat);
            nbChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, maxStnd, minStnd, false, false, gridlinesFormat, ticksFormat);
            cafeChartFormat.ValuesAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, double.NaN, double.NaN, false, false, gridlinesFormat, ticksFormat);
            cafeChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, maxStnd, minStnd, false, false, gridlinesFormat, ticksFormat);
            co2_mmtChartFormat.ValuesAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, double.NaN, double.NaN, false, false, gridlinesFormat, ticksFormat);
            co2_mmtChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, maxStnd, minStnd, false, false, gridlinesFormat, ticksFormat);
            co2_dlrChartFormat.ValuesAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, double.NaN, double.NaN, false, false, gridlinesFormat, ticksFormat);
            co2_dlrChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, maxStnd, minStnd, false, false, gridlinesFormat, ticksFormat);
            fsChartFormat.ValuesAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, double.NaN, double.NaN, false, false, gridlinesFormat, ticksFormat);
            fsChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, maxStnd, minStnd, false, false, gridlinesFormat, ticksFormat);
            bcChartFormat.ValuesAxisFormat = valuesAxisFormat;
            bcChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, bcMaxStnd, bcMinStnd, false, false, gridlinesFormat, ticksFormat);
            mbcChartFormat.ValuesAxisFormat = valuesAxisFormat;
            mbcChartFormat.CategoryAxisFormat = new XlAxisFormat(false, null, true, false, double.NaN, double.NaN,
                double.NaN, mbcMaxStnd, mbcMinStnd, false, false, gridlinesFormat, ticksFormat);
            int mbcIterCount = iterCount + ((smoothing) ? 0 : 1);
            for (int j = 0; j < yearCount; j++)
            {
                nbChartFormat.AddSeries(new XlSeriesFormat(bufName + " " + (minYear + j),
                    new XlRange(new XlCell(nbSRow + j, 2), new XlCell(nbSRow + j, iterCount + 1)),
                    new XlRange(new XlCell(stndSRow + j, 2), new XlCell(stndSRow + j, iterCount + 1))));
                cafeChartFormat.AddSeries(new XlSeriesFormat(bufName + " " + (minYear + j),
                    new XlRange(new XlCell(cafeSRow + j, 2), new XlCell(cafeSRow + j, iterCount + 1)),
                    new XlRange(new XlCell(stndSRow + j, 2), new XlCell(stndSRow + j, iterCount + 1))));
                co2_mmtChartFormat.AddSeries(new XlSeriesFormat(bufName + " " + (minYear + j),
                    new XlRange(new XlCell(co2_mmtSRow + j, 2), new XlCell(co2_mmtSRow + j, iterCount + 1)),
                    new XlRange(new XlCell(stndSRow + j, 2), new XlCell(stndSRow + j, iterCount + 1))));
                co2_dlrChartFormat.AddSeries(new XlSeriesFormat(bufName + " " + (minYear + j),
                    new XlRange(new XlCell(co2_dlrSRow + j, 2), new XlCell(co2_dlrSRow + j, iterCount + 1)),
                    new XlRange(new XlCell(stndSRow + j, 2), new XlCell(stndSRow + j, iterCount + 1))));
                fsChartFormat.AddSeries(new XlSeriesFormat(bufName + " " + (minYear + j),
                    new XlRange(new XlCell(fsSRow + j, 2), new XlCell(fsSRow + j, iterCount + 1)),
                    new XlRange(new XlCell(stndSRow + j, 2), new XlCell(stndSRow + j, iterCount + 1))));
                bcChartFormat.AddSeries(new XlSeriesFormat(bufName + " " + (minYear + j),
                    new XlRange(new XlCell(bcSRow + j, 2), new XlCell(bcSRow + j, iterCount + 1)),
                    new XlRange(new XlCell(stndSRow + j, 2), new XlCell(stndSRow + j, iterCount + 1))));
                mbcChartFormat.AddSeries(new XlSeriesFormat(bufName + " " + (minYear + j),
                    new XlRange(new XlCell(mbcSRow + j, 3), new XlCell(mbcSRow + j, mbcIterCount)),
                    new XlRange(new XlCell(stndSRow + j, 3), new XlCell(stndSRow + j, mbcIterCount))));
            }
            XlSeriesFormat oneToOne = new XlSeriesFormat("1:1", new object[] {1, 1}, new object[] {0, iterCount});
            oneToOne.Color = XlColor.Black;
            oneToOne.ThickLine = true;
            bcChartFormat.AddSeries(oneToOne);
            mbcChartFormat.AddSeries(oneToOne);
            buffer.AddChartInfo(nbChartFormat);
            buffer.AddChartInfo(cafeChartFormat);
            buffer.AddChartInfo(co2_mmtChartFormat);
            buffer.AddChartInfo(co2_dlrChartFormat);
            buffer.AddChartInfo(fsChartFormat);
            buffer.AddChartInfo(bcChartFormat);
            buffer.AddChartInfo(mbcChartFormat);
        }
        void SetupCharts_DrawShape()
        {
            bool isPredefined = (this._ffMode == FunctionalFitMode.None);
            bool fpBased      = FunctionInformation.IsAreaBasedFunction(this._fncType[this._minOptYear - this._minYear]);
            OutputBuffer buffer     = Buffers[this._mfrCount];
            string       sheetName  = buffer.Name;
            string       chartName  = sheetName + "-Shape";
            string       chartTitle = "Function Type " + this._fncType[this._minOptYear - this._minYear] + ", using " +
                ((isPredefined) ? "Predefined Shape" : this._ffMode.ToString());
            double minScaleY = Math.Floor  (this._minBCoef - 1);
            double maxScaleY = Math.Ceiling(this._maxACoef + 1);
            double minScaleX = Math.Floor  ((fpBased) ? Math.Floor  ((this._minFp - 1) / 5) * 5 :
                Math.Floor  ((this._minCw - 1) / 250) * 250);
            double maxScaleX = Math.Ceiling((fpBased) ? Math.Ceiling((this._maxFp + 1) / 5) * 5 :
                Math.Ceiling((this._maxCw + 1) / 250) * 250);
            double xScaleOff = (fpBased) ? 0.1 : 10.0;
            int    xScaleDif = (int)Math.Ceiling((maxScaleX - minScaleX) / xScaleOff) + 1;
            string shapeFpCw     = "fpCw";
            string shapeFpCwName = sheetName + "!" + shapeFpCw;
            string shapeFpCwExpr = "=" + minScaleX + "+(ROW(OFFSET(" + "$A$1,0,0," + xScaleDif + ",1))-1)*" + xScaleOff;
            buffer.AddExcelName(shapeFpCwName, shapeFpCwExpr);
            XlChartFormat chartFormat = new XlChartFormat(XlChartType.XYScatterLinesNoMarkers, chartName,
                chartTitle, XlColor.White, new XlSeriesFormat[0], null, null);
            XlTickLabelsFormat ticksFormat = new XlTickLabelsFormat(XlTickLabelsFormat.DefaultFont, "#,##0.0");
            XlGridlinesFormat gridlinesFormat = new XlGridlinesFormat(true, true, false, XlColor.Gray25);
            chartFormat.ValuesAxisFormat   = new XlAxisFormat(true, new XlAxisTitleFormat("Target (mpg)", 90), true, false,
                double.NaN, 5, 1, maxScaleY, minScaleY, false, false, gridlinesFormat, ticksFormat);
            chartFormat.CategoryAxisFormat = new XlAxisFormat(true,
                new XlAxisTitleFormat((fpBased) ? "Footprint (sf)" : "Weight (lbs)"), true, false,
                double.NaN, xScaleOff * 50, double.NaN, maxScaleX, minScaleX, false, false, gridlinesFormat, ticksFormat);
            for (int i = 0; i < this._yearCount; i++)
            {   
                int  fncType       = this._fncType[i];
                bool isLogistic    = FunctionInformation.IsLogisticFunction   (fncType);
                bool isExponential = FunctionInformation.IsExponentialFunction(fncType);
                bool isLinear      = FunctionInformation.IsLinearFunction     (fncType);
                int  coefCount     = FunctionInformation.GetCoefficientsCount (fncType);
                if (!isLogistic && !isLinear) { continue; }
                this.SetupCharts_DrawShape_Helper(buffer, sheetName, chartFormat, i, shapeFpCw, shapeFpCwName,
                    isLogistic, isExponential, isLinear, coefCount);
            } 
            buffer.AddChartInfo(chartFormat);
        }
        void SetupCharts_DrawShape_Helper(OutputBuffer buffer, string sheetName, XlChartFormat chartFormat, int yrOffset,
            string shapeFpCw, string shapeFpCwName, bool isLogistic, bool isExponential, bool isLinear, int coefCount)
        {
            int    rowOffset = 2             + yrOffset;    
            int    year      = this._minYear + yrOffset;
            string yearStr   = year.ToString();
            string shapeName = sheetName + "!sh_" + yearStr;
            string shapeExpr = string.Empty;
            if (isLogistic)
            {
                shapeExpr = "=EVALUATE(\"1/(1/$B" + rowOffset + "+(1/$C" + rowOffset + "-1/$B" + rowOffset +
                    ")*EXP((fpCw-$D" + rowOffset + ")/$E" + rowOffset + ")/(1+EXP((fpCw-$D" + rowOffset + ")/$E" +
                    rowOffset + ")))+" + shapeFpCw + "*0\")";
            }
            else if (isExponential)
            {
            }
            else if (isLinear)
            {
                string lnrFc = "$D" + rowOffset + "*fpCw+$E" + rowOffset;
                string uBoundExpr = "if(" + lnrFc + ">1/$C" + rowOffset + ",1/$C" + rowOffset + "," + lnrFc      + ")";
                string lBoundExpr = "if(" + lnrFc + "<1/$B" + rowOffset + ",1/$B" + rowOffset + "," + uBoundExpr + ")";
                shapeExpr = "=EVALUATE(\"1/(" + ((coefCount == 3) ? uBoundExpr : lBoundExpr) + ")+" + shapeFpCw + "*0\")";
            }
            buffer.AddExcelName(shapeName, shapeExpr);
            XlSeriesFormat seriesFormat = new XlSeriesFormat(yearStr, "=" + shapeName, "=" + shapeFpCwName);
            if (year < this._minOptYear)
            {   
                seriesFormat.SolidLine = false;
            }
            chartFormat.AddSeries(seriesFormat);
        }
        #region 
        public override void WriteBuffers(Input template, bool templateFirstBuffer)
        {
            this.SetupCharts();
            base.WriteBuffers(template, templateFirstBuffer);
        }
        protected override void WriteBuffer()
        {
            base.WriteBuffer();
            if (!this._cancelWrite)
            {   
                int row1 = 2;
                int row2 = 1 + this._yearCount;
                for (int i = 0; i < SectionCount; i++)
                {
                    this._xlu.SetIndent(new XlCell(row1, 1), new XlCell(row2, 1), 1);
                    row1 += this._yearCount + 2;
                    row2 += this._yearCount + 2;
                    if      (i == SectionMainStart - 1                   ) { row2++; }
                    else if (i >= SectionMainStart && i <= SectionMainEnd) { row1++; if (i < SectionMainEnd) { row2++; } }
                }
            }
        }
        protected override void SetFont(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2, XlCell dataCell1,
            XlCell dataCell2)
        {
            base.SetFont(rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            XlFont font = new XlFont("Arial", 10, false, true, false);
            int row = 1;
            for (int i = 0; i < SectionCount; i++)
            {
                XlCell cell = new XlCell(row++, 1);
                this._xlu.SetFont(cell, cell, font);
                if (i != SectionOpt)
                {   
                    for (int j = 0; j < this._yearCount; j++)
                    {   
                        if (this._minYear + j < this._minOptYear) { row++; continue; }
                        cell = new XlCell(row++, this._optIdx[j] - this._iterOffset + 2);
                        this._xlu.SetFont(cell, cell, font);
                    }
                } else { row += this._yearCount; }
                row++;
                if (i >= SectionMainStart && i <= SectionMainEnd) { row++; }
            }
        }
        protected override void SetNumberFormat(XlCell dataCell1, XlCell dataCell2)
        {
            dataCell1.Column = 2;
            base.SetNumberFormat(dataCell1, dataCell2);
            dataCell1.Row = this._bcSRow;
            this._xlu.SetNumberFormat(dataCell1, dataCell2, XlNumberFormats.NumericDecimal4);
            this._xlu.SetNumberFormat(new XlCell(this._yearCount + 3, 2), new XlCell(this._yearCount + 3, dataCell2.Column),
                XlNumberFormats.Numeric);
            dataCell1.Row = 2;                   dataCell1.Column = 7;
            dataCell2.Row = this._yearCount + 1; dataCell2.Column = 7;
            this._xlu.SetNumberFormat(dataCell1, dataCell2, XlNumberFormats.Numeric);
        }
        protected override void SetBorders(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2, XlCell dataCell1,
            XlCell dataCell2)
        {
            int cols = this.Buffer.Size.Columns;
            int row1 = 1;
            int row2 = 1 + this._yearCount;
            int row3 = 4 + this._yearCount;
            this._xlu.SetBorders(new XlCell(1, 2), new XlCell(1, 7), false, true, false, false, XlColor.DarkBlue);
            this._xlu.SetBorders(new XlCell(this._yearCount + 3, 2), new XlCell(this._yearCount + 3, dataCell2.Column),
                false, true, false, false, XlColor.DarkBlue);
            for (int i = 0; i < SectionCount; i++)
            {
                if (i != SectionOpt)
                {   
                    for (int j = 0; j < this._yearCount; j++)
                    {   
                        if (this._minYear + j < this._minOptYear) { row3++; continue; }
                        XlCell cell = new XlCell(row3++, this._optIdx[j] - this._iterOffset + 2);
                        this._xlu.SetBorders(cell, cell, true, false, true, false, XlColor.Blue);
                    }
                    row3 += 2;
                    if (i >= SectionMainStart && i <= SectionMainEnd) { row3++; }
                }
                this._xlu.SetBorders(new XlCell(row1, 1), new XlCell(row1, (i == SectionOpt) ? 7 : cols),
                    true, true, false, false, true, true, XlColor.DarkBlue);
                this._xlu.SetBorders(new XlCell(row1, 1), new XlCell(row2, 1),
                    false, false, false, true, true, false, XlColor.DarkBlue);
                if (i == SectionOpt)
                {   
                    this._xlu.SetBorders(new XlCell(row1, 5), new XlCell(row2, 5),
                        false, false, false, true, true, false, XlColor.DarkBlue);
                }
                else if (i >= SectionMainStart && i <= SectionMainEnd)
                {   
                    this._xlu.SetBorders(new XlCell(row2, 1), new XlCell(row2, (i == SectionOpt) ? 7 : cols),
                        true, false, false, false, true, false, XlColor.DarkBlue);
                }
                row1 += this._yearCount + 2;
                row2 += this._yearCount + 2;
                if      (i == SectionMainStart - 1                   ) { row2++; }
                else if (i >= SectionMainStart && i <= SectionMainEnd) { row1++; if (i < SectionMainEnd) { row2++; } }
            }
        }
        protected override void AlignText(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2, XlCell dataCell1,
            XlCell dataCell2)
        {
            base.AlignText (rHeader1, rHeader2, cHeader1, cHeader2, dataCell1, dataCell2);
            this._xlu.AlignText(new XlCell(1, 2), new XlCell(1, 7), new XlTextAlignment(XlHAlign.Center, XlVAlign.Bottom));
            this._xlu.AlignText(new XlCell(this._yearCount + 3, 2), new XlCell(this._yearCount + 3, dataCell2.Column),
                new XlTextAlignment(XlHAlign.Center, XlVAlign.Bottom));
        }
        protected override void SetColor(XlCell rHeader1, XlCell rHeader2, XlCell cHeader1, XlCell cHeader2, XlCell dataCell1,
            XlCell dataCell2)
        {
            this._xlu.SetFillColor(XlCell.Empty, XlCell.Empty, XlColor.White);
            int cols = this.Buffer.Size.Columns;
            int row = 1;
            for (int i = 0; i < SectionCount; i++)
            {
                this._xlu.SetFillColor(new XlCell(row, 1), new XlCell(row, (i == SectionOpt) ? 7 : cols), XlColor.LightTurqoise);
                row++;
                if (i != SectionOpt)
                {   
                    for (int j = 0; j < this._yearCount; j++)
                    {   
                        if (this._minYear + j < this._minOptYear) { row++; continue; }
                        XlCell cell = new XlCell(row++, this._optIdx[j] - this._iterOffset + 2);
                        this._xlu.SetFillColor(cell, cell, XlColor.LightYellow);
                    }
                } else { row += this._yearCount; }
                row++;
                if (i >= SectionMainStart && i <= SectionMainEnd) { row++; }
            }
        }
        #endregion
        #endregion
        #region 
        readonly static string[] Sections = new string[] {"Optimized", "Standards", "CAFE", "Tech Costs", "Total Fines",
            "Benefits", "Disc Benefits", "Net Benefits", "CO-2 Emission Savings (mmT)", "CO-2 Monetized Savings (k $)",
            "Fuel Savings (k gal.)", "B:C Ratio", "MB:C Ratio"};
        readonly static int SectionCount = Sections.Length;
        const int SectionOpt       =  0;
        const int SectionStnd      =  1;
        const int SectionCAFE      =  2;
        const int SectionMainStart =  3;
        const int SectionMainEnd   = 10;
        const int SectionNB        =  7;
        const int SectionCO2_MMT   =  8;
        const int SectionCO2_Dlr   =  9;
        const int SectionFS        = 10;
        const int SectionBCR       = 11;
        const int SectionMBCR      = 12;
        bool                 _isOpt;
        OptimizationSettings _optSettings;
        Scenario _scen;
        bool _initialized;
        int _minYear;
        int _maxYear;
        int _minOptYear;
        int _yearCount;
        int    _iterCount;      
        int    _iterOffset;     
        int      _mfrCount;
        string[] _mfrNames;
        int _cafeSRow;
        int _nbSRow;
        int _co2_mmtSRow;
        int _co2_dlrSRow;
        int _fsSRow;
        int _bcSRow;
        int _mbcSRow;
        OptimizationType  _optType;
        FunctionalFitMode _ffMode;
        int[]  _fncType;
        double _minFp, _maxFp, _minCw, _maxCw;
        double _minBCoef, _maxACoef;
        int[] _optIdx;
        double _minStnd   , _maxStnd;
        double _bcMinStnd , _bcMaxStnd;
        double _mbcMinStnd, _mbcMaxStnd;
        bool   _smoothing;      
        #endregion
    }
}

