using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using RC = Volpe.Cafe.RegulatoryClass;
namespace Volpe.Cafe.Data
{
    [Serializable]
    public class Vehicle : ICloneable
    {
        #region 
        private Vehicle()
        {
            this._vehicleType  = VehicleType.None;
            this._engine       = null;
            this._transmission = null;
            this._namePlate    = null;
            this._manufacturer = null;
            this._predecessor  = null;
            this._successors   = new VehicleCollection(8);
            this._regClass     = RegulatoryClass.Unregulated;
            this._techClass    = TechnologyClass.None;
            this._mobile6Class = Mobile6Class.None;
        }
        public Vehicle(VehicleDescription description)
            : this()
        {
            this._description = description;
            this._modelingData = new VehicleModelingData();
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Vehicle Clone()
        {
            Vehicle veh = new Vehicle();
            if (this._validYears != null)
            {
                veh._validYears = new ModelYear[this._validYears.Length];
                for (int i = 0; i < this._validYears.Length; i++)
                {
                    veh._validYears[i] = new ModelYear(this._validYears[i].Year);
                }
            }
            veh._description  = this._description.Clone();
            veh._modelingData = this._modelingData.Clone();
            veh._vehicleType  = this._vehicleType;
            veh._engine       = this._engine;
            veh._transmission = this._transmission;
            veh._namePlate    = this._namePlate;
            veh._manufacturer = this._manufacturer;
            veh._predecessor  = this._predecessor;
            veh._successors.AddRange(this._successors);
            veh._regClass     = this._regClass;
            veh._mobile6Class = this._mobile6Class;
            return veh;
        }
        #endregion
        internal void Initialize(Manufacturer manufacturer, Vehicle[] vehicles)
        {
            this._manufacturer = manufacturer;
            this._engine = null;
            this._transmission = null;
            this._predecessor = null;
            Engine[] engs = manufacturer.Engines.ToArray();
            Transmission[] trns = manufacturer.Transmissions.ToArray();
            for (int i = 0; i < engs.Length; i++)
            {
                if (engs[i].Description.Code == this._description.EngineCode)
                {   
                    this._engine = engs[i];
                    this._engine.Vehicles.Add(this);
                    break;
                }
            }
            for (int i = 0; i < trns.Length; i++)
            {
                if (trns[i].Description.Code == this._description.TransmissionCode)
                {   
                    this._transmission = trns[i];
                    this._transmission.Vehicles.Add(this);
                    break;
                }
            }
            for (int i = 0; i < vehicles.Length; i++)
            {
                if (this._description.PredecessorCode == vehicles[i]._description.Code)
                {   
                    this._predecessor = vehicles[i];
                    vehicles[i]._successors.Add(this);
                    break;
                }
            }
            int yearCount = this._description.Sales.Length;
            int slMsrpYrs = 0;
            for (int i = 0; i < yearCount; i++)
            {
                if (this._description.Sales[i] > 0 && this._description.Msrp[i] > 0) { slMsrpYrs++; }
            }
            this._validYears = new ModelYear[slMsrpYrs];
            for (int i = 0, j = 0; i < yearCount; i++)
            {
                if (this._description.Sales[i] > 0 && this._description.Msrp[i] > 0) { this._validYears[j++] = new ModelYear(i + ModelYear.MinYear); }
            }
            if (this._engine == null      ) { throw new Exception("\n\nVehicle Code " + this._description.Code + " has an invalid Engine Code specified."); }
            if (this._transmission == null) { throw new Exception("\n\nVehicle Code " + this._description.Code + " has an invalid Transmission Code specified."); }
            this.AssignTypeAndMobile6Class();
            this.AssignTechnologyClass();
        }
        internal void ValidatePredecessor()
        {
            if (this._description.PredecessorCode == 0) { return; }
            if (this._predecessor == null)
            {   
                throw new Exception("\n\nVehicle Code " + this._description.Code + " has an invalid Predecessor Code specifed.");
            }
            else if (this._predecessor == this || this._predecessor._description.Code == this._description.Code)
            {   
                throw new Exception("\n\nVehicle Code " + this._description.Code + " lists itself as the predecessor.");
            }
            else if (this._validYears[0].Index < this._predecessor._validYears[0].Index)
            {   
                throw new Exception("\n\nVehicle Code " + this._description.Code + " lists a possible successor as the predecessor.");
            }
        }
        private void AssignTypeAndMobile6Class()
        {
            string style  = this._description.Style;
            double gvwr   = this._description.GVWR;
            double weight = this._description.CurbWeight;
            string fuel   = this._engine.Description.Fuel;
            bool   isDom  = (this._description.Origin == 'D');
            if (Global.StringCompareAny(style, new string[] {"WAGON", "SEDAN", "CONVERTIBLE", "COUPE", "HATCHBACK"}, false))
            {   
                this._vehicleType = VehicleType.PassengerAuto;
                this._mobile6Class = (fuel == "D") ? Mobile6Class.Lddv : Mobile6Class.Ldgv;
            }
            else if (gvwr > 8500 || weight > 6000)
            {   
                this._vehicleType = VehicleType.HeavyLT;
                this._mobile6Class = (fuel == "D") ? Mobile6Class.Hddv2b : Mobile6Class.Hdgv2b;
            }
            else
            {   
                this._vehicleType = VehicleType.LightTruck;
                this._mobile6Class = (fuel == "D") ?
                    ((gvwr <= 6000) ? Mobile6Class.Lddt1 : Mobile6Class.Lddt2) :
                    ((gvwr <= 6000) ? Mobile6Class.Ldgt1 : Mobile6Class.Ldgt2);
            }
            this._vehicleType |= ((isDom) ? VehicleType.Domestic : VehicleType.Imported);
        }
        private void AssignTechnologyClass()
        {
            string[] techClasses =
                new string[] {
                                 "SUBCOMPACT PC", "SUBCOMPACT PERF. PC", "COMPACT PC", "COMPACT PERF. PC",
                                 "MIDSIZE PC"   , "MIDSIZE PERF. PC"   , "LARGE PC"  , "LARGE PERF. PC"  ,
                                 "MINIVAN LT"   , "SMALL LT"           , "MIDSIZE LT", "LARGE LT"
                             };
            this._techClass = TechnologyClass.None;
            for (int i = 0; i < techClasses.Length; i++)
            {
                if (Global.StringCompare(this._description.TechnologyClass, techClasses[i], true))
                {
                    this._techClass = (TechnologyClass)(i + 1);
                }
            }
        }
        internal void SetEngine(Engine engine)
        {
            this._engine = engine;
            this._description.EngineCode = engine.Description.Code;
        }
        internal void SetTransmission(Transmission transmission)
        {
            this._transmission = transmission;
            this._description.TransmissionCode = transmission.Description.Code;
        }
        internal void SetNamePlate(NamePlate namePlate)
        {
            this._namePlate = namePlate;
        }
        internal void SetModelingData(VehicleModelingData value)
        {
            this._modelingData = value;
        }
        public void SetEngineAndTransmissionFromPredecessor()
        {
            if (this._predecessor != null)
            {   
                Engine       predEng = this._predecessor._engine;
                Transmission predTrn = this._predecessor._transmission;
                if (predEng.Parent != null && predEng.Parent.Description.Code == this._engine.Description.Code)
                {   
                    predEng.Vehicles.Add(this);
                    predEng.ModelYearVehicles.Add(this);
                    this._engine.Vehicles.Remove(this);
                    this._engine.ModelYearVehicles.Remove(this);
                    this.SetEngine(predEng);
                }
                if (predTrn.Parent != null && predTrn.Parent.Description.Code == this._transmission.Description.Code)
                {   
                    predTrn.Vehicles.Add(this);
                    predTrn.ModelYearVehicles.Add(this);
                    this._transmission.Vehicles.Remove(this);
                    this._transmission.ModelYearVehicles.Remove(this);
                    this.SetTransmission(predTrn);
                }
            }
        }
        public bool IsValid(ModelYear year)
        {
            return this.IsValid(year.Index);
        }
        public bool IsValid(int yrIndex)
        {
            if (yrIndex < 0 || yrIndex > this._description.Sales.Length)
            {   
                return false;
            }
            else
            {
                return (this._description.Sales[yrIndex] > 0) && (this._description.Msrp[yrIndex] > 0) &&
                    (this._description.Price[yrIndex] > 0);
            }
        }
        public void UpdateTypeAndMobile6Class()
        {
            if (this._mobile6Class == Mobile6Class.None)
            {
                throw new InvalidOperationException("The vehicle being updated is not valid or does not have a " +
                    "valid Mobile6 Class assignment.  Updating the Mobile6 Class or Vehicle Type of invalid " +
                    "vehicles is not allowed.");
            }
            this.AssignTypeAndMobile6Class();
        }
        internal void SetTypeAndMobile6Class(VehicleType type, Mobile6Class mobile6Class)
        {
            this._vehicleType  = type;
            this._mobile6Class = mobile6Class;
        }
        public int[] GetRedesignYears()
        {
            return this.GetRedesignYears(true);
        }
        private int[] GetRedesignYears(bool checkPred)
        {
            if (checkPred && this._predecessor != null)
            {   
                return this._predecessor.GetRedesignYears(true);
            }
            else
            {   
                int[] redesignYears = this._description.RedesignYears;
                for (int i = 0, successorCount = this._successors.Count; i < successorCount; i++)
                {   
                    this.MergeRedesignRefreshYears(ref redesignYears, this._successors[i].GetRedesignYears(false));
                }
                return redesignYears;
            }
        }
        public int[] GetRefreshYears()
        {
            return this.GetRefreshYears(true);
        }
        private int[] GetRefreshYears(bool checkPred)
        {
            if (checkPred && this._predecessor != null)
            {   
                return this._predecessor.GetRefreshYears(true);
            }
            else
            {   
                int[] refreshYears = this._description.RefreshYears;
                for (int i = 0, successorCount = this._successors.Count; i < successorCount; i++)
                {   
                    this.MergeRedesignRefreshYears(ref refreshYears, this._successors[i].GetRefreshYears(false));
                }
                return refreshYears;
            }
        }
        private void MergeRedesignRefreshYears(ref int[] value1, int[] value2)
        {
            int v1Len = value1.Length, v2Len = value2.Length;
            int[] value = new int[v1Len + v2Len];
            int vLen = v1Len;
            Array.Copy(value1, value, vLen);
            for (int i = 0; i < v2Len; i++)
            {
                bool duplicate = false;
                for (int j = 0; j < vLen; j++)
                {   
                    if (value[j] == value2[i]) { duplicate = true; break; }
                }
                if (!duplicate) { value[vLen++] = value2[i]; }
            }
            value1 = new int[vLen];
            Array.Copy(value, value1, vLen);
            Array.Sort(value1);
        }
        public void GetRedesignState(ModelYear modelYear, out bool atRedesign, out bool inShadowOfRedesign)
        {
            atRedesign = inShadowOfRedesign = false;    
            int year = modelYear.Year;
            int[] redesignYears = this.GetRedesignYears();
            for (int i = 0, count = redesignYears.Length; i < count; i++)
            {
                if (redesignYears[i] == year) { atRedesign = true; break; }
            }
            if (!atRedesign)
            {
                for (int i = 0, count = redesignYears.Length; i < count; i++)
                {
                    int delta = Math.Abs(redesignYears[i] - year);    
                    if (0 < delta && delta < 5) { inShadowOfRedesign = true; break; }
                }
            }
        }
        public void GetRefreshState(ModelYear modelYear, out bool atRefresh, out bool inShadowOfRefresh)
        {
            atRefresh = inShadowOfRefresh = false;
            int year = modelYear.Year;
            int[] refreshYears = this.GetRefreshYears();
            for (int i = 0, count = refreshYears.Length; i < count; i++)
            {
                if (refreshYears[i] == year) { atRefresh = true; break; }
            }
            if (!atRefresh)
            {
                for (int i = 0, count = refreshYears.Length; i < count; i++)
                {
                    int delta = Math.Abs(refreshYears[i] - year);    
                    if (0 < delta && delta < 2) { inShadowOfRefresh = true; break; }
                }
            }
        }
        #endregion
        #region 
        public ModelYear[] ValidYears { get { return this._validYears; } }
        public ModelYear MinYear { get { return (this._validYears == null) ? null : this._validYears[0]; } }
        public ModelYear MaxYear { get { return (this._validYears == null) ? null : this._validYears[this._validYears.Length - 1]; } }
        public VehicleDescription Description { get { return this._description; } }
        public VehicleModelingData ModelingData { get { return this._modelingData; } }
        public VehicleType VehicleType { get { return this._vehicleType; } }
        public Engine Engine { get { return this._engine; } }
        public Transmission Transmission { get { return this._transmission; } }
        public NamePlate NamePlate { get { return this._namePlate; } }
        public Manufacturer Manufacturer { get { return this._manufacturer; } }
        public Vehicle Predecessor { get { return this._predecessor; } }
        public VehicleCollection Successors { get { return this._successors; } }
        public bool IsRcAuto { get { return (this._regClass == RC.DomesticAuto || this._regClass == RC.ImportedAuto); } }
        public bool IsRcTruck { get { return (this._regClass == RegulatoryClass.LightTruck); } }
        public RegulatoryClass RegClass { get { return this._regClass; } set { this._regClass = value; } }
        public TechnologyClass TechnologyClass { get { return this._techClass; } }
        public Mobile6Class Mobile6Class { get { return this._mobile6Class; } }
        #endregion
        #region 
        private ModelYear[] _validYears;
        private VehicleDescription _description;
        private VehicleModelingData _modelingData;
        private VehicleType _vehicleType;
        private Engine _engine;
        private Transmission _transmission;
        private NamePlate _namePlate;
        private Manufacturer _manufacturer;
        private Vehicle _predecessor;
        private VehicleCollection _successors;
        private RegulatoryClass _regClass;
        private TechnologyClass _techClass;
        private Mobile6Class _mobile6Class;
        #endregion
    }
}

