using System;
using Volpe.Cafe;
using Volpe.Cafe.Collections;
using Volpe.Cafe.Data;
using Volpe.Cafe.Settings;
using AdjUE = Volpe.Cafe.Parameters.AdjustedUpstreamEmissions;
namespace Volpe.Cafe.Model
{
    [Serializable]
    public class Effects
    {
        #region 
        private Effects(Scenario scenario, ModelYear year, Industry data, ModelingSettings settings)
        {
            this._data     = data;
            this._settings = settings;
            this._isBaseline = scenario.IsBaseline;
            this._myIdx      = year.Index;
            this._mYear      = year.Year;
            this._mfrCount   = data.ManufacturerCount;
            Parameters parameters = settings.Parameters;
            Parameters.EmissionDamageCosts emissionCosts = parameters.EmissionCosts;
            this._dmgCostCO  = emissionCosts.CO ;
            this._dmgCostVOC = emissionCosts.VOC;
            this._dmgCostNOX = emissionCosts.NOX;
            this._dmgCostPM  = emissionCosts.PM ;
            this._dmgCostSOX = emissionCosts.SOX;
            Parameters.EconomicDataValues  economicValues = parameters.EconomicValues;
            Parameters.ExternalCostsValues externalCosts  = parameters.ExternalCosts;
            Parameters.EconomicCostsValues economicCosts  = parameters.EconomicCosts;
            this._testVsOnRoadGap        = economicValues.TestVsOnRoadGap;
            this._reboundEffect          = economicValues.ReboundEffect;
            this._discountRate           = economicValues.DiscountRate;
            this._vehicleTravelTimeValue = economicValues.VehicleTravelTimeValue;
            this._vmtBaseYear            = economicValues.VMTBaseYear;
            this._vmtCarGrowth           = economicValues.VMTCarGrowthRate;
            this._vmtTruckGrowth         = economicValues.VMTTruckGrowthRate;
            this._carCongestion          = externalCosts .CarCongestion;
            this._carAccident            = externalCosts .CarAccident;
            this._carNoise               = externalCosts .CarNoise;
            this._ltCongestion           = externalCosts .TruckCongestion;
            this._ltAccident             = externalCosts .TruckAccident;
            this._ltNoise                = externalCosts .TruckNoise;
            this._monopsonyCost          = economicCosts .Monopsony;
            this._priceShockCost         = economicCosts .PriceShock;
            this._militarySecurityCost   = economicCosts .MilitarySecurity;
            this._convGasMix             = parameters.GasolineMix.GasolineConventional;
        }
        #endregion
        #region 
        public static void Process(Scenario scenario, ModelYear year, Industry data, ModelingSettings settings)
        {
            Effects eff = new Effects(scenario, year, data, settings);
            eff.CalculateAggregatedEffects();
        }
        private void CalculateVehicleEffects(int mfrIdx, AggregateEffectsData ied, ManufacturerModelingData mmd,
            VehicleCollection vc)
        {
            Parameters     parameters = this._settings.Parameters;
            EmissionsRates emissions  = this._settings.EmissionsRates;
            Estimates      fuelPriceEstimates       = this._settings.OperatingModes.FuelPriceEstimates;
            bool           scaleConsumerBenefits    = this._settings.ParametersOverrides.ScaleConsumerBenefits;
            double         scaleConsumerBenefitsPct = this._settings.ParametersOverrides.ConsumerBenefits;
            CO2Estimates   co2Estimates             = this._settings.OperatingModes.CO2Estimates;
            double[]       co2Cost;
            double         co2DiscRate;
            parameters.EmissionCosts.GetCO2Estimates(co2Estimates, out co2Cost, out co2DiscRate);
            for (int i = 0; i < vc.Count; i++)
            {    
                Vehicle veh = vc[i];
                VehicleDescription vd = veh.Description;
                Mobile6Class    mblClass = veh.Mobile6Class;
                RegulatoryClass regClass = veh.RegClass;
                bool isCar = (mblClass == Mobile6Class.Ldgv || mblClass == Mobile6Class.Lddv);
                string fuel         = veh.Engine.Description.Fuel;
                double fuelEconomy  = vd .FuelEconomy;
                double fuelCapacity = vd .FuelCapacity;
                double sales        = vd .Sales[this._myIdx];
                double[] survRate = parameters.GetVehicleSurvivability(veh);
                double[] milesDrv = parameters.GetVehicleMilesDriven  (veh, fuelPriceEstimates);
                double[] fuelPrice, fuelTaxArr, preTaxFuelArr;
                double fuelLowImp, fuelReducedRef, domReducedRef, impReducedRef;
                double massDensity, carbonContent, soxRate;
                AdjUE adjCO, adjVOC, adjNOX, adjPM, adjSOX, adjCO2;
                this.InitEffectsByFuel(fuel, out fuelPrice, out fuelTaxArr, out preTaxFuelArr, out fuelLowImp,
                    out fuelReducedRef, out domReducedRef, out impReducedRef, out massDensity, out carbonContent,
                    out soxRate, out adjCO, out adjVOC, out adjNOX, out adjPM, out adjSOX, out adjCO2);
                double[] coRate  = this.GetEmissionsRates(emissions.CO , mblClass, this._convGasMix, this._myIdx);
                double[] vocRate = this.GetEmissionsRates(emissions.VOC, mblClass, this._convGasMix, this._myIdx);
                double[] noxRate = this.GetEmissionsRates(emissions.NOX, mblClass, this._convGasMix, this._myIdx);
                double[] pmRate  = this.GetEmissionsRates(emissions.PM , mblClass, this._convGasMix, this._myIdx);
                double baselineFe = (this._isBaseline) ? fuelEconomy :
                    this._data.ModelingData.BaselineInfo[this._myIdx].FinalFuelEconomy[mfrIdx][i];
                ied.Sales      [mblClass] += sales;
                ied.FuelEconomy[mblClass] += (sales / fuelEconomy);     
                for (int j = 0; j < Global.CY; j++)
                {
                    double fuelPrice_j  = fuelPrice [Math.Min(this._myIdx + j, fuelPrice .Length - 1)];
                    double fuelTax_j    = fuelTaxArr[Math.Min(this._myIdx + j, fuelTaxArr.Length - 1)];
                    double preTaxFuel_j = preTaxFuelArr[Math.Min(this._myIdx + j, preTaxFuelArr.Length - 1)];
                    double discRatePow  = Math.Pow(1 + this._discountRate, -j);
                    double co2DiscRatePow = Math.Pow(1 + co2DiscRate, -j);
                    double fleet       = sales * survRate[j];
                    double cpm         = fuelPrice_j / fuelEconomy * (1 - this._testVsOnRoadGap);
                    double baselineCpm = fuelPrice_j / baselineFe  * (1 - this._testVsOnRoadGap);
                    double baselineVmt = fleet * milesDrv[j] * 1e-3;
                    if (isCar) { baselineVmt *= Math.Pow(1 + this._vmtCarGrowth  , this._mYear + j - this._vmtBaseYear); }
                    else       { baselineVmt *= Math.Pow(1 + this._vmtTruckGrowth, this._mYear + j - this._vmtBaseYear); }
                    double vmt;
                    if (this._isBaseline) { vmt = baselineVmt; }
                    else
                    {   
                        vmt = baselineVmt + (baselineVmt * ((cpm / baselineCpm) - 1) * this._reboundEffect);
                    }
                    double gallons = vmt / (fuelEconomy * (1 - this._testVsOnRoadGap));
                    double fuelTax    = gallons * fuelTax_j;
                    double preTaxFuel = gallons * preTaxFuel_j;
                    if (scaleConsumerBenefits)
                    {
                        fuelTax    *= scaleConsumerBenefitsPct;
                        preTaxFuel *= scaleConsumerBenefitsPct;
                    }
                    double discFuelTax    = fuelTax * discRatePow;
                    double discPreTaxFuel = preTaxFuel * discRatePow;
                    const double refuelPct  = 0.55;
                    const double refuelTime = 5D / 60D;
                    double driveSurplus  = -0.5 * (baselineVmt * ((cpm / baselineCpm) - 1) * this._reboundEffect) * (baselineCpm - cpm) + ((baselineVmt - vmt) * cpm);
                    double refuelSurplus = -((milesDrv[j] / (refuelPct * fuelCapacity * (1 - this._testVsOnRoadGap) * baselineFe)) - (milesDrv[j] * (1 + ((cpm / baselineCpm) - 1) * this._reboundEffect) / (refuelPct * fuelCapacity * (1 - this._testVsOnRoadGap) * fuelEconomy))) * fleet * (refuelTime * this._vehicleTravelTimeValue * 1e-3);
                    if (scaleConsumerBenefits)
                    {
                        driveSurplus  *= scaleConsumerBenefitsPct;
                        refuelSurplus *= scaleConsumerBenefitsPct;
                    }
                    double discDriveSurplus  = driveSurplus * discRatePow;
                    double discRefuelSurplus = refuelSurplus * discRatePow;
                    double congestionCosts, accidentCosts, noiseCosts;
                    if (isCar)
                    {
                        congestionCosts = vmt * this._carCongestion;
                        accidentCosts   = vmt * this._carAccident;
                        noiseCosts      = vmt * this._carNoise;
                    }
                    else
                    {
                        congestionCosts = vmt * this._ltCongestion;
                        accidentCosts   = vmt * this._ltAccident;
                        noiseCosts      = vmt * this._ltNoise;
                    }
                    double discCongestionCosts = congestionCosts * discRatePow;
                    double discAccidentCosts   = accidentCosts * discRatePow;
                    double discNoiseCosts      = noiseCosts * discRatePow;
                    double economicCosts       = gallons * (fuelLowImp + fuelReducedRef * impReducedRef) * (this._monopsonyCost + this._priceShockCost + this._militarySecurityCost);
                    double discEconomicCosts   = economicCosts * discRatePow;
                    double coEmissions  = this.CalculateEmission(domReducedRef, fuelReducedRef, adjCO , gallons, vmt, coRate , this._myIdx, j);
                    double vocEmissions = this.CalculateEmission(domReducedRef, fuelReducedRef, adjVOC, gallons, vmt, vocRate, this._myIdx, j);
                    double noxEmissions = this.CalculateEmission(domReducedRef, fuelReducedRef, adjNOX, gallons, vmt, noxRate, this._myIdx, j);
                    double pmEmissions  = this.CalculateEmission(domReducedRef, fuelReducedRef, adjPM , gallons, vmt, pmRate , this._myIdx, j);
                    double soxEmissions = ((fuelReducedRef * (domReducedRef * (adjSOX.Extraction + adjSOX.Transportation) + adjSOX.Refining) + adjSOX.TSD) + soxRate) * gallons * Effects.EmFract;
                    double co2Emissions = ((adjCO2.Extraction + adjCO2.Transportation + adjCO2.Refining + adjCO2.TSD) + (massDensity * carbonContent * Effects.FourtyFourTwelfths)) * gallons * 1e-9;
                    double coDamage, coDiscDamage, vocDamage, vocDiscDamage, noxDamage, noxDiscDamage, pmDamage, pmDiscDamage, soxDamage, soxDiscDamage, co2Damage, co2DiscDamage;
                    this.CalculateEmissionCosts(coEmissions , this._dmgCostCO , discRatePow, out coDamage , out coDiscDamage , 1e-3);
                    this.CalculateEmissionCosts(vocEmissions, this._dmgCostVOC, discRatePow, out vocDamage, out vocDiscDamage, 1e-3);
                    this.CalculateEmissionCosts(noxEmissions, this._dmgCostNOX, discRatePow, out noxDamage, out noxDiscDamage, 1e-3);
                    this.CalculateEmissionCosts(pmEmissions , this._dmgCostPM , discRatePow, out pmDamage , out pmDiscDamage , 1e-3);
                    this.CalculateEmissionCosts(soxEmissions, this._dmgCostSOX, discRatePow, out soxDamage, out soxDiscDamage, 1e-3);
                    this.CalculateEmissionCosts(co2Emissions, co2Cost[this._myIdx + j], co2DiscRatePow, out co2Damage, out co2DiscDamage, 1e3);
                    ied.Vmt                [j][mblClass] += vmt;
                    ied.Gallons            [j][mblClass] += gallons;
                    ied.FuelTax            [j][mblClass] += fuelTax;
                    ied.DiscFuelTax        [j][mblClass] += discFuelTax;
                    ied.PreTaxFuel         [j][mblClass] += preTaxFuel;
                    ied.DiscPreTaxFuel     [j][mblClass] += discPreTaxFuel;
                    ied.DriveSurplus       [j][mblClass] += driveSurplus;
                    ied.DiscDriveSurplus   [j][mblClass] += discDriveSurplus;
                    ied.RefuelSurplus      [j][mblClass] += refuelSurplus;
                    ied.DiscRefuelSurplus  [j][mblClass] += discRefuelSurplus;
                    ied.CongestionCosts    [j][mblClass] += congestionCosts;
                    ied.DiscCongestionCosts[j][mblClass] += discCongestionCosts;
                    ied.AccidentCosts      [j][mblClass] += accidentCosts;
                    ied.DiscAccidentCosts  [j][mblClass] += discAccidentCosts;
                    ied.NoiseCosts         [j][mblClass] += noiseCosts;
                    ied.DiscNoiseCosts     [j][mblClass] += discNoiseCosts;
                    ied.EconomicCosts      [j][mblClass] += economicCosts;
                    ied.DiscEconomicCosts  [j][mblClass] += discEconomicCosts;
                    ied.COEmissions        [j][mblClass] += coEmissions;
                    ied.CODamage           [j][mblClass] += coDamage;
                    ied.CODiscDamage       [j][mblClass] += coDiscDamage;
                    ied.VOCEmissions       [j][mblClass] += vocEmissions;
                    ied.VOCDamage          [j][mblClass] += vocDamage;
                    ied.VOCDiscDamage      [j][mblClass] += vocDiscDamage;
                    ied.NOXEmissions       [j][mblClass] += noxEmissions;
                    ied.NOXDamage          [j][mblClass] += noxDamage;
                    ied.NOXDiscDamage      [j][mblClass] += noxDiscDamage;
                    ied.PMEmissions        [j][mblClass] += pmEmissions;
                    ied.PMDamage           [j][mblClass] += pmDamage;
                    ied.PMDiscDamage       [j][mblClass] += pmDiscDamage;
                    ied.SOXEmissions       [j][mblClass] += soxEmissions;
                    ied.SOXDamage          [j][mblClass] += soxDamage;
                    ied.SOXDiscDamage      [j][mblClass] += soxDiscDamage;
                    ied.CO2Emissions       [j][mblClass] += co2Emissions;
                    ied.CO2Damage          [j][mblClass] += co2Damage;
                    ied.CO2DiscDamage      [j][mblClass] += co2DiscDamage;
                    ied.RcGallons             [regClass] += gallons;
                    ied.RcPreTaxFuel          [regClass] += preTaxFuel;
                    ied.RcDiscPreTaxFuel      [regClass] += discPreTaxFuel;
                    ied.RcDriveSurplus        [regClass] += driveSurplus;
                    ied.RcDiscDriveSurplus    [regClass] += discDriveSurplus;
                    ied.RcRefuelSurplus       [regClass] += refuelSurplus;
                    ied.RcDiscRefuelSurplus   [regClass] += discRefuelSurplus;
                    ied.RcCongestionCosts     [regClass] += congestionCosts;
                    ied.RcDiscCongestionCosts [regClass] += discCongestionCosts;
                    ied.RcAccidentCosts       [regClass] += accidentCosts;
                    ied.RcDiscAccidentCosts   [regClass] += discAccidentCosts;
                    ied.RcNoiseCosts          [regClass] += noiseCosts;
                    ied.RcDiscNoiseCosts      [regClass] += discNoiseCosts;
                    ied.RcEconomicCosts       [regClass] += economicCosts;
                    ied.RcDiscEconomicCosts   [regClass] += discEconomicCosts;
                    ied.RcCODamage            [regClass] += coDamage;
                    ied.RcCODiscDamage        [regClass] += coDiscDamage;
                    ied.RcVOCDamage           [regClass] += vocDamage;
                    ied.RcVOCDiscDamage       [regClass] += vocDiscDamage;
                    ied.RcNOXDamage           [regClass] += noxDamage;
                    ied.RcNOXDiscDamage       [regClass] += noxDiscDamage;
                    ied.RcPMDamage            [regClass] += pmDamage;
                    ied.RcPMDiscDamage        [regClass] += pmDiscDamage;
                    ied.RcSOXDamage           [regClass] += soxDamage;
                    ied.RcSOXDiscDamage       [regClass] += soxDiscDamage;
                    ied.RcCO2Damage           [regClass] += co2Damage;
                    ied.RcCO2DiscDamage       [regClass] += co2DiscDamage;
                    mmd.FuelConsumption   [regClass] += gallons;
                    mmd.SocialBenefits    [regClass] += (preTaxFuel + driveSurplus + refuelSurplus +
                        economicCosts + congestionCosts + accidentCosts + noiseCosts +
                        co2Damage + coDamage  + vocDamage + noxDamage + pmDamage + soxDamage);
                    mmd.DiscSocialBenefits[regClass] += (discPreTaxFuel + discDriveSurplus + discRefuelSurplus +
                        discEconomicCosts + discCongestionCosts + discAccidentCosts + discNoiseCosts +
                        co2DiscDamage + coDiscDamage + vocDiscDamage + noxDiscDamage + pmDiscDamage + soxDiscDamage);
                    mmd.CO2Emissions      [regClass] += co2Emissions;
                    mmd.CO2DamageCosts    [regClass] += co2Damage;
                    mmd.CO2DiscDamageCosts[regClass] += co2DiscDamage;
                } 
            } 
        }
        private double CalculateEmission(double domReducedRef, double fuelReducedRef, AdjUE adjEm,
            double gallons, double vmt, double[] rate, int myIdx, int cyIdx)
        {
            return ((fuelReducedRef * (domReducedRef * (adjEm.Extraction + adjEm.Transportation) + adjEm.Refining) + adjEm.TSD) *
                gallons + vmt * rate[Math.Min(cyIdx + myIdx, rate.Length - 1)]) * Effects.EmFract;
        }
        private void CalculateEmissionCosts(double emission, double dmgCost, double discRatePowMy, out double damage,
            out double discDamage, double factor)
        {
            damage = emission * dmgCost * factor;
            discDamage = damage * discRatePowMy;
        }
        private void InitEffectsByFuel(string fuel, out double[] fuelPrice, out double[] fuelTax, out double[] preTaxFuel,
            out double fuelLowImp, out double fuelReducedRef, out double domReducedRef, out double impReducedRef,
            out double massDensity, out double carbonContent, out double soxRate, out AdjUE adjCO, out AdjUE adjVOC,
            out AdjUE adjNOX, out AdjUE adjPM, out AdjUE adjSOX, out AdjUE adjCO2)
        {
            Estimates  fuelPriceEstimates = this._settings.OperatingModes.FuelPriceEstimates;
            Parameters parameters         = this._settings.Parameters;
            switch (fuel)
            {
                case "D":
                    fuelPrice      = parameters.ForecastData.GetDieselFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.DieselFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetDieselPreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.DieselImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.DieselImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.DieselImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.DieselImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.CODieselEmissions;
                    adjVOC         = parameters.VOCDieselEmissions;
                    adjNOX         = parameters.NOXDieselEmissions;
                    adjPM          = parameters.PMDieselEmissions;
                    adjSOX         = parameters.SOXDieselEmissions;
                    adjCO2         = parameters.CO2DieselEmissions;
                    massDensity    = parameters.DieselMassDensity;
                    carbonContent  = parameters.DieselCarbonContent;
                    soxRate        = parameters.DieselSOXRate;
                    break;
                case "E85":
                    fuelPrice      = parameters.ForecastData.GetE85FuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.E85FuelTax;
                    preTaxFuel     = parameters.ForecastData.GetE85PreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.E85ImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.E85ImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.E85ImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.E85ImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COE85Emissions;
                    adjVOC         = parameters.VOCE85Emissions;
                    adjNOX         = parameters.NOXE85Emissions;
                    adjPM          = parameters.PME85Emissions;
                    adjSOX         = parameters.SOXE85Emissions;
                    adjCO2         = parameters.CO2E85Emissions;
                    massDensity    = parameters.E85MassDensity;
                    carbonContent  = parameters.E85CarbonContent;
                    soxRate        = parameters.E85SOXRate;
                    break;
                case "CNG":
                    fuelPrice      = parameters.ForecastData.GetCNGFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.CNGFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetCNGPreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.CNGImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.CNGImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.CNGImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.CNGImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COCNGEmissions;
                    adjVOC         = parameters.VOCCNGEmissions;
                    adjNOX         = parameters.NOXCNGEmissions;
                    adjPM          = parameters.PMCNGEmissions;
                    adjSOX         = parameters.SOXCNGEmissions;
                    adjCO2         = parameters.CO2CNGEmissions;
                    massDensity    = parameters.CNGMassDensity;
                    carbonContent  = parameters.CNGCarbonContent;
                    soxRate        = parameters.CNGSOXRate;
                    break;
                case "H":
                    fuelPrice      = parameters.ForecastData.GetHydrogenFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.HydrogenFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetHydrogenPreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.HydrogenImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.HydrogenImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.HydrogenImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.HydrogenImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COHydrogenEmissions;
                    adjVOC         = parameters.VOCHydrogenEmissions;
                    adjNOX         = parameters.NOXHydrogenEmissions;
                    adjPM          = parameters.PMHydrogenEmissions;
                    adjSOX         = parameters.SOXHydrogenEmissions;
                    adjCO2         = parameters.CO2HydrogenEmissions;
                    massDensity    = parameters.HydrogenMassDensity;
                    carbonContent  = parameters.HydrogenCarbonContent;
                    soxRate        = parameters.HydrogenSOXRate;
                    break;
                case "SP1":
                    fuelPrice      = parameters.ForecastData.GetSpare1FuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.Spare1FuelTax;
                    preTaxFuel     = parameters.ForecastData.GetSpare1PreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.Spare1ImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.Spare1ImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.Spare1ImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.Spare1ImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COSpare1Emissions;
                    adjVOC         = parameters.VOCSpare1Emissions;
                    adjNOX         = parameters.NOXSpare1Emissions;
                    adjPM          = parameters.PMSpare1Emissions;
                    adjSOX         = parameters.SOXSpare1Emissions;
                    adjCO2         = parameters.CO2Spare1Emissions;
                    massDensity    = parameters.Spare1MassDensity;
                    carbonContent  = parameters.Spare1CarbonContent;
                    soxRate        = parameters.Spare1SOXRate;
                    break;
                case "SP2":
                    fuelPrice      = parameters.ForecastData.GetSpare2FuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.Spare2FuelTax;
                    preTaxFuel     = parameters.ForecastData.GetSpare2PreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.Spare2ImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.Spare2ImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.Spare2ImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.Spare2ImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COSpare2Emissions;
                    adjVOC         = parameters.VOCSpare2Emissions;
                    adjNOX         = parameters.NOXSpare2Emissions;
                    adjPM          = parameters.PMSpare2Emissions;
                    adjSOX         = parameters.SOXSpare2Emissions;
                    adjCO2         = parameters.CO2Spare2Emissions;
                    massDensity    = parameters.Spare2MassDensity;
                    carbonContent  = parameters.Spare2CarbonContent;
                    soxRate        = parameters.Spare2SOXRate;
                    break;
                default:
                    fuelPrice      = parameters.ForecastData.GetGasolineFuelPrice(fuelPriceEstimates);
                    fuelTax        = parameters.ForecastData.GasolineFuelTax;
                    preTaxFuel     = parameters.ForecastData.GetGasolinePreTaxFuelPrice(fuelPriceEstimates);
                    fuelLowImp     = parameters.GasolineImportAssumptions.FuelSavingsLeadingToLowerImports;
                    fuelReducedRef = parameters.GasolineImportAssumptions.FuelSavingsLeadingToReducedRefining;
                    domReducedRef  = parameters.GasolineImportAssumptions.ReducedRefiningFromDomesticCrude;
                    impReducedRef  = parameters.GasolineImportAssumptions.ReducedRefiningFromImportedCrude;
                    adjCO          = parameters.COGasolineEmissions;
                    adjVOC         = parameters.VOCGasolineEmissions;
                    adjNOX         = parameters.NOXGasolineEmissions;
                    adjPM          = parameters.PMGasolineEmissions;
                    adjSOX         = parameters.SOXGasolineEmissions;
                    adjCO2         = parameters.CO2GasolineEmissions;
                    massDensity    = parameters.GasolineMassDensity;
                    carbonContent  = parameters.GasolineCarbonContent;
                    soxRate        = parameters.GasolineSOXRate;
                    break;
            }
        }
        private double[] GetEmissionsRates(EmissionsRates.Emissions em, Mobile6Class mblClass, double convGasMix, int myIdx)
        {
            int ubound = em.Diesel.Ldv.GetUpperBound(0) + 1;
            double[,] emRate = null;
            double[,] gasRfg = null;
            double[]  rate   = new double[ubound];
            switch (mblClass)
            {
                case Mobile6Class.Ldgv:   emRate = em.Gasoline.Ldv;   gasRfg = em.GasolineRfg.Ldv;   break;
                case Mobile6Class.Ldgt1:  emRate = em.Gasoline.Ldt1;  gasRfg = em.GasolineRfg.Ldt1;  break;
                case Mobile6Class.Ldgt2:  emRate = em.Gasoline.Ldt2;  gasRfg = em.GasolineRfg.Ldt2;  break;
                case Mobile6Class.Hdgv2b: emRate = em.Gasoline.Hdv2b; gasRfg = em.GasolineRfg.Hdv2b; break;
                case Mobile6Class.Lddv:   emRate = em.Diesel.Ldv;   break;
                case Mobile6Class.Lddt1:  emRate = em.Diesel.Ldt1;  break;
                case Mobile6Class.Lddt2:  emRate = em.Diesel.Ldt2;  break;
                case Mobile6Class.Hddv2b: emRate = em.Diesel.Hdv2b; break;
            }
            if (gasRfg != null)
            {   
                for (int i = 0; i < ubound; i++)
                {
                    rate[i] = emRate[myIdx, i] * convGasMix + gasRfg[myIdx, i] * (1 - convGasMix);
                }
            }
            else if (emRate != null)
            {   
                for (int i = 0; i < ubound; i++)
                {
                    rate[i] = emRate[myIdx, i];
                }
            }
            return rate;
        }
        private void CalculateAggregatedEffects()
        {
            AggregateEffectsData ied = this._data.ModelingData.EffectsData; 
            ied.Reset();
            ManufacturerCollection mfrs = this._data.Manufacturers;
            for (int i = 0; i < this._mfrCount; i++)
            {
                Manufacturer mfr = mfrs[i];
                ManufacturerModelingData mmd = mfr.ModelingData;            
                VehicleCollection vc = mfr.ModelYearVehicles;
                mmd.FuelConsumption   .Clear();
                mmd.SocialBenefits    .Clear();
                mmd.DiscSocialBenefits.Clear();
                mmd.CO2Emissions      .Clear();
                mmd.CO2DamageCosts    .Clear();
                mmd.CO2DiscDamageCosts.Clear();
                this.CalculateVehicleEffects(mfr.Index, ied, mmd, vc);
            } 
            ied.RcSocialBenefits = ied.RcPreTaxFuel + ied.RcDriveSurplus + ied.RcRefuelSurplus +
                ied.RcEconomicCosts + ied.RcCongestionCosts + ied.RcAccidentCosts + ied.RcNoiseCosts +
                ied.RcCO2Damage + ied.RcCODamage + ied.RcVOCDamage + ied.RcNOXDamage + ied.RcPMDamage + ied.RcSOXDamage;
            ied.RcDiscSocialBenefits = ied.RcDiscPreTaxFuel + ied.RcDiscDriveSurplus + ied.RcDiscRefuelSurplus +
                ied.RcDiscEconomicCosts + ied.RcDiscCongestionCosts + ied.RcDiscAccidentCosts + ied.RcDiscNoiseCosts +
                ied.RcCO2DiscDamage + ied.RcCODiscDamage + ied.RcVOCDiscDamage + ied.RcNOXDiscDamage + ied.RcPMDiscDamage +
                ied.RcSOXDiscDamage;
            for (int i = 1; i <= 8; i++)
            {
                Mobile6Class mblClass = (Mobile6Class)i;
                double fe = ied.FuelEconomy[mblClass];
                ied.FuelEconomy[mblClass] = (fe == 0) ? 0 : ied.Sales[mblClass] / fe;
            }
        }
        #endregion
        #region 
        private const double EmFract = 1e6 / 908e6;
        private const double OneTwelfth = 1D / 12D;
        private const double FourtyFourTwelfths = 44D / 12D;
        Industry         _data;
        ModelingSettings _settings;
        bool   _isBaseline;
        int    _myIdx, _mYear;
        int    _mfrCount;
        double _dmgCostCO, _dmgCostVOC, _dmgCostNOX, _dmgCostPM, _dmgCostSOX;
        double _testVsOnRoadGap, _reboundEffect, _discountRate, _vehicleTravelTimeValue;
        double _vmtBaseYear, _vmtCarGrowth, _vmtTruckGrowth;
        double _carCongestion, _carAccident, _carNoise,
               _ltCongestion , _ltAccident , _ltNoise;
        double _monopsonyCost, _priceShockCost, _militarySecurityCost, _convGasMix;
        #endregion
    }
}

