using System;
using System.Drawing;
using System.Runtime.Serialization;
using Volpe.Cafe.Data;
using Volpe.Cafe.IO;
using Volpe.Cafe.Settings;
using Volpe.Utils;
namespace Volpe.Cafe
{
    [Serializable]
    public class Parameters : ICloneable
    {
        #region 
        [Serializable]
        public struct VehicleSurvivability : ICloneable
        {
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public VehicleSurvivability Clone()
            {
                VehicleSurvivability vs = new VehicleSurvivability();
                int length = this.Ldgv.Length;
                vs.Initialize(length);
                for (int i = 0; i < length; i++)
                {
                    vs.Ldgv  [i] = this.Ldgv[i];
                    vs.Lddv  [i] = this.Lddv[i];
                    vs.Ldgt1 [i] = this.Ldgt1[i];
                    vs.Lddt1 [i] = this.Lddt1[i];
                    vs.Ldgt2 [i] = this.Ldgt2[i];
                    vs.Lddt2 [i] = this.Lddt2[i];
                    vs.Hdgv2b[i] = this.Hdgv2b[i];
                    vs.Hddv2b[i] = this.Hddv2b[i];
                }
                return vs;
            }
            #endregion
            public void Initialize(int length)
            {
                this.Ldgv   = new double[length];
                this.Lddv   = new double[length];
                this.Ldgt1  = new double[length];
                this.Lddt1  = new double[length];
                this.Ldgt2  = new double[length];
                this.Lddt2  = new double[length];
                this.Hdgv2b = new double[length];
                this.Hddv2b = new double[length];
            }
            public double[] Ldgv;
            public double[] Lddv;
            public double[] Ldgt1;
            public double[] Lddt1;
            public double[] Ldgt2;
            public double[] Lddt2;
            public double[] Hdgv2b;
            public double[] Hddv2b;
        }
        [Serializable]
        public struct MilesDriven : ICloneable
        {
            #region 
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public MilesDriven Clone()
            {
                MilesDriven md = new MilesDriven();
                int length = this.Van.Length;
                md.Initialize(length);
                Array.Copy(this.Automobile    , md.Automobile    , length);     
                Array.Copy(this.LowAutomobile , md.LowAutomobile , length);
                Array.Copy(this.HighAutomobile, md.HighAutomobile, length);
                Array.Copy(this.Pickup        , md.Pickup        , length);     
                Array.Copy(this.LowPickup     , md.LowPickup     , length);
                Array.Copy(this.HighPickup    , md.HighPickup    , length);
                Array.Copy(this.Van           , md.Van           , length);     
                Array.Copy(this.LowVan        , md.LowVan        , length);
                Array.Copy(this.HighVan       , md.HighVan       , length);
                Array.Copy(this.Suv           , md.Suv           , length);     
                Array.Copy(this.LowSuv        , md.LowSuv        , length);
                Array.Copy(this.HighSuv       , md.HighSuv       , length);
                Array.Copy(this.LtDefault     , md.LtDefault     , length);     
                return md;
            }
            #endregion
            public double[] GetAutomobileMilesDriven(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowAutomobile :
                    (priceEstimates == Estimates.High) ? this.HighAutomobile : this.Automobile;
            }
            public double[] GetPickupMilesDriven(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowPickup :
                    (priceEstimates == Estimates.High) ? this.HighPickup : this.Pickup;
            }
            public double[] GetVanMilesDriven(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowVan :
                    (priceEstimates == Estimates.High) ? this.HighVan : this.Van;
            }
            public double[] GetSuvMilesDriven(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowSuv :
                    (priceEstimates == Estimates.High) ? this.HighSuv : this.Suv;
            }
            public void Initialize(int length)
            {
                this.Automobile     = new double[length];   
                this.LowAutomobile  = new double[length];
                this.HighAutomobile = new double[length];
                this.Pickup         = new double[length];   
                this.LowPickup      = new double[length];
                this.HighPickup     = new double[length];
                this.Van            = new double[length];   
                this.LowVan         = new double[length];
                this.HighVan        = new double[length];
                this.Suv            = new double[length];   
                this.LowSuv         = new double[length];
                this.HighSuv        = new double[length];
                this.LtDefault      = new double[length];   
            }
            #endregion
            #region 
            internal double[] Automobile;
            internal double[] LowAutomobile;
            internal double[] HighAutomobile;
            internal double[] Pickup;
            internal double[] LowPickup;
            internal double[] HighPickup;
            internal double[] Van;
            internal double[] LowVan;
            internal double[] HighVan;
            internal double[] Suv;
            internal double[] LowSuv;
            internal double[] HighSuv;
            internal double[] LtDefault;
            #endregion
        }
        [Serializable]
        public struct ForecastDataValues : ICloneable
        {
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public ForecastDataValues Clone()
            {
                ForecastDataValues fd = new ForecastDataValues();
                int length = this.GasolineFuelPrice.Length;
                fd.Initialize(length);
                for (int i = 0; i < length; i++)
                {
                    fd.GasolineFuelPrice          [i] = this.GasolineFuelPrice[i];
                    fd.LowGasolineFuelPrice       [i] = this.LowGasolineFuelPrice[i];
                    fd.HighGasolineFuelPrice      [i] = this.HighGasolineFuelPrice[i];
                    fd.DieselFuelPrice            [i] = this.DieselFuelPrice[i];
                    fd.LowDieselFuelPrice         [i] = this.LowDieselFuelPrice[i];
                    fd.HighDieselFuelPrice        [i] = this.HighDieselFuelPrice[i];
                    fd.E85FuelPrice               [i] = this.E85FuelPrice[i];
                    fd.LowE85FuelPrice            [i] = this.LowE85FuelPrice[i];
                    fd.HighE85FuelPrice           [i] = this.HighE85FuelPrice[i];
                    fd.CNGFuelPrice               [i] = this.CNGFuelPrice[i];
                    fd.LowCNGFuelPrice            [i] = this.LowCNGFuelPrice[i];
                    fd.HighCNGFuelPrice           [i] = this.HighCNGFuelPrice[i];
                    fd.HydrogenFuelPrice          [i] = this.HydrogenFuelPrice[i];
                    fd.LowHydrogenFuelPrice       [i] = this.LowHydrogenFuelPrice[i];
                    fd.HighHydrogenFuelPrice      [i] = this.HighHydrogenFuelPrice[i];
                    fd.Spare1FuelPrice            [i] = this.Spare1FuelPrice[i];
                    fd.LowSpare1FuelPrice         [i] = this.LowSpare1FuelPrice[i];
                    fd.HighSpare1FuelPrice        [i] = this.HighSpare1FuelPrice[i];
                    fd.Spare2FuelPrice            [i] = this.Spare2FuelPrice[i];
                    fd.LowSpare2FuelPrice         [i] = this.LowSpare2FuelPrice[i];
                    fd.HighSpare2FuelPrice        [i] = this.HighSpare2FuelPrice[i];
                    fd.GasolineFuelTax            [i] = this.GasolineFuelTax[i];
                    fd.DieselFuelTax              [i] = this.DieselFuelTax[i];
                    fd.E85FuelTax                 [i] = this.E85FuelTax[i];
                    fd.CNGFuelTax                 [i] = this.CNGFuelTax[i];
                    fd.HydrogenFuelTax            [i] = this.HydrogenFuelTax[i];
                    fd.Spare1FuelTax              [i] = this.Spare1FuelTax[i];
                    fd.Spare2FuelTax              [i] = this.Spare2FuelTax[i];
                    fd.GasolinePreTaxFuelPrice    [i] = this.GasolinePreTaxFuelPrice[i];
                    fd.LowGasolinePreTaxFuelPrice [i] = this.LowGasolinePreTaxFuelPrice[i];
                    fd.HighGasolinePreTaxFuelPrice[i] = this.HighGasolinePreTaxFuelPrice[i];
                    fd.DieselPreTaxFuelPrice      [i] = this.DieselPreTaxFuelPrice[i];
                    fd.LowDieselPreTaxFuelPrice   [i] = this.LowDieselPreTaxFuelPrice[i];
                    fd.HighDieselPreTaxFuelPrice  [i] = this.HighDieselPreTaxFuelPrice[i];
                    fd.E85PreTaxFuelPrice         [i] = this.E85PreTaxFuelPrice[i];
                    fd.LowE85PreTaxFuelPrice      [i] = this.LowE85PreTaxFuelPrice[i];
                    fd.HighE85PreTaxFuelPrice     [i] = this.HighE85PreTaxFuelPrice[i];
                    fd.CNGPreTaxFuelPrice         [i] = this.CNGPreTaxFuelPrice[i];
                    fd.LowCNGPreTaxFuelPrice      [i] = this.LowCNGPreTaxFuelPrice[i];
                    fd.HighCNGPreTaxFuelPrice     [i] = this.HighCNGPreTaxFuelPrice[i];
                    fd.HydrogenPreTaxFuelPrice    [i] = this.HydrogenPreTaxFuelPrice[i];
                    fd.LowHydrogenPreTaxFuelPrice [i] = this.LowHydrogenPreTaxFuelPrice[i];
                    fd.HighHydrogenPreTaxFuelPrice[i] = this.HighHydrogenPreTaxFuelPrice[i];
                    fd.Spare1PreTaxFuelPrice      [i] = this.Spare1PreTaxFuelPrice[i];
                    fd.LowSpare1PreTaxFuelPrice   [i] = this.LowSpare1PreTaxFuelPrice[i];
                    fd.HighSpare1PreTaxFuelPrice  [i] = this.HighSpare1PreTaxFuelPrice[i];
                    fd.Spare2PreTaxFuelPrice      [i] = this.Spare2PreTaxFuelPrice[i];
                    fd.LowSpare2PreTaxFuelPrice   [i] = this.LowSpare2PreTaxFuelPrice[i];
                    fd.HighSpare2PreTaxFuelPrice  [i] = this.HighSpare2PreTaxFuelPrice[i];
                    fd.E85Share                   [i] = this.E85Share[i];
                }
                return fd;
            }
            #endregion
            public double[] GetFuelPrice(string fuel, Estimates priceEstimates)
            {
                switch (fuel)
                {
                    case "D"  : return this.GetDieselFuelPrice  (priceEstimates);
                    case "E85": return this.GetE85FuelPrice     (priceEstimates);
                    case "CNG": return this.GetCNGFuelPrice     (priceEstimates);
                    case "H"  : return this.GetHydrogenFuelPrice(priceEstimates);
                    case "SP1": return this.GetSpare1FuelPrice  (priceEstimates);
                    case "SP2": return this.GetSpare2FuelPrice  (priceEstimates);
                    default   : return this.GetGasolineFuelPrice(priceEstimates);
                }
            }
            public double[] GetPreTaxFuelPrice(string fuel, Estimates priceEstimates)
            {
                switch (fuel)
                {
                    case "D"  : return this.GetDieselPreTaxFuelPrice  (priceEstimates);
                    case "E85": return this.GetE85PreTaxFuelPrice     (priceEstimates);
                    case "CNG": return this.GetCNGPreTaxFuelPrice     (priceEstimates);
                    case "H"  : return this.GetHydrogenPreTaxFuelPrice(priceEstimates);
                    case "SP1": return this.GetSpare1PreTaxFuelPrice  (priceEstimates);
                    case "SP2": return this.GetSpare2PreTaxFuelPrice  (priceEstimates);
                    default   : return this.GetGasolinePreTaxFuelPrice(priceEstimates);
                }
            }
            public double[] GetGasolineFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowGasolineFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighGasolineFuelPrice : this.GasolineFuelPrice;
            }
            public double[] GetDieselFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowDieselFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighDieselFuelPrice : this.DieselFuelPrice;
            }
            public double[] GetE85FuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowE85FuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighE85FuelPrice : this.E85FuelPrice;
            }
            public double[] GetCNGFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowCNGFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighCNGFuelPrice : this.CNGFuelPrice;
            }
            public double[] GetHydrogenFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowHydrogenFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighHydrogenFuelPrice : this.HydrogenFuelPrice;
            }
            public double[] GetSpare1FuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowSpare1FuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighSpare1FuelPrice : this.Spare1FuelPrice;
            }
            public double[] GetSpare2FuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowSpare2FuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighSpare2FuelPrice : this.Spare2FuelPrice;
            }
            public double[] GetGasolinePreTaxFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowGasolinePreTaxFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighGasolinePreTaxFuelPrice : this.GasolinePreTaxFuelPrice;
            }
            public double[] GetDieselPreTaxFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowDieselPreTaxFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighDieselPreTaxFuelPrice : this.DieselPreTaxFuelPrice;
            }
            public double[] GetE85PreTaxFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowE85PreTaxFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighE85PreTaxFuelPrice : this.E85PreTaxFuelPrice;
            }
            public double[] GetCNGPreTaxFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowCNGPreTaxFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighCNGPreTaxFuelPrice : this.CNGPreTaxFuelPrice;
            }
            public double[] GetHydrogenPreTaxFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowHydrogenPreTaxFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighHydrogenPreTaxFuelPrice : this.HydrogenPreTaxFuelPrice;
            }
            public double[] GetSpare1PreTaxFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowSpare1PreTaxFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighSpare1PreTaxFuelPrice : this.Spare1PreTaxFuelPrice;
            }
            public double[] GetSpare2PreTaxFuelPrice(Estimates priceEstimates)
            {
                return (priceEstimates == Estimates.Low) ? this.LowSpare2PreTaxFuelPrice :
                    (priceEstimates == Estimates.High) ? this.HighSpare2PreTaxFuelPrice : this.Spare2PreTaxFuelPrice;
            }
            public void Initialize(int length)
            {
                this.GasolineFuelPrice           = new double[length];
                this.LowGasolineFuelPrice        = new double[length];
                this.HighGasolineFuelPrice       = new double[length];
                this.DieselFuelPrice             = new double[length];
                this.LowDieselFuelPrice          = new double[length];
                this.HighDieselFuelPrice         = new double[length];
                this.E85FuelPrice                = new double[length];
                this.LowE85FuelPrice             = new double[length];
                this.HighE85FuelPrice            = new double[length];
                this.CNGFuelPrice                = new double[length];
                this.LowCNGFuelPrice             = new double[length];
                this.HighCNGFuelPrice            = new double[length];
                this.HydrogenFuelPrice           = new double[length];
                this.LowHydrogenFuelPrice        = new double[length];
                this.HighHydrogenFuelPrice       = new double[length];
                this.Spare1FuelPrice             = new double[length];
                this.LowSpare1FuelPrice          = new double[length];
                this.HighSpare1FuelPrice         = new double[length];
                this.Spare2FuelPrice             = new double[length];
                this.LowSpare2FuelPrice          = new double[length];
                this.HighSpare2FuelPrice         = new double[length];
                this.GasolineFuelTax             = new double[length];
                this.DieselFuelTax               = new double[length];
                this.E85FuelTax                  = new double[length];
                this.CNGFuelTax                  = new double[length];
                this.HydrogenFuelTax             = new double[length];
                this.Spare1FuelTax               = new double[length];
                this.Spare2FuelTax               = new double[length];
                this.GasolinePreTaxFuelPrice     = new double[length];
                this.LowGasolinePreTaxFuelPrice  = new double[length];
                this.HighGasolinePreTaxFuelPrice = new double[length];
                this.DieselPreTaxFuelPrice       = new double[length];
                this.LowDieselPreTaxFuelPrice    = new double[length];
                this.HighDieselPreTaxFuelPrice   = new double[length];
                this.E85PreTaxFuelPrice          = new double[length];
                this.LowE85PreTaxFuelPrice       = new double[length];
                this.HighE85PreTaxFuelPrice      = new double[length];
                this.CNGPreTaxFuelPrice          = new double[length];
                this.LowCNGPreTaxFuelPrice       = new double[length];
                this.HighCNGPreTaxFuelPrice      = new double[length];
                this.HydrogenPreTaxFuelPrice     = new double[length];
                this.LowHydrogenPreTaxFuelPrice  = new double[length];
                this.HighHydrogenPreTaxFuelPrice = new double[length];
                this.Spare1PreTaxFuelPrice       = new double[length];
                this.LowSpare1PreTaxFuelPrice    = new double[length];
                this.HighSpare1PreTaxFuelPrice   = new double[length];
                this.Spare2PreTaxFuelPrice       = new double[length];
                this.LowSpare2PreTaxFuelPrice    = new double[length];
                this.HighSpare2PreTaxFuelPrice   = new double[length];
                this.E85Share                    = new double[length];
            }
            internal double[] GasolineFuelPrice;
            internal double[] LowGasolineFuelPrice;
            internal double[] HighGasolineFuelPrice;
            internal double[] DieselFuelPrice;
            internal double[] LowDieselFuelPrice;
            internal double[] HighDieselFuelPrice;
            internal double[] E85FuelPrice;
            internal double[] LowE85FuelPrice;
            internal double[] HighE85FuelPrice;
            internal double[] CNGFuelPrice;
            internal double[] LowCNGFuelPrice;
            internal double[] HighCNGFuelPrice;
            internal double[] HydrogenFuelPrice;
            internal double[] LowHydrogenFuelPrice;
            internal double[] HighHydrogenFuelPrice;
            internal double[] Spare1FuelPrice;
            internal double[] LowSpare1FuelPrice;
            internal double[] HighSpare1FuelPrice;
            internal double[] Spare2FuelPrice;
            internal double[] LowSpare2FuelPrice;
            internal double[] HighSpare2FuelPrice;
            public double[] GasolineFuelTax;
            public double[] DieselFuelTax;
            public double[] E85FuelTax;
            public double[] CNGFuelTax;
            public double[] HydrogenFuelTax;
            public double[] Spare1FuelTax;
            public double[] Spare2FuelTax;
            internal double[] GasolinePreTaxFuelPrice;
            internal double[] LowGasolinePreTaxFuelPrice;
            internal double[] HighGasolinePreTaxFuelPrice;
            internal double[] DieselPreTaxFuelPrice;
            internal double[] LowDieselPreTaxFuelPrice;
            internal double[] HighDieselPreTaxFuelPrice;
            internal double[] E85PreTaxFuelPrice;
            internal double[] LowE85PreTaxFuelPrice;
            internal double[] HighE85PreTaxFuelPrice;
            internal double[] CNGPreTaxFuelPrice;
            internal double[] LowCNGPreTaxFuelPrice;
            internal double[] HighCNGPreTaxFuelPrice;
            internal double[] HydrogenPreTaxFuelPrice;
            internal double[] LowHydrogenPreTaxFuelPrice;
            internal double[] HighHydrogenPreTaxFuelPrice;
            internal double[] Spare1PreTaxFuelPrice;
            internal double[] LowSpare1PreTaxFuelPrice;
            internal double[] HighSpare1PreTaxFuelPrice;
            internal double[] Spare2PreTaxFuelPrice;
            internal double[] LowSpare2PreTaxFuelPrice;
            internal double[] HighSpare2PreTaxFuelPrice;
            internal double[] E85Share;
        }
        [Serializable]
        public struct EconomicDataValues : ISerializable
        {
            #region 
            EconomicDataValues(SerializationInfo info, StreamingContext context)
            {
                this.ReboundEffect          = info.GetDouble("ReboundEffect");
                this.DiscountRate           = info.GetDouble("DiscountRate");
                this.PaybackPeriod          = info.GetDouble("PaybackPeriod");
                this.Kf                     = info.GetDouble("Kf");
                this.TestVsOnRoadGap        = info.GetDouble("TestVsOnRoadGap");
                this.VehicleTravelTimeValue = info.GetDouble("VehicleTravelTimeValue");
                try
                {
                    this.VMTBaseYear        = info.GetInt32 ("VMTBaseYear");
                    this.VMTCarGrowthRate   = info.GetDouble("VMTCarGrowthRate");
                    this.VMTTruckGrowthRate = info.GetDouble("VMTTruckGrowthRate");
                }
                catch
                {
                    this.VMTBaseYear        = 0;
                    this.VMTCarGrowthRate   = 0;
                    this.VMTTruckGrowthRate = 0;
                }
            }
            #endregion
            #region 
            #region 
            public void GetObjectData(SerializationInfo info, StreamingContext context)
            {
                info.AddValue("ReboundEffect"         , this.ReboundEffect);
                info.AddValue("DiscountRate"          , this.DiscountRate);
                info.AddValue("PaybackPeriod"         , this.PaybackPeriod);
                info.AddValue("Kf"                    , this.Kf);
                info.AddValue("TestVsOnRoadGap"       , this.TestVsOnRoadGap);
                info.AddValue("VehicleTravelTimeValue", this.VehicleTravelTimeValue);
                info.AddValue("VMTBaseYear"           , this.VMTBaseYear);
                info.AddValue("VMTCarGrowthRate"      , this.VMTCarGrowthRate);
                info.AddValue("VMTTruckGrowthRate"    , this.VMTTruckGrowthRate);
            }
            #endregion
            #endregion
            #region 
            public double ReboundEffect;
            public double DiscountRate;
            public double PaybackPeriod;
            public double Kf;
            public double TestVsOnRoadGap;
            public double VehicleTravelTimeValue;
            public int VMTBaseYear;
            public double VMTCarGrowthRate;
            public double VMTTruckGrowthRate;
            #endregion
        }
        [Serializable]
        public struct EconomicCostsValues
        {
            public double Monopsony;
            public double PriceShock;
            public double MilitarySecurity;
            public double Total;
            public double TotalBBL;
        }
        [Serializable]
        public struct ExternalCostsValues
        {
            public double CarCongestion;
            public double CarAccident;
            public double CarNoise;
            public double TruckCongestion;
            public double TruckAccident;
            public double TruckNoise;
        }
        [Serializable]
        public struct EmissionDamageCosts : ICloneable, ISerializable
        {
            #region 
            EmissionDamageCosts(SerializationInfo info, StreamingContext context)
            {
                this.CO  = info.GetDouble("CO" );
                this.VOC = info.GetDouble("VOC");
                this.NOX = info.GetDouble("NOX");
                this.PM  = info.GetDouble("PM" );
                this.SOX = info.GetDouble("SOX");
                try
                {
                    this.CO2Low      = (double[])info.GetValue("CO2Low"     , typeof(double[]));
                    this.CO2Average  = (double[])info.GetValue("CO2Average" , typeof(double[]));
                    this.CO2High     = (double[])info.GetValue("CO2High"    , typeof(double[]));
                    this.CO2VeryHigh = (double[])info.GetValue("CO2VeryHigh", typeof(double[]));
                }
                catch
                {   
                    double co2        = info.GetDouble("CO2");
                    double co2Pct     = info.GetDouble("CO2Pct");
                    int    co2RefYear = info.GetInt32 ("CO2RefYear");
                    int yrCount = 2100 - ModelYear.MinYear;
                    this.CO2Low      = new double[yrCount];
                    this.CO2Average  = new double[yrCount];
                    this.CO2High     = new double[yrCount];
                    this.CO2VeryHigh = new double[yrCount];
                    for (int i = 0; i < yrCount; i++)
                    {   
                        double co2_i = co2 * Math.Pow(1 + co2Pct, ModelYear.MinYear + i - co2RefYear);
                        this.CO2Low     [i] = co2_i;
                        this.CO2Average [i] = co2_i;
                        this.CO2High    [i] = co2_i;
                        this.CO2VeryHigh[i] = co2_i;
                    }
                }
                try
                {
                    this.CO2DiscRateLow      = info.GetDouble("CO2DiscRateLow");
                    this.CO2DiscRateAverage  = info.GetDouble("CO2DiscRateAverage");
                    this.CO2DiscRateHigh     = info.GetDouble("CO2DiscRateHigh");
                    this.CO2DiscRateVeryHigh = info.GetDouble("CO2DiscRateVeryHigh");
                }
                catch
                {
                    this.CO2DiscRateLow      = 0;
                    this.CO2DiscRateAverage  = 0;
                    this.CO2DiscRateHigh     = 0;
                    this.CO2DiscRateVeryHigh = 0;
                }
            }
            #endregion
            #region 
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public EmissionDamageCosts Clone()
            {
                EmissionDamageCosts edc = new EmissionDamageCosts();
                edc.CO  = this.CO;
                edc.VOC = this.VOC;
                edc.NOX = this.NOX;
                edc.PM  = this.PM;
                edc.SOX = this.SOX;
                edc.CO2Low      = (double[])Global.CloneArray(this.CO2Low     , typeof(double));
                edc.CO2Average  = (double[])Global.CloneArray(this.CO2Average , typeof(double));
                edc.CO2High     = (double[])Global.CloneArray(this.CO2High    , typeof(double));
                edc.CO2VeryHigh = (double[])Global.CloneArray(this.CO2VeryHigh, typeof(double));
                edc.CO2DiscRateLow      = this.CO2DiscRateLow;
                edc.CO2DiscRateAverage  = this.CO2DiscRateAverage;
                edc.CO2DiscRateHigh     = this.CO2DiscRateHigh;
                edc.CO2DiscRateVeryHigh = this.CO2DiscRateVeryHigh;
                return edc;
            }
            #endregion
            #region 
            public void GetObjectData(SerializationInfo info, StreamingContext context)
            {
                info.AddValue("CO" , this.CO);
                info.AddValue("VOC", this.VOC);
                info.AddValue("NOX", this.NOX);
                info.AddValue("PM" , this.PM);
                info.AddValue("SOX", this.SOX);
                info.AddValue("CO2Low"     , this.CO2Low);
                info.AddValue("CO2Average" , this.CO2Average);
                info.AddValue("CO2High"    , this.CO2High);
                info.AddValue("CO2VeryHigh", this.CO2VeryHigh);
                info.AddValue("CO2DiscRateLow"     , this.CO2DiscRateLow);
                info.AddValue("CO2DiscRateAverage" , this.CO2DiscRateAverage);
                info.AddValue("CO2DiscRateHigh"    , this.CO2DiscRateHigh);
                info.AddValue("CO2DiscRateVeryHigh", this.CO2DiscRateVeryHigh);
            }
            #endregion
            public void GetCO2Estimates(CO2Estimates co2Estimates, out double[] co2Cost, out double co2DiscRate)
            {
                co2Cost =
                    (co2Estimates == CO2Estimates.Low     ) ? this.CO2Low :
                    (co2Estimates == CO2Estimates.High    ) ? this.CO2High :
                    (co2Estimates == CO2Estimates.VeryHigh) ? this.CO2VeryHigh : this.CO2Average;
                co2DiscRate =
                    (co2Estimates == CO2Estimates.Low     ) ? this.CO2DiscRateLow :
                    (co2Estimates == CO2Estimates.High    ) ? this.CO2DiscRateHigh :
                    (co2Estimates == CO2Estimates.VeryHigh) ? this.CO2DiscRateVeryHigh : this.CO2DiscRateAverage;
            }
            #endregion
            #region 
            public double CO;
            public double VOC;
            public double NOX;
            public double PM;
            public double SOX;
            public double[] CO2Low;
            public double[] CO2Average;
            public double[] CO2High;
            public double[] CO2VeryHigh;
            public double CO2DiscRateLow;
            public double CO2DiscRateAverage;
            public double CO2DiscRateHigh;
            public double CO2DiscRateVeryHigh;
            #endregion
        }
        [Serializable]
        public struct FuelTypes
        {
            public double GasolineConventional;
            public double GasolineFederal;
            public double GasolineCalifornia;
            public double DieselConventional;
            public double DieselLowSulfur;
            public double E85;
            public double CNG;
            public double Hydrogen;
            public double Spare1;
            public double Spare2;
        }
        [Serializable]
        public struct TotalUpstreamEmissions
        {
            public FuelTypes Extraction;
            public FuelTypes Transportation;
            public FuelTypes Refining;
            public FuelTypes TSD;
        }
        [Serializable]
        public struct AdjustedUpstreamEmissions
        {
            public double Extraction;
            public double Transportation;
            public double Refining;
            public double TSD;
        }
        [Serializable]
        public struct FuelImportAssumptions
        {
            public double FuelSavingsLeadingToLowerImports;
            public double FuelSavingsLeadingToReducedRefining;
            public double ReducedRefiningFromDomesticCrude;
            public double ReducedRefiningFromImportedCrude;
        }
        [Serializable]
        public struct MonteCarloValues : ICloneable, ISerializable
        {
            #region 
            MonteCarloValues(SerializationInfo info, StreamingContext context)
            {
                this.DiscountRates                          = (double[])info.GetValue("DiscountRates", typeof(double[]));
                this.FuelPathLow                            = info.GetDouble("FuelPathLow");
                this.FuelPathReference                      = info.GetDouble("FuelPathReference");
                this.FuelPathHigh                           = info.GetDouble("FuelPathHigh");
                this.ReboundAlphaShape                      = info.GetDouble("ReboundAlphaShape");
                this.ReboundBetaShape                       = info.GetDouble("ReboundBetaShape");
                this.ReboundScale                           = info.GetDouble("ReboundScale");
                this.ReboundBase                            = info.GetDouble("ReboundBase");
                this.MonopsonyMean                          = info.GetDouble("MonopsonyMean");
                this.MonopsonyStandardDeviation             = info.GetDouble("MonopsonyStandardDeviation");
                this.PriceShockMean                         = info.GetDouble("PriceShockMean");
                this.PriceShockStandardDeviation            = info.GetDouble("PriceShockStandardDeviation");
                this.MilitarySecurityAlternativeCost        = info.GetDouble("MilitarySecurityAlternativeCost");
                this.MilitarySecurityAlternativeProbability = info.GetDouble("MilitarySecurityAlternativeProbability");
                try
                {
                    this.CO2PathLow      = info.GetDouble("CO2PathLow");
                    this.CO2PathAverage  = info.GetDouble("CO2PathAverage");
                    this.CO2PathHigh     = info.GetDouble("CO2PathHigh");
                    this.CO2PathVeryHigh = info.GetDouble("CO2PathVeryHigh");
                }
                catch
                {
                    this.CO2PathLow      = 0.25D;
                    this.CO2PathAverage  = 0.50D;
                    this.CO2PathHigh     = 0.20D;
                    this.CO2PathVeryHigh = 0.05D;
                }
            }
            #endregion
            #region 
            #region 
            object ICloneable.Clone()
            {
                return this.Clone();
            }
            public MonteCarloValues Clone()
            {
                MonteCarloValues mcv                       = new MonteCarloValues();
                mcv.DiscountRates                          = (double[])Global.CloneArray(this.DiscountRates, typeof(double));
                mcv.FuelPathLow                            = this.FuelPathLow;
                mcv.FuelPathReference                      = this.FuelPathReference;
                mcv.FuelPathHigh                           = this.FuelPathHigh;
                mcv.ReboundAlphaShape                      = this.ReboundAlphaShape;
                mcv.ReboundBetaShape                       = this.ReboundBetaShape;
                mcv.ReboundScale                           = this.ReboundScale;
                mcv.ReboundBase                            = this.ReboundBase;
                mcv.CO2PathLow                             = this.CO2PathLow;
                mcv.CO2PathAverage                         = this.CO2PathAverage;
                mcv.CO2PathHigh                            = this.CO2PathHigh;
                mcv.CO2PathVeryHigh                        = this.CO2PathVeryHigh;
                mcv.MonopsonyMean                          = this.MonopsonyMean;
                mcv.MonopsonyStandardDeviation             = this.MonopsonyStandardDeviation;
                mcv.PriceShockMean                         = this.PriceShockMean;
                mcv.PriceShockStandardDeviation            = this.PriceShockStandardDeviation;
                mcv.MilitarySecurityAlternativeCost        = this.MilitarySecurityAlternativeCost;
                mcv.MilitarySecurityAlternativeProbability = this.MilitarySecurityAlternativeProbability;
                return mcv;
            }
            #endregion
            #region 
            public void GetObjectData(SerializationInfo info, StreamingContext context)
            {
                info.AddValue("DiscountRates"                         , this.DiscountRates);
                info.AddValue("FuelPathLow"                           , this.FuelPathLow);
                info.AddValue("FuelPathReference"                     , this.FuelPathReference);
                info.AddValue("FuelPathHigh"                          , this.FuelPathHigh);
                info.AddValue("ReboundAlphaShape"                     , this.ReboundAlphaShape);
                info.AddValue("ReboundBetaShape"                      , this.ReboundBetaShape);
                info.AddValue("ReboundScale"                          , this.ReboundScale);
                info.AddValue("ReboundBase"                           , this.ReboundBase);
                info.AddValue("CO2PathLow"                            , this.CO2PathLow);
                info.AddValue("CO2PathAverage"                        , this.CO2PathAverage);
                info.AddValue("CO2PathHigh"                           , this.CO2PathHigh);
                info.AddValue("CO2PathVeryHigh"                       , this.CO2PathVeryHigh);
                info.AddValue("MonopsonyMean"                         , this.MonopsonyMean);
                info.AddValue("MonopsonyStandardDeviation"            , this.MonopsonyStandardDeviation);
                info.AddValue("PriceShockMean"                        , this.PriceShockMean);
                info.AddValue("PriceShockStandardDeviation"           , this.PriceShockStandardDeviation);
                info.AddValue("MilitarySecurityAlternativeCost"       , this.MilitarySecurityAlternativeCost);
                info.AddValue("MilitarySecurityAlternativeProbability", this.MilitarySecurityAlternativeProbability);
            }
            #endregion
            #endregion
            #region 
            public double[] DiscountRates;
            public double FuelPathLow;
            public double FuelPathReference;
            public double FuelPathHigh;
            public double ReboundAlphaShape;
            public double ReboundBetaShape;
            public double ReboundScale;
            public double ReboundBase;
            public double CO2PathLow;
            public double CO2PathAverage;
            public double CO2PathHigh;
            public double CO2PathVeryHigh;
            public double MonopsonyMean;
            public double MonopsonyStandardDeviation;
            public double PriceShockMean;
            public double PriceShockStandardDeviation;
            public double MilitarySecurityAlternativeCost;
            public double MilitarySecurityAlternativeProbability;
            #endregion
        }
        #endregion
        #region 
        private Parameters()
        {
        }
        public Parameters(string path)
            : this(path, "")
        {
        }
        public Parameters(string path, string password)
            : this()
        {
            Input input = this.LoadFile(path, password);
            try
            {   
                this.ParseFile(input);
            }
            catch (Exception ex)
            {   
                throw new InputException("Errors were encountered while parsing file: " + ex.Message, path, ex);
            }
            finally
            {   
                input.Close();
            }
            this.PreCalc();
        }
        #endregion
        #region 
        #region 
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        public Parameters Clone()
        {
            Parameters p = new Parameters();
            p._survivability = this._survivability.Clone();
            p._milesDriven   = this._milesDriven.Clone();
            p._forecastData  = this._forecastData.Clone();
            p._economicValues = this._economicValues;
            p._economicCosts  = this._economicCosts;
            p._externalCosts  = this._externalCosts;
            p._emissionCosts  = this._emissionCosts;
            p._coEmissions  = this._coEmissions;
            p._vocEmissions = this._vocEmissions;
            p._noxEmissions = this._noxEmissions;
            p._pmEmissions  = this._pmEmissions;
            p._soxEmissions = this._soxEmissions;
            p._co2Emissions = this._co2Emissions;
            p._energyDensity = this._energyDensity;
            p._massDensity   = this._massDensity;
            p._carbonContent = this._carbonContent;
            p._soxRate       = this._soxRate;
            p._gasolineMix   = this._gasolineMix;
            p._gasolineAssumptions = this._gasolineAssumptions;
            p._dieselAssumptions   = this._dieselAssumptions;
            p._e85Assumptions      = this._e85Assumptions;
            p._cngAssumptions      = this._cngAssumptions;
            p._hydrogenAssumptions = this._hydrogenAssumptions;
            p._spare1Assumptions   = this._spare1Assumptions;
            p._spare2Assumptions   = this._spare2Assumptions;
            p._gasolineEnergyDensity = this._gasolineEnergyDensity;
            p._dieselEnergyDensity   = this._dieselEnergyDensity;
            p._e85EnergyDensity      = this._e85EnergyDensity;
            p._cngEnergyDensity      = this._cngEnergyDensity;
            p._hydrogenEnergyDensity = this._hydrogenEnergyDensity;
            p._spare1EnergyDensity   = this._spare1EnergyDensity;
            p._spare2EnergyDensity   = this._spare2EnergyDensity;
            p._gasolineMassDensity = this._gasolineMassDensity;
            p._dieselMassDensity   = this._dieselMassDensity;
            p._e85MassDensity      = this._e85MassDensity;
            p._cngMassDensity      = this._cngMassDensity;
            p._hydrogenMassDensity = this._hydrogenMassDensity;
            p._spare1MassDensity   = this._spare1MassDensity;
            p._spare2MassDensity   = this._spare2MassDensity;
            p._gasolineCarbonContent = this._gasolineCarbonContent;
            p._dieselCarbonContent   = this._dieselCarbonContent;
            p._e85CarbonContent      = this._e85CarbonContent;
            p._cngCarbonContent      = this._cngCarbonContent;
            p._hydrogenCarbonContent = this._hydrogenCarbonContent;
            p._spare1CarbonContent   = this._spare1CarbonContent;
            p._spare2CarbonContent   = this._spare2CarbonContent;
            p._gasolineSoxRate = this._gasolineSoxRate;
            p._dieselSoxRate   = this._dieselSoxRate;
            p._e85SoxRate      = this._e85SoxRate;
            p._cngSoxRate      = this._cngSoxRate;
            p._hydrogenSoxRate = this._hydrogenSoxRate;
            p._spare1SoxRate   = this._spare1SoxRate;
            p._spare2SoxRate   = this._spare2SoxRate;
            p._coGasolineEmissions = this._coGasolineEmissions;
            p._coDieselEmissions   = this._coDieselEmissions;
            p._coE85Emissions      = this._coE85Emissions;
            p._coCngEmissions      = this._coCngEmissions;
            p._coHydrogenEmissions = this._coHydrogenEmissions;
            p._coSpare1Emissions   = this._coSpare1Emissions;
            p._coSpare2Emissions   = this._coSpare2Emissions;
            p._vocGasolineEmissions = this._vocGasolineEmissions;
            p._vocDieselEmissions   = this._vocDieselEmissions;
            p._vocE85Emissions      = this._vocE85Emissions;
            p._vocCngEmissions      = this._vocCngEmissions;
            p._vocHydrogenEmissions = this._vocHydrogenEmissions;
            p._vocSpare1Emissions   = this._vocSpare1Emissions;
            p._vocSpare2Emissions   = this._vocSpare2Emissions;
            p._noxGasolineEmissions = this._noxGasolineEmissions;
            p._noxDieselEmissions   = this._noxDieselEmissions;
            p._noxE85Emissions      = this._noxE85Emissions;
            p._noxCngEmissions      = this._noxCngEmissions;
            p._noxHydrogenEmissions = this._noxHydrogenEmissions;
            p._noxSpare1Emissions   = this._noxSpare1Emissions;
            p._noxSpare2Emissions   = this._noxSpare2Emissions;
            p._pmGasolineEmissions = this._pmGasolineEmissions;
            p._pmDieselEmissions   = this._pmDieselEmissions;
            p._pmE85Emissions      = this._pmE85Emissions;
            p._pmCngEmissions      = this._pmCngEmissions;
            p._pmHydrogenEmissions = this._pmHydrogenEmissions;
            p._pmSpare1Emissions   = this._pmSpare1Emissions;
            p._pmSpare2Emissions   = this._pmSpare2Emissions;
            p._soxGasolineEmissions = this._soxGasolineEmissions;
            p._soxDieselEmissions   = this._soxDieselEmissions;
            p._soxE85Emissions      = this._soxE85Emissions;
            p._soxCngEmissions      = this._soxCngEmissions;
            p._soxHydrogenEmissions = this._soxHydrogenEmissions;
            p._soxSpare1Emissions   = this._soxSpare1Emissions;
            p._soxSpare2Emissions   = this._soxSpare2Emissions;
            p._co2GasolineEmissions = this._co2GasolineEmissions;
            p._co2DieselEmissions   = this._co2DieselEmissions;
            p._co2E85Emissions      = this._co2E85Emissions;
            p._co2CngEmissions      = this._co2CngEmissions;
            p._co2HydrogenEmissions = this._co2HydrogenEmissions;
            p._co2Spare1Emissions   = this._co2Spare1Emissions;
            p._co2Spare2Emissions   = this._co2Spare2Emissions;
            p._monteCarlo = this._monteCarlo.Clone();
            return p;
        }
        #endregion
        public double[] GetVehicleSurvivability(Vehicle vehicle)
        {
            switch (vehicle.Mobile6Class)
            {
                case Mobile6Class.Ldgv:   return this.Survivability.Ldgv;
                case Mobile6Class.Lddv:   return this.Survivability.Lddv;
                case Mobile6Class.Ldgt1:  return this.Survivability.Ldgt1;
                case Mobile6Class.Lddt1:  return this.Survivability.Lddt1;
                case Mobile6Class.Ldgt2:  return this.Survivability.Ldgt2;
                case Mobile6Class.Lddt2:  return this.Survivability.Lddt2;
                case Mobile6Class.Hdgv2b: return this.Survivability.Hdgv2b;
                case Mobile6Class.Hddv2b: return this.Survivability.Hddv2b;
                default:                  return new double[36];
            }
        }
        public double[] GetVehicleMilesDriven(Vehicle vehicle, Estimates priceEstimates)
        {
            string style = vehicle.Description.Style.ToUpper();
            switch (style)
            {
                case "PICKUP" : return this._milesDriven.GetPickupMilesDriven(priceEstimates);
                case "UTILITY": return this._milesDriven.GetSuvMilesDriven(priceEstimates);
                case "VAN"    : return this._milesDriven.GetVanMilesDriven(priceEstimates);
                default       : return this._milesDriven.GetAutomobileMilesDriven(priceEstimates);
            }
        }
        private Input LoadFile(string path, string password)
        {
            return new Input(path, password);
        }
        private void ParseFile(Input input)
        {
            string[] sheets = input.SheetNames;
            string[] names = {
                                 "Vehicle Age Data", "Forecast Data", "Economic Values",
                                 "Upstream Emissions", "Fuel Properties", "Monte-Carlo",
                                 "NMNL Market Share", "Safety Values"
                             };
            int[] indexes = new int[names.Length];
            for (int i = 0; i < indexes.Length; i++) { indexes[i] = -1; }
            for (int i = 0; i < sheets.Length; i++)
            {
                if (indexes[0] == -1 && (input.Compare(sheets[i], names[0], true) ||
                    input.Compare(sheets[i], names[0].Replace("Age", ""), true)))      { indexes[0] = i; }
                else if (indexes[1] == -1 && input.Compare(sheets[i], names[1], true)) { indexes[1] = i; }
                else if (indexes[2] == -1 && input.Compare(sheets[i], names[2], true)) { indexes[2] = i; }
                else if (indexes[3] == -1 && input.Compare(sheets[i], names[3], true)) { indexes[3] = i; }
                else if (indexes[4] == -1 && input.Compare(sheets[i], names[4], true)) { indexes[4] = i; }
                else if (indexes[5] == -1 && input.Compare(sheets[i], names[5], true)) { indexes[5] = i; }
                else if (indexes[6] == -1 && input.Compare(sheets[i], names[6], true)) { indexes[6] = i; }
                else if (indexes[7] == -1 && input.Compare(sheets[i], names[7], true)) { indexes[7] = i; }
            }
            if (indexes[5] == -1) { indexes[5] = -2; }  
            if (indexes[6] == -1) { indexes[6] = -2; }  
            if (indexes[7] == -1) { indexes[7] = -2; }  
            input.VerifyIndexes(indexes, names);
            this.ParseVehicleAgeData(input, indexes[0]);
            this.ParseForecastData(input, indexes[1]);
            this.ParseEconomicValues(input, indexes[2]);
            this.ParseUpstreamEmissions(input, indexes[3]);
            this.ParseFuelProperties(input, indexes[4]);
            if (indexes[5] != -2) { this.ParseMonteCarlo(input, indexes[5]); }
            this.ParseAdditional(input);
        }
        private void ParseVehicleAgeData(Input input, int sheetIndex)
        {
            input.ActivateWorksheet(sheetIndex, new XlCell(5, 2), new XlCell(40, 22));
            this._survivability.Initialize(36);
            this._milesDriven.Initialize(36);
            for (int i = 0; i < 36; i++)
            {
                this._survivability.Ldgv  [i] = input.GetDouble(i, 0);
                this._survivability.Lddv  [i] = input.GetDouble(i, 1);
                this._survivability.Ldgt1 [i] = input.GetDouble(i, 2);
                this._survivability.Ldgt2 [i] = input.GetDouble(i, 3);
                this._survivability.Lddt1 [i] = input.GetDouble(i, 4);
                this._survivability.Lddt2 [i] = input.GetDouble(i, 5);
                this._survivability.Hdgv2b[i] = input.GetDouble(i, 6);
                this._survivability.Hddv2b[i] = input.GetDouble(i, 7);
                this._milesDriven.Automobile    [i] = input.GetDouble(i,  8);   
                this._milesDriven.Pickup        [i] = input.GetDouble(i,  9);
                this._milesDriven.Van           [i] = input.GetDouble(i, 10);
                this._milesDriven.Suv           [i] = input.GetDouble(i, 11);
                this._milesDriven.LtDefault     [i] = input.GetDouble(i, 12);
                this._milesDriven.LowAutomobile [i] = input.GetDouble(i, 13);   
                this._milesDriven.LowPickup     [i] = input.GetDouble(i, 14);
                this._milesDriven.LowVan        [i] = input.GetDouble(i, 15);
                this._milesDriven.LowSuv        [i] = input.GetDouble(i, 16);
                this._milesDriven.HighAutomobile[i] = input.GetDouble(i, 17);   
                this._milesDriven.HighPickup    [i] = input.GetDouble(i, 18);
                this._milesDriven.HighVan       [i] = input.GetDouble(i, 19);
                this._milesDriven.HighSuv       [i] = input.GetDouble(i, 20);
            }
        }
        private void ParseForecastData(Input input, int sheetIndex)
        {
            input.ActivateWorksheet(sheetIndex, new XlCell(6, 2), new XlCell(56, 51));
            this._forecastData.Initialize(51);
            for (int i = 0; i < 51; i++)
            {    
                this._forecastData.LowGasolineFuelPrice [i] = input.GetDouble(i,  0);
                this._forecastData.GasolineFuelPrice    [i] = input.GetDouble(i,  1);
                this._forecastData.HighGasolineFuelPrice[i] = input.GetDouble(i,  2);
                this._forecastData.LowDieselFuelPrice   [i] = input.GetDouble(i,  3);
                this._forecastData.DieselFuelPrice      [i] = input.GetDouble(i,  4);
                this._forecastData.HighDieselFuelPrice  [i] = input.GetDouble(i,  5);
                this._forecastData.LowE85FuelPrice      [i] = input.GetDouble(i,  6);
                this._forecastData.E85FuelPrice         [i] = input.GetDouble(i,  7);
                this._forecastData.HighE85FuelPrice     [i] = input.GetDouble(i,  8);
                this._forecastData.LowCNGFuelPrice      [i] = input.GetDouble(i,  9);
                this._forecastData.CNGFuelPrice         [i] = input.GetDouble(i, 10);
                this._forecastData.HighCNGFuelPrice     [i] = input.GetDouble(i, 11);
                this._forecastData.LowHydrogenFuelPrice [i] = input.GetDouble(i, 12);
                this._forecastData.HydrogenFuelPrice    [i] = input.GetDouble(i, 13);
                this._forecastData.HighHydrogenFuelPrice[i] = input.GetDouble(i, 14);
                this._forecastData.LowSpare1FuelPrice   [i] = input.GetDouble(i, 15);
                this._forecastData.Spare1FuelPrice      [i] = input.GetDouble(i, 16);
                this._forecastData.HighSpare1FuelPrice  [i] = input.GetDouble(i, 17);
                this._forecastData.LowSpare2FuelPrice   [i] = input.GetDouble(i, 18);
                this._forecastData.Spare2FuelPrice      [i] = input.GetDouble(i, 19);
                this._forecastData.HighSpare2FuelPrice  [i] = input.GetDouble(i, 20);
                this._forecastData.GasolineFuelTax[i] = input.GetDouble(i, 21);
                this._forecastData.DieselFuelTax  [i] = input.GetDouble(i, 22);
                this._forecastData.E85FuelTax     [i] = input.GetDouble(i, 23);
                this._forecastData.CNGFuelTax     [i] = input.GetDouble(i, 24);
                this._forecastData.HydrogenFuelTax[i] = input.GetDouble(i, 25);
                this._forecastData.Spare1FuelTax  [i] = input.GetDouble(i, 26);
                this._forecastData.Spare2FuelTax  [i] = input.GetDouble(i, 27);
                this._forecastData.LowGasolinePreTaxFuelPrice [i] = input.GetDouble(i, 28);
                this._forecastData.GasolinePreTaxFuelPrice    [i] = input.GetDouble(i, 29);
                this._forecastData.HighGasolinePreTaxFuelPrice[i] = input.GetDouble(i, 30);
                this._forecastData.LowDieselPreTaxFuelPrice   [i] = input.GetDouble(i, 31);
                this._forecastData.DieselPreTaxFuelPrice      [i] = input.GetDouble(i, 32);
                this._forecastData.HighDieselPreTaxFuelPrice  [i] = input.GetDouble(i, 33);
                this._forecastData.LowE85PreTaxFuelPrice      [i] = input.GetDouble(i, 34);
                this._forecastData.E85PreTaxFuelPrice         [i] = input.GetDouble(i, 35);
                this._forecastData.HighE85PreTaxFuelPrice     [i] = input.GetDouble(i, 36);
                this._forecastData.LowCNGPreTaxFuelPrice      [i] = input.GetDouble(i, 37);
                this._forecastData.CNGPreTaxFuelPrice         [i] = input.GetDouble(i, 38);
                this._forecastData.LowHydrogenPreTaxFuelPrice [i] = input.GetDouble(i, 39);
                this._forecastData.HydrogenPreTaxFuelPrice    [i] = input.GetDouble(i, 40);
                this._forecastData.HighHydrogenPreTaxFuelPrice[i] = input.GetDouble(i, 41);
                this._forecastData.LowSpare1PreTaxFuelPrice   [i] = input.GetDouble(i, 42);
                this._forecastData.Spare1PreTaxFuelPrice      [i] = input.GetDouble(i, 43);
                this._forecastData.HighSpare1PreTaxFuelPrice  [i] = input.GetDouble(i, 44);
                this._forecastData.LowSpare2PreTaxFuelPrice   [i] = input.GetDouble(i, 45);
                this._forecastData.Spare2PreTaxFuelPrice      [i] = input.GetDouble(i, 46);
                this._forecastData.HighSpare2PreTaxFuelPrice  [i] = input.GetDouble(i, 47);
                this._forecastData.E85Share[i] = input.GetDouble(i, 48);
            }
        }
        private void ParseEconomicValues(Input input, int sheetIndex)
        {
            input.ActivateWorksheet(sheetIndex);
            this._economicValues.ReboundEffect          = input.GetDouble( 2, 1);
            this._economicValues.DiscountRate           = input.GetDouble( 3, 1);
            this._economicValues.PaybackPeriod          = input.GetDouble( 4, 1);
            this._economicValues.Kf                     = input.GetDouble( 5, 1);
            this._economicValues.TestVsOnRoadGap        = input.GetDouble( 6, 1);
            this._economicValues.VehicleTravelTimeValue = input.GetDouble( 7, 1);
            this._economicValues.VMTBaseYear            = input.GetInt32 (33, 1);
            this._economicValues.VMTCarGrowthRate       = input.GetDouble(34, 1);
            this._economicValues.VMTTruckGrowthRate     = input.GetDouble(35, 1);
            this._economicCosts.Monopsony        = input.GetDouble( 9, 1);
            this._economicCosts.PriceShock       = input.GetDouble(10, 1);
            this._economicCosts.MilitarySecurity = input.GetDouble(11, 1);
            this._economicCosts.Total            = input.GetDouble(12, 1);
            this._economicCosts.TotalBBL         = input.GetDouble(13, 1);
            this._externalCosts.CarCongestion   = input.GetDouble(16, 1);
            this._externalCosts.CarAccident     = input.GetDouble(17, 1);
            this._externalCosts.CarNoise        = input.GetDouble(18, 1);
            this._externalCosts.TruckCongestion = input.GetDouble(20, 1);
            this._externalCosts.TruckAccident   = input.GetDouble(21, 1);
            this._externalCosts.TruckNoise      = input.GetDouble(22, 1);
            this._emissionCosts.CO         = input.GetDouble(24, 1);
            this._emissionCosts.VOC        = input.GetDouble(25, 1);
            this._emissionCosts.NOX        = input.GetDouble(26, 1);
            this._emissionCosts.PM         = input.GetDouble(27, 1);
            this._emissionCosts.SOX        = input.GetDouble(28, 1);
            this._emissionCosts.CO2DiscRateLow      = input.GetDouble(37, 1);
            this._emissionCosts.CO2DiscRateAverage  = input.GetDouble(38, 1);
            this._emissionCosts.CO2DiscRateHigh     = input.GetDouble(39, 1);
            this._emissionCosts.CO2DiscRateVeryHigh = input.GetDouble(40, 1);
            const int co2Len = 61;      
            this._emissionCosts.CO2Low      = new double[co2Len];
            this._emissionCosts.CO2Average  = new double[co2Len];
            this._emissionCosts.CO2High     = new double[co2Len];
            this._emissionCosts.CO2VeryHigh = new double[co2Len];
            for (int i = 3, rows = input.Rows; i < rows; i++)
            {
                int co2YearIndex = input.GetInt32(i, 3) - ModelYear.MinYear;
                if (co2YearIndex < 0 || co2YearIndex > 60) { continue; }    
                this._emissionCosts.CO2Low     [co2YearIndex] = input.GetDouble(i, 4);
                this._emissionCosts.CO2Average [co2YearIndex] = input.GetDouble(i, 5);
                this._emissionCosts.CO2High    [co2YearIndex] = input.GetDouble(i, 6);
                this._emissionCosts.CO2VeryHigh[co2YearIndex] = input.GetDouble(i, 7);
            }
        }
        private void ParseUpstreamEmissions(Input input, int sheetIndex)
        {
            input.ActivateWorksheet(sheetIndex, new XlCell(5, 3), new XlCell(64, 6));
            this._coEmissions.Extraction.GasolineConventional = input.GetDouble(0, 0);
            this._coEmissions.Extraction.GasolineFederal      = input.GetDouble(1, 0);
            this._coEmissions.Extraction.GasolineCalifornia   = input.GetDouble(2, 0);
            this._coEmissions.Extraction.DieselConventional   = input.GetDouble(3, 0);
            this._coEmissions.Extraction.DieselLowSulfur      = input.GetDouble(4, 0);
            this._coEmissions.Extraction.E85                  = input.GetDouble(5, 0);
            this._coEmissions.Extraction.CNG                  = input.GetDouble(6, 0);
            this._coEmissions.Extraction.Hydrogen             = input.GetDouble(7, 0);
            this._coEmissions.Extraction.Spare1               = input.GetDouble(8, 0);
            this._coEmissions.Extraction.Spare2               = input.GetDouble(9, 0);
            this._coEmissions.Transportation.GasolineConventional = input.GetDouble(0, 1);
            this._coEmissions.Transportation.GasolineFederal      = input.GetDouble(1, 1);
            this._coEmissions.Transportation.GasolineCalifornia   = input.GetDouble(2, 1);
            this._coEmissions.Transportation.DieselConventional   = input.GetDouble(3, 1);
            this._coEmissions.Transportation.DieselLowSulfur      = input.GetDouble(4, 1);
            this._coEmissions.Transportation.E85                  = input.GetDouble(5, 1);
            this._coEmissions.Transportation.CNG                  = input.GetDouble(6, 1);
            this._coEmissions.Transportation.Hydrogen             = input.GetDouble(7, 1);
            this._coEmissions.Transportation.Spare1               = input.GetDouble(8, 1);
            this._coEmissions.Transportation.Spare2               = input.GetDouble(9, 1);
            this._coEmissions.Refining.GasolineConventional = input.GetDouble(0, 2);
            this._coEmissions.Refining.GasolineFederal      = input.GetDouble(1, 2);
            this._coEmissions.Refining.GasolineCalifornia   = input.GetDouble(2, 2);
            this._coEmissions.Refining.DieselConventional   = input.GetDouble(3, 2);
            this._coEmissions.Refining.DieselLowSulfur      = input.GetDouble(4, 2);
            this._coEmissions.Refining.E85                  = input.GetDouble(5, 2);
            this._coEmissions.Refining.CNG                  = input.GetDouble(6, 2);
            this._coEmissions.Refining.Hydrogen             = input.GetDouble(7, 2);
            this._coEmissions.Refining.Spare1               = input.GetDouble(8, 2);
            this._coEmissions.Refining.Spare2               = input.GetDouble(9, 2);
            this._coEmissions.TSD.GasolineConventional = input.GetDouble(0, 3);
            this._coEmissions.TSD.GasolineFederal      = input.GetDouble(1, 3);
            this._coEmissions.TSD.GasolineCalifornia   = input.GetDouble(2, 3);
            this._coEmissions.TSD.DieselConventional   = input.GetDouble(3, 3);
            this._coEmissions.TSD.DieselLowSulfur      = input.GetDouble(4, 3);
            this._coEmissions.TSD.E85                  = input.GetDouble(5, 3);
            this._coEmissions.TSD.CNG                  = input.GetDouble(6, 3);
            this._coEmissions.TSD.Hydrogen             = input.GetDouble(7, 3);
            this._coEmissions.TSD.Spare1               = input.GetDouble(8, 3);
            this._coEmissions.TSD.Spare2               = input.GetDouble(9, 3);
            this._vocEmissions.Extraction.GasolineConventional = input.GetDouble(10, 0);
            this._vocEmissions.Extraction.GasolineFederal      = input.GetDouble(11, 0);
            this._vocEmissions.Extraction.GasolineCalifornia   = input.GetDouble(12, 0);
            this._vocEmissions.Extraction.DieselConventional   = input.GetDouble(13, 0);
            this._vocEmissions.Extraction.DieselLowSulfur      = input.GetDouble(14, 0);
            this._vocEmissions.Extraction.E85                  = input.GetDouble(15, 0);
            this._vocEmissions.Extraction.CNG                  = input.GetDouble(16, 0);
            this._vocEmissions.Extraction.Hydrogen             = input.GetDouble(17, 0);
            this._vocEmissions.Extraction.Spare1               = input.GetDouble(18, 0);
            this._vocEmissions.Extraction.Spare2               = input.GetDouble(19, 0);
            this._vocEmissions.Transportation.GasolineConventional = input.GetDouble(10, 1);
            this._vocEmissions.Transportation.GasolineFederal      = input.GetDouble(11, 1);
            this._vocEmissions.Transportation.GasolineCalifornia   = input.GetDouble(12, 1);
            this._vocEmissions.Transportation.DieselConventional   = input.GetDouble(13, 1);
            this._vocEmissions.Transportation.DieselLowSulfur      = input.GetDouble(14, 1);
            this._vocEmissions.Transportation.E85                  = input.GetDouble(15, 1);
            this._vocEmissions.Transportation.CNG                  = input.GetDouble(16, 1);
            this._vocEmissions.Transportation.Hydrogen             = input.GetDouble(17, 1);
            this._vocEmissions.Transportation.Spare1               = input.GetDouble(18, 1);
            this._vocEmissions.Transportation.Spare2               = input.GetDouble(19, 1);
            this._vocEmissions.Refining.GasolineConventional = input.GetDouble(10, 2);
            this._vocEmissions.Refining.GasolineFederal      = input.GetDouble(11, 2);
            this._vocEmissions.Refining.GasolineCalifornia   = input.GetDouble(12, 2);
            this._vocEmissions.Refining.DieselConventional   = input.GetDouble(13, 2);
            this._vocEmissions.Refining.DieselLowSulfur      = input.GetDouble(14, 2);
            this._vocEmissions.Refining.E85                  = input.GetDouble(15, 2);
            this._vocEmissions.Refining.CNG                  = input.GetDouble(16, 2);
            this._vocEmissions.Refining.Hydrogen             = input.GetDouble(17, 2);
            this._vocEmissions.Refining.Spare1               = input.GetDouble(18, 2);
            this._vocEmissions.Refining.Spare2               = input.GetDouble(19, 2);
            this._vocEmissions.TSD.GasolineConventional = input.GetDouble(10, 3);
            this._vocEmissions.TSD.GasolineFederal      = input.GetDouble(11, 3);
            this._vocEmissions.TSD.GasolineCalifornia   = input.GetDouble(12, 3);
            this._vocEmissions.TSD.DieselConventional   = input.GetDouble(13, 3);
            this._vocEmissions.TSD.DieselLowSulfur      = input.GetDouble(14, 3);
            this._vocEmissions.TSD.E85                  = input.GetDouble(15, 3);
            this._vocEmissions.TSD.CNG                  = input.GetDouble(16, 3);
            this._vocEmissions.TSD.Hydrogen             = input.GetDouble(17, 3);
            this._vocEmissions.TSD.Spare1               = input.GetDouble(18, 3);
            this._vocEmissions.TSD.Spare2               = input.GetDouble(19, 3);
            this._noxEmissions.Extraction.GasolineConventional = input.GetDouble(20, 0);
            this._noxEmissions.Extraction.GasolineFederal      = input.GetDouble(21, 0);
            this._noxEmissions.Extraction.GasolineCalifornia   = input.GetDouble(22, 0);
            this._noxEmissions.Extraction.DieselConventional   = input.GetDouble(23, 0);
            this._noxEmissions.Extraction.DieselLowSulfur      = input.GetDouble(24, 0);
            this._noxEmissions.Extraction.E85                  = input.GetDouble(25, 0);
            this._noxEmissions.Extraction.CNG                  = input.GetDouble(26, 0);
            this._noxEmissions.Extraction.Hydrogen             = input.GetDouble(27, 0);
            this._noxEmissions.Extraction.Spare1               = input.GetDouble(28, 0);
            this._noxEmissions.Extraction.Spare2               = input.GetDouble(29, 0);
            this._noxEmissions.Transportation.GasolineConventional = input.GetDouble(20, 1);
            this._noxEmissions.Transportation.GasolineFederal      = input.GetDouble(21, 1);
            this._noxEmissions.Transportation.GasolineCalifornia   = input.GetDouble(22, 1);
            this._noxEmissions.Transportation.DieselConventional   = input.GetDouble(23, 1);
            this._noxEmissions.Transportation.DieselLowSulfur      = input.GetDouble(24, 1);
            this._noxEmissions.Transportation.E85                  = input.GetDouble(25, 1);
            this._noxEmissions.Transportation.CNG                  = input.GetDouble(26, 1);
            this._noxEmissions.Transportation.Hydrogen             = input.GetDouble(27, 1);
            this._noxEmissions.Transportation.Spare1               = input.GetDouble(28, 1);
            this._noxEmissions.Transportation.Spare2               = input.GetDouble(29, 1);
            this._noxEmissions.Refining.GasolineConventional = input.GetDouble(20, 2);
            this._noxEmissions.Refining.GasolineFederal      = input.GetDouble(21, 2);
            this._noxEmissions.Refining.GasolineCalifornia   = input.GetDouble(22, 2);
            this._noxEmissions.Refining.DieselConventional   = input.GetDouble(23, 2);
            this._noxEmissions.Refining.DieselLowSulfur      = input.GetDouble(24, 2);
            this._noxEmissions.Refining.E85                  = input.GetDouble(25, 2);
            this._noxEmissions.Refining.CNG                  = input.GetDouble(26, 2);
            this._noxEmissions.Refining.Hydrogen             = input.GetDouble(27, 2);
            this._noxEmissions.Refining.Spare1               = input.GetDouble(28, 2);
            this._noxEmissions.Refining.Spare2               = input.GetDouble(29, 2);
            this._noxEmissions.TSD.GasolineConventional = input.GetDouble(20, 3);
            this._noxEmissions.TSD.GasolineFederal      = input.GetDouble(21, 3);
            this._noxEmissions.TSD.GasolineCalifornia   = input.GetDouble(22, 3);
            this._noxEmissions.TSD.DieselConventional   = input.GetDouble(23, 3);
            this._noxEmissions.TSD.DieselLowSulfur      = input.GetDouble(24, 3);
            this._noxEmissions.TSD.E85                  = input.GetDouble(25, 3);
            this._noxEmissions.TSD.CNG                  = input.GetDouble(26, 3);
            this._noxEmissions.TSD.Hydrogen             = input.GetDouble(27, 3);
            this._noxEmissions.TSD.Spare1               = input.GetDouble(28, 3);
            this._noxEmissions.TSD.Spare2               = input.GetDouble(29, 3);
            this._soxEmissions.Extraction.GasolineConventional = input.GetDouble(30, 0);
            this._soxEmissions.Extraction.GasolineFederal      = input.GetDouble(31, 0);
            this._soxEmissions.Extraction.GasolineCalifornia   = input.GetDouble(32, 0);
            this._soxEmissions.Extraction.DieselConventional   = input.GetDouble(33, 0);
            this._soxEmissions.Extraction.DieselLowSulfur      = input.GetDouble(34, 0);
            this._soxEmissions.Extraction.E85                  = input.GetDouble(35, 0);
            this._soxEmissions.Extraction.CNG                  = input.GetDouble(36, 0);
            this._soxEmissions.Extraction.Hydrogen             = input.GetDouble(37, 0);
            this._soxEmissions.Extraction.Spare1               = input.GetDouble(38, 0);
            this._soxEmissions.Extraction.Spare2               = input.GetDouble(39, 0);
            this._soxEmissions.Transportation.GasolineConventional = input.GetDouble(30, 1);
            this._soxEmissions.Transportation.GasolineFederal      = input.GetDouble(31, 1);
            this._soxEmissions.Transportation.GasolineCalifornia   = input.GetDouble(32, 1);
            this._soxEmissions.Transportation.DieselConventional   = input.GetDouble(33, 1);
            this._soxEmissions.Transportation.DieselLowSulfur      = input.GetDouble(34, 1);
            this._soxEmissions.Transportation.E85                  = input.GetDouble(35, 1);
            this._soxEmissions.Transportation.CNG                  = input.GetDouble(36, 1);
            this._soxEmissions.Transportation.Hydrogen             = input.GetDouble(37, 1);
            this._soxEmissions.Transportation.Spare1               = input.GetDouble(38, 1);
            this._soxEmissions.Transportation.Spare2               = input.GetDouble(39, 1);
            this._soxEmissions.Refining.GasolineConventional = input.GetDouble(30, 2);
            this._soxEmissions.Refining.GasolineFederal      = input.GetDouble(31, 2);
            this._soxEmissions.Refining.GasolineCalifornia   = input.GetDouble(32, 2);
            this._soxEmissions.Refining.DieselConventional   = input.GetDouble(33, 2);
            this._soxEmissions.Refining.DieselLowSulfur      = input.GetDouble(34, 2);
            this._soxEmissions.Refining.E85                  = input.GetDouble(35, 2);
            this._soxEmissions.Refining.CNG                  = input.GetDouble(36, 2);
            this._soxEmissions.Refining.Hydrogen             = input.GetDouble(37, 2);
            this._soxEmissions.Refining.Spare1               = input.GetDouble(38, 2);
            this._soxEmissions.Refining.Spare2               = input.GetDouble(39, 2);
            this._soxEmissions.TSD.GasolineConventional = input.GetDouble(30, 3);
            this._soxEmissions.TSD.GasolineFederal      = input.GetDouble(31, 3);
            this._soxEmissions.TSD.GasolineCalifornia   = input.GetDouble(32, 3);
            this._soxEmissions.TSD.DieselConventional   = input.GetDouble(33, 3);
            this._soxEmissions.TSD.DieselLowSulfur      = input.GetDouble(34, 3);
            this._soxEmissions.TSD.E85                  = input.GetDouble(35, 3);
            this._soxEmissions.TSD.CNG                  = input.GetDouble(36, 3);
            this._soxEmissions.TSD.Hydrogen             = input.GetDouble(37, 3);
            this._soxEmissions.TSD.Spare1               = input.GetDouble(38, 3);
            this._soxEmissions.TSD.Spare2               = input.GetDouble(39, 3);
            this._pmEmissions.Extraction.GasolineConventional = input.GetDouble(40, 0);
            this._pmEmissions.Extraction.GasolineFederal      = input.GetDouble(41, 0);
            this._pmEmissions.Extraction.GasolineCalifornia   = input.GetDouble(42, 0);
            this._pmEmissions.Extraction.DieselConventional   = input.GetDouble(43, 0);
            this._pmEmissions.Extraction.DieselLowSulfur      = input.GetDouble(44, 0);
            this._pmEmissions.Extraction.E85                  = input.GetDouble(45, 0);
            this._pmEmissions.Extraction.CNG                  = input.GetDouble(46, 0);
            this._pmEmissions.Extraction.Hydrogen             = input.GetDouble(47, 0);
            this._pmEmissions.Extraction.Spare1               = input.GetDouble(48, 0);
            this._pmEmissions.Extraction.Spare2               = input.GetDouble(49, 0);
            this._pmEmissions.Transportation.GasolineConventional = input.GetDouble(40, 1);
            this._pmEmissions.Transportation.GasolineFederal      = input.GetDouble(41, 1);
            this._pmEmissions.Transportation.GasolineCalifornia   = input.GetDouble(42, 1);
            this._pmEmissions.Transportation.DieselConventional   = input.GetDouble(43, 1);
            this._pmEmissions.Transportation.DieselLowSulfur      = input.GetDouble(44, 1);
            this._pmEmissions.Transportation.E85                  = input.GetDouble(45, 1);
            this._pmEmissions.Transportation.CNG                  = input.GetDouble(46, 1);
            this._pmEmissions.Transportation.Hydrogen             = input.GetDouble(47, 1);
            this._pmEmissions.Transportation.Spare1               = input.GetDouble(48, 1);
            this._pmEmissions.Transportation.Spare2               = input.GetDouble(49, 1);
            this._pmEmissions.Refining.GasolineConventional = input.GetDouble(40, 2);
            this._pmEmissions.Refining.GasolineFederal      = input.GetDouble(41, 2);
            this._pmEmissions.Refining.GasolineCalifornia   = input.GetDouble(42, 2);
            this._pmEmissions.Refining.DieselConventional   = input.GetDouble(43, 2);
            this._pmEmissions.Refining.DieselLowSulfur      = input.GetDouble(44, 2);
            this._pmEmissions.Refining.E85                  = input.GetDouble(45, 2);
            this._pmEmissions.Refining.CNG                  = input.GetDouble(46, 2);
            this._pmEmissions.Refining.Hydrogen             = input.GetDouble(47, 2);
            this._pmEmissions.Refining.Spare1               = input.GetDouble(48, 2);
            this._pmEmissions.Refining.Spare2               = input.GetDouble(49, 2);
            this._pmEmissions.TSD.GasolineConventional = input.GetDouble(40, 3);
            this._pmEmissions.TSD.GasolineFederal      = input.GetDouble(41, 3);
            this._pmEmissions.TSD.GasolineCalifornia   = input.GetDouble(42, 3);
            this._pmEmissions.TSD.DieselConventional   = input.GetDouble(43, 3);
            this._pmEmissions.TSD.DieselLowSulfur      = input.GetDouble(44, 3);
            this._pmEmissions.TSD.E85                  = input.GetDouble(45, 3);
            this._pmEmissions.TSD.CNG                  = input.GetDouble(46, 3);
            this._pmEmissions.TSD.Hydrogen             = input.GetDouble(47, 3);
            this._pmEmissions.TSD.Spare1               = input.GetDouble(48, 3);
            this._pmEmissions.TSD.Spare2               = input.GetDouble(49, 3);
            this._co2Emissions.Extraction.GasolineConventional = input.GetDouble(50, 0);
            this._co2Emissions.Extraction.GasolineFederal      = input.GetDouble(51, 0);
            this._co2Emissions.Extraction.GasolineCalifornia   = input.GetDouble(52, 0);
            this._co2Emissions.Extraction.DieselConventional   = input.GetDouble(53, 0);
            this._co2Emissions.Extraction.DieselLowSulfur      = input.GetDouble(54, 0);
            this._co2Emissions.Extraction.E85                  = input.GetDouble(55, 0);
            this._co2Emissions.Extraction.CNG                  = input.GetDouble(56, 0);
            this._co2Emissions.Extraction.Hydrogen             = input.GetDouble(57, 0);
            this._co2Emissions.Extraction.Spare1               = input.GetDouble(58, 0);
            this._co2Emissions.Extraction.Spare2               = input.GetDouble(59, 0);
            this._co2Emissions.Transportation.GasolineConventional = input.GetDouble(50, 1);
            this._co2Emissions.Transportation.GasolineFederal      = input.GetDouble(51, 1);
            this._co2Emissions.Transportation.GasolineCalifornia   = input.GetDouble(52, 1);
            this._co2Emissions.Transportation.DieselConventional   = input.GetDouble(53, 1);
            this._co2Emissions.Transportation.DieselLowSulfur      = input.GetDouble(54, 1);
            this._co2Emissions.Transportation.E85                  = input.GetDouble(55, 1);
            this._co2Emissions.Transportation.CNG                  = input.GetDouble(56, 1);
            this._co2Emissions.Transportation.Hydrogen             = input.GetDouble(57, 1);
            this._co2Emissions.Transportation.Spare1               = input.GetDouble(58, 1);
            this._co2Emissions.Transportation.Spare2               = input.GetDouble(59, 1);
            this._co2Emissions.Refining.GasolineConventional = input.GetDouble(50, 2);
            this._co2Emissions.Refining.GasolineFederal      = input.GetDouble(51, 2);
            this._co2Emissions.Refining.GasolineCalifornia   = input.GetDouble(52, 2);
            this._co2Emissions.Refining.DieselConventional   = input.GetDouble(53, 2);
            this._co2Emissions.Refining.DieselLowSulfur      = input.GetDouble(54, 2);
            this._co2Emissions.Refining.E85                  = input.GetDouble(55, 2);
            this._co2Emissions.Refining.CNG                  = input.GetDouble(56, 2);
            this._co2Emissions.Refining.Hydrogen             = input.GetDouble(57, 2);
            this._co2Emissions.Refining.Spare1               = input.GetDouble(58, 2);
            this._co2Emissions.Refining.Spare2               = input.GetDouble(59, 2);
            this._co2Emissions.TSD.GasolineConventional = input.GetDouble(50, 3);
            this._co2Emissions.TSD.GasolineFederal      = input.GetDouble(51, 3);
            this._co2Emissions.TSD.GasolineCalifornia   = input.GetDouble(52, 3);
            this._co2Emissions.TSD.DieselConventional   = input.GetDouble(53, 3);
            this._co2Emissions.TSD.DieselLowSulfur      = input.GetDouble(54, 3);
            this._co2Emissions.TSD.E85                  = input.GetDouble(55, 3);
            this._co2Emissions.TSD.CNG                  = input.GetDouble(56, 3);
            this._co2Emissions.TSD.Hydrogen             = input.GetDouble(57, 3);
            this._co2Emissions.TSD.Spare1               = input.GetDouble(58, 3);
            this._co2Emissions.TSD.Spare2               = input.GetDouble(59, 3);
        }
        private void ParseFuelProperties(Input input, int sheetIndex)
        {
            input.ActivateWorksheet(sheetIndex, new XlCell(4, 2), new XlCell(35, 8));
            this._energyDensity.GasolineConventional = input.GetDouble(0, 0);
            this._energyDensity.GasolineFederal      = input.GetDouble(1, 0);
            this._energyDensity.GasolineCalifornia   = input.GetDouble(2, 0);
            this._energyDensity.DieselConventional   = input.GetDouble(3, 0);
            this._energyDensity.DieselLowSulfur      = input.GetDouble(4, 0);
            this._energyDensity.E85                  = input.GetDouble(5, 0);
            this._energyDensity.CNG                  = input.GetDouble(6, 0);
            this._energyDensity.Hydrogen             = input.GetDouble(7, 0);
            this._energyDensity.Spare1               = input.GetDouble(8, 0);
            this._energyDensity.Spare2               = input.GetDouble(9, 0);
            this._massDensity.GasolineConventional   = input.GetDouble(0, 1);
            this._massDensity.GasolineFederal        = input.GetDouble(1, 1);
            this._massDensity.GasolineCalifornia     = input.GetDouble(2, 1);
            this._massDensity.DieselConventional     = input.GetDouble(3, 1);
            this._massDensity.DieselLowSulfur        = input.GetDouble(4, 1);
            this._massDensity.E85                    = input.GetDouble(5, 1);
            this._massDensity.CNG                    = input.GetDouble(6, 1);
            this._massDensity.Hydrogen               = input.GetDouble(7, 1);
            this._massDensity.Spare1                 = input.GetDouble(8, 1);
            this._massDensity.Spare2                 = input.GetDouble(9, 1);
            this._carbonContent.GasolineConventional = input.GetDouble(0, 2);
            this._carbonContent.GasolineFederal      = input.GetDouble(1, 2);
            this._carbonContent.GasolineCalifornia   = input.GetDouble(2, 2);
            this._carbonContent.DieselConventional   = input.GetDouble(3, 2);
            this._carbonContent.DieselLowSulfur      = input.GetDouble(4, 2);
            this._carbonContent.E85                  = input.GetDouble(5, 2);
            this._carbonContent.CNG                  = input.GetDouble(6, 2);
            this._carbonContent.Hydrogen             = input.GetDouble(7, 2);
            this._carbonContent.Spare1               = input.GetDouble(8, 2);
            this._carbonContent.Spare2               = input.GetDouble(9, 2);
            this._soxRate.GasolineConventional = input.GetDouble(0, 3);
            this._soxRate.GasolineFederal      = input.GetDouble(1, 3);
            this._soxRate.GasolineCalifornia   = input.GetDouble(2, 3);
            this._soxRate.DieselConventional   = input.GetDouble(3, 3);
            this._soxRate.DieselLowSulfur      = input.GetDouble(4, 3);
            this._soxRate.E85                  = input.GetDouble(5, 3);
            this._soxRate.CNG                  = input.GetDouble(6, 3);
            this._soxRate.Hydrogen             = input.GetDouble(7, 3);
            this._soxRate.Spare1               = input.GetDouble(8, 3);
            this._soxRate.Spare2               = input.GetDouble(9, 3);
            this._gasolineAssumptions.FuelSavingsLeadingToLowerImports    = input.GetDouble(14, 0);
            this._gasolineAssumptions.FuelSavingsLeadingToReducedRefining = input.GetDouble(15, 0);
            this._gasolineAssumptions.ReducedRefiningFromDomesticCrude    = input.GetDouble(16, 0);
            this._gasolineAssumptions.ReducedRefiningFromImportedCrude    = input.GetDouble(17, 0);
            this._dieselAssumptions.FuelSavingsLeadingToLowerImports    = input.GetDouble(14, 1);
            this._dieselAssumptions.FuelSavingsLeadingToReducedRefining = input.GetDouble(15, 1);
            this._dieselAssumptions.ReducedRefiningFromDomesticCrude    = input.GetDouble(16, 1);
            this._dieselAssumptions.ReducedRefiningFromImportedCrude    = input.GetDouble(17, 1);
            this._e85Assumptions.FuelSavingsLeadingToLowerImports    = input.GetDouble(14, 2);
            this._e85Assumptions.FuelSavingsLeadingToReducedRefining = input.GetDouble(15, 2);
            this._e85Assumptions.ReducedRefiningFromDomesticCrude    = input.GetDouble(16, 2);
            this._e85Assumptions.ReducedRefiningFromImportedCrude    = input.GetDouble(17, 2);
            this._cngAssumptions.FuelSavingsLeadingToLowerImports    = input.GetDouble(14, 3);
            this._cngAssumptions.FuelSavingsLeadingToReducedRefining = input.GetDouble(15, 3);
            this._cngAssumptions.ReducedRefiningFromDomesticCrude    = input.GetDouble(16, 3);
            this._cngAssumptions.ReducedRefiningFromImportedCrude    = input.GetDouble(17, 3);
            this._hydrogenAssumptions.FuelSavingsLeadingToLowerImports    = input.GetDouble(14, 4);
            this._hydrogenAssumptions.FuelSavingsLeadingToReducedRefining = input.GetDouble(15, 4);
            this._hydrogenAssumptions.ReducedRefiningFromDomesticCrude    = input.GetDouble(16, 4);
            this._hydrogenAssumptions.ReducedRefiningFromImportedCrude    = input.GetDouble(17, 4);
            this._spare1Assumptions.FuelSavingsLeadingToLowerImports    = input.GetDouble(14, 5);
            this._spare1Assumptions.FuelSavingsLeadingToReducedRefining = input.GetDouble(15, 5);
            this._spare1Assumptions.ReducedRefiningFromDomesticCrude    = input.GetDouble(16, 5);
            this._spare1Assumptions.ReducedRefiningFromImportedCrude    = input.GetDouble(17, 5);
            this._spare2Assumptions.FuelSavingsLeadingToLowerImports    = input.GetDouble(14, 6);
            this._spare2Assumptions.FuelSavingsLeadingToReducedRefining = input.GetDouble(15, 6);
            this._spare2Assumptions.ReducedRefiningFromDomesticCrude    = input.GetDouble(16, 6);
            this._spare2Assumptions.ReducedRefiningFromImportedCrude    = input.GetDouble(17, 6);
            this._gasolineMix.GasolineConventional = input.GetDouble(22, 0);
            this._gasolineMix.GasolineFederal      = input.GetDouble(23, 0);
            this._gasolineMix.GasolineCalifornia   = input.GetDouble(24, 0);
            this._gasolineMix.DieselConventional   = input.GetDouble(25, 0);
            this._gasolineMix.DieselLowSulfur      = input.GetDouble(26, 0);
            this._gasolineMix.E85                  = input.GetDouble(27, 0);
            this._gasolineMix.CNG                  = input.GetDouble(28, 0);
            this._gasolineMix.Hydrogen             = input.GetDouble(29, 0);
            this._gasolineMix.Spare1               = input.GetDouble(30, 0);
            this._gasolineMix.Spare2               = input.GetDouble(31, 0);
        }
        private void ParseMonteCarlo(Input input, int sheetIndex)
        {
            input.ActivateWorksheet(sheetIndex);
            string[] discRates = input.GetString(2, 1).Split(',');
            int discRatesCount = discRates.Length;
            this._monteCarlo.DiscountRates = new double[discRatesCount];
            for (int i = 0; i < discRatesCount; i++)
            {
                this._monteCarlo.DiscountRates[i] = Global.GetDouble(discRates[i]);
            }
            this._monteCarlo.FuelPathLow       = input.GetDouble( 4, 1);
            this._monteCarlo.FuelPathReference = input.GetDouble( 5, 1);
            this._monteCarlo.FuelPathHigh      = input.GetDouble( 6, 1);
            this._monteCarlo.ReboundAlphaShape = input.GetDouble( 8, 1);
            this._monteCarlo.ReboundBetaShape  = input.GetDouble( 9, 1);
            this._monteCarlo.ReboundScale      = input.GetDouble(10, 1);
            this._monteCarlo.ReboundBase       = input.GetDouble(11, 1);
            this._monteCarlo.CO2PathLow      = input.GetDouble(13, 1);
            this._monteCarlo.CO2PathAverage  = input.GetDouble(14, 1);
            this._monteCarlo.CO2PathHigh     = input.GetDouble(15, 1);
            this._monteCarlo.CO2PathVeryHigh = input.GetDouble(16, 1);
            this._monteCarlo.MonopsonyMean              = input.GetDouble(18, 1);
            this._monteCarlo.MonopsonyStandardDeviation = input.GetDouble(19, 1);
            this._monteCarlo.PriceShockMean              = input.GetDouble(21, 1);
            this._monteCarlo.PriceShockStandardDeviation = input.GetDouble(22, 1);
            this._monteCarlo.MilitarySecurityAlternativeCost        = input.GetDouble(24, 1);
            this._monteCarlo.MilitarySecurityAlternativeProbability = input.GetDouble(25, 1);
        }
        protected virtual void ParseAdditional(Input input)
        {
        }
        private void PreCalc()
        {
            this.PreCalcFuelTypes(this._energyDensity,
                out this._gasolineEnergyDensity, out this._dieselEnergyDensity,
                out this._e85EnergyDensity, out this._cngEnergyDensity, out this._hydrogenEnergyDensity,
                out this._spare1EnergyDensity, out this._spare2EnergyDensity);
            this.PreCalcFuelTypes(this._massDensity,
                out this._gasolineMassDensity, out this._dieselMassDensity,
                out this._e85MassDensity, out this._cngMassDensity, out this._hydrogenMassDensity,
                out this._spare1MassDensity, out this._spare2MassDensity);
            this.PreCalcFuelTypes(this._carbonContent,
                out this._gasolineCarbonContent, out this._dieselCarbonContent,
                out this._e85CarbonContent, out this._cngCarbonContent, out this._hydrogenCarbonContent,
                out this._spare1CarbonContent, out this._spare2CarbonContent);
            this.PreCalcFuelTypes(this._soxRate,
                out this._gasolineSoxRate, out this._dieselSoxRate,
                out this._e85SoxRate, out this._cngSoxRate, out this._hydrogenSoxRate,
                out this._spare1SoxRate, out this._spare2SoxRate);
            this.PreCalcEmissions(this._coEmissions, out this._coGasolineEmissions,
                out this._coDieselEmissions, out this._coE85Emissions, out this._coCngEmissions,
                out this._coHydrogenEmissions, out this._coSpare1Emissions, out this._coSpare2Emissions);
            this.PreCalcEmissions(this._vocEmissions, out this._vocGasolineEmissions,
                out this._vocDieselEmissions, out this._vocE85Emissions, out this._vocCngEmissions,
                out this._vocHydrogenEmissions, out this._vocSpare1Emissions, out this._vocSpare2Emissions);
            this.PreCalcEmissions(this._noxEmissions, out this._noxGasolineEmissions,
                out this._noxDieselEmissions, out this._noxE85Emissions, out this._noxCngEmissions, 
                out this._noxHydrogenEmissions, out this._noxSpare1Emissions, out this._noxSpare2Emissions);
            this.PreCalcEmissions(this._pmEmissions, out this._pmGasolineEmissions,
                out this._pmDieselEmissions, out this._pmE85Emissions, out this._pmCngEmissions,
                out this._pmHydrogenEmissions, out this._pmSpare1Emissions, out this._pmSpare2Emissions);
            this.PreCalcEmissions(this._soxEmissions, out this._soxGasolineEmissions,
                out this._soxDieselEmissions, out this._soxE85Emissions, out this._soxCngEmissions,
                out this._soxHydrogenEmissions, out this._soxSpare1Emissions, out this._soxSpare2Emissions);
            this.PreCalcEmissions(this._co2Emissions, out this._co2GasolineEmissions,
                out this._co2DieselEmissions, out this._co2E85Emissions, out this._co2CngEmissions,
                out this._co2HydrogenEmissions, out this._co2Spare1Emissions, out this._co2Spare2Emissions);
        }
        private void PreCalcFuelTypes(FuelTypes value, out double gasoline, out double diesel,
            out double e85, out double cng, out double hydrogen, out double spare1, out double spare2)
        {
            gasoline = this._gasolineMix.GasolineConventional * value.GasolineConventional +
                       this._gasolineMix.GasolineFederal      * value.GasolineFederal      +
                       this._gasolineMix.GasolineCalifornia   * value.GasolineCalifornia;
            diesel   = this._gasolineMix.DieselConventional * value.DieselConventional +
                       this._gasolineMix.DieselLowSulfur    * value.DieselLowSulfur;
            e85      = this._gasolineMix.E85      * value.E85;
            cng      = this._gasolineMix.CNG      * value.CNG;
            hydrogen = this._gasolineMix.Hydrogen * value.Hydrogen;
            spare1   = this._gasolineMix.Spare1   * value.Spare1;
            spare2   = this._gasolineMix.Spare2   * value.Spare2;
        }
        private void PreCalcEmissionsFuelTypes(FuelTypes value, out double gasoline, out double diesel,
            out double e85, out double cng, out double hydrogen, out double spare1, out double spare2)
        {
            this.PreCalcFuelTypes(value, out gasoline, out diesel, out e85, out cng, out hydrogen, out spare1, out spare2);
            gasoline *= (this._gasolineEnergyDensity * 1e-6);
            diesel   *= (this._dieselEnergyDensity   * 1e-6);
            e85      *= (this._e85EnergyDensity      * 1e-6);
            cng      *= (this._cngEnergyDensity      * 1e-6);
            hydrogen *= (this._hydrogenEnergyDensity * 1e-6);
            spare1   *= (this._spare1EnergyDensity   * 1e-6);
            spare2   *= (this._spare2EnergyDensity   * 1e-6);
        }
        private void PreCalcEmissions(TotalUpstreamEmissions value, out AdjustedUpstreamEmissions gasoline,
            out AdjustedUpstreamEmissions diesel, out AdjustedUpstreamEmissions e85, 
            out AdjustedUpstreamEmissions cng, out AdjustedUpstreamEmissions hydrogen,
            out AdjustedUpstreamEmissions spare1, out AdjustedUpstreamEmissions spare2)
        {
            this.PreCalcEmissionsFuelTypes(value.Extraction, out gasoline.Extraction, out diesel.Extraction, 
                out e85.Extraction, out cng.Extraction, out hydrogen.Extraction, 
                out spare1.Extraction, out spare2.Extraction);
            this.PreCalcEmissionsFuelTypes(value.Transportation, out gasoline.Transportation, out diesel.Transportation,
                out e85.Transportation, out cng.Transportation, out hydrogen.Transportation, 
                out spare1.Transportation, out spare2.Transportation);
            this.PreCalcEmissionsFuelTypes(value.Refining, out gasoline.Refining, out diesel.Refining, 
                out e85.Refining, out cng.Refining, out hydrogen.Refining,
                out spare1.Refining, out spare2.Refining);
            this.PreCalcEmissionsFuelTypes(value.TSD, out gasoline.TSD, out diesel.TSD,
                out e85.TSD, out cng.TSD, out hydrogen.TSD, out spare1.TSD, out spare2.TSD);
        }
        public void UpdateForMonteCarlo(double discountRate, double reboundEffect, double monopsony, double priceShock,
            double militarySecurity)
        {
            this._economicValues.DiscountRate     = discountRate;
            this._economicValues.ReboundEffect    = reboundEffect;
            this._economicCosts .Monopsony        = monopsony;
            this._economicCosts .PriceShock       = priceShock;
            this._economicCosts .MilitarySecurity = militarySecurity;
            this._economicCosts .Total            = (monopsony + priceShock + militarySecurity);
            this._economicCosts .TotalBBL         = this._economicCosts.Total * 42;
        }
        public void UpdateParametersOverrides(ParametersOverrides overrides)
        {
            if (overrides.OverrideReboundEffect         ) { this._economicValues.ReboundEffect          = overrides.ReboundEffect; }
            if (overrides.OverrideDiscountRate          ) { this._economicValues.DiscountRate           = overrides.DiscountRate; }
            if (overrides.OverrideVehicleTravelTimeValue) { this._economicValues.VehicleTravelTimeValue = overrides.VehicleTravelTimeValue; }
            if (overrides.OverrideMilitarySecurity      ) { this._economicCosts .MilitarySecurity       = overrides.MilitarySecurity; }
        }
        #endregion
        #region 
        public Parameters.VehicleSurvivability Survivability { get { return this._survivability; } }
        public Parameters.MilesDriven VehicleMilesDriven { get { return this._milesDriven; } }
        public Parameters.ForecastDataValues ForecastData { get { return this._forecastData; } }
        public Parameters.EconomicDataValues EconomicValues { get { return this._economicValues; } }
        public Parameters.EconomicCostsValues EconomicCosts { get { return this._economicCosts; } }
        public Parameters.ExternalCostsValues ExternalCosts { get { return this._externalCosts; } }
        public Parameters.EmissionDamageCosts EmissionCosts { get { return this._emissionCosts; } }
        public Parameters.TotalUpstreamEmissions COEmissions  { get { return this._coEmissions; } }
        public Parameters.TotalUpstreamEmissions VOCEmissions { get { return this._vocEmissions; } }
        public Parameters.TotalUpstreamEmissions NOXEmissions { get { return this._noxEmissions; } }
        public Parameters.TotalUpstreamEmissions PMEmissions  { get { return this._pmEmissions; } }
        public Parameters.TotalUpstreamEmissions SOXEmissions { get { return this._soxEmissions; } }
        public Parameters.TotalUpstreamEmissions CO2Emissions { get { return this._co2Emissions; } }
        public Parameters.FuelTypes EnergyDensity { get { return this._energyDensity; } }
        public Parameters.FuelTypes MassDensity   { get { return this._massDensity;   } }
        public Parameters.FuelTypes CarbonContent { get { return this._carbonContent; } }
        public Parameters.FuelTypes SOXRate { get { return this._soxRate; } }
        public Parameters.FuelTypes GasolineMix   { get { return this._gasolineMix;   } }
        public Parameters.FuelImportAssumptions GasolineImportAssumptions { get { return this._gasolineAssumptions; } }
        public Parameters.FuelImportAssumptions DieselImportAssumptions   { get { return this._dieselAssumptions;   } }
        public Parameters.FuelImportAssumptions E85ImportAssumptions      { get { return this._e85Assumptions;      } }
        public Parameters.FuelImportAssumptions CNGImportAssumptions      { get { return this._cngAssumptions;      } }
        public Parameters.FuelImportAssumptions HydrogenImportAssumptions { get { return this._hydrogenAssumptions; } }
        public Parameters.FuelImportAssumptions Spare1ImportAssumptions   { get { return this._spare1Assumptions;   } }
        public Parameters.FuelImportAssumptions Spare2ImportAssumptions   { get { return this._spare2Assumptions;   } }
        public double GasolineEnergyDensity { get { return this._gasolineEnergyDensity; } }
        public double DieselEnergyDensity   { get { return this._dieselEnergyDensity;   } }
        public double E85EnergyDensity      { get { return this._e85EnergyDensity;      } }
        public double CNGEnergyDensity      { get { return this._cngEnergyDensity;      } }
        public double HydrogenEnergyDensity { get { return this._hydrogenEnergyDensity; } }
        public double Spare1EnergyDensity   { get { return this._spare1EnergyDensity;   } }
        public double Spare2EnergyDensity   { get { return this._spare2EnergyDensity;   } }
        public double GasolineMassDensity { get { return this._gasolineMassDensity; } }
        public double DieselMassDensity   { get { return this._dieselMassDensity;   } }
        public double E85MassDensity      { get { return this._e85MassDensity;      } }
        public double CNGMassDensity      { get { return this._cngMassDensity;      } }
        public double HydrogenMassDensity { get { return this._hydrogenMassDensity; } }
        public double Spare1MassDensity   { get { return this._spare1MassDensity;   } }
        public double Spare2MassDensity   { get { return this._spare2MassDensity;   } }
        public double GasolineCarbonContent { get { return this._gasolineCarbonContent; } }
        public double DieselCarbonContent   { get { return this._dieselCarbonContent;   } }
        public double E85CarbonContent      { get { return this._e85CarbonContent;      } }
        public double CNGCarbonContent      { get { return this._cngCarbonContent;      } }
        public double HydrogenCarbonContent { get { return this._hydrogenCarbonContent; } }
        public double Spare1CarbonContent   { get { return this._spare1CarbonContent;   } }
        public double Spare2CarbonContent   { get { return this._spare2CarbonContent;   } }
        public double GasolineSOXRate { get { return this._gasolineSoxRate; } }
        public double DieselSOXRate   { get { return this._dieselSoxRate;   } }
        public double E85SOXRate      { get { return this._e85SoxRate;      } }
        public double CNGSOXRate      { get { return this._cngSoxRate;      } }
        public double HydrogenSOXRate { get { return this._hydrogenSoxRate; } }
        public double Spare1SOXRate   { get { return this._spare1SoxRate;   } }
        public double Spare2SOXRate   { get { return this._spare2SoxRate;   } }
        public Parameters.AdjustedUpstreamEmissions COGasolineEmissions { get { return this._coGasolineEmissions; } }
        public Parameters.AdjustedUpstreamEmissions CODieselEmissions   { get { return this._coDieselEmissions;   } }
        public Parameters.AdjustedUpstreamEmissions COE85Emissions      { get { return this._coE85Emissions;      } }
        public Parameters.AdjustedUpstreamEmissions COCNGEmissions      { get { return this._coCngEmissions;      } }
        public Parameters.AdjustedUpstreamEmissions COHydrogenEmissions { get { return this._coHydrogenEmissions; } }
        public Parameters.AdjustedUpstreamEmissions COSpare1Emissions   { get { return this._coSpare1Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions COSpare2Emissions   { get { return this._coSpare2Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions VOCGasolineEmissions { get { return this._vocGasolineEmissions; } }
        public Parameters.AdjustedUpstreamEmissions VOCDieselEmissions   { get { return this._vocDieselEmissions;   } }
        public Parameters.AdjustedUpstreamEmissions VOCE85Emissions      { get { return this._vocE85Emissions;      } }
        public Parameters.AdjustedUpstreamEmissions VOCCNGEmissions      { get { return this._vocCngEmissions;      } }
        public Parameters.AdjustedUpstreamEmissions VOCHydrogenEmissions { get { return this._vocHydrogenEmissions; } }
        public Parameters.AdjustedUpstreamEmissions VOCSpare1Emissions   { get { return this._vocSpare1Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions VOCSpare2Emissions   { get { return this._vocSpare2Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions NOXGasolineEmissions { get { return this._noxGasolineEmissions; } }
        public Parameters.AdjustedUpstreamEmissions NOXDieselEmissions   { get { return this._noxDieselEmissions;   } }
        public Parameters.AdjustedUpstreamEmissions NOXE85Emissions      { get { return this._noxE85Emissions;      } }
        public Parameters.AdjustedUpstreamEmissions NOXCNGEmissions      { get { return this._noxCngEmissions;      } }
        public Parameters.AdjustedUpstreamEmissions NOXHydrogenEmissions { get { return this._noxHydrogenEmissions; } }
        public Parameters.AdjustedUpstreamEmissions NOXSpare1Emissions   { get { return this._noxSpare1Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions NOXSpare2Emissions   { get { return this._noxSpare2Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions PMGasolineEmissions { get { return this._pmGasolineEmissions; } }
        public Parameters.AdjustedUpstreamEmissions PMDieselEmissions   { get { return this._pmDieselEmissions;   } }
        public Parameters.AdjustedUpstreamEmissions PME85Emissions      { get { return this._pmE85Emissions;      } }
        public Parameters.AdjustedUpstreamEmissions PMCNGEmissions      { get { return this._pmCngEmissions;      } }
        public Parameters.AdjustedUpstreamEmissions PMHydrogenEmissions { get { return this._pmHydrogenEmissions; } }
        public Parameters.AdjustedUpstreamEmissions PMSpare1Emissions   { get { return this._pmSpare1Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions PMSpare2Emissions   { get { return this._pmSpare2Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions SOXGasolineEmissions { get { return this._soxGasolineEmissions; } }
        public Parameters.AdjustedUpstreamEmissions SOXDieselEmissions   { get { return this._soxDieselEmissions;   } }
        public Parameters.AdjustedUpstreamEmissions SOXE85Emissions      { get { return this._soxE85Emissions;      } }
        public Parameters.AdjustedUpstreamEmissions SOXCNGEmissions      { get { return this._soxCngEmissions;      } }
        public Parameters.AdjustedUpstreamEmissions SOXHydrogenEmissions { get { return this._soxHydrogenEmissions; } }
        public Parameters.AdjustedUpstreamEmissions SOXSpare1Emissions   { get { return this._soxSpare1Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions SOXSpare2Emissions   { get { return this._soxSpare2Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions CO2GasolineEmissions { get { return this._co2GasolineEmissions; } }
        public Parameters.AdjustedUpstreamEmissions CO2DieselEmissions   { get { return this._co2DieselEmissions;   } }
        public Parameters.AdjustedUpstreamEmissions CO2E85Emissions      { get { return this._co2E85Emissions;      } }
        public Parameters.AdjustedUpstreamEmissions CO2CNGEmissions      { get { return this._co2CngEmissions;      } }
        public Parameters.AdjustedUpstreamEmissions CO2HydrogenEmissions { get { return this._co2HydrogenEmissions; } }
        public Parameters.AdjustedUpstreamEmissions CO2Spare1Emissions   { get { return this._co2Spare1Emissions;   } }
        public Parameters.AdjustedUpstreamEmissions CO2Spare2Emissions   { get { return this._co2Spare2Emissions;   } }
        public Parameters.MonteCarloValues MonteCarlo { get { return this._monteCarlo; } }
        #endregion
        #region 
        private Parameters.VehicleSurvivability _survivability;
        private Parameters.MilesDriven _milesDriven;
        private Parameters.ForecastDataValues _forecastData;
        private Parameters.EconomicDataValues _economicValues;
        private Parameters.EconomicCostsValues _economicCosts;
        private Parameters.ExternalCostsValues _externalCosts;
        private Parameters.EmissionDamageCosts _emissionCosts;
        private Parameters.TotalUpstreamEmissions _coEmissions;
        private Parameters.TotalUpstreamEmissions _vocEmissions;
        private Parameters.TotalUpstreamEmissions _noxEmissions;
        private Parameters.TotalUpstreamEmissions _pmEmissions;
        private Parameters.TotalUpstreamEmissions _soxEmissions;
        private Parameters.TotalUpstreamEmissions _co2Emissions;
        private Parameters.FuelTypes _energyDensity;
        private Parameters.FuelTypes _massDensity;
        private Parameters.FuelTypes _carbonContent;
        private Parameters.FuelTypes _soxRate;
        private Parameters.FuelTypes _gasolineMix;
        private Parameters.FuelImportAssumptions _gasolineAssumptions;
        private Parameters.FuelImportAssumptions _dieselAssumptions;
        private Parameters.FuelImportAssumptions _e85Assumptions;
        private Parameters.FuelImportAssumptions _cngAssumptions;
        private Parameters.FuelImportAssumptions _hydrogenAssumptions;
        private Parameters.FuelImportAssumptions _spare1Assumptions;
        private Parameters.FuelImportAssumptions _spare2Assumptions;
        private double _gasolineEnergyDensity;
        private double _dieselEnergyDensity;
        private double _e85EnergyDensity;
        private double _cngEnergyDensity;
        private double _hydrogenEnergyDensity;
        private double _spare1EnergyDensity;
        private double _spare2EnergyDensity;
        private double _gasolineMassDensity;
        private double _dieselMassDensity;
        private double _e85MassDensity;
        private double _cngMassDensity;
        private double _hydrogenMassDensity;
        private double _spare1MassDensity;
        private double _spare2MassDensity;
        private double _gasolineCarbonContent;
        private double _dieselCarbonContent;
        private double _e85CarbonContent;
        private double _cngCarbonContent;
        private double _hydrogenCarbonContent;
        private double _spare1CarbonContent;
        private double _spare2CarbonContent;
        private double _gasolineSoxRate;
        private double _dieselSoxRate;
        private double _e85SoxRate;
        private double _cngSoxRate;
        private double _hydrogenSoxRate;
        private double _spare1SoxRate;
        private double _spare2SoxRate;
        private Parameters.AdjustedUpstreamEmissions _coGasolineEmissions;
        private Parameters.AdjustedUpstreamEmissions _coDieselEmissions;
        private Parameters.AdjustedUpstreamEmissions _coE85Emissions;
        private Parameters.AdjustedUpstreamEmissions _coCngEmissions;
        private Parameters.AdjustedUpstreamEmissions _coHydrogenEmissions;
        private Parameters.AdjustedUpstreamEmissions _coSpare1Emissions;
        private Parameters.AdjustedUpstreamEmissions _coSpare2Emissions;
        private Parameters.AdjustedUpstreamEmissions _vocGasolineEmissions;
        private Parameters.AdjustedUpstreamEmissions _vocDieselEmissions;
        private Parameters.AdjustedUpstreamEmissions _vocE85Emissions;
        private Parameters.AdjustedUpstreamEmissions _vocCngEmissions;
        private Parameters.AdjustedUpstreamEmissions _vocHydrogenEmissions;
        private Parameters.AdjustedUpstreamEmissions _vocSpare1Emissions;
        private Parameters.AdjustedUpstreamEmissions _vocSpare2Emissions;
        private Parameters.AdjustedUpstreamEmissions _noxGasolineEmissions;
        private Parameters.AdjustedUpstreamEmissions _noxDieselEmissions;
        private Parameters.AdjustedUpstreamEmissions _noxE85Emissions;
        private Parameters.AdjustedUpstreamEmissions _noxCngEmissions;
        private Parameters.AdjustedUpstreamEmissions _noxHydrogenEmissions;
        private Parameters.AdjustedUpstreamEmissions _noxSpare1Emissions;
        private Parameters.AdjustedUpstreamEmissions _noxSpare2Emissions;
        private Parameters.AdjustedUpstreamEmissions _pmGasolineEmissions;
        private Parameters.AdjustedUpstreamEmissions _pmDieselEmissions;
        private Parameters.AdjustedUpstreamEmissions _pmE85Emissions;
        private Parameters.AdjustedUpstreamEmissions _pmCngEmissions;
        private Parameters.AdjustedUpstreamEmissions _pmHydrogenEmissions;
        private Parameters.AdjustedUpstreamEmissions _pmSpare1Emissions;
        private Parameters.AdjustedUpstreamEmissions _pmSpare2Emissions;
        private Parameters.AdjustedUpstreamEmissions _soxGasolineEmissions;
        private Parameters.AdjustedUpstreamEmissions _soxDieselEmissions;
        private Parameters.AdjustedUpstreamEmissions _soxE85Emissions;
        private Parameters.AdjustedUpstreamEmissions _soxCngEmissions;
        private Parameters.AdjustedUpstreamEmissions _soxHydrogenEmissions;
        private Parameters.AdjustedUpstreamEmissions _soxSpare1Emissions;
        private Parameters.AdjustedUpstreamEmissions _soxSpare2Emissions;
        private Parameters.AdjustedUpstreamEmissions _co2GasolineEmissions;
        private Parameters.AdjustedUpstreamEmissions _co2DieselEmissions;
        private Parameters.AdjustedUpstreamEmissions _co2E85Emissions;
        private Parameters.AdjustedUpstreamEmissions _co2CngEmissions;
        private Parameters.AdjustedUpstreamEmissions _co2HydrogenEmissions;
        private Parameters.AdjustedUpstreamEmissions _co2Spare1Emissions;
        private Parameters.AdjustedUpstreamEmissions _co2Spare2Emissions;
        private Parameters.MonteCarloValues _monteCarlo;
        #endregion
    }
}

